# flcheck --
#
# This file implements package flcheck, a fusion logic modelchecker 
# using the CUDD BDD and BuDDy packages
#
# Copyright (C) 2009-2025  Antonio Cau, Ben Moszkowski and Helge Janicke

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# FL derived
#####################################

# for historical reason and is defined
proc and { a b } {
    return "( ($a) and ($b) )"
}

proc or { a b } {
    return "( ($a) or ($b) )"
}

proc true {} {
   return "true"
}

proc false {} {
   return "false"
}

proc not { a } {
    return "(not ($a))"
}

proc imp {a b} {
    return "(($a) imp ($b))"
}

proc equiv {a b} {
   # set x "not{$a}"
   # set y "not{$b}"
   # return "{{$a and $b} or {$x and $y}}"
   return "( ($a) equiv ($b) )" 
}

proc l_and {alist} {
    set x "true"
    foreach e $alist {
        set x "( ($x) and ($e) )"
    }
    return $x
}

proc l_or {alist} {
    set x "false"
    foreach e $alist {
        set x "( ($x) or ($e) )"
    }
    return $x
}

proc tl_and {alist} {
    set x "true"
    foreach e $alist {
        set x "(and $x $e)"
    }
    return $x
}

proc tl_or {alist} {
    set x "false"
    foreach e $alist {
        set x "(or $x $e)"
    }
    return $x
}

#############
# derived future fusion expressions

proc len_e { n } {
    if {$n < 0 } { 
        return "test(false)"
    } else {
      if {$n == 0 } {
          return "(test(true))"
      } else {
          return "(step(true);[len_e [expr $n -1]])"
      }
    }
}

proc len_r_e { n } {
    return "([len_e $n])"
}

proc len_eq_e { n } {
    return "([len_e $n])"
}

proc len_r_eq_e { n } {
    return "([len_e $n])"
}

proc len_le_e { n } {
    if { $n == 0 } {
        return "(test(true))"
    } else {
        return "(test(true) or ([len_le_e [expr $n -1]];step(true)))"
    }
}

proc len_r_le_e { n } {
    return "([len_le_e $n])"
}

proc len_lt_e { n } {
    if { $n == 0 } {
        return "(test(false))"
    } else {
        return "([len_le_e [expr $n -1]])"
    }
}

proc len_r_lt_e { n } {
    return"([len_lt_e $n])"
}

proc len_ge_e { n } {
    return "(([len_eq_e $n]);chopstar(step(true)))"
}

proc len_r_ge_e { n } {
    return "([len_ge_e $n])"
}

proc len_gt_e { n } {
    return "([len_ge_e [incr n]])"
}

proc len_r_gt_e { n } {
    return "([len_gt_e $n])"
}

proc keep_e { e } {
    return "(chopstar(step($e)))"
}

proc keep_r_e { e } {
    return "([keep_e $e])"
}

proc keep_eq_e { k e } {
    if { $k == 0 } {
        return "(test(true))"
    } else {
        return "(step($e);([keep_eq_e [expr $k -1] $e]))"
    }
}

proc keep_r_eq_e { k e } {
    return "([keep_eq_e $k $e])"
}

proc keep_le_e { k e } {
    if { $k == 0 } {
        return "(test(true))"
    } else {
        return "(test(true) or (step($e);([keep_le_e [expr $k -1] $e])))"
    }
}

proc keep_r_le_e { k e } {
    return "([keep_le_e $k $e])"
}

proc keep_lt_e { k e } {
    if { $k == 0 } {
        return "(test(false))"
    } else {
        return "([keep_le_e [expr $k -1] $e])"
    }
}

proc keep_r_lt_e { k e } {
    return "([keep_lt_e $k $e])"
}

proc keep_ge_e { k e } {
    return "(([keep_eq_e $k $e]);([keep_e $e]))"
}

proc keep_r_ge_e { k e } {
    return "([keep_ge_e $k $e])"
}

proc keep_gt_e { k e } {
    return "([keep_ge_e [incr k] $e])"
}

proc keep_r_gt_e { k e } {
    return "([keep_gt_e $k $e])"
}

proc stable_e { e } {
    set tt "(($e) equiv (next ($e)))"
    return "([keep_e $tt])"
}

proc stable_r_e { e } {
    return "([stable_e $e])"
}

proc stable_eq_e { k e } {
    set tt "(($e) equiv (next ($e)))"
    return "([keep_eq_e $k $tt])"
}

proc stable_r_eq_e { k e } {
    return "([stable_eq_e  $k $e])"
}

proc stable_ge_e { k e } {
    set tt "(($e) equiv (next ($e)))"
    return "([keep_ge_e $k $tt])"
}

proc stable_r_ge_e { k e } {
    return "([stable_ge_e $k $e])"
}

proc true_e {} {
    return "(chopstar(step(true)))"
}

proc true_r_e {} {
    return "([true_e])"
}

proc sometimes_e { e } {
    return "([true_e];($e))"
}

proc sometimes_r_e { e } {
    return "([sometimes_e $e])"
}

proc skip_e {} {
    return "step(true)"
}

proc skip_r_e {} {
    return "([skip_e])"
}

proc empty_e {} {
    return "test(true)"
}

proc empty_r_e {} {
    return "([empty_e])"
}

proc star_e { e } {
    return "(chopstar($e))"
}

proc star_r_e { e } {
    return "([star_e $e])"
}

proc star_eq_e { k e } {
    if { $k == 0 } {
        return "test(true)"
    } else {
        return "(($e);([star_eq_e [expr $k -1] $e]))"
    }
}

proc star_r_eq_e { k e } {
    return "([star_eq_e $k $e])"
}

proc star_le_e { k e } {
    if { $k == 0 } {
        return "test(true)"
    } else {
        return "(test(true) or (($e);([star_le_e [expr $k-1] $e])))"
    }
}

proc star_r_le_e { k e } {
    return "([star_le_e $k $e])"
}

proc star_lt_e { k e } {
    if { $k == 0 } {
        return "test(false)"
    } else {
        return "([star_le_e [expr $k -1] $e])"
    }
}

proc star_r_lt_e { k e } {
    return "([star_lt_e $k $e])"
}

proc star_ge_e { k e } {
    return "([star_eq_e $k $e];[star_e $e])"
}

proc star_r_ge_e { k e } {
    return "([star_ge_e $k $e])"
}

proc star_gt_e { k e } {
    return "([star_ge_e [incr k] $e])"
}

proc star_r_gt_e { k e } {
    return "([star_gt_e $k $e])"
}

#############
# derived past fusion expressions

proc len_l_e { n } {
    if {$n < 0 } { 
        return "test(false)"
    } else {
      if {$n == 0 } {
          return "(test(true))"
      } else {
          return "([len_l_e [expr $n -1]];pstep(true))"
      }
    }
}

proc len_l_eq_e { n } {
    return "([len_l_e $n])"
}

proc len_l_le_e { n } {
    if { $n == 0 } {
        return "(test(true))"
    } else {
        return "(test(true) or (pstep(true);[len_l_le_e [expr $n -1]]))"
    }
}

proc len_l_lt_e { n } {
    if { $n == 0 } {
        return "(test(false))"
    } else {
        return "([len_l_le_e [expr $n -1]])"
    }
}

proc len_l_ge_e { n } {
    return "(pchopstar(pstep(true));([len_l_eq_e $n]))"
}

proc len_l_gt_e { n } {
    return "([len_l_ge_e [incr n]])"
}

proc keep_l_e { e } {
    return "(pchopstar(pstep($e)))"
}

proc keep_l_eq_e { k e } {
    if { $k == 0 } {
        return "(test(true))"
    } else {
        return "(([keep_l_eq_e [expr $k -1] $e]);pstep($e))"
    }
}

proc keep_l_le_e { k e } {
    if { $k == 0 } {
        return "(test(true))"
    } else {
        return "(test(true) or (([keep_l_le_e [expr $k -1] $e]);pstep($e)))"
    }
}

proc keep_l_lt_e { k e } {
    if { $k == 0 } {
        return "(test(false))"
    } else {
        return "([keep_l_le_e [expr $k -1] $e])"
    }
}

proc keep_l_ge_e { k e } {
    return "(([keep_l_e $e]);([keep_l_eq_e $k $e]))"
}

proc keep_l_gt_e { k e } {
    return "([keep_l_ge_e [incr k] $e])"
}

proc stable_l_e { e } {
    set tt "(($e) equiv (prev ($e)))"
    return "([keep_l_e $tt])"
}

proc stable_l_eq_e { k e } {
    set tt "(($e) equiv (prev ($e)))"
    return "([keep_l_eq_e $k $tt])"
}

proc stable_l_ge_e { k e } {
    set tt "(($e) equiv (prev ($e)))"
    return "([keep_l_ge_e $k $tt])"
}

proc true_l_e {} {
    return "pchopstar(pstep(true))"
}

proc sometimes_l_e { a } {
    return "($a);[true_l_e]"
}

proc skip_l_e {} {
    return "pstep(true)"
}

proc empty_l_e {} {
    return "test(true)"
}

proc star_l_e { e } {
    return "(pchopstar($e))"
}

proc star_l_eq_e { k e } {
    if { $k == 0 } {
        return "test(true)"
    } else {
        return "(([star_l_eq_e [expr $k -1] $e]);($e))"
    }
}

proc star_l_le_e { k e } {
    if { $k == 0 } {
        return "test(true)"
    } else {
        return "(test(true) or (([star_l_le_e [expr $k-1] $e]);($e)))"
    }
}

proc star_l_lt_e { k e } {
    if { $k == 0 } {
        return "test(false)"
    } else {
        return "([star_l_le_e [expr $k -1] $e])"
    }
}

proc star_l_ge_e { k e } {
    return "([star_l_e $e];[star_l_eq_e $k $e])"
}

proc star_l_gt_e { k e } {
    return "([star_l_ge_e [incr k] $e])"
}


########################
# derived right fusion logic

proc len_r { n } {
    return "(<[len_e $n]>[empty_r])"
}

proc next_r {x} {
    return "(<[skip_e]>($x))"
}

proc next { x } {
    return "(next ($x))"
}

proc next_eq_r { k x } {
    return "(<[len_eq_e $k]>($x))"
}

proc next_ge_r { k x } {
    return "(<[len_ge_e $k]>($x))"
}

proc next_gt_r { k x } {
    return "(<[len_gt_e $k]>($x))"
}

proc next_le_r { k x } {
    return "(<[len_le_e $k]>($x))"
}

proc next_lt_r { k x } {
    return "(<[len_lt_e $k]>($x))"
}

proc sometimes_r { x } { 
    return "(<[true_e]>($x))"
}

proc sometimes_u { x } {
    return "(true until ($x))"
}

proc always_r { x } {
    return "not(<[true_e]>(not($x)))"
}

proc always_u { x } {
    return "not(true until (not ($x)))"
}

proc diamond { x } {
    return "[sometimes_u ($x)]" 
}

proc diamond_r { x } {
    return "([diamond $x])"
}

proc diamondplus { x } {
    return "[next [diamond ($x)]]" 
}

proc diamondplus_r { x } {
    return "([diamondplus $x])"
}

proc box { x } {
    return "[always_u ($x)]" 
}

proc box_r { x } {
    return "[always_u ($x)]" 
}

proc more_r {} {    
    return "(<[skip_e]>(true))"
}

proc more {} {
    return "(next true)"
}

proc empty_r {} {    
    return "(not([more_r]))"
}

proc empty {} {
    return "(not(next true))"
}

proc box_m { x } {
    return "[box [imp [more] $x]]" 
}

proc box_m_r { x } {
    return "[box [imp [more] $x]]" 
}

proc diamond_m { x } {
    return "([diamond [and [more] $x]])"
}

proc diamond_m_r { x } {
    return "([diamond [and [more] $x]])"
}

proc di_fin_r { w } {
    return "(<[true_e];test($w)>true)"
} 

proc di_e_r { e } {
    return "(<$e>(true))"
}

# di x = ( x and finite ); true
#proc di_r { x } {
#    return "((($x) and [finite_u])<[skip_l_e]>)"
#}

proc bi_fin_r { w } {
    return "not [di_fin_r [not $w]]"
}

proc bi_e_r { e } {
    return "[not [di_e_r [not $e]]]"
}

proc skip_r {} {    
    return "(<[skip_e]>([empty_r]))"
}

proc skip {} {
    return "[skip_r]" 
}

proc gets_r { x y } {
    set tt "((next ($x)) equiv ($y))" 
    return  "([box_m $tt])"
}

proc assign { x y } {
    return "((next ($x)) equiv ($y))" 
}

proc assign_r { x y } {
    return "((next ($x)) equiv ($y))" 
}

proc finite_r {} {
    return "(<[true_e]>[empty_r])"
}

proc finite_u {} {
    return "(true until [empty])"
}

proc fin_r { x } {
    return "[box [imp [empty] $x]]"
}

proc sfin_r { x } {
    return "[not [fin_r [not $x]]]"
}

proc dd_r { e x } {
    return "(<$e>($x))"
}

proc bb_r { e x } {
    return "(not (<$e>(not ($x))))"
}

proc sometimes_plus_r { x } {
    return "(next [sometimes_r ($x)])"
}

proc sometimes_m_r { x } {
    set tt "([more_r] and ($x))"
    return "([sometimes_r $tt])"
}

proc weaknext_r { x } {
    return "[not [next_r [not $x]]]"
}

proc weaknext { x } {
    return "(not (next (not ($x))))"
}

proc weaknext_eq_r { k x } {
    set tt "(not ($x))" 
    return "[not [next_eq_r $k $tt]]"
}

proc weaknext_ge_r { k x } {
    set tt "(not ($x))" 
    return "[not [next_ge_r $k $tt]]"
}

proc weaknext_gt_r { k x } {
    set tt "(not ($x))" 
    return "[not [next_gt_r $k $tt]]"
}

proc weaknext_le_r { k x } {
    set tt "(not ($x))" 
    return "[not [next_le_r $k $tt]]"
}

proc weaknext_lt_r { k x } {
    set tt "(not ($x))" 
    return "(not ([next_lt_r $k $tt]))"
}

proc weakuntil { x y } {
    set tt "($x until $y)"
    return "([or $tt [box_m $x]])" 
}

proc len_eq_r { n } {
    return "([len_r $n])"
}

proc len_ge_r { n } {
    return "(<[len_e $n]>(true))" 
}

proc len_gt_r { n } {
    return "(<[len_e $n]>[more])"
}

proc len_le_r { n } {
    if { $n == 0 } {
        return "(<test true>[empty])"
    } else {
        return "((<test true>[empty]) or (<[skip_e]>([len_le_r [expr $n -1]])))"
    }
}

proc len_lt_r { n } {
    if { $n == 0 } {
        return "false"
    } else {
        return "[len_le_r [expr $n -1]]"
    }
}

proc while_r { x e } {
    return "(<chopstar(test($x);$e);test(not $x)>[empty])"
}

proc keep_r { e } {
    return "(<[keep_e $e]>([empty]))"
}

proc keep_eq_r { k e } {
    return "(<[keep_eq_e $k $e]>([empty]))"
}

proc keep_le_r { k e } {
    return "(<[keep_le_e $k $e]>([empty]))"
}

proc keep_lt_r { k e } {
    return "(<[keep_lt_e $k $e]>([empty]))"
}

proc keep_ge_r { k e } {
    return "(<[keep_ge_e $k $e]>([empty]))"
}

proc keep_gt_r { k e } {
    return "(<[keep_gt_e $k $e]>([empty]))"
}

proc keepnow_r { e } {
    return "(<step($e)>(true))"
}

proc sfin_u { x } {
    set tt "([empty] and ($x))" 
    return "([diamond $tt])"
}

proc stable_r { x } {
    set tt "(($x) equiv (next ($x)))" 
    return "([keep_r $tt])"
}

proc stable_eq_r { k x } {
    set tt "(($x) equiv (next ($x)))" 
    return "([keep_eq_r $k $tt])"
}

proc stable_ge_r { k x } {
    set tt "(($x) equiv (next ($x)))" 
    return "([keep_ge_r $k $tt])"
}

proc stablenow_r { x } {
    return "(([more]) and (($x) equiv (next ($x))))"
}



########################
# derived past fusion logic


proc len_l { n } {
    return "([empty_l] <[len_l_e $n]>)"
}

proc prev_l {x} {
    return "(($x)<[skip_l_e]>)"
}

proc prev { x } {
    return "(prev ($x))"
}

proc prev_eq_l { k x } {
    return "(($x)<[len_l_eq_e $k]>)"
}

proc prev_ge_l { k x } {
    return "(($x)<[len_l_ge_e $k]>)"
}

proc prev_gt_l { k x } {
    return "(($x)<[len_l_gt_e $k]>)"
}

proc prev_le_l { k x } {
    return "(($x)<[len_l_le_e $k]>)"
}

proc prev_lt_l { k x } {
    return "(($x)<[len_l_lt_e $k]>)"
}

proc sometimes_l { x } { 
    return "(($x)<[true_l_e]>)"
}

proc sometimes_s { x } {
    return "(true since ($x))"
}

proc always_l { x } {
    return "not((not($x))<[true_l_e]>)"
}

proc always_s { x } {
    return "not(true since (not ($x)))"
}

proc diamond_l { x } {
    return "[sometimes_s $x]" 
}

proc diamondplus_l { x } {
    return "[prev [diamond_l $x]]" 
}

proc box_l { x } {
    return "[always_s $x]" 
}

proc more_l {} {
    return "((true) <[skip_l_e]>)"
}

proc more_p {} {
    return "(prev true)"
}

proc empty_l {} {    
    return "[not [more_l]]"
}

proc empty_p {} {    
    return "[not [more_p]]"
}

proc first {} {
    return "[not [prev true]]"
}

proc box_m_l { x } {
    return "[box_l [imp [more_p] $x]]" 
}

proc diamond_m_l { x } {
    return "([diamond_l [and [more_p] $x]])"
}

proc di_fin_l { w } {
    return "(true<test($w);[true_l_e]>)"
}

proc di_e_l { e } {
    return "((true)<$e>)"
}

proc bi_fin_l { w } {
    return "[not [di_fin_l [not $w]]]"
}

proc skip_l {} {    
    return "([empty_l])<[skip_l_e]>"
}

proc gets_l { x y } {
    set tt "((prev ($x)) equiv ($y))" 
    return  "([box_m_l $tt])"
}

proc assign_l { x y } {
    return "((prev ($x)) equiv ($y))" 
}

proc fin_l { x } {
    return "[box_l [imp [empty_l] $x]]"
}

proc sfin_l { x } {
    return "[not [fin_l [not $x]]]"
}

proc dd_l { x e } {
    return "(($x)<$e>)"
}

proc bb_l { x e } {
    return "(not((not ($x))<$e>))"
}

proc sometimes_plus_l { x } {
    return "(prev [sometimes_l ($x)])"
}

proc sometimes_m_l { x } {
    set tt "([more_l] and ($x))"
    return "([sometimes_l $tt])"
}

proc weakprev_l { x } {
    return "[not [prev_l [not $x]]]"
}

proc weakprev { x } {
    return "(not (prev (not ($x))))"
}
proc weakprev_eq_l { k x } {
    set tt "(not ($x))" 
    return "[not [prev_eq_l $k $tt]]"
}

proc weakprev_ge_l { k x } {
    set tt "(not ($x))" 
    return "[not [prev_ge_l $k $tt]]"
}

proc weakprev_gt_l { k x } {
    set tt "(not ($x))" 
    return "[not [prev_gt_l $k $tt]]"
}

proc weakprev_le_l { k x } {
    set tt "(not ($x))" 
    return "[not [prev_le_l $k $tt]]"
}

proc weakprev_lt_l { k x } {
    set tt "(not ($x))" 
    return "[not [prev_lt_l $k $tt]]"
}

proc weaksince { x y } {
    set tt "($x since $y)"
    return "([or $tt [box_m_l $x]])" 
}

proc len_eq_l { n } {
    return "([len_l $n])"
}

proc len_ge_l { n } {
    return "((true)<[len_l_e $n]>)" 
}

proc len_gt_l { n } {
    return "([more_l]<[len_l_e $n]>)"
}

proc len_le_l { n } {
    if { $n == 0 } {
        return "([empty_l]<test true>)"
    } else {
        return "(([empty_l]<test true>) or (([len_le_l [expr $n -1]])<[skip_l_e]>))"
    }
}

proc len_lt_l { n } {
    if { $n == 0 } {
        return "false"
    } else {
        return "[len_le_l [expr $n -1]]"
    }
}

proc while_l { x e } {
    return "([empty_l]<test(not $x);pchopstar($e;test($x))>)"
}

proc keep_l { e } {
    return "(([empty_l])<[keep_l_e $e]>)"
}

proc keep_eq_l { k e } {
    return "(([empty_l])<[keep_l_eq_e $k $e]>)"
}

proc keep_le_l { k e } {
    return "(([empty_l])<[keep_l_le_e $k $e]>)"
}

proc keep_lt_l { k e } {
    return "(([empty])<[keep_l_lt_e $k $e]>)"
}

proc keep_ge_l { k e } {
    return "(([empty_l])<[keep_l_ge_e $k $e]>)"
}

proc keep_gt_l { k e } {
    return "(([empty_l])<[keep_l_gt_e $k $e]>)"
}

proc keepnow_l { e } {
    return "((true)<pstep($e)>)"
}

proc sfin_s { x } {
    set tt "([empty_l] and ($x))" 
    return "([diamond_s $tt])"
}

proc stable_l { x } {
    set tt "(($x) equiv (prev ($x)))" 
    return "([keep_l $tt])"
}

proc stable_eq_l { k x } {
    set tt "(($x) equiv (prev ($x)))" 
    return "([keep_eq_l $k $tt])"
}

proc stable_ge_l { k x } {
    set tt "(($x) equiv (prev ($x)))" 
    return "([keep_ge_l $k $tt])"
}

proc stablenow_l { x } {
    return "(([more_l]) and (($x) equiv (prev ($x))))"
}

