#!/bin/sh
# flcheck.tcl \
exec tclsh8.6 "$0" ${1+"$@"}

# flcheck --
#
# This file implements package flcheck, a fusion logic modelchecker 
# using the CUDD and BuDDy BDD packages
#
# Copyright (C) 2009-2025  Antonio Cau, Ben Moszkowski and Helge Janicke

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Libraries needed
#

# Tk library
  package require Tk
# tile library
  package require tile
# fusion logic parser/lexer
  if {![info exist ::starkit::topdir]} {
    source fusion_logic.tcl
  } else {
     source $::starkit::topdir/fusion_logic.tcl
  }
# tcldot is broken so we use dot directly
#load ./libtcldot.so.0.0.0

# Tile theme used by GUI
#
namespace eval tile::theme::strl {

    package provide tile::theme::strl 0.9

    variable colors
    array set colors {
	-frame  	"#eadaa5"
	-lighter	"#f5f5f5"
	-window	 	"#ffffff"
	-selectbg	"#ffff33"
	-selectfg	"#000000"
	-disabledfg	"#666666"
    }

    ttk::style theme create strl -parent clam -settings {

	ttk::style configure . \
	    -borderwidth 	1 \
	    -background 	$colors(-frame) \
	    -fieldbackground	$colors(-window) \
	    -troughcolor	$colors(-lighter) \
	    -selectbackground	$colors(-selectbg) \
	    -selectforeground	$colors(-selectfg) \
	    ;
	ttk::style map . -foreground [list disabled $colors(-disabledfg)]

	## Buttons.
	#
	ttk::style configure TButton \
            anchor center -width -11 -padding "1 1" -relief raised -shiftrelief 1 \
	    -highlightthickness 1 -highlightcolor $colors(-frame) -foreground black

	ttk::style map TButton -relief {
	    {pressed !disabled} 	sunken
	    {active !disabled} 	raised
	} -highlightcolor {alternate black}

        ttk::style configure Red.TButton \
	    -anchor center -width -11 -padding "1 1" -relief raised -shiftrelief 1 \
	    -highlightthickness 1 -highlightcolor $colors(-frame) -foreground red

        ttk::style configure TMenubutton \
	    -anchor center -width -11 -padding "1 1" -relief raised -shiftrelief 1 \
	    -highlightthickness 1 -highlightcolor $colors(-frame)

        ttk::style map TMenubutton -relief {
	    {pressed !disabled} 	sunken
	    {active !disabled} 	raised
	} -highlightcolor {alternate black}

        

	ttk::style configure TMenubutton -relief raised -padding {10 2}
	ttk::style configure TRadiobutton -padding 1
	ttk::style configure TCheckbutton -padding 1

	## Toolbar buttons.
	#
        ttk::style configure RedToolbutton \
	    -width 0 -relief flat -borderwidth 2 -padding 3 \
	    -background $colors(-frame) -foreground red ;
        ttk::style map RedToolbutton -background [list active $colors(-selectbg)] 
	ttk::style map RedToolbutton -foreground [list active $colors(-selectfg)] 
	ttk::style map RedToolbutton -relief {
	    disabled 	flat
	    selected	sunken  
	    pressed 	sunken  
	    active  	raised
	}
        
	ttk::style configure Toolbutton \
	    -width 0 -relief flat -borderwidth 2 -padding 3 \
	    -background $colors(-frame) -foreground #000000 ;
	ttk::style map Toolbutton -background [list active $colors(-selectbg)] 
	ttk::style map Toolbutton -foreground [list active $colors(-selectfg)] 
	ttk::style map Toolbutton -relief {
	    disabled 	flat
	    selected	sunken  
	    pressed 	sunken  
	    active  	raised
	}

	## Entry widgets.
	#
	ttk::style configure TEntry \
	    -selectborderwidth 1 -padding 2 -insertwidth 2 -font TkTextFont
	ttk::style configure TCombobox \
	    -selectborderwidth 1 -padding 2 -insertwidth 2 -font TkTextFont

	## Notebooks.
	#
        ttk::style configure TNotebook -padding 0 -tabmargins {2 2 1 0}
        ttk::style configure TNotebook.Tab -padding {4 2}
        ttk::style map TNotebook.Tab -expand [list selected {2 2 1 0}]

	## Labelframes.
	#
	ttk::style configure TLabelframe -borderwidth 2 -relief groove

	## Scrollbars.
	#
        ttk::style configure TScrollbar \
	    -width 12 -arrowsize 12 
	
        
	ttk::style layout Vertical.TScrollbar {
	    Scrollbar.trough -children {
		Scrollbar.uparrow -side top
		Scrollbar.downarrow -side bottom
		Scrollbar.uparrow -side bottom
		Vertical.Scrollbar.thumb -side top -expand true -sticky ns
	    }
	}

	ttk::style layout Horizontal.TScrollbar {
	    Scrollbar.trough -children {
		Scrollbar.leftarrow -side left
		Scrollbar.rightarrow -side right
		Scrollbar.leftarrow -side right
		Horizontal.Scrollbar.thumb -side left -expand true -sticky we
	    }
	}
    }
}


# from http://wiki.tcl.tk/1109
#
# plus new addto
# adapted for use with text widget
#
namespace eval history {
    proc add? {w} {
        variable $w
        variable n$w
        upvar 0 $w hist
        #set s [set ::[$w cget -textvariable]]
        set s [$w get limit end-1c]
        if {$s == ""} return
        if [string compare $s [lindex $hist end]] {
            lappend hist $s
            set n$w [llength $hist]
        }
    }
    proc addto {w s} {
        variable $w
        variable n$w
        upvar 0 $w hist
        if {$s == ""} return
        if [string compare $s [lindex $hist end]] {
            lappend hist $s
            set n$w [llength $hist]
        }
    }
    proc move {w where} {
        variable $w
        variable n$w
        upvar 0 $w hist
        incr n$w $where
        if {[set n$w]<0} {set n$w 0}
        if {[set n$w]>=[llength $hist]+1} {
            set n$w [llength $hist]
        }
        #set ::[$w cget -textvar] [lindex $hist [set n$w]]
        $w delete limit end-1c
        $w mark set insert end
        $w mark set limit insert
        $w insert insert [lindex $hist [set n$w]]
        $w see end
    }
    proc for {type name args} {
        switch -- $type {
            text {
                uplevel $type $name $args
                bind $name <Up> {::history::move %W -1; break}
                bind $name <Down> {::history::move %W 1}
                bind $name <Next> {::history::move %W 99999}
               # bind $name <Return> {::history::add? %W}
                variable $name {}
                variable n$name 0
            }
            default {error "usage: ::history::for text <w> <args>"}
        }
    }
 }
# namespace eval history {
#     proc add? {w} {
#         variable $w
#         variable n$w
#         upvar 0 $w hist
#         set s [set ::[$w cget -textvariable]]
#         if {$s == ""} return
#         if [string compare $s [lindex $hist end]] {
#             lappend hist $s
#             set n$w [llength $hist]
#         }
#     }
#     proc move {w where} {
#         variable $w
#         variable n$w
#         upvar 0 $w hist
#         incr n$w $where
#         if {[set n$w]<0} {set n$w 0}
#         if {[set n$w]>=[llength $hist]+1} {
#             set n$w [llength $hist]
#         }
#         set ::[$w cget -textvar] [lindex $hist [set n$w]]
#     }
#     proc for {type name args} {
#         switch -- $type {
#             ttk::entry {
#                 uplevel $type $name $args
#                 bind $name <Up> {history::move %W -1}
#                 bind $name <Down> {history::move %W 1}
#                 bind $name <Next> {history::move %W 99999}
#                 bind $name <Return> {history::add? %W}
#                 variable $name {}
#                 variable n$name 0
#             }
#             default {error "usage: history::for entry <w> <args>"}
#         }
#     }
#  }




# bdd_tcl package
#

namespace eval ::bdd_tcl {
global c BDD 

# choose a bdd package
proc init_bdd {} {
  global BDD
  if [string equal $BDD "CUDD"] {
   if {![info exist ::starkit::topdir]} {
     source cudd.tcl
   } else {
     source $::starkit::topdir/cudd.tcl
   }
  } else {
   if [string equal $BDD "Buddy"] {
       if {![info exist ::starkit::topdir]} {
           source buddy.tcl 
       } else {
           source $::starkit::topdir/buddy.tcl
     }
   }
  }
}


#treelist routines
#

proc graphInit _g {upvar 1 $_g g; catch {unset g}; set g(nodes) {}}

proc addNode {_g node args} {
    upvar 1 $_g g
    set id [llength $g(nodes)]
    set g($id) [concat $node $args]
    lappend g(nodes) $id
    set id
 }

proc addEdge {_g from to args} {upvar 1 $_g g; set g($from,$to) $args}

proc isTerminal {_g id} {
    upvar 1 $_g g
    expr {[array names g $id,*]==""} ;# no edges leading out
 }

proc sons {_g id} {
    upvar 1 $_g g
    regsub -all $id, [array names g $id,*] "" res
    set res
 }

proc edges _g {upvar 1 $_g g; array names g *,*}

proc nodes _g {upvar 1 $_g g; set g(nodes)}

proc treelist2graph {L _g} {
    upvar 1 $_g g
    set from [addNode g [lindex $L 0]]
    foreach i [lrange $L 1 end] {
        set to [treelist2graph $i g]
        addEdge g $from $to
    }
    set from
 }

proc layout _g {
    upvar 1 $_g g
    set xt 0; set yt 0; set dx -60; set dy -30
    foreach id [lrevert [nodes g]] {
        if [isTerminal g $id] {
            set g(xy:$id) [list $xt $yt]
            incr xt $dx
        } else {
            set xs {}; set ys {}
            foreach i [sons g $id] {
                foreach {x y} $g(xy:$i) break
                lappend xs $x
                lappend ys $y
            }
            set x [expr {([max $xs]+[min $xs])/2}] ;# x:center between sons
            set y [expr {[min $ys]+$dy}]  ;# y: one layer above highest son
            set g(xy:$id) [list $x $y]
        }
    }
 }

proc draw {_g w} {
  global bddfont
    upvar 1 $_g g
    foreach i [edges g] {
        foreach {from to} [split $i ,] break
        eval $w create line $g(xy:$from) $g(xy:$to) -width 2 -tag edge
    }
    foreach i [nodes g] {
        set id [eval $w create text $g(xy:$i) -text [list $g($i)] -tag node]
        eval $w create rect [$w bbox $id] -fill white -outline white -tag node
        $w raise $id
    }
 }

proc max L {lindex [lsort -integer $L] end}

proc min L {lindex [lsort -integer $L] 0}

proc lrevert L {
    for {set res {}; set i [llength $L]} {$i>0} {#see loop} {
        lappend res [lindex $L [incr i -1]]
    }
    set res
 }


proc drawtree { tl tabtext} {
global c tab

    graphInit g
    treelist2graph [lindex $tl 0] g

    new_tab .top18 $tab $tabtext
    incr tab
    #puts "tab: $tab"
    layout g
    $c delete edge node
    draw g $c
    foreach {x y} [$c bbox all] break
    $c move all [expr 5-$x] [expr 5-$y]
    zoom $c 0.8
#    foreach {- - x1 y1} [$c bbox all] break
#    $c config -width [expr $x1+10] -height [expr $y1+10]

}



# reduce routines
#


#
# c(e) where e is a future fusion expression
#
# L: tree list (fusion expression) 
#
proc  c_fe {L} {
    global reduce_error

   set root [lindex $L 0]

   switch $root {
       fetest { 
           return "{fetest {stnot true}}"
       }
       festep { 
           return "{festep {[lindex $L 1]}}"
       }
       feinit {
           set x "{fechop {fetest ([lindex $L 1])} {fechopstar {festep true}}}"
           set y [c_fe $x]
           return $y
       }
       fefin {
           set x "{fechop {fechopstar {festep true}} {fetest ([lindex $L 1])}}"
           set y [c_fe $x]
           return $y
       }
       fechop { 
           set x [c_fe [lindex $L 1]]
           set y [c_fe [lindex $L 2]]
           return "{feor {fechop $x {[lindex $L 2]}} {fechop {[lindex $L 1]} $y}}"
       }
       feor { 
           set x [c_fe [lindex $L 1]]
           set y [c_fe [lindex $L 2]]
           return "{feor $x $y}"
       }
       feiand  {
           # (x iand z) = (test(x);z) 
           set x1 "{fechop {fetest {[lindex $L 1]}} {[lindex $L 2]}}"
           set x2 [c_fe $x1]
           return $x2
       }
       fefand  {
           # (x fand z) = (z;test(x))
           set x1 "{fechop {[lindex $L 2]} {fetest {[lindex $L 1]}}}"
           set x2 [c_fe $x1]
           return $x2
       }
       fechopstar { 
           set y $L
           set x [c_fe [lindex $L 1]]
           return "{fechop {$x} {$y}}"
       }
       default { 
           if {[llength $L] == 1 } {
               return "{$L}"
           } else {
               if {[llength $L] == 2 } {
                   set oper [lindex $L 0]
                   set x [c_fe [lindex $L 1]]
                   return "{$oper {$x}}"
               } else {
                   if {[llength $L] == 3 } {
                       set oper [lindex $L 0]
                       set x [c_fe [lindex $L 1]]
                       set y [c_fe [lindex $L 2]]
                       return "{$oper {$x} {$y}}"
                   } else {
                       set reduce_error 1
                       insert "error in c_fe" red
                   }
               }
           }
       }
   }
}

#
# c(e) where e is a past fusion expression
#
# L: tree list (fusion expression) 
#
proc  c_pfe {L} {
    global reduce_error

    set root [lindex $L 0]

    switch $root {
       fetest { 
           return "{fetest {stnot true}}"
       }
       fepstep { 
           return "{fepstep {[lindex $L 1]}}"
       }
       fepinit {
           set x "{fepchop {fepchopstar {fepstep true}} {fetest ([lindex $L 1])}}"
           set y [c_pfe $x]
           return $y
       }
       fepfin {
           set x "{fechop {fetest ([lindex $L 1])} {fepchopstar {fepstep true}}}"
           set y [c_pfe $x]
           return $y
       }
       fechop { 
           set x [c_pfe [lindex $L 1]]
           set y [c_pfe [lindex $L 2]]
           return "{feor {fechop $x {[lindex $L 2]}} {fechop {[lindex $L 1]} $y}}"
       }
       feor { 
           set x [c_pfe [lindex $L 1]]
           set y [c_pfe [lindex $L 2]]
           return "{feor $x $y}"
       }
       feandi {
           # (z andi x) = (z;test(x))
           set x1 "{fechop {[lindex $L 1]} {fetest {[lindex $L 2]}}}"
           set x2 [c_pfe $x1]
           return $x2
       }
       feandf {
           # (z andf x) = (test(x);z)
           set x1 "{fechop {fetest {[lindex $L 2]}} {[lindex $L 1]}}"
           set x2 [c_pfe $x1]
           return $x2
       }
       fepchopstar {
           # (pchopstar z) = test(true) or (pchopstar z);z 
           set y $L
           set x [c_pfe [lindex $L 1]]
           return "{fechop {$y} {$x}}"
               }
       default  { 
              if {[llength $L] == 1 } {
                  return "{$L}"
              } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [c_pfe [lindex $L 1]]
                    return "{$oper {$x}}"
                } else {
                   if {[llength $L] == 3 } {
                     set oper [lindex $L 0]
                     set x [c_pfe [lindex $L 1]]
                     set y [c_pfe [lindex $L 2]]
                     return "{$oper {$x} {$y}}"
                   } else {
                       set reduce_error 1
                       insert "error in c_pfe" red
                   }
                }
              }
       }
   }
}



#
# is_state L 
# determine whether L is a state formula
#
# L : tree list (fusion logic formula)
#
proc is_state { L } {
    set pfound [regexp {flpchop|flsince|flprev} $L]
    set ffound [regexp {flchop|fluntil|flnext} $L]

    if { $pfound == 0 && $ffound == 0 } {
        return 1
    } else {
        return 0
    }
}

proc is_state_old { L } {
    set root [lindex $L 0]
    #puts "is_state: root is $root"

    switch -regexp -- $root {
        stor -
        stand -
        stnot -
        stimp -
        stequiv -
        true -
        false  {
            return 1
        }
        {r_[0-9]+} {
            return 1
        }
        flor -
        fland -
        flequiv -
        flimp -
        Eqv -
        And -
        Or -
        Imp    {
            if { [is_state [lindex $L 1]] == 1  && [is_state [lindex $L 2]] == 1  } {
                return 1
            } else {
                return 0
            }
        }
        flnot -
        Not   {
            if { [is_state [lindex $L 1]] == 1} {
                return 1
            } else {
                return 0
            }
        }  
        {r_q[0-9]+}    {
            return 1
        }  
        {[A-Z]+[_A-Z]*} {
            return 1
        }
        default  {
            return 0
        }
    }
}

#
# is_past L 
# determine whether L is a past time formula
#
# L : tree list (fusion logic formula)
#

proc is_past { L } {

    set pfound [regexp {flpchop|flsince|flprev} $L]

    return $pfound
}

proc is_past_old { L } {
  set root [lindex $L 0]
  puts "is_past: root is $root"

    if { [is_state $L] == 1 } {
        return 0
    } else {
        switch -- $root {
            flpchop   {
                return 1
            }
            flsince {
                return 1
            }
            flor -
            fland -
            flequiv -
            flimp {
                if { [is_past [lindex $L 1]] == 1 || [is_past [lindex $L 2]] == 1  } {
                    return 1
                } else {
                    return 0
                }
            }
            flprev {
                return 1
            }
            flnot  {
                if { [is_past [lindex $L 1]] == 1 } {
                    return 1
                } else {
                    return 0
                }
            }  
            default  {
                #puts "default"
                return 0
            }
        }
    }
}

#
# is_future L 
# determine whether L is a future time formula
#
# L : tree list (fusion logic formula)
#

proc is_future { L } {

    set ffound [regexp {flchop|fluntil|flnext} $L]

    return $ffound
}


proc is_future_old { L } {
  set root [lindex $L 0]
  puts "is_state: root is $root"

    if { [is_state $L] == 1 } {
        return 0
    } else {
        switch -- $root {
            flchop   {
                return 1
            }
            fluntil {
                return 1
            }
            flor -
            fland -
            flequiv -
            flimp  {
                if { [is_future [lindex $L 1]] == 1 || [is_future [lindex $L 2]] == 1  } {
                    return 1
                } else {
                    return 0
                }
            }
            flnext {
                return 1
            }
            flnot    {
                if { [is_future [lindex $L 1]] == 1 } {
                    return 1
                } else {
                    return 0
                } 
            }  
            default  {
                #puts "default" 
                return 0
            }
        }
    }
}

##
## shift L k: L \uparrow k
## shift the subscripts of the dependent variables in L by k
##
## L: tree list (invariant)
## k: integer (>0)
##
#proc shift { L k } {

#  set root [lindex $L 0]
 
#  switch -regexp -- $root {
#      {r_[0-9]+} { 
#                   set j [string range $root 2 end]
#                   set jk [expr $k + $j]
#                   return " r_$jk "
#      }
#      
#      default { 
#                if {[llength $L] == 1 } {
#                  return " { $L } "
#                } else {
#                   if {[llength $L] == 2 } {
#                     set oper [lindex $L 0]
#                     set x [shift [lindex $L 1] $k]
#                     return " { $oper $x } "
#                   } else {
#                     if {[llength $L] == 3 } {
#                       set oper [lindex $L 0]
#                       set x [shift [lindex $L 1] $k]
#                       set y [shift [lindex $L 2] $k]
#                       return " { $oper $x $y } "
#                     } else {
#                       puts "error in shift"
#                     }
#                   }
#                }
#      }
#  }
#              
#}


##
## reduce_length L
## determine the number of distinct dependent variables in L
## 
## L: tree list (reduced fusion logic formula)
##
#proc reduce_length { L } {
#
#  set root [lindex $L 0]
#
#  switch $root {
#      Eqv { 
#            return 1
#      }
#      default { 
#                if {[llength $L] == 1 } {
#                  return 0
#                } else {
#                   if {[llength $L] == 2 } {
#                     set oper [lindex $L 0]
#                     set x [reduce_length [lindex $L 1]]
#                     return $x
#                   } else {
#                     if {[llength $L] == 3 } {
#                       set oper [lindex $L 0]
#                       set x [reduce_length [lindex $L 1]]
#                       set y [reduce_length [lindex $L 2]]
#                       set xy [expr $x + $y]
#                       return $xy
#                     } else {
#                       puts "error in reduce_length"
#                     }
#                   }
#                }
#      }
#  }
#
#}

##
## reduce L
## reduction function transforms a fusion logic formula into its corresponding
## invariant, i.e., \Red(L)
##
## L : tree list (fusion logic formula)
##
#proc reduce { L } {
#   global fresh
#   set root [lindex $L 0]
#   puts "reduce: L is $L"
#    if { [is_state $L] == 1 } {
#        return "{ Eqv r_1 {$L} }"
#    } else {
#        switch $root {
#            flchop { 
#                set E [lindex $L 1]
#                set F [lindex $L 2]
#                if { [is_state $F] == 1 } {
#                     puts "pass"
#                     set E_oper [lindex $E 0]
#                     set E_arg1 [lindex $E 1]
#                    if { $E_oper == "fetest" } {
#                        return " { Eqv r_1 { And {$F} {$E_arg1} } }"
#                    } else {
#                      if { $E_oper == "festep" } { 
#                          return " { Eqv r_1 { And {$E_arg1} {trnext {$F} } } }"#          
#                      } else {
#                        if { $E_oper == "feor" } { 
#                            set E_arg2 [lindex $E 2]
#                            set x "{ flor  { flchop {$E_arg1} {$F} } { flchop {$E_arg2} {$F} } }"
#                            set z [reduce [lindex $x 0]]
#                            return $z
#                        } else {
#                          if { $E_oper == "fechop" } { 
#                              set E_arg2 [lindex $E 2]
#                              set x "{ flchop {$E_arg1} { flchop {$E_arg2} {$F} } } "
#                            set z [reduce [lindex $x 0]]
#                            return $z
#                          } else {
#                            if { $E_oper == "fechopstar" } { 
#                              set y [c_fe $E_arg1 ]
#                              puts "y is $y"
#                              set x "{ flor {$F} { flchop $y r_q$fresh } }"
#                              puts "x is $x"
#                              set re "r_q$fresh"
#                              incr fresh
#                              set z [reduce [lindex $x 0]]
#                              set k [reduce_length [lindex $z 0] ]
#                              regsub -all $re $z {r_$k} z
#                              set z1 [subst $z]
#                              incr fresh -1
#                              return $z1
#                            } else { 
#                              puts "error in reduce F=W branch"
#                            }
#                          }
#                        }
#                      }
#                    }
#                } else {
#                  set x [reduce $F ]
#                  set k [reduce_length [lindex $x 0]]
#                  set y "{ flchop {$E} r_q$fresh }"
#                  set re "r_q$fresh"
#                  incr fresh
#                  set z [reduce [lindex $y 0]]
#                  set z1 [shift [lindex $z 0] $k]
#                  regsub -all $re $z1 {r_$k} z1
#                  set z2 [subst $z1] 
#                  incr fresh -1
#                  set z3 "{ And $x $z2 } "
#                  return $z3
#                }
#            }
#            flnot { 
#                     set x [reduce [lindex $L 1] ]
#                     set k [reduce_length [lindex $x 0]]
#                     set k1 [expr $k + 1]
#                     set z "{ And $x { Eqv r_$k1 {Not r_$k} } } "
#                     return $z
#            }
#            flor { 
#                    set x1 [reduce [lindex $L 1] ]
#                    set j [reduce_length [lindex $x1 0] ]
#                    set x2 [reduce [lindex $L 2] ]
#                    set y2 [shift [lindex $x2 0] $j]
#                    set k [reduce_length [lindex $x2 0]]
#                    set jk [expr $j + $k]
#                    set jk1 [expr $jk + 1]
#                    set z "{ And $x1 { And $y2 { Eqv r_$jk1 {Or r_$j r_$jk} } } }"
#                    return $z
#            }
#            default { puts "error in reduce switch" }
#        }
#    }
#}


#
# reverse_fl L
# time reversal of past/future fusion logic formula L\itlRev
# 
# input L: tree list (fusion logic formula)
# output : reversed fusion logic formula L\itlRev
proc reverse_fl { L } {
    global dbg
    
    set root [lindex $L 0]
    # puts "reverse_fl: L is $L"

    if { [is_state $L] == 1 } {
        # this is crucial in the new semantics
        return "{$L}"
    } else {
    
     switch $root {
         flchop {
             # rev(<E1> E2) = rev(E1)<rev(E2)>
             set E2 [lindex $L 2]
             # puts $E2
             set E1 [lindex $L 1]
             # puts $E1
             set z "{flpchop [reverse_fl $E1] [reverse_fe $E2]}"
             # puts "reverse_fl: plpchop z is $z"
             return $z
         }
         flpchop {
             # rev(E1 <E2>) = <rev(E2)>rev(E1)  
             set E2 [lindex $L 2]
             # puts $E2
             set E1 [lindex $L 1]
             # puts $E1
             set z "{flchop [reverse_pfe $E2] [reverse_fl $E1]}"
             # puts "reverse_fl: plpchop z is $z"
             return $z
         }
         flnot {
             set x [reverse_fl [lindex $L 1]]
             set z "{flnot $x}"
             # puts "reverse_fl: plnot z is $z"
             return $z
         }
         flnext {
             set x [reverse_fl [lindex $L 1]]
             set z "{flprev $x}"
             # puts "reverse_fl: plprev z is $z"
             return $z
         }
         flprev {
             set x [reverse_fl [lindex $L 1]]
             set z "{flnext $x}"
             # puts "reverse_fl: plprev z is $z"
             return $z
         }
         flor {
             set x1 [reverse_fl [lindex $L 1] ]
             set x2 [reverse_fl [lindex $L 2] ]
             set z "{flor $x1 $x2}"
             # puts "reverse_fl: flor z is $z"
             return $z
         }
         fland {
             set x1 [reverse_fl [lindex $L 1] ]
             set x2 [reverse_fl [lindex $L 2] ]
             set z  "{fland $x1 $x2}"
             # puts "reverse_fl: pland z is $z"
             return $z
         }
         flequiv {
             set x1 [reverse_fl [lindex $L 1] ]
             set x2 [reverse_fl [lindex $L 2] ]
             set z  "{flequiv $x1 $x2}"
             # puts "reverse_fl: flequiv z is $z"
             return $z
         }
         flimp {
             set x1 [reverse_fl [lindex $L 1] ]
             set x2 [reverse_fl [lindex $L 2] ]
             set z  "{flimp $x1 $x2}"
             # puts "reverse_fl: flimp z is $z"
             return $z
         }
         fluntil {
             set x1 [reverse_fl [lindex $L 1] ]
             set x2 [reverse_fl [lindex $L 2] ]
             set z  "{flsince $x1 $x2}"
             # puts "reverse_fl: flsince z is $z"
             return $z
         }
         flsince {
             set x1 [reverse_fl [lindex $L 1] ]
             set x2 [reverse_fl [lindex $L 2] ]
             set z  "{fluntil $x1 $x2}"
             # puts "reverse_fl: flsince z is $z"
             return $z
         }
         default {
             insert_debug "error in reverse_pl switch" red
         }
     }
 }
}

#
# reverse_fe E
# time reversal of future fusion expression E\itlRev
# 
# input L: tree list (future fusion exxpression)
# output : past fusion expression E\itlRev
proc reverse_fe { E } {

   set root [lindex $E 0]
   # puts "reverse_fe: E is $E"

     switch $root {
         fetest { 
             set Y [lindex $E 1]
             set z "{fetest {$Y}}"
             # puts "reverse_fe: fetest z is $z"
             return $z
         }
         festep {
             set Y [lindex $E 1]
             set z "{fepstep [reverse_ftr $Y]}"
             # puts "reverse_fe: festep z is $z"
             return $z
         }
         feinit {
             set Y [lindex $E 1]
             set z "{fepinit $Y}"
             # puts "reverse_fe: feinit z is $z"
             return $z
         }
         fefin {
             set Y [lindex $E 1]
             set z "{fepfin $Y}"
             # puts "reverse_fe: fefin z is $z"
             return $z
         }
         fechop { 
             set E1 [lindex $E 1]
             set E2 [lindex $E 2]
             set z "{fechop [reverse_fe $E2] [reverse_fe $E1]}"
             # puts "reverse_fe: fechop z is $z"
             return $z
         }
         fechopstar { 
             set Y [lindex $E 1]
             set z "{fepchopstar [reverse_fe $Y]}"
             # puts "reverse_fe: fechopstar z is $z"
             return $z
         }
         feor {
             set E1 [reverse_fe [lindex $E 1] ]
             set E2 [reverse_fe [lindex $E 2] ]
             set z "{feor $E1 $E2}"
             # puts "reverse_fe: feor z is $z"
             return $z
         }
         feiand {
             # rev(E1 iand y) = rev(y) andi E1
             set E2 [reverse_fe [lindex $E 2] ]
             set z "{feandi $E2 $E1}"
             # puts "reverse_fe: feiand z is $z"
             return $z
         }
         fefand {
             # rev(E1 fand y) = rev(y) andf E1
             set E2 [reverse_fe [lindex $E 2] ]
             set z "{feandf $E2 $E1}"
             # puts "reverse_fe: fefand z is $z"
             return $z
         }
         default {
             puts "error in reverse_fe switch"
         }
     }
}

#
# reverse_pfe E
# time reversal of past fusion expression E\itlRev
# 
# input L: tree list (past fusion exxpression)
# output : future fusion expression E\itlRev
proc reverse_pfe { E } {

   set root [lindex $E 0]
   # puts "reverse_fe: E is $E"

     switch $root {
         fetest { 
             set Y [lindex $E 1]
             set z "{fetest {$Y}}"
             # puts "reverse_pfe: fetest z is $z"
             return $z
         }
         fepstep {
             set Y [lindex $E 1]
             set z "{festep [reverse_ptr $Y]}"
             # puts "reverse_pfe: fepstep z is $z"
             return $z
         }
         fepinit {
             set Y [lindex $E 1]
             set z "{feinit $Y}"
             # puts "reverse_pfe: fepinit z is $z"
             return $z
         }
         fepfin {
             set Y [lindex $E 1]
             set z "{fefin $Y}"
             # puts "reverse_pfe: fepfin z is $z"
             return $z
         }
         fechop { 
             set E1 [lindex $E 1]
             set E2 [lindex $E 2]
             set z "{fechop [reverse_pfe $E2] [reverse_pfe $E1]}"
                  # puts "reverse_pfe: fechop z is $z"
                  return $z
         }
         fepchopstar { 
             set Y [lindex $E 1]
             set z "{fechopstar [reverse_pfe $Y]}"
             # puts "reverse_pfe: fechopstar z is $z"
             return $z
         }
         feor {
             set E1 [reverse_pfe [lindex $E 1] ]
             set E2 [reverse_pfe [lindex $E 2] ]
             set z "{feor $E1 $E2}"
             # puts "reverse_pfe: feor z is $z"
             return $z
         }
         feandi {
             # rev(y andi x) = x iand rev(y)
             set x [lindex $E 2]
             set E1 [reverse_pfe [lindex $E 1] ]
             set z "{feiand $x $E2}"
             # puts "reverse_pfe: feandi z is $z"
             return $z
         }
         feandf {
             # rev(y andf x) = x fand rev(y)
             set x [lindex $E 2]
             set E2 [reverse_pfe [lindex $E 2] ]
             set z "{fefand $x $E2}"
             # puts "reverse_pfe: fefand z is $z"
             return $z
         }
         default {
             puts "error in reverse_pfe switch"
         }
     }
}


#
# reverse_ftr T
# time reversal of future transition expression T\itlRev
# 
# input L: tree list (future transition exxpression)
# output : past transition expression T\itlRev
proc reverse_ftr { T } {

   set root [lindex $T 0]
   # puts "reverse_ftr: T is $T"
 
  switch -regexp -- $root {
      true -
      false {
          set z "{trprev $root}"
          # puts "reverse_ftr: true-false z is $z"
          return $z
      }
      trand -
      tror -
      trequiv -
      trimp {
          set oper [lindex $T 0]
          set x1 [reverse_ftr [lindex $T 1] ]
          set y1 [reverse_ftr [lindex $T 2] ]
          set z "{$oper $x1 $y1}"
          # puts "reverse_tr: and-or z is $z"
          return $z
      }
      trnot {
          set oper [lindex $T 0]
          set x1 [reverse_ftr [lindex $T 1] ]
          set z "{$oper $x1}"
          # puts "reverse_tr: not z is $z"
          return $z
      }
      trnext {
          set x1 [lindex $T 1]
          set z "{$x1}"
          # puts "reverse_ftr: next z is $z"
          return $z
      } 
      {[A-Z]+[_A-Z]*}   { 
          set z "{trprev $root}"
          # puts "reverse_ftr: var z is $z"
          return $z
      }
      default { 
            puts "error in reverse_ftr switch"    
      }
   }
}

#
# reverse_ptr T
# time reversal of past transition expression T\itlRev
# 
# input L: tree list (past transition exxpression)
# output : future transition expression T\itlRev
proc reverse_ptr { T } {

   set root [lindex $T 0]
   # puts "reverse_tr: T is $T"
 
  switch -regexp -- $root {
      true -
      false {
          set z "{trnext $root}" 
          # puts "reverse_ptr: true-false z is $z"
          return $z
      }
      trand -
      tror -
      trequiv -
      trimp {
          set oper [lindex $T 0]
          set x1 [reverse_ptr [lindex $T 1] ]
          set y1 [reverse_ptr [lindex $T 2] ]
          set z "{$oper $x1 $y1}"
          # puts "reverse_ptr: and-or z is $z"
          return $z
      }
      trnot {
          set oper [lindex $T 0]
          set x1 [reverse_ptr [lindex $T 1] ]
          set z "{$oper $x1}"
          # puts "reverse_ptr: not z is $z"
          return $z
      }
      trprev {
          set x1 [lindex $T 1]
          set z "{$x1}"
          # puts "reverse_ptr: prev z is $z"
          return $z
      } 
      {[A-Z]+[_A-Z]*} { 
          set z "{trnext $root}"
          # puts "reverse_ptr: var z is $z"
          return $z
      }
      default { 
            puts "error in reverse_ptr switch"    
      }
   }
}


#
#
proc state_sanitise { xx } {

    set gammaa $xx
    
    set gammaa [regsub -all {(fl|st|tr)or} $gammaa {Or}]
    set gammaa [regsub -all {(fl|st|tr)and} $gammaa {And}]
    set gammaa [regsub -all {(fl|st|tr)equiv} $gammaa {Eqv}]
    set gammaa [regsub -all {(fl|st|tr)imp} $gammaa {Imp}]
    set gammaa [regsub -all {(fl|st|tr)not} $gammaa {Not}]    

    return $gammaa
}


# state_to_flstate
# convert a state formula into a reduce_state form
# L: tree list (state formula)
proc state_to_flstate { L } {

    global dbg

    if { $dbg == 1 } {
        insert_debug "state_to_flstate: L=$L"
    }
    set tt $L
    
    set tt [regsub -all {stor} $tt {flor}]
    set tt [regsub -all {stand} $tt {fland}]
    set tt [regsub -all {stequiv} $tt {flequiv}]
    set tt [regsub -all {stimp} $tt {flimp}]
    set tt [regsub -all {stnot} $tt {flnot}]

    if { $dbg == 1 } {
        insert_debug "state_to_flstate: tt=$tt"
    }
    return $tt
    
}

# tr_to_ptr
# convert a future transition formula into past transition formula
# L: tree list (next transition formula)
proc tr_to_ptr { L } {
    global dbg

    set root [lindex $L 0]


    if { $dbg == 1 } {
        insert_debug "L= $L"
        insert_debug "root = $root"
    }
    
    switch -regexp -- $root {
        true -
        false {
            set z "{$root}"  
            # puts "reverse_tr: true-false z is $z"
            return $z
        }
        stand -
        trand -
        stor -
        tror -
        stequiv -
        trequiv -
        stimp -
        trimp -
        And -
        Or -
        Eqv -
        Imp {
            set oper [lindex $L 0]
            set x1 [tr_to_ptr [lindex $L 1] ]
            set y1 [tr_to_ptr [lindex $L 2] ]
            set z "{$oper $x1 $y1}"
            # puts "tr_to_ptr: and-or z is $z"
            return $z
        }
        stnot -
        trnot -
        Not {
            set oper [lindex $L 0]
            set x1 [tr_to_ptr [lindex $L 1] ]
            set z "{$oper $x1}"
            # puts "tr_to_ptr: not z is $z"
            return $z
        }
        trnext {
            set x1 [lindex $L 1]
            set z "$x1"
            # puts "tr_to_ptr: next z is $z"
            return $z
        }
        {[A-Z]+[_A-Z]*}   { 
            set z "{trprev $root}"
            # puts "ptr_to_tr: var z is $z"
            return $z
        }
        {r_[0-9]+} {
            set z "{trprev $root}"
            return $z
        }
        default { 
            insert "error in tr_to_ptr switch"    red
        }
    }
}

# ptr_to_tr
# convert a past transition formula into future transition formula
# L: tree list (previous transition formula)
proc ptr_to_tr { L } {

    global dbg

    set root [lindex $L 0]

    if { $dbg == 1 } {
        insert_debug "L= $L"
        insert_debug "root = $root"
    }

    switch -regexp -- $root {
        true -
        false {
            set z "{trnext $root}" 
            # puts "reverse_tr: true-false z is $z"
            return $z
        }
        stand -
        trand -
        stor -
        tror -
        stequiv -
        trequiv -
        stimp -
        trimp -
        And -
        Or -
        Eqv -
        Imp {
            set oper [lindex $L 0]
            set x1 [ptr_to_tr [lindex $L 1] ]
            set y1 [ptr_to_tr [lindex $L 2] ]
            set z "{$oper $x1 $y1}"
            # puts "ptr_to_tr: and-or z is $z"
            return $z
        }
        stnot -
        trnot -
        Not {
            set oper [lindex $L 0]
            set x1 [ptr_to_tr [lindex $L 1] ]
            set z "{$oper $x1}"
            # puts "ptr_to_tr: not z is $z"
            return $z
        }
        trprev {
            set x1 [lindex $L 1]
            set z "$x1"
            # puts "ptr_to_tr: next z is $z"
            return $z
        }
        {[A-Z]+[_A-Z]*}   { 
            set z "{trnext $root}"
            # puts "ptr_to_tr: var z is $z"
            return $z
        }
        {r_[0-9]+} {
            set z "{trnext $root}"
            return $z
        }
        default { 
            insert "error in ptr_to_tr switch"  red  
        }
    }
}

#
# reduce_state k L
# reduction function v2 \Red'_k(L)
# 
# k=2, return state formula
# k=1, return NL^1 formula
# k=0, return any formula
# L: tree list (fusion logic formula)
#
proc reduce_state { k L } {

    global depindex
    global infinite
    global dbg
    global reduce_error

    set root [lindex $L 0]

    if { $dbg == 1 } {
        insert_debug "reduce_state: L is ($L)"
        insert_debug "reduce_state: k is $k"
    }
   
   if { [is_state $L] == 1 } { 
       #puts "reduce_state: is state"
       set tt [state_sanitise $L]
     return "{$tt}"
   } else {
     switch $root {
         flchop {
             set E [lindex $L 1]
             set Y [lindex $L 2]
             set E_oper [lindex $E 0]
             set E_arg1 [lindex $E 1]
             switch $E_oper {
                 fetest {
                     set y1 [reduce_state $k $Y]
                     set z "{And {$E_arg1} $y1}"
                     #puts "reduce_state: fetest z is $z"
                     return $z
                 } 
                 festep {
                     if { $k == 2 } {
                         #set tt [string trim $L]  
                         #set silly [array names depindex -exact $tt]
                         #puts "reduce_state: silly $silly"
                         return "r_$depindex($L)"
                     } else {
                         if { $k == 1 } {
                             set y1 [reduce_state 2 $Y]
                             set z "{And {$E_arg1} {flnext $y1}}"
                             #puts "reduce_state: festep z is $z"
                             return $z
                         }
                     }
                 }
                 feiand {
                     # <x1 iand x2>x3 = <test(x1);x2>x3= x1 and <x2>x3
                     set E_arg2 [lindex $E 2]
                     set y1 "{flchop {$E_arg2} {$Y}}"
                     set y2 [reduce_state $k [lindex $y1 0]]
                     set y3 [lindex $E_arg1 0]   
                     set z "{And {$y3} $y2}"
                     #puts "reduce_state: feand z is $z"
                     return $z
                 }
                 fefand {
                     # <x1 fand x2>x3 = <x2;test(x1)>x3 = <x2>(x1 and x3) 
                     set E_arg2 [lindex $E 2]
                     set y3  $E_arg1
                     set y4 [state_to_flstate $y3]
                     set y1 "{flchop {$E_arg2} {fland {$y4} {$Y}}}"
                     set z [reduce_state $k [lindex $y1 0]]
                     return $z
                 }
                 feor {
                     set E_arg2 [lindex $E 2]
                     set x "{flor {flchop {$E_arg1} {$Y}} {flchop {$E_arg2} {$Y}}}"
                     set z [reduce_state $k [lindex $x 0]]
                     #puts "reduce_state: feor z is $z"
                     return $z
                 }
                 fechop {
                     set E_arg2 [lindex $E 2]
                     set x "{flchop {$E_arg1} {flchop {$E_arg2} {$Y}}}"
                     set z [reduce_state $k [lindex $x 0]]
                     #puts "reduce_state: fechop z is $z"
                     return $z
                 }
                 fechopstar {
                     if { $infinite == 1 } {
                         set z "{false}"
                         set reduce_error 1
                     } else {
                         set z "r_$depindex($L)"
                     }
                     #puts "reduce_state: fechopstar z is $z"
                     return $z
                 }
                 feinit {
                     set x "{flchop {fechop {fetest {$E_arg1}} {fechopstar {festep {true}}}} {$Y}}"
                     set z [reduce_state $k [lindex $x 0]]
                     #puts "reduce_state: feinit z is $z"
                     return $z
                 }
                 fefin {
                     set x "{flchop {fechop {fechopstar {festep {true}}} {fetest {$E_arg1}}} {$Y}}"
                     set z [reduce_state $k [lindex $x 0]]
                     #puts "reduce_state: fefin z is $z"
                 }
                 default {
                     set reduce_error 1
                     insert "error in reduce_state, future fusion expression branch" red
                 }
             }
         }
         flpchop {
             set E [lindex $L 2]
             set Y [lindex $L 1]
             set E_oper [lindex $E 0]
             set E_arg1 [lindex $E 1]
             switch $E_oper {
                 fetest {
                     set y1 [reduce_state $k $Y]
                     set z "{And {$E_arg1} $y1}"
                     #puts "reduce_state: fetest z is $z"
                     return $z
                 } 
                 fepstep {
                     # x1 <pstep x2> 
                     if { $k == 2 } {
                         #set tt [string trim $L]  
                         #set silly [array names depindex -exact $tt]
                         #puts "reduce_state: silly $silly"
                         return "r_$depindex($L)"
                     } else {
                         if { $k == 1 } {
                             set y1 [reduce_state 2 $Y]
                             set t1 "{$E_arg1}"
                             #puts "t1=$t1" 
                             #set y2 [tr_to_ptr $t1]
                             set y2 $t1
                             set z "{And $y2 {trprev $y1}}"
                             #puts "reduce_state: festep z is $z"
                             return $z
                         }
                     }
                 }
                 feandi {
                     # x1 < x2 andi x3> = x1 <x2;test(x1)> = x1<x2> and (x1)
                     # sfin x1 = [not [box [imp [empty] [not x1]]] =
                     # {fluntil true {flnot {flimp {flnot {flnext true}} {flnot x1}}}}  
                     set E_arg2 [lindex $E 2]
                     set y3 [state_to_flstate $E_arg2]
                     #set y4 "{fluntil true {flnot {flimp {flnot {flnext true}} {flnot {$y3}}}}}"
                     #puts "y4= $y4" 
                     set y1 "{fland {flpchop {$Y} {$E_arg1}} {$y3}}"
                     set z [reduce_state $k [lindex $y1 0]]
                     return $z
                     #set y1 "{flchop {$Y} {$E_arg1}}"
                     #set y2 [reduce_state $k [lindex $y1 0]]
                     #set y3 [lindex $E_arg2 0]   
                     #set z "{And {$y2} $y3}"
                     #puts "reduce_state: feiand z is $z"
                     #return $z
                 }
                 feandf {
                     # x1 < x2 andf x3> = x1<test(x3);x2>= (x1 and x3)<x2> 
                     set E_arg2 [lindex $E 2]
                     set y3 [state_to_flstate $E_arg2]
                     #set y4 "{fluntil true {flnot {flimp {flnot {flnext true}} {flnot {$y3}}}}}"
                     set y1 "{flpchop {fland {$Y} {$y3}} {$E_arg1}}"
                     set z [reduce_state $k [lindex $y1 0]]
                     return $z
                     #set y4 [state_to_flstate $y3]
                     #set y1 "{flpchop {fland {$y4} {$Y}} {$E_arg1}}"
                     #set z [reduce_state $k [lindex $y1 0]]
                     #return $z
                 }
                 feor {
                     # Y < E_arg1 or E_arg2>
                     set E_arg2 [lindex $E 2]
                     set x "{flor {flpchop {$Y} {$E_arg1}} {flpchop {$Y} {$E_arg2}}}"
                     set z [reduce_state $k [lindex $x 0]]
                     #puts "reduce_state: feor z is $z"
                     return $z
                 }
                 fechop {
                     # Y < E_arg1 ; E_arg2>  (Y<E_arg1>)<E_arg2>
                     set E_arg2 [lindex $E 2]
                     set x "{flpchop {flpchop {$Y} {$E_arg1}} {$E_arg2}}"
                     set z [reduce_state $k [lindex $x 0]]
                     #puts "reduce_state: fechop z is $z"
                     return $z
                 }
                 fepchopstar {
                     #if { $infinite == 1 } {
                     #    set z "{false}"
                     #} else {
                     #    set z "r_$depindex($L)"
                     #}
                     set z "r_$depindex($L)"
                     #puts "reduce_state: fepchopstar z is $z"
                     return $z
                 }
                 fepinit {
                     set x "{flpchop {$Y} {fechop {fepchopstar {fepstep {true}}} {fetest {$E_arg1}}}}"
                     set z [reduce_state $k [lindex $x 0]]
                     #puts "reduce_state: feinit z is $z"
                     return $z
                 }
                 fepfin {
                     set x "{flpchop {$Y} {fetest {$E_arg1}} {fechop {fechopstar {festep {true}}}}}"
                     set z [reduce_state $k [lindex $x 0]]
                     #puts "reduce_state: fefin z is $z"
                 }
                 default {
                     set reduce_error 1
                     insert "error in reduce_state, past fusion expression branch" red
                 }
             }
         }
         fluntil {
             #set Z [lindex $L 2]
             set Y [lindex $L 1]
             #set y1 [reduce_state 1 $Y] 
             
             return "{r_$depindex($L)}"
         }
         flsince {
             #set Z [lindex $L 2]
             set Y [lindex $L 1]
             #set y1 [reduce_state 1 $Y] 
             
             return "{r_$depindex($L)}"
         }
         flnext {
             set Y [lindex $L 1]
             if { $k == 2 } {
                 return "r_$depindex($L)"
             } else {
                 if { $k == 1 } {
                     set y1 [reduce_state 2 $Y]
                     set z "{flnext {$y1}}"
                     #puts "reduce_state: flnext z is $z"
                     return $z
                 }
             }
         }
         flprev {
             set Y [lindex $L 1]
             if { $k == 2 } {
                 return "r_$depindex($L)"
             } else {
                 if { $k == 1 } {
                     set y1 [reduce_state 2 $Y]
                     set z "{flprev {$y1}}"
                     #puts "reduce_state: flprev z is $z"
                     return $z
                 }
             }
         }
         flnot {
             set x [reduce_state [expr max($k,1)] [lindex $L 1]]
             set z "{Not $x}"
             #puts "reduce_state: flnot z is $z"
             return $z
         }
         flor {
             set k1 [expr max($k,1)] 
             set x1 [reduce_state $k1 [lindex $L 1] ]
             set x2 [reduce_state $k1 [lindex $L 2] ]
             set z "{Or $x1 $x2}"
             #puts "reduce_state: flor z is $z"
             return $z
         }
         fland {
             set k1 [expr max($k,1)]
             set x1 [reduce_state $k1 [lindex $L 1] ]
             set x2 [reduce_state $k1 [lindex $L 2] ]
             set z "{And $x1 $x2}"
             #puts "reduce_state: fland z is $z"
             return $z
         }
         flequiv {
             set k1 [expr max($k,1)]
             set x1 [reduce_state $k1 [lindex $L 1] ]
             set x2 [reduce_state $k1 [lindex $L 2] ]
             set z "{Eqv $x1 $x2}"
             #puts "reduce_state: flor z is $z"
             return $z
         }
         flimp {
             set k1 [expr max($k,1)]
             set x1 [reduce_state $k1 [lindex $L 1] ]
             set x2 [reduce_state $k1 [lindex $L 2] ]
             set z "{Imp $x1 $x2}"
             #puts "reduce_state: flor z is $z"
             return $z
         }
         default {
             set reduce_error 1
             insert "error in reduce_state switch" red
         }
     }
   }
}

#
# reduce_transition k L
# reduction function v2 \Red_k(L)
#
# k=0, any
# k=1, NL^1
# k=2, state 
# L: tree list (fusion logic formula)
#
proc reduce_transition { k L } {
    global depindex
    global freshx
    global deplist
    global deplistp
    global pdeplist 
    global LivenessTests
    global LivenessOffsets
    global dbg
    global infinite
    global reduce_error

    #if { $L != [string trim $L] } {
         #puts "trimming whitespaces from front and end" 
         # set L  [string trim $L]
    #}
    set root [lindex $L 0]

    if { $dbg == 1} {
        insert_debug "reduce_transition: L is ($L)"
        insert_debug "reduce_transition: k is $k"
    }
    if { [is_state $L] == 1 } { 
        #puts "reduce_transition: is state"
        return "true"
    } else {
        switch $root {
            flchop {
                set E [lindex $L 1]
                set Y [lindex $L 2]
                set E_oper [lindex $E 0]
                set E_arg1 [lindex $E 1]
                #puts "$E_oper"
                switch $E_oper {
                    fetest {
                        set z [reduce_transition $k $Y]
                        #puts "reduce_transition: fetest z is $z"
                        return $z
                    }
                    festep {
                        if { $k == 2 } {
                            #set tmp [array names depindex -exact $L]
                            #puts "tmp is $tmp"
                            #set tmp [string trim $L]
                            if {[array names depindex -exact $L] == "" } {
                                if { $dbg == 1 } {
                                    insert_debug "assigning ($L) to $freshx"
                                }
                                set depindex($L) $freshx
                                #puts "$depindex($L)"    
                                incr freshx
                            } 
                            #else {
                            #  puts "reuse depindex ( )"
                            #}
                            set x2 [reduce_transition 2 $Y]
                            #puts "x2 is $x2"
                            set x1 [reduce_state 2 $Y]
                            #puts "x1 is $x1"
                            set x3 "{Eqv r_$depindex($L) {And {$E_arg1} {flnext $x1}}}"
                            if { [array names deplist -exact $x3] == "" } {
                                if { $dbg == 1 } {
                                    insert_debug "in Red_T step: adding Dep r_$depindex($L) Eqv ($E_arg1 and next $x1)"
                                }
                                set deplist($x3) "r_$depindex($L)" 
                                set z "{And $x2 $x3}"
                            } else {
                                set z $x2
                            }
                            #puts "reduce_transition: festep z is $z"
                            return $z
                        } else {
                            if { $k == 1 } { 
                                set z [reduce_transition 2 $Y]
                                #puts "reduce_transition: festep z is $z"
                                return $z
                            } else {
                                insert " error in festep of reduce_transition " red
                            }
                        }
                    } 
                    feiand {
                        set E_arg2 [lindex $E 2]
                        set y1 "{flchop {$E_arg2} {$Y}}" 
                        set z [reduce_transition $k [lindex $y1 0]]   
                        return $z
                    }
                    fefand {
                        set E_arg2 [lindex $E 2]
                        set y3  $E_arg1
                        set y4 [state_to_flstate $y3]
                        set y1 "{flchop {$E_arg2} {fland {$y4} {$Y}}}"
                        set z [reduce_transition $k [lindex $y1 0]]
                        return $z
                    }
                    feor {
                        set E_arg2 [lindex $E 2]
                        set x "{flor {flchop {$E_arg1} {$Y}} {flchop {$E_arg2} {$Y}}}"
                        #puts "reduce_transition: feor z is $z"
                        set z [reduce_transition $k [lindex $x 0]]
                        return $z
                    }
                    fechop {
                        set E_arg2 [lindex $E 2]
                        #puts " $E_arg2"     
                        set x "{flchop {$E_arg1} {flchop {$E_arg2} {$Y}}}"
                        set z [reduce_transition $k [lindex $x 0]]
                        #puts "reduce_transition: fechop z is $z"
                        return $z
                    }
                    fechopstar {
                        if { $infinite == 1 } {
                            insert "error: can not use chopstar with infinite time" red
                            set reduce_error 1
                            return "{false}"
                        }
                        #set tmp [array names depindex -exact $L]
                        #puts "tmp is $tmp"
                        if {[array names depindex -exact $L] == "" } {
                            if { $dbg == 1 } {
                                insert_debug "assigning ($L) to $freshx"
                            }
                            set depindex($L) $freshx
                            incr freshx
                        } 
                        #else {
                        # puts "reuse depindex ( )"
                        #}
                        set y1 [c_fe $E_arg1 ]
                        set x1 "{flor {$Y} {flchop $y1 r_$depindex($L)}}"
                        set y3 [reduce_transition 1 [lindex $x1 0]]
                        set y2 [reduce_state 1 [lindex $x1 0]]
                        set z1 "{Eqv r_$depindex($L) $y2}"
                        if { [array names deplist -exact $z1] == "" } {
                            if { $dbg == 1 } {
                                insert_debug "in Red_T chopstar: adding Dep r_$depindex($L) Eqv $y2"
                            }
                            set deplist($z1) "r_$depindex($L)"
                            set z "{And $z1 $y3}"
                        } else {
                            set z $y3
                        }
                        #puts "reduce_transition: fechopstar z is $z"
                        return $z
                    }
                    feinit {
                        set x "{flchop {fechop {fetest {$E_arg1}} {fechopstar {festep {true}}}} {$Y}}"
                        set z [reduce_transition $k [lindex $x 0]]
                        return $z
                    }
                    fefin {
                        set x "{flchop {fechop {fechopstar {festep {true}}} {fetest {$E_arg1}}} {$Y}}"
                        set z [reduce_transition $k [lindex $x 0]]
                        return $z
                    }
                    default {
                        set reduce_error 1
                        insert "error in reduce_transition, future fusion expression branch" red
                    }
                }
            }
            flpchop {
                set E [lindex $L 2]
                set Y [lindex $L 1]
                set E_oper [lindex $E 0]
                set E_arg1 [lindex $E 1]
                #puts "$E_oper"
                #
                switch $E_oper {
                    fetest {
                        # Y <test E_arg1>  
                        set z [reduce_transition $k $Y]
                        #puts "reduce_transition: fetest z is $z"
                        return $z
                    }
                    fepstep {
                        # Y<pstep E_arg1>
                        if { $k == 2 } {
                            #set tmp [array names depindex -exact $L]
                            #puts "tmp is $tmp"
                            #set tmp [string trim $L]
                            if {[array names depindex -exact $L] == "" } {
                                if { $dbg == 1 } {
                                    insert_debug "assigning ($L) to $freshx"
                                }
                                set depindex($L) $freshx
                                #puts "$depindex($L)"    
                                incr freshx
                            } 
                            #else {
                            #  puts "reuse depindex ( )"
                            #}
                            set x2 [reduce_transition 2 $Y]
                            #puts "x2 is $x2"
                            set x1 [reduce_state 2 $Y]
                            #puts "x1 is $x1"
                            # r_$depindex($L) Eqv $E_arg1 and (prev $x1)
                            # not sure whether $E_arg1 needs changing or not?
                            # it must change !
                            set x5 [ptr_to_tr $E_arg1]
                            #puts "x5= $x5" 
                            set x4 "{Eqv {flnext r_$depindex($L)} {And $x5 $x1}}"
                            if { [array names deplist -exact $x4] == "" } {
                                if { $dbg == 1 } {
                                    insert_debug "in Red_T step: adding Dep (next r_$depindex($L)) Eqv ($x5 and $x1)"
                                }
                                set deplistp($x4) "r_$depindex($L)" 
                                set z "{And $x2 $x4}"
                            } else {
                                set z $x2
                            }
                            set x3 "{Eqv r_$depindex($L) {And {$E_arg1} {flprev $x1}}}"
                            if { [array names pdeplist -exact $x3] == "" } {
                                if { $dbg == 1 } {
                                  insert_debug "in Red_T prev: adding PDep r_$depindex($L) Eqv  ($E_arg1) and prev $x1"
                                }
                             set pdeplist($x3) "r_$depindex($L)" 
                             set z "{And $x2 $x3}"
                            } else {
                                set z $x2
                            }
                            #puts "reduce_transition: festep z is $z"
                            return $z
                        } else {
                            if { $k == 1 } { 
                                set z [reduce_transition 2 $Y]
                                #puts "reduce_transition: festep z is $z"
                                return $z
                            } else {
                                insert " error in festep of reduce_transition " red
                            }
                        }
                    } 
                    feandi {
                        # x1 < x2 andi x3> = x1<x2> and  x3
                        set E_arg2 [lindex $E 2]
                        set y3 [state_to_flstate $E_arg2]
                        #set y4 "{fluntil true {flnot {flimp {flnot {flnext true}} {flnot {$y3}}}}}"
                        set y1 "{fland {flpchop {$Y} {$E_arg1}} {$y3}}"
                        set z [reduce_transition $k [lindex $y1 0]]
                        return $z
                        #set y1 "{flpchop {$E_arg2} {$Y}}" 
                        #set z [reduce_transition $k [lindex $y1 0]]   
                        #return $z
                    }
                    feandf {
                        # x1 < x2 fand x3> = x1<test(x3);x2> =(x1 and x3)<x2>
                        set E_arg2 [lindex $E 2]
                        set y3 [state_to_flstate $E_arg2]
                        #set y4 "{fluntil true {flnot {flimp {flnot {flnext true}} {flnot {$y3}}}}}"
                        set y1 "{flpchop {fland {$Y} {$y3}} {$E_arg1}}"
                        set z [reduce_transition $k [lindex $y1 0]]
                        return $z
                        #set y3  $E_arg1
                        #set y4 [state_to_flstate $y3]
                        #set y1 "{flchop {$E_arg2} {fland {$y4} {$Y}}}"
                        #set z [reduce_transition $k [lindex $y1 0]]
                        #return $z
                    }
                    feor {
                        set E_arg2 [lindex $E 2]
                        set x "{flor {flpchop {$Y} {$E_arg1}} {flpchop {$Y} {$E_arg2}}}"
                        #puts "reduce_transition: feor z is $z"
                        set z [reduce_transition $k [lindex $x 0]]
                        return $z
                    }
                    fechop {
                        set E_arg2 [lindex $E 2]
                        #puts " $E_arg2"     
                        set x "{flpchop {flpchop {$Y} {$E_arg1}} {$E_arg2}}"
                        set z [reduce_transition $k [lindex $x 0]]
                        #puts "reduce_transition: fechop z is $z"
                        return $z
                    }
                    fepchopstar {
                        # Y<pchopstar E_arg1> 
                        #if { $infinite == 1 } {
                        #    insert "error: can not use chopstar with infinite time" red
                        #    return "{false}"
                        #}
                        #set tmp [array names depindex -exact $L]
                        #puts "tmp is $tmp"
                        if {[array names depindex -exact $L] == "" } {
                            if { $dbg == 1 } {
                                insert_debug "assigning ($L) to $freshx"
                            }
                            set depindex($L) $freshx
                            incr freshx
                        } 
                        #else {
                        # puts "reuse depindex ( )"
                        #}
                        set y1 [c_pfe $E_arg1 ]
                        set x1 "{flor {$Y} {flpchop r_$depindex($L) $y1}}"
                        set y3 [reduce_transition 1 [lindex $x1 0]]
                        #puts "y3 = $y3"
                        set y2 [reduce_state 1 [lindex $x1 0]]
                        # need to transform y2 to a next transition
                        #puts "y2= $y2"
                        set y4 [ptr_to_tr [lindex $y2 0]]
                        #puts "y4= $y4"
                        #set z1 "{Eqv r_$depindex($L) $y2}"
                        set z1 "{Eqv {flnext r_$depindex($L)} $y4}"
                        if { [array names deplist -exact $z1] == "" } {
                            if { $dbg == 1 } {
                                insert_debug "in Red_T pchopstar: adding (next Dep r_$depindex($L)) Eqv $y4"
                            }
                            set deplistp($z1) "r_$depindex($L)"
                            set z "{And $z1 $y4}"
                        } else {
                            set z $y3
                        }
                        set x3 "{Eqv r_$depindex($L) $y2}"
                        if { [array names pdeplist -exact $x3] == "" } {
                            if { $dbg == 1 } {
                                insert_debug "in Red_T prev: adding PDep r_$depindex($L) Eqv $y2"
                            }
                            set pdeplist($x3) "r_$depindex($L)"
                            set z "{And $x3 $y3}"
                        } else {
                            set z $y3
                        }
                        #puts "reduce_transition: fepchopstar z is $z"
                        return $z
                    }
                    fepinit {
                        set x "{flpchop {$Y} {fepchopstar {fepstep {true}}} {fechop {fetest {$E_arg1}}}}"
                        set z [reduce_transition $k [lindex $x 0]]
                        return $z
                    }
                    fepfin {
                        set x "{flpchop {$Y} {fechop {fetest {$E_arg1}} {fepchopstar {fepstep {true}}}}}"
                        set z [reduce_transition $k [lindex $x 0]]
                        return $z
                    }
                    default {
                        set reduce_error 1
                        insert "error in reduce_transition prev fusion expression branch" red
                    }
                }
            }
            fluntil {
                #set x1 [reduce_state 1 [lindex $L 1] ]
                #set x2 [reduce_state 0 [lindex $L 2] ]
                #puts "reduce_transition: fluntil z is $z"
                set Y1 [lindex $L 1]
                set Y2 [lindex $L 2]
                  
                set y12 [reduce_transition 2 $Y1]
                #puts "in red_T: y12 = $y12" 
                set y1 [reduce_state 2 $Y1]
                #puts "in red_T: y1= [lindex $y1 0]"
                
                set y22 [reduce_transition 2 $Y2]
                #puts "in red_T: y22= $y22"
                if { $k == 2 || $k == 1} {
                    if {[array names depindex -exact $L] == "" } {
                        if { $dbg == 1 } {
                            insert_debug "in red_T k= $k: assigning ($L) to $freshx"
                        }
                        set depindex($L) $freshx
                        incr freshx
                    }
                    set y21 [reduce_state 2 $Y2]
                    #puts "in red_T k= $k: y21= $y21"
                    #puts "in red_T k= $k: L = $L"
                    #puts "in red_T k= $k: r_depindex(L) = r_$depindex($L)"  
                    set rL "r_$depindex($L)"
                    if { [lindex $Y1 0] == "fluntil" } {
                        #puts " y22= $y22 "
                        if { $y12 != "true" } { 
                            set y22 "{And $y12 $y22}"
                        }
                    }
                       
                    set y41 "{Eqv $rL {Or $y21 {And $y1 {flnext $rL}}}}"
                    #puts "silly2: $y41"
                    if { [array names deplist -exact $y41] == "" } {
                        if { $dbg == 1 } {
                            insert_debug "in red_T until k= $k: adding Dep $rL Eqv ($y21 or ($y1 and next $rL))"
                        }
                        set deplist($y41) $rL
                        #puts "silly : $depindex($L) [lindex $y21 0]"
                        if {$infinite == 1} {
                            lappend LivenessOffsets [expr $depindex($L) -1]
                            lappend LivenessTests [lindex $y21 0]
                        }
                        #puts "LivenessOffsets = $LivenessOffsets, LivenessTests = $LivenessTests"
                        if { $y22 == "true" } {
                            set z "$y41"
                        } else {
                            set z "{And $y22 $y41}"
                        }
                    } else {
                        set z $y22
                    }
                    #set z1 "{fluntil $y1 $y22}"
                    #puts "in red_T k= $k: z= $z" 
                    return $z
                } else {
                    set reduce_error 1
                    insert " error in fluntil of reduce_transition " red
                }
            }
            flsince {
                #set x1 [reduce_state 1 [lindex $L 1] ]
                #set x2 [reduce_state 0 [lindex $L 2] ]
                #puts "reduce_transition: fluntil z is $z"
                set Y1 [lindex $L 1]
                set Y2 [lindex $L 2]
                  
                set y12 [reduce_transition 2 $Y1]
                #puts "in red_T: y12 = $y12" 
                set y1 [reduce_state 2 $Y1]
                #puts "in red_T: y1= [lindex $y1 0]"
                
                set y22 [reduce_transition 2 $Y2]
                #puts "in red_T: y22= $y22"
                if { $k == 2 || $k == 1} {
                    if {[array names depindex -exact $L] == "" } {
                        if { $dbg == 1 } {
                            insert_debug "in red_T k= $k: assigning ($L) to $freshx"
                        }
                        set depindex($L) $freshx
                        incr freshx
                    }
                    set y21 [reduce_state 2 $Y2]
                    #puts "in red_T k= $k: y21= $y21"
                    #puts "in red_T k= $k: L = $L"
                    #puts "in red_T k= $k: r_depindex(L) = r_$depindex($L)"  
                    set rL "r_$depindex($L)"
                    if { [lindex $Y1 0] == "fluntil" } {
                        #puts " y22= $y22 "
                        if { $y12 != "true" } { 
                            set y22 "{And $y12 $y22}"
                        }
                    }
                       
                    set y42 "{Eqv {flnext $rL} {Or {flnext $y21} {And {flnext $y1}  $rL}}}"
                    #puts "silly2: $y42"
                    if { [array names deplist -exact $y42] == "" } {
                        if { $dbg == 1 } {
                            insert_debug "in red_T until k= $k: adding Dep (next $rL) Eqv ((next $y21) or ((next $y1) and $rL))"
                        }
                        set deplistp($y42) $rL
                        #puts "silly : $depindex($L) [lindex $y21 0]" 
                        #lappend LivenessOffsets [expr $depindex($L) -1]
                        #lappend LivenessTests [lindex $y21 0]
                        #puts "LivenessOffsets = $LivenessOffsets, LivenessTests = $LivenessTests"
                        if { $y22 == "true" } {
                            set z "$y42"
                        } else {
                            set z "{And $y22 $y42}"
                        }
                    } else {
                        set z $y22
                    }
                    set y41 "{Eqv $rL {Or $y21 {And $y1 {flprev $rL}}}}"
                    #puts "silly2: $y41"
                    if { [array names pdeplist -exact $y41] == "" } {
                        if { $dbg == 1 } {
                            insert_debug "in red_T until k= $k: adding PDep $rL Eqv ($y21 or ($y1 and prev $rL))"
                        }
                        set pdeplist($y41) $rL
                        #puts "silly : $depindex($L) [lindex $y21 0]" 
                        #lappend LivenessOffsets [expr $depindex($L) -1]
                        #lappend LivenessTests [lindex $y21 0]
                        #puts "LivenessOffsets = $LivenessOffsets, LivenessTests = $LivenessTests"
                        if { $y22 == "true" } {
                            set z "$y41"
                        } else {
                            set z "{And $y22 $y41}"
                        }
                    } else {
                        set z $y22
                    }
                    #set z1 "{fluntil $y1 $y22}"
                    #puts "in red_T k= $k: z= $z" 
                    return $z
                } else {
                    set reduce_error 1
                    insert " error in flsince of reduce_transition " red
                }
                
            }
            flnext {
                set Y [lindex $L 1]
                #puts "reduce_transition: Y is $Y"
                if { $k == 2 } {
                    if {[array names depindex -exact $L] == "" } {
                        if { $dbg == 1 } {
                            insert_debug "assigning ($L) to $freshx"
                        }
                        set depindex($L) $freshx
                        incr freshx
                    }
                    set x2 [reduce_transition 2 $Y]
                    #puts "x2 is $x2"
                    set x1 [reduce_state 2 $Y]
                    #puts "x1 is $x1"
                    set x3 "{Eqv r_$depindex($L) {flnext $x1}}"
                    if { [array names deplist -exact $x3] == "" } {
                        if { $dbg == 1 } {
                            insert_debug "in Red_T next: adding Dep r_$depindex($L) Eqv  next $x1"
                        }
                        set deplist($x3) "r_$depindex($L)" 
                        set z "{And $x2 $x3}"
                    } else {
                        set z $x2
                    }
                    #set z "{And $x2 {Eqv r_$depindex($L) {flnext $x1}}}"
                    #puts "reduce_transition: flnext z is $z"
                    return $z
                } else {
                    if { $k == 1 } {      
                        set z [reduce_transition 2 $Y]
                        return $z
                    } else {
                        set reduce_error 1
                        insert " error in flnext of reduce_transition " red
                    }
                }     
            }
            flprev {
                set Y [lindex $L 1]
                #puts "reduce_transition: Y is $Y"
                if { $k == 2 } {
                    if {[array names depindex -exact $L] == "" } {
                        if { $dbg == 1 } {
                            insert_debug "assigning ($L) to $freshx"
                        }
                        set depindex($L) $freshx
                        incr freshx
                    }
                    set x2 [reduce_transition 2 $Y]
                    #puts "x2 is $x2"
                    set x1 [reduce_state 2 $Y]
                    #puts "x1 is $x1"
                    #set depindex(r_$depindex($L)) $freshx
                    #incr freshx
                    #set x5 "{Eqv r_$depindex(r_$depindex($L)) {flnext r_$depindex($L)}}"
                    #set x4  "{Eqv r_$depindex(r_$depindex($L)) $x1}"
                    set x4 "{Eqv {flnext r_$depindex($L)} $x1}"
                    if { [array names deplistp -exact $x4] == "" } {
                        if { $dbg == 1 } {
                            #puts "in Red_T prev: adding Dep (r_$depindex(r_$depindex($L))) Eqv (next r_$depindex($L))" 
                            insert_debug "in Red_T prev: adding Dep (next r_$depindex($L)) Eqv  $x1"
                        }
                        set deplistp($x4) "r_$depindex($L)" 
                        #set z "{And $x2 {And $x5 $x4}}"
                        set z "{And $x2 $x4}" 
                    } else {
                        set z $x2
                    } 
                    set x3 "{Eqv r_$depindex($L) {flprev $x1}}"
                    if { [array names pdeplist -exact $x3] == "" } {
                        if { $dbg == 1 } {
                            insert_debug "in Red_T prev: adding PDep r_$depindex($L) Eqv  prev $x1"
                        }
                        set pdeplist($x3) "r_$depindex($L)" 
                        set z "{And $x2 $x3}"
                    } else {
                        set z $x2
                    }
                    #set z "{And $x2 {Eqv r_$depindex($L) {trprev $x1}}}"
                    #puts "reduce_transition: flnext z is $z"
                    return $z
                } else {
                    if { $k == 1 } {      
                        set z [reduce_transition 2 $Y]
                        return $z
                    } else {
                        set reduce_error 1
                        insert " error in flprev of reduce_transition " red
                    }
                }     
            }
            flnot {
                set k1 [expr max($k,1)]
                set z [reduce_transition $k1 [lindex $L 1] ]
                #puts "reduce_transition: flnot z is $z"
                return $z
            }
            flor {
                set k1 [expr max($k,1)]
                set x1 [reduce_transition $k1 [lindex $L 1] ]
                set x2 [reduce_transition $k1 [lindex $L 2] ]
                set z "{And $x1 $x2}"
                #puts "reduce_transition: flor z is $z"
                return $z
            }
            fland {
                set k1 [expr max($k,1)]
                set x1 [reduce_transition $k1 [lindex $L 1] ]
                set x2 [reduce_transition $k1 [lindex $L 2] ]
                set z "{And $x1 $x2}"
                return $z
            }
            flequiv {
                set k1 [expr max($k,1)]
                set x1 [reduce_transition $k1 [lindex $L 1] ]
                set x2 [reduce_transition $k1 [lindex $L 2] ]
                set z "{And $x1 $x2}"
                #puts "reduce_transition: flor z is $z"
                return $z
            }
            flimp {
                set k1 [expr max($k,1)]
                set x1 [reduce_transition $k1 [lindex $L 1] ]
                set x2 [reduce_transition $k1 [lindex $L 2] ]
                set z "{And $x1 $x2}"
                #puts "reduce_transition: flor z is $z"
                return $z
            }
            default {
                set reduce_error 1
                insert "error in reduce_transition switch" red
            }
        }
    }
}


#
# prev_sub_next L
# transform a PL^1 formula into NL^1 formula
#
# L: tree list (state formula from next_sub)
#
proc prev_sub_next { L } {
    set root [lindex $L 0]

    puts "prev_sub_next: L is $L"

    if { [is_state $root] == 1 } { 
        puts "prev_sub_next: is state"
        return "{flnext $L}"
    } else {
        puts "not is state root = [llength $L]"       
        switch -regexp -- [lindex $root 0] {
            And -
            Imp -
            Eqv -
            Or      {
                #puts "first" 
                set oper [lindex $L 0]
                set x1 [prev_sub_next [lindex $L 1] ]
                set y1 [prev_sub_next [lindex $L 2] ]
                set z "{$oper $x1 $y1}"
                return $z
            }
            Not         {
                #puts "second" 
                set oper [lindex $L 0]
                set x1 [prev_sub_next [lindex $L 1] ]
                set z "{$oper $x1}"
                return $z
            }
            trprev -
            flprev {
                #puts "third" 
                set z [lindex $L 1]
                return $z
            }    
            default {
                puts "default [llength $root]" 
                if {[llength $root] == 1 } {
                    #puts "fourth" 
                    return "{$L}"
                } else {
                    if {[llength $root] == 2 } {
                        put "fifth" 
                        set oper [lindex $L 0]
                        set x [prev_sub_next [lindex $L 1] ]
                        return "{$oper $x}"
                    } else {
                        if {[llength $root] == 3 } {
                            set oper [lindex $L 0]
                            set x [prev_sub_next [lindex $L 1] ]
                            set y [prev_sub_next [lindex $L 2] ]
                            return "{$oper $x $y}"
                        } else {
                            insert "error in prev_sub_next" red
                        }
                    }
                }
            }
        }
    }
}

#
# next_sub_replace L
# replace all variables in L by their primed version
#
# L: tree list (state formula from next_sub)
#
proc next_sub_replace { L } {
    set root [lindex $L 0]

    #puts "next_sub_replace: L is $L"
    
    switch -regexp -- $root {
        And -
        Imp -
        Eqv -
        Or      {
            set oper [lindex $L 0]
            set x1 [next_sub_replace [lindex $L 1] ]
            set y1 [next_sub_replace [lindex $L 2] ]
            set z "{$oper $x1 $y1}"
            return $z
        }
        Not         {
            set oper [lindex $L 0]
            set x1 [next_sub_replace [lindex $L 1] ]
            set z "{$oper $x1}"
            return $z
        }
        {r_[0-9]+} { 
            set j [string range $root 2 end]
            return " pr_$j "
        }
        {[A-Z]+[_A-Z]*}   { 
            return " p$root "
        }
        default { 
            if {[llength $L] == 1 } {
                return "{$L}"
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [next_sub_replace [lindex $L 1] ]
                    return "{$oper $x}"
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [next_sub_replace [lindex $L 1] ]
                        set y [next_sub_replace [lindex $L 2] ]
                        return "{$oper $x $y}"
                    } else {
                        insert "error in next_sub_replace" red
                    }
                }
            }
        }
    }
}

#
# prev_sub_replace L
# replace all variables in L by their primed version
#
# L: tree list (state formula from next_sub)
#
proc prev_sub_replace { L } {
    set root [lindex $L 0]

    #puts "prev_sub_replace: L is $L"
    
    switch -regexp -- $root {
        And -
        Imp -
        Eqv -
        Or      {
            set oper [lindex $L 0]
            set x1 [prev_sub_replace [lindex $L 1] ]
            set y1 [prev_sub_replace [lindex $L 2] ]
            set z "{$oper $x1 $y1}"
            return $z
        }
        Not         {
            set oper [lindex $L 0]
            set x1 [prev_sub_replace [lindex $L 1] ]
            set z "{$oper $x1}"
            return $z
        }
        {r_[0-9]+} { 
            set j [string range $root 2 end]
            return " pr_$j "
        }
        {[A-Z]+[_A-Z]*}   { 
            return " p$root "
        }
        default { 
            if {[llength $L] == 1 } {
                return "{$L}"
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [prev_sub_replace [lindex $L 1] ]
                    return "{$oper $x}"
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [prev_sub_replace [lindex $L 1] ]
                        set y [prev_sub_replace [lindex $L 2] ]
                        return "{$oper $x $y}"
                    } else {
                        insert "error in prev_sub_replace" red
                    }
                }
            }
        }
    }
}

# 
# next_sub L
#  find formula in scope of a next
# 
# L : tree list (fusion logic formula)
#
proc next_sub { L } {
    set root [lindex $L 0]
    
    
    switch -regexp -- $root {
        trnext -
        flnext { 
            set x [next_sub_replace [lindex $L 1] ]
            return $x
        }
        default { 
            if {[llength $L] == 1 } {
                return "{$L}"
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [next_sub [lindex $L 1] ]
                    return "{$oper $x}"
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [next_sub [lindex $L 1] ]
                        set y [next_sub [lindex $L 2] ]
                        return "{$oper $x $y}"
                    } else {
                        insert "error in next_sub" red
                    }
                }
            }
        }
    }           
}

# 
# prev_sub L
#  find formula in scope of a prev
# 
# L : tree list (fusion logic formula)
#
proc prev_sub { L } {
    set root [lindex $L 0]
    
    
    switch -regexp -- $root {
        trprev -
        flprev { 
            set x [prev_sub_replace [lindex $L 1] ]
            return $x
        }
        default { 
            if {[llength $L] == 1 } {
                return "{$L}"
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [prev_sub [lindex $L 1] ]
                    return "{$oper $x}"
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [prev_sub [lindex $L 1] ]
                        set y [prev_sub [lindex $L 2] ]
                        return "{$oper $x $y}"
                    } else {
                        insert puts "error in prev_sub" red
                    }
                }
            }
        }
    }           
}

#
# next_sub_false L
#  replace all variables in L within scope of a next by false
#
# L: tree list (fusion logic formula)
#
proc next_sub_false { L } {
    set root [lindex $L 0]
    
    switch -regexp -- $root {
        trnext -
        flnext { 
            return " false "
        }
        default { 
            if {[llength $L] == 1 } {
                return "{$L}"
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [next_sub_false [lindex $L 1] ]
                    return "{$oper $x}"
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [next_sub_false [lindex $L 1] ]
                        set y [next_sub_false [lindex $L 2] ]
                        return "{$oper $x $y}"
                    } else {
                        insert puts "error in next_sub_false" red
                    }
                }
            }
        }
    }            
}

#
# prev_sub_false L
#  replace all variables in L within scope of a prev by false
#
# L: tree list (fusion logic formula)
#
proc prev_sub_false { L } {
    set root [lindex $L 0]
    
    switch -regexp -- $root {
        trprev -
        flprev { 
            return " false "
        }
        default { 
            if {[llength $L] == 1 } {
                return "{$L}"
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [prev_sub_false [lindex $L 1] ]
                    return "{$oper $x}"
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [prev_sub_false [lindex $L 1] ]
                        set y [prev_sub_false [lindex $L 2] ]
                        return "{$oper $x $y}"
                    } else {
                        insert "error in prev_sub_false" red
                    }
                }
            }
        }
    }            
}

# transform reduced tree list to BDD routines
#

#
# flatten args
#  sanitize the tree list args
#
# args: tree list
#
proc flatten args {
    regsub -all {[{}]} $args {} tmp;    # Ditch braces
    regsub -all { +} $tmp { } tmp;      # Squeeze spaces
    set fl [string trim $tmp];          # Chop leading/trailing white space
    return $fl
}

#
# is_var L
#  check whether L is a variable
# 
# L: tree list (fusion logic formula)
#
proc is_var { L } {
  set root [lindex $L 0]
  #puts "is_var: root is $root" 
  switch -regexp -- $root {
    {r_[0-9]+} { 
                   return 1
    }
    {pr_[0-9]+} { 
                   return 1
    }
    {p[A-Z]+[_A-Z]*} { 
                   return 1
    }
    {r[A-Z]+[_A-Z]*} { 
                   return 1
    }
    default { 
                return 0
      }
  }

}

#
# extract_var_indep L
#  extract the independent variables used in formula L
#
# L: tree list (fusion logic formula)
#
proc extract_var_indep { L } {
    set root [lindex $L 0]
    #puts "extract_var: root is $root" 
    switch -regexp -- $root {
        Eqv -
        And -
        Or -
        Not -
        Imp { 
            if {[llength $L] == 1 } {
                return ""
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [extract_var_indep [lindex $L 1] ]
                    return $x
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [extract_var_indep [lindex $L 1] ]
                        lappend z $x
                        set y [extract_var_indep [lindex $L 2] ]
                        lappend z $y
                        return $z
                    } else {
                        insert "error in extract_var_indep" red
                    }
                }
            }
        }
        {r_[0-9]+} { 
            return ""
        }
        {[A-Z]+[_A-Z]*}   { 
            return $root
        }
        default { 
            if {[llength $L] == 1 } {
                return ""
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [extract_var_indep [lindex $L 1] ]
                    return $x
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [extract_var_indep [lindex $L 1] ]
                        lappend z $x
                        set y [extract_var_indep [lindex $L 2] ]
                        lappend z $y
                        return $z
                    } else {
                        insert "error in extract_var_indep" red
                    }
                }
            }
        }
    }
}

#
# extract_var_dep L
#  extract the variables used in formula L
#
# L: tree list (fusion logic formula)
#
proc extract_var_dep { L } {
    set root [lindex $L 0]
    #puts "extract_var: root is $root" 
    switch -regexp -- $root {
        Eqv -
        And -
        Or -
        Not -
        Imp { 
            if {[llength $L] == 1 } {
                return ""
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [extract_var_dep [lindex $L 1] ]
                    return $x
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [extract_var_dep [lindex $L 1] ]
                        lappend z $x
                        set y [extract_var_dep [lindex $L 2] ]
                        lappend z $y
                        return $z
                    } else {
                        insert "error in extract_var_dep" red
                    }
                }
            }
        }
        {r_[0-9]+} { 
            return $root
        }
        {[A-Z]+[_A-Z]*}   { 
            return ""
        }
        default { 
            if {[llength $L] == 1 } {
                return ""
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [extract_var_dep [lindex $L 1] ]
                    return $x
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [extract_var_dep [lindex $L 1] ]
                        lappend z $x
                        set y [extract_var_dep [lindex $L 2] ]
                        lappend z $y
                        return $z
                    } else {
                        insert "error in extract_var_dep" red
                    }
                }
            }
        }
    }
}

#
# extract_var L
#  extract the variables used in formula L
#
# L: tree list (fusion logic formula)
#
proc extract_var { L } {
    set root [lindex $L 0]
    #puts "extract_var: root is $root" 
    switch -regexp -- $root {
        Eqv -
        And -
        Or -
        Not -
        Imp { 
            if {[llength $L] == 1 } {
                return ""
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [extract_var [lindex $L 1] ]
                    return $x
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [extract_var [lindex $L 1] ]
                        lappend z $x
                        set y [extract_var [lindex $L 2] ]
                        lappend z $y
                        return $z
                    } else {
                        insert "error in extract_var" red
                    }
                }
            }
        }
        {r_[0-9]+} { 
            return $root
        }
        {[A-Z]+[_A-Z]*}   { 
            return $root
        }
        default { 
            if {[llength $L] == 1 } {
                return ""
            } else {
                if {[llength $L] == 2 } {
                    set oper [lindex $L 0]
                    set x [extract_var [lindex $L 1] ]
                    return $x
                } else {
                    if {[llength $L] == 3 } {
                        set oper [lindex $L 0]
                        set x [extract_var [lindex $L 1] ]
                        lappend z $x
                        set y [extract_var [lindex $L 2] ]
                        lappend z $y
                        return $z
                    } else {
                        insert "error in extract_var" red
                    }
                }
            }
        }
    }
}

#
# normalise L
#  replace non-primed independent variables A by pA in L
#
# L: tree list (fusion logic formula)
# 
proc normalise { L } {
    
    set root [lindex $L 0]
    set lroot [llength $L]
    
    switch $lroot {
        1   { 
            set x [lindex $L 0]
            if {[string index $x 0] == "p" } {
                return $x
            } else {
                regsub -all {([A-Z]+[_A-Z]*)} $x {r\1} x
                return $x
            }
        }
        2   { 
            set oper [lindex $L 0]
            set x [normalise [lindex $L 1]]
            return [list $oper $x]
        }
        3   { 
            set oper [lindex $L 0]
            set x [normalise [lindex $L 1]]
            set y [normalise [lindex $L 2]]
            return [list $oper $x $y]
        }
        default { 
            return "error in normalise"
        }
    }
}

#
# tree2bdd_init gamma
#  replace dependent variables in gamma by their bdd equivalent 
#
#  gamma: tree list (init of reduced fusion logic formula)
proc tree2bdd_init { gamma } {
global a
global b
variable indep
    
set gammaa [regsub -all {(r_)([0-9]+)} $gamma {[lindex $a [expr {\2 -1}]]}]
return $gammaa
}

#
# tree2bdd gamma
#  transform gamma into a bdd expression
#
# gamma: tree list (reduced fusion logic formula)
#
proc tree2bdd { gamma } {
    global a
    global b
    variable indep
    
#puts "$gamma ->\n"

set gammaa $gamma
    
while { [regsub -all {\{And true true\}} $gammaa {true}] != $gammaa } {
          set gammaa [regsub -all {\{And true true\}} $gammaa {true}]
}     
set gammaa [regsub -all {\{And true false\}} $gammaa {false}]
set gammaa [regsub -all {\{And false true\}} $gammaa {false}]
set gammaa [regsub -all {\{And false false\}} $gammaa {false}]

set gammaa [regsub -all {\{true\}} $gammaa {[One]} ]
set gammaa [regsub -all {\{false\}} $gammaa {[Zero]} ]
set gammaa [regsub -all {\{(fl|st|tr)not true\}} $gammaa {[Zero]} ]
set gammaa [regsub -all {\{(fl|st|tr)not false\}} $gammaa {[One]} ]
set gammaa [regsub -all {\{} $gammaa \[ ]
set gammaa [regsub -all {\}} $gammaa \] ]
set gammaa [regsub -all {(fl|st|tr)or} $gammaa {Or}]
set gammaa [regsub -all {(fl|st|tr)and} $gammaa {And}]
set gammaa [regsub -all {(fl|st|tr)equiv} $gammaa {Eqv}]
set gammaa [regsub -all {(fl|st|tr)imp} $gammaa {Imp}]
set gammaa [regsub -all {(fl|st|tr)not} $gammaa {Not}]
set gammaa [regsub -all {true} $gammaa {[One]}]
set gammaa [regsub -all {false} $gammaa {[Zero]}]
set gammaa [regsub -all {(pr_)([0-9]+)} $gammaa {[lindex $b [expr {\2 -1}]]} ]
set gammaa [regsub -all {(r_)([0-9]+)} $gammaa {[lindex $a [expr {\2 -1}]]} ]
set gammaa [regsub -all {p([A-Z]+[_A-Z]*)} $gammaa {[lindex $b $indep(\1)]} ]
set gammaa [regsub -all {r([A-Z]+[_A-Z]*)} $gammaa {[lindex $a $indep(\1)]} ]
#set gammaa [regsub -all {\[And \[One\] \[One\]\]}  $gammaa {[One]}]

#puts "$gammaa\n"

return $gammaa

}

#
# input_to_trace inputlist
#  convert input to enforcer to a bdd expression
#
# inputList: tree list (input to enforcer)
#
proc input_to_trace { inputList } {
global a
global b
variable indep

#puts " $inputList "

set trace_a [regsub -all {\(} $inputList \[ ]
set trace_a [regsub -all {\)} $trace_a \] ]
set trace_a [regsub -all {([A-Z]+[_A-Z]*)} $trace_a {[lindex $a $indep(\1)]}]
set trace_a [regsub -all {not} $trace_a {Not}]
#set trace_a [regsub -all {tand} $trace_a {And}]
set trace_a [regsub -all {and} $trace_a {And}]
set trace_a [regsub -all {true} $trace_a {[One]}]
return $trace_a
}

#
# input_to_trace_next inputlist
#  convert input to enforcer to a next bdd expression
#
# inputList: tree list (input to enforcer)
#
proc input_to_trace_next { inputList } {
global a
global b
variable indep

#puts " $inputList "

set trace_b [regsub -all {\(} $inputList \[ ]
set trace_b [regsub -all {\)} $trace_b \] ]
set trace_b [regsub -all {([A-Z]+[_A-Z]*)} $trace_b {[lindex $b $indep(\1)]}]
set trace_b [regsub -all {not} $trace_b {Not}]
#set trace_b [regsub -all {tand} $trace_b {And}]
set trace_b [regsub -all {and} $trace_b {And}]
set trace_b [regsub -all {true} $trace_b {[One]}]

#set trace_b [lreplace $trace_b 0 0] 
#lappend trace_b [One]

return $trace_b 
}


# GUI routines
#


proc openfile {} {
  global current curdir

#    variable TEMPURA
#    global term_spawn_id
    set types {
    { {TCL File}           {.tcl}}
    }
    #puts $curdir
    set tmp $current
    set current \
      [tk_getOpenFile -defaultextension ".tcl" -filetypes $types -initialdir $curdir]
    catch {grab .top18} err
    grab release .top18
    focus .top18
    if {$current != ""} {
        newdir [file dir $current]
        cd $curdir
        set ttt [file rootname $current]
        ::bdd_tcl::insert "Setting current file to $current." blue
        ::bdd_tcl::insert " "
        wm title .top18 "FLCHECK: $ttt"
    } else {
      set current $tmp
    }

}

proc load_file { lfile } {
    variable watch
    variable mtime
    global current curdir curfile

    set curdir [file dir $current]

    set curfile [file join $curdir $lfile]

    if {![file exists $curfile]} {
        insert "File $curfile not found!" red

    } else {
        set ttt [file normalize $curfile]
        insert "Loading $ttt"
        source $curfile
    }

    # if { $watch != 0 } {
    #     after cancel $watch
    # }
    # set mtime [file mtime $curfile]
    # watchfile
    # update idletasks



}

proc watchfile {} {
    variable watch
    variable mtime
    global current curdir curfile
    
    
    if { [file exists $curfile] } {
        set mt [file mtime $curfile]
        if {$mt != $mtime} {
            .top18.cpd21.reloadButton configure -style Red.TButton
        } else {
            .top18.cpd21.reloadButton configure -style TButton
        }
    } else {
        insert "Oh no, $curfile has disappeared.\n" red
    }
    
    set watch [after 2500 watchfile]
}

proc check_file { cfile } {
    global current nodisplay

    # removing white space from front and end of cfile
    set cfile [string trim $cfile]

    if {![file exists $cfile]} {
        set ttt [file normalize $cfile]
        puts "File $ttt not found!" 
        set current ""
    } else {
        #puts "Setting current file to $cfile." 
        set current $cfile
        if { $nodisplay == 0 } {
            set ttt [file rootname $current]
            ::bdd_tcl::insert "Setting current file to $cfile." blue
            ::bdd_tcl::insert " "
            wm title .top18 "FLCHECK: $ttt"
        } else {
            ::bdd_tcl::insert "Setting current file to $cfile." blue
        }
    }
    

}

proc check_batch { cfile } {
    global curbatch

    if {![file exists $cfile]} {
        set ttt [file normalize $cfile]
        ::bdd_tcl::insert "File $ttt not found!" red
        set curbatch ""
    } else {
        set curbatch $cfile
    }  

}

# toggle_dbg --
#
#   ADD COMMENTS HERE
#
# Arguments:
#   args	comments
# Results:
#   Returns ...
#
proc toggle_dbg {} {

    global dbg

    if { $dbg == 0 } {
        set dbg 1
    } else {
        set dbg 0
    }

}

# savefile --
#
#   ADD COMMENTS HERE
#
# Arguments:
#   args	comments
# Results:
#   Returns ...
#
proc savefile {} {
    global out current curdir

    set types {
    {{Output  Files}      {.out} }
    }
    set tmp $curdir
    set save_file \
        [tk_getSaveFile -defaultextension ".out" -filetypes $types -initialdir $curdir    ]
    catch {grab .top18} err
    grab release .top18
    if {$save_file != ""} {
        newdir [file dir $save_file]
        set out_file [file tail $save_file]
        cd $curdir
        set toutput [open $out_file w]
        puts $toutput [$out get 0.0 end]
        close $toutput
        set curdir $tmp
        cd $curdir
    } 

}

# savefile --
#
#   ADD COMMENTS HERE
#
# Arguments:
#   args	comments
# Results:
#   Returns ...
#
proc dbgsavefile {} {
    global tdebug current curdir

    set types {
    {{Debug  Files}      {.dbg} }
    }
    set tmp $curdir
    set save_file \
        [tk_getSaveFile -defaultextension ".dbg" -filetypes $types -initialdir $curdir    ]
    catch {grab .top18} err
    grab release .top18
    if {$save_file != ""} {
        newdir [file dir $save_file]
        set out_file [file tail $save_file]
        cd $curdir
        set toutput [open $out_file w]
        puts $toutput [$tdebug get 0.0 end]
        close $toutput
        set curdir $tmp
        cd $curdir
    } 

}

# newdir --
#
#   ADD COMMENTS HERE
#
# Arguments:
#   args	comments
#    d          directory
# Results:
#   Returns ...
#
proc newdir {d} {
    global curdir
    if {![file isdirectory $d]} {return}
    set curdir "[convertfilename $d]"
}


# convertfilename --
#
#   ADD COMMENTS HERE
#
# Arguments:
#   args	comments
#    f          filename
# Results:
#   Returns ...
#
proc convertfilename {f} {
    return $f
}



proc setTooltip {widget text} { 

    if { $text != "" } {
        # 2) Adjusted timings and added key and button bindings. These seem to
        # make artifacts tolerably rare.
        bind $widget <Any-Enter>    [list after 500 [list ::bdd_tcl::showTooltip %W $text]]
        bind $widget <Any-Leave>    [list after 500 [list destroy %W.tooltip]] 
        bind $widget <Any-KeyPress> [list after 500 [list destroy %W.tooltip]] 
        bind $widget <Any-Button>   [list after 500 [list destroy %W.tooltip]] 
   } 
}

proc showTooltip {widget text} {
    global tcl_platform
    
    if { [string match $widget* [winfo containing  [winfo pointerx .] [winfo pointery .]] ] == 0  } {
        return 
    } 


   catch { destroy $widget.tooltip } 


   set scrh [winfo screenheight $widget]    ; # 1) flashing window fix 
   set scrw [winfo screenwidth $widget]     ; # 1) flashing window fix 
   set tooltip [toplevel $widget.tooltip -bd 1 -bg black] 
   wm geometry $tooltip +$scrh+$scrw        ; # 1) flashing window fix
   wm overrideredirect $tooltip 1 

   if {$tcl_platform(platform) == {windows}} { ; # 3) wm attributes...
       wm attributes $tooltip -topmost 1   ; # 3) assumes...
   }                                           ; # 3) Windows
   pack [label $tooltip.label -bg lightyellow -fg black -text $text -justify left] 
   set width [winfo reqwidth $tooltip.label] 
   set height [winfo reqheight $tooltip.label] 

   set pointer_below_midline [expr [winfo pointery .] > [expr [winfo screenheight .] / 2.0]]                ; # b.) Is the pointer in the bottom half of the screen?

   set positionX [expr [winfo pointerx .] - round($width / 2.0)]    ; # c.) Tooltip is centred horizontally on pointer.
   set positionY [expr [winfo pointery .] + 35 * ($pointer_below_midline * -2 + 1) - round($height / 2.0)]  ; # b.) Tooltip is displayed above or below depending on pointer Y position.

   # a.) Ad-hockery: Set positionX so the entire tooltip widget will be displayed.
   # c.) Simplified slightly and modified to handle horizontally-centred tooltips and the left screen edge.
   if  {[expr $positionX + $width] > [winfo screenwidth .]} {
       set positionX [expr [winfo screenwidth .] - $width]
   } elseif {$positionX < 0} {
       set positionX 0
   }

   wm geometry $tooltip [join  "$width x $height + $positionX + $positionY" {}] 
   raise $tooltip 

   # 2) Kludge: defeat rare artifact by passing mouse over a tooltip to destroy it.
   bind $widget.tooltip <Any-Enter> {destroy %W} 
   bind $widget.tooltip <Any-Leave> {destroy %W} 
} 


proc Window {args} {
    #global vTcl
    
    set cmd [lindex $args 0]
    set name [lindex $args 1]
    set newname [lindex $args 2]
    set rest [lrange $args 3 end]
    
    if {$name == "" || $cmd == ""} {return}
    if {$newname == ""} {
        set newname $name
    }
    set exists [winfo exists $newname]
    switch $cmd {
        show {
            if {$exists == "1" && $name != "."} {wm deiconify $name; return}
            if {[info procs flcheckWindow(pre)$name] != ""} {
                eval "flcheckWindow(pre)$name $newname $rest"
            }
            if {[info procs flcheckWindow$name] != ""} {
                eval "flcheckWindow$name $newname $rest"
            }
            if {[info procs flcheckWindow(post)$name] != ""} {
                eval "flcheckWindow(post)$name $newname $rest"
            }
        }
        hide    { if {$exists} {wm withdraw $newname; return} }
        iconify { if {$exists} {wm iconify $newname; return} }
        destroy { if {$exists} {destroy $newname; return} }
    }
}

proc flcheckWindow. {base} {
    if {$base == ""} {
        set base .
    }
    ###################
    # CREATING WIDGETS
    ###################
#    wm focusmodel $base passive
    wm geometry $base 122x26+0+0
    #wm maxsize $base 1137 870
    wm minsize $base 1 1
    wm overrideredirect $base 0
    wm resizable $base 1 1
    wm withdraw $base
    wm title $base "BDD"
    ###################
    # SETTING GEOMETRY
    ###################
}

# GetSelection --
#
#   ADD COMMENTS HERE
#
# Arguments:
#   args	comments
# Results:
#   Returns ...
#
proc GetSelection {w} {
	if {
	    ![catch {selection get -displayof $w} txt] ||
	    ![catch {selection get -displayof $w -selection CLIPBOARD} txt]
	} {
	    return $txt
	}
	return -code error "could not find default selection"
}


proc flcheckWindow.top18 {base} {
    global out help term entry_widget notebook_widget tdebug
    global bddfont tback back current writing
      
    if {$base == ""} {
        set base .top18
    }
    if {[winfo exists $base]} {
        wm deiconify $base; return
    }
    ###################
    # CREATING WIDGETS
    ###################
    toplevel $base -class Toplevel  
#    wm focusmodel $base passive
    wm geometry $base 800x570+1+1
    #wm maxsize $base 1200 870
    wm minsize $base 1 1
    wm overrideredirect $base 0
    wm resizable $base 1 1
    wm deiconify $base
    wm title $base "FLCHECK:$current"

    wm protocol $base WM_DELETE_WINDOW {
         exit
    }


    # f -> cpd20
    ttk::frame $base.f \
        -borderwidth 0 -relief raised

    if [string equal $::tcl_platform(os) "Darwin"] {
        menu $base.f.01

        $base.f.01 add cascade -label "File" \
           -font $bddfont \
           -underline 0 -menu $base.f.01.02

        $base configure -menu $base.f.01

        menu $base.f.01.02 -relief raised -tearoff 0 

        $base.f.01.02 add command \
          -label Exit   -font $bddfont \
          -command { exit}
        
    }

   
    # menu $base.f.fmenu

    # $base.f.fmenu add cascade -label "File" \
    #    -font $bddfont\
    #    -underline 0 -menu $base.f.fmenu.entry
    # $base configure -menu $base.f.fmenu

    # menu $base.f.fmenu.entry -relief raised  \
    #     -tearoff 0


    # $base.f.fmenu.entry add command \
    #    -font $bddfont\
    #    -label Open -command {::bdd_tcl::openfile}

    
#    $base.f.fmenu.entry add command \
#        -font $bddfont\
#        -label Reload -command {source  fusion_logic_derived.tcl}

    # $base.f.fmenu.entry add command \
    #     -font $bddfont\
    #     -label Exit -command {exit}

    # f0 --> cpd21
    ttk::frame $base.f0 \
        -borderwidth 1 -relief raised

    ttk::button $base.f0.openButton -width 6 -text "OPEN"  \
        -command {::bdd_tcl::openfile}

    bdd_tcl::setTooltip $base.f0.openButton "Open Fusion Logic file"

    # ttk::button $base.f0.reloadButton -width 6 -text "RELOAD"  \
    #     -command {::bdd_tcl::reload} -state disabled

    # bdd_tcl::setTooltip $base.f0.reloadButton "Load Fusion Logic file again"

    ttk::button $base.f0.saveButton -width 6 -text "SAVE"  \
        -command {::bdd_tcl::savefile}

    ::bdd_tcl::setTooltip $base.f0.saveButton "Save Fusion Logic output"

    ttk::button $base.f0.clearButton -width 6 -text "CLEAR"  \
        -command {::bdd_tcl::o_clearnc}

    ::bdd_tcl::setTooltip  $base.f0.clearButton "Clear FLCHECK output"

    ttk::checkbutton $base.f0.dbgButton -width 6 -text "DEBUG"  \
        -variable dbg 

    ::bdd_tcl::setTooltip  $base.f0.dbgButton "Toggle debug"

    ttk::button $base.f0.dclearButton -width 7 -text "DBG CLR"  \
        -command {::bdd_tcl::d_clearnc}

    ::bdd_tcl::setTooltip  $base.f0.dclearButton "Clear DEBUG output"

    ttk::button $base.f0.dbgsaveButton -width 7 -text "DBG SAVE"  \
        -command {::bdd_tcl::dbgsavefile}

    ::bdd_tcl::setTooltip $base.f0.dbgsaveButton "Save DEBUG output"

    
    # f1 --> cpd22
    ttk::frame $base.f1  \
        -borderwidth 1 -height 405 -relief raised \
        -width 169 


    
    ttk::notebook $base.f1.n
    set notebook_widget $base.f1.n

    ttk::frame $base.f1.n.f1
    set frm $base.f1.n.f1

    $base.f1.n add $frm -text "FLCHECK"

    ttk::scrollbar $frm.04\
        -command [list $frm.06 xview] -orient horiz
    ttk::scrollbar $frm.05 \
        -command [list $frm.06 yview] -orient vert
    
    ::history::for text $frm.06 \
        -background $tback \
        -cursor xterm -borderwidth 1 \
        -highlightthickness 0\
        -relief sunken -font $bddfont \
        -wrap char -state normal \
        -height 17 \
        -insertbackground black \
        -insertwidth 4\
        -width 80 \
        -xscrollcommand [list $frm.04 set]\
        -yscrollcommand [list $frm.05 set]
    
        # -background $tback\
        # -relief sunken -font $bddfont\
        # -cursor xterm -borderwidth 1 \
        # -highlightthickness 0\
        # -wrap char -state normal\
        # -height 17 \
        # -insertbackground black \
        # -selectbackground black \
        # -selectforeground $back\
        # -width 80 \ 
        # -xscrollcommand [list $frm.04 set]\
        # -yscrollcommand [list $frm.05 set]

    set out $frm.06
    $out mark set limit insert
    $out mark gravity limit left

    menu $out.pop
    $out.pop add command -label "Cut"   \
             -command {event generate [focus] <<Cut>>}
    $out.pop add command -label "Copy"   \
        -command {event generate [focus] <<Copy>>}
    $out.pop add command -label "Paste"  \
        -command {event generate [focus] <<Paste>>}


    
    # ttk::scrollbar $base.f1.01 \
    #     -command [list $base.f1.03 xview] -orient horiz  
    # ttk::scrollbar $base.f1.02  \
    #     -command [list $base.f1.03 yview] -orient vert 
    # text $base.f1.03 -background $tback \
    #     -font $bddfont\
    #    -cursor xterm -borderwidth 1 \
    #    -highlightthickness 0\
    #    -wrap none -state disabled\
    #    -height 7 \
    #    -insertbackground black \
    #    -selectbackground black \
    #    -selectforeground $back\
    #    -xscrollcommand [list $base.f1.01 set]\
    #    -yscrollcommand [list $base.f1.02 set]
        

    # set entry_widget $base.f1.04

    # history::for ttk::entry $entry_widget -font $bddfont -textvar enter_text
    
    # #ttk::button $base.f1.exit   -text "Exit"  -command "exit"

    # set term $base.f1.03


    ttk::frame $base.f1.n.f2
    set frmh $base.f1.n.f2

    $base.f1.n add $frmh -text "Help"

    ttk::scrollbar $frmh.04\
        -command [list $frmh.06 xview] -orient horiz
    ttk::scrollbar $frmh.05 \
        -command [list $frmh.06 yview] -orient vert
    text $frmh.06 -background $tback\
       -font $bddfont\
       -cursor xterm -borderwidth 1 \
       -highlightthickness 0\
       -wrap none -state disabled\
       -height 17 \
       -insertbackground black \
       -selectbackground black \
       -selectforeground $back\
       -xscrollcommand [list $frmh.04 set]\
       -yscrollcommand [list $frmh.05 set]


    set help $frmh.06

    ttk::frame $base.f1.n.f3
    set frmd $base.f1.n.f3

    $base.f1.n add $frmd -text "Debug"

    ttk::scrollbar $frmd.04\
        -command [list $frmd.06 xview] -orient horiz
    ttk::scrollbar $frmd.05 \
        -command [list $frmd.06 yview] -orient vert
    text $frmd.06 -background $tback\
       -font $bddfont\
       -cursor xterm -borderwidth 1 \
       -highlightthickness 0\
       -wrap none -state disabled\
       -height 17 \
       -insertbackground black \
       -selectbackground black \
       -selectforeground $back\
       -xscrollcommand [list $frmd.04 set]\
       -yscrollcommand [list $frmd.05 set]


    set tdebug $frmd.06

    
    
    bind $notebook_widget  <ButtonRelease-1> {+
        update idletasks
    }

    bind all <Control-plus>  [list ::bdd_tcl::bump_fonts +2]
    bind all <Control-equal> [list ::bdd_tcl::bump_fonts +2]
    bind all <Control-minus> [list ::bdd_tcl::bump_fonts -2]

    bind $out <Return> {

        if { $writing == 0 } {
            
            #set show_output 0
            #::history::add? $out;
            
            set enter_text [$out get limit end-1c];
            ::history::addto $out $enter_text
            #puts "entered: $enter_text" 
            set temp [string range $enter_text 0 3]
            if { $temp == "exit" } {
                exit
            } else {
                if {$enter_text != "" } {
                    
                    #set temp $enter_text
                    #::bdd_tcl::insert_input $enter_text
                    ::bdd_tcl::remove_tabs .top18
                    ::bdd_tcl::insert " "
                    ::bdd_tcl::check $enter_text
                    #::bdd_tcl::insert ">"
                    #set enter_text ""
                    
                    #$out insert end \n;
                    
                    $out mark set insert end
                    $out mark set limit insert
                    $out see end
                    catch {grab .top18} err
                    grab release .top18
                    break
                } else {
                    $out mark set insert end
                    $out mark set limit insert;
                    $out see end
                    break
                }
            }
        } else {
            break
        }
    }

    if {[tk windowingsystem]=="aqua"} {
        bind $out <Button-2> {
            tk_popup $out.pop %X %Y
        }
        bind $out <Control-Button-1> {
            tk_popup $out.pop %X %Y
        }
    } else {
        bind $out <Button-3> {
            tk_popup $out.pop %X %Y
        }
    }


    bind $out <ButtonRelease-1> {
        if {[%W compare insert < limit]} {
            %W configure -insertbackground red
        } else {
            %W configure -insertbackground black
        }
    }

    bind $out <<Paste>> {
        if {![catch {GetSelection %W} txt]} {
            catch {
		if {[%W compare sel.first >= limit]} {
		    %W delete sel.first sel.last
		}
            }
            if {[%W compare insert < limit]} { %W mark set insert end }
            %W configure -insertbackground black
            %W insert insert $txt
	    %W see insert
        }
        break
    }

    bind $out <<Cut>> {
        if {[string match %W [selection own -displayof %W]]} {
	    clipboard clear -displayof %W
	    catch {
		set txt [selection get -displayof %W]
		clipboard append -displayof %W $txt
		if {[%W compare sel.first >= limit]} {
                    %W configure -insertbackground black
		    %W delete sel.first sel.last
		}
	    }
        }
        break
    }

    bind $out <BackSpace>  {
        if {[%W tag nextrange sel 1.0 end ] != ""} {
            %W delete sel.first sel.last
        } elseif {[%W compare insert > limit]} {
            %W delete insert-1c
            %W see insert
        }
        break
    }

    bind $out <Key> {
        if [%W compare insert < limit] {
            %W configure -insertbackground black
            %W mark set insert end
        }
    }

    bind $out <KeyRelease> {+
   	update idletasks
    }

    bind $out <Any-Enter> {+
   	focus %W
    }
    
    # f  --> cpd20
    # f0 --> cpd21
    # f1 --> cpd22
    
    pack $base.f \
        -in $base -anchor center -expand 0 -fill x -side top

    pack $base.f0.openButton -side left -padx 2 -pady 2

    #pack $base.f0.reloadButton -side left -padx 2 -pady 2

    pack $base.f0.saveButton -side left -padx 2 -pady 2

    pack $base.f0.clearButton -side left -padx 2 -pady 2

    pack $base.f0.dbgButton -side right -padx 2 -pady 2

    pack $base.f0.dclearButton -side left -padx 2 -pady 2

    pack $base.f0.dbgsaveButton -side left -padx 2 -pady 2

    pack $base.f0 -fill x
    
    
    pack $base.f1 \
        -in $base -anchor center -expand 1 -fill both -side top

    grid columnconf $base.f1 0 -weight 1
    grid rowconf $base.f1 0 -weight 1
    
   #  grid $base.f1.exit \
   #     -in $base.f1 -column 0 -row 0 -columnspan 2 -rowspan 1 \
   #     -sticky nsew
     grid $base.f1.n \
        -in $base.f1 -row 0 -column 0 -columnspan 2 -rowspan 1 -sticky nsew
    
     # grid $base.f1.01 \
     #   -in $base.f1 -row 2 -column 0 -columnspan 1 -rowspan 1 -sticky nsew
     # grid $base.f1.02 \
     #   -in $base.f1 -row 1 -column 1 -columnspan 1 -rowspan 1 -sticky nsew
     # grid $base.f1.03 \
     #   -in $base.f1 -row 1 -column 0 -columnspan 1 -rowspan 1 -sticky nsew
     # grid $base.f1.04 \
     #    -in $base.f1 -column 0 -row 3 -columnspan 2 -rowspan 1 \
     #    -sticky nesw
     
     

    grid columnconf $frm 0 -weight 1
    grid rowconf $frm 0 -weight 1
    grid $frm.04 \
        -in $frm -column 0 -row 1 -columnspan 1 -rowspan 1 -sticky ew 
    grid $frm.05 \
        -in $frm -column 1 -row 0 -columnspan 1 -rowspan 1 -sticky ns 
    grid $frm.06 \
        -in $frm -column 0 -row 0 -columnspan 1 -rowspan 1 \
        -sticky nesw

    grid columnconf $frmh 0 -weight 1
    grid rowconf $frmh 0 -weight 1
    grid $frmh.04 \
        -in $frmh -column 0 -row 1 -columnspan 1 -rowspan 1 -sticky ew 
    grid $frmh.05 \
        -in $frmh -column 1 -row 0 -columnspan 1 -rowspan 1 -sticky ns 
    grid $frmh.06 \
        -in $frmh -column 0 -row 0 -columnspan 1 -rowspan 1 \
        -sticky nesw

    grid columnconf $frmd 0 -weight 1
    grid rowconf $frmd 0 -weight 1
    grid $frmd.04 \
        -in $frmd -column 0 -row 1 -columnspan 1 -rowspan 1 -sticky ew 
    grid $frmd.05 \
        -in $frmd -column 1 -row 0 -columnspan 1 -rowspan 1 -sticky ns 
    grid $frmd.06 \
        -in $frmd -column 0 -row 0 -columnspan 1 -rowspan 1 \
        -sticky nesw

}

proc bump_fonts {incr} {
    global bddfont
    
    catch {
      set size [font configure $bddfont -size]
      incr size $incr
      if {$size >= 8 && $size <= 32} {
        font configure $bddfont -size $size
      }
    }
}



proc insert {txt {tag normal} } {
    global out nodisplay writing
    if { $nodisplay == 0 } {
        set writing 1
        $out configure -state normal
        $out insert end $txt $tag
        $out see end
        $out insert end "\n"
        $out see end
        $out mark set limit insert
        #$out configure -state disabled
        set writing 0
        update idletasks
    } else {
        puts $txt
    }
 
}

# o_clearnc --
#
#   ADD COMMENTS HERE
#
# Arguments:
#   args	comments
# Results:
#   Returns ...
#
proc o_clearnc {} {
    global out nodisplay
    if { $nodisplay == 0 } {
        $out delete 1.0 {end -1 lines}
        $out mark set limit insert
        update idletasks
    }
}

# d_clearnc --
#
#   ADD COMMENTS HERE
#
# Arguments:
#   args	comments
# Results:
#   Returns ...
#
proc d_clearnc {} {
    global tdebug nodisplay
    if { $nodisplay == 0 } {
        $tdebug configure -state normal
        $tdebug delete 1.0 end
        $tdebug configure -state disabled
        #$tdebug mark set limit insert
        update idletasks
    }
}

proc insert_help {txt {tag normal} } {
  global help  nodisplay


    if {$nodisplay == 0} {
        $help configure -state normal
        $help insert end $txt $tag
        $help see end
        $help insert end "\n"
        $help see end
        $help configure -state disabled
    } else {
        puts $txt
    }
   
}

proc insert_debug {txt {tag normal} } {
  global tdebug  nodisplay


    if {$nodisplay == 0} {
        $tdebug configure -state normal
        $tdebug insert end $txt $tag
        $tdebug see end
        $tdebug insert end "\n"
        $tdebug see end
        $tdebug configure -state disabled
    } else {
        puts $txt
    }
   
}

proc help_begin {} {
  global help  nodisplay

    if {$nodisplay == 0} {
        $help see 1.0
    }
   
}

proc insert_input {txt {tag normal} } {
  global term nodisplay
  if { $nodisplay == 0 } { 
    $term configure -state normal
    $term insert end $txt $tag
    $term see end
    $term insert end "\n"
    $term see end
    $term configure -state disabled
    update idletasks
  } else {
   puts $txt
  }
}

#proc insert1 {c} {
#    global cmd
#    global point
#
#    set cmd [linsert $cmd $point $c]
#    incr point
#}

#proc backspace {} {
#    global cmd
#    global point
#
#    if {0 < $point} {
#        incr point -1
#        set cmd [lreplace $cmd $point $point]
#    }
#
#    return
#}

#proc right {} {
#    global cmd
#    global point
#
#    if {$point < [llength $cmd]} {
#        incr point 1
#    }
#
#    return
#}

#proc left {} {
#    global point
#
#    if {0 < $point} {
#        incr point -1
#    }
#
#    return
#}

#proc up {} {
#    global cmd
#    global point
#    global event
#    global saveCmd
#    global term
#
    # If we're leaving event 1 (the new command) save it.
#
#    if {$event == 1} {
#        set saveCmd [join $cmd ""]
#    }
#
#    # Erase this command and get the previous event.
#
#    control-u
#    $term insert end "\n"
#    if {[expr {1 - [history keep]}] < $event} {
#        incr event -1
#    }
#    set cmd [history event $event]
#    $term insert insert $cmd
#    $term see insert
#    set cmd [split $cmd {}]
#    set point [llength $cmd]
#
#    return
#}

#proc control-u {} {
#    global cmd
#    global point
#    global term
#
#    while {$point > 0} {
#         $term mark set insert insert-1c
#        incr point -1
#    }
#    $term delete insert end
#    $term insert end "\n"
#    $term see end
#    set cmd {}
#
#    return
#}

#proc down {} {
#    global cmd
#    global point
#    global event
#    global saveCmd
#    global term
#
#    # Erase this command and get the next event.
#
#    if {$event == 1} {
#        return
#    }
#    control-u
#    $term insert end "\n"
#    incr event
#    if {$event == 1} {
#        set cmd $saveCmd
#    } else {
#        set cmd [history event $event]
#    }
#
#    $term insert insert $cmd
#    $term see insert
#    set cmd [split $cmd {}]
#    set point [llength $cmd]
#
#}

proc remove_tabs {base} {
  global tab
  #puts "tab: $tab"
  for {set i 4} {$i<$tab} {incr i} {
   #puts "remove tab $i"
   $base.f1.n forget $base.f1.n.f$i
  }
  set tab 4
}

proc new_tab {base fnr {tabtext "BDD"}} {
   global c 
   catch { ttk::frame $base.f1.n.f$fnr }

    set frm $base.f1.n.f$fnr
    if {$tabtext == "BDD"} {
        set tabtext "$tabtext [expr $fnr -5]"
    }
    $base.f1.n add $frm -text $tabtext 

    catch { ttk::scrollbar $frm.04\
        -command [list $frm.06 xview] -orient horiz }
    catch { ttk::scrollbar $frm.05 \
        -command [list $frm.06 yview] -orient vert }
    catch { canvas $frm.06 -width 300 -height 200 -background white\
       -highlightthickness 0\
       -xscrollcommand [list $frm.04 set]\
       -yscrollcommand [list $frm.05 set] }
    
    set c $frm.06

    
     grid $frm.04 \
        -in $frm -column 0 -row 1 -columnspan 3 -rowspan 1 -sticky ew 
     grid $frm.05 \
        -in $frm -column 3 -row 0 -columnspan 1 -rowspan 1 -sticky ns 
     grid $frm.06 \
        -in $frm -column 0 -row 0 -columnspan 3 -rowspan 1 \
        -sticky nesw 
     grid columnconf $frm 0 -weight 1 
     grid columnconf $frm 1 -weight 1 
     grid columnconf $frm 2 -weight 1 
     grid rowconf $frm 0 -weight 1 
    catch { ttk::button $frm.zoomin   -text "Zoom in"  -command "::bdd_tcl::zoom $c 1.25" }
    catch { ttk::button $frm.zoomout  -text "Zoom out" -command "::bdd_tcl::zoom $c 0.8" }
     grid $frm.zoomin $frm.zoomout -sticky ew
    # Set up event bindings for canvas:
    bind $c <3> "::bdd_tcl::zoomMark $c %x %y"
    bind $c <B3-Motion> "::bdd_tcl::zoomStroke $c %x %y"
    bind $c <ButtonRelease-3> "::bdd_tcl::zoomArea $c %x %y"


}

#
# zoom routines from http://wiki.tcl.tk/4844
#

#--------------------------------------------------------
#
#  zoomMark
#
#  Mark the first (x,y) coordinate for zooming.
#
#--------------------------------------------------------
proc zoomMark {c x y} {
    global zoomArea
    set zoomArea(x0) [$c canvasx $x]
    set zoomArea(y0) [$c canvasy $y]
    $c create rectangle $x $y $x $y -outline black -tag zoomArea
}

#--------------------------------------------------------
#
#  zoomStroke
#
#  Zoom in to the area selected by itemMark and
#  itemStroke.
#
#--------------------------------------------------------
proc zoomStroke {c x y} {
    global zoomArea
    set zoomArea(x1) [$c canvasx $x]
    set zoomArea(y1) [$c canvasy $y]
    $c coords zoomArea $zoomArea(x0) $zoomArea(y0) $zoomArea(x1) $zoomArea(y1)
}

#--------------------------------------------------------
#
#  zoomArea
#
#  Zoom in to the area selected by itemMark and
#  itemStroke.
#
#--------------------------------------------------------
proc zoomArea {c x y} {
    global zoomArea

    #--------------------------------------------------------
    #  Get the final coordinates.
    #  Remove area selection rectangle
    #--------------------------------------------------------
    set zoomArea(x1) [$c canvasx $x]
    set zoomArea(y1) [$c canvasy $y]
    $c delete zoomArea

    #--------------------------------------------------------
    #  Check for zero-size area
    #--------------------------------------------------------
    if {($zoomArea(x0)==$zoomArea(x1)) || ($zoomArea(y0)==$zoomArea(y1))} {
        return
    }

    #--------------------------------------------------------
    #  Determine size and center of selected area
    #--------------------------------------------------------
    set areaxlength [expr {abs($zoomArea(x1)-$zoomArea(x0))}]
    set areaylength [expr {abs($zoomArea(y1)-$zoomArea(y0))}]
    set xcenter [expr {($zoomArea(x0)+$zoomArea(x1))/2.0}]
    set ycenter [expr {($zoomArea(y0)+$zoomArea(y1))/2.0}]

    #--------------------------------------------------------
    #  Determine size of current window view
    #  Note that canvas scaling always changes the coordinates
    #  into pixel coordinates, so the size of the current
    #  viewport is always the canvas size in pixels.
    #  Since the canvas may have been resized, ask the
    #  window manager for the canvas dimensions.
    #--------------------------------------------------------
    set winxlength [winfo width $c]
    set winylength [winfo height $c]

    #--------------------------------------------------------
    #  Calculate scale factors, and choose smaller
    #--------------------------------------------------------
    set xscale [expr {$winxlength/$areaxlength}]
    set yscale [expr {$winylength/$areaylength}]
    if { $xscale > $yscale } {
        set factor $yscale
    } else {
        set factor $xscale
    }

    #--------------------------------------------------------
    #  Perform zoom operation
    #--------------------------------------------------------
    zoom $c $factor $xcenter $ycenter $winxlength $winylength
}

#--------------------------------------------------------
#
#  zoom
#
#  Zoom the canvas view, based on scale factor
#  and centerpoint and size of new viewport.
#  If the center point is not provided, zoom
#  in/out on the current window center point.
#
#  This procedure uses the canvas scale function to
#  change coordinates of all objects in the canvas.
#
#--------------------------------------------------------
proc zoom { canvas factor \
    {xcenter ""} {ycenter ""} \
    {winxlength ""} {winylength ""} } {

    #--------------------------------------------------------
    #  If (xcenter,ycenter) were not supplied,
    #  get the canvas coordinates of the center
    #  of the current view.  Note that canvas
    #  size may have changed, so ask the window
    #  manager for its size
    #--------------------------------------------------------
    if { [string equal $xcenter ""] } {
        set winxlength [winfo width $canvas]
        set winylength [winfo height $canvas]
        set xcenter [$canvas canvasx [expr {$winxlength/2.0}]]
        set ycenter [$canvas canvasy [expr {$winylength/2.0}]]
    }

    #--------------------------------------------------------
    #  Scale all objects in the canvas
    #  Adjust our viewport center point
    #--------------------------------------------------------
    $canvas scale all 0 0 $factor $factor
    set xcenter [expr {$xcenter * $factor}]
    set ycenter [expr {$ycenter * $factor}]

    #--------------------------------------------------------
    #  Get the size of all the items on the canvas.
    #
    #  This is *really easy* using
    #      $canvas bbox all
    #  but it is also wrong.  Non-scalable canvas
    #  items like text and windows now have a different
    #  relative size when compared to all the lines and
    #  rectangles that were uniformly scaled with the
    #  [$canvas scale] command.
    #
    #  It would be better to tag all scalable items,
    #  and make a single call to [bbox].
    #  Instead, we iterate through all canvas items and
    #  their coordinates to compute our own bbox.
    #--------------------------------------------------------
    set x0 1.0e30; set x1 -1.0e30 ;
    set y0 1.0e30; set y1 -1.0e30 ;
    foreach item [$canvas find all] {
        switch -exact [$canvas type $item] {
       	"arc" -
       	"line" -
       	"oval" -
       	"polygon" -
       	"rectangle" {
       	        set coords [$canvas coords $item]
       	        foreach {x y} $coords {
       		    if { $x < $x0 } {set x0 $x}
       		    if { $x > $x1 } {set x1 $x}
       		    if { $y < $y0 } {set y0 $y}
       		    if { $y > $y0 } {set y1 $y}
       	        }
       	     }
        }
    }

    #--------------------------------------------------------
    #  Now figure the size of the bounding box
    #--------------------------------------------------------
    set xlength [expr {$x1-$x0}]
    set ylength [expr {$y1-$y0}]

    #--------------------------------------------------------
    #  But ... if we set the scrollregion and xview/yview
    #  based on only the scalable items, then it is not
    #  possible to zoom in on one of the non-scalable items
    #  that is outside of the boundary of the scalable items.
    #
    #  So expand the [bbox] of scaled items until it is
    #  larger than [bbox all], but do so uniformly.
    #--------------------------------------------------------
    foreach {ax0 ay0 ax1 ay1} [$canvas bbox all] {break}

    while { ($ax0<$x0) || ($ay0<$y0) || ($ax1>$x1) || ($ay1>$y1) } {
        # triple the scalable area size
        set x0 [expr {$x0-$xlength}]
        set x1 [expr {$x1+$xlength}]
        set y0 [expr {$y0-$ylength}]
        set y1 [expr {$y1+$ylength}]
        set xlength [expr {$xlength*3.0}]
        set ylength [expr {$ylength*3.0}]
    }

    #--------------------------------------------------------
    #  Now that we've finally got a region defined with
    #  the proper aspect ratio (of only the scalable items)
    #  but large enough to include all items, we can compute
    #  the xview/yview fractions and set our new viewport
    #  correctly.
    #--------------------------------------------------------
    set newxleft [expr {($xcenter-$x0-($winxlength/2.0))/$xlength}]
    set newytop  [expr {($ycenter-$y0-($winylength/2.0))/$ylength}]
    $canvas configure -scrollregion [list $x0 $y0 $x1 $y1]
    $canvas xview moveto $newxleft
    $canvas yview moveto $newytop

    #--------------------------------------------------------
    #  Change the scroll region one last time, to fit the
    #  items on the canvas.
    #--------------------------------------------------------
    $canvas configure -scrollregion [$canvas bbox all]
}

#
# check inp
#  check validity/satisfaction/enforcement of formula inp
#
# inp: string (fusion logic formula plus type of check)
#
proc check { inp } {
    
    variable a
    variable b  
    variable indep
    global c tab fresh freshx depindex deplist deplistp pdeplist LivenessOffsets LivenessTests
    global INPUT nodisplay current infinite dbg reduce_error stdio tdebug
  
     
    set fresh 0
    set freshx 1
    #set pastdepth -1
    
    array unset depindex
    array unset deplist
    array unset deplistp
    array unset pdeplist
    array unset indep
    
    set LivenessOffsets {}
    set LivenessTests {}
    
    if {$current != ""} {
        set ttt [file normalize $current]
        insert "Loading $ttt" blue
        source $current
    }

    insert "input = $inp" blue
    
    set inp1 ""
    if [catch {set inp1 [subst $inp]} err] {
        insert " "
        insert "error in $inp: $err" red
        insert " "
        return
    } 

    
    if { $dbg == 1 } {
        insert_debug "check: $inp1"
    }
    yy_scan_string $inp1
    set loc [yyparse]
    
    if {$loc != "error"} {
        Init-data-manager
        set tl [lindex $loc 1]
        set check [lindex $loc 0]

        if { $check == 2 || $check == 3 } {
            set infinite 1
        } else {
            set infinite 0
        }

        set original_tl $tl
        
        if { $check == 4 || $check == 5 } {
            set pfound [regexp {flpchop|flsince|flprev} $tl]
            set ffound [regexp {flchop|fluntil|flnext} $tl]
            #puts "pfound=$pfound, ffound=$ffound"  
            if { $pfound == 1 } {
                if { $ffound == 1 } {
                    insert "error: formula contains past and future operators." red
                    return
                } else {
                    set past_rev 1
                    insert "reversing input" blue
                    set tl  [reverse_fl [lindex $tl 0]]
                }
            } else {
                set past_rev 0
            }
        } else {
            set past_rev 0
        }
            
        #puts "$check"
        #puts "results: $tl"
        #set tl [regsub -all {\(} $tl \{ ]
        #set tl [regsub -all {\)} $tl \} ]
        #puts "results: ($tl)"
          
      
        ##set tl1 [reduce [lindex $tl 0]]

        set reduce_error 0
        
        if {$check > 5} {
            #set tlreverse [reverse_pl [lindex $tl 0]]
            #puts "reverse: $tlreverse"
            #no reversing anymore
            # for validity we need to complement the formula first
            if { $check == 7 } {
                set tl "{flnot $tl}"
            }
            set tl1 [reduce_transition 2 [lindex $tl 0]]
            set tlst [reduce_state 2 [lindex $tl 0]]
        } else {
            # for validity we need to complement the formula first
            if { $check == 1 || $check == 3 } {
                set tl "{flnot $tl}"
            }
            set tl1 [reduce_transition 2 [lindex $tl 0]] 
            set tlst [reduce_state 2 [lindex $tl 0]]
        }

        if { $reduce_error == 1} {
            return
        } 

        
        #puts "tl1: $tl1"
        if { $dbg == 1 } {
            insert_debug " tlst           : $tlst"
            insert_debug " LivenessOffsets: $LivenessOffsets "
            insert_debug " LivenessTests  : $LivenessTests "
            insert_debug " deplist        :  "     
        }
        if { $dbg == 1 } {
            insert_debug "future initial state: $tlst" blue
            insert_debug " "
            insert_debug "future dependency:" blue
        }
        
        set zz "true"
        set zl [lsort -dictionary [array names deplist]]
        if { [llength $zl] > 1 } {
            foreach dep  $zl {
                #insert " $dep " blue
                if { $dbg == 1} {
                    insert_debug " $dep "
                }
                set zz "{And $zz $dep}"
            }
        } else {
            if { [llength $zl] == 1 } {
                #insert "[lindex $zl 0]" blue
                set zz [lindex $zl 0]
            } else {
                set zz "true"
            }
        }
        if { $dbg == 1 } {
            insert_debug "zz= $zz"
            #puts "length zl= [llength $zl]" 
        }

        if { $dbg == 1 } {
            insert_debug "deplistp:"
        }
        set zzp "true"
        set zlp [lsort -dictionary [array names deplistp]]
        if { [llength $zlp] > 1 } {
            foreach dep  $zlp {
                if { $dbg == 1} {
                    insert_debug " $dep "
                }
                set zzp "{And $zzp $dep}"
            }
        } else {
            if { [llength $zlp] == 1 } {
                set zzp [lindex $zlp 0]
            } else {
                set zzp "true"
            }
        }
        if { $dbg == 1 } {
            insert_debug "zzp= $zzp"
            #puts "length zlp: [llength $zlp]"
        }

        if { $infinite == 0} {
            set zzp "{Imp {flnext true} $zzp}"
        } 
            

        
        #insert " "
        #insert "past dependency:" blue
        set pzz "true"
        set pzl [lsort -dictionary [array names pdeplist]]
        if { [llength $pzl] > 1 } {
            foreach dep  $pzl {
                #insert " $dep " blue
                set pzz "{And $pzz $dep}"
            }
        } else {
            if { [llength $pzl] == 1 } {
                #insert "[lindex $pzl 0]" blue
                set pzz [lindex $pzl 0]
            } else {
                set pzz "true"
            }
        }
        if { $dbg == 1 } {  
            insert_debug "pzz= $pzz"
        }

        # set ptl1 [prev_sub_next $pzz]
        # puts " new: $ptl1" 
        if { [llength $pzl] == 0 } {
            insert " "
            insert "***No past time operators" blue
            insert " "
            set tl1 $zz
            #set pastdepth 0
            set past_tlst $tlst
            set future_tlst $tlst
            set future_tl $tl1
            set past_tl "true" 
            
        } else {

            insert " "
            insert "***Have past time operators" blue
            insert " "
            set future_tl $zz
            set ptl1 $pzz
            
            set p_tlst  [lindex [prev_sub_false [lindex $ptl1 0] ] 0]
            set future_tlst $tlst

            set Ltt "fluntil true $tlst"
            
            set depindex($Ltt) $freshx
            
            
            set x5 "{Eqv r_$depindex($Ltt) {Or $future_tlst {flnext r_$depindex($Ltt)}}}"
            if { $dbg == 1} {
                insert_debug " "
                insert_debug "past/future dependency: " blue
                insert_debug "$x5" blue
                insert_debug " "
            }
            set deplistp($x5) "r_$depindex($Ltt)"
            

            set past_tl "{And $zzp $x5}"
           
            
            
            set newtl1 "{And $future_tl $past_tl}"
            
            if { $dbg == 1 } {
                insert_debug "p_tlst= $p_tlst"
                insert_debug "future_tlst= $future_tlst" 
                insert_debug " assigning ($Ltt) to $freshx"
                insert_debug "x5=$x5"
                insert_debug "newtl1= $newtl1"
                #puts "length past = [llength $pzl]"
            }

            incr freshx

            if { $dbg == 1 } {
                
                insert_debug " "
                insert_debug "past initial state:" blue
                insert_debug "$p_tlst" blue
                insert_debug " "
            }
            if { $infinite == 1} {
                #set newtl1 "{And $zz $zzp}"
                lappend LivenessOffsets [expr $depindex($Ltt) -1]
                lappend LivenessTests [lindex $tlst 0]
                if { $dbg == 1 } {
                    insert_debug "LivenessOffsets = $LivenessOffsets"
                    insert_debug "LivenessTests = $LivenessTests"
                }
            }

            set past_tlst "{And {$p_tlst} r_$depindex($Ltt)}"
            set newtlst  $past_tlst
            

            
            if { $dbg == 1 } {
                insert_debug "newtlst= $newtlst"
                insert_debug "newtl1 = $newtl1"
            }
            
            set tl1 $newtl1
            set tlst $newtlst

            # past_tlst
            # past_tl
            # future_tlst
            # fulture_tl
            
            #set pastdepth [llength $pzl]
            #set pastdepth -1
            
        }
        
        #set k [reduce_length [lindex $tl1 0] ]
    
        set var_for1_d [flatten [extract_var_dep [lindex $tlst 0]]]
        #set p_var_for1_d [flatten [extract_var_dep [lindex $past_tlst 0]]]
        #set f_var_for1_d [flatten [extract_var_dep [lindex $future_tlst 0]]]
        #puts "f_var_for1_d is $f_var_for1_d"
        
        set var_for1_i [flatten [extract_var_indep [lindex $tlst 0]]]
        #set p_var_for1_i [flatten [extract_var_indep [lindex $past_tlst 0]]]
        #set f_var_for1_i [flatten [extract_var_indep [lindex $future_tlst 0]]]
        #puts "var_for1_i is $var_for1_i"
        
        set var_for2_d [flatten [extract_var_dep [lindex $tl1 0]]]
        #set p_var_for2_d [flatten [extract_var_dep [lindex $past_tl 0]]]
        #set f_var_for2_d [flatten [extract_var_dep [lindex $future_tl 0]]]
        #puts "f_var_for2_d is $f_var_for2_d"
        
        set var_for2_i [flatten [extract_var_indep [lindex $tl1 0]]]
        #set p_var_for2_i [flatten [extract_var_indep [lindex $past_tl 0]]]
        #set f_var_for2_i [flatten [extract_var_indep [lindex $future_tl 0]]]
        #puts "var_for2_i is $var_for2_i"
        
        set var_for_3_d [concat $var_for1_d $var_for2_d]
        set var_for_3_i [concat $var_for1_i $var_for2_i]
        #set var_for_3_d [concat $p_var_for1_d $f_var_for1_d $p_var_for2_d $f_var_for2_d]
        #set p_var_for_3_d [concat $p_var_for1_d $p_var_for2_d]
        #set f_var_for_3_d [concat $f_var_for1_d $f_var_for2_d]
        #set var_for_3_i [concat $p_var_for1_i $f_var_for1_i $p_var_for2_i $f_var_for2_i]
        set var_for_d  [lsort -unique $var_for_3_d]
        #set p_var_for_d  [lsort -unique $p_var_for_3_d]
        #set f_var_for_d  [lsort -unique $f_var_for_3_d]
        #puts "f_var_for_d is $f_var_for_d"

        # set fstate "{false}"
        # foreach fv $f_var_for_d {
        #     set fstate "{Or $fv $fstate}"
        # }
        # puts "fstate is $fstate"
        
        set var_for_i  [lsort -unique $var_for_3_i]
        set dep_var_for  [lsort -dictionary $var_for_d]
        #set p_dep_var_for  [lsort -dictionary $p_var_for_d]
        #set f_dep_var_for  [lsort -dictionary $f_var_for_d]
        set indep_var_for  [lsort -dictionary $var_for_i]

        
    
        set gamma1 [normalise [lindex $tlst 0]]

        if { [is_var $gamma1] == 1 } {
            set gamma1 $gamma1
        } else {
            set gamma1 "{$gamma1}"
        }
        #puts "gamma1 is after $gamma1"
        
        set p_gamma1 [normalise [lindex $past_tlst 0]]

        if { [is_var $p_gamma1] == 1 } {
            set p_gamma1 $p_gamma1
        } else {
            set p_gamma1 "{$p_gamma1}"
        }
        #puts "p_gamma1 is after $p_gamma1"

        set f_gamma1 [normalise [lindex $future_tlst 0]]

        # set f_state [normalise [lindex $fstate 0]]

        # if { [is_var $f_state] == 1 } {
        #     set f_state $f_state
        # } else {
        #     set f_state "{$f_state}"
        # }
        # puts "f_state is $f_state"


                                
        if { [is_var $f_gamma1] == 1 } {
            set f_gamma1 $f_gamma1
        } else {
            set f_gamma1 "{$f_gamma1}"
        }
        #puts "f_gamma1 is after $f_gamma1"

        
        #puts "before: $tl1"
        set gamma2 [normalise [lindex [next_sub [lindex $tl1 0] ] 0] ]
        set gamma2 "{$gamma2}"
        #puts "after: $gamma2" 

        set p_gamma2 [normalise [lindex [next_sub [lindex $past_tl 0] ] 0] ]
        set p_gamma2 "{$p_gamma2}"
        set f_gamma2 [normalise [lindex [next_sub [lindex $future_tl 0] ] 0] ]
        set f_gamma2 "{$f_gamma2}"

        set gamma3 [normalise [lindex [next_sub_false [lindex $tl1 0] ] 0] ]
        set gamma3 "{$gamma3}"

        set p_gamma3 [normalise [lindex [next_sub_false [lindex $past_tl 0] ] 0] ]
        set p_gamma3 "{$p_gamma3}"
        set f_gamma3 [normalise [lindex [next_sub_false [lindex $future_tl 0] ] 0] ]
        set f_gamma3 "{$f_gamma3}"

        insert "dependent variables : $dep_var_for"
     
        insert "independent variables : "
        set zzi [split $indep_var_for " "]
        foreach zzz $zzi {
            if { [string range $zzz 0 5] == "AUTHON" } {
                insert "  $zzz" red
            } else { 
                if { [string range $zzz 0 5] == "AUTHOP" } {
                    insert "  $zzz" green
                } else {
                    insert "  $zzz"
                }
            }
        }
    
        if {$nodisplay==0} {
            #drawtree $tl  "Input"
            # if {$check>5} {
            #     drawtree $tlreverse "Reverse"
            # }
            if { $check == 4 || $check == 5} {
                if { $past_rev == 1 } {
                    drawtree $original_tl "Input"
                    drawtree $tl "Reverse Input"
                } else {
                    drawtree $tl  "Input"
                }
            } else {
                drawtree $tl  "Input"
            }
            drawtree $tl1 "Reduce"
            drawtree $gamma1 "Initial"
            drawtree $f_gamma1 "F Initial"
            drawtree $gamma2 "Transition"
            drawtree $gamma3 "Final"
            #drawtree $p_gamma3 "P Final"
            #drawtree $f_gamma3 "F Final"
            update idletasks
        }

        set zll [lsort -dictionary [array names deplist]]
        set pzll [lsort -dictionary [array names deplistp]]

        #puts "zll: $zll"
        #puts "pzll: $pzll" 
        #set k [llength $dep_var_for]
        set kzl [llength $zll]
        set kpzl [llength $pzll]
        set k [expr $kzl+$kpzl]
        set kfresh [expr $freshx -1]
        #puts "kfresh = $kfresh"
        #puts "k=$k, kzl=$kzl, kpzl=$kpzl"
        
        set i $k
        #puts "k= $k  length dep_var_for= [llength $dep_var_for]"
        
        foreach x $indep_var_for {
            array set indep [list $x $i]
            incr i
        }

        #puts "indep: [array get indep]"

        set suffix_list [concat $dep_var_for $indep_var_for]

        #puts "suffix_list is $suffix_list"

      
        set liveness_length [llength $LivenessTests]
        #puts "liveness_length = $liveness_length, invariant length = $k" 
        if { $infinite == 1 } then {
            if { $check == 2 } {
                set checkinf 0
            } else {
                set checkinf 1
            }    
            set t_init [time {init_my_basic_vars_infinite $suffix_list $liveness_length $k $checkinf} ]
        } else {
            set t_init [time {init_my_basic_vars $suffix_list}]
        }
        insert "init time: $t_init" green
        update idletasks

        set DeltaFunc {}
        foreach ele $LivenessTests {
            set xx [normalise $ele]
          
            if { [is_var $xx] == 1 } {
                set xx $xx
            } else {
                set xx "{$xx}"
            }
            #puts $xx
            lappend DeltaFunc [subst [tree2bdd $xx]]
        }
        #puts " $LivenessTestsa" 
          
        #insert "start gamma1 manipulations"
        #puts "gamma1 is $gamma1"
        set gamma1a [tree2bdd $gamma1]
        set p_gamma1a [tree2bdd $p_gamma1]
        set f_gamma1a [tree2bdd $f_gamma1]
        #set f_state_gamma [tree2bdd $f_state]
        
        #puts "gamma1a: $gamma1a"

        #insert "start gamma2 manipulations"
        #puts "gamma2 is $gamma2"
        set gamma2a [tree2bdd $gamma2]
        set p_gamma2a [tree2bdd $p_gamma2]
        set f_gamma2a [tree2bdd $f_gamma2]
        #puts "gamma2a: $gamma2a"

        #insert "start gamma3 manipulations"
        #puts "gamma3 is $gamma3"
        set gamma3a [tree2bdd $gamma3]
        set p_gamma3a [tree2bdd $p_gamma3]
        set f_gamma3a [tree2bdd $f_gamma3]
        #puts "gamma3a: $gamma3a"

        set InitFunc [subst $gamma1a]
        set p_InitFunc [subst $p_gamma1a]
        set f_InitFunc [subst $f_gamma1a]
        #set f_StateFunc [subst $f_state_gamma]

        set RhoFunc [subst $gamma2a]
        set p_RhoFunc [subst $p_gamma2a]
        set f_RhoFunc [subst $f_gamma2a]
      
        set TauFunc [subst $gamma3a]
        set p_TauFunc [subst $p_gamma3a]
        set f_TauFunc [subst $f_gamma3a]

        # InitFunc = p_InitFunc
        # RhoFunc = [* p_RhoFunc f_RhoFunc]
        # p_RhoFunc Implies f_InitFunc
        # TauFunc = [* p_TauFunc f_TauFunc]
        # p_InitFunc  --p_TauFunc--> p_RhoFunc
        # f_InitFunc  --f_TauFunc--> f_RhoFunc 
      

        #insert "start bdd manipulations"
        update idletasks

        switch  -regexp -- $check {
    
            4 -
            8 {
                set InputTrace {}
                #set InputTrace_next {}
                foreach el $INPUT {
                    #puts "el: $el"
                    set el_a [input_to_trace $el]
                    #set el_b [input_to_trace_next $el]
                    set el_bdd [subst $el_a]
                    #set el_bdd_next [subst $el_b]
                    #puts "el_bdd: $el_bdd"
                    lappend InputTrace $el_bdd
                    #lappend InputTrace_next $el_bdd_next
                    
                }
                #set InputTrace_next [lreplace $InputTrace_next 0 0]
                #lappend InputTrace_next [One]
                #puts "InputTrace: $InputTrace"
                if { $past_rev == 0 } {
                    set t_enforce [time {enforce $suffix_list $k $InitFunc $RhoFunc $TauFunc $InputTrace}]
                } else {
                    set t_enforce [time {p_enforce $suffix_list $k $InitFunc $RhoFunc $TauFunc $InputTrace}]
                }
                insert "enforce time: $t_enforce"  blue
                set t_quit [time {Quit-data-manager}]
                insert "quit time   : $t_quit" blue
                insert " "
                    
            }
            
            5 -
            9 {
                if { $past_rev == 0 } {
                    i_enforce $suffix_list $k $InitFunc $RhoFunc $TauFunc
                } else {
                    p_i_enforce $suffix_list $k $InitFunc $RhoFunc $TauFunc
                }
                set t_quit [time {Quit-data-manager}]
                insert "quit time: $t_quit" blue
                insert " "
            }
            
            0 -
            1 -
            2 -
            3 -
            6 -
            7 {
                
                #puts "at invariant"
                if { $infinite == 1 } {
                    set livelength [llength DeltaFunc]
                    set t [time {invariant_infinite $suffix_list $checkinf $kpzl $kzl $livelength $DeltaFunc $InitFunc $f_InitFunc  $p_RhoFunc $f_RhoFunc $p_TauFunc $f_TauFunc}]
                } else {
                    set t [time {invariant $suffix_list $check $kpzl $kzl $InitFunc $f_InitFunc $p_RhoFunc $f_RhoFunc $p_TauFunc $f_TauFunc}]
                }
                
                insert "invariant time: $t" blue
                
                set t_quit [time {Quit-data-manager}]
                insert "quit time     : $t_quit" blue
                insert " "
            }
            default {
                insert " "
                insert "syntax error" red
                insert " "
            }    
        }
       
    } else {
        insert " "
        insert "error in parsing" red
        insert " "
    }
}

proc show_help {} {
    global help Version

    
    ::bdd_tcl::insert_help "FLCHECK is a decision procedure for Fusion Logic" blue
    ::bdd_tcl::insert_help " "
    ::bdd_tcl::insert_help "Copyright (C) 2009-2025  Antonio Cau, Ben Moszkowski and Helge Janicke"
    ::bdd_tcl::insert_help " "
    ::bdd_tcl::insert_help "This program is free software: you can redistribute it "
    ::bdd_tcl::insert_help "and/or modify it under the terms of the GNU Lesser General"
    ::bdd_tcl::insert_help "Public License as published by the Free Software Foundation,"
    ::bdd_tcl::insert_help "either version 3 of the License, or (at your option) any later"
    ::bdd_tcl::insert_help "version."
    ::bdd_tcl::insert_help " "
    ::bdd_tcl::insert_help "This program is distributed in the hope that it will be useful, "
    ::bdd_tcl::insert_help "but WITHOUT ANY WARRANTY; without even the implied warranty of"
    ::bdd_tcl::insert_help "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the"
    ::bdd_tcl::insert_help "GNU General Public License for more details."
    ::bdd_tcl::insert_help " "
    ::bdd_tcl::insert_help "Contact Antonio Cau (cau.researcher@gmail.com)"
    ::bdd_tcl::insert_help " "
    ::bdd_tcl::insert_help "Version: $Version "
    ::bdd_tcl::insert_help " "
    ::bdd_tcl::insert_help "It is written in Tcl/Tk \[1\] and uses the CUDD bdd library \[2\] of
Fabio Somenzi and the BuDDy library package of Jorn Lind-Nielsen \[3\].
Swig \[4\] is used to write the Tcl/Tk interface to both CUDD and BuDDy. The
first version of the fusion logic decision procedure was written by
Ben Moszkowski using perldDD \[5\] (perl \[6\] interface to CUDD) and clisp\[7\]. 

The formal details of the decision procedure are in \[8,9,10\]. The
enforcement technique is described in \[11\]. "

  ::bdd_tcl::insert_help "\nSyntax of Fusion Logic\n" blue

  ::bdd_tcl::insert_help "variable   

  v ::= \[A-Z\]+\[_A-Z\]*

state formula  

  w ::= v | true | false | w or w | w and w | ( w ) | not w
          | w equiv w | w imp w

future transition formula  
  ft ::= v | true | false | ft or ft | ft and ft | ( ft ) | not ft | next w
           | ft equiv ft | ft imp ft

past transition formula  
  pt ::= v | true | false | pt or pt | pt and pt | ( pt ) | not pt | prev w
          | pt equiv pt | pt imp pt

future fusion expression   
  
  fe ::= chopstar fe | step ft | test w | fe or fe | fe ; fe | ( fe ) |
         w iand fe | w fand fe | init w | fin w

past fusion expression   
  
  pe ::= pchopstar pe | pstep pt | test w | pe or pe | pe ; pe | ( pe ) |
         pe andi w | pe andf w | pinit w | pfin w

fusion logic formula 

  f ::= w | f or f | f and f | not f | ( f ) | f equiv f | f imp f |
        <fe> f | f until f | next f | f <pe> | f since f | prev f 
          

Precedence of operators:
(from lowest to highest)
until
since
or (fusion logic formula)
and (fusion logic formula)
equiv (fusion logic formula)
imp (fusion logic formula)
or (state formula)
and (state formula)
equiv (state formula)
imp (state formula)
or (future/past transition formula)
and (future/past transition formula)
equiv (future/past transition formula)
imp (future/past transition formula)
or (fusion expression)
iand (future fusion expression)
fand (future fusion expression)
andi (past fusion expression)
andf (past fusion expression)
< >  (future/past fusion logic chop)
;    (future/past fusion expression chop)
chopstar
pchopstar
test
step
pstep
next
prev
not

"

  ::bdd_tcl::insert_help "\nUsing TCL syntax to define abbreviations\n" blue

  ::bdd_tcl::insert_help "TCL variables holds as values strings

assignment of string \"not A\" to TCL variable y is as follows

set y \"not A\"

To refer to the value of TCL variable y is as follows

set z \"\$y or B\"

The value of TCL variable z is then

\"not A or B\"

Using TCL syntax to define derived operators

Let f_1 and f_2 be two fusion logic formulae then

`f_1 imp f_2' denotes the derived fusion logic formula

`not ( f_1 ) or f_2'

The following TCL code 

proc imp {a b} {
    return \"(not(\$a) or \$b)\"
}

defines the derived fusion logic operator `imp'

The derived fusion logic operator `imp' is used as follows

set x \"\[imp A B\]\"

The value of the TCL variable x:

\"( not ( A ) or B )\"

"
  ::bdd_tcl::insert_help "The following derived operators are defined
(see fusion_logic_derived.tcl for their definitions) " blue

::bdd_tcl::insert_help "  

\[and f1 f2\]  \[or f1 f2\]  \[imp f1 f2\]  \[equiv f1 f2\]   

\[true\]  \[false\]  \[not f\]  \[l_and flist\]  \[l_or flist\] 

"
  ::bdd_tcl::insert_help "future fusion expressions (n integer, e future fusion expression): " blue

  ::bdd_tcl::insert_help "
 
\[len_e n\]  \[keep_e e\]  \[stable_e e\]

\[len_r_e n\]  \[keep_r_e e\]  \[stable_r_e e\]

\[len_eq_e n\]  \[keep_eq_e n e\]  \[stable_eq_e n e\]  \[stable_ge_e n e\]

\[len_r_eq_e n\]  \[keep_r_eq_e n e\]  \[stable_r_eq_e n e\]  \[stable_r_ge_e n e\]

\[len_le_e n\]  \[len_lt_e n\]  \[len_ge_e n\] \[len_gt_e n\]

\[len_r_le_e n\]  \[len_r_lt_e n\]  \[len_r_ge_e n\] \[len_r_gt_e n\]

\[keep_le_e n e\]  \[keep_lt_e n e\]  \[keep_ge_e n e\]  \[keep_gt_e n e\]

\[keep_r_le_e n e\]  \[keep_r_lt_e n e\]  \[keep_r_ge_e n e\]  \[keep_r_gt_e n e\]

\[true_e\]  \[sometimes_e e\]  \[skip_e\]  \[empty_e\]

\[true_r_e\]  \[sometimes_r_e e\]  \[skip_r_e\]  \[empty_r_e\]

\[star_e e\]  \[star_eq_e n e\]  \[star_le_e n e\]
\[star_lt_e n e\]  \[star_ge_e n e\]  \[star_gt_e n e\]

\[star_r_e e\]  \[star_r_eq_e n e\]  \[star_r_le_e n e\]
\[star_r_lt_e n e\]  \[star_r_ge_e n e\]  \[star_r_gt_e n e\]

"
  ::bdd_tcl::insert_help "past fusion expressions (n an integer, e past expression): " blue

  ::bdd_tcl::insert_help "
 
\[len_l_e n\]  \[keep_l_e e\]  \[stable_l_e e\]

\[len_l_eq_e n\]  \[keep_l_eq_e n e\]  \[stable_l_eq_e n e\]  \[stable_l_ge_e n e\] 

\[len_l_le_e n\]  \[len_l_lt_e n\]  \[len_l_ge_e n\] \[len_l_gt_e n\] 

\[keep_l_le_e n e\]  \[keep_l_lt_e n e\]  \[keep_l_ge_e n e\]  \[keep_l_gt_e n e\] 

\[true_l_e\]  \[sometimes_l_e e\]  \[skip_l_e\]  \[empty_l_e\]

\[star_l_e e\]  \[star_l_eq_e n e\]  \[star_l_le_e n e\]
\[star_l_lt_e n e\]  \[star_l_ge_e n e\]  \[star_l_gt_e n e\]

"
  ::bdd_tcl::insert_help "future fusion formulae
(n is integer, e is future fusion expression, f is fusion formula, w is state formula): " blue

 ::bdd_tcl::insert_help "

\[next_r f\]  \[next f\]  \[next_eq_r n f\]
\[next_ge_r n f\]  \[next_gt_r n f\]   \[next_le_r n f\]  \[next_lt_r n f\] 

\[sometimes_r f\]  \[sometimes_u f\]  \[diamond f\]  \[diamond_r f\]

\[always_r f\]  \[always_u f\]  \[box f\] \[box_r f\] 

\[diamondplus f\] \[diamondplus_r f\]

\[more_r\] \[more\]  \[empty_r\]  \[empty\]

\[box_m f\]  \[box_m_r f\] \[diamond_m f\]  \[diamond_m_r f\]

\[di_fin_r w\]  \[di_e_r e\]  \[bi_fin_r w\]  \[bi_e_r e\]

\[skip_r\]  \[skip\]

\[gets_r f1 f2\]  \[assign f1 f2\] \[assign_r f1 f2\]

\[finite_r \] \[finite_u\]

\[fin_r f\]  \[sfin_r f\]

\[dd_r e f\]  \[bb_r e f\]

\[sometimes_plus_r f\]  \[sometimes_m_r f\]

\[weaknext_r f\]  \[weaknext f\]

\[weaknext_eq_r n f\]  \[weaknext_ge_r n f\]  \[weaknext_gt_r n f\] 

\[weaknext_le_r n f\]  \[weaknext_lt_r n f\] 

\[weakuntil f1 f2\]

\[len_r n\] \[len_eq_r n\] 

\[len_ge_r n\]  \[len_gt_r n\]  \[len_le_r n\]  \[len_lt_r n\] (

\[while_r w e\]  \[keep_r e\]

\[keep_eq_r n e\] \[keep_le_r n e\] \[keep_lt_r n e\] \[keep_ge_r n e\] \[keep_gt_r n e\]

\[keepnow_r e\]  \[stablenow_r f\]

\[sfin_r f\]  \[sfin_u f\]

\[stable_r e\]  \[stable_eq_r n e\]  \[stable_ge_r n e\] 

"

  ::bdd_tcl::insert_help "past fusion logic:
(n is integer, e is past fusion expression, f is fusion formula, w is state formula): " blue

  ::bdd_tcl::insert_help "

\[prev_l f\]  \[prev f\]  \[prev_eq_l n f\]
\[prev_ge_l n f\]  \[prev_gt_l n f\]   \[prev_le_l n f\]  \[prev_lt_l n f\] 

\[sometimes_l f\]  \[sometimes_s f\]  \[diamond_l f\]  \[diamondplus_l f\]

\[always_l f\]  \[always_s f\]  \[box_l f\] 

\[more_l\]  \[empty_l\] \[first\] \[skip_l\] 

\[box_m_l f\]  \[diamond_m_l f\] 

\[di_fin_l w\]  \[di_e_l e\]  \[bi_fin_l w\]  \[bi_e_l e\]

\[gets_l f1 f2\]  \[assign_l f1 f2\]

\[fin_l f\]  \[sfin_l f\]

\[dd_l e f\]  \[bb_l e f\]

\[sometimes_plus_l f\]  \[sometimes_m_l f\]

\[weakprev_l f\]  \[weakprev f\]

\[weakprev_eq_l n f\]  \[weakprev_ge_l n f\]  \[weakprev_gt_l n f\] 

\[weakprev_le_l n f\]  \[weakprev_lt_l n f\] 

\[weaksince f1 f2\]

\[len_l n\] \[len_eq_l n\] 

\[len_ge_l n\]  \[len_gt_l n\]  \[len_le_l n\]  \[len_lt_l n\] (

\[while_l w e\]  \[keep_l e\]

\[keep_eq_l n e\] \[keep_le_l n e\] \[keep_lt_l n e\] \[keep_ge_l n e\] \[keep_gt_l n e\]

\[keepnow_l e\]  \[stablenow_l f\]

\[sfin_s f\]

\[stable_l e\]  \[stable_eq_l n e\]  \[stable_ge_l n e\]


"

  ::bdd_tcl::insert_help "\nChecking Validity and Satisfiability\n" blue

  ::bdd_tcl::insert_help "check fusion logic formula f for validity with finite time

  vld f 

check fusion logic formula f for satisfiability with finite time

  sat f "

 ::bdd_tcl::insert_help "check fusion logic formula f for validity with infinite time

  vld_i f  

check fusion logic formula f for satisfiability with infinite time

  sat_i f  " 

  ::bdd_tcl::insert_help "\nChecking Enforcement\n" blue

  ::bdd_tcl::insert_help "enforce  fusion logic formula f wrt a given input 

Let TCL variable INPUT be the input for which enforcement of f is
checked  

  enf f   

interactive enforce fusion logic formula f

  i_enf f 

You will be prompted for the input trace state by state where `end' will
denote end of the input."

  ::bdd_tcl::insert_help "\nCommand line options\n" blue

  ::bdd_tcl::insert_help "flcheck.tcl without command line options will start the graphical user
interface.  In the FLCHECK tab window of the GUI one can type the
check ones want to perform, for instance, vld (A or B). The output of the
decision procedure is the shown. The history of the checks
performed can be accessed by using the up and down cursor keys.


The following command line options are available:

flcheck.tcl \[-buddy\] \[-dbg\] \[-showhelp\] \[-file <filename> \] \[-stdio\]
            \[ ( -enf | -i_enf ) \"formula\" \]

-buddy:
  use the BuDDY bdd library instead of the CUDD bdd library

-dbg:
  print debugging information

-showhelp:
  show this help on stdout.

-file <filename>:
  use definitions in the file <filename>

-stdio:
  Reads input from stdin and writes output to stdout without GUI.
  For example:
   ./flcheck -file test-flux.tcl -stdio < stdio-test.in 
   where stdio-test is a file with contents:
    sat \[x_0\]
    sat_i \[x_0\]
    sat \[and \[more\] \[len_r 4\]\]
    load \"example-small.tcl\"
    sat \[test_formula\]
    exit
 
   The output of these checks is then shown on stdout.

   Note: one can use, in the input, the command 'exit' to terminate and
   command 'load \"some file\"' to load a file. 


-enf   \"formula\":
  check the enforcement of fusion \"formula\" without GUI (finite time)

-i_enf \"formula\":
  check interactively the enforcement of fusion \"formula\" without GUI (finite time)

"

 
  ::bdd_tcl::insert_help "\nReferences\n" blue

    ::bdd_tcl::insert_help "
\[1\] Tcl Developer Xchange.
    https://www.tcl-lang.org/

\[2\] CUDD library. 
    Fabio Somenzi
    University of Colorado at Boulder
    https://vlsi.colorado.edu/~fabio (dead link)
    archived versions at
    https://github.com/davidkebo/cudd/tree/main/cudd_versions
    

\[3\] BuDDy library package.
    Jorn Lind-Nielsen
    https://sourceforge.net/projects/buddy/

\[4\] Simplified Wrapper and Interface Generator.
    https://www.swig.org/

\[5\] perldDD.
    Fabio Somenzi
    University of Colorado at Boulder.
    https://vlsi.colorado.edu/~fabio (dead link)

\[6\] The Perl language.
    https://www.perl.org/

\[7\] Gnu ANSI Common Lisp implementation. 
    https://clisp.sourceforge.io/

\[8\] Ben Moszkowski. 
    A Hierarchical Analysis of Propositional Temporal Logic based on Intervals.
    We Will Show Them: Essays in Honour of Dov Gabbay, p. 371--440. 
    College Publications (formerly KCL Publications), 2005. 

\[9\] Ben Moszkowski.
    A Hierarchical Completeness Proof for Propositional Interval Temporal
    Logic with Finite Time.
    Journal of Applied Non-Classical Logics, V 14:1--2, p. 55--104, 2004.

\[10\] Antonio Cau, Helge Janicke and Ben Moszkowski.
    A Decision Procedure for Fusion Logic. Seminar talk slides, see
    file itl-fl.pdf.

\[11\] Antonio Cau, Helge Janicke and Ben Moszkowski.
    Verification and Enforcement of Access Control Policies. 
    Formal Methods in System Design, 43.3, (2013).

"

  ::bdd_tcl::help_begin

}

}
# end ::bdd_tcl namespace 

proc get_stdin {} {
    global bddfont
    
    set  enter_text [chan gets stdin]
    set temp [string range $enter_text 0 3]
    #::bdd_tcl::check $enter_text

    switch -exact -- $temp {
        exit {
            puts "End of processing input."
            exit
        }
        load {
            #set temp $enter_text
            
            set tt [string range $enter_text 5 end]
            # removing quotes
            regsub -all {\"|\'|\`} $tt {} tt
            #puts $tt
            #puts "tt=$tt, [pwd]"
            
            ::bdd_tcl::check_file $tt
        }
        default {
            if { $temp != "" } {
                ::bdd_tcl::check $enter_text
            }
        }
    }

    if { [chan eof stdin] } {
        puts "End of processing input."
        exit
    }
    
}


# main --
#
#   ADD COMMENTS HERE
#
# Arguments:
#   args	comments
# Results:
#   Returns ...
#
proc main {argc argv} {
    global cmd point event tab argv0
    global fore back tback oback bddfont out help notebook_widget
    global current curdir  BDD nodisplay dbg reduce_error stdio writing Version


    # GUI paremeters
    #
    set Version "2.0"
    set cmd {}
    set point 0
    set event 1
    #set bdd_tcl::count 0
    set tab 4
    set current ""
    #set curdir [pwd]
    set BDD "CUDD"
    set nodisplay 0
    set dbg 0
    set reduce_error 0
    set stdio 0
    set writing 0
    set check_file 0


    set fore black
    set back #99aecb
    tk_setPalette $back
    set tback white
    set oback white
    if [string equal $::tcl_platform(platform) "windows"] {
        set bddfont        {TkFixedFont}
    } else {
        set bddfont        {TkFixedFont}
    }
    #set bddfont {Courier 14} 
    ttk::style configure . -font $bddfont
    option add *Text.background $tback widgetDefault

    ttk::setTheme strl
    
    #set out .top18.f1.n.f1.06


    #puts "argv = $argv " 
    set arglen [llength $argv]
    set index 0
    set tmp_file ""
    while {$index < $arglen} {
        set arg [lindex $argv $index]
        #puts "checking arg $arg, index = $index"  
        switch -exact -- $arg {
            {-buddy}   {set BDD "Buddy"}
            {-stdio}   {set stdio 1;set nodisplay 1}
            {-dbg}     {set dbg 1}
            {-file}    {set check_file 1; set tmp_file "[lindex $argv [incr index]]"}
            {-i_enf_l} {set nodisplay 1; set input_tcl "i_enf [lindex $argv [incr index]]"}
            {-i_enf_r} {set nodisplay 1; set stdio 0;set input_tcl "i_enf [lindex $argv [incr index]]"}
            {-i_enf}   {set nodisplay 1; set stdio 0;set input_tcl "i_enf [lindex $argv [incr index]]"}
            {-enf_l}   {set nodisplay 1; set stdio 0;set input_tcl "enf [lindex $argv [incr index]]"}
            {-enf_r}   {set nodisplay 1; set stdio 0;set input_tcl "enf [lindex $argv [incr index]]"}
            {-enf}     {set nodisplay 1; set stdio 0;set input_tcl "enf [lindex $argv [incr index]]"}
            {-showhelp} {set nodisplay 1;::bdd_tcl::show_help;exit}
            default  {set nodisplay 0}
        }
        incr index
    }

    



    if [string equal $BDD "CUDD"] {
        # CUDD library
        if {![info exist ::starkit::topdir]} { 
            if [string equal $::tcl_platform(platform) "windows"] { 
                load win64/cudd_tcl.dll
            } else {
                if [string equal $::tcl_platform(os) "SunOS"] {
                    puts "Platform not suported"
                    exit
                } else {
                    if [string equal $::tcl_platform(os) "Linux"] {
                        load linux64/cudd_tcl.so   
                    } else {
                        if [string equal $::tcl_platform(os) "Darwin"] {
                            load macos/cudd_tcl.dylib
                        } else {
                            puts "Platform not suported yet"
                            exit
                        }
                    }
                }
            }
        } else {
            package require cudd_tcl
        }
    } else {
        if [string equal $BDD "Buddy"] {
            #
            #puts "Buddy is not supported anymore" 
            #exit
            # Buddy library
            if {![info exist ::starkit::topdir]} {
                if [string equal $::tcl_platform(platform) "windows"] { 
                    load win64/buddy_tcl.dll
                } else {
                    if [string equal $::tcl_platform(os) "SunOS"] {
                        puts "Platform not suported"
                        exit
                    } else {
                        if [string equal $::tcl_platform(os) "Linux"] {
                            load linux64/buddy_tcl.so
                        } else {
                            if [string equal $::tcl_platform(os) "Darwin"] {
                                load macos/buddy_tcl.dylib
                            } else {
                                puts "Platform not suported yet"
                                exit
                            }
                        }
                    }
                }
            } else {
                package require buddy_tcl
            }
        } else {
            puts "no bdd selected"
            exit
        }
    }

    bdd_tcl::init_bdd 

    if {![info exist ::starkit::topdir]} {
        source fusion_logic_derived.tcl
    } else {
        source $::starkit::topdir/fusion_logic_derived.tcl
    }

    set tempdir [pwd]
    if {[info exists ::starkit::topdir]} {
        set ttt [file dirname $::starkit::topdir]
        cd $ttt
        set curdir $ttt
    } else {
        set ttt [file dirname $argv0]
        cd $ttt
        set curdir $ttt
    }

    
    if { $nodisplay == 0 } { 

        ::bdd_tcl::Window show .
        ::bdd_tcl::Window show .top18
        
        $out tag configure blue -foreground blue
        $out tag configure normal -foreground black
        $out tag configure red  -foreground red
        $out tag configure green  -foreground green

        $help tag configure blue -foreground blue
        $help tag configure normal -foreground black
        $help tag configure red  -foreground red
        $help tag configure green  -foreground green

        ::bdd_tcl::show_help

        ::bdd_tcl::insert "FLCHECK is a decision procedure for Fusion Logic."
        ::bdd_tcl::insert " "
        ::bdd_tcl::insert "Copyright (C) 2009-2025  Antonio Cau, Ben Moszkowski"
        ::bdd_tcl::insert " "
        ::bdd_tcl::insert "This program is free software: you can redistribute it "
        ::bdd_tcl::insert "and/or modify it under the terms of the GNU Lesser General"
        ::bdd_tcl::insert "Public License as published by the Free Software Foundation,"
        ::bdd_tcl::insert "either version 3 of the License, or (at your option) any later"
        ::bdd_tcl::insert "version."
        ::bdd_tcl::insert " "
        ::bdd_tcl::insert "This program is distributed in the hope that it will be useful, "
        ::bdd_tcl::insert "but WITHOUT ANY WARRANTY; without even the implied warranty of"
        ::bdd_tcl::insert "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the"
        ::bdd_tcl::insert "GNU General Public License for more details."
        ::bdd_tcl::insert " "
        ::bdd_tcl::insert "Contact Antonio Cau (cau.researcher@gmail.com)"
        ::bdd_tcl::insert " "
        ::bdd_tcl::insert "Version: $Version "
        ::bdd_tcl::insert " "
        
        if { $check_file == 1 } {
            ::bdd_tcl::check_file $tmp_file
        }
        focus .top18
        
        #  set ttt [file normalize fusion_logic_derived.tcl]
        #  ::bdd_tcl::insert "loading $ttt"

        bind $notebook_widget <ButtonRelease-1> {+
            update idletasks
        }

    } else {
        
        #::bdd_tcl::Window hide .
        #::bdd_tcl::Window hide .top18
        wm withdraw .
        #wm deiconify .
        
        if { $stdio == 0 } {
            if { $check_file == 1 } {
                ::bdd_tcl::check_file $tmp_file
            }
            ::bdd_tcl::check $input_tcl
            exit
        } else {
            if { $check_file == 1 } {
                ::bdd_tcl::check_file $tmp_file
            }
            puts "Begin processing input." 
            chan configure stdin -buffering line -blocking 0
            chan event stdin readable get_stdin
        }
        
        
    }
}

main $argc $argv


