# flcheck --
#
# This file implements package flcheck, a fusion logic modelchecker 
# using the CUDD and BuDDy BDD packages
#
# Copyright (C) 2009-2025  Antonio Cau, Ben Moszkowski and Helge Janicke

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

#
#

::bdd_tcl::load_file policy_library.tcl

#########################################################################
# Role-Based Access Control policy verfication example
#########################################################################
#
# the set of users is { AC, HJ } 

proc users {} {
    return [list  AC HJ ]
}

# the set of roles { USER, ADMIN }

proc roles {} {
    return [list ADMIN USER]
}

# the set of all subjects is the union of users and roles

proc subjects {} {
    return [concat [users] [roles] ]
}

# the set of objects is { RM, SERVER } 
proc objects {} {
    return [list RM SERVER]
}

# the set of actions 
# { rm-activate-admin, rm-activate-user, server-create }

proc actions { object } {
    if { $object == "RM" } { 
        return [actions_activate] 
    } else {
      if {$object == "SERVER"} {
        return "CREATE"
      } else {
         return {}
      }
    }
}

proc actions_activate {} {
    set actions {}
    foreach a [roles] {
      lappend actions "ACTIVATE$a"
    }
     return $actions 
}


proc action_activate { role } {
   return "ACTIVATE$role"
}


# the user AC is assigned the role ADMIN 

proc rule_activate_1 {} {
    global consequences
    set c [autho_plus AC RM [action_activate ADMIN]]
    array set consequences [list $c 0]
    #puts "$c : $consequences($c)"
    return [r_sfb_0 true $c]
}

# the user HJ is assigned the role USER 

proc rule_activate_2 {} {
    global consequences
    set c [autho_plus HJ RM [action_activate USER]]
    array set consequences [list $c 0]
    return [r_sfb_0 true $c]
}

# the user HJ is temporarily promoted to act as ADMIN if user AC is ill

proc rule_activate_3 {} {
    global consequences
    set c [autho_plus HJ RM [action_activate ADMIN]]
    array set consequences [list $c 0]
    return [r_sfb_0 [ill AC] $c]
}

# any user that has called in sick cannot activate any role

proc rule_activate_4 {} {
   global consequences
   set l_rules {}
    foreach user [users] {
      foreach action [actions "RM"] {
          set x [ill $user]
          set c [autho_minus $user "RM" $action]
          array set consequences [list $c 0]
          #puts "$c : $consequences($c)"
          set rule [r_sfb_0 $x $c]
          lappend l_rules $rule
      }
    }
    set z [l_and $l_rules]
    return $z
}

# assign permission < SERVER, CREATE > to the role ADMIN 

proc rule_create_1 {} {
    global consequences
    set c [autho_plus ADMIN SERVER CREATE]
    array set consequences [list $c 0]
    return "[r_sfb_true $c]"
}


#
# Conflict Resolution, Decision Rule:
# As we define our RBAC example as a hybrid policy, e.g. both positive
# and negative authorisations are present in the same policy, we can
# create conflicts.  It is not generally necessary to remove conflicts
# between positive and negative rules, however there must be an
# unambiguous definition of which decision is being taken in case a
# conflict arises.  We capture this in a standard decision rule:

proc rule_conflict_resolution {} {
  set l_rules {}
    foreach subject [subjects] {
     foreach object [objects] {
      foreach action [actions $object] {
          set x1 "[not [autho_minus $subject $object $action]] "
          set x [and [autho_plus $subject $object $action] $x1]
          set y [autho_derived $subject $object $action]
          set rule [r_sfb_0 $y $x]
          #puts "$rule"
          lappend l_rules $rule
      }
     }
    }
    #puts "$l_rules"
    set z [l_and $l_rules]
    #puts "$z"
    return $z
}

#\end{small}
#%\subsubsection{Conflict Analysis and Property Verification}
#For the analysis of the policy, we expand the policy into its normal
#form by expanding the sets and then complete the policy specification
#with a set of default rules of the form $\itlFalse \itlWFollows
#c(s,o,a)$ where $c\in\{\autho^d,\autho^-,\autho^+\},s\in S, o \in O, a
#\in A_o$ and $a(s,o,a)$ does not occur as any consequence. It has been
#shown in \cite{Siewe2005} that the resulting policy is a refinement of
#the original specification. We refer to this specification as the
#model $M$ of our policy.

proc default_rules {} {
   global consequences
    set l_rules {}
    foreach c [array names consequences] {
    #puts "$c : $consequences($c)"
        if { $consequences($c) == 1 } {
            set x [false]
            set rule [r_sfb_0 $x $c]
            lappend l_rules $rule
        }
    }
    if { [llength $l_rules] == 0 } {
      return "true"
    } else {
      set z [l_and $l_rules]
      return $z
    }
}

proc model_1 {} {
    collect_set
    set x [list \
            [rule_activate_1] \
            [rule_activate_2] \
            [rule_activate_3] \
            [rule_activate_4] \
            [rule_create_1] \
          ]
    set y [default_rules]
    return [and [l_and $x] $y]
}

# Determining Conflicts:
# To check whether our policy model contains conflicting rules we
# check the validity of model_1 imp prop_1
#
# test for validity
# not valid
#
# A counter example is generated, for which the predicate
# ill(AC) is true, leading to both the positive authorisation
# autho^+(AC, RM, act(ADMIN)) and the negative authorisation 
# autho^-(AC, RM, act(ADMIN). 

proc property_1 {} {
    foreach subject [subjects] {
     foreach object [objects] {
      foreach action [actions $object] {
          set x "[autho_minus $subject $object $action]"
          set y "[autho_plus $subject $object $action]"
          set rule " not ( [and $x $y] ) "
          lappend l_rules $rule
      }
     }
    }
    set z [l_and $l_rules]
    return [always_r $z]
}

proc test_1 {} { 
    return "[imp [always_r [model_1]] [property_1]]"
}

# It remains to check whether for this conflict
# our decision rule  yields the desired outcome.  In this
# case we are satisfied that the denial indeed takes precedence and that
# AC is not allowed to act in the role ADMIN.
# 
# test for validity
# not valid
 
proc model_2 {} {
    collect_set
    set x [list \
            [rule_activate_1] \
            [rule_activate_2] \
            [rule_activate_3] \
            [rule_activate_4] \
            [rule_create_1] \
            [rule_conflict_resolution] \
          ]
    set y [default_rules]
    return [and [l_and $x] $y]
}

proc test_2 {} { 
    return "[imp [always_r [model_2]] [property_1]]"
} 


# Checking Dynamic Separation of Duty:
# We check whether the user HJ and the user AC can
# possibly act in the role ADMIN at the same time. 
#
# test for validity
# valid
# so indeed HJ and AC cannot possibly assume the role
# ADMIN at the same time. 

proc property_3 {} {
    set x [autho_derived AC RM [action_activate ADMIN]]
    set y [autho_derived HJ RM [action_activate ADMIN]]
    set z [and $x $y]
    set z1 "[not $z ]"
    return "[always_r $z1]"
}

proc test_3 {} { 
    return "[imp [always_r [model_2]] [property_3]]"
}

# Checking Healthiness condition:
# The intent of conditionally promoting user HJ was to ensure
# that there is always a user that can act in the role
# ADMIN.   
#
# test for validity
# not valid
# The counter example generated provides that if both users are ill,
# no user can activate the role ADMIN.

proc property_4 {} {
    set x [autho_derived AC RM [action_activate ADMIN]]
    set y [autho_derived HJ RM [action_activate ADMIN]]
    set z "(($x) or ($y))"
    return "[always_r $z]"
}


proc test_4 {} { 
    return "[imp [always_r [model_2]] [property_4]]"
}

# Checks with additional Assumptions:
# Given assumption that both users are not ill then 
# model_2 satisfies property_4. 
#
# test for validity
# valid

proc not_both_ill {} {
  set x "[and [ill HJ] [ill AC]]"
    set y "[not $x]"
  return "$y"
}

proc test_5 {} { 
    return "[imp [always_r [and [model_2] [not_both_ill]]] [property_4]]"
}

############################################################################
#
# 


proc autho_plus { s o a } {
   set x "_"
   return "AUTHOP$x$s$x$o$x$a"
} 

proc autho_minus { s o a } {
   set x "_"
   return "AUTHON$x$s$x$o$x$a"
} 

proc autho_derived { s o a } {
   set x "_"
   return "AUTHO$x$s$x$o$x$a"
} 

proc ill { user } {
  set y "ILL"
  set x "_"
  return "$y$x$user"
}

proc not_ill { user } {
  set y "ILL"
  set x "_"
    return "[not $y$x$user]"
}


proc collect_set {} {
    global consequences
    array unset consequences
    foreach subject [subjects] {
     foreach object [objects] {
      foreach action [actions $object] {
          array set consequences [list [autho_plus $subject $object $action] 1]
          array set consequences [list [autho_minus $subject $object $action] 1]
      }
     }
    }
}





