# flcheck --
#
# This file implements package flcheck, a fusion logic modelchecker 
# using the CUDD BDD package
#
# Copyright (C) 2009-2025  Antonio Cau, Ben Moszkowski and Helge Janicke

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.



# Initialise the CUDD BDD manager
# enable SIFT automatic dynamic reordering 
# 
# creates variables: manager
proc Init-data-manager {} {
  variable manager [Cudd_Init 0 0 $::bdd_tcl::CUDD_UNIQUE_SLOTS $::bdd_tcl::CUDD_CACHE_SLOTS 0] 
  
  Cudd_AutodynEnable $manager 0
  Cudd_EnableReorderingReporting $manager

  #Cudd_PrintInfoStd $manager
    
}


# Terminate the CUDD BDD manager
#
proc Quit-data-manager {} {
   variable manager
 Cudd_Quit $manager
}

# Debug CUDD BDD manager
# Checks for inconsistencies in the manager heap
#
proc Debug {} {
  variable manager
 Cudd_DebugCheck $manager
}

# Print debug CUDD BDD manager
# Prints to the standard output a BDD and its statistics
#
proc Print {node nvars prlevel} {
  variable manager
  Cudd_PrintDebug $manager $node $nvars $prlevel
}

# Prints out statistics and settings for a CUDD manager
#
proc Info {} {
  variable manager
  Cudd_PrintInfoStd $manager
}

# Reports the number of live nodes in BDDs
#
proc LiveNodes {} {
  variable manager
  Cudd_ReadNodeCount $manager
}

# Reports the peak number of nodes
#
proc PeakNodes {} {
  variable manager
  Cudd_ReadPeakNodeCount $manager
}

# Reports the peak number of live nodes
#
proc PeakLiveNodes {} {
  variable manager
  Cudd_ReadPeakLiveNodeCount $manager
}

# Creates a new BDD variable
#
proc bddVar0 {} {
  variable manager
  set node [Cudd_bddNewVar $manager]
  Cudd_Ref $node
  return $node
}

# Creates a new BDD variable
# adjacent to cnode
proc AdjVar0 {cnode} {
  variable manager
  set ind [Cudd_NodeReadIndex $cnode]
  set lev [Cudd_ReadPerm $manager $ind]
  set node [Cudd_bddNewVarAtLevel $manager $lev]
  Cudd_Ref $node
  return $node
}

# Creates a new BDD variable
# and assigns it to node
proc bddVar {node} {
  variable manager
  upvar 1 $node var
  set var [Cudd_bddNewVar $manager]
  Cudd_Ref $var
}


# One node in BDD manager
#
proc One {} {
  variable manager
  set node [Cudd_ReadOne $manager]
  Cudd_Ref $node
  return $node 
}

# Zero node in BDD manager
#
proc Zero {} {
  variable manager
  return [! [One]]
}

# Computes the NAND of two BDDs a and b
#
proc Nand {a b} {
  variable manager
  set node [Cudd_bddNand $manager $a $b]
  Cudd_Ref $node
  return $node 
}

# Computes the AND of two BDDs a and b
#
proc And {a b} {
  variable manager
  set node [Cudd_bddAnd $manager $a $b]
  Cudd_Ref $node
  return $node
}

# Computes the OR of two BDDs a and b
#
proc Or {a b} {
  variable manager
  set node [Cudd_bddOr $manager $a $b]
  Cudd_Ref $node
  return $node 
}

# Computes the XOR of two BDDs a and b
#
proc Xor {a b} {
  variable manager
  set node [Cudd_bddXor $manager $a $b]
  Cudd_Ref $node
  return $node 
}

# Computes the XNOR of two BDDs a and b
#
proc Xnor {a b} {
  variable manager 
  set node [Cudd_bddXnor $manager $a $b]
  Cudd_Ref $node
  return $node  
}

# Computes the NOR of two BDDs a and b
#
proc Nor {a b} {
  variable manager 
  set node [Cudd_bddNor $manager $a $b]
  Cudd_Ref $node
  return $node  
}

# Computes the NOT of BDDs a 
#
proc Not {a} {
  variable manager
  set node [Cudd_Not $a]
  Cudd_Ref $node
  return $node
}

# Returns 1 if BDD a is less than or equal to b
#
proc Leq {a b} {
   variable manager
    return [Cudd_bddLeq $manager $a $b]
}

# Computes the IMP of two BDDs a and b
#
proc Imp {a b} {
    variable manager
    return [Or [Not $a] $b]
}

# Computes the EQV of two BDDs a and b
#
proc Eqv {a b} {
    return [Xnor $a $b] 
}

#Existentially abstracts all the variables in cube from f
#
proc Exists {cube f} {
   variable manager
   set node [Cudd_bddExistAbstract $manager $f $cube]
   Cudd_Ref $node
   return $node  
}

# Takes the AND of two BDDs and simultaneously abstracts the variables in cube
#
proc AndExists {cube f g} {
   variable manager 
   set node [Cudd_bddAndAbstract $manager $f $g $cube]
   Cudd_Ref $node
   return $node
}

#
proc Shuffle { varlist } {
    variable manager

    #set permutation {}

    set Cuddperm [new_int_array [llength $varlist]]

    set i 0
    
    foreach ele $varlist {

        #puts " ww: $ele "
        set ind [Cudd_NodeReadIndex $ele]

        #puts $ind

        array_set_int $Cuddperm $i $ind

        #lappend permutation $ind

        incr i
    }

    #puts "permutation $permutation "

    if { [llength $varlist] > 0 } {
        Cudd_ShuffleHeap $manager $Cuddperm
    }

    delete_int_array $Cuddperm

}

# Swaps two sets of variables of the same size (a and b) in the BDD f
#
proc SwapVariables {f a b} {
   variable manager
   
    if {[llength $a] != [llength $b]} {
     insert "SwapVariables: given variable lists of different size" red
    }
    set len [llength $a]
    set tempa [new_node_array $len]
    set tempb [new_node_array $len]
    for {set i 0} {$i<$len} {incr i} {
        array_set_node $tempa $i [lindex $a $i]
        array_set_node $tempb $i [lindex $b $i]
    }
    set node [Cudd_bddSwapVariables $manager $f $tempa $tempb $len]
    delete_node_array $tempa
    delete_node_array $tempb
    Cudd_Ref $node
    return $node
}

# Computes the AND of a list of BDDs
#
proc RAnd {name alist} {
    upvar $name x
    if {![info exist x]} {
      set x [One]
    }
    foreach e $alist {
        set x [And $x $e]
    }
}

# Computes the OR of a list of BDDs
#
proc ROr {name alist} {
    upvar $name x
    if {![info exist x]} {
      set x [Zero]
    }
    foreach e $alist {
        set x [Or $x $e]
    }
}

# defines * to be an abbreviation for ::bdd_tcl::And 
interp alias {} * {} ::bdd_tcl::And
# defines + to be an abbreviation for ::bdd_tcl::Or 
interp alias {} + {} ::bdd_tcl::Or
# defines ! to be an abbreviation for ::bdd_tcl::Not
interp alias {} ! {} ::bdd_tcl::Not
# defines ^ to be an abbreviation for ::bdd_tcl::Xor
interp alias {} ^ {} ::bdd_tcl::Xor
# defines Eqv to be an abbreviation for ::bdd_tcl::Eqv
interp alias {} Eqv {} ::bdd_tcl::Eqv

# BDD routines
#

# Initialise the BDD of normal and primed  (in)dependent variables  
# for infinite time
proc init_my_basic_vars_infinite {suffix_name_list liveness_length invariant_length negate_init_formula} {
    
    variable a {}
    variable b {}
    variable manager
    variable my_names
    variable a_cube [One]
    variable b_cube [One]
    variable a_cube_without_independent_vars [One]
    variable num_vars
    variable total_num_vars
    variable num_of_independent_vars
    variable COUNTER
   
    
    set num_vars [llength $suffix_name_list]
    set total_num_vars  [expr 2*$num_vars + $liveness_length]
    
    set num_of_independent_vars [expr $num_vars - $invariant_length]
    
   insert "number of variables $num_vars"
   insert "number of independent variables $num_of_independent_vars" 
   insert "total number of variables  $total_num_vars"
    
    if { $negate_init_formula == 1} {
        set COUNTER  "counter-model"
    } else {
        set COUNTER "model"
    }
    
    
    set my_names {}
    for {set j 0} {$j< $total_num_vars} {incr j} {
        
        #set tmp [bddVar0]
        lappend a [bddVar0]
        lappend b [bddVar0]
        if { $j < $num_vars } {
            set suffix_name [lindex $suffix_name_list $j]
        } else {
            if { $j < 2*$num_vars } {
                set tt [lindex $suffix_name_list [expr $j-$num_vars]]
                set suffix_name "stable$tt"
            } else {
                set tt [expr $j - 2*$num_vars]
                set suffix_name "live$tt"
            }
        }
        
        #[AdjVar0 $tmp]
        #insert "j: $j"
        lappend my_names "a_$suffix_name"
        lappend my_names "b_$suffix_name"
        
        set a_cube  [* $a_cube [lindex $a $j]]
        set b_cube  [* $b_cube [lindex $b $j]]
        if { $j < $invariant_length || $j >= $num_vars } {
            set a_cube_without_independent_vars [* $a_cube_without_independent_vars [lindex $a $j]]
        }
    }
     # foreach ele $a {
     #     puts [Cudd_NodeReadIndex $ele]
     # }
     # foreach ele $b {
     #     puts [Cudd_NodeReadIndex $ele]
     # }
    #puts "$my_names"
    #insert "finished init variables"
}

# Initialise the BDD of normal and primed  (in)dependent variables  
# for finite time
proc init_my_basic_vars {suffix_name_list} {
   variable a {}
   variable b {}
   variable manager
   variable my_names
   variable a_cube [One]
   variable b_cube [One]
      
    set num_vars [llength $suffix_name_list]
    insert "number of variables $num_vars"
    set my_names {}
    for {set j 0} {$j< $num_vars} {incr j} {
        set suffix_name [lindex $suffix_name_list $j]
        #set tmp [bddVar0]
        lappend a [bddVar0]
        lappend b [bddVar0]
        #[AdjVar0 $tmp]
        #insert "j: $j"
        lappend my_names "a_$suffix_name"
        lappend my_names "b_$suffix_name"
        
        set a_cube  [* $a_cube [lindex $a $j]]
        set b_cube  [* $b_cube [lindex $b $j]]
    }
    #insert "finished init variables"
}

#
#
proc init_my_liveness_vars { LivenessTests num_vars total_num_vars} {

    variable a 
    variable b 
    variable a_cube 
    variable b_cube 
    variable manager
    variable my_names
    global LivenessOffsets
    variable a_halves_are_equal_BDD
    variable a_right_equals_b_right_BDD
    variable initialise_tests_BDD
    variable update_tests_BDD
    variable achieve_all_tests_BDD
    

    #LivenessOffsetsFn
    #LivenessTestsFn

    shuffle_variables $num_vars
    
    set initialise_tests_BDD [One]
    set update_tests_BDD [One]
    set achieve_all_tests_BDD [One]
    
    #puts "$num_vars $LivenessOffsets" 
    set k 0
    #if { [llength $LivenessOffsets] == 0 } {
        #puts "no liveness tests"
        #puts " here init_liveness: [expr 2*$num_vars], $total_num_vars" 
    #}
    #if { [llength $LivenessOffsets] > 0 } then {
    for {set j [expr 2*$num_vars]} { $j< $total_num_vars } {incr j} {
        #puts "here: [lindex $LivenessOffsets $k]" 
        set initialise_tests_BDD [* $initialise_tests_BDD [Eqv [lindex $a $j] [! [lindex $a [lindex $LivenessOffsets $k]]]]]
        set update_tests_BDD [* $update_tests_BDD [Eqv [lindex $b $j] [+ [lindex $a $j]  [lindex $LivenessTests $k]]]]
        
        set achieve_all_tests_BDD [* $achieve_all_tests_BDD [Imp [lindex $a [lindex $LivenessOffsets $k]] [lindex $a $j]]]
        incr k
    }
    #}
    
    set a_halves_are_equal_BDD [One]
    set a_right_equals_b_right_BDD [One]

    insert "***Starting hard part of init_my_liveness_vars."

    for {set j 0} {$j< $num_vars} {incr j} {
        set jj [expr $j+$num_vars]
        set a_halves_are_equal_BDD [* $a_halves_are_equal_BDD [Eqv [lindex $a $j] [lindex $a $jj]]]
        set a_right_equals_b_right_BDD [* $a_right_equals_b_right_BDD [Eqv [lindex $b $jj] [lindex $a $jj]]]
    }
    insert "***Ended hard part of init_my_liveness_vars."

    #if { $output_graphs >= 2 } {
        #insert "***outputing \$initialise_tests_BDD, etc."
        #insert "***my_names=$my_names,"
        #my_dot([$a_cube], ["\$a_cube"]);
	#my_dot([$b_cube], ["\$b_cube"]);
	#my_dot([$initialise_tests_BDD], ["\$initialise_tests_BDD"]);
	#my_dot([$update_tests_BDD], ["\$update_tests_BDD"]);
	#my_dot([$achieve_all_tests_BDD], ["\$achieve_all_tests_BDD"]);
	#my_dot([$a_right_equals_b_right_BDD], ["\$a_right_equals_b_right_BDD"]);
    #}
    
}

#
# Here we shuffle the order of variables to speed things up.
proc shuffle_variables { num_vars } {
    variable a 
    variable b 
    variable a_cube 
    variable b_cube 
    variable manager
    variable my_names
    global LivenessOffsets

    #puts "shuffle $num_vars" 

    #puts "length a = [llength $a]" 
    set ShuffleList {}
    set LivenessHash [dict create]

    for {set j 0} { $j < [llength $LivenessOffsets] } { incr j } {
        dict set LivenessHash [lindex $LivenessOffsets $j] $j
    }
    #puts " LiveHash= $LivenessHash " 
    for {set j 0 } { $j< $num_vars } { incr j } {
        
        lappend ShuffleList [lindex $a $j]
        lappend ShuffleList [lindex $b $j]
        lappend ShuffleList [lindex $b [expr $j+$num_vars]]
        lappend ShuffleList [lindex $a [expr $j+$num_vars]]
        #puts " loop I: j=$j, [expr $j+$num_vars] "
        
        if { [dict exists $LivenessHash $j] } {
            
            #puts "loop II: [expr 2*$num_vars + [dict get $LivenessHash $j]] "
            
            #puts " silly a [lindex $a [expr 2*$num_vars + [dict get $LivenessHash $j]]] "
            #puts " silly b [lindex $b [expr 2*$num_vars + [dict get $LivenessHash $j]]] "
            if { [lindex $a [expr 2*$num_vars + [dict get $LivenessHash $j]]] != {} } {
                lappend ShuffleList [lindex $a [expr 2*$num_vars + [dict get $LivenessHash $j]]]
            }
            if { [lindex $b [expr 2*$num_vars + [dict get $LivenessHash $j]]] != {} } {
                lappend ShuffleList [lindex $b [expr 2*$num_vars + [dict get $LivenessHash $j]]]
            }
        }
    }
    insert "*** ShuffleList= [llength $ShuffleList]"
    
    Shuffle $ShuffleList
    
}

#
#
proc analyse_initial_segment_model { Init f_Init negate_init_formula p_Rho f_Rho p_Tau} {

    variable a 
    variable b 
    variable a_cube 
    variable b_cube 
    variable manager
    variable my_names
    variable COUNTER
    variable PeriodicState
    variable IndepVarsInPeriodicState
    variable max_iterations
    variable SuccWList
    variable SuccW
    global pj dbg
    
    insert "***Now determining initial segment of $COUNTER."

    set Rho [* $p_Rho $f_Rho]
    
    set SuccW $Init

    if { $negate_init_formula == 1} {
        insert "***Testing for validity with infinite time"
        #set SuccW [Not $SuccW]
    } else {
        insert "***Testing for satisfiability with infinite time"
    }

    set SuccWStar $SuccW
    set Convergence 0
    set SuccWList {}
    set pj 0
    set j 0

    # set p_Convergence 0
    
    # while { ([* $SuccW [* $p_Tau $f_Init]] == [Zero]) && ($j < $max_iterations) } {
    #     insert "***Starting past iteration $j."
    #     check_node_stats_infinite
    #     lappend SuccWList $SuccW
    #     set SuccW [SwapVariables [AndExists $a_cube $SuccW $p_Rho] $a $b]
    #     set OldSuccWStar $SuccWStar
    #     set SuccWStar [+ $SuccWStar $SuccW]
    #     if { $OldSuccWStar == $SuccWStar } {
    #         set p_Convergence 1 
    #     } else {
    #       set p_Convergence 0
    #     }

    #     incr j
    #     # 
    #     if { ([* $SuccW [* $p_Tau $f_Init]] == [One]) } {
    #         if { $past_j == 0 } {
    #             set past_j $j
    #         }
    #     }
    #     if { $p_Convergence } { break }

    # }

    # if { $dbg == 1 } {
    #     puts "past j= $j, p_Convergence=$p_Convergence"
    # }
    
    # set past_j $j
    # set f_Convergence 0

    #set p_SuccW $SuccW

    while { ([* $SuccW [* $PeriodicState $IndepVarsInPeriodicState]] == [Zero]) && ($j < $max_iterations) } {
        insert "***Starting iteration $j."
        check_node_stats_infinite

        #counting number of past iterations
        # if { [* $p_SuccW  $p_Tau] == [Zero] } {
        #     #puts "p_tau : zero"
        #     incr pj
        # } else {
        #     #puts "p_tau : one"
        #     #set pj $j
        # }
        
        lappend SuccWList $SuccW
        #set p_SuccW [SwapVariables [AndExists $a_cube $SuccW $p_Rho] $a $b]
        set SuccW [SwapVariables [AndExists $a_cube $SuccW $Rho] $a $b]
        set OldSuccWStar $SuccWStar
        set SuccWStar [+ $SuccWStar $SuccW]
        if { $OldSuccWStar == $SuccWStar } {
            set Convergence 1 
        } else {
          set Convergence 0
        }

        incr j

        if { $Convergence } { break }

    }

    if { $dbg == 1} {
        insert_debug "pj= $pj, future j = $j, Convergence=$Convergence"
    }
    
    if {$j==1} {
       set iterations "iteration"
    } else {
       set iterations "iterations"
    }

    insert "$j $iterations performed."

    if { [* $SuccW [* $PeriodicState $IndepVarsInPeriodicState] ] == [Zero] } {
        if {  $Convergence == 0 } {
            insert "***No convergence after $max_iterations iterations!"
        } else {
            if { $negate_init_formula == 0} {
                insert " "
                insert  "Unsatisfiable with infinite time." red
                insert " "
            } else {
                insert " "
                insert  "Valid with infinite time." green
                insert " "
            }
        }
    } else {
        if { $negate_init_formula == 0} {
            insert " "
            insert  "Satisfiable with infinite time." green
            insert " "
        } else {
            insert " "
            insert "Not valid with infinite time." red
            insert " "
        }
    }

    return $j
}

#
#
proc print_initial_segment_model { suffix_name_list invariant_length j Rho f_Init} {

   variable a 
   variable b 
   variable a_cube 
   variable b_cube 
   variable manager
   variable my_names
   variable COUNTER
   variable SuccWList
   variable SuccW
   variable PeriodicState
   variable IndepVarsInPeriodicState
   variable PeriodicStateStringList
   variable InitialStateStringList
   global pj dbg

   #puts "past_j = $past_j" 

   insert "***Now start to find a $COUNTER of initial segment."
   insert "SuccWList = [expr [llength $SuccWList] ]"

   set InitialStateStringList {}
   if { [llength $suffix_name_list] == $invariant_length } {
       set number_of_states [expr [llength $SuccWList] + 1]
       #set p_number_of_states $pj
       
       insert "No independent variables."
       
       set SuccW [* $SuccW [* $PeriodicState $IndepVarsInPeriodicState]]
       while { $j >= 0 } {
           
           set j1 [expr $j+1]
           #insert "j = $j"

           if { $dbg == 1 } {
               set k 0
           } else {
               set k $invariant_length
           }
           #set k $invariant_length

           if { [* $SuccW $f_Init] == [Zero] } {
               if { $dbg == 1 } {
                   insert_debug "j=$j, zero"
               }
           } else {
               set pj $j
               if { $dbg == 1 } {
                   insert_debug "j=$j, one"
               }
           }
           
           if {$j !=0} {
               set TempBDD [lindex $SuccWList end]
               #remove last element from SuccWlist
               #puts "here print: [llength $SuccWList]" 
               set SuccWList [lreplace $SuccWList end end]
               set SuccW [AndExists $b_cube $TempBDD  [* $Rho [SwapVariables $SuccW $a $b]]]
           }

           incr j -1
       }

            
       
       if {$number_of_states == 1 } { 
                 set states "state"  
       } else {
                 set states "states"
       }
       if {$pj == 1} {
           set pstates "state"
       } else {
           set pstates "states"
       }
       if {$pj > 0} {
           set ptext " with $pj past $pstates."
       } else {
           set ptext "."
       }
       insert "Can use any interval with exactly $number_of_states $states as a $COUNTER of initial segment$ptext"
   } else {
       set SuccW [* $SuccW [* $PeriodicState $IndepVarsInPeriodicState]]
       while { $j >= 0 } {
           
           set j1 [expr $j+1]
           #insert "j = $j"
           
           
           set StateString ""
           if { $dbg == 1 } {
               set k 0
           } else {
               set k $invariant_length
           }
           #set k $invariant_length
           
           while { $k < [llength $suffix_name_list] } {
               
               append StateString "[lindex $suffix_name_list $k]="
               if { [* $SuccW [lindex $a $k] ] == [Zero] } {
                      append StateString "0 "
               } else {
                   set SuccW [* $SuccW [lindex $a $k]]
                   append StateString "1 "
               }

               incr k
           }
           if { [* $SuccW $f_Init] == [Zero] } {
               if { $dbg == 1 } {
                   insert_debug "j=$j, zero"
               }
           } else {
               set pj $j
               if { $dbg == 1 } {
                   insert_debug "j=$j, one"
               }
           }
          
           if {$j !=0} {
               set TempBDD [lindex $SuccWList end]
               #remove last element from SuccWlist
               #puts "here print: [llength $SuccWList]" 
               set SuccWList [lreplace $SuccWList end end]
               set SuccW [AndExists $b_cube $TempBDD  [* $Rho [SwapVariables $SuccW $a $b]]]
           }
           lappend InitialStateStringList $StateString
           incr j -1
       }
       
       set number_of_states [llength $InitialStateStringList]
       set temp [expr max([llength $PeriodicStateStringList],[llength $InitialStateStringList])]
       set state_width_string "$temp"
       set state_width [string length $state_width_string]
       if {$number_of_states == 1 } { 
                 set states "state"  
       } else {
           set states "states"
       }
       if {$pj == 1} {
           set pstates "state"
       } else {
           set pstates "states"
       }
       # 
       if {$pj > 0} {
           set ptext " with $pj past $pstates:"
       } else {
           set ptext ":"
       }
       insert "Here is a $COUNTER of an initial segment with $number_of_states $states$ptext"
       for {set j [expr [llength $InitialStateStringList] -1]} {$j>=0} {incr j -1} {
           set k [expr [llength $InitialStateStringList]-$j-1 ]
           set z [lindex $InitialStateStringList $j]
           set z1 [split $z " "]
           set z2 [lsort -dictionary $z1]
           # 
           if {$k == $pj  } {
               #insert "   State $k: "
               insert "   State $k (current): "
           } else {
               insert "   State $k: "
           }
           foreach z3 $z2 {
               insert "  $z3"
           }
           insert " "
       }
   }
    
}

#
#
proc compute_periodic_segment_model { suffix_name_list invariant_length j} {

   variable a 
   variable b 
   variable a_cube 
   variable b_cube 
   variable manager
   variable my_names
   variable COUNTER
   variable SuccInfList
   variable SuccInf
   variable a_halves_are_equal_BDD
   variable RhoInf
   variable PeriodicState
   variable IndepVarsInPeriodicState
   variable PeriodicStateStringList
   variable periodic_number_of_states
   global dbg
   

    set first_time 1
    set IndepVarsInPeriodicState [One]

    insert "***Now start to find a $COUNTER of periodic segment."
    #insert "SuccInfList= $SuccInfList"

    set PeriodicStateStringList {}

   if {[llength $suffix_name_list] == $invariant_length  } {
       set number_of_states [expr [llength $SuccInfList]+1]
       set periodic_number_of_states $number_of_states
        insert "No independent variables."
        if {$number_of_states == 1 } { 
                 set states "state"  
        } else {
            set states "states"
        }
        insert "Can use any interval with exactly $number_of_states $states as a $COUNTER of periodic segment."
    } else {
        insert "***Initially looking at states in reverse order: "

        set SuccInf [* $SuccInf $a_halves_are_equal_BDD]
        
        while { $j >= 0 } {

            set j1 [expr $j+1]

            insert "j = $j"

            set StateString ""

            if { $dbg == 1 } {
                set k 0
            } else {
                set k $invariant_length
            }
            #set k $invariant_length

            while { $k < [llength $suffix_name_list] } {
                append StateString "[lindex $suffix_name_list $k]="
                if { [* $SuccInf [lindex $a $k] ] == [Zero] } {
                    append StateString "0 "
                    if { $first_time == 1 } {
                        set IndepVarsInPeriodicState [* $IndepVarsInPeriodicState [! [lindex $a $k]]]
                    }
                } else {
                    set SuccInf [* $SuccInf [lindex $a $k]]
                    append StateString "1 "
                    if { $first_time == 1 } {
                        set IndepVarsInPeriodicState [* $IndepVarsInPeriodicState [lindex $a $k]]
                    }
                }

                incr k
            }
            if { $j != 0 } {
                set TempBDD [lindex $SuccInfList end]
                #remove last element from SuccWlist
                set SuccInfList [lreplace $SuccInfList end end]
                set SuccInf [AndExists $b_cube $TempBDD  [* $RhoInf [SwapVariables $SuccInf $a $b]]]
            }
            
            lappend PeriodicStateStringList $StateString

            set first_time 0

            incr j -1
        }
    }                 
        
}

#
#
proc  print_periodic_segment_model { suffix_name_list invariant_length} {

   variable a 
   variable b 
   variable a_cube 
   variable b_cube 
   variable manager
   variable my_names
   variable PeriodicStateStringList
   variable InitialStateStringList
   variable SuccInfList
   variable COUNTER
   variable SuccWList
   variable periodic_number_of_states
   global pj dbg

   set number_of_states [llength $PeriodicStateStringList]

   set temp [expr max([llength $PeriodicStateStringList], [llength $InitialStateStringList])]

   set state_width_string "$temp"
   set state_width [string length $state_width_string]

   if { [llength $suffix_name_list] == $invariant_length } {
       #puts "xx= [expr [llength $SuccInfList]+1] yy=[expr [llength $SuccWList]+1]"
       
       #set number_of_states [expr [llength $SuccWList]+2-$past_j]
       set number_of_states $periodic_number_of_states
       insert "No independent variables."
       if {$number_of_states == 1 } {
           set states "state"  
       } else {
          set states "states"
       }
       insert "Can use any interval with exactly $number_of_states $states as a $COUNTER of an (overlapping) periodic segment."
   } else {
       if {$number_of_states == 1 } { 
                 set states "state"  
       } else {
           set states "states"
       }
       insert "Here is a $COUNTER of an overlapping periodic segment with $number_of_states $states:"
       for {set j [expr [llength $PeriodicStateStringList] -1]} { $j>=0 } {incr j -1} {
           set k [expr [llength $PeriodicStateStringList] -$j]
           set z [lindex $PeriodicStateStringList $j]
           set z1 [split $z " "]
           set z2 [lsort -dictionary $z1]
           foreach z3 $z2 {
               insert "  $z3"
           }
           insert " "
       }
   }
    
}

# show the current number of live nodes 
#
proc check_node_stats_infinite {} { 
    variable manager
    variable max_LiveNodes
  
    insert "Cudd current number of live nodes=[LiveNodes]" blue

    if { [LiveNodes] > $max_LiveNodes } {
        insert "***Current number of live nodes [LiveNodes] exceeds limit present $max_LiveNodes" red
    }
}




# show the current number of live nodes 
#
proc check_node_stats {} { 
  variable manager
  
  insert "Cudd current number of live nodes=[LiveNodes]" blue
}

# give a dot-based representation of the BDD in tab
#
proc dot {onodes iname oname} {
  variable manager
  global c tab bddfont nodisplay
  if {$nodisplay==0 } {
    set leno [llength $onodes]
    set tempo [new_node_array $leno]
    for {set i 0} {$i<$leno} {incr i} {
        array_set_node $tempo $i [lindex $onodes $i]
    }
    set tempin [new_name_array [llength $iname]]
    for {set i 0} {$i<[llength $iname]} {incr i} {
        array_set_name $tempin $i [lindex $iname $i]
    }
    set tempout [new_name_array [llength $oname]]
    for {set i 0} {$i<[llength $oname]} {incr i} {
        array_set_name $tempout $i [lindex $oname $i]
    }
    
    new_tab .top18 $tab
    incr tab
    #puts "tab; $tab"
    Cudd_DumpDotty $manager $leno $tempo $tempin $tempout
    insert "start generating dot graph" blue
    if [string equal $::tcl_platform(platform) "windows"] { 
      set f [exec "c:/Program Files/Graphviz2.20/bin/dot.exe" -Ttk -Nstyle=filled < "./dotty.dot" > dotty.tcl]
    } else {
      set f [exec dot -Ttk -Nstyle=filled < "./dotty.dot" > dotty.tcl]
    }
    $c delete node edge
    source dotty.tcl   
    $c itemconfigure label -font $bddfont
    $c itemconfigure edge -width 2
    zoom $c 0.8
    insert "end generating dot graph" blue
    delete_node_array $tempo
    delete_name_array $tempi
    delete_name_array $tempout  
   }
}

#
# invariant_infinite
# check satifiability/validity of an infinite fusion logic formula in 
# terms of bdd operations 
#
proc  invariant_infinite {suffix_name_list negate_init_formula p_invariant_length f_invariant_length liveness_length Delta Init f_Init p_Rho f_Rho p_Tau f_Tau} {
    variable a 
    variable b 
    variable a_cube 
    variable b_cube 
    variable manager
    variable my_names
    variable a_cube_without_independent_vars
    
    variable initialise_tests_BDD
    variable update_tests_BDD
    variable achieve_all_tests_BDD
    variable a_halves_are_equal_BDD
    variable a_right_equals_b_right_BDD
    
    global LivenessOffsets
    variable num_vars
    variable total_num_vars
    variable max_LiveNodes
    variable COUNTER
    variable SuccInfList
    variable SuccInf
    variable RhoInf
    variable PeriodicState
    variable IndepVarsInPeriodicState
    variable max_iterations
    variable SuccWList
    variable SuccW
    variable PeriodicStateStringList
    variable InitialStateStringList

    set Rho [* $p_Rho $f_Rho]
    set Tau [* $p_Tau $f_Tau]
    set invariant_length [expr $p_invariant_length+$f_invariant_length]
    
    set max_iterations 50
    set max_LiveNodes 1500000

    #puts "$num_vars $total_num_vars $LivenessOffsets $suffix_name_list" 
    #set num_vars [llength $suffix_name_list]
    #set tt [expr $num_vars-$invariant_length]
    
    init_my_liveness_vars $Delta $num_vars $total_num_vars

    set SuccW $Init
    
    if {$negate_init_formula == 0} {
        insert "***Testing for satisfiability with infinite time."
    } else {
        insert "***Testing for validity with infinite time."
        #set SuccW [Not $SuccW]
    }

    set SuccWStar $SuccW
    set OldSuccWStar [Not $SuccWStar]
    set Convergence 0

    set j 0

    
    while { ($j == 0 || $SuccW != [Zero]) && ($j < $max_iterations) } {
        insert "***Starting iteration $j."
        check_node_stats_infinite

        set SuccW [SwapVariables [AndExists $a_cube $SuccW $Rho] $a $b]
        set OldSuccWStar $SuccWStar
        set SuccWStar [+ $SuccWStar $SuccW]

        if { $OldSuccWStar == $SuccWStar } {
            set Convergence 1 
        } else {
          set Convergence 0
        }
        
        incr j

        if { $Convergence } { break }

    }

    

    set max_number_of_states $j

    if {$j==1} {
       set iterations "iteration"
    } else {
       set iterations "iterations"
    }

    insert "$j $iterations performed."

    if {  $Convergence == 0 } {
        insert "***No convergence after $max_iterations iterations!" red
        return 
    }

    if { $SuccW == [Zero] } {
        insert "***All $COUNTER models are finite!"
        if { $negate_init_formula == 0} {
            insert " "
            insert  "Unsatisfiable with infinite time." red
            insert " "
        } else {
            insert " "
            insert "Valid with infinite time." green
            insert " "
        }
        return 
    }

    insert "***Need to analyse more."

    set SuccWStarAndMore [SwapVariables [AndExists $b_cube $SuccWStar $Rho] $a $b]

    insert "***Now look for infinitary states."

    set SuccInf [* [* $SuccWStarAndMore $a_halves_are_equal_BDD] $initialise_tests_BDD]
    set SuccInfStar $SuccInf
    set RhoInf [* $Rho [* $a_right_equals_b_right_BDD $update_tests_BDD]]
    set SuccInfList {}

    set Convergence 0

    set InfLoopTest_BDD [* $a_halves_are_equal_BDD $achieve_all_tests_BDD]

    set j 0

    while { ( $j == 0 || [* $SuccInf $InfLoopTest_BDD] == [Zero]) && $j < $max_iterations } {

        insert "***Starting infinitary iteration $j."

        lappend SuccInfList $SuccInf

        check_node_stats_infinite

        set SuccInf [SwapVariables [AndExists $a_cube $SuccInf $RhoInf] $a $b]

        set OldSuccInfStar $SuccInfStar

        set SuccInfStar [+ $SuccInfStar $SuccInf]

        if { ($OldSuccInfStar == $SuccInfStar) } {
            set Convergence 1
        } else {
            set Convergence 0
        }

        incr j

        if { $Convergence } { break }
    }

    insert "$j iteration(s) performed."

    if { $Convergence } {
        insert "***Convergence."
    }

    if { $j >=  $max_iterations && !$Convergence } {
        insert "Unable to converge when looking for infinitary states."
        return 1
    }

    set SuccInf [* $SuccInf $InfLoopTest_BDD]

    if { $SuccInf == [Zero] } {
        if { $negate_init_formula == 0 } {
            insert " "
            insert "Unsatisfiable with infinite time." red
            insert " "
        } else {
            insert " "
            insert "Valid with infinite time." green
            insert " "
        }
    } else {
        if {  $negate_init_formula == 0 } {
            insert " "
            insert "Satisfiable with infinite time." green
            insert " "
        } else {
            insert " "
            insert "Not valid with infinite time." red
            insert " "
        }
        set PeriodicState $SuccInf
        compute_periodic_segment_model $suffix_name_list $invariant_length $j
        set tj [analyse_initial_segment_model $Init $f_Init $negate_init_formula $p_Rho $f_Rho $p_Tau]
        print_initial_segment_model $suffix_name_list $invariant_length  $tj $Rho $f_Init
        print_periodic_segment_model $suffix_name_list $invariant_length
    }

    insert "Cudd current number of live nodes=[LiveNodes]" blue
    insert "Cudd peak number of live nodes=[PeakLiveNodes]" blue
    insert "----------------------------------------------"

    return 0
}


#
# invariant ...
#  check satifiability/validity of a fusion logic formula in 
#  terms of bdd operations 
#
proc  invariant {suffix_name_list negate_init_formula p_invariant_length f_invariant_length Init f_Init p_Rho f_Rho p_Tau f_Tau} {
    
    variable a 
    variable b 
    variable a_cube 
    variable b_cube 
    variable manager
    variable my_names
    variable num_vars
    global dbg

    
    set Tau [* $p_Tau $f_Tau]
    set Rho [* $p_Rho $f_Rho]
    set invariant_length [expr $p_invariant_length+$f_invariant_length]
    
    set num_vars [llength $suffix_name_list]
    set tt [expr $num_vars-$invariant_length]
   
    #insert "*** # of dependencies=$invariant_length, # of independent variables=$tt."
    set max_iterations 100
   
    set SuccW $Init
    
    if {$negate_init_formula == 0} {
        insert "***Testing for satisfiability with finite time."
        set rev 0
    } else {
        if {$negate_init_formula == 1} {
            insert "***Testing for validity with finite time."
            #set SuccW [Not $SuccW]
            set rev 0
        } else {
            if {$negate_init_formula == 6} {
              insert "***Testing for reverse satisfiability with finite time."
              set rev 1
            } else {
               if {$negate_init_formula == 7} {
                 insert "***Testing for validity with finite time."
                 #set SuccW [Not $SuccW] 
                 set rev 1
               } else {
                insert "***Do not know what to test."
             }
            }
        }
    }

    
    set SuccWStar $SuccW
    set OldSuccWStar [Not $SuccWStar]
    set Convergence 0
    set SuccWList {}
    #set FirstSuccW $SuccW
    
    set TempBDD [* $SuccW $Tau]
    #dot [list $Rho] $my_names [list "Transition function" ]
    #dot [list $SuccW] $my_names [list "initial state" ]
    #dot [list $SuccW $TempBDD] $my_names [list "First Loop: Initial SuccW" "TempBDD"]
    
    set j 0
    set pj 0
    #set p_SuccW $SuccW
    
    while { ([* $SuccW $Tau] == [Zero]) && ($j < $max_iterations)  } {
        
        insert "***Starting iteration $j."
        check_node_stats
        
        lappend SuccWList $SuccW
        #set p_SuccW [SwapVariables [AndExists $a_cube $SuccW $p_Rho] $a $b]
        set SuccW [SwapVariables [AndExists $a_cube $SuccW $Rho] $a $b]
        set OldSuccWStar $SuccWStar
        set SuccWStar [+ $SuccWStar $SuccW]
        if { $OldSuccWStar == $SuccWStar } {
            set Convergence 1 
        } else {
          set Convergence 0
        }  
        #  dot [list $SuccW] $my_names [list "First Loop: SuccW after j=$j."]
        incr j
        if { $Convergence } { break }
    }


    if { $dbg == 1 } {
        insert_debug "pj = $pj, j= $j, Convergence=$Convergence"
    }
    
    if {$j==1} {
       set iterations "iteration"
    } else {
       set iterations "iterations"
    }
    insert "$j $iterations performed."
    if { [* $SuccW $Tau] == [Zero] } {
        if { $Convergence == 0} {
          insert "***No convergence after $max_iterations iterations!"
        } else {
            if { $negate_init_formula == 0} {
                insert " "
                insert  "Unsatisfiable with finite time." red
                insert " "
            } else {
                if { $negate_init_formula == 1} {
                    insert " "
                    insert "Valid with finite time." green
                    insert " "
               } else {
                   if { $negate_init_formula == 6} {
                       insert " "
                       insert  "Unsatisfiable using time reversal with finite time." red
                       insert " "
                } else {
                    if { $negate_init_formula == 7} {
                        insert " "
                        insert "Valid using time reversal with finite time." green
                        insert " "
                 }
                }
               }
             }
            }
    } else {
        # dot [list $Tau] $my_names [list "Tau"]
        
        if { $negate_init_formula == 0} {
            insert " "
            insert "Satisfiable with finite time." green
            insert " "
            set Counter "model"
       } else {
           if { $negate_init_formula == 1} {
               insert " "
               insert "Not valid with finite time." red
               insert " "
               set Counter "counter-model"
         } else {
             if { $negate_init_formula == 6} {
                 insert " "
                 insert "Satisfiable using time reversal with finite time." green
                 insert " "
                 set Counter "model"
             } else {
                 insert " "
                 insert "Not valid using time reversal with finite time." red
                 insert " "
                 set Counter "counter-model"
           }
         }
       }
       insert "***Now start to find a $Counter."
       #insert "Length of SuccWList = [llength $SuccWList]" 
       set StateStringList {}
        if {[llength $suffix_name_list] == $invariant_length } {
            set number_of_states [expr $j+1]
            
            insert "No independent variables."

            set SuccW [* $SuccW $Tau]
            while {$j>=0} {

                if { [* $SuccW $f_Init] == [Zero] } {
                    if { $dbg == 1 } {
                        insert_debug "j=$j, zero"
                    }
                } else {
                    set pj $j
                    if { $dbg == 1 } {
                        insert_debug "j=$j, one"
                    }
                }
                if {$j !=0} {
                    #puts "length SuccWList= [llength $SuccWList]" 
                    #puts "SuccWList 0 = [lindex $SuccWList 0], FirstSuccW = $FirstSuccW"
                    set TempBDD [lindex $SuccWList end]
                    #remove last element from SuccWlist
                    set SuccWList [lreplace $SuccWList end end]
                    #dot [list $TempBDD] $my_names [list "Model: TempBDD for j=$j"]
                    set SuccW [AndExists $b_cube $TempBDD  [* $Rho [SwapVariables $SuccW $a $b]]]
                }
                incr j -1
            }
            
            if {$number_of_states == 1 } { 
                 set states "state"  
            } else {
                 set states "states"
            }
            
            if {$pj == 1} {
                set pstates "state"
            } else {
                set pstates "states"
            }
            
            if {$pj > 0} {
                set ptext " with $pj past $pstates."
            } else {
                set ptext "."
            }
            #set ptext "." 
            insert "Can use any interval with exactly $number_of_states $states as a $Counter$ptext"
        } else {
            set SuccW [* $SuccW $Tau]
            while {$j>=0} {
                #insert "j=$j"
                set StateString ""
                if { $dbg == 1 } {
                    set k 0
                } else {
                    set k $invariant_length
                }
                #dot [list $SuccW] $my_names [list "Model: Initial SuccW for j=$j k=$k"]
                while { $k < [llength $suffix_name_list] } {
                  append StateString "[lindex $suffix_name_list $k]=" 
                  if { [* $SuccW [lindex $a $k] ] == [Zero] } {
                      append StateString "0 "
                  } else {
                      set SuccW [* $SuccW [lindex $a $k]]
                      append StateString "1 "
                  }
                 #   dot [list $SuccW] $my_names [list "Model: Final SuccW for j=$j k=$k"]
                  incr k
                }
                if { [* $SuccW $f_Init] == [Zero] } {
                    if { $dbg == 1 } {
                        insert_debug "j=$j, zero"
                    }
                } else {
                    set pj $j
                    if { $dbg == 1 } {
                        insert_debug "j=$j, one"
                    }
                }
                if {$j !=0} {
                    #puts "length SuccWList= [llength $SuccWList]" 
                    #puts "SuccWList 0 = [lindex $SuccWList 0], FirstSuccW = $FirstSuccW"
                    set TempBDD [lindex $SuccWList end]
                    #remove last element from SuccWlist
                    set SuccWList [lreplace $SuccWList end end]
                    #dot [list $TempBDD] $my_names [list "Model: TempBDD for j=$j"]
                    set SuccW [AndExists $b_cube $TempBDD  [* $Rho [SwapVariables $SuccW $a $b]]]
                }
                lappend StateStringList $StateString
                incr j -1
            }
            set number_of_states [llength $StateStringList]
            if {$number_of_states == 1 } { 
                 set states "state"  
            } else {
                 set states "states"
            }
            if {$pj == 1} {
                set pstates "state"
            } else {
                set pstates "states"
            }
            # 
            if {$pj > 0} {
                set ptext " with $pj past $pstates:"
            } else {
                set ptext ":"
            }
            insert "Here is a $Counter with $number_of_states $states$ptext"
            for {set j [expr [llength $StateStringList] -1]} {$j>=0} {incr j -1} {
              set k [expr [llength $StateStringList] -$j-1]
              set z [lindex $StateStringList $j]
              set z1 [split $z " "]
              set z2 [lsort -dictionary $z1]
              if { $rev == 0 } {
                  #  
                  if {$k == $pj } {
                      #insert "   State $k (current): "
                      insert "   State $k (current): "
                  } else {
                      insert "   State $k: "
                  }
              } else {
                  set k1 [expr $number_of_states-$k-1]
                  # 
                  if {$k1 == $pj} {
                      insert "   State $k1 (current): "
                      #insert "   State $k1: "
                  } else {
                      insert "   State $k1: "
                  }
              }
              foreach z3 $z2 {
                if { [string range $z3 0 5] == "AUTHON" } {
                     insert "  $z3" red
                } else { 
                  if { [string range $z3 0 5] == "AUTHOP" } {
                     insert "  $z3" green
                  } else {
                     insert "  $z3"
                  }
                }
              }
              insert " "   
            }
        }
   }
    insert "Cudd current number of live nodes=[LiveNodes]" blue
    insert "Cudd peak number of live nodes=[PeakLiveNodes]" blue
    insert "----------------------------------------------"
}

#
# enforce ...
# enforce a right fusion logic formula wrt  inputtrace in terms of bdd operations
#
proc  enforce {suffix_name_list invariant_length Init Rho Tau Inputtrace} {
   variable a 
   variable b 
   variable a_cube 
   variable b_cube 
   variable manager
   variable my_names
   global dbg

   set num_vars [llength $suffix_name_list]
   set tt [expr $num_vars-$invariant_length]
   
   #insert "*** # of dependencies=$invariant_length, # of independent variables=$tt."
   set max_iterations 100

    #set input_0 [lindex $Inputtrace 0]
    #puts "enf: input_0 : $input_0"  
    #RAnd input_0_and $input_0 
    
    set SuccW $Init 
    insert "***Testing enforcer against input trace."
    
    #set SuccWStar $SuccW
    #set OldSuccWStar [Not $SuccWStar]
    #set Convergence 0
    #set SuccWList {}
    #set FirstSuccW $SuccW
    #set TempBDD [* $SuccW $Tau]
    #dot [list $Rho] $my_names [list "Transition function" ]
    #dot [list $SuccW $TempBDD] $my_names [list "First Loop: Initial SuccW" "TempBDD"]
    set j 0
    
    foreach input_element $Inputtrace {      
      # RAnd input_element_and $input_element 
      #puts "input_el: $input_element"
       
        set SuccW [* $SuccW $input_element]
       
       set SuccW [* $SuccW $Tau]

       #    [* $SuccW $Tau]
       # if { $SuccW == [Zero] } { 
       #   insert "not a final state" red 
       # } else {
       #   insert "final state" green
       # }

       set StateString ""
       if { $dbg == 1 } {
           set k 0
       } else {
           set k $invariant_length
       }
       #set k $invariant_length
       #set k 0
       while { $k < [llength $suffix_name_list] } {
          append StateString "[lindex $suffix_name_list $k]="
          if { [* $SuccW [lindex $a $k] ] == [Zero] } {
             append StateString "0 "
          } else {
             set SuccW [* $SuccW [lindex $a $k]]
             append StateString "1 "
          }
          incr k
       }
       set z1 [split $StateString " "]
       set z2 [lsort -dictionary $z1]
       if { $SuccW == [Zero] } {
           insert " "
           insert "   State $j (not final): " red
           #insert "not a final state" red 
       } else {
           insert " "
           insert "   State $j (final): " green
           #insert "final state" green
       }
       #insert "   State $j: "
       incr j
       foreach z3 $z2 {
         if { [string range $z3 0 5] == "AUTHON" } {
            insert "  $z3" red
         } else { 
            if { [string range $z3 0 5] == "AUTHOP" } {
               insert "  $z3" green
            } else {
               insert "  $z3"
            }
         }
       }   

     set SuccW [SwapVariables [AndExists $a_cube $SuccW $Rho] $a $b]
     #set SuccW  [AndExists $b_cube [SwapVariables $SuccW $a $b] $Rho]
     
     # set SuccW $Init
    }
    insert "Cudd current number of live nodes=[LiveNodes]" blue
    insert "Cudd peak number of live nodes=[PeakLiveNodes]" blue
    insert "----------------------------------------------"
}

#
# p_enforce ...
# enforce a left fusion logic formula wrt  inputtrace in terms of bdd operations
#
proc  p_enforce {suffix_name_list invariant_length Init Rho Tau Inputtrace} {
   variable a 
   variable b 
   variable a_cube 
   variable b_cube 
   variable manager
   variable my_names
   global dbg

   set num_vars [llength $suffix_name_list]
   set tt [expr $num_vars-$invariant_length]
   
   #insert "*** # of dependencies=$invariant_length, # of independent variables=$tt."
   set max_iterations 100

    #set input_0 [lindex $Inputtrace 0]
    #puts "enf: input_0 : $input_0"  
    #RAnd input_0_and $input_0 
    
    set SuccW $Tau 
    insert "***Testing enforcer against input trace using time reversal."
    
    #set SuccWStar $SuccW
    #set OldSuccWStar [Not $SuccWStar]
    #set Convergence 0
    #set SuccWList {}
    #set FirstSuccW $SuccW
    #set TempBDD [* $SuccW $Tau]
    #dot [list $Rho] $my_names [list "Transition function" ]
    #dot [list $SuccW $TempBDD] $my_names [list "First Loop: Initial SuccW" "TempBDD"]
    set j 0
    
    foreach input_element $Inputtrace {      
      # RAnd input_element_and $input_element 
      #puts "input_el: $input_element"
       
       set SuccW [* $SuccW $input_element]
       
       set SuccW [* $SuccW $Init]

       # [* $SuccW $Tau]
       # if { $SuccW == [Zero] } { 
       #   insert "not a first state" red 
       # } else {
       #   insert "first state" green
       # }

       set StateString ""
       set k $invariant_length
       while { $k < [llength $suffix_name_list] } {
          append StateString "[lindex $suffix_name_list $k]="
          if { [* $SuccW [lindex $a $k] ] == [Zero] } {
             append StateString "0 "
          } else {
             set SuccW [* $SuccW [lindex $a $k]]
             append StateString "1 "
          }
          incr k
       }
       set z1 [split $StateString " "]
       set z2 [lsort -dictionary $z1]
       if { $SuccW == [Zero] } {
           insert " "
           insert "   State $j (not first): " red
       } else {
           insert " "
           insert "   State $j (first): " green
       }
       
       incr j
       foreach z3 $z2 {
         if { [string range $z3 0 5] == "AUTHON" } {
            insert "  $z3" red
         } else { 
            if { [string range $z3 0 5] == "AUTHOP" } {
               insert "  $z3" green
            } else {
               insert "  $z3"
            }
         }
       }   

       set SuccW [AndExists $b_cube [SwapVariables $SuccW $a $b] $Rho]

     
     # set SuccW $Init
    }
    insert "Cudd current number of live nodes=[LiveNodes]" blue
    insert "Cudd peak number of live nodes=[PeakLiveNodes]" blue
    insert "----------------------------------------------"
}

#
# i_enforce ...
# interactive enforcement of a right fusion logic formula in terms of bdd operations
#
proc  i_enforce {suffix_name_list invariant_length Init Rho Tau} {
   variable a 
   variable b 
   variable a_cube 
   variable b_cube 
   variable manager
   variable my_names
   variable indep

   set num_vars [llength $suffix_name_list]
   set tt [expr $num_vars-$invariant_length]
   
   #insert "*** # of dependencies=$invariant_length, # of independent variables=$tt."
   set max_iterations 100

    #set input_0 [lindex $Inputtrace 0]
    #puts "enf: input_0 : $input_0"  
    #RAnd input_0_and $input_0 
    
    set SuccW $Init 
    insert "***Testing enforcer against interactive input."
    
    #set SuccWStar $SuccW
    #set OldSuccWStar [Not $SuccWStar]
    #set Convergence 0
#    set SuccWList {}
    #set FirstSuccW $SuccW
    #set TempBDD [* $SuccW $Tau]
    #dot [list $Rho] $my_names [list "Transition function" ]
    #dot [list $SuccW $TempBDD] $my_names [list "First Loop: Initial SuccW" "TempBDD"]
    set j 0
     
     set g_input ""
     while { $g_input != "end" } {
      puts "new input"
      
      gets stdin g_input
      update 

      if { $g_input != "end" } {
         #puts "g_input: $g_input"
         set g_input_a [subst $g_input]
         #puts "g_input_a: $g_input_a"
         set inp [tl_and $g_input_a]
         #puts "inp: $inp"
         set inp_a [input_to_trace $inp]
         #puts "inp_a: $inp_a"
         set inp_bdd [subst $inp_a]
         # puts "el_bdd: $el_bdd"
         # RAnd input_element_and $input_element 
         #puts "input_el: $input_element"
          
 
       set SuccW [* $SuccW $inp_bdd]
       
       set SuccW [* $SuccW $Tau]
       # [* $SuccW $Tau]
       # if { $SuccW == [Zero] } { 
       #   puts "not a final state" 
       # } else {
       #   puts "final state" 
       # }

       set StateString ""
       set k $invariant_length
       while { $k < [llength $suffix_name_list] } {
          append StateString "[lindex $suffix_name_list $k]="
          if { [* $SuccW [lindex $a $k] ] == [Zero] } {
             append StateString "0 "
          } else {
             set SuccW [* $SuccW [lindex $a $k]]
             append StateString "1 "
          }
          incr k
          update
       }
       set z1 [split $StateString " "]
       set z2 [lsort -dictionary $z1]
       if { $SuccW == [Zero] } {
           insert " "
           insert "   State $j (not final): " red
       } else {
           insert " "
           insert "   State $j (final): " green
       }
      
       incr j
       foreach z3 $z2 {
         if { [string range $z3 0 5] == "AUTHON" } {
            puts "  $z3" 
         } else { 
            if { [string range $z3 0 5] == "AUTHOP" } {
               puts "  $z3" 
            } else {
               puts "  $z3"
            }
         }
       }   

     set SuccW [SwapVariables [AndExists $a_cube $SuccW $Rho] $a $b]
     
     update 
     # set SuccW $Init
    
    puts "Cudd current number of live nodes=[LiveNodes]" 
    puts "Cudd peak number of live nodes=[PeakLiveNodes]" 
    puts "----------------------------------------------"
      } else { 
        return
      }
     }
}
#
# p_i_enforce ...
# interactive enforcement of a left fusion logic formula in terms of bdd operations
#
proc  p_i_enforce {suffix_name_list invariant_length Init Rho Tau} {
   variable a 
   variable b 
   variable a_cube 
   variable b_cube 
   variable manager
   variable my_names
   variable indep

   set num_vars [llength $suffix_name_list]
   set tt [expr $num_vars-$invariant_length]
   
   #insert "*** # of dependencies=$invariant_length, # of independent variables=$tt."
   set max_iterations 100

    #set input_0 [lindex $Inputtrace 0]
    #puts "enf: input_0 : $input_0"  
    #RAnd input_0_and $input_0 
    
    set SuccW $Tau 
    insert "***Testing enforcer against interactive input using time reversal."
    
    #set SuccWStar $SuccW
    #set OldSuccWStar [Not $SuccWStar]
    #set Convergence 0
#    set SuccWList {}
    #set FirstSuccW $SuccW
    #set TempBDD [* $SuccW $Tau]
    #dot [list $Rho] $my_names [list "Transition function" ]
    #dot [list $SuccW $TempBDD] $my_names [list "First Loop: Initial SuccW" "TempBDD"]
    set j 0
     
     set g_input ""
     while { $g_input != "end" } {
      puts "new input"
      
      gets stdin g_input
      update 

      if { $g_input != "end" } {
         #puts "g_input: $g_input"
         set g_input_a [subst $g_input]
         #puts "g_input_a: $g_input_a"
         set inp [tl_and $g_input_a]
         #puts "inp: $inp"
         set inp_a [input_to_trace $inp]
         #puts "inp_a: $inp_a"
         set inp_bdd [subst $inp_a]
         # puts "el_bdd: $el_bdd"
         # RAnd input_element_and $input_element 
         #puts "input_el: $input_element"
          
 
       set SuccW [* $SuccW $inp_bdd]
       
       set SuccW [* $SuccW $Init]
       # [* $SuccW $Tau]
       # if { $SuccW == [Zero] } { 
       #   puts "not a first state" 
       # } else {
       #   puts "first state" 
       # }

       set StateString ""
       set k $invariant_length
       while { $k < [llength $suffix_name_list] } {
          append StateString "[lindex $suffix_name_list $k]="
          if { [* $SuccW [lindex $a $k] ] == [Zero] } {
             append StateString "0 "
          } else {
             set SuccW [* $SuccW [lindex $a $k]]
             append StateString "1 "
          }
          incr k
          update
       }
       set z1 [split $StateString " "]
       set z2 [lsort -dictionary $z1]
       if { $SuccW == [Zero] } {
           insert " "
           insert "   State $j (not first): " red
       } else {
           insert " "
           insert "   State $j (first): " green
       }  
       
       incr j
       foreach z3 $z2 {
         if { [string range $z3 0 5] == "AUTHON" } {
            puts "  $z3" 
         } else { 
            if { [string range $z3 0 5] == "AUTHOP" } {
               puts "  $z3" 
            } else {
               puts "  $z3"
            }
         }
       }   

     set SuccW [AndExists $b_cube [SwapVariables $SuccW $a $b] $Rho]
     
     update 
     # set SuccW $Init
    
    puts "Cudd current number of live nodes=[LiveNodes]" 
    puts "Cudd peak number of live nodes=[PeakLiveNodes]" 
    puts "----------------------------------------------"
      } else { 
        return
      }
     }
}
