(*  Title     : A Shallow Encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

section \<open>Until operator\<close>   

theory Until
imports Semantics  SChopTheorems  
begin
(*
sledgehammer_params [minimize=true,isar_proofs=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="cvc4 z3 e spass vampire " ]
*)

text \<open> 
This theory introduces the weak and strong versions of the until operator. 
The theorems from \cite{vegapaper2019} are proven in a mostly deductive style. 
\<close>

subsection \<open>Definitions\<close>

definition until_d :: " ('a :: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
 where "until_d F G \<equiv> \<lambda>s. ( (\<exists> k . k \<le> nlength s \<and> ( (ndropn k s ) \<Turnstile> G) \<and> 
                               (\<forall>j. j < k  \<longrightarrow>  ( (ndropn j s ) \<Turnstile> F) )) )"

syntax 
 "_until_d"  :: "[lift,lift] \<Rightarrow> lift"        ("(_ \<U> _)" [84,84] 83)

syntax (ASCII)
 "_until_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ until _)" [84,84] 83)

translations
 "_until_d"    \<rightleftharpoons> "CONST until_d"

definition suntil_d :: " ('a :: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
 where "suntil_d F G \<equiv> LIFT(\<circle>(F \<U> G))"

definition wait_d ::  " ('a :: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
  where "wait_d F G \<equiv> LIFT(\<box> F \<or> F \<U> G)"

definition release_d ::  " ('a :: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
  where "release_d F G \<equiv> LIFT(\<not>((\<not> F) \<U> (\<not> G)))"

syntax 
 "_wait_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ \<W> _)" [84,84] 83)
 "_release_d"  :: "[lift,lift] \<Rightarrow> lift"        ("(_ \<R> _)" [84,84] 83)
 "_suntil_d"   :: "[lift,lift] \<Rightarrow> lift"        ("(_ \<U>\<^sup>s _)" [84,84] 83)

syntax (ASCII)
 "_wait_d"        :: "[lift,lift] \<Rightarrow> lift"        ("(_ wait _)" [84,84] 83)
 "_release_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ release _)" [84,84] 83)
 "_suntil_d"      :: "[lift,lift] \<Rightarrow> lift"        ("(_ suntil _)" [84,84] 83)

translations
 "_wait_d"       \<rightleftharpoons> "CONST wait_d" 
 "_release_d"    \<rightleftharpoons> "CONST release_d" 
 "_suntil_d"     \<rightleftharpoons> "CONST suntil_d"

definition srelease_d ::  " ('a :: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
  where "srelease_d F G \<equiv> LIFT(\<not>((\<not> F) \<W> (\<not> G)))"

syntax 
 "_srelease_d"  :: "[lift,lift] \<Rightarrow> lift"        ("(_ \<M> _)" [84,84] 83)

syntax (ASCII)
 "_srelease_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ srelease _)" [84,84] 83)

translations
 "_srelease_d"    \<rightleftharpoons> "CONST srelease_d" 

subsection \<open>Axioms\<close>

subsubsection \<open>NextUntil\<close>

lemma NextUntilsema:
 assumes "( \<sigma> \<Turnstile> \<circle>(f \<U> g))"
 shows   "( \<sigma> \<Turnstile> (\<circle> f) \<U> (\<circle> g))"
proof -
 have 0: "0 < nlength \<sigma> \<and> 
         (\<exists>k. k\<le>nlength (ndropn (Suc 0) \<sigma>) \<and> g ((ndropn (Suc k) \<sigma>)) \<and> 
           (\<forall>j<k. f ( (ndropn (Suc j) \<sigma>))  ))"
   using assms zero_enat_def by (auto simp add: next_defs until_d_def ndropn_ndropn)  
 have 1: "0 < nlength \<sigma> "
   using "0" by auto
 have 2: "(\<exists>k. k\<le>nlength (ndropn (Suc 0) \<sigma>) \<and> g ((ndropn (Suc k) \<sigma>)) \<and> 
          (\<forall>j<k. f (( (ndropn (Suc j) \<sigma>)) ))) "
   using "0" by auto
 obtain k where 3: "k\<le>nlength (ndropn (Suc 0) \<sigma>) \<and> g (( (ndropn (Suc k) \<sigma>))) \<and> 
      (\<forall>j<k. f (( (ndropn (Suc j) \<sigma>)) )) "
   using "2" by auto   
 have 4: "g (( (ndropn (Suc k) \<sigma>))) " 
   using "3" by auto
 have 5: " k\<le> nlength \<sigma>" 
   using "3" "0" 
   by (metis diff_le_self dual_order.order_iff_strict enat_ile enat_ord_simps(1) idiff_enat_enat 
        less_le_trans ndropn_nlength not_less) 
 have 6: "0 < nlength (ndropn k \<sigma>) " 
   using "1" "3"
   by (metis gr_zeroI iless_Suc_eq is_NNil_ndropn leD le_numeral_extra(3) ndropn_0 
       ndropn_Suc_conv_ndropn nlength_NCons zero_enat_def) 
 have 7: "(\<forall>j<k. 0 < nlength (ndropn j \<sigma>) \<and> f (( (ndropn (Suc j) \<sigma>))))" 
   using "3" "5"
   by (metis enat_ord_simps(2) is_NNil_ndropn ndropn_0 not_less order.trans zero_le)
 have 71: " nlength \<sigma> - enat k \<noteq> enat 0" 
   using "6" zero_enat_def by auto
 have 72: "(\<forall>j<k. nlength \<sigma> - enat j \<noteq> enat 0 \<and> f (ndropn (Suc j) \<sigma>))" 
   using "7" zero_enat_def by auto
 have 8: "\<exists>k. k\<le>nlength \<sigma> \<and>
            nlength \<sigma> - enat k \<noteq> enat 0 \<and>
            g (( (ndropn (Suc k) \<sigma>))) \<and> 
            (\<forall>j<k. nlength \<sigma> - enat j \<noteq> enat 0 \<and> f (ndropn (Suc j) \<sigma>))"
   using "3" "5" "6" "71" "7" "72"  by blast
 from 8 show ?thesis 
by (simp add: next_defs until_d_def ndropn_ndropn)
qed  

lemma NextUntilsemb:
 assumes "(\<sigma> \<Turnstile> (\<circle> f) \<U> (\<circle> g))"
 shows   "( \<sigma> \<Turnstile> \<circle>(f \<U> g))"
proof -
 have 1: "\<exists>k. k\<le>nlength \<sigma> \<and> 0 < nlength (ndropn k \<sigma>) \<and> g ( (ndropn (Suc k) \<sigma>)) \<and> 
           (\<forall>j<k. j < nlength \<sigma> \<and> f ( (ndropn (Suc j) \<sigma>)))" 
   using assms 
   by (auto simp add: next_defs until_d_def ndropn_ndropn)
       (metis  is_NNil_ndropn le_zero_eq  ndropn_0  ndropn_nlength not_less zero_enat_def)
 obtain k where 2: "k\<le>nlength \<sigma> \<and> 0 < nlength (ndropn k \<sigma>) \<and>
            g ( (ndropn (Suc k) \<sigma>)) \<and> 
           (\<forall>j<k. j < nlength \<sigma> \<and> f ( (ndropn (Suc j) \<sigma>)))"
   using 1 by auto
 have 3: "0 < nlength \<sigma>  "  
   using "2" by auto
 have 4: "k\<le>nlength (ndropn (Suc 0) \<sigma>)" 
   by auto
     (metis "2" "3" add.commute add.right_neutral enat_min iless_Suc_eq ndropn_0 
      ndropn_Suc_conv_ndropn ndropn_nlength nlength_NCons zero_enat_def)
 have 5: "g ( (ndropn (Suc k) \<sigma>)) " 
   using "2" by auto
 have 6: "(\<forall>j<k. j < nlength \<sigma> \<and> f ( (ndropn (Suc j) \<sigma>)))" 
   using "2" by blast
 have 7: "0 < nlength \<sigma> \<and> 
          (\<exists>k. k\<le>nlength (ndropn (Suc 0) \<sigma>) \<and> g ( (ndropn (Suc k) \<sigma>)) \<and>
             (\<forall>j<k. f ( (ndropn (Suc j) \<sigma>)) )  )" 
   using "2" "3" "4" by blast
 from 7 show ?thesis by (auto simp: next_defs until_d_def zero_enat_def ndropn_ndropn)
qed

lemma NextUntilsem:
 "\<sigma> \<Turnstile> \<circle>(f \<U> g) = (\<circle> f) \<U> (\<circle> g)"
using NextUntilsema NextUntilsemb using unl_lift2 by blast

lemma NextUntil:
 "\<turnstile> \<circle>(f \<U> g) = (\<circle> f) \<U> (\<circle> g)"
using NextUntilsem Valid_def by blast

subsubsection \<open>UntilNextUntil\<close>

lemma UntilNextUntilsema:
 assumes "0 < nlength \<sigma> \<and> 
          (\<exists>k. 0< k \<and> k \<le> nlength \<sigma> \<and> g ((ndropn k \<sigma>)) \<and> (\<forall>j. 0 < j \<and> j < k \<longrightarrow> f ((ndropn j \<sigma>))))"
 shows   "( \<sigma>) \<Turnstile> \<circle> ( f \<U> g)"
proof -
 have 1: "0 < nlength \<sigma> \<and> 
          (\<exists>k. 0<k \<and> k \<le> nlength \<sigma> \<and> g ((ndropn k \<sigma>)) \<and> (\<forall>j. 0 < j \<and> j < k \<longrightarrow> f ((ndropn j \<sigma>)))) "
   using assms by auto
 have 3: "(\<exists>k. 0<k \<and> k \<le> nlength \<sigma> \<and> g (( (ndropn k \<sigma>))) \<and> (\<forall>j. 0 < j \<and> j < k \<longrightarrow> f ((ndropn j \<sigma>))) )"
   using "1" by auto
 obtain k where 4: "0< (Suc k) \<and> (Suc k) \<le> nlength \<sigma> \<and> g ( (ndropn (Suc k) \<sigma>)) \<and> 
      (\<forall>j. 0 < j \<and> j < (Suc k) \<longrightarrow> f ( (ndropn j \<sigma>)))"
   using "3" by (metis Suc_pred)
 have 5: "k\<le>nlength (ndropn (Suc 0) \<sigma>)" 
   by (metis "4" One_nat_def add.commute co.enat.sel(2) eSuc_enat epred_conv_minus le_cases min_absorb1 
      ndropn_nlength ntaken_all ntaken_ndropn_swap_nlength ntaken_nlength one_enat_def plus_1_eSuc(2) 
      plus_1_eq_Suc)
 have 6: "g ( (ndropn (Suc k) \<sigma>))"
   using "4" by auto
 have 7: "(\<forall>j<k. f ( (ndropn (Suc j) \<sigma>)))"
   using "4" by blast  
 have 8: "(\<exists>k. k\<le>nlength (ndropn (Suc 0) \<sigma>) \<and> g ((ndropn (Suc k) \<sigma>)) \<and> (\<forall>j<k. f ((ndropn (Suc j) \<sigma>)))) " 
   using "4" "5" by blast
 have 9: "0 < nlength \<sigma> \<and>
          (\<exists>k. k\<le>nlength (ndropn (Suc 0) \<sigma>) \<and> g ((ndropn (Suc k) \<sigma>)) \<and> (\<forall>j<k. f ((ndropn (Suc j) \<sigma>))))"
   using "1" "8" by blast 
 from 9 show ?thesis by (auto simp add: next_defs until_d_def zero_enat_def ndropn_ndropn)
qed

lemma UntilNextUntilsemb:
 assumes " \<sigma> \<Turnstile> \<circle> ( f \<U> g)"
 shows   "0 < nlength \<sigma> \<and> 
          (\<exists>k. 0< k \<and> k \<le> nlength \<sigma> \<and> g ((ndropn k \<sigma>)) \<and> (\<forall>j. 0 < j \<and> j < k \<longrightarrow> f ((ndropn j \<sigma>))))"
proof -
 have 1: "0 < nlength \<sigma> \<and> 
          (\<exists>k. k\<le>nlength(ndropn (Suc 0) \<sigma>) \<and> g ((ndropn (Suc k) \<sigma>)) \<and> (\<forall>j<k. f ((ndropn (Suc j) \<sigma>))))"
   using assms by (auto simp add: next_defs until_d_def ndropn_ndropn) (simp add: zero_enat_def)  
 have 2: "(\<exists>k. k\<le>nlength(ndropn (Suc 0) \<sigma>) \<and> g ((ndropn (Suc k) \<sigma>)) \<and> (\<forall>j<k. f ((ndropn (Suc j) \<sigma>))))" 
   using "1" by auto
 obtain k where 3: "k\<le>nlength(ndropn (Suc 0) \<sigma>) \<and> g ((ndropn (Suc k) \<sigma>)) \<and> 
         (\<forall>j<k. f ( (ndropn (Suc j) \<sigma>))) "
   using "2" by auto
 have 4: "0< (Suc k)"
  by simp
 have 5: "g ( (ndropn (Suc k) \<sigma>)) " 
   using "3" by auto
 have 6: "(Suc k) \<le> nlength \<sigma>"
   using  3 by auto
    (metis "1" dual_order.eq_iff eSuc_enat is_NNil_ndropn le_cases ndropn_0 ndropn_Suc_conv_ndropn 
       ndropn_ndropn ndropn_nlength nlength_NCons plus_1_eq_Suc zero_enat_def)
 have 7: "(\<forall>j. 0 < j \<and> j < (Suc k) \<longrightarrow> f ( (ndropn j \<sigma>)))"
   using "3" less_Suc_eq_0_disj by auto
   have 8: "(\<exists>k. 0<k \<and> k \<le> nlength \<sigma> \<and> g ((ndropn k \<sigma>)) \<and> (\<forall>j. 0 < j \<and> j < k \<longrightarrow> f ((ndropn j \<sigma>))))"   
   using "3" "6" "7" by blast
 show ?thesis using "1" "8" by blast
qed

lemma UntilNextUntilsem:
 "( \<sigma> \<Turnstile> \<circle> ( f \<U> g)) = 
  (0 < nlength \<sigma> \<and> 
    (\<exists>k. 0< k \<and> k \<le> nlength \<sigma> \<and> g ((ndropn k \<sigma>)) \<and> (\<forall>j. 0 < j \<and> j < k \<longrightarrow> f ((ndropn j \<sigma>))))) "
using UntilNextUntilsema[of \<sigma> g f] UntilNextUntilsemb[of f g \<sigma>] by meson 
 
lemma UntilNextUntilsem1:
 "(\<sigma> \<Turnstile> f \<U> g) = (\<sigma> \<Turnstile> (g \<or> (f \<and> \<circle>(f \<U> g))))"
unfolding  UntilNextUntilsem 
proof 
  assume a: "(\<sigma> \<Turnstile> f \<U> g) "
  show "(\<sigma> \<Turnstile> g \<or>
          f \<and>
          (\<lambda>\<sigma>.  0 < nlength \<sigma> \<and> 
                (\<exists>k. 0<k \<and> k \<le> nlength \<sigma> \<and> g((ndropn k \<sigma>)) \<and> (\<forall>j. 0 < j \<and> j < k \<longrightarrow> f((ndropn j \<sigma>))))
          ))"
    using a by (simp add: until_d_def) (metis enat_0_iff(2) i0_less ndropn_0 neq0_conv not_le)
   next
  next
   assume b: "(\<sigma> \<Turnstile> g \<or>
          f \<and>
          (\<lambda>\<sigma>. 0 < nlength \<sigma> \<and> 
               (\<exists>k. 0<k \<and> k \<le> nlength \<sigma> \<and> g((ndropn k \<sigma>)) \<and> (\<forall>j. 0 < j \<and> j < k \<longrightarrow> f((ndropn j \<sigma>))))))"
   show "(\<sigma> \<Turnstile> f \<U> g) " 
    using b by (simp add: until_d_def) (metis i0_lb linorder_cases ndropn_0 not_less_zero zero_enat_def)
qed

lemma UntilNextUntil:
 "\<turnstile> f \<U> g = (g \<or> (f \<and> \<circle>(f \<U> g)))"
by (simp add: UntilNextUntilsem1 Valid_def) 

subsubsection \<open>NotUntilFalse\<close>

lemma NotUntilFalse:
 "\<turnstile> \<not> (f \<U> #False)"
by (simp add: intI until_d_def)

subsubsection \<open>UntilOrDist\<close>

lemma UntilOrDistsem:
 " \<sigma> \<Turnstile> f \<U> (g \<or> h) = ( f \<U> g \<or> f \<U> h) " 
by (auto simp add: until_d_def) 

lemma UntilOrDist:
 "\<turnstile> f \<U> (g \<or> h) = ( f \<U> g \<or> f \<U> h)"
using UntilOrDistsem Valid_def by blast

subsubsection \<open>UntilRightDistOr\<close>

lemma UntilRightDistOr:
 "\<turnstile> f \<U> h \<or> g \<U> h \<longrightarrow> (f \<or> g) \<U> h"
by (auto simp add: Valid_def until_d_def)

subsubsection \<open>UntilLeftDistAnd\<close>

lemma UntilLeftDistAnd:
 "\<turnstile> f \<U> (g \<and> h) \<longrightarrow> f \<U> g \<and> f \<U> h"
by (auto simp add: Valid_def until_d_def)

subsubsection \<open>UntilAndDist\<close>

lemma UntilAndDistsem:
 "\<sigma> \<Turnstile> (f \<and> g) \<U> h = ((f \<U> h) \<and> (g \<U> h))"
 by (auto simp add: until_d_def ) 
   (metis (no_types, lifting) less_le_trans not_less_iff_gr_or_eq order.order_iff_strict) 

lemma UntilAndDist:
 "\<turnstile> (f \<and> g) \<U> h = ((f \<U> h) \<and> (g \<U> h))"
using UntilAndDistsem Valid_def by blast

subsubsection \<open>untilNotImp\<close>

lemma UntilNotImp:
 "\<turnstile> f \<U> g \<and> (\<not> g) \<U> h \<longrightarrow> f \<U> h"
by (simp add: Valid_def until_d_def)
   (metis not_less_iff_gr_or_eq order.strict_trans)

subsubsection \<open>UntilUntil\<close>

lemma UntilUntilsem:
 "(\<sigma> \<Turnstile> f \<U> g)  = (\<sigma> \<Turnstile> f \<U> (f \<U> g)) "
proof auto
 show "\<sigma> \<Turnstile> (f \<U> g)  \<Longrightarrow> \<sigma> \<Turnstile> (f \<U> (f \<U> g))  " 
   by (simp add: until_d_def) 
      (metis enat_add_sub_same enat_le_plus_same(1) enat_ord_code(4) enat_ord_simps(4) gen_nlength_def 
       ndropn_0 nlength_code not_less_zero)
 show " \<sigma> \<Turnstile> (f \<U> (f \<U> g))  \<Longrightarrow> \<sigma> \<Turnstile> (f \<U> g)  " 
   proof -
    assume a: "\<sigma> \<Turnstile> (f \<U> (f \<U> g))" 
    show   "\<sigma> \<Turnstile> (f \<U> g)" 
     proof -
      have 1: "\<exists>k. enat k \<le> nlength \<sigma> \<and>
               (\<exists>ka. enat ka \<le> nlength (ndropn k \<sigma>) \<and> g (ndropn ka (ndropn k \<sigma>)) \<and>
                     (\<forall>j<ka. f (ndropn j (ndropn k \<sigma>)))) \<and>
               (\<forall>j<k. f (ndropn j \<sigma>))" 
        using a unfolding until_d_def by blast
      obtain k where 2: 
        "enat k \<le> nlength \<sigma> \<and>
          (\<exists>ka. enat ka \<le> nlength (ndropn k \<sigma>) \<and> g (ndropn ka (ndropn k \<sigma>)) \<and> 
            (\<forall>j<ka. f (ndropn j (ndropn k \<sigma>)))) \<and>
           (\<forall>j<k. f (ndropn j \<sigma>))" 
        using 1 by auto
      have 3: "(\<exists>ka. enat ka \<le> nlength (ndropn k \<sigma>) \<and> g (ndropn ka (ndropn k \<sigma>)) \<and> 
                  (\<forall>j<ka. f (ndropn j (ndropn k \<sigma>))))" 
        using 2 by auto
      obtain ka where 4: 
       "enat ka \<le> nlength (ndropn k \<sigma>) \<and> g (ndropn ka (ndropn k \<sigma>)) \<and> 
        (\<forall>j<ka. f (ndropn j (ndropn k \<sigma>)))"
        using 3 by auto
      have 41: " enat ka \<le> nlength \<sigma> - (enat k)"
        using "4" by auto
      have 5: " enat (ka+k) \<le> nlength \<sigma>"  
        using 2 41 by auto 
             (metis add.commute antisym_conv2 enat.simps(3) enat_add_sub_same enat_min le_iff_add 
              less_imp_le order_refl plus_enat_simps(1))
      have 6: "g (ndropn (ka+k) \<sigma>)" 
        by (metis "4" add.commute ndropn_ndropn)
      have 7: "(\<forall>j<(ka+k). f (ndropn j \<sigma>)) " 
        by (metis "2" "4" add_diff_inverse_nat less_diff_conv2 linorder_not_less ndropn_ndropn)
      have 8: "\<exists>k. k \<le> nlength \<sigma> \<and> g (ndropn k \<sigma>) \<and> (\<forall>j<k. f (ndropn j \<sigma>)) "
        using 5 6 7 by blast
      show ?thesis unfolding until_d_def by (simp add: "8") 
     qed
   qed
qed

lemma UntilUntil: 
"\<turnstile> f \<U> g = f \<U> (f \<U> g)"
using UntilUntilsem by fastforce

subsubsection \<open>UntilRightor\<close>

lemma UntilRightOr:
 "\<turnstile> f \<U> ( g \<U> h ) \<longrightarrow> (f \<or> g) \<U> h"
proof (auto simp add: Valid_def until_d_def ndropn_ndropn)
  fix w :: "'a nellist"  
  fix k 
  fix ka
  assume a0: "enat k \<le> nlength w" 
  assume a1: "\<forall>j<k. f (ndropn j w)" 
  assume a2: "enat ka \<le> nlength w - enat k"
  assume a3: "h (ndropn (k + ka) w)"  
  assume a4: "\<forall>j<ka. g (ndropn (k + j) w)" 
  show "\<exists>k. enat k \<le> nlength w \<and> h (ndropn k w) \<and> (\<forall>j<k. f (ndropn j w) \<or> g (ndropn j w)) " 
  proof -
   have 1: "ka+k \<le> nlength w"
     by (metis a0 a2 add.commute dual_order.order_iff_strict enat.simps(3) enat_add_sub_same 
         enat_less_enat_plusI2 less_eqE plus_enat_simps(1)) 
   have 2: "h (ndropn (ka+k) w)"
     using  a3 by (simp add: add.commute)
   have 3: "(\<forall>j<(ka+k). f (ndropn j w) \<or> g (ndropn j w))" 
     by (metis a1 a4 add_diff_inverse_nat less_diff_conv2 not_less)
   show ?thesis using 1 2 3 by blast
  qed
qed



subsubsection \<open> UntilRightAnd\<close>


lemma UntilRightAndsem:
assumes "(\<sigma> \<Turnstile> f \<U> (g \<and> h))"
shows   "(\<sigma> \<Turnstile> (f \<U> g) \<U> h)"
proof -
 have 1: "\<exists>k. k\<le>nlength \<sigma> \<and> g ((ndropn k \<sigma>)) \<and> h ((ndropn k \<sigma>)) \<and> (\<forall>j<k. f ((ndropn j \<sigma>)))"
   using assms by (simp add: until_d_def) 
 obtain k where 2: "k\<le>nlength \<sigma> \<and> g ((ndropn k \<sigma>)) \<and> h ((ndropn k \<sigma>)) \<and> (\<forall>j<k. f ((ndropn j \<sigma>)))"
   using "1" by auto
 have 3: "h ( (ndropn k \<sigma>))"
   using "2" by auto
 have 4: "k\<le>nlength \<sigma>"  
   using "2" by auto
 have 5: "(\<forall>j<k. 
           \<exists>ka. ka\<le>nlength (ndropn j \<sigma>) \<and> g ((ndropn (ka + j) \<sigma>)) \<and> (\<forall>ja<ka. f ((ndropn (ja + j) \<sigma>))))"
   proof auto
      fix j
      assume a0: "j < k"
      show " \<exists>ka. enat ka \<le> nlength \<sigma> - enat j \<and> g (ndropn (ka + j) \<sigma>) \<and> (\<forall>ja<ka. f (ndropn (ja + j) \<sigma>))"  
      proof -
       have 51: "k-j \<le> nlength (ndropn j \<sigma>)"
         using "4" a0 
         by (metis enat_minus_mono1 idiff_enat_enat ndropn_nlength)
       have 52: " g ( (ndropn ((k-j) + j) \<sigma>))" 
         by (simp add: "2" a0 less_imp_le_nat)
       have 53: "(\<forall>ja<(k-j). f ( (ndropn (ja + j) \<sigma>)))" 
         using "2" less_diff_conv by blast
       show ?thesis
         using "51" "52" "53" by auto 
      qed
   qed
 have 6: "\<exists>k. k\<le>nlength \<sigma> \<and>
       h ( (ndropn k \<sigma>)) \<and>
       (\<forall>j<k. \<exists>k\<le>nlength (ndropn j \<sigma>). g ((ndropn (k + j) \<sigma>)) \<and> (\<forall>ja<k. f ((ndropn (ja + j) \<sigma>))))" 
   using "2" "5" by blast
 from 6 show ?thesis by (simp add: until_d_def ndropn_ndropn add.commute) 
qed

lemma UntilRightAnd:
 "\<turnstile> f \<U> (g \<and> h) \<longrightarrow> (f \<U> g) \<U> h"
using UntilRightAndsem Valid_def by auto


subsubsection \<open>DiamondEqvTrueUntil\<close>

lemma DiamondEqvTrueUntil:
 "\<turnstile> \<diamond> f = #True \<U> f" 
by (simp add: Valid_def sometimes_defs until_d_def)

subsubsection \<open>TrueUntilImpNotUntil\<close>

lemma nellist_ndropn_first_upto:
assumes "(\<exists>i\<le> k. f ( (ndropn i xs)))"
shows   "(\<exists>i\<le> k. f ( (ndropn i xs)) \<and> (\<forall>j<i . \<not> (f ( (ndropn j xs)))))"
using assms 
proof (induct k arbitrary: xs)
case 0
then show ?case by simp
next
case (Suc k)
then show ?case 
by (metis le_Suc_eq less_Suc_eq_le)
qed

lemma nellist_ndropn_first:
assumes "(\<exists>i\<le>nlength xs. f ( (ndropn i xs)))"
shows   "(\<exists>i\<le>nlength xs. f ( (ndropn i xs)) \<and> (\<forall>j. j<i \<longrightarrow> \<not> (f ( (ndropn j xs)))))"
proof (cases "nfinite xs")
case True
then show ?thesis using assms nellist_ndropn_first_upto[of _ f xs] nfinite_nlength_enat[of xs] 
   by force
next
case False
then show ?thesis using assms nellist_ndropn_first_upto[of _ f xs]
proof -
assume a1: "\<And>k. \<exists>i\<le>k. f (ndropn i xs) \<Longrightarrow> \<exists>i\<le>k. f (ndropn i xs) \<and> (\<forall>j<i. \<not> f (ndropn j xs))"
obtain nn :: nat where
f2: "f (ndropn nn xs) \<and> enat nn \<le> nlength xs"
using assms by blast
then have "\<forall>e. \<not> e \<le> enat nn \<or> e \<le> nlength xs"
by force
then show ?thesis
using f2 a1 by (meson enat_ord_simps(1) less_imp_le_nat)
qed 
qed

lemma NotSuffixFirstfinite:
assumes "(\<exists>n\<le>nlength xs. \<not> f ( (ndropn n xs)))"
shows   "(\<exists>n \<le> nlength xs. \<not> f ( (ndropn n xs)) \<and> (\<forall>k. k<n \<longrightarrow> f ( (ndropn k xs))))"
using assms nellist_ndropn_first[of "xs" "LIFT(\<not> f)"] by auto  

lemma TrueUntilImpNotUntilsem:
 assumes " \<sigma>  \<Turnstile> #True \<U> g"
 shows   " \<sigma>  \<Turnstile> (\<not>g) \<U> g"
using assms 
by (simp add: until_d_def nellist_ndropn_first)

lemma TrueUntilImpNotUntil:
"\<turnstile> #True \<U> g \<longrightarrow> (\<not>g) \<U> g"
  by (simp add: intI nellist_ndropn_first   until_d_def)

subsubsection \<open>WaitNotDistUntil\<close>

lemma WaitNotDistUntilsem1:
assumes "(\<sigma> \<Turnstile> \<not>(f \<W> g))"
shows   "( \<sigma> \<Turnstile> ((\<not> g) \<U> ((\<not> f) \<and> (\<not>g))))"
proof -
 have 1: "(\<forall>k. g ( (ndropn k \<sigma>)) \<longrightarrow> k \<le> nlength \<sigma> \<longrightarrow> (\<exists>j<k. \<not> f ( (ndropn j \<sigma>)))) \<and>
          (\<exists>n\<le>nlength \<sigma>. \<not> f ( (ndropn n \<sigma>)))"
   using assms by (simp add: wait_d_def until_d_def always_defs)
 have 2: "(\<forall>k. k \<le> nlength \<sigma> \<longrightarrow> \<not> g ( (ndropn k \<sigma>)) \<or> (\<exists>j<k. \<not> f ( (ndropn j \<sigma>))))"
   using "1" by auto
 have 3: "(\<exists>n\<le>nlength \<sigma>. \<not> f ( (ndropn n \<sigma>)))"  
   using "1" by auto
 obtain n where 4: "n\<le> nlength \<sigma> \<and> \<not> f ( (ndropn n \<sigma>)) \<and>
                    (\<forall> k <n . f ( (ndropn k \<sigma>)))"
   using "3" using NotSuffixFirstfinite by blast
 have 16: "n \<le> nlength \<sigma>"
   by (simp add: "4") 
 have 17: "\<not> g ( (ndropn n \<sigma>))" 
   using "1" "4" by blast
 have 18: "(\<forall>j<n. \<not> g ( (ndropn j \<sigma>)))"
   by (metis "2" "4" dual_order.strict_iff_order dual_order.strict_trans1 enat_ord_simps(2)) 
 have 19: "\<exists>k\<le>nlength \<sigma>. \<not> f ( (ndropn k \<sigma>)) \<and> \<not> g ( (ndropn k \<sigma>)) \<and> (\<forall>j<k. \<not> g ( (ndropn j \<sigma>)))"
   using 16 17 18 4 by blast
 have 20: "( \<sigma> \<Turnstile> ((\<not> g) \<U> (\<not> f \<and> \<not>g)))" 
   using "19" by (simp add: until_d_def)
 show ?thesis using "20" by auto 
qed

lemma WaitNotDistUntilsem2:
assumes "(\<sigma> \<Turnstile> ((\<not> g) \<U> ((\<not> f) \<and> (\<not>g))))"
shows   "( \<sigma> \<Turnstile> \<not>(f \<W> g))"
using assms not_less_iff_gr_or_eq by (auto simp add: always_defs wait_d_def until_d_def)

lemma WaitNotDistUntilsem:
 "(\<sigma> \<Turnstile> (\<not>(f \<W> g))  =  ((\<not> g) \<U> ((\<not> f) \<and> (\<not>g)))) " 
using WaitNotDistUntilsem1 WaitNotDistUntilsem2 
unl_lift2 by blast

lemma WaitNotDistUntil:
 "\<turnstile> (\<not>(f \<W> g)) = ((\<not> g) \<U> (\<not> f \<and> \<not>g))"
using WaitNotDistUntilsem   Valid_def by (metis )

subsubsection \<open>UntilInduction\<close>

lemma LFPUntilsem1:
assumes "\<forall>n\<le>nlength \<sigma>.
            (g ( (ndropn n \<sigma>)) \<longrightarrow> h ( (ndropn n \<sigma>))) \<and>
            (f ( (ndropn n \<sigma>)) \<and> n < nlength \<sigma> \<and> h ( (ndropn (Suc n) \<sigma>)) \<longrightarrow>
             h ( (ndropn n \<sigma>)))"
         "k \<le> nlength \<sigma>"
         "g ( (ndropn k \<sigma>)) " 
         "\<forall>j<k. f ( (ndropn j \<sigma>))"
shows     " h ( \<sigma>)"
using assms
proof (induct k arbitrary: \<sigma> )
case 0
then show ?case by auto
next
case (Suc k)
then show ?case
    proof -
     have 1: "g (ndropn (Suc k) \<sigma>) \<longrightarrow> h  (ndropn (Suc k ) \<sigma>) "
       using Suc.prems(1) Suc.prems(2) by blast
     have 2: "(f ( (ndropn k \<sigma>)) \<and> k < nlength \<sigma> \<and> h ( (ndropn (Suc k) \<sigma>)) \<longrightarrow>
             h ( (ndropn k \<sigma>)))" 
       by (simp add: Suc.prems(1) less_le_not_le)
     have 3: "k < nlength \<sigma>"
       using Suc.prems(2) Suc_ile_eq by auto 
     have 4: "f ( (ndropn k \<sigma>))"
       by (simp add: Suc.prems(4)) 
     have 5: "h ( (ndropn k \<sigma>))"
       using "1" "2" "3" "4" Suc.prems(3) by blast         
     have 6: " h ( (ndropn 0 \<sigma>)) "   
       using zero_induct[of "\<lambda>k . h (ndropn k  \<sigma>)" k] 
       3 5 Suc.prems nellist_ndropn_first[of \<sigma> h]
       by (metis Suc_ile_eq less_Suc_eq_0_disj less_imp_le not_less_eq)          
     show ?thesis 
       using "6" by auto      
    qed
qed

lemma LFPUntilsem:
  "\<sigma> \<Turnstile> \<box>((g \<or> (f \<and> \<circle>h)) \<longrightarrow> h) \<longrightarrow> (f \<U> g \<longrightarrow> h)"
using LFPUntilsem1[of \<sigma> g h f]  
by (auto simp add: always_defs next_defs until_d_def ndropn_ndropn)
   (metis canonically_ordered_monoid_add_class.lessE enat.simps(3) enat_add_sub_same zero_enat_def)

lemma LFPUntil:
  "\<turnstile> \<box>((g \<or> (f \<and> \<circle>h)) \<longrightarrow> h) \<longrightarrow> (f \<U> g \<longrightarrow> h)"
using LFPUntilsem  Valid_def 
by (metis )

lemma UntilInduction_a:
 "\<turnstile> \<box>(f \<longrightarrow> ((\<circle> f) \<and> g) \<or> h) \<longrightarrow> (f \<longrightarrow> \<box> g \<or> g \<U> h)"
proof -
 have 1: "\<turnstile> (\<box> g \<or> g \<U> h) = g \<W> h "
   by (auto simp add: wait_d_def)
 have 2: "\<turnstile> (f \<longrightarrow> \<box> g \<or> g \<U> h) = ( (\<not> h) \<U> (\<not> g \<and> \<not>h) \<longrightarrow> \<not> f)"
   using "1" WaitNotDistUntil by fastforce 
 have 3: "\<turnstile> \<box>( ((\<not>g \<and> \<not>h) \<or> (\<not>h \<and> \<circle>(\<not> f)))\<longrightarrow> \<not>f) \<longrightarrow> ( (\<not> h) \<U> (\<not> g \<and> \<not>h) \<longrightarrow> \<not> f)  "
   using LFPUntil by blast
 have 4: "\<turnstile> (f \<longrightarrow> ((\<circle> f) \<and> g) \<or> h) \<longrightarrow> ( ((\<not>g \<and> \<not>h) \<or> (\<not>h \<and> \<circle>(\<not> f)))\<longrightarrow> \<not>f)"
   using NextImpNotNextNot[of f] by auto
 have 5: "\<turnstile> \<box>(f \<longrightarrow> ((\<circle> f) \<and> g) \<or> h) \<longrightarrow> \<box>( ((\<not>g \<and> \<not>h) \<or> (\<not>h \<and> \<circle>(\<not> f)))\<longrightarrow> \<not>f)" 
   using "4" by (rule ImpBoxRule)
 show ?thesis 
   using "2" "3" "5" by fastforce
qed

lemma UntilInduction_b:
 "\<turnstile> \<box>(f \<longrightarrow> (\<circle>f) \<or> g) \<longrightarrow> (f \<longrightarrow> \<box> f \<or> f \<U> g)"
proof -
 have 1: "\<turnstile> (\<box> f \<or> f \<U> g) = f \<W> g"
   by (auto simp add: wait_d_def)
 have 2: "\<turnstile> (f \<longrightarrow> \<box> f \<or> f \<U> g) = ( (\<not> g) \<U> (\<not> f \<and> \<not> g) \<longrightarrow> \<not> f)"
   using "1" WaitNotDistUntil by fastforce 
 have 3: "\<turnstile> \<box> ( ((\<not> f \<and> \<not> g) \<or> (\<not>g \<and> \<circle> (\<not> f))) \<longrightarrow> \<not> f) \<longrightarrow> ( (\<not> g) \<U> (\<not> f \<and> \<not> g) \<longrightarrow> \<not> f)"
   using LFPUntil by blast
 have 4: "\<turnstile>(f \<longrightarrow> (\<circle>f) \<or> g) \<longrightarrow>   ( ((\<not> f \<and> \<not> g) \<or> (\<not>g \<and> \<circle> (\<not> f))) \<longrightarrow> \<not> f)"
   using  NextImpNotNextNot[of "f"] by auto
 have 5: "\<turnstile> \<box>(f \<longrightarrow> (\<circle>f) \<or> g) \<longrightarrow>   \<box>( ((\<not> f \<and> \<not> g) \<or> (\<not>g \<and> \<circle> (\<not> f))) \<longrightarrow> \<not> f)"
   using "4" BoxImpBoxRule by blast
 show ?thesis 
  using "2" "3" "5" by fastforce
qed   


subsection \<open>Theorems\<close>

lemma NextFalseSUntil:
 "\<turnstile> \<circle> g = #False \<U>\<^sup>s g"
proof -
 have 1: "\<turnstile> #False \<U> g = g"
   using UntilNextUntil[of "LIFT(#False)" g]  by auto
 show ?thesis unfolding suntil_d_def using "1" inteq_reflection by force
qed

lemma WNextUntil:
 "\<turnstile> wnext(f \<U> g) = (empty \<or> (\<circle> f) \<U> (\<circle> g))"
by (meson NextUntil Prop06 WnextEqvEmptyOrNext)

lemma UntilRelease:
 "\<turnstile> f \<R> g = (\<not> ( (\<not> f) \<U> (\<not> g)))"
by (simp add: release_d_def)

lemma SReleaseWait:
 "\<turnstile> f \<M> g = (\<not> (\<not> f) \<W> (\<not> g))" 
by (simp add: srelease_d_def)

lemma ReleaseUntil:
 "\<turnstile> f \<U> g = (\<not> ( (\<not> f) \<R> (\<not> g)))"
by (simp add: release_d_def)

lemma WaitSRelease:
 "\<turnstile> f \<W> g = (\<not> (\<not> f) \<M> (\<not> g))"
by (simp add: srelease_d_def)

lemma NotUntilRelease:
 "\<turnstile> \<not>(f \<U> g) = (\<not> f) \<R> (\<not> g)"
by (simp add: ReleaseUntil)

lemma NotWaitSRelease:
 "\<turnstile> \<not>(f \<W> g) = (\<not> f) \<M> (\<not> g)"
by (simp add: WaitSRelease)

lemma NotReleaseUntil:
 "\<turnstile> \<not>(f \<R> g) = (\<not> f) \<U> (\<not> g)"
by (simp add: UntilRelease)

lemma NotSReleaseWait:
 "\<turnstile> \<not>(f \<M> g) = (\<not> f) \<W> (\<not> g)"
by (simp add: SReleaseWait)

lemma BoxEqvFalseRelease:
 "\<turnstile> \<box> f = #False \<R> f"
unfolding release_d_def 
by (metis DiamondEqvTrueUntil Prop11 always_d_def int_simps(3) inteq_reflection lift_imp_neg)

lemma UntilTrue:
 " \<turnstile> f \<U> #True"
using UntilNextUntil by fastforce

lemma UntilIdempotent:
 "\<turnstile> f \<U> f = f"
using UntilNextUntil[of f f] by auto 

lemma UntilImpUntil:
 assumes "\<turnstile> f0 \<longrightarrow> f1"
         "\<turnstile> g0 \<longrightarrow> g1"
 shows   "\<turnstile> f0 \<U> g0 \<longrightarrow> f1 \<U> g1"
using assms 
by (metis Prop10 Prop12 UntilAndDist UntilLeftDistAnd int_eq)

lemma UntilEqvUntil:
 assumes "\<turnstile> f0 = f1"
         "\<turnstile> g0 = g1"
 shows   "\<turnstile> f0 \<U> g0 = f1 \<U> g1"
proof -
 have 1: "\<turnstile> f0 \<longrightarrow> f1"
   using assms by auto
 have 2: "\<turnstile> g0 \<longrightarrow> g1"
   using assms by auto
 have 3: "\<turnstile> f0 \<U> g0 \<longrightarrow> f1 \<U> g1" 
   using 1 2 UntilImpUntil[of "f0" "f1" "g0" "g1"] by auto
 have 4: "\<turnstile> f1 \<longrightarrow> f0"
   using assms by auto
 have 5: "\<turnstile> g1 \<longrightarrow> g0"
   using assms by auto  
 have 6: "\<turnstile> f1 \<U> g1 \<longrightarrow> f0 \<U> g0"
   using 4 5 UntilImpUntil[of "f1" "f0" "g1" "g0"] by auto
 from 3 6 show ?thesis by fastforce
qed

lemma UntilRightDistImp:
 "\<turnstile> (f \<longrightarrow> g) \<U> h \<longrightarrow> (f \<U> h \<longrightarrow> g \<U> h)"
proof -
 have 1: "\<turnstile> (f \<longrightarrow> g) \<U> h \<longrightarrow> (f \<U> h \<longrightarrow> g \<U> h) =
            ((f \<longrightarrow> g) \<U> h \<and> f \<U> h \<longrightarrow> g \<U> h)" 
   by auto
 have 2: "\<turnstile> ((f \<longrightarrow> g) \<U> h \<and> f \<U> h) = ((f \<longrightarrow> g) \<and> f) \<U> h "
   by (simp add: UntilAndDist int_iffD1 int_iffD2 int_iffI)
 have 3: "\<turnstile> ((f \<longrightarrow> g) \<and> f) = (f \<and> g)"
   by auto
 have 4: "\<turnstile> h = h"
   by auto 
 have 5: "\<turnstile> ((f \<longrightarrow> g) \<and> f) \<U> h = (f \<and> g) \<U> h"
   using "3" "4" using UntilEqvUntil by blast
 have 6: "\<turnstile> (f \<and> g) \<U> h = (f \<U> h \<and> g \<U> h)"
   by (simp add: UntilAndDist) 
 show ?thesis 
   using "2" "5" "6" by fastforce
qed    

lemma FalseUntil:
 "\<turnstile> #False \<U> g = g"
 by (metis Prop10 Prop12 TrueW UntilNextUntil int_simps(14) int_simps(21) int_simps(25) int_simps(3)
      inteq_reflection)

lemma UntilExclMid:
 "\<turnstile> f \<U> g \<or> f \<U> (\<not> g)"
 using UntilOrDist UntilTrue by fastforce

lemma NotUntilImp:
 "\<turnstile> (\<not> f) \<U> (g \<U> h) \<and> f \<U> h \<longrightarrow> g \<U> h"
proof -
 have 1: "\<turnstile> (\<not> f) \<U> (g \<U> h) \<longrightarrow> (\<not> f \<or> g) \<U> h"
   by (simp add: UntilRightOr)
 have 2: "\<turnstile> (\<not> f \<or> g) = (f \<longrightarrow> g)" 
   by auto
 have 3: "\<turnstile> h = h"
   by auto
 have 4: "\<turnstile>  (\<not> f \<or> g) \<U> h = (f \<longrightarrow> g) \<U> h"
   by (simp add: "2" UntilEqvUntil)
 have 5: "\<turnstile> (f \<longrightarrow> g) \<U> h \<longrightarrow> (f \<U> h \<longrightarrow> g \<U> h)"
   by (simp add: UntilRightDistImp) 
 have 6: "\<turnstile> (\<not> f) \<U> (g \<U> h) \<longrightarrow> (f \<U> h \<longrightarrow> g \<U> h)"
   using "1" "4" "5" by fastforce
 from 6 show ?thesis by auto
qed

lemma UntilNotImpa:
 "\<turnstile> f \<U> ((\<not> g) \<U> h) \<and> g \<U> h \<longrightarrow> f \<U> h"
proof -
 have 1: "\<turnstile> f \<U> ((\<not> g) \<U> h) \<longrightarrow> (f \<or> (\<not> g)) \<U> h "
   by (simp add: UntilRightOr) 
 have 2: "\<turnstile> (f \<or> (\<not> g)) = (g \<longrightarrow> f)"
   by auto
 have 3: "\<turnstile> h = h"
   by auto
 have 4: "\<turnstile> (f \<or> (\<not> g)) \<U> h = (g \<longrightarrow> f) \<U> h"
   by (simp add: "2" UntilEqvUntil) 
 have 5: "\<turnstile> (g \<longrightarrow> f) \<U> h \<longrightarrow> ( g \<U> h \<longrightarrow> f \<U> h)"
   by (simp add: UntilRightDistImp) 
 have 6: "\<turnstile>  f \<U> ((\<not> g) \<U> h) \<longrightarrow> ( g \<U> h \<longrightarrow> f \<U> h)"
  using "1" "4" "5" by fastforce
 from 6 show ?thesis by auto
qed

lemma UntilNotUntilImp:
 "\<turnstile> f \<U> g \<and> (\<not> g) \<U> f \<longrightarrow> f"
proof -
 have 1: "\<turnstile> f \<U> g \<and> (\<not> g) \<U> f \<longrightarrow> f \<U> f"
   using UntilNotImp by auto
 have 2: "\<turnstile>  f \<U> f = f "
   using UntilIdempotent by auto
 from 1 2 show ?thesis by fastforce
qed

lemma AndNotUntilImp:
 "\<turnstile> f \<and> (\<not> f) \<U> g \<longrightarrow> g"
proof -
 have 1: "\<turnstile> f = f \<U> f"
   by (simp add: UntilIdempotent int_iffD1 int_iffD2 int_iffI)
 have 2: "\<turnstile> g =  #False \<U> g"
   by (meson FalseUntil Prop11)
 have 3: "\<turnstile> f \<U> f \<and> (\<not> f) \<U> g \<longrightarrow> #False \<U> g"
   by (metis "1" FalseUntil UntilNotImp inteq_reflection)
 from 1 2 3 show ?thesis by fastforce
qed

lemma UntilImpOr:
 "\<turnstile> f \<U> g \<longrightarrow> f \<or> g"
proof -
 have "\<turnstile> f \<and> \<circle>(f \<U> g) \<longrightarrow> f \<or> g"
   by force 
 then show ?thesis 
   using UntilNextUntil[of f g] by auto
qed

lemma UntilIntro:
 "\<turnstile> g \<longrightarrow> f \<U> g"
proof -
 have 1: "\<turnstile> g = #False \<U> g"
   by (meson FalseUntil Prop11)
 have 2: "\<turnstile> #False \<longrightarrow> f"
   by auto
 have 3: "\<turnstile> g \<longrightarrow> g"
   by auto
 have 4: "\<turnstile> #False \<U> g \<longrightarrow> f \<U> g"
   by (simp add: UntilImpUntil)
 from 1 4 show ?thesis by fastforce
qed

lemma OrImpUntil:
 "\<turnstile> f \<and> g \<longrightarrow> f \<U> g"
by (simp add: Prop01 Prop05 UntilIntro)

lemma UntilAbsorp_a:
 "\<turnstile> (f \<or> f \<U> g) = (f \<or> g)"
proof -
 have 1: "\<turnstile> (f \<or> f \<U> g) \<longrightarrow> f \<or> g"
   using UntilImpOr by fastforce
 have 2: "\<turnstile> f \<or> g \<longrightarrow> (f \<or> f \<U> g)"
   using UntilIntro by fastforce
 from 1 2 show ?thesis by fastforce
qed

lemma UntilAbsorp_b:
 "\<turnstile> (f \<U> g \<or> g) = f \<U> g"
using UntilNextUntil by fastforce

lemma UntilAbsorp_c:
 "\<turnstile> (f \<U> g \<and> g) = g"
using UntilIntro by fastforce

lemma UntilAbsorp_d:
 "\<turnstile> (f \<U> g \<or> (f \<and> g)) = f \<U> g"
using UntilNextUntil by fastforce 

lemma UntilAbsorp_e:
 "\<turnstile> (f \<U> g \<and> (f \<or> g)) = f \<U> g"
by (meson Prop10 Prop11 UntilImpOr)

lemma LeftUntilAbsorp:
 "\<turnstile> f \<U> (f \<U> g) = f \<U> g"
by (meson Prop11 UntilUntil)

lemma RightUntilAbsorp:
 "\<turnstile> (f \<U> g) \<U> g = f \<U> g"
by (metis Prop11 UntilAbsorp_b UntilAbsorp_c UntilImpOr UntilRightAnd UntilUntil inteq_reflection)

lemma UntilAbsorpAndDiamond:
 "\<turnstile> (f \<U> g \<and> \<diamond> g) = f \<U> g"
by (metis DiamondEqvTrueUntil Prop11 Prop12 UntilIdempotent UntilImpUntil int_simps(12) inteq_reflection)

lemma UntilAbsorpOrDiamond:
 "\<turnstile> (f \<U> g \<or> \<diamond> g) = \<diamond> g"
using UntilAbsorpAndDiamond by fastforce

lemma UntilAbsorpDiamond:
 "\<turnstile> f \<U> (\<diamond> g) = \<diamond> g"
using DiamondDiamondEqvDiamond UntilAbsorpOrDiamond UntilAbsorp_b by fastforce

lemma UntilImpDiamond:
 "\<turnstile> f \<U> g \<longrightarrow> \<diamond> g"
using UntilAbsorpAndDiamond by fastforce

lemma AlwaysImpNotUntilNot:
 "\<turnstile> \<box> f \<longrightarrow> \<not>(g \<U> (\<not> f))"
by (simp add: UntilImpDiamond always_d_def)

lemma UntilAlwaysAndDist:
 "\<turnstile> \<box> f \<and> g \<U> h \<longrightarrow> (f \<and> g) \<U> (f \<and> h)"
proof -
 have 1: "\<turnstile> \<box>(h \<or> g \<and> \<circle> ((f \<and> g) \<U> (f \<and> h)) \<longrightarrow> (f \<and> g) \<U> (f \<and> h)) \<longrightarrow> 
               g \<U> h \<longrightarrow> (f \<and> g) \<U> (f \<and> h)" 
   using LFPUntil by blast
 have 2: "\<turnstile>  f \<longrightarrow> (h \<or> g \<and> \<circle> ((f \<and> g) \<U> (f \<and> h)) \<longrightarrow> (f \<and> g) \<U> (f \<and> h))"
   using UntilNextUntil[of "LIFT(f \<and>g)" "LIFT(f \<and> h)"] by auto  
 have 3: "\<turnstile>  \<box> f \<longrightarrow> \<box>(h \<or> g \<and> \<circle> ((f \<and> g) \<U> (f \<and> h)) \<longrightarrow> (f \<and> g) \<U> (f \<and> h))" 
   using 2 BoxImpBoxRule by blast
 have 4: "\<turnstile> \<box> f  \<longrightarrow> (g \<U> h \<longrightarrow> (f \<and> g) \<U> (f \<and> h))"
   using "1" "3" lift_imp_trans by blast
 show ?thesis using 4 by fastforce
qed
 
lemma UntilAndImp:
 "\<turnstile> \<box> f \<and> \<diamond> g \<longrightarrow> f \<U> g"
proof -
have 1: "\<turnstile> \<diamond> g = #True \<U> g"
  by (simp add: DiamondEqvTrueUntil)
have 2: "\<turnstile> \<box> f \<and> #True \<U> g \<longrightarrow> (f \<and> #True) \<U> (f \<and> g)"
  using UntilAlwaysAndDist by blast 
have 3: "\<turnstile> (f \<and> #True) \<U> (f \<and> g) = f  \<U> (f \<and> g)"
  by simp
have 4: "\<turnstile> f  \<U> (f \<and> g) \<longrightarrow> (f \<U> f) \<U> g  " 
  by (simp add: UntilRightAnd)
have 5: "\<turnstile> (f \<U> f) = f "
  by (simp add: UntilIdempotent)
have 6: "\<turnstile> (f \<U> f) \<U> g  = f \<U> g"
  by (simp add: "5" UntilEqvUntil)
show ?thesis 
by (metis "1" "2" "3" "4" "5" inteq_reflection lift_imp_trans)
qed

lemma UntilRightMono:
 "\<turnstile> \<box>(f \<longrightarrow> g) \<longrightarrow> (h \<U> f \<longrightarrow> h \<U> g)"
proof -
 have 1: "\<turnstile> \<box>(f \<longrightarrow> g) \<and> h \<U> f \<longrightarrow> ( (f \<longrightarrow> g) \<and> h) \<U> ((f\<longrightarrow>g) \<and> f)"
   using UntilAlwaysAndDist by blast 
 have 2: "\<turnstile> ((f \<longrightarrow> g) \<and> h) \<U> ((f\<longrightarrow>g) \<and> f)  \<longrightarrow> h \<U> ((f\<longrightarrow>g) \<and> f)"
   by (meson Prop12 UntilImpUntil int_iffD2 lift_and_com)
 have 3: "\<turnstile> ((f\<longrightarrow>g) \<and> f) \<longrightarrow> g"
   by auto
 have 4: "\<turnstile>  h \<U> ((f\<longrightarrow>g) \<and> f) \<longrightarrow> h \<U> g"
   by (simp add: "3" UntilImpUntil)
 show ?thesis 
   by (meson "1" "2" "4" Prop09 lift_imp_trans)
qed

lemma UntilLeftMono:
 "\<turnstile> \<box> (f \<longrightarrow> g) \<longrightarrow> (f \<U> h \<longrightarrow> g \<U> h)"
proof -
 have 1: "\<turnstile> \<box> (f \<longrightarrow> g) \<and> f \<U> h \<longrightarrow> ((f\<longrightarrow>g) \<and> f) \<U> ((f\<longrightarrow>g) \<and>h) "
   by (simp add: UntilAlwaysAndDist)
 have 2: "\<turnstile> ((f\<longrightarrow>g) \<and> f) \<U> ((f\<longrightarrow>g) \<and>h) \<longrightarrow> ((f\<longrightarrow>g) \<and> f) \<U> h"
   by (meson Prop12 UntilLeftDistAnd)
 have 3:"\<turnstile> ((f\<longrightarrow>g) \<and> f) \<longrightarrow> g"
   by auto
 have 4: "\<turnstile> ((f\<longrightarrow>g) \<and> f) \<U> h \<longrightarrow> g \<U> h"
   by (simp add: "3" UntilImpUntil)
 show ?thesis 
   by (meson "1" "2" "4" Prop09 lift_imp_trans)
qed  

lemma UntilCatRule:
" \<turnstile> \<box> ( (f \<longrightarrow> g \<U> h) \<and> (h \<longrightarrow> g \<U> i)) \<longrightarrow> (f \<longrightarrow> (g \<U> i)) "
proof -
 have 1: "\<turnstile> \<box> ( (f \<longrightarrow> g \<U> h) \<and> (h \<longrightarrow> g \<U> i)) \<longrightarrow> \<box> (f \<longrightarrow> g \<U> h)"
   by (metis BoxElim BoxEqvBoxBox BoxImpBoxRule Prop12 inteq_reflection)
 have 2: "\<turnstile> \<box> ( (f \<longrightarrow> g \<U> h) \<and> (h \<longrightarrow> g \<U> i)) \<longrightarrow> \<box> (h \<longrightarrow> g \<U> i) "
   by (metis BoxElim BoxEqvBoxBox BoxImpBoxRule Prop12 inteq_reflection)
 have 3: "\<turnstile> \<box> (h \<longrightarrow> g \<U> i) \<longrightarrow> \<box> ( g \<U> h \<longrightarrow> g \<U> (g \<U> i))"
   by (metis BoxEqvBoxBox BoxImpBoxRule UntilRightMono inteq_reflection) 
 have 4: "\<turnstile> \<box> ( g \<U> h \<longrightarrow> g \<U> (g \<U> i)) \<longrightarrow> \<box>( g \<U> h \<longrightarrow> g \<U> i)"
   by (metis BoxEqvBoxBox UntilUntil int_iffD1 inteq_reflection)
 have 5: "\<turnstile>  \<box> (f \<longrightarrow> g \<U> h) \<longrightarrow> (f \<longrightarrow> g \<U> h)"
   by (simp add: BoxElim)
 have 6: "\<turnstile> \<box>( g \<U> h \<longrightarrow> g \<U> i) \<longrightarrow> (g \<U> h \<longrightarrow> g \<U> i)"
   by (simp add: BoxElim)
 have 7: "\<turnstile> (f \<longrightarrow> g \<U> h) \<and> (g \<U> h \<longrightarrow> g \<U> i) \<longrightarrow> (f \<longrightarrow> g \<U> i)"
   by auto
 have 8: "\<turnstile> \<box> ( (f \<longrightarrow> g \<U> h) \<and> (h \<longrightarrow> g \<U> i)) \<longrightarrow> 
            (f \<longrightarrow> g \<U> h) \<and> (g \<U> h \<longrightarrow> g \<U> i) "
   using "1" "2" "3" "4" "5" "6" by fastforce
 from 7 8 show ?thesis by auto
qed
 
lemma UntilStrengthen:
 "\<turnstile> \<box> ( (f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> (f \<U> g \<longrightarrow> h \<U> i)"
proof -
 have 11: "\<turnstile> \<box> ( (f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> \<box>(f \<longrightarrow> h)"
   by (meson BoxImpBoxRule Prop12 int_iffD2 lift_and_com)
 have 1: "\<turnstile> \<box> ( (f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> (f \<U> g \<longrightarrow> h \<U> g)"
   using 11 UntilLeftMono[of "f" "h" "g"]    by fastforce
 have 21: "\<turnstile> \<box> ( (f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> \<box>(g \<longrightarrow> i)"
   by (simp add: BoxImpBoxRule Prop01 Prop05)
 have 2: "\<turnstile> \<box> ( (f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> ( h \<U> g \<longrightarrow> h \<U> i) "
   using 21 UntilRightMono[of "g" "i" "h"]  by fastforce 
 have 3: "\<turnstile> \<box> ( (f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> (f \<U> g \<longrightarrow> h \<U> g) \<and> ( h \<U> g \<longrightarrow> h \<U> i)"
   using "1" "2" by fastforce
 have 4: "\<turnstile> (f \<U> g \<longrightarrow> h \<U> g) \<and> ( h \<U> g \<longrightarrow> h \<U> i) \<longrightarrow> (f \<U> g \<longrightarrow> h \<U> i)"
   by auto
 from 3 4 show ?thesis by auto
qed

lemma UntilInduction:
 "\<turnstile> \<box> (f \<longrightarrow> \<not> g \<and> \<circle> f) \<longrightarrow> (f \<longrightarrow> \<not>(h \<U> g))"
proof -
 have 1: "\<turnstile> \<box> (\<not> g) \<longrightarrow> \<not>(h \<U> g)"
   by (simp add: UntilImpDiamond always_d_def)
 have 15: "\<turnstile> (f \<longrightarrow> \<not> g \<and> \<circle> f) \<longrightarrow> (g \<or>  \<circle> (\<not> f) \<longrightarrow> \<not> f)"
   using NextImpNotNextNot[of "f"] by fastforce
 have 16: "\<turnstile> (f \<longrightarrow> \<not> g \<and> \<circle> f) \<longrightarrow> (g \<or>  #True \<and> \<circle> (\<not> f) \<longrightarrow> \<not> f)"
   using 15 by auto
 have 2: "\<turnstile> \<box> (f \<longrightarrow> \<not> g \<and> \<circle> f) \<longrightarrow> \<box>(g \<or> #True \<and> \<circle> (\<not> f) \<longrightarrow> \<not> f)"
   using 16 BoxImpBoxRule by blast
 have 3: "\<turnstile> \<box> (f \<longrightarrow> \<not> g \<and> \<circle> f) \<longrightarrow> (#True \<U> g \<longrightarrow> \<not> f)"
   using "2" LFPUntil[of "g" "LIFT(#True)" "LIFT(\<not> f)"]  
   by fastforce
 have 4: "\<turnstile> (#True \<U> g \<longrightarrow> \<not> f) \<longrightarrow> (f \<longrightarrow> \<not>(#True \<U> g))"
   by auto
 have 5: "\<turnstile>  \<not>(#True \<U> g) = \<box> (\<not> g)"
   using BoxEqvFalseRelease NotUntilRelease inteq_reflection by fastforce
 from 5 4 3 1 show ?thesis by fastforce
qed

lemma UntilBoxImp:
 "\<turnstile> f \<U> (\<box> g) \<longrightarrow> \<box>(f \<U> g)"
proof -
 have 1: "\<turnstile> f \<U> \<box>g \<longrightarrow> f \<U> g" 
   by (meson BoxElim BoxGen MP UntilRightMono)
 have 2: "\<turnstile> wnext (f \<U> \<box>g) = (empty \<or> \<circle> (f \<U> \<box>g))"
   by (meson WnextEqvEmptyOrNext) 
 have 3: "\<turnstile> \<box> g = (g \<and> wnext (\<box>g))"
   by (metis (no_types) BoxEqvAndWnextBox) 
 have 4:  "\<turnstile>f \<U> \<box>g = (\<box>g \<or> f \<and> \<circle> (f \<U> \<box>g))"
   by (meson UntilNextUntil) 
 have 5: "\<turnstile> g \<and> wnext (\<box>g) \<longrightarrow> empty \<or> \<circle> (f \<U> \<box>g) "
   by (metis NextUntil Prop01 Prop05 Prop08 UntilIntro WnextEqvEmptyOrNext int_iffD1 
       inteq_reflection)
 have 6: "\<turnstile> more \<and> f \<U> \<box>g \<longrightarrow> empty \<or> \<circle> (f \<U> \<box>g)"
   using 5 3 4 by fastforce
 have 7:  " \<turnstile> more \<and> f \<U> \<box>g \<longrightarrow> \<circle> (f \<U> \<box>g)"
   using 2 6 WnextAndMoreEqvNext by fastforce
 from 1 7 show ?thesis using BoxIntro[of "LIFT(f \<U> (\<box> g))" "LIFT(f \<U> g)"]  
   by auto
qed

lemma UntilBoxEqvBox:
 "\<turnstile> f \<U> (\<box> f) = \<box> f"
proof -
 have 1: "\<turnstile>  f \<U> (\<box> f) \<longrightarrow> \<box>(f \<U> f)"
   using UntilBoxImp[of "f" "f"] by auto
 have 2: "\<turnstile> \<box>(f \<U> f) = \<box> f" 
   by (simp add: BoxEqvBox UntilIdempotent)
 have 3: "\<turnstile> \<box> f \<longrightarrow> f \<U> (\<box> f)" 
   by (simp add: UntilIntro)
 from 1 2 3 show ?thesis by fastforce
qed

lemma UntilRightStrengthen:
 "\<turnstile> f \<U> (g \<and> h) \<longrightarrow> f \<U> (g \<U> h)"
by (meson BoxGen MP OrImpUntil UntilRightMono) 

lemma UntilLeftStrengthen:
 "\<turnstile> (f \<and> g) \<U> h \<longrightarrow> (f \<U> g) \<U>  h"
by (simp add: OrImpUntil UntilImpUntil)

lemma UntilLeftAndOrder:
 "\<turnstile> (f \<and> g) \<U> h \<longrightarrow> f \<U> (g \<U> h)"
by (metis Prop12 UntilIdempotent UntilImpUntil UntilIntro inteq_reflection)

lemma UntilFrameNext:
 "\<turnstile> \<box> f \<longrightarrow> (\<circle> g \<longrightarrow> \<circle> (f \<U> g))"
by (simp add: NextImpNext Prop01 Prop05 Prop09 UntilIntro)

lemma UntilFrameDiamond:
 "\<turnstile> \<box> f \<longrightarrow> (\<diamond> g \<longrightarrow> \<diamond> (f \<U> g))"
by (meson NowImpDiamond Prop09 UntilAndImp lift_imp_trans)

lemma UntilFrameBox:
 "\<turnstile> \<box> f \<longrightarrow> (\<box> g \<longrightarrow> \<box> (f \<U> g))"
by (simp add: BoxAndBoxImpBoxRule OrImpUntil Prop09)

lemma UntilImpNot:
 "\<turnstile> f \<U> g \<longrightarrow> (f \<and> \<not>g) \<U> g"
proof -
 have 1: "\<turnstile> f \<U> g \<longrightarrow> \<diamond> g" 
   by (simp add: UntilImpDiamond)
 have 2: "\<turnstile> \<diamond> g = #True \<U> g"
   by (simp add: DiamondEqvTrueUntil)
 have 3: "\<turnstile> #True \<U> g \<longrightarrow> (\<not>g) \<U> g"
   by (simp add: TrueUntilImpNotUntil)
 have 4: "\<turnstile> (f \<U> g \<and> (\<not>g) \<U> g) = (f \<and> \<not>g) \<U> g"
   using UntilAndDist by fastforce 
 show ?thesis 
   by (meson "1" "2" "3" "4" Prop10 int_iffD1 lift_imp_trans)
qed
 
lemma UntilAndRule:
 "\<turnstile> f \<U> g = (f \<and> \<not>g) \<U> g"
proof -
 have 1: "\<turnstile> (f \<and> \<not>g) \<U> g \<longrightarrow> f \<U> g" 
   using UntilAndDist by fastforce
 show ?thesis by (simp add: "1" UntilImpNot int_iffI)
qed

lemma UntilWait:
 "\<turnstile> f \<U> g = (f \<W> g \<and> \<diamond> g)"
proof -
 have 1: "\<turnstile> f \<U> g \<longrightarrow> f \<W> g \<and> \<diamond> g" 
   by (simp add: Prop05 Prop12 UntilImpDiamond wait_d_def)
 have 2: "\<turnstile> (f \<W> g \<and> \<diamond> g)  = ((\<box> f \<or> f \<U> g) \<and> \<diamond> g) "
   by (auto simp add: wait_d_def)
 have 3: "\<turnstile> ((\<box> f \<or> f \<U> g) \<and> \<diamond> g) = ( (\<box> f \<and> \<diamond>g) \<or> (f \<U> g \<and> \<diamond> g))"
   by auto
 have 4: "\<turnstile> (\<box> f \<and> \<diamond>g) \<longrightarrow> f \<U> g"
   by (simp add: UntilAndImp)
 have 5: "\<turnstile> (f \<U> g \<and> \<diamond> g) \<longrightarrow> f \<U> g"
   by auto
 show ?thesis 
   using "1" "2" "4" by fastforce 
qed

lemma WaitLeftDistAnd:
 "\<turnstile> f \<W> (g \<and> h) \<longrightarrow> f \<W> g \<and> f \<W> h"
proof -
 have 1: "\<turnstile> f \<W> (g \<and> h) \<longrightarrow> f \<W> g" 
   unfolding wait_d_def
   by (metis Prop08 Prop12 UntilAbsorp_a UntilLeftDistAnd int_iffD1 inteq_reflection)
 have 2: "\<turnstile> f \<W> (g \<and> h) \<longrightarrow>  f \<W> h" 
   unfolding wait_d_def
   by (metis Prop08 Prop12 UntilAbsorp_a UntilLeftDistAnd int_iffD1 inteq_reflection)
 show ?thesis by (simp add: "1" "2" Prop12)
qed

lemma WaitRightDistAnd:
 "\<turnstile> (f \<and> g) \<W> h = (f \<W> h \<and> g \<W> h)"
proof -
 have 1: "\<turnstile> \<box>(f \<and> g) = (\<box> f \<and> \<box> g)"
   by (metis BoxAndBoxEqvBoxRule inteq_reflection lift_and_com) 
 have 2: "\<turnstile> (f \<and> g) \<U> h = (f \<U> h \<and> g \<U> h)"
   by (simp add: UntilAndDist) 
 have 3: "\<turnstile> ((\<box> f \<and> \<box> g) ) \<longrightarrow> ((\<box>f \<or> f \<U> h) \<and> (\<box>g \<or> g \<U> h)) "
   by (simp add: intI)
 have 4: "\<turnstile> (f \<U> h \<and> g \<U> h) \<longrightarrow> ((\<box>f \<or> f \<U> h) \<and> (\<box>g \<or> g \<U> h))" 
   by auto
 have 5: "\<turnstile> ((\<box> f \<and> \<box> g) \<or> (f \<U> h \<and> g \<U> h)) \<longrightarrow> ((\<box>f \<or> f \<U> h) \<and> (\<box>g \<or> g \<U> h))" 
   using 3 4 by fastforce
 have 6: "\<turnstile> \<box> f \<and> \<box> g \<longrightarrow> ((\<box> f \<and> \<box> g) \<or> (f \<U> h \<and> g \<U> h))" 
   by auto
 have 7: "\<turnstile> \<box>f \<and> g \<U> h \<longrightarrow> ((\<box> f \<and> \<box> g) \<or> (f \<U> h \<and> g \<U> h))"
   by (metis "2" Prop05 Prop12 UntilAlwaysAndDist UntilLeftDistAnd inteq_reflection lift_imp_trans)
 have 8: "\<turnstile> f \<U> h \<and> \<box> g \<longrightarrow> ((\<box> f \<and> \<box> g) \<or> (f \<U> h \<and> g \<U> h))"
 by (metis Prop05 Prop08 Prop12 UntilAlwaysAndDist UntilAndDist UntilLeftDistAnd inteq_reflection 
     lift_and_com)
 have 9: "\<turnstile> f \<U> h \<and> g \<U> h \<longrightarrow> ((\<box> f \<and> \<box> g) \<or> (f \<U> h \<and> g \<U> h))"
   by auto
 have 10: "\<turnstile> ((\<box>f \<or> f \<U> h) \<and> (\<box>g \<or> g \<U> h)) \<longrightarrow> ((\<box> f \<and> \<box> g) \<or> (f \<U> h \<and> g \<U> h))"
   using 6 7 8 9 by fastforce
 have 11: "\<turnstile> ((\<box> f \<and> \<box> g) \<or> (f \<U> h \<and> g \<U> h)) = ((\<box>f \<or> f \<U> h) \<and> (\<box>g \<or> g \<U> h))"
   using 5 10 by auto
 have 12: "\<turnstile>  (\<box>(f \<and> g) \<or> (f \<and> g) \<U> h) = ((\<box> f \<and> \<box> g) \<or> (f \<U> h \<and> g \<U> h))  "
   using "1" "2" by fastforce 
 show ?thesis unfolding wait_d_def using 11 12 
   by (meson Prop04 UntilIdempotent)
qed

lemma WaitAndRule:
 "\<turnstile> f \<W> g = (f \<and> \<not> g) \<W> g"
proof -
 have 1: "\<turnstile> (f \<W> g \<and> (\<not>g ) \<W> g) = (f \<and> \<not> g) \<W> g " 
   by (meson Prop11 WaitRightDistAnd)
 have 2: "\<turnstile> (\<not>g ) \<W> g"
   by (metis (no_types, opaque_lifting) DiamondEqvTrueUntil FalseUntil UntilAndRule UntilExclMid 
       always_d_def int_simps(17) int_simps(4) inteq_reflection wait_d_def)
 show ?thesis 
   using "1" "2" by fastforce
qed

lemma WaitUntilb:
 "\<turnstile> f \<W>  g = (\<box> ( f \<and> \<not> g) \<or> f \<U> g)"
proof -
 have 1: "\<turnstile> f \<W> g = (f \<and> \<not> g) \<W> g" 
   by (simp add: WaitAndRule)
 have 2: "\<turnstile> (f \<and> \<not> g) \<W> g = (\<box> (f \<and> \<not> g) \<or> (f \<and> \<not> g) \<U> g)"
   by (auto simp add: wait_d_def)
 have 3: "\<turnstile> (f \<and> \<not> g) \<U> g = f \<U> g"
   by (meson Prop11 UntilAndRule)
 show ?thesis 
   using "1" "2" "3" by fastforce
qed

lemma UntilNotDistWait:
 "\<turnstile> (\<not> (f \<U> g)) = (\<not> g) \<W> (\<not> f \<and> \<not> g)"
proof -
 have 1: "\<turnstile> (\<not> ((\<not> g) \<W> (\<not> f \<and> \<not> g))) = (\<not>(\<not> f \<and> \<not> g)) \<U> ( \<not> (\<not> g) \<and> \<not> (\<not> f \<and> \<not> g))"
   using WaitNotDistUntil by blast  
 have 2: "\<turnstile> (\<not>(\<not> f \<and> \<not> g)) = (f \<or> g)"
   by auto
 have 3: "\<turnstile> ( \<not> (\<not> g) \<and> \<not> (\<not> f \<and> \<not> g)) = g"
   by auto
 have 4: "\<turnstile> (\<not>(\<not> f \<and> \<not> g)) \<U> ( \<not> (\<not> g) \<and> \<not> (\<not> f \<and> \<not> g)) =
            (f \<or> g) \<U> g "
   using "2" "3" UntilEqvUntil by blast
 have 5: "\<turnstile> (f \<or> g) \<U> g = ((f \<or> g) \<and> \<not>g) \<U> g"
   by (simp add: UntilAndRule)
 have 6: "\<turnstile> ((f \<or> g) \<and> \<not>g) = (f \<and> \<not> g)"
   by auto
 have 7: "\<turnstile> ((f \<or> g) \<and> \<not>g) \<U> g = (f \<and> \<not>g) \<U> g"
   using "6" inteq_reflection by fastforce 
 have 8: "\<turnstile> (f \<and> \<not>g) \<U> g = f \<U> g"
   by (meson Prop11 UntilAndRule)
 have 9: "\<turnstile> f \<U> g = (\<not> ((\<not> g) \<W> (\<not> f \<and> \<not> g)))"
   using "1" "4" "5" "7" "8" by fastforce
 show ?thesis using "9" by auto
qed   
   
lemma UntilImpWait:
 "\<turnstile> f \<U> g \<longrightarrow> f \<W> g"
by (meson Prop03 WaitUntilb)

lemma WaitAndDist:
 "\<turnstile> (\<box> f \<and> g \<W> h) \<longrightarrow> (f \<and> g) \<W> (f \<and> h)"
proof -
 have 1: "\<turnstile> (\<box> f \<and> g \<W> h) = (\<box> f \<and> (\<box> g \<or> g \<U> h))"
   by (auto simp add: wait_d_def)
 have 2: "\<turnstile> (\<box> f \<and> (\<box> g \<or> g \<U> h)) = ((\<box>f \<and> \<box>g) \<or> (\<box> f \<and> g \<U> h))"
   by auto
 have 3: "\<turnstile> (\<box>f \<and> \<box>g) = \<box>(f \<and> g)"
   by (simp add: BoxAndBoxEqvBoxRule)
 have 4: "\<turnstile> (\<box> f \<and> g \<U> h) \<longrightarrow> (f \<and> g) \<U> (f \<and> h)"
   by (simp add: UntilAlwaysAndDist)
 have 5: "\<turnstile> ((\<box>f \<and> \<box>g) \<or> (\<box> f \<and> g \<U> h)) \<longrightarrow> \<box>(f \<and> g) \<or> (f \<and> g) \<U> (f \<and> h)"
   using "3" "4" by fastforce
 have 6: "\<turnstile> (\<box>(f \<and> g) \<or> (f \<and> g) \<U> (f \<and> h)) = (f \<and> g) \<W> (f \<and> h)"
   by (auto simp add: wait_d_def)
 show ?thesis 
   using "1" "5" "6" by fastforce
qed

lemma WaitDiamondOr:
 "\<turnstile> f \<W> (\<diamond> g) = (\<box> f \<or> \<diamond> g)"
proof -
 have 1: "\<turnstile> f \<W> (\<diamond> g) = (\<box> f \<or> f \<U> (\<diamond> g))"
   by (auto simp add: wait_d_def)
 have 2: "\<turnstile> f \<U> (\<diamond> g) = \<diamond> g"
   by (simp add: UntilAbsorpDiamond)
 show ?thesis using "1" "2" Prop06 by blast
qed

lemma WaitBoxImp:
 "\<turnstile> f \<W> (\<box> g) \<longrightarrow> \<box> ( f \<W> g)"
proof -
 have 1: "\<turnstile> f \<W> (\<box> g) = (\<box> f \<or> f \<U> (\<box> g))"
   by (auto simp add: wait_d_def)
 have 2: "\<turnstile> \<box> f = \<box> (\<box> f)"
   by (simp add: BoxEqvBoxBox)
 have 3: "\<turnstile> f \<U> (\<box> g) \<longrightarrow> \<box>(f \<U> g)"
   by (simp add: UntilBoxImp)
 have 4: "\<turnstile> (\<box> f \<or> f \<U> (\<box> g)) \<longrightarrow> (\<box> (\<box> f) \<or> \<box>(f \<U> g))"
   using "2" "3" by fastforce
 have 5: "\<turnstile> \<box> (\<box> f) \<longrightarrow> \<box>( \<box> f \<or> f \<U> g) "
   by (metis BoxImpBoxRule Prop08 UntilIdempotent UntilIntro int_simps(11) int_simps(25) 
       inteq_reflection)
 have 6: "\<turnstile> \<box>(f \<U> g) \<longrightarrow> \<box>( \<box> f \<or> f \<U> g)"
   by (metis BoxImpBoxRule UntilImpWait wait_d_def) 
 have 7: "\<turnstile> (\<box> (\<box> f) \<or> \<box>(f \<U> g)) \<longrightarrow> \<box>( \<box> f \<or> f \<U> g)"
   using "5" "6" by fastforce
 have 6: "\<turnstile> \<box>( \<box> f \<or> f \<U> g) = \<box> ( f \<W> g)"
   by (simp add: wait_d_def) 
 show ?thesis 
   by (metis "4" "7" lift_imp_trans wait_d_def)
qed

lemma WaitAbsorbBox:
 "\<turnstile> f \<W> (\<box> f) = \<box> f"
by (metis Prop02 Prop11 UntilBoxEqvBox UntilImpWait inteq_reflection wait_d_def)

lemma BoxImpWait:
 "\<turnstile> \<box> f \<longrightarrow> f \<W> g"
by (auto simp add: wait_d_def)

lemma WaitDistNext:
 "\<turnstile> \<circle> ( f \<W> g) = (\<circle> f) \<W> (\<circle> g) "
\<comment> \<open>nitpick finds counterexample, does not hold because of finite intervals\<close>
oops


lemma WaitDistNextInfinite:
 "\<turnstile> inf \<longrightarrow> \<circle> ( f \<W> g) = (\<circle> f) \<W> (\<circle> g) "
proof -
 have 1: "\<turnstile> \<circle> (\<box> f \<or> f \<U> g) \<longrightarrow> (\<circle> (\<box> f) \<or> \<circle>(f \<U> g))"
   by (simp add: ChopOrImpRule next_d_def)
 have 2: "\<turnstile> (\<circle> (\<box> f) \<or> \<circle>(f \<U> g)) \<longrightarrow> \<circle> (\<box> f \<or> f \<U> g)"
   by (metis BoxImpWait NextImpNext Prop02 UntilImpWait wait_d_def)
 have 21: "\<turnstile> \<circle> (\<diamond>(\<not> f)) = \<diamond>(\<circle>(\<not>f))"
   by (metis (no_types, lifting) ChopAssoc FiniteChopSkipEqvSkipChopFinite inteq_reflection 
       next_d_def sometimes_d_def)
 have 22: "\<turnstile> (\<not>( \<circle> f)) = (empty \<or> \<circle>( \<not>f))"
   using WnextEqvEmptyOrNext[of "LIFT(\<not>f)"] unfolding wnext_d_def by simp 
 have 23: "\<turnstile> inf \<longrightarrow> \<circle>(\<not>f) = (\<not>( \<circle> f))"
   using "22" MoreAndInfEqvInf WnextAndMoreEqvNext WnextEqvEmptyOrNext by fastforce
 have 24: "\<turnstile> inf \<longrightarrow>  \<diamond>(\<circle>(\<not>f)) = \<diamond>(\<not>( \<circle> f))" 
   unfolding sometimes_d_def using 23 
   InfRightChopEqvChop[of "LIFT(\<circle>(\<not>f))" "LIFT((\<not> (\<circle> f)))" "LIFT(finite)"]
   by simp
 have 25: "\<turnstile> inf \<longrightarrow> \<circle> (\<diamond>(\<not> f)) = \<diamond>(\<not>( \<circle> f))"
   using 21 24 by fastforce
 have 26: "\<turnstile> inf \<longrightarrow>  (empty \<or> \<circle>(\<diamond>(\<not> f))) = \<circle> (\<diamond>(\<not> f))"
   using MoreAndInfEqvInf WnextAndMoreEqvNext WnextEqvEmptyOrNext by fastforce      
 have 27: "\<turnstile> inf \<longrightarrow> wnext (\<diamond>(\<not> f)) = \<circle> (\<diamond>(\<not> f))"    
   by (metis "26" WnextEqvEmptyOrNext inteq_reflection)
 have 28: "\<turnstile> inf \<longrightarrow> (\<not> (\<circle> (\<not> \<diamond>(\<not> f)))) = \<diamond>(\<not> \<circle> f)" 
   using 21 24 27 unfolding wnext_d_def  by fastforce
 have 3: "\<turnstile> inf \<longrightarrow> \<circle> (\<box> f) = \<box> (\<circle> f)"  
   unfolding always_d_def using 28 by fastforce
 have 4: "\<turnstile> \<circle>(f \<U> g) = \<circle> f \<U> \<circle> g" 
   by (simp add: NextUntil)
 have 5: "\<turnstile> \<circle> (\<box> f \<or> f \<U> g) = (\<circle> (\<box> f) \<or> \<circle>(f \<U> g))"
   using "1" "2" int_iffI by blast
 have 6: "\<turnstile> inf \<longrightarrow> \<circle> ( f \<W> g) = \<circle> (\<box> f \<or> f \<U> g)"
   by (simp add: wait_d_def) 
 have 7: "\<turnstile> inf \<longrightarrow> \<circle> (\<box> f \<or> f \<U> g) = (\<circle> (\<box> f) \<or> \<circle>(f \<U> g))"
   using "5" by auto 
 have 8: "\<turnstile>  inf \<longrightarrow> (\<circle> (\<box> f) \<or> \<circle>(f \<U> g)) = (\<box> (\<circle> f) \<or> \<circle> f \<U> \<circle> g)" 
   using 4 3 by fastforce
 show ?thesis 
   by (metis "5" "8" inteq_reflection wait_d_def) 
qed

lemma WnextAlwaysEqvAlwaysWnext:
  "\<turnstile> finite \<longrightarrow> wnext (\<box> f) = \<box>(wnext f)"
by (metis (mono_tags, lifting) BoxEqvFiniteYields FiniteChopSkipEqvSkipChopFinite 
    SkipYieldsEqvWeakNext YieldsYieldsEqvChopYields intI inteq_reflection unl_lift2)

lemma WaitExpand: 
 "\<turnstile> f \<W> g = (g \<or> (f \<and> \<circle>(f \<W> g)))"
\<comment> \<open> nitpick finds counterexample, does not hold because of finite intervals\<close>
oops

lemma WaitExpand: 
 "\<turnstile> f \<W> g = (g \<or> (f \<and> wnext(f \<W> g)))"
proof -
 have 1: "\<turnstile>  f \<W> g = (\<box> f \<or> f \<U> g)"
   by (simp add: wait_d_def)
 have 2: "\<turnstile> \<box> f = (f \<and> wnext(\<box> f))"
   by (simp add: BoxEqvAndWnextBox)
 have 3: "\<turnstile> f \<U> g = (g \<or> (f \<and> \<circle> (f \<U> g)))"
   using UntilNextUntil by blast 
 have 4: "\<turnstile> (f \<and> wnext(\<box> f)) = (f \<and> (empty \<or> \<circle> (\<box> f)))"
   using "2" BoxEqvAndEmptyOrNextBox[of f]  by fastforce
 have 5: "\<turnstile>  wnext(f \<W> g) = (empty \<or> \<circle> (f \<W> g))"
   using WnextEqvEmptyOrNext by blast
 have 6: "\<turnstile> (f \<and> (empty \<or> \<circle> (\<box> f))) = ( (f \<and> empty) \<or> (f \<and> \<circle> (\<box> f)))"
   by auto
 have 7: "\<turnstile> (( (f \<and> empty) \<or> (f \<and> \<circle> (\<box> f))) \<or>
             (g \<or> (f \<and> \<circle> (f \<U> g))) ) =
             (g \<or> (f \<and> (empty \<or> \<circle> (\<box>f)  \<or> \<circle> (f \<U> g) ) ))"
   by auto
 have 8: "\<turnstile> (\<circle> (\<box>f)  \<or> \<circle> (f \<U> g)) = \<circle>(\<box>f \<or> f \<U> g)"
   by (metis ChopOrEqv Prop11 next_d_def)
 show ?thesis 
   by (metis "1" "2" "3" "4" "5" "6" "7" "8" inteq_reflection)
qed

lemma InfImpWnextEqnNext:
 "\<turnstile> inf \<longrightarrow> wnext f = \<circle> f"
proof -
 have 1: "\<turnstile> wnext f = (empty \<or> \<circle> f)"
   by (simp add: WnextEqvEmptyOrNext)
 have 2: "\<turnstile> inf \<longrightarrow> more"
   using MoreAndInfEqvInf by auto  
 have 3: "\<turnstile> inf \<longrightarrow> (empty \<or> \<circle> f) =  \<circle> f"
    unfolding empty_d_def using "2"  by fastforce
 show ?thesis
   by (metis "1" "3" inteq_reflection)
qed

lemma WaitExpandInfinite: 
 "\<turnstile> inf \<longrightarrow> f \<W> g = (g \<or> (f \<and> \<circle>(f \<W> g)))"
proof -
 have 1: "\<turnstile> f \<W> g = (g \<or> (f \<and> wnext(f \<W> g)))"
   using WaitExpand by blast
 have 2: "\<turnstile> inf \<longrightarrow>  wnext(f \<W> g) = \<circle> (f \<W> g) "
   using InfImpWnextEqnNext by blast
 have 3: "\<turnstile> inf \<longrightarrow> (g \<or> (f \<and> wnext(f \<W> g))) = (g \<or> (f \<and> \<circle>(f \<W> g)))"
   using 2 by auto
 show ?thesis using "1" "3" by fastforce 
qed

lemma WaitExclMid:
 "\<turnstile> f \<W> g \<or> f \<W> (\<not> g)"
using WaitExpand
proof -
 have 1: "\<turnstile> f \<W> g = (g \<or> f \<and> wnext (f \<W> g)) " 
   by (simp add: WaitExpand)
 have 2: "\<turnstile> f \<W> (\<not> g) = ((\<not> g) \<or> f \<and> wnext ( f \<W> (\<not> g)))"
   by (simp add: WaitExpand) 
 have 3: "\<turnstile> (f \<W> g \<or> f \<W> (\<not> g)) = 
            ( (g \<or> f \<and> wnext (f \<W> g)) \<or> ((\<not> g) \<or> f \<and> wnext ( f \<W> (\<not> g)))) "
   using "1" "2" by fastforce
 from 3 show ?thesis by fastforce
qed
 
lemma WaitleftZero:
 "\<turnstile> #True \<W> g"
by (meson BoxGen BoxImpWait MP TrueW)

lemma WaitLeftDistOr:
 "\<turnstile> f \<W> (g \<or> h) = (f \<W> g \<or> f \<W> h)"
proof -
 have 1: "\<turnstile> f \<W> (g \<or> h) = ( \<box> f \<or> f \<U> (g \<or> h))"
   by (simp add: wait_d_def)
 have 2: "\<turnstile> (f \<W> g \<or> f \<W> h) = ((\<box> f \<or> f \<U> g) \<or> (\<box> f \<or> f \<U> h))  "
   by (simp add: wait_d_def)
 have 3: "\<turnstile> f \<U> (g \<or> h) = (f \<U> g \<or> f \<U> h)"
   by (simp add: UntilOrDist) 
 from 1 2 3 show ?thesis by fastforce
qed

lemma WaitRightDistOr:
 "\<turnstile> f \<W> h \<or> g \<W> h \<longrightarrow> (f \<or> g) \<W> h"
proof -
 have 0:"\<turnstile> \<box> g \<longrightarrow> \<box> (f \<or> g)"
   by (simp add: BoxImpBoxRule intI)
 have 1: "\<turnstile> \<box>f \<longrightarrow> \<box> (f \<or> g)"
   by (simp add: BoxImpBoxRule intI)
 have 11: "\<turnstile> \<box> f \<or> \<box> g \<longrightarrow> \<box> (f \<or> g)" 
   using "0" "1" Prop02 by blast
 have 2: "\<turnstile>  f \<W> h = (\<box> f \<or> f \<U> h)"
   by (simp add: wait_d_def)
 have 3:"\<turnstile> (f \<or> g) \<W> h = (\<box> (f \<or> g) \<or> (f \<or> g) \<U> h)"
   by (simp add: wait_d_def)
 have 4: "\<turnstile> g \<W> h = (\<box>g  \<or> g \<U> h)" 
   by (simp add: wait_d_def)
 have 5: "\<turnstile> f \<U> h \<or> g \<U> h \<longrightarrow> (f \<or> g) \<U> h" 
   using UntilRightDistOr by simp
 have 6:"\<turnstile> (f \<W> h \<or> g \<W> h) = ((\<box> f \<or> \<box> g) \<or> (f \<U> h \<or> g \<U> h))" 
   using "2" "4" by fastforce
 from 11 5 6 3 show ?thesis
   by (meson BoxImpWait Prop02 Prop11 UntilImpWait lift_imp_trans)
qed 

lemma WaitOrRule:
 "\<turnstile> f \<W> g = (f \<or> g) \<W> g"
proof -
 have 1: "\<turnstile> f \<W> g \<longrightarrow> (f \<or> g) \<W> g"
   by (metis (no_types, lifting) Prop03 Prop10 UntilAbsorp_a WaitNotDistUntil int_iffD1 int_simps(14) 
       int_simps(32) int_simps(33) inteq_reflection)
 have 2: "\<turnstile> (f \<or> g) \<W> g \<longrightarrow> f \<W> g"
   by (metis (no_types, lifting) Prop03 Prop10 WaitNotDistUntil int_iffD2 int_simps(14) 
       int_simps(32) int_simps(33) inteq_reflection) 
 from 1 2 show ?thesis by fastforce
qed

lemma UntilOrRule:
 "\<turnstile> f \<U> g = (f \<or> g) \<U> g"
by (metis UntilWait WaitOrRule inteq_reflection)
 
lemma WaitRule:
 "\<turnstile> (\<not> f) \<W> f"
by (metis BoxGen BoxImpWait MP WaitOrRule int_eq_true int_simps(29) inteq_reflection)

lemma UntilRule:
 "\<turnstile> (\<not> f) \<U> f = \<diamond> f"
using DiamondEqvTrueUntil UntilOrRule inteq_reflection by fastforce

lemma WaitImpRule:
 "\<turnstile> (f \<longrightarrow> g) \<W> f"
proof -
 have 1: "\<turnstile> (f \<longrightarrow> g) \<W> f = ((f\<longrightarrow>g) \<or>f) \<W> f"
   by (simp add: WaitOrRule)
 have 2: "\<turnstile> (f\<longrightarrow>g) \<or>f"
   by auto
 have 3: "\<turnstile> ((f\<longrightarrow>g) \<or>f) \<W> f = #True \<W> f"
   by (metis "1" "2" int_eq_true inteq_reflection)
 show ?thesis 
   using "1" "3" WaitleftZero by fastforce
qed

lemma DiamondUntilImpRule:
 "\<turnstile> \<diamond> f \<longrightarrow> (f \<longrightarrow> g) \<U> f"
using UntilWait WaitImpRule by fastforce

lemma WaitNotDist:
 "\<turnstile> (\<not> (f \<W> g)) = (f \<and> \<not> g) \<U> (\<not> f \<and> \<not> g)"
proof -
 have 1: "\<turnstile> (\<not> (f \<W> g)) = (\<not> g ) \<U> (\<not> f \<and> \<not> g)"
   using WaitNotDistUntil by blast 
 have 2: "\<turnstile> (\<not> g ) \<U> (\<not> f \<and> \<not> g) = (\<not> g \<and> \<not>(\<not> f \<and> \<not> g)) \<U> (\<not> f \<and> \<not> g)"
   using UntilAndRule by blast
 have 3: "\<turnstile> (\<not> g \<and> \<not>(\<not> f \<and> \<not> g)) = (f \<and> \<not> g)"
   by auto
 have 4: "\<turnstile> (\<not> g \<and> \<not>(\<not> f \<and> \<not> g)) \<U> (\<not> f \<and> \<not> g) = (f \<and> \<not> g) \<U> (\<not> f \<and> \<not> g)"
   using "3" inteq_reflection by force
 show ?thesis using "1" "2" "4" by fastforce
qed

lemma UntilNotDist:
 "\<turnstile> (\<not> (f \<U> g)) = (f \<and> \<not> g) \<W> (\<not> f \<and> \<not> g)"
proof -
 have 1: "\<turnstile> (\<not> (f \<U> g)) = (\<not> g ) \<W> (\<not> f \<and> \<not> g)"
   using UntilNotDistWait by blast
 have 2: "\<turnstile>  (\<not> g ) \<W> (\<not> f \<and> \<not> g) = (\<not> g \<and> \<not>(\<not> f \<and> \<not> g)) \<W> (\<not> f \<and> \<not> g) "
   by (simp add: WaitAndRule)
 have 3: "\<turnstile> (\<not> g \<and> \<not>(\<not> f \<and> \<not> g)) = (f \<and> \<not> g)"
   by auto
 have 4: "\<turnstile> (\<not> g \<and> \<not>(\<not> f \<and> \<not> g)) \<W> (\<not> f \<and> \<not> g) = (f \<and> \<not> g) \<W> (\<not> f \<and> \<not> g)"
   using "3" inteq_reflection by force
 show ?thesis using "1" "2" "4" by fastforce
qed

lemma UntilDuala:
 "\<turnstile> (\<not> ( (\<not> f) \<U> (\<not> g))) = g \<W> (f \<and> g)"
proof -
 have 1: "\<turnstile> (\<not> ( (\<not> f) \<U> (\<not> g))) = (\<not> f \<and> \<not> (\<not> g)) \<W> ( \<not>(\<not> f) \<and> \<not> (\<not> g))"
   using UntilNotDist by blast 
 have 2: "\<turnstile> (\<not> f \<and> g) \<W> ( f \<and> g) = g \<W> ( f \<and> g) "
   using "1" UntilNotDistWait int_eq by fastforce
 show ?thesis 
 using "1" "2" by fastforce
qed

lemma UntilDualb:
 "\<turnstile> (\<not> ( (\<not> f) \<U> (\<not> g))) = (\<not> f \<and> g) \<W> (f \<and> g)"
proof -
 have 1: "\<turnstile> (\<not> ( (\<not> f) \<U> (\<not> g))) = (\<not> f \<and> \<not> (\<not> g)) \<W> ( \<not>(\<not> f) \<and> \<not> (\<not> g))"
   using UntilNotDist by blast 
 show ?thesis 
   using "1" by auto
qed

lemma WaitDuala:
 "\<turnstile> (\<not> ( (\<not> f) \<W> (\<not> g))) = g \<U> (f \<and> g)"
proof -
 have 1: "\<turnstile> (\<not> ( (\<not> f) \<W> (\<not> g))) = (\<not> f \<and> \<not> (\<not> g)) \<U> ( \<not>(\<not> f) \<and> \<not> (\<not> g))"
   using WaitNotDist by blast 
 have 2: "\<turnstile> (\<not> f \<and> g) \<U> ( f \<and> g) = g \<U> ( f \<and> g) "
   using "1" WaitNotDistUntil int_eq by fastforce
 show ?thesis 
   using "1" "2" by fastforce
qed

lemma WaitDualb:
 "\<turnstile> (\<not> ( (\<not> f) \<W> (\<not> g))) = (\<not> f \<and> g) \<U> (f \<and> g)"
proof -
 have 1: "\<turnstile> (\<not> ( (\<not> f) \<W> (\<not> g))) = (\<not> f \<and> \<not> (\<not> g)) \<U> ( \<not>(\<not> f) \<and> \<not> (\<not> g))"
   using WaitNotDist by blast 
 show ?thesis using "1" by auto
qed

lemma WaitIdempotent:
 "\<turnstile> f \<W> f = f"
by (meson BoxElim Prop02 Prop12 UntilIdempotent UntilImpWait UntilIntro WaitUntilb int_iffD1 
    int_iffI lift_imp_trans)

lemma WaitRightZero:
 "\<turnstile> f \<W> #True "
by (meson MP TrueW UntilImpWait UntilIntro)

lemma WaitLeftIdentity:
 "\<turnstile> #False \<W> g = g"
by (metis (no_types, lifting) UntilAbsorp_c UntilNotDistWait WaitDuala WaitIdempotent WaitSRelease 
    int_eq int_simps(17) int_simps(3) srelease_d_def)

lemma WaitImpOr:
 "\<turnstile> f \<W> g \<longrightarrow> f \<or> g"
 by (metis Prop03 WaitIdempotent WaitLeftDistOr WaitOrRule inteq_reflection)

lemma BoxOrImpWait:
 "\<turnstile> \<box>(f \<or> g) \<longrightarrow> f \<W> g"
using BoxImpWait WaitOrRule by fastforce

lemma BoxImpImpWait:
 "\<turnstile> \<box>(\<not> g \<longrightarrow> f) \<longrightarrow> f \<W> g"
proof -
 have 1: "\<turnstile> (\<not> g \<longrightarrow> f) = (f \<or> g)"
   by auto
 have 2: "\<turnstile> \<box> (\<not> g \<longrightarrow> f) = \<box>(f \<or> g)" 
   using "1" BoxEqvBox by blast
 show ?thesis using "2" BoxOrImpWait by fastforce
qed

lemma WaitInsertion:
 "\<turnstile> g \<longrightarrow> f \<W> g"
by (simp add: Prop05 UntilIntro wait_d_def)

lemma WaitFrameNext:
 "\<turnstile> \<box> f \<longrightarrow> (\<circle> g \<longrightarrow> \<circle> (f \<W> g))"
by (simp add: NextImpNext Prop01 Prop05 Prop09 WaitInsertion)

lemma WaitFrameDiamond:
 "\<turnstile> \<box> f \<longrightarrow> (\<diamond> g \<longrightarrow> \<diamond> (f \<W> g))"
by (simp add: DiamondImpDiamond Prop01 Prop05 Prop09 WaitInsertion)

lemma WaitFrameBox:
 "\<turnstile> \<box> f \<longrightarrow> (\<box> g \<longrightarrow> \<box> (f \<W> g))"
by (meson BoxAndBoxImpBoxRule OrImpUntil Prop09 UntilImpWait lift_imp_trans)

lemma WaitInductiona:
 "\<turnstile> \<box> (f \<longrightarrow> (\<circle> f \<and> g) \<or> h) \<longrightarrow> (f \<longrightarrow> g \<W> h)"
by (simp add: UntilInduction_a wait_d_def)

lemma WaitInductionb:
 "\<turnstile> \<box> (f \<longrightarrow> \<circle> f \<or> g) \<longrightarrow> (f \<longrightarrow> f \<W> g)"
by (simp add: UntilInduction_b wait_d_def)

lemma WaitInductionc:
 "\<turnstile> \<box>(f \<longrightarrow> \<circle> f) \<longrightarrow> (f \<longrightarrow> f \<W> g)"
proof -
 have 1: "\<turnstile> (f \<longrightarrow> \<circle> f) \<longrightarrow>  (f \<longrightarrow> wnext f)"
   unfolding wnext_d_def using NextImpNotNextNot[of f] by auto
 have 2: "\<turnstile> \<box>(f \<longrightarrow> \<circle> f) \<longrightarrow> \<box> (f \<longrightarrow> wnext f)"
   using "1" BoxImpBoxRule by blast
 show ?thesis by (meson 2 BoxImpWait BoxInduct Prop09 lift_imp_trans) 
qed

lemma WaitInductiond:
 "\<turnstile> \<box> (f \<longrightarrow> g \<and> \<circle> f) \<longrightarrow> (f \<longrightarrow> f \<W> g)"
proof -
 have 1: "\<turnstile> (f \<longrightarrow> g \<and> \<circle> f) \<longrightarrow>  (f \<longrightarrow>  wnext f)" 
   unfolding wnext_d_def using NextImpNotNextNot[of f] by auto
 have 2: "\<turnstile> \<box>(f \<longrightarrow> g \<and> \<circle> f) \<longrightarrow> \<box> (f \<longrightarrow>  wnext f)"
   using "1" BoxImpBoxRule by blast
 show ?thesis  by (meson 2 BoxImpWait BoxInduct Prop09 lift_imp_trans) 
qed

lemma WaitAbsorba:
 "\<turnstile> (f \<or> f \<W> g) = (f \<or> g)"
proof -
 have 1: "\<turnstile> (f \<or> f \<W> g) \<longrightarrow> (f \<or> g) "
   using WaitImpOr by fastforce
 have 2: "\<turnstile> f \<or> g \<longrightarrow> f \<or> f \<W> g"
   using WaitInsertion by fastforce
 show ?thesis using "1" "2" int_iffI by blast
qed

lemma WaitAbsorbb:
 "\<turnstile> (f \<W> g \<or> g) = f \<W> g"
using WaitInsertion[of g f] by auto 

lemma WaitAbsorbc:
 "\<turnstile> (f \<W> g \<and> g) = g"
using WaitInsertion by fastforce

lemma WaitAbsorbd:
 "\<turnstile> (f \<W> g \<and> (f \<or> g)) = f \<W> g"
by (meson Prop10 Prop11 WaitImpOr)

lemma WaitAbsorbe:
 "\<turnstile> (f \<W> g \<or> (f \<and> g)) = f \<W> g"
unfolding wait_d_def
using UntilAbsorp_d by fastforce

lemma WaitLeftAbsorb:
 "\<turnstile> f \<W> (f \<W> g) = f \<W> g"
by (metis (no_types, lifting) BoxEqvBoxBox UntilUntil WaitAbsorbBox WaitAbsorba 
    WaitLeftDistOr inteq_reflection wait_d_def)

lemma WaitRightAbsorb:
 "\<turnstile> (f \<W> g) \<W> g = f \<W> g"
by (metis (no_types, lifting) LeftUntilAbsorp Prop10 WaitInsertion WaitNotDistUntil int_iffD1
      int_iffI int_simps(32) inteq_reflection)

lemma WaitBox:
 "\<turnstile> \<box> f = f \<W> #False"
by (metis (no_types, lifting) BoxGen DiamondNotEqvNotBox UntilAbsorpAndDiamond UntilAbsorp_c 
     int_eq_true int_simps(2) int_simps(25) inteq_reflection wait_d_def)

lemma WaitDiamond:
 "\<turnstile> \<diamond> f = (\<not>( (\<not> f) \<W> #False))"
using DiamondNotEqvNotBox WaitBox by fastforce

lemma WaitImp:
 "\<turnstile> f \<W> g \<longrightarrow> \<box> f \<or> \<diamond> g"
by (metis Prop08 UntilImpDiamond WaitAbsorbb WaitImpOr WaitRightAbsorb int_eq wait_d_def)

lemma WaitRightUntilAbsorb:
 "\<turnstile> f \<W> (f \<U> g) = f \<W> g"
by (metis UntilUntil WaitOrRule inteq_reflection wait_d_def)

lemma WaitLeftUntilAbsorb:
 "\<turnstile> (f \<U> g) \<W> g = f \<U> g"
by (metis Prop11 RightUntilAbsorp UntilAbsorp_b UntilImpWait WaitImpOr inteq_reflection)

lemma UntilRightWaitAbsorb:
 "\<turnstile> f \<U> (f \<W> g) = f \<W> g"
using UntilImpWait UntilIntro WaitLeftAbsorb by fastforce

lemma UntilLeftWaitAbsorb:
 "\<turnstile> (f \<W> g) \<U> g = f \<U> g"
by (metis UntilWait WaitRightAbsorb inteq_reflection)

lemma WaitDiamondAbsorb:
 "\<turnstile> (\<diamond> g) \<W> g = \<diamond> g"
by (metis DiamondEqvTrueUntil WaitLeftUntilAbsorb inteq_reflection)

lemma WaitAndBoxAbsorb:
 "\<turnstile> (\<box> f \<and> f \<W> g) = \<box> f"
by (meson BoxImpWait NotDiamondNotEqvBox Prop04 Prop10)

lemma WaitOrBoxAbsorb:
 "\<turnstile> (\<box> f \<or> f \<W> g) = f \<W> g"
by (metis UntilRightWaitAbsorb WaitLeftAbsorb inteq_reflection wait_d_def)

lemma WaitAndBoxImpBox:
 "\<turnstile> f \<W> g \<and> \<box> (\<not> g) \<longrightarrow> \<box> f"
by (metis (no_types, opaque_lifting) Prop02 Prop05 Prop07 Prop08 UntilIdempotent UntilImpDiamond 
    UntilIntro always_d_def int_simps(25) int_simps(4) inteq_reflection wait_d_def)

lemma BoxImpUntilOrBox:
 "\<turnstile> \<box> f \<longrightarrow> f \<U> g \<or> \<box> (\<not> g)"
proof -
 have 1: "\<turnstile> (\<box> f \<longrightarrow> f \<U> g \<or> \<box> (\<not> g)) = 
            ((\<box> f \<and> \<diamond> g)  \<longrightarrow> f \<U> g)"
   by (auto simp add: always_d_def)
 have 2: "\<turnstile> (\<box> f \<and> \<diamond> g)  \<longrightarrow> f \<U> g"
   using UntilAndImp by blast
 show ?thesis 
   using "1" "2" by fastforce
qed

lemma NotBoxAndWaitImpDiamond:
 "\<turnstile> \<not>( \<box> f) \<and> f \<W> g \<longrightarrow> \<diamond> g"
using WaitImp by fastforce

lemma DiamondImpNotBoxOrUntil:
 "\<turnstile> \<diamond> g \<longrightarrow> \<not> (\<box> f) \<or> f \<U> g"
proof -
 have 1: "\<turnstile> \<diamond> g \<and> \<box> f \<longrightarrow> f \<U> g"
   using UntilAndImp by fastforce
 show ?thesis using "1" by auto
qed

lemma WaitRightDistImp:
 "\<turnstile> (f \<longrightarrow> g) \<W> h \<longrightarrow> (f \<W> h \<longrightarrow> g \<W> h)"
proof -
 have 0: "\<turnstile> \<box>(f \<longrightarrow> g) \<and> \<box>f  \<longrightarrow> \<box>g"
   by (simp add: BoxAndBoxImpBoxRule intI)   
 have 1: "\<turnstile> \<box>(f \<longrightarrow> g) \<and> \<box>f  \<longrightarrow> \<box>g \<or> g \<U> h"
   using "0" by auto
 have 2: "\<turnstile> \<box>(f \<longrightarrow> g) \<and> f \<U> h \<longrightarrow> ( (f \<longrightarrow> g) \<and> f) \<U> ( (f \<longrightarrow> g) \<and> h)"
   using UntilAlwaysAndDist[of "LIFT(f \<longrightarrow>g)" "f" "h"] by auto
 have 3: "\<turnstile> (f \<longrightarrow> g) \<and> f \<longrightarrow> g"
   by auto
 have 4: "\<turnstile> (f \<longrightarrow> g) \<and> h \<longrightarrow> h"
   by auto 
 have 5: "\<turnstile> ( (f \<longrightarrow> g) \<and> f) \<U> ( (f \<longrightarrow> g) \<and> h) \<longrightarrow> \<box>g \<or> g \<U> h"
   by (simp add: "3" "4" Prop05 UntilImpUntil)
 have 6: "\<turnstile>\<box>(f \<longrightarrow> g) \<and> f \<U> h \<longrightarrow> \<box>g \<or> g \<U> h "
   using "2" "5" lift_imp_trans by blast
 have 7: "\<turnstile>\<box>(f \<longrightarrow> g) \<longrightarrow> \<box>f \<or> f \<U> h \<longrightarrow> \<box>g \<or> g \<U> h"
   by (simp add: "1" "6" Prop09 Prop12)
 have 8: "\<turnstile> \<box> f \<and> (f \<longrightarrow> g) \<U> h \<longrightarrow> (f \<and> (f \<longrightarrow>g)) \<U> (f \<and> h)"
   by (simp add: UntilAlwaysAndDist)
 have 9: "\<turnstile> f \<and> (f \<longrightarrow>g) \<longrightarrow> g"
   by auto
 have 10: "\<turnstile> f \<and> h \<longrightarrow> h"
   by auto
 have 11: "\<turnstile> (f \<and> (f \<longrightarrow>g)) \<U> (f \<and> h) \<longrightarrow> \<box>g \<or> g \<U> h"
   by (simp add: "10" "9" Prop05 UntilImpUntil)
 have 12: "\<turnstile>  (f \<longrightarrow> g) \<U> h \<and> \<box>f \<longrightarrow> \<box>g \<or> g \<U> h"
   using 8 11 by fastforce
 have 13:"\<turnstile> (f \<longrightarrow> g) \<U> h \<and> f \<U> h \<longrightarrow> \<box>g \<or> g \<U> h"
   by (metis "3" Prop05 UntilAndDist UntilImpUntil UntilIntro UntilUntil inteq_reflection)    
 have 14: "\<turnstile> (f \<longrightarrow> g) \<U> h \<longrightarrow> \<box>f \<or> f \<U> h \<longrightarrow> \<box>g \<or> g \<U> h"
   by (simp add: "12" "13" Prop09 Prop12)
 show ?thesis unfolding wait_d_def using 7 14 Prop02 by blast 
qed

lemma WaitLeftMono:
 "\<turnstile> \<box> (f \<longrightarrow> g) \<longrightarrow> (f \<W> h \<longrightarrow> g \<W> h)"
by (meson BoxImpWait WaitRightDistImp lift_imp_trans)

lemma WaitRightMono:
 "\<turnstile> \<box> (f \<longrightarrow> g) \<longrightarrow> (h \<W> f \<longrightarrow> h \<W> g)"
proof -
 have 1: "\<turnstile> \<box> (f \<longrightarrow> g) \<longrightarrow> (h \<U> f \<longrightarrow> h \<U> g)"
   by (simp add: UntilRightMono)
 have 2: "\<turnstile> \<box> (f \<longrightarrow> g) \<longrightarrow> (\<box> h  \<longrightarrow> \<box> h \<or> h \<U> g)"
   by auto
 have 3: "\<turnstile> \<box> (f \<longrightarrow> g) \<longrightarrow> (h \<U> f  \<longrightarrow> \<box> h \<or> h \<U> g)"
   using "1" by auto
 have 4: "\<turnstile> \<box> (f \<longrightarrow> g) \<longrightarrow> (\<box> h \<or> h \<U> f \<longrightarrow> \<box> h \<or> h \<U> g)"
   using "2" "3" by fastforce
 from 4 show ?thesis by (simp add: wait_d_def)
qed

lemma WaitStrengthen:
 "\<turnstile> \<box>((f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> (f \<W> g \<longrightarrow> h \<W> i)"
proof -
 have 1: "\<turnstile> \<box>((f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> (f \<W> g \<longrightarrow> h \<W> g)"
   by (meson BoxAndBoxEqvBoxRule Prop01 Prop05 Prop11 WaitLeftMono lift_and_com lift_imp_trans)
 have 2: "\<turnstile>  \<box>((f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> (h \<W> g \<longrightarrow> h \<W> i) "
   by (meson BoxElim BoxImpBoxBox BoxImpBoxRule Prop12 WaitRightMono lift_imp_trans)
 have 3: "\<turnstile> \<box> ( (f \<longrightarrow> h) \<and> (g \<longrightarrow> i)) \<longrightarrow> (f \<W> g \<longrightarrow> h \<W> g) \<and> ( h \<W> g \<longrightarrow> h \<W> i)"
   using "1" "2" by fastforce
 have 4: "\<turnstile> (f \<W> g \<longrightarrow> h \<W> g) \<and> ( h \<W> g \<longrightarrow> h \<W> i) \<longrightarrow> (f \<W> g \<longrightarrow> h \<W> i)"
   by auto
 from 3 4 show ?thesis by auto
qed

lemma WaitCatRule:
 "\<turnstile> \<box>( (f \<longrightarrow> g \<W> h) \<and> (h \<longrightarrow> g \<W> i)) \<longrightarrow> (f \<longrightarrow> g \<W> i)"
proof -
 have 1: "\<turnstile> \<box> ( (f \<longrightarrow> g \<W>  h) \<and> (h \<longrightarrow> g \<W>  i)) \<longrightarrow> \<box> (f \<longrightarrow> g \<W> h)"
   by (metis BoxElim BoxEqvBoxBox BoxImpBoxRule Prop12 inteq_reflection)
 have 2: "\<turnstile> \<box> ( (f \<longrightarrow> g \<W> h) \<and> (h \<longrightarrow> g \<W> i)) \<longrightarrow> \<box> (h \<longrightarrow> g \<W> i) "
   by (metis BoxElim BoxEqvBoxBox BoxImpBoxRule Prop12 inteq_reflection)
 have 3: "\<turnstile> \<box> (h \<longrightarrow> g \<W> i) \<longrightarrow> \<box> ( g \<W> h \<longrightarrow> g \<W> (g \<W> i))"
   by (metis BoxEqvBoxBox BoxImpBoxRule WaitRightMono inteq_reflection) 
 have 4: "\<turnstile> \<box> ( g \<W> h \<longrightarrow> g \<W> (g \<W> i)) \<longrightarrow> \<box>( g \<W> h \<longrightarrow> g \<W> i)"
   by (metis BoxEqvBoxBox WaitLeftAbsorb int_iffD1 inteq_reflection)
 have 5: "\<turnstile>  \<box> (f \<longrightarrow> g \<W> h) \<longrightarrow> (f \<longrightarrow> g \<W> h)"
   by (simp add: BoxElim)
 have 6: "\<turnstile> \<box>( g \<W> h \<longrightarrow> g \<W> i) \<longrightarrow> (g \<W> h \<longrightarrow> g \<W> i)"
   by (simp add: BoxElim)
 have 7: "\<turnstile> (f \<longrightarrow> g \<W> h) \<and> (g \<W> h \<longrightarrow> g \<W> i) \<longrightarrow> (f \<longrightarrow> g \<W> i)"
   by auto
 have 8: "\<turnstile> \<box> ( (f \<longrightarrow> g \<W> h) \<and> (h \<longrightarrow> g \<W> i)) \<longrightarrow> 
            (f \<longrightarrow> g \<W> h) \<and> (g \<W> h \<longrightarrow> g \<W> i) "
   using "1" "2" "3" "4" "5" "6"by fastforce 
 from 7 8 show ?thesis by auto
qed

lemma LeftUntilWaitImp:
 "\<turnstile> (f \<U> g) \<W> h \<longrightarrow> (f \<W> g) \<W> h"
 by (meson BoxGen MP UntilImpWait WaitLeftMono)

lemma RightWaitUntilImp:
 "\<turnstile> f \<W> (g \<U> h) \<longrightarrow> f \<W> (g \<W> h)"
by (meson BoxGen MP UntilImpWait WaitRightMono)

lemma RightUntilUntilImp:
 "\<turnstile> f \<U> (g \<U> h) \<longrightarrow> f \<U> (g \<W> h)"
by (meson BoxGen MP UntilImpWait UntilRightMono)

lemma LeftUntilUntilImp:
 "\<turnstile> (f \<U> g) \<U> h \<longrightarrow> (f \<W> g) \<U> h"
by (simp add: UntilImpUntil UntilImpWait)

lemma LeftUntilOrStrengthen:
 "\<turnstile> (f \<U> g) \<U> h \<longrightarrow> (f \<or> g) \<U> h"
by (simp add: UntilImpOr UntilImpUntil)

lemma LeftWaitOrStrengthen:
 "\<turnstile> (f \<W> g) \<W> h \<longrightarrow> (f \<or> g) \<W> h"
by (meson BoxGen MP WaitImpOr WaitLeftMono)

lemma RightWaitOrStrengthen:
 "\<turnstile> f \<W> (g \<W> h) \<longrightarrow> f \<W> (g \<or> h)"
by (meson BoxGen MP WaitImpOr WaitRightMono)

lemma BoxImpBoxOr:
 "\<turnstile> \<box> f \<longrightarrow> \<box>(f \<or> g)"
 by (metis BoxEqvBoxBox BoxImpBoxRule BoxImpWait Prop12 WaitAbsorbd inteq_reflection)

lemma RightWaitOrOrder:
 "\<turnstile> f \<W> (g \<W> h) \<longrightarrow> (f \<or> g) \<W> h"
proof -
 have 1: "\<turnstile> f \<W> (g \<W> h) = (\<box> f \<or> f \<U> (\<box> g \<or> g \<U> h))"
   by (simp add: wait_d_def)
 have 2: "\<turnstile> (f \<or> g) \<W> h = (\<box> (f \<or> g) \<or> (f \<or> g) \<U> h)"
   by (simp add: wait_d_def)
 have 3: "\<turnstile> \<box> f \<longrightarrow> (\<box> (f \<or> g) \<or> (f \<or> g) \<U> h)"
   using BoxImpBoxOr by fastforce
 have 4: "\<turnstile> f \<U> (\<box> g \<or> g \<U> h) = (f \<U> (\<box> g) \<or> f \<U> (g \<U> h))"
   using UntilOrDist by blast
 have 5: "\<turnstile> f \<U> (g \<U> h) \<longrightarrow> (\<box> (f \<or> g) \<or> (f \<or> g) \<U> h)"
   by (simp add: Prop05 UntilRightOr)
 have 6: "\<turnstile> f \<U> (\<box> g) \<longrightarrow> (\<box> (f \<or> g) \<or> (f \<or> g) \<U> h)"
   by (metis BoxImpBoxRule BoxImpWait UntilBoxImp UntilImpOr lift_imp_trans wait_d_def) 
 have 7: "\<turnstile> (f \<U> (\<box> g) \<or> f \<U> (g \<U> h)) \<longrightarrow> (\<box> (f \<or> g) \<or> (f \<or> g) \<U> h)"
   using "5" "6" by fastforce
 show ?thesis 
   using "1" "2" "3" "4" "7" by fastforce
qed

lemma RightWaitAndOrder:
 "\<turnstile> f \<W> (g \<and> h) \<longrightarrow> f \<W> (g \<W> h)"
by (metis Prop03 WaitAbsorbe WaitLeftDistOr inteq_reflection)

lemma UntilOrder:
 "\<turnstile> ((\<not> f) \<U> g \<or> (\<not> g) \<U> f) = \<diamond>(f \<or> g)"
proof -
 have 1: "\<turnstile> ((\<not> f) \<U> g \<or> (\<not> g) \<U> f) \<longrightarrow> \<diamond>(f \<or> g)"
   using UntilAbsorpAndDiamond UntilOrDist by fastforce 
 have 2: "\<turnstile> \<diamond>(f \<or> g) = #True \<U> (f \<or>  g) "
   by (metis DiamondEqvTrueUntil )
 have 3: "\<turnstile> #True \<U> (f \<or>  g) = (\<not> (f \<or> g) ) \<U> (f \<or> g)" 
   using "2" UntilRule by fastforce
 have 4: "\<turnstile> (\<not> (f \<or> g) ) \<U> (f \<or> g) = (\<not>f \<and> \<not>g) \<U> (f \<or> g)"
   by (metis UntilAbsorp_c int_eq int_simps(14) int_simps(33))
 have 5: "\<turnstile> (\<not>f \<and> \<not>g) \<U> (f \<or> g) \<longrightarrow> (\<not> f \<and> \<not>g) \<U> f  \<or> (\<not> f \<and> \<not>g) \<U> g"
   by (simp add: UntilOrDist int_iffD1)
 have 6: "\<turnstile> (\<not> f \<and> \<not>g) \<U> f \<longrightarrow> (\<not> g) \<U> f" 
   by (metis UntilAndRule int_iffD2 inteq_reflection lift_and_com)
 have 7: "\<turnstile> (\<not> f \<and> \<not>g) \<U> g \<longrightarrow> (\<not> f) \<U> g"
   by (metis UntilAndRule int_iffD2)
 have 8: "\<turnstile> (\<not> f \<and> \<not>g) \<U> f  \<or> (\<not> f \<and> \<not>g) \<U> g \<longrightarrow> (\<not> f) \<U> g \<or> (\<not> g) \<U> f"
   using 6 7 by fastforce 
 have 9: "\<turnstile>  \<diamond>(f \<or> g) \<longrightarrow> (\<not> f) \<U> g \<or> (\<not> g) \<U> f  "
   using 2 3 4 5 8 by fastforce
 show ?thesis  using  1 9 by fastforce
qed

lemma WaitOrder: 
 "\<turnstile> (\<not> f) \<W> g \<or> (\<not> g) \<W> f"
proof -
 have 1: "\<turnstile> (\<not> f) \<W> g = (\<box> (\<not> f) \<or> (\<not> f) \<U> g)"
   by (simp add: wait_d_def)
 have 2:"\<turnstile> (\<not> g) \<W> f = (\<box> (\<not> g) \<or> (\<not> g) \<U> f)" 
   by (simp add: wait_d_def)
 have 3: "\<turnstile> ((\<box> (\<not> f) \<or> (\<not> f) \<U> g) \<or> (\<box> (\<not> g) \<or> (\<not> g) \<U> f)) =
           ( (\<box> (\<not> f) \<or> \<box> (\<not> g)) \<or> ((\<not> f) \<U> g \<or> (\<not> g) \<U> f))"
   by auto
 have 4: "\<turnstile> ((\<not> f) \<U> g \<or> (\<not> g) \<U> f) = \<diamond>(f \<or> g)"
   using UntilOrder by blast
 have 5: "\<turnstile> (\<box> (\<not> f) \<or> \<box> (\<not> g)) = (\<not> (\<diamond> f) \<or> \<not> (\<diamond> g))"
   by (simp add: always_d_def)
 have 6: "\<turnstile> \<diamond>(f \<or> g) = (\<diamond>f \<or> \<diamond> g)"
   by (simp add: ChopOrEqv sometimes_d_def)
 have 7: "\<turnstile> (\<box> (\<not> f) \<or> \<box> (\<not> g)) \<or> \<diamond>(f \<or> g) "
   using 5 6 by fastforce
 show ?thesis 
   using 1 2 4 7 by fastforce
qed

lemma WaitImpOrder:
 "\<turnstile> f \<W> g \<and> (\<not> g) \<W> h \<longrightarrow> f \<W> h"
proof -
 have 1: "\<turnstile> f \<W> g = (\<box>f \<or> f \<U> g) "
   by (simp add: wait_d_def)
 have 2: "\<turnstile> f \<W> h = (\<box> f \<or> f \<U> h)"
   by (simp add: wait_d_def)
 have 3: "\<turnstile> (\<not> g) \<W> h = (\<box> (\<not> g) \<or> (\<not> g) \<U> h)"
   by (simp add: wait_d_def)
 have 4: "\<turnstile> \<box> f \<and> (\<box> (\<not> g) \<or> (\<not> g) \<U> h) \<longrightarrow> (\<box> f \<or> f \<U> h)"
   by auto
 have 5: "\<turnstile> (\<box>f \<or> f \<U> g) \<and> \<box> (\<not> g) \<longrightarrow> (\<box> f \<or> f \<U> h)"
   using "1" WaitAndBoxImpBox by fastforce
 have 6: "\<turnstile> f \<U> g \<and> (\<not> g) \<U> h \<longrightarrow> (\<box> f \<or> f \<U> h)"
   by (simp add: Prop05 UntilNotImp)
 have 7: "\<turnstile> \<box> f \<and> (\<not> g) \<U> h \<longrightarrow> (\<box> f \<or> f \<U> h)"
   by auto
 have 8: "\<turnstile> (\<box>f \<or> f \<U> g) \<and>  (\<not> g) \<U> h \<longrightarrow> (\<box> f \<or> f \<U> h)"
   using "6" "7" by fastforce
 have 9: "\<turnstile> (\<box>f \<or> f \<U> g) \<and> (\<box> (\<not> g)\<or> (\<not> g) \<U> h) \<longrightarrow> (\<box> f \<or> f \<U> h)"
   using "5" "8" by fastforce
 show ?thesis by (simp add: "9" wait_d_def) 
qed




end 