(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

section \<open>Time reversal operator\<close>

theory TimeReversal
imports
 SChopTheorems FOTheorems Omega 
begin

text \<open>
 Time reversal operator is defined in \cite{moszkowski14:_compos}. 
 Note: in this theory only finite intervals can be reversed. 
\<close>

subsection \<open>Definition\<close>

definition freverse_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "freverse_d F \<equiv> \<lambda> s. nfinite s \<and> (nrev s \<Turnstile> F)" 

definition ereverse_d :: "('a::world, 'b) formfun \<Rightarrow> ('a, 'b) formfun"
where "ereverse_d f \<equiv> \<lambda> s. (if nfinite s then (nrev s \<Turnstile> f) else (\<some> (x::'b ). x=x))" 

syntax
 "_freverse_d"      :: "lift \<Rightarrow> lift"        ("(_\<^sup>r)" [85] 85)
 "_ereverse_d"      :: "lift \<Rightarrow> lift"        ("(_\<^sup>R)" [100] 99)

syntax (ASCII)
 "_freverse_d"      :: "lift \<Rightarrow> lift"        ("(frev _)" [85] 85)
 "_ereverse_d"      :: "lift \<Rightarrow> lift"        ("(erev _)" [100] 99)

translations
 "_freverse_d"     \<rightleftharpoons> "CONST freverse_d" 
 "_ereverse_d"     \<rightleftharpoons> "CONST ereverse_d" 

definition wfreverse_d ::  "('a::world) formula \<Rightarrow> 'a formula"
where "wfreverse_d F \<equiv> LIFT (\<not>((\<not>F)\<^sup>r)) " 

syntax
 "_wfreverse_d"      :: "lift \<Rightarrow> lift"        ("(wrev _)" [85] 85)

syntax (ASCII)
 "_wfreverse_d"      :: "lift \<Rightarrow> lift"        ("(wrev _)" [85] 85)

translations
 "_wfreverse_d"     \<rightleftharpoons> "CONST wfreverse_d" 

subsection \<open>Finite theorems\<close>

lemma FiniteChopEqv: 
 assumes "\<turnstile> finite \<longrightarrow> f = g "
 shows   "\<turnstile> finite \<longrightarrow> f;h = g;h " 
using assms 
by (auto simp add: Valid_def itl_defs) 

lemma FiniteTransEqv: 
 assumes "\<turnstile> finite \<longrightarrow> f = g " 
         "\<turnstile> finite \<longrightarrow> g = h "
 shows "\<turnstile> finite \<longrightarrow> f = h " 
using assms 
by (auto simp add: Valid_def itl_defs) 

lemma FiniteOrEqv: 
 assumes "\<turnstile> finite \<longrightarrow> f1 = f2 " 
         "\<turnstile> finite \<longrightarrow> g1 = g2 "
 shows "\<turnstile> finite \<longrightarrow> (f1 \<or> g1) = (f2 \<or> g2) " 
using assms 
by (auto simp add: Valid_def itl_defs) 

lemma FiniteAndEqv: 
 assumes "\<turnstile> finite \<longrightarrow> f1 = f2 " 
         "\<turnstile> finite \<longrightarrow> g1 = g2 "
 shows "\<turnstile> finite \<longrightarrow> (f1 \<and> g1) = (f2 \<and> g2) " 
using assms 
by (auto simp add: Valid_def itl_defs) 

lemma FiniteNotEqv:
 assumes "\<turnstile> finite \<longrightarrow> f1 = f2"
 shows    "\<turnstile> finite \<longrightarrow> (\<not> f1) = (\<not> f2)" 
 using assms 
by (auto simp add: Valid_def itl_defs)  


lemma FiniteWNextEqv: 
assumes "\<turnstile> finite \<longrightarrow> f =g" 
shows   "\<turnstile> finite \<longrightarrow> wnext f = wnext g" 
using assms 
by (auto simp add: Valid_def itl_defs) 

lemma FiniteWPrevEqv: 
assumes "\<turnstile> finite \<longrightarrow> f =g" 
shows   "\<turnstile> finite \<longrightarrow> wprev f = wprev g" 
using assms 
by (auto simp add: Valid_def itl_defs) 

lemma FiniteInitonlyEqv: 
assumes " \<turnstile> finite \<longrightarrow> f = g "
 shows  " \<turnstile> finite \<longrightarrow> initonly f = initonly g" 
using assms
by (auto simp add: Valid_def itl_defs)



lemma FiniteKeepEqv: 
assumes " \<turnstile> finite \<longrightarrow> f = g "
 shows  " \<turnstile> finite \<longrightarrow> keep f = keep g" 
using assms
by (auto simp add: Valid_def keep_d_def itl_defs nsubn_nfinite)


lemma FiniteHaltEqv: 
assumes " \<turnstile> finite \<longrightarrow> f = g "
 shows  " \<turnstile> finite \<longrightarrow> halt f = halt g" 
using assms
by (auto simp add: Valid_def itl_defs)


lemma FiniteDaAlt: 
 "\<turnstile> finite \<longrightarrow> da f = (#True;f);#True" 
unfolding da_d_def
by (metis ChopAssoc FiniteChopEqv int_simps(13) int_simps(9) inteq_reflection)

lemma ChopCsImpCSChop:
 "\<turnstile> (f \<and> finite);(schopstar f) \<longrightarrow> (schopstar f);(f \<and> finite)"
by (metis SChopstar_WPowerstar WPowerstar_slide_var int_iffD2 inteq_reflection)


lemma CSChopImpChopCS:
 "\<turnstile> (schopstar f);(f \<and> finite) \<longrightarrow> (f \<and> finite);(schopstar f)"
by (metis SChopstar_WPowerstar WPowerstar_slide_var int_iffD1 inteq_reflection)


lemma NextDiamondEqvDiamondNext:
 " \<turnstile> \<circle>( \<diamond> f) = \<diamond>(\<circle>f)"
proof -
 have  1: "\<turnstile> finite;skip = skip;finite  " 
   by (simp add: FiniteChopSkipEqvSkipChopFinite)
 hence 2: "\<turnstile> (finite;skip);f =( skip;finite);f " using LeftChopEqvChop by blast
 have  3: "\<turnstile> (finite;skip);f =  finite;(skip;f)  "
  by (meson ChopAssoc ChopEmpty Prop04)
 have  4: "\<turnstile> ( skip;finite);f =  skip;(finite;f) "
   by (meson ChopAssoc Prop11)
 from 2 3 4 show ?thesis by (metis int_eq next_d_def sometimes_d_def)
qed

lemma WeakNextBoxInduct:
 assumes "\<turnstile> wnext  (\<box> f) \<longrightarrow> f "
 shows   "\<turnstile> finite \<longrightarrow> f "
proof -
 have   1: "\<turnstile> wnext  (\<box> f) \<longrightarrow> f" using assms by blast
 hence  2: "\<turnstile> \<not> f  \<longrightarrow> \<not> ( wnext  (\<box> f))  "  by fastforce 
 hence  3: "\<turnstile> \<not> f  \<longrightarrow>   \<circle>  (\<not> (\<box> f)) "  by (simp add: wnext_d_def) 
 have   4: "\<turnstile> (\<not> (\<box> f)) =   (\<diamond> (\<not> f)) " by (auto simp: always_d_def)
 hence  5: "\<turnstile> \<circle>(\<not> (\<box> f)) =  \<circle> (\<diamond> (\<not> f)) " using NextEqvNext by blast
 have   6: "\<turnstile> \<not> f  \<longrightarrow>   \<circle>  (\<diamond> (\<not> f)) " using "3" "5" by fastforce
 have   7: "\<turnstile> \<circle>  (\<diamond> (\<not> f)) = \<diamond>( \<circle> (\<not> f))  " using NextDiamondEqvDiamondNext by blast
 have   8: "\<turnstile> \<not> f  \<longrightarrow>   \<diamond>( \<circle> (\<not> f))  " using "6" "7" by fastforce
 have   9: "\<turnstile> \<diamond>(\<not> f)  \<longrightarrow>   \<diamond>(\<diamond>( \<circle> (\<not> f))) " using "8" DiamondImpDiamond by blast
 have  10: "\<turnstile> \<diamond>(\<diamond>( \<circle> (\<not> f))) = \<diamond>( \<circle> (\<not> f)) " using DiamondDiamondEqvDiamond by blast
 have  11: "\<turnstile> \<diamond>(\<not> f)  \<longrightarrow> \<diamond>( \<circle> (\<not> f)) " using "9" "10" by fastforce
 have  12: "\<turnstile> \<diamond>(\<not> f)  \<longrightarrow> \<circle> (\<diamond> (\<not> f)) " using "7" "11" by fastforce
 hence 13: "\<turnstile> finite \<longrightarrow> \<not>( \<diamond>(\<not> f))"  using NextLoop by blast
 hence 14: "\<turnstile> finite \<longrightarrow> \<box>f " by (simp add: always_d_def)
 have  15: "\<turnstile> \<box>f \<longrightarrow> f " using BoxElim by blast
 from 14 15 show ?thesis by (metis lift_imp_trans)
qed

subsection \<open>Time reversal Rules\<close>


lemma WRevRev:
 "\<turnstile> (wrev f) = (finite \<longrightarrow> f\<^sup>r)"
by (simp add: Valid_def wfreverse_d_def freverse_d_def finite_defs nfinite_nrev)

lemma RevWRev:
 "\<turnstile> (frev f) = (finite \<and> wrev f) " 
by (auto simp add: Valid_def wfreverse_d_def freverse_d_def finite_defs nfinite_nrev)

lemma RevWRevAlt: 
 "\<turnstile> (frev f) = (\<not>((wrev (\<not>f)))) "
unfolding wfreverse_d_def
by simp 

lemma EExRev :
 "\<turnstile> (\<exists>\<exists> x. F x)\<^sup>r = (\<exists>\<exists> x. (F x)\<^sup>r)"
by (simp add: Valid_def exist_state_d_def freverse_d_def)

lemma EExWRev :
 "\<turnstile> (wrev (\<exists>\<exists> x. F x)) = (\<exists>\<exists> x. (wrev (F x)))"
by (simp add: Valid_def exist_state_d_def wfreverse_d_def freverse_d_def)


lemma rev_const :
 "\<turnstile> finite \<longrightarrow> (#c)\<^sup>R = #c"
by (auto simp add: Valid_def itl_defs ereverse_d_def)

lemma rev_fun1 :
 "\<turnstile> finite \<longrightarrow> (f<x>)\<^sup>r = f<x\<^sup>r>"
by (auto simp: itl_defs freverse_d_def)

lemma rev_fun1_alt :
 "\<turnstile>  (f<x>)\<^sup>r = (f<x\<^sup>r> \<and> finite)"
by (auto simp: itl_defs freverse_d_def)

lemma wrev_fun1 :
 "\<turnstile> finite \<longrightarrow>(wrev (f<x>)) = f<(wrev x)> "
by (auto simp add: itl_defs wfreverse_d_def   freverse_d_def)

lemma wrev_fun1_alt :
 "\<turnstile>  (wrev (f<x>)) = (finite \<longrightarrow> f<(wrev x)> ) "
by (auto simp add: itl_defs wfreverse_d_def   freverse_d_def)

lemma rev_fun1_e :
 "\<turnstile> finite \<longrightarrow> (f<x>)\<^sup>R = f<x\<^sup>R>"
by (auto simp: itl_defs ereverse_d_def)

lemma rev_fun2:
 "\<turnstile> finite \<longrightarrow> (f<x,y>)\<^sup>r = f<x\<^sup>r,y\<^sup>r>"
by (auto simp: itl_defs freverse_d_def)

lemma rev_fun2_alt:
 "\<turnstile>  (f<x,y>)\<^sup>r = (f<x\<^sup>r,y\<^sup>r> \<and> finite)"
by (auto simp: itl_defs freverse_d_def)

lemma wrev_fun2:
 "\<turnstile> finite \<longrightarrow> (wrev (f<x,y>)) = f<(wrev x),(wrev y)>"
by (auto simp add: itl_defs wfreverse_d_def   freverse_d_def)

lemma wrev_fun2_alt :
 "\<turnstile>  (wrev (f<x,y>)) = (finite \<longrightarrow> f<(wrev x), (wrev y)> ) "
by (auto simp add: itl_defs wfreverse_d_def   freverse_d_def)

lemma rev_fun2_e:
 "\<turnstile> finite \<longrightarrow> (f<x,y>)\<^sup>R = f<x\<^sup>R,y\<^sup>R>"
by (auto simp: itl_defs ereverse_d_def)

lemma rev_fun3:
 "\<turnstile> finite \<longrightarrow> (f<x,y,z>)\<^sup>r = f<x\<^sup>r,y\<^sup>r,z\<^sup>r>"
by (auto simp: itl_defs freverse_d_def)

lemma rev_fun3_alt:
 "\<turnstile>  (f<x,y,z>)\<^sup>r = (f<x\<^sup>r,y\<^sup>r,z\<^sup>r> \<and> finite)"
by (auto simp: itl_defs freverse_d_def)

lemma wrev_fun3:
 "\<turnstile> finite \<longrightarrow> (wrev (f<x,y,z>)) = f<(wrev x),(wrev y),(wrev z)>"
by (auto simp add: itl_defs wfreverse_d_def   freverse_d_def)

lemma wrev_fun3_alt :
 "\<turnstile>  (wrev (f<x,y,z>)) = (finite \<longrightarrow> f<(wrev x), (wrev y), (wrev z)> ) "
by (auto simp add: itl_defs wfreverse_d_def   freverse_d_def)

lemma rev_fun3_e:
 "\<turnstile> finite \<longrightarrow> (f<x,y,z>)\<^sup>R = f<x\<^sup>R,y\<^sup>R,z\<^sup>R>"
by (auto simp: itl_defs ereverse_d_def)

lemma rev_forall:
 "\<turnstile> (\<forall> x. P x)\<^sup>r = (\<forall> x. (P x)\<^sup>r)"
by (auto simp: freverse_d_def)

lemma wrev_forall:
 "\<turnstile> (wrev (\<forall> x. P x)) = (\<forall> x. (wrev (P x)))"
by (auto simp: wfreverse_d_def freverse_d_def )

lemma rev_exists:
 "\<turnstile> (\<exists> x. P x)\<^sup>r = (\<exists> x. (P x)\<^sup>r)"
by (auto simp: freverse_d_def)

lemma wrev_exists:
 "\<turnstile> (wrev (\<exists> x. P x)) = (\<exists> x. (wrev (P x)))"
by (auto simp: wfreverse_d_def freverse_d_def )

lemma rev_exists1:
 "\<turnstile> (\<exists>! x. P x)\<^sup>r = (\<exists>! x. (P x)\<^sup>r)"
by (auto simp: freverse_d_def)


lemma wrev_exists1: 
 "\<turnstile> wrev (\<exists>! x. P x) = (finite \<longrightarrow> (\<exists>! x. wrev (P x))) "
proof -
 have 1: "\<turnstile> wrev (\<exists>! x. P x) = (finite \<longrightarrow> frev (\<exists>! x. P x)) "
 using WRevRev by blast
 have 2: "\<turnstile> (finite \<longrightarrow> frev (\<exists>! x. P x)) = (finite \<longrightarrow> (\<exists>! x. frev (P x))) "
 by (metis int_simps(27) inteq_reflection rev_exists1) 
 have 3: "\<And> x. \<turnstile> frev (P x) = (wrev (P x) \<and> finite)"
 by (metis RevWRev inteq_reflection lift_and_com) 
 have 4: "\<turnstile> ( (\<exists>! x. frev (P x))) = (\<exists>! x.  wrev (P x) \<and> finite) "
    using 3 by (simp add: Valid_def freverse_d_def itl_defs)
 have 5: "\<turnstile> (\<exists>! x.  wrev (P x) \<and> finite) = ((\<exists>! x. wrev (P x)) \<and> finite) " 
  by auto
 have 6: "\<turnstile> (finite \<longrightarrow> ((\<exists>! x. wrev (P x)) \<and> finite)) = (finite \<longrightarrow> (\<exists>! x. wrev (P x))) "
 by force   
 show ?thesis
 by (metis "1" "2" "4" "5" "6" inteq_reflection) 
qed  
    
   

lemma rev_current:
 "\<turnstile> finite \<longrightarrow> ($v)\<^sup>R = (!v)"
by (auto simp: nnth_nrev itl_def ereverse_d_def nfirst_nrev)

lemma rev_next:
 "\<turnstile>  finite \<longrightarrow> (v$)\<^sup>R = (v!)"
by (auto simp:  itl_defs ereverse_d_def nfinite_nrev) 
 (metis One_nat_def enat_ord_simps(2) epred_conv_minus idiff_enat_enat less_Suc0 linorder_not_less 
  nfinite_nlength_enat nnth_nrev one_enat_def the_enat.simps)

lemma rev_penult:
 "\<turnstile>  finite \<longrightarrow> (v!)\<^sup>R = (v$)"
by (auto simp:  itl_defs ereverse_d_def nfinite_nrev)
 (metis One_nat_def Suc_ile_eq epred_enat i0_less ndropn_nfirst nfinite_ndropn 
    nfinite_nlength_enat nlast_nrev nrev_ndropn ntaken_nlast the_enat.simps zero_enat_def)

lemma rev_fin:
 "\<turnstile> finite \<longrightarrow> (!v)\<^sup>R = ($v)"
by (auto simp:  itl_defs ereverse_d_def nfinite_nrev nlast_nrev)

lemma EqvReverseReverse:
 "\<turnstile> finite \<longrightarrow> (f\<^sup>r)\<^sup>r = f"
by (simp add: Valid_def itl_defs freverse_d_def nfinite_nrev nrev_nrev)

lemma EqvWReverseWReverse:
 "\<turnstile> finite \<longrightarrow> (wrev (wrev f)) = f"
by (simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def nfinite_nrev nrev_nrev)

lemma EqvReverseReverseAlt:
 "\<turnstile>  (f\<^sup>r)\<^sup>r = (f \<and> finite)"
by (auto simp add: Valid_def itl_defs freverse_d_def nfinite_nrev nrev_nrev)

lemma EqvWReverseWReverseAlt:
 "\<turnstile>  (wrev (wrev f)) = (finite \<longrightarrow> f)"
by (simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def nfinite_nrev nrev_nrev)

lemma ReverseEqv:
  "(\<turnstile> finite \<longrightarrow> f) \<longleftrightarrow> (\<turnstile> finite \<longrightarrow>f\<^sup>r)"
by (simp add: Valid_def itl_defs freverse_d_def)
   (metis nfinite_nrev nrev_nrev)

lemma WReverseEqv:
  "(\<turnstile> finite \<longrightarrow> f) \<longleftrightarrow> (\<turnstile> finite \<longrightarrow> wrev f)"
by (simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def)
   (metis nfinite_nrev nrev_nrev)

lemma ReverseEqvAlt:
  "(\<turnstile>  f \<and> finite) \<longleftrightarrow> (\<turnstile> f\<^sup>r)"
by (auto simp add: Valid_def itl_defs freverse_d_def)

lemma WReverseEqvAlt:
  "(\<turnstile> finite \<longrightarrow> f) \<longleftrightarrow> (\<turnstile>  wrev f)"
by (simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def)
  (metis nfinite_nrev nrev_nrev)

lemma RevSkip:
 "\<turnstile> skip\<^sup>r = skip"
using nlength_eq_enat_nfiniteD by (simp add: Valid_def freverse_d_def itl_defs, blast )

lemma WRevSkip:
 "\<turnstile> (wrev skip) = (inf \<or> skip)"
by (auto simp add: Valid_def wfreverse_d_def freverse_d_def itl_defs)

lemma RevFinite:
 "\<turnstile> finite\<^sup>r = finite" 
by (simp add: Valid_def freverse_d_def finite_defs nfinite_nrev)

lemma WRevFinite:
 "\<turnstile> (wrev finite) "
by (auto simp add: Valid_def wfreverse_d_def freverse_d_def itl_defs nfinite_nrev)

lemma NotRevInfinite: 
 "\<turnstile> \<not> ( inf\<^sup>r) " 
by (simp add: Valid_def freverse_d_def infinite_defs nfinite_nrev)

lemma WRevInfinite: 
 "\<turnstile> (wrev inf) = inf " 
by (simp add: Valid_def wfreverse_d_def freverse_d_def infinite_defs nfinite_nrev)

lemma RevChop:
"\<turnstile> finite \<longrightarrow> (f;g)\<^sup>r = (g\<^sup>r;f\<^sup>r)"
by ( simp add: Valid_def itl_defs freverse_d_def nfinite_nrev)
 (metis diff_diff_cancel diff_le_self enat_ord_simps(1) nfinite_nlength_enat nrev_ndropn 
       nrev_ntaken the_enat.simps)

lemma RevYields: 
 "\<turnstile> finite \<longrightarrow> (f yields g)\<^sup>r = ((g\<^sup>r) revyields f\<^sup>r)"
unfolding yields_d_def revyields_d_def
 by (meson FiniteChopEqv FiniteNotEqv FiniteTransEqv RevChop rev_fun1)


lemma WRevChop:
"\<turnstile> finite \<longrightarrow> (wrev (f;g)) = ((wrev g);(wrev f))"
by ( simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def nfinite_nrev)
 (metis diff_diff_cancel diff_le_self enat_ord_simps(1) nfinite_nlength_enat nrev_ndropn 
       nrev_ntaken the_enat.simps)

lemma RevChopAlt:
"\<turnstile>  (f;g)\<^sup>r = (g\<^sup>r;f\<^sup>r)"
by ( simp add: Valid_def itl_defs freverse_d_def nfinite_nrev)
 (metis diff_diff_cancel diff_le_self enat_ord_simps(1) nfinite_nlength_enat nrev_ndropn 
       nrev_ntaken the_enat.simps)

lemma RevSChopAlt:
"\<turnstile>  (f\<frown>g)\<^sup>r = (g\<^sup>r\<frown>f\<^sup>r)"
by (metis (no_types, opaque_lifting) EqvReverseReverseAlt RevChopAlt int_simps(27) 
  inteq_reflection rev_fun2_alt schop_d_def)
 

lemma WRevChopAlt:
"\<turnstile>  (wrev (f;g)) = ((wrev g);(wrev f))"
by ( simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def nfinite_nrev )
 (metis diff_diff_cancel diff_le_self enat_ord_simps(1) nfinite_nlength_enat nrev_ndropn 
  nrev_ntaken the_enat.simps)

lemma WRevSChopAlt:
"\<turnstile>  (wrev (f\<frown>g) ) = ((wrev g);(wrev f))"
unfolding schop_d_def
by (metis (no_types, opaque_lifting) EqvReverseReverseAlt EqvWReverseWReverse FiniteImpAnd 
WRevChopAlt int_simps(4) inteq_reflection rev_fun1_alt wfreverse_d_def) 

lemma RAnd: 
 "\<turnstile> (f \<and> g)\<^sup>r = (f\<^sup>r \<and> g\<^sup>r)"
by (auto simp add: Valid_def freverse_d_def)

lemma WRAnd: 
 "\<turnstile> (wrev (f \<and> g)) = (wrev f \<and> wrev g)"
by (auto simp add: Valid_def wfreverse_d_def freverse_d_def)

lemma RevAndFinite: 
 "\<turnstile> (f\<^sup>r \<and> finite) = (f \<and> finite)\<^sup>r"
by (metis RAnd RevFinite inteq_reflection)

lemma RevAndFiniteAlt: 
 "\<turnstile> (f\<^sup>r \<and> finite) = (f)\<^sup>r"
 by (metis int_simps(20) inteq_reflection rev_fun2_alt)

lemma WRevAndFinite: 
 "\<turnstile> (wrev f)  = (wrev (f \<and> finite)) " 
proof -
 have 1: "\<turnstile> (wrev (f \<and> finite)) = ((wrev f) \<and> wrev finite) "
 by (simp add: WRAnd) 
 have 2: "\<turnstile> ((wrev f) \<and> wrev finite) = (wrev f) "
 using RevFinite WRevRev by fastforce 
 show ?thesis
 using "1" "2" by fastforce  
qed
 
lemma WRevAndFiniteAlt: 
 "\<turnstile> ((wrev f) \<and> finite) = (frev f) "
 by (metis int_simps(4) inteq_reflection rev_fun1_alt wfreverse_d_def) 

lemma WRevFiniteImp:
 "\<turnstile> (finite \<longrightarrow> wrev f) = wrev f"
 by (metis FiniteImp RevFinite WRevAndFiniteAlt WRevRev inteq_reflection) 


lemma RMoreEqvMore:
 "\<turnstile> finite \<longrightarrow> more\<^sup>r = more"
by (simp add: Valid_def itl_defs freverse_d_def )

lemma WRMoreEqvMore:
 "\<turnstile> finite \<longrightarrow> wrev more = more"
by (simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def )


lemma RMoreEqvMoreAlt:
 "\<turnstile>  more\<^sup>r = (more \<and> finite)"
by (auto simp add: Valid_def itl_defs freverse_d_def )

lemma WRMoreEqvMoreAlt:
 "\<turnstile>  wrev more = more"
by (auto simp add: Valid_def itl_defs wfreverse_d_def  freverse_d_def 
    nfinite_conv_nlength_enat zero_enat_def)


lemma REmptyEqvEmpty:
 "\<turnstile> empty\<^sup>r = empty"
by (auto simp add: Valid_def itl_defs freverse_d_def nlength_eq_enat_nfiniteD zero_enat_def)

lemma WREmptyEqvEmpty:
 "\<turnstile> wrev empty = (finite \<longrightarrow> empty)"
by (auto simp add: Valid_def itl_defs wfreverse_d_def  freverse_d_def 
    nlength_eq_enat_nfiniteD zero_enat_def)


lemma RImpRule:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows " \<turnstile> (f\<^sup>r) \<longrightarrow> (g\<^sup>r)"
using assms 
by (auto simp add: Valid_def itl_defs freverse_d_def)

lemma WRImpRule:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows " \<turnstile> (wrev f) \<longrightarrow> (wrev g)"
using assms 
by (auto simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def)


lemma REqvRule:
 assumes "\<turnstile> f = g"
 shows " \<turnstile> (f\<^sup>r) = (g\<^sup>r)"
using assms 
using inteq_reflection by force

lemma WREqvRule:
 assumes "\<turnstile> f = g"
 shows " \<turnstile> (wrev f) = (wrev g)"
using assms
by (simp add: Prop11 WRImpRule) 


lemma FPowerFinite: 
"\<turnstile> (fpower (f \<and> more) n) = ((fpower (f \<and> more) n) \<and> finite) " 
proof (induct n)
case 0
then show ?case 
by (metis EmptyImpFinite Prop10 fpower_d_def wpow_0)
next
case (Suc n)
then show ?case 
  proof -
   have 1: "\<turnstile> fpower (f \<and> more) (Suc n) = (f \<and> fmore);( fpower (f \<and> more) n)" 
     by (metis AndMoreSChopEqvAndFmoreChop fpower_d_def schop_d_def wpow_Suc)
   have 2: "\<turnstile> (f \<and> fmore);( fpower (f \<and> more) n) = 
              ((f \<and> fmore);( fpower (f \<and> more) n) \<and> finite)"
    by (metis ChopImpChop FiniteChopFiniteEqvFinite Prop01 Prop05 Prop10 Suc fmore_d_def int_eq int_iffD1)   
   show ?thesis 
   by (metis "1" "2" inteq_reflection)
  qed
qed


lemma RevFPowerChop:
 "\<turnstile> (fpower (f \<and> more)  n)\<^sup>r = (fpower ((f \<and> more)\<^sup>r) n)"
proof 
 (induct n)
 case 0
 then show ?case using REmptyEqvEmpty 
 by (metis fpower_d_def wpow_0)
 next
 case (Suc n)
 then show ?case 
  proof -
   have 1: "\<turnstile>   (fpower (f \<and> more) (Suc n))\<^sup>r = ((f \<and> fmore);(fpower (f \<and> more) n))\<^sup>r "
    by (metis AndMoreSChopEqvAndFmoreChop ChopEmpty fpower_d_def inteq_reflection schop_d_def wpow_Suc)
   have 2:  "\<turnstile> ((f \<and> fmore);(fpower (f \<and> more) n))\<^sup>r = ((fpower (f \<and> more) n);(f \<and> fmore))\<^sup>r "
       by (metis (no_types, lifting) AndMoreAndFiniteEqvAndFmore FPowerCommute REqvRule inteq_reflection)
   have 3: "\<turnstile> (fpower (f \<and> more) n) = ((fpower (f \<and> more) n) \<and> finite) " 
     by (simp add: FPowerFinite)
   have 4: "\<turnstile> ((fpower (f \<and> more) n);(f \<and> fmore))\<^sup>r = ((f \<and> fmore)\<^sup>r;(fpower (f \<and> more) n)\<^sup>r)"
    by (metis  RevChopAlt )
   show ?thesis 
   by (metis "2" "4" AndMoreAndFiniteEqvAndFmore RevAndFinite Suc fpower_d_def inteq_reflection wpow_Suc)
 qed
qed


lemma FPowerWRevFiniteAbsorb: 
 "\<turnstile> (fpower (finite \<and> wrev (f \<and> more)) n) = (fpower ( wrev (f \<and> more)) n) " 
proof (induct n)
case 0
then show ?case 
by (metis REmptyEqvEmpty fpower_d_def inteq_reflection wpow_0)
next
case (Suc n)
then show ?case
by (metis (no_types, opaque_lifting) EqvReverseReverseAlt EqvWReverseWReverseAlt RevWRev WRevAndFinite 
    WRevRev fpower_d_def int_eq int_simps(27)) 
qed

lemma WRevFPowerChop:
 "\<turnstile> wrev (fpower (f \<and> more)  n) = (finite \<longrightarrow> fpower (wrev (f \<and> more)) n)"
proof -
 have 1: "\<turnstile> wrev (fpower (f \<and> more)  n) = (finite \<longrightarrow> frev (fpower (f \<and> more)  n)) " 
    by (simp add: WRevRev)
 have 2: "\<turnstile> (finite \<longrightarrow> frev (fpower (f \<and> more)  n)) = 
            (finite \<longrightarrow>  (fpower ((f \<and> more)\<^sup>r) n)) " 
    using RevFPowerChop by fastforce
 have 3: "\<turnstile> (fpower ((f \<and> more)\<^sup>r) n) = (fpower (finite \<and> wrev (f \<and> more)) n) " 
      by (metis RevFPowerChop RevWRev inteq_reflection)
 have 4: "\<turnstile> (fpower (finite \<and> wrev (f \<and> more)) n) = (fpower ( wrev (f \<and> more)) n) "
   by (simp add: FPowerWRevFiniteAbsorb)
 show ?thesis 
 using "1" "2" "3" "4" by fastforce
qed

lemma RevChopstar:
 "\<turnstile> (schopstar f)\<^sup>r = (schopstar (f\<^sup>r))"
proof -
 have 1: "\<turnstile> (schopstar f) = (\<exists>n. fpower (f \<and> more) n)"
    by (simp add: FPowerstardef schopstar_d_def) 
 have 2: "\<turnstile> (schopstar f)\<^sup>r = (\<exists> n. fpower (f \<and> more) n)\<^sup>r " 
      using REqvRule "1" by blast
 have 3: "\<turnstile> (\<exists> n. fpower (f \<and> more) n)\<^sup>r = (\<exists> n. (fpower (f \<and> more) n)\<^sup>r)" 
      by (simp add: rev_exists)
 have 4: "\<turnstile> (\<exists> n. (fpower (f \<and> more) n)\<^sup>r) = (\<exists> n. (fpower ((f \<and> more)\<^sup>r) n))" 
      by (simp add: RevFPowerChop ExEqvRule)  
 have 5: "\<turnstile>  (f \<and> fmore)\<^sup>r = (f\<^sup>r \<and> fmore)"
    by (metis (no_types, opaque_lifting) FiniteImpAnd RAnd RMoreEqvMore RevAndFinite 
        inteq_reflection itl_def(26))
 hence 6: "\<turnstile> (\<exists> n. (fpower ((f \<and> more)\<^sup>r) n)) = (\<exists> n. (fpower ((f\<^sup>r \<and> more)) n))"
 by (metis AndMoreAndFiniteEqvAndFmore FPowerstar_WPowerstar FPowerstardef RAnd RMoreEqvMoreAlt 
     fmore_d_def inteq_reflection rev_fun2_alt)
 have 7: "\<turnstile> (\<exists> n. (fpower ((f\<^sup>r \<and> more)) n)) = schopstar (f\<^sup>r)"
  by (simp add: fpowerstar_d_def schopstar_d_def)  
 from 2 3 4 6 7 show ?thesis by fastforce
qed

lemma WRevAndFMore: 
 "\<turnstile> finite \<longrightarrow> wrev (f \<and> fmore) = (wrev f \<and> fmore)" 
proof -
 have 1: "\<turnstile> finite \<longrightarrow> wrev (f \<and> fmore) = (f \<and> fmore)\<^sup>r" 
 using WRevAndFiniteAlt[of "LIFT (f \<and> fmore)"] by auto
 have 2: "\<turnstile> finite \<longrightarrow> (f \<and> fmore)\<^sup>r = (f\<^sup>r \<and> fmore) " 
   by (metis EqvReverseReverseAlt RMoreEqvMoreAlt fmore_d_def inteq_reflection rev_fun2)
 have 3: "\<turnstile> finite \<longrightarrow> (f\<^sup>r \<and> fmore) = (wrev f \<and> fmore) " 
   by (metis (no_types, lifting) "2" AndMoreAndFiniteEqvAndFmore EqvReverseReverseAlt RAnd 
       RMoreEqvMoreAlt WRevAndFiniteAlt fmore_d_def inteq_reflection)
 show ?thesis 
 by (metis (no_types, lifting) "1" AndMoreAndFiniteEqvAndFmore WRAnd WRMoreEqvMoreAlt WRevAndFinite 
     WRevAndFiniteAlt inteq_reflection)
qed

lemma WRevChopstar:
 "\<turnstile> wrev (schopstar f) = (finite \<longrightarrow> schopstar (wrev f))"
proof -
 have 1: "\<turnstile> (schopstar f) = (\<exists>n. fpower (f \<and> more) n)"
    by (simp add: FPowerstardef schopstar_d_def) 
 have 2: "\<turnstile> wrev (schopstar f) = wrev (\<exists> n. fpower (f \<and> more) n) " 
      using WREqvRule "1" by blast
 have 3: "\<turnstile> wrev (\<exists> n. fpower (f \<and> more) n) = (\<exists> n. wrev (fpower (f \<and> more) n))" 
      by (simp add: wrev_exists)
 have 4: "\<turnstile> (\<exists> n. wrev (fpower (f \<and> more) n)) = (\<exists> n. (finite \<longrightarrow> fpower (wrev (f \<and> more)) n))" 
      by (simp add: WRevFPowerChop ExEqvRule)  
 have 5: "\<turnstile>  finite \<longrightarrow> wrev (f \<and> fmore) = (wrev f \<and> fmore)"
     by (simp add: WRevAndFMore)
 hence 6: "\<turnstile> (\<exists> n. (finite \<longrightarrow> fpower (wrev (f \<and> more)) n)) = 
             (\<exists> n. (finite \<longrightarrow> fpower ((wrev f \<and> more)) n))"
      by (metis (no_types, lifting) ExEqvRule WRAnd WRMoreEqvMoreAlt WRevFPowerChop inteq_reflection)
 have 65: "\<turnstile> (\<exists> n. (finite \<longrightarrow> fpower ((wrev f \<and> more)) n)) = 
             (finite \<longrightarrow> (\<exists> n. ( fpower ((wrev f \<and> more)) n))) " 
      by auto
 have 7: "\<turnstile> (finite \<longrightarrow> (\<exists> n. ( fpower ((wrev f \<and> more)) n))) =
            (finite \<longrightarrow> schopstar (wrev f))" 
  by (simp add: fpowerstar_d_def schopstar_d_def)  
 from 2 3 4 6 7 show ?thesis 
 by (metis "65" inteq_reflection)
qed

lemma RevLen: 
 "\<turnstile> (len k)\<^sup>r = len k" 
by (auto simp add: Valid_def itl_defs freverse_d_def len_defs nlength_eq_enat_nfiniteD)

lemma WRevLen:
 "\<turnstile> wrev (len k) = (finite \<longrightarrow> len k)"
by (auto simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def len_defs nlength_eq_enat_nfiniteD)

lemma NotRevOmega: 
 "\<turnstile> \<not> frev (omega f) "
 by (meson MP NotRevInfinite RImpRule lift_imp_neg omega_imp_inf)

lemma WRevOmega: 
"\<turnstile> wrev (omega f) = inf "
proof -
 have 1: "\<turnstile> wrev (omega f) \<longrightarrow> inf" 
   by (metis WRImpRule WRevInfinite inteq_reflection omega_imp_inf) 
 have 2: "\<turnstile> inf \<longrightarrow> wrev (omega f) "
   by (simp add: freverse_d_def intI itl_defs(4) wfreverse_d_def) 
 show ?thesis
 by (simp add: "1" "2" Prop11) 
qed
   


lemmas all_rev = rev_const rev_fun1 rev_fun2 rev_fun3 rev_fun1_alt rev_fun2_alt rev_fun3_alt
       rev_forall rev_exists rev_exists1 rev_current rev_next rev_penult rev_fin RevSkip 
       RevChop RevChopAlt RevChopstar RevFinite RevAndFinite RevLen

lemmas all_wrev =  wrev_fun1 wrev_fun2 wrev_fun3 wrev_fun1_alt wrev_fun2_alt wrev_fun3_alt
       wrev_forall wrev_exists  wrev_exists1 WRevSkip 
       WRevChop WRevChopAlt WRevChopstar WRevFinite WRevAndFinite WRevLen

lemmas all_rev_b = rev_fun1_alt rev_fun2_alt rev_fun3_alt  rev_forall rev_exists RevChopAlt
  rev_exists1    RevSkip  RevChopstar RevFinite RevAndFinite RevLen

lemmas all_wrev_b =  wrev_fun1_alt wrev_fun2_alt wrev_fun3_alt
       wrev_forall wrev_exists  WRevSkip WRevChopAlt WRevChopstar WRevAndFinite WRevLen

lemmas all_rev_unl = all_rev[THEN intD]

lemmas all_wrev_unl = all_wrev[THEN intD]

lemmas all_rev_eq = all_rev_b[THEN inteq_reflection]

lemmas all_wrev_eq = all_wrev_b[THEN inteq_reflection]


subsection \<open>Properties of Time Reversal\<close>

lemma NotRev:
 "\<turnstile> (\<not>(f\<^sup>r)) = (wrev (\<not>f)) "
by (simp add: wfreverse_d_def) 

lemma NotWRev:
 "\<turnstile> (\<not>(wrev f)) = (\<not>f)\<^sup>r" 
  by (simp add: wfreverse_d_def)


lemma RNot:
 "\<turnstile> finite \<longrightarrow> (\<not>f)\<^sup>r = (\<not> f\<^sup>r)"
by (simp add: rev_fun1)

lemma RNotAlt:
 "\<turnstile>  (\<not>f)\<^sup>r = ((\<not> f\<^sup>r) \<and> finite)"
by (simp add: rev_fun1_alt)

lemma WRNot:
 "\<turnstile> finite \<longrightarrow> wrev (\<not>f) = (\<not> (wrev f))"
by (simp add: wrev_fun1)

lemma WRNotAlt:
 "\<turnstile>  wrev (\<not>f) = (finite \<longrightarrow> \<not> (wrev f) )"
by (simp add: wrev_fun1_alt)

lemma RRNot:
 "\<turnstile> finite \<longrightarrow>(\<not>(f\<^sup>r))\<^sup>r = (\<not>f)"
by (simp add: freverse_d_def intI itl_defs(5) nfinite_nrev nrev_nrev)

lemma WRWRNot:
 "\<turnstile> finite \<longrightarrow> wrev (\<not>(wrev f)) = (\<not>f) " 
  by (metis (no_types, lifting) EqvWReverseWReverse EqvWReverseWReverseAlt WRevAndFiniteAlt WRevRev
       all_wrev_eq(1) all_wrev_eq(9) inteq_reflection)

lemma RRNotAlt:
 "\<turnstile> (\<not>(f\<^sup>r))\<^sup>r = (\<not>f \<and> finite)"
by (auto simp add: Valid_def freverse_d_def intI itl_defs(5) nfinite_nrev nrev_nrev )

lemma WRWRNotAlt:
 "\<turnstile>  wrev (\<not>(wrev f)) = (finite \<longrightarrow> \<not>f) " 
by (metis EqvReverseReverseAlt EqvWReverseWReverseAlt WRevRev all_wrev_eq(9) int_simps(4) 
    inteq_reflection wfreverse_d_def)

lemma RTrue:
 "\<turnstile> finite \<longrightarrow> (#True)\<^sup>r = #True"
by (metis int_simps(1) inteq_reflection rev_fun2)

lemma WRTrue:
 "\<turnstile> finite \<longrightarrow> wrev (#True) = #True"
by (metis int_simps(1) inteq_reflection wrev_fun2)

lemma RTrueAlt:
 "\<turnstile>  (#True)\<^sup>r = finite"
by (simp add: freverse_d_def intI itl_defs(5))

lemma WRTrueAlt:
 "\<turnstile>  wrev (#True) "
by (meson Prop11 Prop12 WReverseEqvAlt int_simps(17))

lemma ROr:
 "\<turnstile> finite \<longrightarrow>(f \<or> g)\<^sup>r = (f\<^sup>r \<or> g\<^sup>r)"
by (simp add: rev_fun2)

lemma WROr:
 "\<turnstile> finite \<longrightarrow> wrev (f \<or> g) = (wrev f \<or> wrev g)"
by (simp add: wrev_fun2)


lemma ROrAlt:
 "\<turnstile> (f \<or> g)\<^sup>r = (f\<^sup>r \<or> g\<^sup>r)"
by (auto simp add: Valid_def freverse_d_def) 

lemma WROrAlt:
 "\<turnstile> wrev (f \<or> g) = (wrev f \<or> wrev g)"
by (auto simp add: Valid_def wfreverse_d_def freverse_d_def) 

lemma RROr:
 "\<turnstile> finite \<longrightarrow> (f\<^sup>r \<or> g\<^sup>r)\<^sup>r = (f \<or> g)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow>(f\<^sup>r \<or> g\<^sup>r)\<^sup>r = ((f\<^sup>r)\<^sup>r \<or> (g\<^sup>r)\<^sup>r)" using ROr by blast
 have 2: "\<turnstile> finite \<longrightarrow> ((f\<^sup>r)\<^sup>r \<or> (g\<^sup>r)\<^sup>r) = (f \<or> g)" using EqvReverseReverse 
 by (simp add: freverse_d_def intI itl_defs(5) nfinite_nrev nrev_nrev) 
 from 1 2 show ?thesis by fastforce
qed

lemma WRWROr:
 "\<turnstile> finite \<longrightarrow> wrev (wrev f \<or> wrev g) = (f \<or> g)"
by (metis EqvWReverseWReverse WROrAlt inteq_reflection)


lemma RROrAlt:
 "\<turnstile>  (f\<^sup>r \<or> g\<^sup>r)\<^sup>r = ((f \<or> g) \<and> finite)"
by (auto simp add:  Valid_def freverse_d_def itl_defs nfinite_nrev nrev_nrev) 

lemma WRWROrAlt:
 "\<turnstile>  wrev (wrev f \<or> wrev g) = (finite \<longrightarrow> f \<or> g)"
by (metis EqvWReverseWReverseAlt WROrAlt inteq_reflection)


lemma RRAnd:
 " \<turnstile> finite \<longrightarrow> (f\<^sup>r \<and> g\<^sup>r)\<^sup>r = (f \<and> g)"
proof -
 have 1: "\<turnstile> (f\<^sup>r \<and> g\<^sup>r)\<^sup>r = ((f\<^sup>r)\<^sup>r \<and> (g\<^sup>r)\<^sup>r)" using RAnd by blast
 have 2: "\<turnstile> finite \<longrightarrow> ((f\<^sup>r)\<^sup>r \<and> (g\<^sup>r)\<^sup>r) = (f \<and> g)"
   by (metis EqvReverseReverse RAnd inteq_reflection)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRAnd:
 " \<turnstile> finite \<longrightarrow> wrev (wrev f \<and> wrev g) = (f \<and> g)"
by (metis EqvWReverseWReverse WRAnd inteq_reflection)

lemma RRAndAlt:
 " \<turnstile>  (f\<^sup>r \<and> g\<^sup>r)\<^sup>r = (f \<and> g \<and> finite)"
by (auto simp add:  Valid_def freverse_d_def itl_defs nfinite_nrev nrev_nrev) 

lemma WRWRAndAlt:
 " \<turnstile>  wrev (wrev f \<and> wrev g) = (finite \<longrightarrow> (f \<and> g))"
by (metis EqvWReverseWReverseAlt WRAnd inteq_reflection)

lemma RAndEmptyEqvAndEmpty:
 "\<turnstile> (f \<and> empty)\<^sup>r = (f \<and> empty)"
by ( simp add: Valid_def itl_defs freverse_d_def)
 (metis ndropn_0 ndropn_nlast nlength_eq_enat_nfiniteD nrev_NNil the_enat_0 zero_enat_def)

lemma WRAndEmptyEqvAndEmpty:
 "\<turnstile> wrev (f \<and> empty) = (finite \<longrightarrow> (f \<and> empty))"
by ( simp add: Valid_def itl_defs wfreverse_d_def freverse_d_def)
 (metis ndropn_0 ndropn_nlast nrev_NNil the_enat_0)

lemma RNextEqvPrev:
 "\<turnstile> finite \<longrightarrow> (\<circle> f)\<^sup>r = prev (f\<^sup>r)"
by (metis RevChop RevSkip inteq_reflection next_d_def prev_d_def)

lemma WRNextEqvPrev:
 "\<turnstile> finite \<longrightarrow> wrev (\<circle> f) = prev (wrev f)"
proof -
 have 3: "\<turnstile>  finite \<longrightarrow> wrev f ; (finite \<longrightarrow> skip) = wrev f ; ( skip)  " 
   by (metis EqvWReverseWReverse EqvWReverseWReverseAlt FiniteRightChopEqvChop inteq_reflection)  
 show ?thesis 
 by (metis "3" WRevRev all_rev_eq(8) all_wrev_eq(7) inteq_reflection next_d_def prev_d_def)
qed


lemma RNextEqvPrevAlt:
 "\<turnstile> (\<circle> f)\<^sup>r = prev (f\<^sup>r)"
 by (metis RevChopAlt all_rev_eq(8) next_d_def prev_d_def)

lemma WRNextEqvPrevAlt:
 "\<turnstile>  wrev (\<circle> f) = (finite \<longrightarrow> prev (wrev f))"
by (metis (no_types, lifting) EqvWReverseWReverseAlt WRevAndFiniteAlt WRevRev all_rev_eq(8) 
    all_wrev_eq(7) all_wrev_eq(9) inteq_reflection next_d_def prev_d_def)

lemma RRNextEqvPrev:
 "\<turnstile> finite \<longrightarrow>(\<circle> (f\<^sup>r))\<^sup>r = prev (f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow>(\<circle> (f\<^sup>r))\<^sup>r = prev ((f\<^sup>r)\<^sup>r)" using RNextEqvPrev by blast
 have 2: "\<turnstile> finite \<longrightarrow> (f\<^sup>r)\<^sup>r = f"
     by (simp add: EqvReverseReverse) 
 have 3: "\<turnstile> finite \<longrightarrow> prev ((f\<^sup>r)\<^sup>r) = prev f" 
  unfolding prev_d_def using 2 by (simp add: FiniteChopEqv)
 from 1 3 show ?thesis using FiniteTransEqv by blast
qed

lemma WRWRNextEqvPrev:
 "\<turnstile> finite \<longrightarrow> wrev (\<circle> (wrev f)) = prev (f)"
by (metis EqvWReverseWReverse FiniteChopEqv FiniteTransEqv WRNextEqvPrev prev_d_def)

lemma RRNextEqvPrevAlt:
 "\<turnstile> (\<circle> (f\<^sup>r))\<^sup>r = (finite \<and> prev f )"
by (metis EqvReverseReverseAlt RevChopAlt all_rev_eq(8) inteq_reflection lift_and_com next_d_def prev_d_def)

lemma WRWRNextEqvPrevAlt:
 "\<turnstile>  wrev (\<circle> (wrev f)) = (finite \<longrightarrow> prev  f)"
by (metis (no_types, lifting) EqvWReverseWReverseAlt WRevAndFiniteAlt all_rev_eq(8) 
    all_wrev_eq(7) all_wrev_eq(9) inteq_reflection next_d_def prev_d_def)

lemma RWNextEqvWPrev:
 "\<turnstile> finite \<longrightarrow> (wnext f)\<^sup>r = wprev(f\<^sup>r)"
proof -
 have 1: "\<turnstile> finite\<longrightarrow>(wnext f)\<^sup>r = (empty \<or> next f)\<^sup>r"
 by (metis ROr ROrAlt WnextEqvEmptyOrNext inteq_reflection) 
 have 2: "\<turnstile> finite \<longrightarrow> (empty \<or> next f)\<^sup>r = (empty\<^sup>r \<or> (next f)\<^sup>r) " 
   by (simp add: ROr)
 have 3: "\<turnstile> finite \<longrightarrow> (empty\<^sup>r \<or> (next f)\<^sup>r) = (empty \<or> (prev (f\<^sup>r)))" 
   using REmptyEqvEmpty RNextEqvPrev[of f] by fastforce 
 have 4: "\<turnstile> finite \<longrightarrow> (empty \<or> (prev (f\<^sup>r))) = wprev(f\<^sup>r)"
     by (metis WprevEqvEmptyOrPrev int_simps(1) int_simps(12) inteq_reflection)
 show ?thesis 
 by (metis "3" FiniteTransEqv WnextEqvEmptyOrNext WprevEqvEmptyOrPrev inteq_reflection rev_fun2)   
qed

lemma WRWNextEqvWPrev:
 "\<turnstile> finite \<longrightarrow> wrev (wnext f) = wprev(wrev f)"
by (metis (no_types, opaque_lifting) EqvReverseReverse EqvWReverseWReverseAlt FiniteTransEqv
    RNextEqvPrevAlt WRevRev int_simps(4) inteq_reflection wfreverse_d_def wnext_d_def 
    wprev_d_def wrev_fun1)


lemma RWNextEqvWPrevAlt:
 "\<turnstile> (wnext f)\<^sup>r = wprev(f\<^sup>r)"
proof -
 have 1: "\<turnstile> (wnext f)\<^sup>r = (empty \<or> next f)\<^sup>r"
 by (simp add: REqvRule WnextEqvEmptyOrNext)
 have 2: "\<turnstile>  (empty \<or> next f)\<^sup>r = (empty\<^sup>r \<or> (next f)\<^sup>r) "
 by (simp add: ROrAlt) 
 have 3: "\<turnstile>  (empty\<^sup>r \<or> (next f)\<^sup>r) = (empty \<or> (prev (f\<^sup>r)))"
 by (metis "2" REmptyEqvEmpty RNextEqvPrevAlt inteq_reflection) 
 have 4: "\<turnstile>  (empty \<or> (prev (f\<^sup>r))) = wprev(f\<^sup>r)"
 by (meson WprevEqvEmptyOrPrev int_iffD1 int_iffD2 int_iffI)
 show ?thesis
 by (metis "1" "2" "3" "4" inteq_reflection) 
qed

lemma WRWNextEqvWPrevAlt:
 "\<turnstile>  wrev (wnext f) = wprev(wrev f)"
by (metis (no_types, lifting) RNextEqvPrevAlt int_simps(4) inteq_reflection itl_def(20)
    wfreverse_d_def wprev_d_def)

lemma RRWNextEqvWPrev:
 "\<turnstile> finite \<longrightarrow> (wnext (f\<^sup>r))\<^sup>r = wprev(f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow>(wnext (f\<^sup>r))\<^sup>r = wprev ((f\<^sup>r)\<^sup>r)" using RWNextEqvWPrev by blast
 have 2: "\<turnstile> finite \<longrightarrow> (f\<^sup>r)\<^sup>r = f" 
    by (simp add: EqvReverseReverse)
 have 3: "\<turnstile> finite \<longrightarrow> wprev ((f\<^sup>r)\<^sup>r) = wprev f" 
    by (simp add: "2" FiniteWPrevEqv)
 from 1 3 show ?thesis using FiniteTransEqv by blast
qed

lemma WRWRWNextEqvWPrev:
 "\<turnstile> finite \<longrightarrow> wrev (wnext (wrev f)) = wprev(f)"
using EqvWReverseWReverse FiniteTransEqv FiniteWPrevEqv WRWNextEqvWPrev by blast

lemma RRWNextEqvWPrevAlt:
 "\<turnstile>  (wnext (f\<^sup>r))\<^sup>r = (finite \<and> wprev f)"
by (metis FiniteImpAnd ROrAlt RRWNextEqvWPrev WnextEqvEmptyOrNext all_rev_eq(2) inteq_reflection lift_and_com)

lemma WRWRWNextEqvWPrevAlt:
 "\<turnstile>  wrev (wnext (wrev f)) = (finite \<longrightarrow> wprev(f)) "
using WRWRWNextEqvWPrev WRevRev by fastforce

lemma RPrevEqvNext:
"\<turnstile> finite \<longrightarrow> (prev f)\<^sup>r = \<circle> (f\<^sup>r)"
by (metis RevChop RevSkip inteq_reflection next_d_def prev_d_def)

lemma WRPrevEqvNext:
"\<turnstile> finite \<longrightarrow> wrev (prev f) = \<circle> (wrev f)"
by (metis (no_types, opaque_lifting) EqvWReverseWReverse EqvWReverseWReverseAlt FiniteChopEqv 
    WRevRev all_rev_eq(8) all_wrev_eq(7) inteq_reflection next_d_def prev_d_def)


lemma RPrevEqvNextAlt:
"\<turnstile>  (prev f)\<^sup>r = \<circle> (f\<^sup>r)"
by (metis RevChopAlt all_rev_eq(8) next_d_def prev_d_def)

lemma WRPrevEqvNextAlt:
"\<turnstile>  wrev (prev f) = (finite \<longrightarrow> \<circle> (wrev f))"
by (metis (no_types, lifting) EqvWReverseWReverseAlt WRWRNextEqvPrevAlt WRevAndFiniteAlt WRevRev 
    all_rev_eq(10) all_rev_eq(2) inteq_reflection)

lemma RRPrevEqvNext:
"\<turnstile> finite \<longrightarrow> (prev (f\<^sup>r))\<^sup>r = \<circle> (f)"
proof -
 have 1: "\<turnstile> finite\<longrightarrow>(prev (f\<^sup>r))\<^sup>r = \<circle> ((f\<^sup>r)\<^sup>r)" using RPrevEqvNext by blast
 have 2: "\<turnstile> finite \<longrightarrow>\<circle> ((f\<^sup>r)\<^sup>r) = \<circle> f"
   by (simp add: EqvReverseReverse FiniteRightChopEqvChop next_d_def)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRPrevEqvNext:
"\<turnstile> finite \<longrightarrow> wrev (prev (wrev f)) = \<circle> (f)"
by (metis EqvWReverseWReverse FiniteRightChopEqvChop FiniteTransEqv WRPrevEqvNext next_d_def)


lemma RRPrevEqvNextAlt:
"\<turnstile>  (prev (f\<^sup>r))\<^sup>r = (finite \<and> \<circle> f)"
by (metis EqvReverseReverseAlt RNextEqvPrevAlt inteq_reflection lift_and_com)

lemma WRWRPrevEqvNextAlt:
"\<turnstile>  wrev (prev (wrev f)) = (finite \<longrightarrow> \<circle> f)"
by (metis EqvWReverseWReverseAlt WRNextEqvPrevAlt WRPrevEqvNextAlt WRWRNextEqvPrevAlt inteq_reflection)

lemma RWPrevEqvWNext:
 "\<turnstile> finite \<longrightarrow>(wprev f)\<^sup>r = wnext(f\<^sup>r)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (wprev f)\<^sup>r = (empty \<or> prev f)\<^sup>r"
   by (metis WprevEqvEmptyOrPrev int_simps(1) int_simps(12) inteq_reflection) 
 have 2: "\<turnstile> finite \<longrightarrow> (empty \<or> prev f)\<^sup>r = (empty\<^sup>r \<or> (prev f)\<^sup>r)"
   by (simp add: ROr) 
 have 3: "\<turnstile> finite \<longrightarrow> (prev f)\<^sup>r = (next (f\<^sup>r))" 
   by (simp add: RPrevEqvNext)
 have 4: "\<turnstile> finite \<longrightarrow> empty\<^sup>r = empty" 
   using REmptyEqvEmpty by auto
 have 5: "\<turnstile> finite \<longrightarrow> (empty\<^sup>r \<or> (prev f)\<^sup>r) = (empty \<or> (next (f\<^sup>r)))"
   by (simp add: "3" "4" FiniteOrEqv)
 have 6: "\<turnstile> finite \<longrightarrow> (empty \<or> (next (f\<^sup>r))) = wnext(f\<^sup>r) " 
   by (metis WnextEqvEmptyOrNext int_simps(1) int_simps(12) inteq_reflection)
 show ?thesis 
 by (metis "5" FiniteTransEqv WnextEqvEmptyOrNext WprevEqvEmptyOrPrev inteq_reflection rev_fun2)
qed

lemma WRWPrevEqvWNext:
 "\<turnstile> finite \<longrightarrow> wrev (wprev f) = wnext(wrev f)"
by (metis (no_types, opaque_lifting) RPrevEqvNextAlt WRTrueAlt WReverseEqvAlt int_simps(1)
     int_simps(4) inteq_reflection wfreverse_d_def wnext_d_def wprev_d_def)

lemma RWPrevEqvWNextAlt:
 "\<turnstile> (wprev f)\<^sup>r = wnext(f\<^sup>r)"
proof -
 have 1: "\<turnstile>  (wprev f)\<^sup>r = (empty \<or> prev f)\<^sup>r"
   by (metis WprevEqvEmptyOrPrev int_simps(1)  inteq_reflection) 
 have 2: "\<turnstile>  (empty \<or> prev f)\<^sup>r = (empty\<^sup>r \<or> (prev f)\<^sup>r)"
   by (simp add: ROrAlt) 
 have 3: "\<turnstile>  (prev f)\<^sup>r = (next (f\<^sup>r))"
 by (simp add: RPrevEqvNextAlt) 
 have 4: "\<turnstile>  empty\<^sup>r = empty" 
   using REmptyEqvEmpty by auto
 have 5: "\<turnstile>  (empty\<^sup>r \<or> (prev f)\<^sup>r) = (empty \<or> (next (f\<^sup>r)))"
 by (metis "2" "3" REmptyEqvEmpty inteq_reflection)
 have 6: "\<turnstile>  (empty \<or> (next (f\<^sup>r))) = wnext(f\<^sup>r) " 
   by (metis WnextEqvEmptyOrNext  inteq_reflection)
 show ?thesis
 by (metis "1" "2" "5" "6" inteq_reflection) 
qed

lemma WRWPrevEqvWNextAlt:
 "\<turnstile>  wrev (wprev f) = (finite \<longrightarrow> wnext(wrev f))"
by (metis (no_types, opaque_lifting) EqvReverseReverseAlt RPrevEqvNextAlt WRWRNotAlt 
    WRevAndFiniteAlt int_simps(4) inteq_reflection wfreverse_d_def wnext_d_def wprev_d_def)

lemma RRWPrevEqvWNext:
 "\<turnstile> finite \<longrightarrow> (wprev (f\<^sup>r))\<^sup>r = wnext(f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (wprev (f\<^sup>r))\<^sup>r = wnext ((f\<^sup>r)\<^sup>r)" using RWPrevEqvWNext by blast
 have 2: "\<turnstile> finite \<longrightarrow> wnext ((f\<^sup>r)\<^sup>r) = wnext f" 
   by (simp add: EqvReverseReverse FiniteWNextEqv)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRWPrevEqvWNext:
 "\<turnstile> finite \<longrightarrow> wrev (wprev (wrev f)) = wnext(f)"
by (metis EqvWReverseWReverse WRWNextEqvWPrevAlt inteq_reflection)


lemma RRWPrevEqvWNextAlt:
 "\<turnstile>  (wprev (f\<^sup>r))\<^sup>r = (finite \<and> wnext f)"
by (meson EqvReverseReverseAlt Prop04 REqvRule RWNextEqvWPrevAlt lift_and_com)

lemma WRWRWPrevEqvWNextAlt:
 "\<turnstile>  wrev (wprev (wrev f)) = (finite \<longrightarrow> wnext f)"
by (metis EqvWReverseWReverseAlt WRWNextEqvWPrevAlt inteq_reflection)


lemma RDiamondEqvDi:
 "\<turnstile> finite \<longrightarrow> (\<diamond>f)\<^sup>r = di (f\<^sup>r)"
 by (metis FiniteRightChopEqvChop FiniteTransEqv Prop11 RevChop all_rev_eq(10) di_d_def 
    int_simps(9) sometimes_d_def)

lemma WRDiamondEqvDi:
 "\<turnstile> finite \<longrightarrow> wrev (\<diamond>f) = di (wrev f)"
unfolding di_d_def sometimes_d_def
by (metis FiniteRightChopEqvChop FiniteTransEqv Prop11 WRevChop WReverseEqv  int_simps(9)
    lift_imp_trans)

lemma RDiamondEqvDiAlt:
 "\<turnstile>  (\<diamond>f)\<^sup>r =  (f\<^sup>r);finite"
 by (metis RevChopAlt all_rev_eq(10) sometimes_d_def)

lemma WRDiamondEqvDiAlt:
 "\<turnstile>  wrev (\<diamond>f) = di (wrev f)"
unfolding di_d_def sometimes_d_def
by (metis WRevRev all_rev_eq(10) all_wrev_eq(7) int_simps(13) inteq_reflection )

lemma RRDiamondEqvDi:
 "\<turnstile> finite \<longrightarrow> (\<diamond>(f\<^sup>r))\<^sup>r = di (f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (\<diamond> (f\<^sup>r))\<^sup>r = di ((f\<^sup>r)\<^sup>r)" using RDiamondEqvDi by blast
 have 2: "\<turnstile> finite \<longrightarrow>di ((f\<^sup>r)\<^sup>r) = di f"
  by (simp add: EqvReverseReverse FiniteChopEqv di_d_def)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRDiamondEqvDi:
 "\<turnstile> finite \<longrightarrow> wrev (\<diamond>(wrev f)) = di (f)"
by (metis EqvWReverseWReverse FiniteChopEqv FiniteTransEqv WRDiamondEqvDi di_d_def)

lemma RRDiamondEqvDiAlt:
 "\<turnstile>  (\<diamond>(f\<^sup>r))\<^sup>r =  (finite \<and> di f)"
by (metis (no_types, opaque_lifting) EqvReverseReverseAlt FiniteImpAnd RRAndAlt RRDiamondEqvDi 
    all_rev_eq(10) all_rev_eq(6) inteq_reflection lift_and_com sometimes_d_def)

lemma WRWRDiamondEqvDiAlt:
 "\<turnstile>  wrev (\<diamond>(wrev f)) =  (finite \<longrightarrow> di f)"
using WRWRDiamondEqvDi WRevRev by fastforce


lemma RBoxEqvBi:
 "\<turnstile> finite \<longrightarrow> (\<box> f)\<^sup>r = bi (f\<^sup>r)"
unfolding always_d_def bi_d_def 
by (metis FiniteChopEqv FiniteNotEqv FiniteTransEqv RDiamondEqvDi RNot di_d_def)

lemma WRBoxEqvBi:
 "\<turnstile> finite \<longrightarrow> wrev (\<box> f) = bi (wrev f)"
unfolding always_d_def bi_d_def 
by (simp add: FiniteNotEqv RDiamondEqvDi wfreverse_d_def)

lemma RBoxEqvBiAlt:
 "\<turnstile>  (\<box> f)\<^sup>r = (bi (f\<^sup>r) \<and> finite)"
 by (metis FiniteImpAnd RAnd RBoxEqvBi RTrueAlt int_simps(17) inteq_reflection lift_and_com)

lemma WRBoxEqvBiAlt:
 "\<turnstile>  wrev (\<box> f) = (finite \<longrightarrow> bi (wrev f))"
 using WRBoxEqvBi WRevRev by fastforce


lemma RRBoxEqvBi:
 "\<turnstile> finite \<longrightarrow> (\<box> (f\<^sup>r))\<^sup>r = bi (f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (\<box> (f\<^sup>r))\<^sup>r = bi ((f\<^sup>r)\<^sup>r)" using RBoxEqvBi by blast
 have 2: "\<turnstile> finite \<longrightarrow> bi ((f\<^sup>r)\<^sup>r) = bi f" 
  by (simp add: EqvReverseReverse FiniteChopEqv FiniteNotEqv bi_d_def di_d_def)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRBoxEqvBi:
 "\<turnstile> finite \<longrightarrow> wrev (\<box> (wrev f)) = bi (f)"
 by (metis (no_types, opaque_lifting) EqvWReverseWReverse FiniteChopEqv FiniteNotEqv FiniteTransEqv 
     WRBoxEqvBi bi_d_def di_d_def)


lemma RRBoxEqvBiAlt:
 "\<turnstile>  (\<box> (f\<^sup>r))\<^sup>r = (bi f \<and> finite)"
 by (metis FiniteImpAnd RAnd RRBoxEqvBi RTrueAlt int_simps(17) inteq_reflection lift_and_com)

lemma WRWRBoxEqvBiAlt:
 "\<turnstile>  wrev (\<box> (wrev f)) = (finite \<longrightarrow> bi f)"
 using WRWRBoxEqvBi WRevRev by fastforce

lemma RDiEqvDiamond:
 "\<turnstile> finite \<longrightarrow> (di f)\<^sup>r = \<diamond> (f\<^sup>r)"
unfolding sometimes_d_def di_d_def
by (meson FiniteChopEqv FiniteTransEqv Prop11 RTrue RevChop int_simps(8))

lemma WRDiEqvDiamond:
 "\<turnstile> finite \<longrightarrow> wrev (di f) = \<diamond> (wrev f)"
 by (metis (no_types, opaque_lifting) EqvWReverseWReverseAlt RTrueAlt WRevAndFiniteAlt
     all_rev_eq(2) all_rev_eq(6) all_wrev_eq(2) all_wrev_eq(9) di_d_def int_simps(1) int_simps(12)
     inteq_reflection sometimes_d_def)


lemma RDiEqvDiamondAlt:
 "\<turnstile>  (di f)\<^sup>r = \<diamond> (f\<^sup>r)"
 by (metis RTrueAlt RevChopAlt di_d_def inteq_reflection sometimes_d_def)

lemma WRDiEqvDiamondAlt:
 "\<turnstile>  wrev (di f) = (finite \<longrightarrow> \<diamond> (wrev f))"
 by (metis (no_types, lifting) EqvWReverseWReverseAlt WRWRDiamondEqvDiAlt WRevAndFiniteAlt
     WRevRev all_wrev_eq(9) inteq_reflection)


lemma RRDiEqvDiamond:
 "\<turnstile> finite \<longrightarrow> (di (f\<^sup>r))\<^sup>r = \<diamond> (f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (di (f\<^sup>r))\<^sup>r = \<diamond> ((f\<^sup>r)\<^sup>r)" using RDiEqvDiamond by blast
 have 2: "\<turnstile> finite \<longrightarrow> \<diamond> ((f\<^sup>r)\<^sup>r) = \<diamond> f" 
   by (simp add: EqvReverseReverse FiniteRightChopEqvChop sometimes_d_def)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRDiEqvDiamond:
 "\<turnstile> finite \<longrightarrow> wrev (di (wrev f)) = \<diamond> (f)"
 by (metis EqvWReverseWReverse WRDiamondEqvDiAlt int_eq)



lemma RRDiEqvDiamondAlt:
 "\<turnstile>  (di (f\<^sup>r))\<^sup>r = (finite \<and> \<diamond> f)"
 by (metis EqvReverseReverseAlt RDiEqvDiamondAlt all_rev_eq(10) all_rev_eq(6) inteq_reflection
     lift_and_com sometimes_d_def)

lemma WRWRDiEqvDiamondAlt:
 "\<turnstile>  wrev (di (wrev f)) = (finite \<longrightarrow> \<diamond> f)"
 by (meson EqvWReverseWReverseAlt Prop04 WRDiamondEqvDiAlt WREqvRule)

lemma RBiEqvBox:
 "\<turnstile> finite \<longrightarrow>(bi f)\<^sup>r = \<box> (f\<^sup>r)"
unfolding always_d_def bi_d_def 
by (metis FiniteNotEqv FiniteRightChopEqvChop FiniteTransEqv RDiEqvDiamond RNot sometimes_d_def)

lemma WRBiEqvBox:
 "\<turnstile> finite \<longrightarrow> wrev (bi f) = \<box> (wrev f)"
unfolding always_d_def bi_d_def
by (metis (no_types, opaque_lifting) EqvReverseReverse EqvReverseReverseAlt FiniteTransEqv
    RDiEqvDiamondAlt all_rev_eq(1) int_simps(4) inteq_reflection wfreverse_d_def wrev_fun1) 



lemma RBiEqvBoxAlt:
 "\<turnstile> (bi f)\<^sup>r = (\<box> (f\<^sup>r) \<and> finite)"
 by (metis FiniteImpAnd RAnd RBiEqvBox RTrueAlt int_simps(17) inteq_reflection lift_and_com)


lemma WRBiEqvBoxAlt:
 "\<turnstile> wrev (bi f) = (finite \<longrightarrow> \<box> (wrev f))"
 using WRBiEqvBox WRevRev by fastforce

lemma RRBiEqvBox:
 "\<turnstile> finite \<longrightarrow> (bi (f\<^sup>r))\<^sup>r = \<box> (f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (bi (f\<^sup>r))\<^sup>r = \<box> ((f\<^sup>r)\<^sup>r)" using RBiEqvBox by blast
 have 2: "\<turnstile> finite\<longrightarrow>\<box> ((f\<^sup>r)\<^sup>r) = \<box> f" 
  by (simp add: EqvReverseReverse FiniteNotEqv FiniteRightChopEqvChop always_d_def sometimes_d_def)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRBiEqvBox:
 "\<turnstile> finite \<longrightarrow> wrev (bi (wrev f)) = \<box> (f)"
 by (metis (no_types, opaque_lifting) EqvWReverseWReverse FiniteNotEqv FiniteRightChopEqvChop 
     FiniteTransEqv WRBiEqvBox always_d_def sometimes_d_def)

lemma RRBiEqvBoxAlt:
 "\<turnstile>  (bi (f\<^sup>r))\<^sup>r = (\<box> f \<and> finite)"
 by (meson FiniteImpAnd Prop04 Prop11 RBiEqvBox RBiEqvBoxAlt RRBiEqvBox)

lemma WRWRBiEqvBoxAlt:
 "\<turnstile>  wrev (bi (wrev f)) = (finite \<longrightarrow> \<box> f)"
 by (metis EqvWReverseWReverseAlt WRBiEqvBoxAlt WRBoxEqvBiAlt WRWRBoxEqvBiAlt inteq_reflection)


lemma RDaEqvDa:
 "\<turnstile> finite \<longrightarrow> (da f)\<^sup>r = da(f\<^sup>r)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (finite;(f;#True))\<^sup>r = (f;#True)\<^sup>r; finite\<^sup>r" using RevChop by blast
 have 2: "\<turnstile> finite \<longrightarrow> (f;#True)\<^sup>r; finite\<^sup>r =  (f;#True)\<^sup>r; finite"
 by (simp add: all_rev_eq(10))
 have 3: "\<turnstile> finite \<longrightarrow> (f;#True)\<^sup>r; finite = (#True\<^sup>r;f\<^sup>r);finite "
   by (simp add: FiniteChopEqv RevChop)
 have 4: "\<turnstile> finite \<longrightarrow>(#True\<^sup>r;f\<^sup>r);finite = (#True;f\<^sup>r);finite "
  using FiniteChopEqv RTrue by blast
 have 5: "\<turnstile> finite \<longrightarrow>(#True;f\<^sup>r);finite = #True;(f\<^sup>r;finite)" 
  using ChopAssoc by fastforce
 have 6: "\<turnstile> finite \<longrightarrow>(finite;(f;#True))\<^sup>r = #True;(f\<^sup>r;finite)" using "1" "2" "3" "4" "5" by fastforce
 have 7: "\<turnstile> finite \<longrightarrow> #True;(f\<^sup>r;finite) = finite;(f\<^sup>r;#True) "
 by (meson FiniteChopEqv FiniteRightChopEqvChop FiniteTransEqv Prop11 int_simps(8) int_simps(9)) 
 from 6 7 show ?thesis
   unfolding da_d_def 
   using FiniteTransEqv by blast
qed

lemma WRDaEqvDa:
 "\<turnstile> finite \<longrightarrow> wrev (da f) = da(wrev f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> wrev (finite;(f;#True)) = wrev (f;#True); wrev finite"
 by (simp add: WRevChop) 
 have 2: "\<turnstile> finite \<longrightarrow> wrev (f;#True); wrev finite =  wrev (f;#True); finite"
 by (metis FiniteRightChopEqvChop InfEqvNotFinite all_rev_eq(10) finite_d_def inteq_reflection
      wfreverse_d_def wrev_fun1)
 have 3: "\<turnstile> finite \<longrightarrow> wrev (f;#True) ; finite = (wrev #True;wrev f);finite "
   by (simp add: FiniteChopEqv WRevChop)
 have 4: "\<turnstile> finite \<longrightarrow> (wrev #True;wrev f);finite = (#True;wrev f);finite "
  using FiniteChopEqv WRTrue by blast
 have 5: "\<turnstile> finite \<longrightarrow> (#True;wrev f);finite = #True;(wrev f;finite)" 
  using ChopAssoc by fastforce
 have 6: "\<turnstile> finite \<longrightarrow> wrev (finite;(f;#True)) = #True;(wrev f;finite)" 
  using "1" "2" "3" "4" "5" by fastforce
 have 7: "\<turnstile> finite \<longrightarrow> #True;(wrev f;finite) = finite;(wrev f;#True) "
 by (meson FiniteChopEqv FiniteRightChopEqvChop FiniteTransEqv Prop11 int_simps(8) int_simps(9)) 
 from 6 7 show ?thesis
 by (metis FiniteTransEqv da_d_def)
qed 

lemma RDaEqvDaAlt:
 "\<turnstile>  (da f)\<^sup>r = (finite \<and> da (f\<^sup>r))"
 by (metis (no_types, lifting) EqvReverseReverseAlt FiniteImpAnd RDaEqvDa RRAndAlt WRevAndFiniteAlt 
     inteq_reflection lift_and_com)

lemma WRDaEqvDaAlt:
 "\<turnstile>  wrev (da f) = (finite \<longrightarrow> da (wrev f))"
 by (metis FiniteImp FiniteImpAnd Prop10 WRDaEqvDa WRevAndFiniteAlt WRevRev int_iffD1 inteq_reflection)

lemma RRDaEqvDa:
 "\<turnstile> finite \<longrightarrow> (da (f\<^sup>r))\<^sup>r = da(f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (da (f\<^sup>r))\<^sup>r = da ((f\<^sup>r)\<^sup>r)" using RDaEqvDa by blast
 have 2: "\<turnstile> finite \<longrightarrow> da ((f\<^sup>r)\<^sup>r) = da f"
  by (simp add: EqvReverseReverse FiniteChopEqv FiniteRightChopEqvChop da_d_def)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRDaEqvDa:
 "\<turnstile> finite \<longrightarrow> wrev (da (wrev f)) = da(f)"
 by (metis EqvWReverseWReverse FiniteChopEqv FiniteRightChopEqvChop FiniteTransEqv WRDaEqvDa da_d_def)


lemma RRDaEqvDaAlt:
 "\<turnstile>  (da (f\<^sup>r))\<^sup>r = (finite \<and> da f)"
 by (metis FiniteImpAnd RDaEqvDa RDaEqvDaAlt RRDaEqvDa inteq_reflection lift_and_com)

lemma WRWRDaEqvDaAlt:
 "\<turnstile>  wrev (da (wrev f)) = (finite \<longrightarrow> da f)"
 by (metis EqvWReverseWReverseAlt WRDaEqvDaAlt WRevFiniteImp inteq_reflection)

lemma RBaEqvBa:
 "\<turnstile> finite \<longrightarrow> (ba f)\<^sup>r = ba(f\<^sup>r)"
unfolding ba_d_def 
by (metis FiniteChopEqv FiniteNotEqv FiniteRightChopEqvChop FiniteTransEqv RDaEqvDa RNot itl_def(12))

lemma WRBaEqvBa:
 "\<turnstile> finite \<longrightarrow> wrev (ba f) = ba(wrev f)"
unfolding ba_d_def wfreverse_d_def
by (simp add: FiniteNotEqv RDaEqvDa)  

lemma RBaEqvBaAlt:
 "\<turnstile>  (ba f)\<^sup>r = (ba(f\<^sup>r) \<and> finite)"
 by (metis FiniteImpAnd RAnd RBaEqvBa RTrueAlt int_simps(17) inteq_reflection lift_and_com)

lemma WRBaEqvBaAlt:
 "\<turnstile>  wrev (ba f) = (finite \<longrightarrow> ba(wrev f))"
 by (metis (no_types, opaque_lifting) EqvWReverseWReverseAlt FiniteImpAnd RDaEqvDa WRevAndFiniteAlt 
     WRevRev all_wrev_eq(9) ba_d_def int_simps(4) inteq_reflection wfreverse_d_def)

lemma RRBaEqvBa:
 "\<turnstile> finite \<longrightarrow> (ba (f\<^sup>r))\<^sup>r = ba(f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (ba (f\<^sup>r))\<^sup>r = ba ((f\<^sup>r)\<^sup>r)" using RBaEqvBa by blast
 have 2: "\<turnstile> finite \<longrightarrow> ba ((f\<^sup>r)\<^sup>r) = ba f"
 by (simp add: EqvReverseReverse FiniteChopEqv FiniteNotEqv FiniteRightChopEqvChop ba_d_def da_d_def)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRBaEqvBa:
 "\<turnstile> finite \<longrightarrow> wrev (ba (wrev f)) = ba(f)"
 by (metis (no_types, opaque_lifting) EqvWReverseWReverse FiniteChopEqv FiniteNotEqv 
     FiniteRightChopEqvChop FiniteTransEqv WRBaEqvBa ba_d_def da_d_def)

lemma RRBaEqvBaAlt:
 "\<turnstile>  (ba (f\<^sup>r))\<^sup>r = (ba f \<and> finite)"
 by (meson FiniteImpAnd Prop04 Prop11 RBaEqvBa RBaEqvBaAlt RRBaEqvBa)

lemma WRWRBaEqvBaAlt:
 "\<turnstile>  wrev (ba (wrev f)) = (finite \<longrightarrow> ba f )"
 by (metis (no_types, opaque_lifting) EqvReverseReverseAlt EqvWReverseWReverseAlt FiniteImpAnd 
     WRWRBaEqvBa all_wrev_eq(9) int_simps(4) inteq_reflection wfreverse_d_def)

lemma RInitEqvFin:
 "\<turnstile> finite \<longrightarrow> (init f)\<^sup>r = fin(f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (init f)\<^sup>r = ((f \<and> empty);#True)\<^sup>r"
   by (metis init_d_def int_simps(1) int_simps(12) inteq_reflection lift_and_com) 
 have 2: "\<turnstile> finite \<longrightarrow> ((f \<and> empty);#True)\<^sup>r = (#True;(f \<and> empty)\<^sup>r)" 
   by (meson FiniteChopEqv FiniteTransEqv RTrue RevChop)    
 have 3: "\<turnstile> #True;(f \<and> empty)\<^sup>r = #True;(f\<^sup>r \<and> empty)" 
      by (metis RAnd REmptyEqvEmpty RightChopEqvChop int_eq)
 have 4: "\<turnstile> finite \<longrightarrow> #True;(f\<^sup>r \<and> empty) = #True;(f \<and> empty)" 
     using RAndEmptyEqvAndEmpty 
     by (metis "3" FiniteRightChopEqvChop REmptyEqvEmpty inteq_reflection rev_fun2)
 have 5: "\<turnstile> #True;(f \<and> empty) = fin(f)"  
      using FinEqvTrueChopAndEmpty by fastforce
from 1 2 3 4 5 show ?thesis by fastforce
qed

lemma WRInitEqvFin:
 "\<turnstile> finite \<longrightarrow> wrev (init f) = fin(f)"
 by (metis EqvWReverseWReverse EqvWReverseWReverseAlt FiniteTransEqv RInitEqvFin WRevRev inteq_reflection)

lemma RInitEqvFinAlt:
 "\<turnstile>  (init f)\<^sup>r = (sfin(f))"
proof -
 have 1: "\<turnstile>  (init f)\<^sup>r = ((f \<and> empty);#True)\<^sup>r"
   by (metis init_d_def int_simps(1) inteq_reflection lift_and_com) 
 have 2: "\<turnstile>  ((f \<and> empty);#True)\<^sup>r = (finite;(f \<and> empty)\<^sup>r)"
 by (metis RTrueAlt RevChopAlt inteq_reflection) 
 have 3: "\<turnstile> finite;(f \<and> empty)\<^sup>r = finite;(f\<^sup>r \<and> empty)" 
      by (metis RAnd REmptyEqvEmpty RightChopEqvChop int_eq)
 have 4: "\<turnstile>  finite;(f\<^sup>r \<and> empty) = finite;(f \<and> empty)"
 by (metis "3" RAndEmptyEqvAndEmpty inteq_reflection) 
 have 5: "\<turnstile> finite;(f \<and> empty) = sfin(f)"
 by (metis DiamondSChopdef FiniteChopEqvDiamond SFinEqvTrueSChopAndEmpty inteq_reflection)  
from 1 2 3 4 5 show ?thesis by fastforce
qed

lemma WRInitEqvFinAlt:
 "\<turnstile>  wrev (init f) = (finite \<longrightarrow> fin(f))"
 by (metis EqvWReverseWReverseAlt FiniteImpAnd RInitEqvFin WRevRev all_wrev_eq(9) inteq_reflection)

lemma RFinEqvInit:
 "\<turnstile> finite \<longrightarrow> (fin f)\<^sup>r = init (f)"
proof -
 have 1: "\<turnstile> fin f = #True;(f \<and> empty)" 
      using FinEqvTrueChopAndEmpty by auto
 have 2: "\<turnstile> (fin f)\<^sup>r = (#True;(f \<and> empty))\<^sup>r" 
      using "1" REqvRule by blast
 have 3: "\<turnstile> finite \<longrightarrow> (#True;(f \<and> empty))\<^sup>r = (f \<and> empty)\<^sup>r;#True" 
   by (meson FiniteRightChopEqvChop FiniteTransEqv RTrue RevChop)
 have 4: "\<turnstile> (f\<and> empty)\<^sup>r;#True = (f\<^sup>r \<and> empty);#True" 
      using LeftChopEqvChop RAnd REmptyEqvEmpty  by (metis int_eq)
 have 5: "\<turnstile> (f\<and> empty)\<^sup>r;#True = (f\<and> empty);#True"
      by (simp add: RAndEmptyEqvAndEmpty LeftChopEqvChop)
 have 6: "\<turnstile> (f \<and> empty);#True = init(f)" 
      by (simp add: AndChopCommute init_d_def)
 from 1 2 3 4 5 6 show ?thesis by fastforce
qed

lemma WRFinEqvInit:
 "\<turnstile> finite \<longrightarrow> wrev (fin f) = init (f)"
 by (metis EqvWReverseWReverse EqvWReverseWReverseAlt FiniteTransEqv RFinEqvInit WRevRev inteq_reflection)

lemma RFinEqvInitAlt:
 "\<turnstile>  (fin f)\<^sup>r =  (finite \<and> init f )"
proof -
 have 1: "\<turnstile> fin f = #True;(f \<and> empty)" 
      using FinEqvTrueChopAndEmpty by auto
 have 2: "\<turnstile> (fin f)\<^sup>r = (#True;(f \<and> empty))\<^sup>r" 
      using "1" REqvRule by blast
 have 3: "\<turnstile>  (#True;(f \<and> empty))\<^sup>r = (f \<and> empty)\<^sup>r;finite"
 by (metis RTrueAlt RevChopAlt inteq_reflection) 
 have 4: "\<turnstile> (f\<and> empty)\<^sup>r;finite = (f\<^sup>r \<and> empty);finite" 
      using LeftChopEqvChop RAnd REmptyEqvEmpty  by (metis int_eq)
 have 5: "\<turnstile> (f\<and> empty)\<^sup>r;finite = (f\<and> empty);finite"
      by (simp add: RAndEmptyEqvAndEmpty LeftChopEqvChop)
 have 6: "\<turnstile> (f\<and> empty);finite = (finite \<and> init f ) "
 by (meson InitAndEmptyEqvAndEmpty LeftChopEqvChop Prop04 StateAndEmptyChop lift_and_com)
   
 from 1 2 3 4 5 6 show ?thesis by fastforce
qed

lemma WRFinEqvInitAlt:
 "\<turnstile>  wrev (fin f) =  (finite \<longrightarrow> init f)"
 by (metis EqvWReverseWReverseAlt FiniteImpAnd WRInitEqvFin all_wrev_eq(9) inteq_reflection)

lemma RHaltEqvInitonly:
 "\<turnstile> finite \<longrightarrow> (halt f)\<^sup>r = initonly (f\<^sup>r) " 
proof -
 have 1: "\<turnstile> (halt f)\<^sup>r = (\<box> ( empty = f ))\<^sup>r" by (simp add: halt_d_def)
 have 2: "\<turnstile> finite \<longrightarrow> (\<box> ( empty = f ))\<^sup>r = bi ( (empty = f)\<^sup>r)" by (simp add: RBoxEqvBi)
 have 3: "\<turnstile> finite \<longrightarrow> (empty = f)\<^sup>r = (empty = f\<^sup>r)" by (metis REmptyEqvEmpty inteq_reflection rev_fun2)
 hence 4: "\<turnstile> finite \<longrightarrow> bi ( (empty = f)\<^sup>r) = bi(empty = f\<^sup>r)" 
   by (metis (no_types, opaque_lifting) FiniteChopEqv FiniteNotEqv bi_d_def di_d_def)
 have 5: "\<turnstile> bi(empty = f\<^sup>r) = initonly(f\<^sup>r)" by (simp add: initonly_d_def)
 from 1 2 4 5 show ?thesis by fastforce
qed

lemma WRHaltEqvInitonly:
 "\<turnstile> finite \<longrightarrow> wrev (halt f) = initonly (wrev f) "
 by (metis (no_types, opaque_lifting) EqvReverseReverse FiniteImpAnd FiniteInitonlyEqv 
    RHaltEqvInitonly WRevAndFiniteAlt WReverseEqvAlt all_wrev_eq(2) all_wrev_eq(9) int_eq) 

lemma RHaltEqvInitonlyAlt:
 "\<turnstile>  (halt f)\<^sup>r = (initonly (f\<^sup>r) \<and> finite) " 
proof -
 have 1: "\<turnstile> (halt f)\<^sup>r = (\<box> ( empty = f ))\<^sup>r" by (simp add: halt_d_def)
 have 2: "\<turnstile>  (\<box> ( empty = f ))\<^sup>r = (bi ( (empty = f)\<^sup>r) \<and> finite)" by (simp add: RBoxEqvBiAlt)
 have 3: "\<turnstile>  (empty = f)\<^sup>r = ((empty = f\<^sup>r) \<and> finite)"
  using REmptyEqvEmpty FiniteImpAnd[of "LIFT (empty = f)\<^sup>r "]
  by (metis (no_types, opaque_lifting) RAnd RTrueAlt int_simps(17) inteq_reflection lift_and_com rev_fun2) 
 hence 4: "\<turnstile>  (bi ( (empty = f)\<^sup>r) \<and> finite) = (bi( (empty = f\<^sup>r) \<and> finite) \<and> finite)"
 by (metis "2" inteq_reflection) 
 have 39:  "\<turnstile> bi finite = finite "
 by (metis AndInfChopEqvAndInf InfEqvNotFinite bi_d_def di_d_def finite_d_def int_simps(17) inteq_reflection) 
 have 40: "\<turnstile> (bi( (empty = f\<^sup>r) \<and> finite) \<and> finite) = (bi( (empty = f\<^sup>r)) \<and> finite) "
 by (metis BiAndEqv BiElim 39 Prop10 Prop12 inteq_reflection)
 have 5: "\<turnstile> (bi(empty = f\<^sup>r) \<and> finite) = (initonly(f\<^sup>r) \<and> finite)" by (simp add: initonly_d_def)
 from 1 2 4 5 show ?thesis by (metis "40" inteq_reflection) 
qed

lemma WRHaltEqvInitonlyAlt:
 "\<turnstile>  wrev (halt f) = (finite \<longrightarrow> initonly (wrev f)) "
 using WRHaltEqvInitonly WRevRev by fastforce 



lemma RInitonlyEqvHalt:
 "\<turnstile> finite \<longrightarrow> (initonly f)\<^sup>r = halt(f\<^sup>r)" 
proof -
 have 1: "\<turnstile> (initonly f)\<^sup>r = (bi (empty = f))\<^sup>r" by (simp add: initonly_d_def)
 have 2: "\<turnstile> finite \<longrightarrow> (bi (empty = f))\<^sup>r = \<box>((empty = f)\<^sup>r)" by (simp add: RBiEqvBox)
 have 3: "\<turnstile> finite \<longrightarrow> (empty = f)\<^sup>r = (empty = f\<^sup>r)" by (metis REmptyEqvEmpty inteq_reflection rev_fun2)
 hence 4: "\<turnstile> finite \<longrightarrow>\<box> ((empty = f)\<^sup>r) = \<box>(empty = f\<^sup>r)"
  by (metis (no_types, opaque_lifting) FiniteNotEqv FiniteRightChopEqvChop always_d_def sometimes_d_def)
 have 5: "\<turnstile> \<box>(empty = f\<^sup>r) = halt(f\<^sup>r)" by (simp add: halt_d_def)
 from 1 2 4 5 show ?thesis by fastforce
qed

lemma WRInitonlyEqvHalt:
 "\<turnstile> finite \<longrightarrow> wrev (initonly f) = halt(wrev f)"
 by (metis (no_types, lifting) RInitonlyEqvHalt WRHaltEqvInitonlyAlt WRevAndFiniteAlt WReverseEqvAlt 
     all_wrev_eq(2) all_wrev_eq(9) inteq_reflection) 



lemma RInitonlyEqvHaltAlt:
 "\<turnstile>  (initonly f)\<^sup>r = (halt(f\<^sup>r) \<and> finite)"
by (metis (no_types, lifting) EqvReverseReverseAlt FiniteImpAnd RInitonlyEqvHalt RRAndAlt 
   WRevAndFiniteAlt all_rev_eq(10) inteq_reflection)


lemma WRInitonlyEqvHaltAlt:
 "\<turnstile>  wrev (initonly f) = (finite \<longrightarrow> halt(wrev f))"
 by (metis (no_types, lifting) EqvWReverseWReverseAlt RInitonlyEqvHaltAlt WRHaltEqvInitonlyAlt 
      WRevAndFiniteAlt WRevRev all_wrev_eq(9) inteq_reflection)


lemma RRHaltEqvInitonly:
 "\<turnstile> finite \<longrightarrow> (halt (f\<^sup>r))\<^sup>r = initonly (f) "
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (halt (f\<^sup>r))\<^sup>r = initonly ((f\<^sup>r)\<^sup>r)" using RHaltEqvInitonly by blast
 have 2: "\<turnstile> finite \<longrightarrow> initonly ((f\<^sup>r)\<^sup>r) = initonly(f) "
  by (simp add: EqvReverseReverse FiniteInitonlyEqv)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRHaltEqvInitonly:
 "\<turnstile> finite \<longrightarrow> wrev (halt (wrev f)) = initonly (f) "
 by (meson EqvWReverseWReverse FiniteInitonlyEqv FiniteTransEqv WRHaltEqvInitonly)

lemma RRHaltEqvInitonlyAlt:
 "\<turnstile>  (halt (f\<^sup>r))\<^sup>r = (initonly (f) \<and> finite) "
 by (metis EqvReverseReverse FiniteImpAnd FiniteInitonlyEqv RHaltEqvInitonlyAlt inteq_reflection)

lemma WRWRHaltEqvInitonlyAlt:
 "\<turnstile>  wrev (halt (wrev f)) = (finite \<longrightarrow> initonly (f) ) "
 by (metis (no_types, lifting) EqvWReverseWReverseAlt WRInitonlyEqvHaltAlt WRevAndFiniteAlt WRevRev
     all_wrev_eq(9) inteq_reflection)

lemma RRInitonlyEqvHalt :
 "\<turnstile> finite \<longrightarrow> (initonly (f\<^sup>r))\<^sup>r = halt(f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (initonly (f\<^sup>r))\<^sup>r = halt((f\<^sup>r)\<^sup>r)" using RInitonlyEqvHalt by blast
 have 2: "\<turnstile> finite \<longrightarrow> halt((f\<^sup>r)\<^sup>r) = halt(f)"
  by (simp add: EqvReverseReverse FiniteHaltEqv)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRInitonlyEqvHalt :
 "\<turnstile> finite \<longrightarrow> wrev (initonly (wrev f)) = halt(f)"
 by (meson EqvWReverseWReverse FiniteHaltEqv FiniteTransEqv WRInitonlyEqvHalt)

lemma RRInitonlyEqvHaltAlt :
 "\<turnstile>  (initonly (f\<^sup>r))\<^sup>r = (halt(f) \<and> finite)"
 by (metis EqvReverseReverse FiniteHaltEqv FiniteImpAnd RInitonlyEqvHaltAlt inteq_reflection)

lemma WRWRInitonlyEqvHaltAlt :
 "\<turnstile>  wrev (initonly (wrev f)) = (finite \<longrightarrow> halt(f) )"
 by (metis EqvWReverseWReverseAlt WRHaltEqvInitonlyAlt WRInitonlyEqvHaltAlt WRWRHaltEqvInitonlyAlt 
    inteq_reflection)


lemma RKeepEqvKeep :
 "\<turnstile> finite \<longrightarrow> (keep f)\<^sup>r = keep(f\<^sup>r)"
proof -
 have 1: "\<turnstile> (keep f)\<^sup>r = (ba(skip \<longrightarrow> f))\<^sup>r" by (simp add: keep_d_def)
 have 2: "\<turnstile> finite \<longrightarrow> (ba(skip \<longrightarrow> f))\<^sup>r = ba((skip \<longrightarrow> f)\<^sup>r)" by (simp add:RBaEqvBa)
 have 3: "\<turnstile> finite \<longrightarrow> (skip \<longrightarrow> f)\<^sup>r = (skip \<longrightarrow> f\<^sup>r)"
 by (metis all_rev_eq(8) rev_fun2) 
 hence 4: "\<turnstile> finite \<longrightarrow> ba((skip \<longrightarrow> f)\<^sup>r) = ba(skip \<longrightarrow> f\<^sup>r)"
   by (simp add: FiniteChopEqv FiniteNotEqv FiniteRightChopEqvChop ba_d_def da_d_def)
 have 5: "\<turnstile> ba(skip \<longrightarrow> f\<^sup>r) = keep(f\<^sup>r)" by (simp add: keep_d_def)
 from 1 2 4 5 show ?thesis by fastforce
qed

lemma WRKeepEqvKeep :
 "\<turnstile> finite \<longrightarrow> wrev (keep f) = keep(wrev f)"
 by (metis (no_types, opaque_lifting) EqvReverseReverseAlt FiniteImpAnd RKeepEqvKeep WRevAndFiniteAlt 
      WReverseEqvAlt all_wrev_eq(2) all_wrev_eq(9) inteq_reflection)


lemma RKeepEqvKeepAlt :
 "\<turnstile>  (keep f)\<^sup>r = (keep(f\<^sup>r) \<and> finite)"
using FiniteImpAnd[of "LIFT (keep f)\<^sup>r"] RKeepEqvKeep
by (metis (no_types, opaque_lifting) Prop11 Prop12 RImpRule RTrueAlt int_simps(12) int_simps(13)
   inteq_reflection)

lemma WRKeepEqvKeepAlt :
 "\<turnstile>  wrev (keep f) = (finite \<longrightarrow> keep(wrev f))"
 using WRKeepEqvKeep WRevRev by fastforce

lemma RRKeepEqvKeep :
 "\<turnstile> finite \<longrightarrow> (keep (f\<^sup>r))\<^sup>r = keep(f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (keep (f\<^sup>r))\<^sup>r = keep( (f\<^sup>r)\<^sup>r )" using RKeepEqvKeep by blast
 have 2: "\<turnstile> finite \<longrightarrow> keep( (f\<^sup>r)\<^sup>r ) = keep(f)" 
   by (simp add: EqvReverseReverse FiniteKeepEqv)
 from 1 2 show ?thesis by fastforce
qed

lemma WRWRKeepEqvKeep :
 "\<turnstile> finite \<longrightarrow> wrev (keep (wrev f)) = keep(f)"
 by (meson EqvWReverseWReverse FiniteKeepEqv FiniteTransEqv WRKeepEqvKeep)



lemma RRKeepEqvKeepAlt :
 "\<turnstile>  (keep (f\<^sup>r))\<^sup>r = (keep(f) \<and> finite)"
 by (metis EqvReverseReverse FiniteImpAnd FiniteKeepEqv RKeepEqvKeepAlt inteq_reflection)

lemma WRWRKeepEqvKeepAlt :
 "\<turnstile>  wrev (keep (wrev f)) = (finite \<longrightarrow> keep(f) )"
 by (metis (no_types, lifting) EqvWReverseWReverseAlt WRKeepEqvKeepAlt WRevAndFiniteAlt WRevRev 
all_wrev_eq(9) inteq_reflection)


lemma RassignEqvTAssign:
 "\<turnstile> finite \<longrightarrow> ($v = e)\<^sup>R = (v \<leftarrow> e\<^sup>R)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> ($v = e)\<^sup>R = (($v)\<^sup>R = e\<^sup>R)"
    using rev_fun2_e by blast
 have 20: "\<turnstile> finite \<longrightarrow> ($v)\<^sup>R = (!v) " 
   by (simp add: rev_current)
 have 2: "\<turnstile> finite \<longrightarrow> (($v)\<^sup>R = e\<^sup>R) = ((!v) = e\<^sup>R)"
   using 20 by (auto simp add: Valid_def itl_defs) 
 have 3: "\<turnstile> finite \<longrightarrow> ((!v) = e\<^sup>R) = (v \<leftarrow> e\<^sup>R)" by (simp add: intI temporal_assign_d_def)
 from 1 2 3 show ?thesis by fastforce
qed

lemma RassignEqvTAssignAlt:
 "\<turnstile>  (($v = e)\<^sup>R \<and> finite) = ((v \<leftarrow> e\<^sup>R) \<and> finite)"
 by (simp add: FiniteImpAnd RassignEqvTAssign)


lemma RTAssignEqvAssign:
 "\<turnstile> finite \<longrightarrow> (v \<leftarrow> e)\<^sup>r = ($v = e\<^sup>R)" 
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (v \<leftarrow> e)\<^sup>r = (finite \<longrightarrow> !v = e)\<^sup>r"
    unfolding temporal_assign_d_def 
    by (metis int_simps(1) int_simps(12) inteq_reflection)
 have 2: "\<turnstile> finite \<longrightarrow> (finite \<longrightarrow> !v = e)\<^sup>r = ($v = e\<^sup>R)"
  by (auto simp add: Valid_def itl_defs freverse_d_def ereverse_d_def nfinite_nrev nlast_nrev) 
 from 1 2 show ?thesis by fastforce
qed

lemma WRTAssignEqvAssign:
 "\<turnstile> finite \<longrightarrow> wrev (v \<leftarrow> e) = ($v = e\<^sup>R)"
 by (metis (no_types, opaque_lifting) FiniteNotEqv FiniteTransEqv RTAssignEqvAssign int_simps(4) 
     inteq_reflection rev_fun1 wfreverse_d_def) 

lemma RTAssignEqvAssignAlt:
 "\<turnstile>  (v \<leftarrow> e)\<^sup>r = (($v = e\<^sup>R) \<and> finite)"
 by (metis FiniteImpAnd RAnd RTAssignEqvAssign RTrueAlt int_simps(17) inteq_reflection lift_and_com) 

lemma WRTAssignEqvAssignAlt:
 "\<turnstile>  wrev (v \<leftarrow> e) = (finite \<longrightarrow> ($v = e\<^sup>R) )"
 using RTAssignEqvAssignAlt WRevRev by fastforce

lemma RNextAssignEqvPrevAssign:
 "\<turnstile> finite \<longrightarrow> (v := e)\<^sup>r = (v =: e\<^sup>R)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (v := e)\<^sup>r = (v$ = e)\<^sup>r" by (simp add: REqvRule intI next_assign_d_def)
 have 20: "\<turnstile> finite \<longrightarrow> (v$ = e)\<^sup>r = ((v$)\<^sup>R = e\<^sup>R) "
 by (simp add: ereverse_d_def freverse_d_def intI itl_defs(5))
 have 21: "\<turnstile> finite \<longrightarrow> ((v$)\<^sup>R = e\<^sup>R) = (v! = e\<^sup>R) "
   by ( simp add: Valid_def itl_defs freverse_d_def ereverse_d_def)
    (metis One_nat_def Suc_ile_eq epred_conv_minus gr_zeroI idiff_enat_enat nfinite_nlength_enat 
     nnth_nrev one_enat_def the_enat.simps zero_enat_def)
 have 2: "\<turnstile> finite \<longrightarrow> (v$ = e)\<^sup>r = (v! = e\<^sup>R)"
 using "20" "21" FiniteTransEqv by blast
 have 3: "\<turnstile> finite \<longrightarrow> (v! = e\<^sup>R) = (v =: e\<^sup>R)"
  by (auto simp add: Valid_def itl_defs freverse_d_def ereverse_d_def)
 from 1 2 3 show ?thesis by fastforce
qed

lemma WRNextAssignEqvPrevAssign:
 "\<turnstile> finite \<longrightarrow> wrev (v := e) = (v =: e\<^sup>R)"
 by (metis EqvWReverseWReverse EqvWReverseWReverseAlt FiniteTransEqv RNextAssignEqvPrevAssign WRevRev 
   inteq_reflection)

lemma RNextAssignEqvPrevAssignAlt:
 "\<turnstile>  (v := e)\<^sup>r = ((v =: e\<^sup>R) \<and> finite)"
using FiniteImpAnd[of "LIFT (v := e)\<^sup>r" ]  RNextAssignEqvPrevAssign[of v e]
by (metis (no_types, opaque_lifting) Prop11 Prop12 RImpRule RTrueAlt int_simps(12) int_simps(13)
    inteq_reflection) 

lemma WRNextAssignEqvPrevAssignAlt:
 "\<turnstile>  wrev (v := e) = (finite \<longrightarrow> (v =: e\<^sup>R))"
 by (metis EqvWReverseWReverseAlt RNextAssignEqvPrevAssignAlt WRevRev all_wrev_eq(9) inteq_reflection)

lemma RPrevAssignEqvNextAssign:
 "\<turnstile> finite \<longrightarrow> (v =: e)\<^sup>r = (v := e\<^sup>R)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (v =: e)\<^sup>r = (v! = e)\<^sup>r"
     by (auto simp add: Valid_def itl_defs freverse_d_def ereverse_d_def nfinite_nrev)
 have 2: "\<turnstile> finite \<longrightarrow> (v! = e)\<^sup>r = (v$ = e\<^sup>R)"
  by ( simp add: Valid_def itl_defs freverse_d_def ereverse_d_def nfinite_nrev)
   (metis One_nat_def Suc_ile_eq epred_enat i0_less ndropn_nfirst nfinite_ndropn nfinite_nlength_enat 
    nlast_nrev nrev_ndropn ntaken_nlast the_enat.simps zero_enat_def)
 have 3: "\<turnstile> finite \<longrightarrow> (v$ = e\<^sup>R) = (v := e\<^sup>R)" by (simp add: next_assign_d_def)
 from 1 2 3 show ?thesis by fastforce
qed

lemma WRPrevAssignEqvNextAssign:
 "\<turnstile> finite \<longrightarrow> wrev (v =: e) = (v := e\<^sup>R)"
 by (metis (no_types, opaque_lifting) FiniteNotEqv FiniteTransEqv RPrevAssignEqvNextAssign int_simps(4) 
     inteq_reflection rev_fun1 wfreverse_d_def)

lemma RPrevAssignEqvNextAssignAlt:
 "\<turnstile>  (v =: e)\<^sup>r = ((v := e\<^sup>R) \<and> finite)"
using FiniteImpAnd[of "LIFT (v =: e)\<^sup>r"]  RPrevAssignEqvNextAssign
by (metis (no_types, opaque_lifting) Prop11 Prop12 RImpRule RTrueAlt int_simps(12) 
    int_simps(13) inteq_reflection) 

lemma WRPrevAssignEqvNextAssignAlt:
 "\<turnstile>  wrev (v =: e) = (finite \<longrightarrow> (v := e\<^sup>R) )"
 by (metis EqvWReverseWReverseAlt RPrevAssignEqvNextAssignAlt WRevRev all_wrev_eq(9) inteq_reflection)



lemma RGetsEqvBaSkipImp:
 "\<turnstile> finite \<longrightarrow> (v gets e)\<^sup>r = ba(skip \<longrightarrow> ($v = e\<^sup>R))"
proof -
 have 01: "\<turnstile> (skip \<longrightarrow> finite \<longrightarrow>  !v = e) =
              (skip  \<longrightarrow>  !v = e)  " 
      using nlength_eq_enat_nfiniteD by ( simp add: Valid_def itl_defs, blast)
 have 1: "\<turnstile> finite \<longrightarrow>(v gets e)\<^sup>r = (ba(skip \<longrightarrow> (!v = e)))\<^sup>r" 
   unfolding gets_d_def keep_d_def temporal_assign_d_def
   by (metis "01" EqvReverseReverse REmptyEqvEmpty int_simps(1) inteq_reflection)
 have 2: "\<turnstile> finite \<longrightarrow>(ba(skip \<longrightarrow> (!v = e)))\<^sup>r = ba ( (skip \<longrightarrow> (!v = e))\<^sup>r )" 
      by (simp add: RBaEqvBa)
 have 3: "\<turnstile> finite \<longrightarrow> (skip \<longrightarrow> (!v = e))\<^sup>r = (skip \<longrightarrow> ($v = e\<^sup>R))" 
       by (auto simp add: Valid_def itl_defs freverse_d_def ereverse_d_def nfinite_nrev nlast_nrev)
 hence 4: "\<turnstile> finite \<longrightarrow> ba ((skip \<longrightarrow> (!v = e))\<^sup>r) = ba (skip \<longrightarrow> ($v = e\<^sup>R))" 
    by (simp add: FiniteChopEqv FiniteNotEqv FiniteRightChopEqvChop ba_d_def da_d_def)
 from 1 2 4 show ?thesis by fastforce
qed

lemma WRGetsEqvBaSkipImp:
 "\<turnstile> finite \<longrightarrow> wrev (v gets e) = ba(skip \<longrightarrow> ($v = e\<^sup>R))"
 by (metis FiniteKeepEqv FiniteTransEqv WRKeepEqvKeep WRTAssignEqvAssign gets_d_def keep_d_def)

   
lemma RGetsEqvBaSkipImpAlt:
 "\<turnstile>  (v gets e)\<^sup>r = (ba(skip \<longrightarrow> ($v = e\<^sup>R)) \<and> finite)"
 by (metis FiniteImpAnd RAnd RGetsEqvBaSkipImp RTrueAlt int_simps(17) inteq_reflection lift_and_com)

lemma WRGetsEqvBaSkipImpAlt:
 "\<turnstile>  wrev (v gets e) = (finite \<longrightarrow> ba(skip \<longrightarrow> ($v = e\<^sup>R)) )"
 by (metis EqvWReverseWReverseAlt RGetsEqvBaSkipImpAlt WRevRev all_wrev_eq(9) inteq_reflection)
  
lemma RIfThenElse:
 "\<turnstile> finite \<longrightarrow>(if\<^sub>i f0 then f1 else f2)\<^sup>r = if\<^sub>i (f0\<^sup>r) then (f1\<^sup>r) else (f2\<^sup>r)"
unfolding ifthenelse_d_def
by (auto simp add: Valid_def itl_defs freverse_d_def ereverse_d_def)

lemma WRIfThenElse:
 "\<turnstile> finite \<longrightarrow> wrev (if\<^sub>i f0 then f1 else f2) = if\<^sub>i (wrev f0) then (wrev f1) else (wrev f2)"
unfolding ifthenelse_d_def
by (metis (no_types, lifting) FiniteAndEqv FiniteOrEqv WRAnd WRNot WROrAlt int_simps(1) 
    inteq_reflection wrev_fun2)

lemma RIfThenElseAlt:
 "\<turnstile> (if\<^sub>i f0 then f1 else f2)\<^sup>r = (if\<^sub>i (f0\<^sup>r) then (f1\<^sup>r) else (f2\<^sup>r) \<and> finite)"
using RIfThenElse[of f0 f1 f2] 
FiniteImpAnd[of "LIFT (if\<^sub>i f0 then f1 else f2)\<^sup>r"]
by (metis RAnd RTrueAlt int_simps(17) inteq_reflection lift_and_com)   

lemma WRIfThenElseAlt:
 "\<turnstile> wrev (if\<^sub>i f0 then f1 else f2) = (finite \<longrightarrow> if\<^sub>i (wrev f0) then (wrev f1) else (wrev f2) )"
 by (metis (no_types, lifting) FiniteImp FiniteImpAnd Prop12 WRIfThenElse WRevAndFiniteAlt
   WRevFinite WRevRev int_iffD1 int_iffI inteq_reflection)




lemma RWhile:
 "\<turnstile> (init f \<and> swhile f0 do f1)\<^sup>r = ( fin(f) \<and> (schopstar ((f0\<^sup>r) \<and> (f1\<^sup>r))) \<and> init (\<not>(f0)) )" 
proof -
 have 1: "\<turnstile> (init f \<and> swhile f0 do f1)\<^sup>r = ( init f \<and> (schopstar (f0 \<and> f1)) \<and> sfin (\<not>f0) )\<^sup>r" 
       by (simp add: swhile_d_def)
 have 2: "\<turnstile> ( init f \<and> (schopstar (f0 \<and> f1)) \<and> sfin (\<not>f0) )\<^sup>r = 
            ((init f)\<^sup>r \<and> (schopstar (f0 \<and> f1))\<^sup>r \<and> (sfin (\<not>f0))\<^sup>r)"
   by (metis RAnd inteq_reflection)
 have 3: "\<turnstile> finite \<longrightarrow> (init f)\<^sup>r = fin(f)" 
     by (simp add: RInitEqvFin)
 have 4: "\<turnstile> (schopstar (f0 \<and> f1))\<^sup>r = (schopstar ((f0\<^sup>r) \<and> (f1\<^sup>r)))"
    by (metis RAnd RevChopstar inteq_reflection) 
 have 40: "\<turnstile> finite \<longrightarrow> (sfin (\<not>f0)) = (fin (\<not>f0)) "
   by (metis (no_types, opaque_lifting) FinEqvTrueChopAndEmpty Finprop(4) int_simps(4) 
       inteq_reflection sfin_d_def)
 have 41: "\<turnstile> finite \<longrightarrow> (sfin (\<not>f0))\<^sup>r = (fin (\<not>f0))\<^sup>r "
 by (metis "40" Prop12 ReverseEqv all_rev_eq(2))
 have 5: "\<turnstile> finite \<longrightarrow>(sfin (\<not>f0))\<^sup>r = init (\<not>(f0))" 
   using "41" FiniteTransEqv RFinEqvInit by blast
 have 50: "\<turnstile> (schopstar (f0 \<and> f1))\<^sup>r \<longrightarrow> finite "
 using "4" SChopstar_finite by fastforce 
 have 51: "\<turnstile> (schopstar ((f0\<^sup>r) \<and> (f1\<^sup>r))) \<longrightarrow> finite " 
    using SChopstar_finite by blast
 have 6: "\<turnstile> ((init f)\<^sup>r \<and> (schopstar (f0 \<and> f1))\<^sup>r \<and> (sfin (\<not>f0))\<^sup>r) = 
            ( fin(f) \<and> (schopstar ((f0\<^sup>r) \<and> (f1\<^sup>r))) \<and> init (\<not>(f0)) ) "
    using 3 4 5 50 51 by fastforce
 from 1 2 6 show ?thesis by fastforce
qed

lemma WRWhile:
 "\<turnstile> wrev (init f \<and> swhile f0 do f1) = 
    (finite \<longrightarrow> fin(f) \<and> (schopstar ((wrev f0) \<and> (wrev f1))) \<and> init (\<not>(f0)) )"
 by (metis (no_types, opaque_lifting) EqvReverseReverseAlt RAnd RWhile SChopstar_Chopstar 
   WRevAndFiniteAlt WRevRev all_wrev_eq(9) int_eq)
 

lemma AAxRev:
 "\<turnstile> (\<forall>\<forall> x. F x)\<^sup>r = (\<forall>\<forall> x. (F x)\<^sup>r)"
proof -
 have 1: "\<turnstile> (\<forall>\<forall> x. F x) = (\<not>(\<exists>\<exists> x. \<not>(F x)))" using AAxDef by blast
 have 2: "\<turnstile> (\<forall>\<forall> x. F x)\<^sup>r = (\<not>(\<exists>\<exists> x. \<not>(F x)))\<^sup>r" using REqvRule "1" by blast
 have 3: "\<turnstile>  (\<not>(\<exists>\<exists> x. \<not>(F x)))\<^sup>r = (\<not>((\<exists>\<exists> x.(\<not> (F x)))\<^sup>r) \<and> finite)"
 using  FiniteImpAnd[of "LIFT (\<not>(\<exists>\<exists> x. \<not>(F x)))\<^sup>r"]
 by (simp add: rev_fun1_alt)
 have 4: "\<turnstile> ((\<exists>\<exists> x.(\<not> (F x)))\<^sup>r) = ((\<exists>\<exists> x.(\<not> (F x))\<^sup>r))" by (simp add: EExRev)
 hence 5: "\<turnstile> (\<not>((\<exists>\<exists> x.(\<not> (F x)))\<^sup>r) \<and> finite) = (\<not>(\<exists>\<exists> x.(\<not> (F x))\<^sup>r) \<and> finite) " by auto
 have 51: "\<And> x. \<turnstile>  ((\<not> (F x))\<^sup>r)= ((\<not>( (F x)\<^sup>r) \<and> finite))"
 using rev_fun1_alt by blast
 hence 52: "\<turnstile>  (\<exists>\<exists> x.((\<not> (F x))\<^sup>r)) = (\<exists>\<exists> x.(\<not>( (F x)\<^sup>r) \<and> finite))" 
    using EExEqvRule[of "(\<lambda>x. LIFT( (\<not> (F x)))\<^sup>r )" "(\<lambda>x. LIFT((\<not>( (F x)\<^sup>r)) \<and> finite)) " ] 
    by fastforce
 hence 6: "\<turnstile> (\<not>(\<exists>\<exists> x.(\<not> (F x))\<^sup>r) \<and> finite ) = (\<not>(\<exists>\<exists> x.(\<not>( (F x)\<^sup>r) \<and> finite)) \<and> finite)"  
    by auto  
 have 60: "\<turnstile> (\<not>(\<exists>\<exists> x.(\<not>( (F x)\<^sup>r) \<and> finite)) \<and> finite) = 
             (\<not>(\<exists>\<exists> x. (\<not> (( (F x)\<^sup>r) \<or> \<not>finite))) \<and> finite) "
     by (auto simp add: Valid_def exist_state_d_def)   
 have 7: "\<turnstile> (\<not>(\<exists>\<exists> x.(\<not>(( (F x)\<^sup>r) \<or> \<not>finite)) ) \<and> finite  ) = ((\<forall>\<forall> x. ((F x)\<^sup>r \<or> \<not>finite)) \<and> finite) " using AAxDef 
    by (simp add: forall_state_d_def)
 have 69: "\<turnstile> ((\<forall>\<forall> x. ((F x)\<^sup>r \<or> \<not>finite)) \<and> finite) = (\<forall>\<forall> x. (finite \<and> ((F x)\<^sup>r \<or> \<not>finite))) " 
    using AAxAndImport[of "LIFT finite"  "\<lambda>x. LIFT (F x)\<^sup>r \<or> \<not>finite"   ] by auto
 have 70: "\<And>x. \<turnstile>  (finite \<and> ((F x)\<^sup>r \<or> \<not>finite)) =  (F x)\<^sup>r " 
       by (auto simp add: freverse_d_def itl_defs(5))
 have 71: "\<turnstile> ((\<forall>\<forall> x. (finite \<and> ((F x)\<^sup>r \<or> \<not>finite)))) =  (\<forall>\<forall> x. (F x)\<^sup>r)" 
     by (simp add: "70" AAxEqvRule)
 from 1 2 3 5 6 7 70 show ?thesis 
 by (metis "60" "69" "71" inteq_reflection)
qed

lemma AAxWRev:
 "\<turnstile> wrev (\<forall>\<forall> x. F x) = (\<forall>\<forall> x.  wrev (F x))"
proof -
 have 1: "\<turnstile> wrev (\<forall>\<forall> x. F x) = (finite \<longrightarrow> frev (\<forall>\<forall> x. F x))"
 by (simp add: WRevRev) 
 have 2: "\<turnstile> (finite \<longrightarrow> frev (\<forall>\<forall> x. F x)) = (finite \<longrightarrow>  (\<forall>\<forall> x. frev (F x)))" 
   using AAxRev[of F] by auto
 have 3: "\<turnstile> (finite \<longrightarrow>  (\<forall>\<forall> x. frev (F x))) = (\<forall>\<forall> x. \<not> finite \<or> frev (F x)) " 
  using AAxOrImport[of "LIFT \<not>finite" "\<lambda>x. LIFT frev (F x)"] by auto
 have 4: " \<And> x. \<turnstile> (\<not> finite \<or> frev (F x)) = (wrev (F x))"
 using WRevRev by fastforce 
 have 5: "\<turnstile> (\<forall>\<forall> x. \<not> finite \<or> frev (F x)) = (\<forall>\<forall> x.  wrev (F x))"
 by (meson "4" AAxEqvRule) 
 show ?thesis
 by (metis "1" "2" "3" "5" inteq_reflection)  
qed
 




end