(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)
section \<open>Weak chop operator\<close>

theory Theorems
 imports
   ITL
begin
(*
 sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                        isar_proofs=false,
                    provers="z3 vampire cvc4 e spass " ]
*)


text \<open>
We give the proofs of a list of Finite and Infinite ITL theorems. These proofs and theorems are from 
\cite{Ben:ImpReaPro} but adapted for infinite and finite intervals.
\<close>

subsection \<open>Propositional reasoning\<close>

text \<open>
This is a list of propositional logic theorems used in the proofs of the ITL theorems. 
\<close>

lemma IfThenElseImp:
 "\<turnstile> (if\<^sub>i  g  then  f  else f1) \<longrightarrow> (( g \<longrightarrow> f) \<and> (\<not>g \<longrightarrow> f1)) "
by (simp add: itl_def Valid_def) 

lemma Prop01:
 assumes "\<turnstile> f \<longrightarrow> \<not> g \<or> h"
 shows   "\<turnstile> g \<and> f \<longrightarrow> h"
using assms by auto

lemma Prop02:
 assumes "\<turnstile> f \<longrightarrow> g"
         "\<turnstile> f1 \<longrightarrow> g"
  shows  "\<turnstile> f \<or> f1  \<longrightarrow> g"
using assms by fastforce

lemma Prop03:
  assumes "\<turnstile> f = (g \<or> h) "
  shows   "\<turnstile> h \<longrightarrow> f"
using assms by auto  

lemma Prop04:
 assumes "\<turnstile> f = h"
         "\<turnstile> f = h1"
 shows   "\<turnstile> h1 = h"
using assms using int_eq by auto

lemma Prop05:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows "\<turnstile> f \<longrightarrow> h \<or> g "
using assms by auto

lemma Prop06:
 assumes "\<turnstile> f = (g \<or> h) "
         "\<turnstile> h = h1"
 shows "\<turnstile> f = (g \<or> h1) "
using assms by fastforce

lemma Prop07:
 assumes "\<turnstile> f \<longrightarrow> g \<or> h "
 shows   "\<turnstile> f \<and> \<not> g \<longrightarrow> h"
using assms by auto

lemma Prop08:
 assumes "\<turnstile> f \<longrightarrow> g \<or> h"
         "\<turnstile> h \<longrightarrow> h1"
 shows   "\<turnstile> f \<longrightarrow> g \<or> h1"
using assms by fastforce

lemma Prop09:
 assumes "\<turnstile> f \<and> g \<longrightarrow> h"
 shows   "\<turnstile> f \<longrightarrow> (g \<longrightarrow> h)"
using assms by auto

lemma Prop10:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows   "\<turnstile> f = (f \<and> g)"
using assms by auto 

lemma Prop11:
 "(\<turnstile> f = f1) = ( (\<turnstile> f \<longrightarrow> f1) \<and>  (\<turnstile> f1 \<longrightarrow> f) )"
by (auto simp: Valid_def)

lemma Prop12:
 "(\<turnstile> f  \<longrightarrow> ( f1 \<and> f2)) = ( (\<turnstile> f \<longrightarrow> f1) \<and> (\<turnstile> f \<longrightarrow> f2))"
by (auto simp: Valid_def)

lemma Prop13:
 assumes "\<turnstile> f \<longrightarrow> g \<or> h"
 shows   "\<turnstile> f \<and> \<not>h \<longrightarrow> g"
using assms by (auto simp: Valid_def)


subsection \<open>State formulas\<close>

text \<open>
The \<open>init\<close> operator denotes state formulas, i.e., ITL formula that only constrain
the first state of an interval. 
\<close>

lemma Initprop :
  "\<turnstile> ((init f) \<and> (init g)) = init(f \<and> g)"
  "\<turnstile> (\<not> (init f)) = init ( \<not> f )"
  "\<turnstile> ((init f) \<or> (init g)) = init (f \<or> g)" 
  "\<turnstile> init #True" 
by (auto simp: itl_defs)

lemma Finprop :
 "\<turnstile> ((#True;(f \<and> empty)) \<and> (#True;(g \<and> empty))) = (#True;((f \<and> g) \<and> empty)) "
 "\<turnstile> ((#True;(f \<and> empty)) \<or> (#True;(g \<and> empty))) = (#True;((f \<or> g) \<and> empty)) "
 "\<turnstile> (#True;((#True) \<and> empty))"
 "\<turnstile> finite \<longrightarrow> (\<not> (#True;(f \<and> empty))) = (#True;(\<not>f \<and> empty))"
 "\<turnstile> (\<not> (#True;(f \<and> empty))) = ( (#True;(\<not>f \<and> empty)) \<and> finite)"
using nfinite_nlength_enat 
by (auto simp: finalt_defs finite_defs zero_enat_def, 
    auto simp add: itl_defs nfinite_nlength_enat zero_enat_def, force)
  
subsection \<open>finite and inf properties\<close>

lemma EmptyImpFinite:
 "\<turnstile> empty \<longrightarrow> finite"
using nlength_eq_enat_nfiniteD by (auto simp add: itl_defs zero_enat_def)

lemma SkipChopFiniteImpFinite:
 "\<turnstile> skip;finite \<longrightarrow> finite"
using nfinite_ndropn nlength_eq_enat_nfiniteD
by (simp add: Valid_def itl_defs,force  )

lemma FiniteChopSkipImpFinite:
 "\<turnstile> finite;skip \<longrightarrow> finite"
using nlength_eq_enat_nfiniteD 
by (simp add: Valid_def itl_defs, force)

lemma FiniteChopSkipImpMore:
 "\<turnstile> finite;skip \<longrightarrow> more"
using nlength_eq_enat_nfiniteD one_enat_def
by (simp add: Valid_def itl_defs, force)

lemma FiniteAndMoreImpFiniteChopSkip:
 "\<turnstile> finite \<and> more \<longrightarrow> finite;skip"
by (auto simp add: Valid_def itl_defs zero_enat_def)
   (metis Suc_ile_eq diff_diff_cancel diff_le_self enat_ord_simps(1) idiff_enat_enat nfinite_nlength_enat)

lemma FiniteChopSkipEqvFiniteAndMore:
 "\<turnstile> finite;skip = (finite \<and> more)"
 by (simp add: FiniteAndMoreImpFiniteChopSkip FiniteChopSkipImpFinite FiniteChopSkipImpMore 
     Prop12 int_iffI)
  
lemma FiniteChopSkipEqvSkipChopFinite:
 "\<turnstile> finite;skip = skip;finite"
by (auto simp add: Valid_def itl_defs)
   (metis enat.distinct(1) enat_add_sub_same enat_le_plus_same(2) le_iff_add ,
    metis eSuc_enat enat.simps(3) enat_add_sub_same idiff_enat_0_right iless_Suc_eq le_zero_eq 
    less_eqE min.orderE  nlength_eq_enat_nfiniteD not_le one_eSuc plus_1_eSuc(2),
    metis add.commute enat.simps(3) enat_add_sub_same idiff_enat_enat le_iff_add 
    nfinite_nlength_enat)

lemma FiniteAndEmptyEqvEmpty:
 "\<turnstile> (finite \<and> empty) = empty"
by (auto simp add: Valid_def itl_defs nlength_eq_enat_nfiniteD 
    zero_enat_def)

lemma FiniteChopFiniteEqvFinite:
 "\<turnstile> finite;finite = finite"
by (auto simp add: Valid_def itl_defs) (metis  zero_enat_def zero_le) 

lemma InfChopInfEqvInf:
 "\<turnstile> inf;inf = inf"
by (simp add: Valid_def itl_defs)

lemma InfChopFiniteEqvInf:
 "\<turnstile> inf;finite = inf"
by (simp add: Valid_def itl_defs)

lemma FiniteChopInfEqvInf:
 "\<turnstile> finite;inf = inf"
by (auto simp add: Valid_def itl_defs) (metis  zero_enat_def zero_le)

lemma InfEqvNotFinite:
 "\<turnstile> inf = (\<not> finite)"
by (simp add: Valid_def itl_defs)

lemma FiniteEqvNotInf:
 "\<turnstile> finite = (\<not> inf)"
by (simp add: Valid_def itl_defs)

lemma ChopTrueAndFiniteEqvAndFiniteChopFinite:
 "\<turnstile> ((f;#True) \<and> finite) = (f \<and> finite);finite"
by (auto simp add: Valid_def itl_defs)

lemma TrueChopAndFiniteEqvAndFiniteChopFinite:
 "\<turnstile> ((#True;f) \<and> finite) = finite;(f \<and> finite)"
by (auto simp add: Valid_def itl_defs)

lemma FiniteChopMoreEqvMore:
 "\<turnstile> finite;more = more"
by (auto simp add: Valid_def itl_defs)
   (metis idiff_0_right zero_enat_def zero_le)

lemma ChopAndFiniteDist:
 "\<turnstile> ((f;g) \<and> finite) = (f \<and> finite);(g \<and> finite)"
by (auto simp add: Valid_def itl_defs)

lemma FiniteOrInfinite:
 "\<turnstile> finite \<or> inf"
by (simp add: Valid_def itl_defs)

lemma FiniteImpAnd:
 assumes "\<turnstile> finite \<longrightarrow> f = g"
 shows "\<turnstile> (f \<and> finite) = (g \<and> finite)"
using assms by (auto simp add: Valid_def itl_defs)

lemma FmoreEqvSkipChopFinite:
 "\<turnstile> fmore = skip;finite"
by (metis FiniteChopSkipEqvFiniteAndMore FiniteChopSkipEqvSkipChopFinite 
          fmore_d_def inteq_reflection lift_and_com)
 
lemma FiniteImp:
 "\<turnstile> (f \<and> finite \<longrightarrow> g) = (f \<and> finite \<longrightarrow> g \<and> finite)"
by (simp add: itl_defs Valid_def)

lemma ChopAndInf:
 "\<turnstile> ((f;g) \<and> inf) = (f;(g \<and> inf))"
by (auto simp add: Valid_def itl_defs)

lemma ChopAndInfB:
 "\<turnstile> ((f;g) \<and> inf) = ((f \<and> inf) \<or> (f \<and> finite);(g \<and> inf)  )"
by (auto simp add: Valid_def itl_defs)

lemma MoreAndInfEqvInf:
 "\<turnstile> (more \<and> inf) = inf"
by (metis ChopAndInf EmptyImpFinite FiniteChopMoreEqvMore InfEqvNotFinite Prop11 Prop12 empty_d_def
          finite_d_def int_simps(32) inteq_reflection)

lemma AndInfChopAndInfEqvAndInf:
 "\<turnstile> (f \<and> inf);(f \<and> inf) = (f \<and> inf)"
by (auto simp add: Valid_def itl_defs)

lemma AndInfChopEqvAndInf:
 "\<turnstile> (f \<and> inf);g = (f \<and> inf)"
by (auto simp add: Valid_def itl_defs)

lemma AndMoreAndInfEqvAndInf:
 "\<turnstile> ((f \<and> more) \<and> inf) = (f \<and> inf)"
by (auto simp add: Valid_def itl_defs nlength_eq_enat_nfiniteD zero_enat_def)
   (metis gr_zeroI nlength_eq_enat_nfiniteD zero_enat_def)

lemma AndMoreAndFiniteEqvAndFmore:
 "\<turnstile> ((f \<and> more) \<and> finite) = (f \<and> fmore)"
by (auto simp add: Valid_def itl_defs)

lemma NotFmoreAndEmpty:
 "\<turnstile> \<not> (empty \<and> fmore)"
by (auto simp add: fmore_d_def empty_d_def)

lemma NotFmoreAndInf:
 "\<turnstile> \<not> ((f \<and> inf) \<and> fmore)"
by (auto simp add: fmore_d_def finite_d_def infinite_d_def)

lemma FmoreChopAnd:
 "\<turnstile> (((f \<and> more);g) \<and> fmore) = ((f \<and> fmore);(g \<and> finite))"
by (auto simp add: Valid_def itl_defs)

lemma NotEmptyAndInf:
 "\<turnstile> \<not>(empty \<and> inf)"
by (auto simp add: Valid_def itl_defs nlength_eq_enat_nfiniteD zero_enat_def)

lemma OrFiniteInf:
 "\<turnstile>  f = (( f \<and> finite) \<or> ( f \<and> inf))"
by (auto simp add: Valid_def itl_defs)

lemma AndInfEqvChopFalse:
 "\<turnstile> (f \<and> inf) = f;#False"
by (auto simp add: Valid_def itl_defs)


lemma EmptyOrMoreSplit: 
 "\<turnstile> f = ((f \<and> empty) \<or> (f \<and> more)) " 
unfolding empty_d_def by auto


subsection \<open>Basic Theorems\<close>

lemma BiChopImpChop :
    " \<turnstile> bi (f \<longrightarrow> f1) \<longrightarrow> f;g \<longrightarrow> f1;g "
 proof -
 have  1: "\<turnstile> g \<longrightarrow> g" by auto
 hence 2: "\<turnstile> \<box> ( g \<longrightarrow> g) " by (rule BoxGen)
 have  3: "\<turnstile> bi ( f \<longrightarrow> f1) \<and> \<box>(g \<longrightarrow> g) \<longrightarrow> f;g \<longrightarrow> f1;g " by (rule BiBoxChopImpChop)
 from 2 3 show ?thesis by fastforce
qed 

lemma AndChopA:
    " \<turnstile> (f \<and> f1);g \<longrightarrow> f;g "
proof -
 have  1: "\<turnstile> f \<and> f1 \<longrightarrow> f" by auto
 hence 2: "\<turnstile> bi (f \<and> f1 \<longrightarrow> f)" by (rule BiGen)
 have  3: "\<turnstile> bi (f \<and> f1 \<longrightarrow> f) \<longrightarrow> (f  \<and> f1);g \<longrightarrow> f;g" by (rule BiChopImpChop)
 from 2 3 show ?thesis using MP by blast
qed

lemma AndChopB:
    "\<turnstile> (f \<and> f1);g \<longrightarrow> f1;g "
proof -
 have  1: "\<turnstile> f \<and> f1 \<longrightarrow> f1" by auto
 hence 2: "\<turnstile>  bi (f \<and> f1 \<longrightarrow> f1)" by (rule BiGen)
 have  3: "\<turnstile>  bi (f \<and> f1 \<longrightarrow> f1) \<longrightarrow> (f \<and> f1);g \<longrightarrow> f1;g" by (rule BiChopImpChop)
 from 2 3 show ?thesis using MP by blast
qed

lemma NextChop:
  "\<turnstile> (\<circle> f);g  = \<circle>(f;g)"
proof -
 have 1: "\<turnstile> skip;(f;g)  = (skip;f);g" by (rule ChopAssoc)
 show ?thesis by (metis "1" int_eq next_d_def)
qed

lemma BoxChopImpChop :
    " \<turnstile>  \<box> (g \<longrightarrow> g1) \<longrightarrow> f;g \<longrightarrow> f;g1 "
 proof -
 have  1: "\<turnstile> g \<longrightarrow> g" by auto
 hence 2: "\<turnstile> bi ( g \<longrightarrow> g) " by (rule BiGen)
 have  3: "\<turnstile> bi ( f \<longrightarrow> f) \<and> \<box>(g \<longrightarrow> g1) \<longrightarrow> f;g \<longrightarrow> f;g1 " by (rule BiBoxChopImpChop)
 from 2 3 show ?thesis by fastforce
qed 

lemma LeftChopImpChop:
 assumes "\<turnstile> f\<longrightarrow> f1" 
 shows   "\<turnstile> f;g \<longrightarrow> f1;g"
proof -
 have  1: "\<turnstile> f\<longrightarrow> f1" using assms by auto
 hence 2: "\<turnstile> bi (f \<longrightarrow> f1)" by (rule BiGen)
 have  3: "\<turnstile> bi (f \<longrightarrow> f1) \<longrightarrow> f;g \<longrightarrow> f1;g" by (rule BiChopImpChop)
 from 2 3 show ?thesis using MP by blast
qed

lemma RightChopImpChop:
 assumes "\<turnstile> g \<longrightarrow> g1" 
 shows   "\<turnstile> f;g \<longrightarrow> f;g1"
proof -
 have  1: "\<turnstile> g \<longrightarrow> g1" using assms by auto
 hence 2: "\<turnstile> \<box> (g \<longrightarrow> g1)" by (rule BoxGen)
 have  3: "\<turnstile> \<box> (g \<longrightarrow> g1) \<longrightarrow> f;g \<longrightarrow> f;g1" by (rule BoxChopImpChop)
 from 2 3 show ?thesis using MP by blast
qed
 
lemma RightChopEqvChop:
 assumes "\<turnstile> g = g1"
 shows   "\<turnstile> (f;g) = (f;g1)"
using assms  RightChopImpChop[of "g" "g1" "f"] RightChopImpChop[of "g1" "g" "f"]
by fastforce

lemma InfRightChopEqvChop:
assumes "\<turnstile> inf \<longrightarrow> g = g1" 
shows   "\<turnstile> inf \<longrightarrow> (f;g) = (f;g1)"
using assms 
by (auto simp add: Valid_def itl_defs)

lemma ChopOrEqv:
 " \<turnstile> f;(g \<or> g1) = (f;g \<or> f;g1) "
proof -
 have  1: "\<turnstile> g \<longrightarrow> g \<or> g1" by auto
 hence 2: "\<turnstile> f;g \<longrightarrow> f;(g \<or> g1)" by (rule RightChopImpChop)
 have  3: "\<turnstile> g1 \<longrightarrow> g \<or> g1" by auto 
 hence 4: "\<turnstile> f;g1 \<longrightarrow> f;(g \<or> g1)" by (rule RightChopImpChop)
 from 2 4 show ?thesis by (meson ChopOrImp Prop02 Prop11) 
qed

lemma OrChopEqv:
 " \<turnstile> (f \<or> f1);g = (f;g \<or> f1;g) "
proof -
 have  1: "\<turnstile> f \<longrightarrow> f \<or> f1" by auto
 hence 2: "\<turnstile> f;g \<longrightarrow> (f\<or> f1);g" by (rule LeftChopImpChop)
 have  3: "\<turnstile> f1 \<longrightarrow> f \<or> f1" by auto 
 hence 4: "\<turnstile> f1;g \<longrightarrow> (f\<or> f1);g" by (rule LeftChopImpChop)
 from 2 4 show ?thesis 
 by (meson OrChopImp int_iffI Prop02)
qed

lemma OrChopImpRule:
 assumes "\<turnstile> f \<longrightarrow> f1 \<or> f2"
 shows   "\<turnstile> f;g \<longrightarrow> (f1;g) \<or> (f2;g)"
proof -
 have  1: "\<turnstile> f \<longrightarrow> f1 \<or> f2" using assms by auto
 hence 2: "\<turnstile> f;g \<longrightarrow> (f1 \<or> f2);g" by (rule LeftChopImpChop)
 have  3: "\<turnstile> (f1 \<or> f2); g = (f1;g \<or> f2;g)" by (rule OrChopEqv) 
 from 2 3 show ?thesis by fastforce
qed

lemma  LeftChopEqvChop:
 assumes "\<turnstile> f = f1"
 shows   "\<turnstile> f;g = (f1;g)"
proof -
 have  1: "\<turnstile> f = f1" using assms by auto
 hence 2: "\<turnstile> f \<longrightarrow> f1" by auto
 hence 3: "\<turnstile> f;g \<longrightarrow> f1;g" by (rule LeftChopImpChop)
 have "\<turnstile> f1 \<longrightarrow> f" using "1" by auto
 hence 4: "\<turnstile> f1;g \<longrightarrow> f;g" by (rule LeftChopImpChop)
 from 3 4 show ?thesis by (simp add: int_iffI)
qed

lemma OrChopEqvRule:
 assumes "\<turnstile> f = (f1 \<or> f2)"
 shows   "\<turnstile> f;g = ((f1;g) \<or> (f2;g))"
proof -
 have  1: "\<turnstile> f = (f1 \<or> f2)" using assms by auto
 hence 2: "\<turnstile> f;g = ((f1 \<or> f2);g)" by (rule LeftChopEqvChop) 
 have  3: "\<turnstile> (f1 \<or> f2);g = (f1;g \<or> f2;g)" by (rule OrChopEqv)
 from 2 3 show ?thesis by fastforce
qed

lemma NextImpNext:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows   "\<turnstile> \<circle> f \<longrightarrow> \<circle> g"
proof -
 have  1: "\<turnstile> f \<longrightarrow> g" using assms by auto
 hence 2: "\<turnstile> \<box> (f \<longrightarrow> g)" by (rule BoxGen)
 have  3: "\<turnstile> \<box> (f \<longrightarrow> g) \<longrightarrow> (skip;f) \<longrightarrow> (skip;g)" by (rule BoxChopImpChop)
 have  4: "\<turnstile>(skip;f) \<longrightarrow> (skip;g)" by (metis 2 3 MP)
 from 4 show ?thesis by (metis next_d_def)
qed

lemma ChopOrImpRule:
 assumes " \<turnstile> g \<longrightarrow> g1 \<or> g2"
 shows   " \<turnstile> f;g \<longrightarrow> (f;g1) \<or> (f;g2)"
proof -
 have  1: "\<turnstile> g \<longrightarrow> g1 \<or> g2" using assms by auto
 hence 2: "\<turnstile> f;g \<longrightarrow> f;(g1 \<or> g2)" by (rule RightChopImpChop)
 have  3: "\<turnstile> f;(g1 \<or> g2) = (f;g1 \<or> f;g2)" by (rule ChopOrEqv)
 from 2 3 show ?thesis by fastforce
qed

lemma NextImpDist:
 "\<turnstile> \<circle> (f \<longrightarrow> g) \<longrightarrow> \<circle> f \<longrightarrow> \<circle> g"
proof -
 have  1: "\<turnstile> (\<not> (f \<longrightarrow> g)) = (f \<and> \<not> g)" by auto
 hence 2: "\<turnstile> skip;(\<not> (f \<longrightarrow> g)) = skip;(f \<and> \<not> g)" by (rule RightChopEqvChop)
 have  3: "\<turnstile> f \<longrightarrow> g \<or> (f \<and> \<not> g)" by auto
 hence 4: "\<turnstile> skip;f \<longrightarrow> (skip;g) \<or> (skip;(f \<and> \<not> g))" by (rule ChopOrImpRule)
 hence 5: "\<turnstile> \<not> (skip;(f \<and> \<not> g)) \<longrightarrow> (skip;f) \<longrightarrow> (skip;g)" by auto
 have  6: "\<turnstile> \<not> (skip;(\<not>(f \<longrightarrow> g))) \<longrightarrow> (skip;f) \<longrightarrow> (skip;g)" using "2" "5" by fastforce
 hence 7: "\<turnstile> \<not> (\<circle>(\<not>(f \<longrightarrow> g))) \<longrightarrow> (\<circle> f) \<longrightarrow> (\<circle> g)" by (simp add: next_d_def)
 have  8: "\<turnstile> \<circle>(f \<longrightarrow> g) \<longrightarrow> \<not> (\<circle>(\<not>(f \<longrightarrow> g)))" by (rule NextImpNotNextNot)
 from 7 8 show ?thesis using lift_imp_trans by blast
qed

lemma FiniteChopImpDiamond:
 "\<turnstile> (f \<and> finite);g  \<longrightarrow> \<diamond> g"
proof -
 have  1: "\<turnstile> f \<and> finite \<longrightarrow> finite" by auto
 hence 2: "\<turnstile> (f \<and> finite);g \<longrightarrow> finite;g" by (rule LeftChopImpChop)
 from 2 show ?thesis by (simp add: sometimes_d_def)
qed


lemma NowImpDiamond:
 "\<turnstile> f \<longrightarrow> \<diamond> f"
proof -
 have  1: "\<turnstile> empty;f = f" by (rule EmptyChop)
 have  2: "\<turnstile> empty \<longrightarrow> finite" by (rule EmptyImpFinite)
 hence 3: "\<turnstile> empty;f \<longrightarrow> finite;f" by (rule LeftChopImpChop)
 have  4: " \<turnstile> f \<longrightarrow> finite;f" using "1" "3" by fastforce
 from 4 show ?thesis by (simp add: sometimes_d_def)
qed

lemma BoxElim:
 "\<turnstile> \<box> f \<longrightarrow> f "
proof -
  have  1: "\<turnstile> \<not> f \<longrightarrow> \<diamond> (\<not> f)" by (rule NowImpDiamond)
  hence 2: "\<turnstile> \<not> (\<diamond> (\<not> f)) \<longrightarrow> f " by auto
  from 2 show ?thesis by (metis always_d_def)
qed

lemma NextDiamondImpDiamond:
 "\<turnstile> \<circle> (\<diamond> f) \<longrightarrow> \<diamond> f"
proof -
 have  1: "\<turnstile> skip;(finite;f) = ((skip;finite);f)" by (rule ChopAssoc)
 hence 2: "\<turnstile> (skip;finite);f = skip;(finite;f)" by auto 
 hence 3: "\<turnstile> (skip;finite);f = \<circle>(\<diamond>f)" by (simp add: next_d_def sometimes_d_def)
 have  4: "\<turnstile> (skip;finite);f \<longrightarrow> \<diamond> f" 
 by (simp add: SkipChopFiniteImpFinite LeftChopImpChop sometimes_d_def)
 from 3 4 show ?thesis by fastforce
qed

lemma BoxImpNowAndWeakNext:
 " \<turnstile> \<box> f \<longrightarrow> (f \<and> wnext ( \<box> f) )"
proof -
  have  1: "\<turnstile> \<not> f \<longrightarrow> \<diamond> (\<not> f) " by (rule NowImpDiamond)
  hence 2: "\<turnstile> \<not> (\<diamond> (\<not> f)) \<longrightarrow> f" by auto
  hence 3: "\<turnstile> \<box> f \<longrightarrow> f" by (metis always_d_def)
  have  4: "\<turnstile> \<circle> ( \<diamond> (\<not> f)) \<longrightarrow> \<diamond> ( \<not> f ) " by (rule NextDiamondImpDiamond)
  have  5: "\<turnstile> \<not> \<not> (\<diamond> (\<not> f)) \<longrightarrow> \<diamond>( \<not> f ) " by auto
  hence 6: "\<turnstile> \<circle> ( \<not> \<not> (\<diamond> (\<not> f)) )  \<longrightarrow> \<circle> (\<diamond>( \<not> f )) " by (rule NextImpNext)
  have  7: "\<turnstile> \<circle> ( \<not> \<not> (\<diamond> (\<not> f)) )  \<longrightarrow> \<diamond> ( \<not> f )" using "4" "6" by auto  
  hence 8: "\<turnstile> \<circle> ( \<not>( \<box> f)) \<longrightarrow> \<diamond> ( \<not> f ) " by (simp add: always_d_def)
  hence 9: "\<turnstile> \<not> (\<diamond> ( \<not> f ))  \<longrightarrow> \<not> ( \<circle> ( \<not>( \<box> f)))" by auto
  hence 10:"\<turnstile> \<box>f \<longrightarrow> wnext ( \<box> f )" by (simp add: always_d_def wnext_d_def) 
  from 3 10 show ?thesis by fastforce
qed

lemma BoxImpBoxRule:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows   "\<turnstile> \<box> f\<longrightarrow> \<box> g" 
proof - 
 have  1: "\<turnstile> f \<longrightarrow> g" using assms by auto
 hence 2: "\<turnstile> \<not> g \<longrightarrow>  \<not> f" by auto
 hence 3: "\<turnstile> \<box>(\<not> g \<longrightarrow>  \<not> f)" by (rule BoxGen)
 have  4: "\<turnstile>  \<box>(\<not> g \<longrightarrow>  \<not> f) \<longrightarrow> (finite;(\<not>g)) \<longrightarrow> (finite;(\<not>f))" by (rule BoxChopImpChop)
 have  5: "\<turnstile> (finite;(\<not>g)) \<longrightarrow> (finite;(\<not>f))" using "3" "4" MP by blast
 hence 6: "\<turnstile> \<diamond> (\<not>g)  \<longrightarrow> \<diamond>(\<not>f)" by (simp add: sometimes_d_def)
 hence 7: "\<turnstile>  \<not> (\<diamond>(\<not>f) ) \<longrightarrow> \<not>( \<diamond> (\<not>g))" by auto
 from 7 show ?thesis by (simp add: always_d_def)
qed

lemma BoxImpDist:
 " \<turnstile>  \<box>(f \<longrightarrow> g) \<longrightarrow> \<box> f \<longrightarrow> \<box> g"
proof -
 have  1: "\<turnstile> (f \<longrightarrow> g) \<longrightarrow> (\<not>  g \<longrightarrow> \<not>  f) " 
   by auto
 hence 2: "\<turnstile> \<box>(f \<longrightarrow> g) \<longrightarrow> \<box>(\<not>  g \<longrightarrow> \<not>  f) " 
   by (rule BoxImpBoxRule)
 have  3: "\<turnstile> \<box>((\<not>  g) \<longrightarrow> \<not>  f) \<longrightarrow>  (finite; (\<not>  g)) \<longrightarrow> (finite; (\<not>  f))" 
   by (rule BoxChopImpChop)
 have  4: "\<turnstile> \<box>(f \<longrightarrow> g) \<longrightarrow>   (finite; (\<not>  g)) \<longrightarrow> (finite; (\<not>  f))" 
   using "2" "3" lift_imp_trans by blast   
 hence 5: "\<turnstile> \<box>(f \<longrightarrow> g) \<longrightarrow>   \<diamond>(\<not>  g) \<longrightarrow> \<diamond>(\<not>  f) " 
   by (simp add: sometimes_d_def)
 hence 6: "\<turnstile> \<box>(f \<longrightarrow> g) \<longrightarrow>   \<not>( \<diamond>(\<not>  f)) \<longrightarrow> \<not>( \<diamond>(\<not>  g)) " 
   by auto
 from 6 show ?thesis by (simp add: always_d_def)
qed

lemma DiamondEmptyEqvFinite:
 " \<turnstile>   \<diamond> empty = finite"
proof -
 have 1: "\<turnstile> finite;  empty  = finite " by (rule ChopEmpty)
 from 1 show ?thesis by (simp add: sometimes_d_def)
qed

lemma NextEqvNext:
 assumes "\<turnstile>   f= g"
 shows   "\<turnstile> \<circle> f = \<circle> g"
proof -
 have  1: "\<turnstile> f= g" using assms by auto
 hence 2: "\<turnstile> skip;f = skip;g"  by (rule RightChopEqvChop) 
 from 1 show ?thesis by (metis "2" next_d_def)
qed
 
lemma NextAndNextImpNextRule:
 assumes "\<turnstile> (f \<and> g) \<longrightarrow> h " 
 shows   "\<turnstile> (\<circle> f \<and> \<circle> g) \<longrightarrow> \<circle> h "
using assms 
by  (simp add: Valid_def itl_defs)

lemma NextAndNextEqvNextRule:
 assumes "\<turnstile> (f \<and> g) = h "
 shows   "\<turnstile> (\<circle> f \<and> \<circle> g) = \<circle> h "
using assms
by (simp add: NextAndNextImpNextRule NextImpNext Prop11 Prop12) 

lemma WeakNextEqvWeakNext:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile> wnext  f = wnext  g "
using assms using inteq_reflection by force

lemma DiamondImpDiamond:
 assumes "\<turnstile> f \<longrightarrow> g "
 shows   "\<turnstile> \<diamond> f \<longrightarrow> \<diamond> g "
using assms by (simp add: RightChopImpChop sometimes_d_def)

lemma DiamondEqvDiamond:
 assumes "\<turnstile> f = g " 
 shows   "\<turnstile> \<diamond> f = \<diamond> g "
using assms using int_eq by force

lemma BoxEqvBox:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile> \<box> f = \<box> g "
using assms using inteq_reflection by force

lemma BoxAndBoxImpBoxRule:
 assumes "\<turnstile> f \<and> g \<longrightarrow> h "
 shows   "\<turnstile> \<box> f \<and> \<box> g \<longrightarrow> \<box> h "
using assms by (auto simp: itl_defs Valid_def)

lemma BoxAndBoxEqvBoxRule:
 assumes "\<turnstile> (f \<and> g) = h "
 shows   "\<turnstile> (\<box> f \<and> \<box> g) = \<box> h " 
using assms BoxAndBoxImpBoxRule BoxImpBoxRule by (metis int_iffD1 int_iffD2 int_iffI Prop12)

lemma ImpBoxRule:
 assumes "\<turnstile>   f \<longrightarrow> g "
 shows   "\<turnstile> \<box> f \<longrightarrow> \<box> g "
using assms by (simp add: BoxImpBoxRule)

lemma WnextEqvEmptyOrNext:
 "\<turnstile> wnext f = (empty \<or> \<circle> f)"
by (auto simp: Valid_def itl_defs zero_enat_def)

lemma BoxIntro:
 assumes "\<turnstile>  f \<longrightarrow> g" 
         "\<turnstile>  more  \<and> f \<longrightarrow> \<circle> f "
 shows   "\<turnstile> f \<longrightarrow> \<box> g "
proof -
 have   1: "\<turnstile>  more  \<and> f \<longrightarrow>\<circle> f " 
   using assms by auto
 hence  2: "\<turnstile> f\<longrightarrow> (empty \<or> \<circle>f)"
   unfolding empty_d_def by fastforce 
 hence  3: "\<turnstile> f \<longrightarrow> wnext f"
   by (metis WnextEqvEmptyOrNext inteq_reflection) 
 hence  4: "\<turnstile> \<box>(f \<longrightarrow> wnext f)" 
   by (rule BoxGen)
 have   5: "\<turnstile> (\<box> (f \<longrightarrow> wnext f)) \<and> f \<longrightarrow> \<box> f  " 
   by (rule BoxInduct)
 hence  6: "\<turnstile> (\<box> (f \<longrightarrow> wnext f)) \<longrightarrow> (f \<longrightarrow> \<box>f)" 
   by fastforce
 have   7: "\<turnstile> f \<longrightarrow> \<box>f " 
   using "4" "6" MP by blast
 have   8: "\<turnstile> \<box>f \<longrightarrow> f" 
   by (rule BoxElim)
 have   9: "\<turnstile> f = \<box> f" 
   using "7" "8" by fastforce
 have  10: "\<turnstile> f \<longrightarrow> g" 
   using assms by auto
 hence 11: "\<turnstile> \<box>f \<longrightarrow> \<box> g" 
   by (rule ImpBoxRule)
 from 7 9 11 show ?thesis  
   using lift_imp_trans by blast 
qed

lemma NextLoop:
 assumes "\<turnstile> f  \<longrightarrow> \<circle> f" 
 shows   "\<turnstile> finite \<longrightarrow> \<not>  f "
proof -
 have   1: "\<turnstile> f\<longrightarrow> \<circle> f " 
   using assms by auto
 hence  2: "\<turnstile> f \<longrightarrow> (more \<and> wnext f)"
   unfolding more_d_def wnext_d_def
   by (metis NextImpNext NextImpNotNextNot Prop05 Prop10 Prop12 int_iffD1 int_simps(29) lift_imp_trans)
 hence  3: "\<turnstile> f \<longrightarrow> wnext f " 
   by auto
 hence  4: "\<turnstile> \<box>(f \<longrightarrow> wnext f)" 
   by (rule BoxGen)
 have   5: "\<turnstile> \<box> (f \<longrightarrow> wnext f) \<and> f \<longrightarrow> \<box> f  " 
   by (rule BoxInduct)
 hence  6: "\<turnstile> \<box> (f \<longrightarrow> wnext f) \<longrightarrow> (f \<longrightarrow> \<box>f)" 
   by fastforce
 have   7: "\<turnstile> f \<longrightarrow> \<box>f "  
   using "4" "6" MP by blast
 have   8: "\<turnstile> \<box>f \<longrightarrow> f" 
   by (rule BoxElim)
 have   9: "\<turnstile> f = \<box> f" 
   using "7" "8" by fastforce
 have  10: "\<turnstile> f \<longrightarrow> more" 
   using "2" by auto
 hence 11: "\<turnstile> \<box> f \<longrightarrow> \<box> more" 
   by (rule ImpBoxRule)
 have  12: "\<turnstile> finite = (\<not>(\<box> more)) "
   by (metis DiamondEmptyEqvFinite InfEqvNotFinite always_d_def empty_d_def finite_d_def int_eq) 
 from 7 9 11 12 show ?thesis 
   by fastforce
qed

lemma NotEmptyAndNext:
 "\<turnstile> \<not>(empty \<and> \<circle> f)"
by (auto simp: Valid_def itl_defs zero_enat_def) 

lemma BoxEqvAndWnextBox:
 "\<turnstile> \<box> f = (f \<and> wnext ( \<box> f))"
proof -
 have 1: "\<turnstile> \<box> f \<longrightarrow> f \<and> wnext ( \<box> f)" 
   using BoxImpNowAndWeakNext by blast
 have 2: "\<turnstile> f \<and> wnext ( \<box> f) \<longrightarrow> f" 
   by auto
 have 3: "\<turnstile> more \<and> (f \<and> wnext ( \<box> f) ) \<longrightarrow> \<circle> (f \<and> wnext ( \<box> f) )" 
   using "1" NextImpNext WnextEqvEmptyOrNext empty_d_def int_iffD1 
   by (metis Prop01 Prop05 Prop08)
 have 4: "\<turnstile> f \<and> wnext ( \<box> f) \<longrightarrow> \<box> f" 
   using "2" "3" BoxIntro by blast
 from 1 4 show ?thesis by fastforce
qed

lemma BoxEqvAndEmptyOrNextBox:
 "\<turnstile> \<box>f = (f \<and> (empty  \<or>  \<circle>(\<box> f)))"
using BoxEqvAndWnextBox WnextEqvEmptyOrNext by (metis int_eq)

lemma BoxEqvBoxBox:
 "\<turnstile> \<box>f = \<box> (\<box> f)"
using BoxGen BoxInduct 
by (metis BoxImpNowAndWeakNext MP int_iffI Prop09 Prop12)

lemma BoxBoxImpBox:
 "\<turnstile> \<box>(\<box>h) \<longrightarrow> \<box> h"
by (simp add: BoxElim)

lemma BoxImpBoxBox:
 "\<turnstile> \<box> h  \<longrightarrow>  \<box>(\<box>h)"
 by (simp add: BoxEqvBoxBox int_iffD1)

lemma DiamondIntroC:
 assumes "\<turnstile> f \<longrightarrow> \<circle> g"
 shows   "\<turnstile> f \<longrightarrow> \<diamond> g"
using assms 
by (metis (no_types, lifting) ChopAssoc FiniteChopSkipEqvSkipChopFinite NextChop 
          NextDiamondImpDiamond NowImpDiamond inteq_reflection lift_imp_trans next_d_def 
          sometimes_d_def)

lemma DiamondIntro:
 assumes "\<turnstile> (f \<and> \<not>  g) \<longrightarrow> \<circle> f "
 shows   "\<turnstile> f \<and> finite \<longrightarrow> \<diamond> g "
proof -
 have   1: "\<turnstile> f \<and> \<not>  g \<longrightarrow> \<circle> f " 
   using assms by auto
 hence  2: "\<turnstile> f \<and> \<not>  g \<and> (\<box> (\<not> g))\<longrightarrow> (\<circle> f) \<and> (\<box> (\<not> g))" 
   by auto
 have   3: "\<turnstile> (\<box> (\<not> g)) \<longrightarrow> \<not>  g" 
   by (rule BoxElim)
 hence  4: "\<turnstile> \<box> (\<not> g) = ((\<box> (\<not> g)) \<and> \<not>  g)" 
   using BoxImpBoxBox BoxBoxImpBox by fastforce
 have   5: "\<turnstile> f \<and> (\<box> (\<not> g))\<longrightarrow> \<circle> f \<and> \<box> (\<not> g) " 
   using "2" "4"  by fastforce
 have   6: "\<turnstile> \<box> (\<not> g) = ((\<not> g) \<and> wnext(\<box> (\<not> g)))  "  
   using BoxEqvAndWnextBox by metis
 have   7: "\<turnstile> \<circle> f \<and> \<box> (\<not> g) \<longrightarrow>  \<circle> f \<and> wnext(\<box> (\<not> g)) " 
   using "6" by auto
 have   8: "\<turnstile> f \<and> (\<box> (\<not> g))\<longrightarrow> \<circle> f \<and> wnext(\<box> (\<not> g)) " 
   using "5" "7" using lift_imp_trans by blast
 hence  9: "\<turnstile> f \<and> (\<box> (\<not> g))\<longrightarrow> more \<and> wnext f \<and> wnext(\<box> (\<not> g))" 
   using zero_enat_def by (auto simp: Valid_def itl_defs)
 hence 10: "\<turnstile> f \<and> (\<box> (\<not> g))\<longrightarrow>wnext f \<and> wnext(\<box> (\<not> g)) " 
   by auto
 hence 11: "\<turnstile> f \<and> (\<box> (\<not> g))\<longrightarrow> wnext (f \<and> \<box> (\<not> g)) " 
   by (auto simp: Valid_def itl_defs)
 hence 12: "\<turnstile> \<box>(f \<and> (\<box> (\<not> g))\<longrightarrow> wnext (f \<and> \<box> (\<not> g))) " 
   by (rule BoxGen)
 have  13: "\<turnstile> \<box>(f \<and> (\<box> (\<not> g))\<longrightarrow> wnext (f \<and> \<box> (\<not> g))) \<and> f \<and> (\<box> (\<not> g)) \<longrightarrow> \<box>(f \<and> (\<box> (\<not> g))) " 
   by (rule BoxInduct)
 hence 14: "\<turnstile> \<box>(f \<and> (\<box> (\<not> g))\<longrightarrow> wnext (f \<and> \<box>(\<not> g))) \<longrightarrow> ((f \<and> (\<box>(\<not> g))) \<longrightarrow> \<box>(f \<and> (\<box> (\<not> g)))) " 
   by fastforce
 have  15: "\<turnstile> ((f \<and> (\<box> (\<not> g))) \<longrightarrow> \<box>(f \<and> (\<box> (\<not> g)))) " 
   using "12" "14" MP by blast
 have  16: "\<turnstile> \<box>(f \<and> (\<box> (\<not> g))) \<longrightarrow> (f \<and> (\<box> (\<not> g))) " 
   by (rule BoxElim)
 have  17: "\<turnstile> \<box>(f \<and> (\<box> (\<not> g))) = (f \<and> (\<box> (\<not> g)))" 
   using "16" "15" by fastforce
 have  18: "\<turnstile> (f \<and> (\<box> (\<not> g))) \<longrightarrow> more" 
   using "9" by auto
 hence 19: "\<turnstile> \<box>(f \<and> (\<box> (\<not> g))) \<longrightarrow> \<box> more" 
   by (rule ImpBoxRule)
 have  20: "\<turnstile> finite = (\<not>(\<box> more)) "
   by (metis DiamondEmptyEqvFinite InfEqvNotFinite always_d_def empty_d_def finite_d_def int_eq)   
 have  21: "\<turnstile> finite \<longrightarrow> \<not>(f \<and> (\<box> (\<not> g)))" 
   using "17" "19" "20" by fastforce
 hence 22: "\<turnstile> finite \<longrightarrow> \<not> f \<or> \<not> (\<box> (\<not> g)) " 
   by auto
 have  23: "\<turnstile> (\<not> (\<box> (\<not> g))) = \<diamond> g" 
   by (auto simp: always_d_def)
 from 22 23 show ?thesis by fastforce
qed
 
lemma DiamondIntroB:
 assumes "\<turnstile> (f \<and> \<not>  g) \<longrightarrow> \<circle> (f \<and> \<not> g) "
 shows   "\<turnstile> f \<and> finite \<longrightarrow> \<diamond> g "
proof -
 have  1: "\<turnstile> (f \<and> \<not>  g) \<longrightarrow> \<circle> (f \<and> \<not>g) " using assms by auto
 hence 2: "\<turnstile> finite \<longrightarrow> \<not>(f \<and> \<not> g) " by (rule NextLoop)
 hence 3: "\<turnstile> f \<and> finite \<longrightarrow> g" by auto
 have  4: "\<turnstile> g \<longrightarrow> \<diamond> g" by (rule NowImpDiamond)
 from 3 4 show ?thesis using lift_imp_trans by blast
qed

lemma NextContra :
 assumes "\<turnstile> (f \<and> \<not>  g) \<longrightarrow> (\<circle> f \<and> \<not>( \<circle> g)) "
 shows   "\<turnstile> f \<and> finite \<longrightarrow> g "
proof -
 have  1: "\<turnstile> (f\<and> \<not>  g)\<longrightarrow> (\<circle> f \<and> \<not>( \<circle> g)) " using assms by auto
 hence 2: "\<turnstile> \<not>( f \<longrightarrow> g) \<longrightarrow> \<circle> ( \<not>(f \<longrightarrow> g)) " by (auto simp: itl_defs Valid_def)
 hence 3: "\<turnstile> finite \<longrightarrow> \<not> \<not>( f \<longrightarrow> g) " by (rule NextLoop)
 from 3 show ?thesis by auto
qed

lemma DiamondDiamondEqvDiamond:
 " \<turnstile> \<diamond>(\<diamond> f) = \<diamond> f"
proof -
 have  1: "\<turnstile> finite;finite = finite" by (simp add: FiniteChopFiniteEqvFinite)
 hence 2: "\<turnstile> (finite;finite);f = finite;f " using LeftChopEqvChop by blast
 have  3: "\<turnstile> (finite;finite);f = finite;(finite;f) " using ChopAssoc by fastforce
 from 2 3 show ?thesis by (metis inteq_reflection sometimes_d_def)
qed

lemma WeakNextDiamondInduct:
 assumes "\<turnstile> wnext  (\<diamond> f) \<longrightarrow> f "
 shows   "\<turnstile> finite \<longrightarrow> f "
proof -
 have  1: "\<turnstile> wnext  (\<diamond> f) \<longrightarrow> f " using assms by blast
 hence 2: "\<turnstile> \<not> f \<longrightarrow> \<not>( wnext  (\<diamond> f))" by fastforce  
 hence 3: "\<turnstile> \<not> f \<longrightarrow> \<circle>( \<not> (\<diamond> f)) " by (simp add: wnext_d_def)
 have  4: "\<turnstile> f \<longrightarrow> \<diamond> f" by (rule NowImpDiamond)
 hence 5: "\<turnstile> \<not>(\<diamond> f) \<longrightarrow> \<not> f  " by auto
 have  6: "\<turnstile> \<not>f \<longrightarrow> \<circle>( \<not>f)" using "3" "5" using NextImpNext lift_imp_trans by blast  
 hence 7: "\<turnstile> finite \<longrightarrow> \<not>\<not> f" by (rule NextLoop)
 from 7 show ?thesis by auto
qed

lemma EmptyNextInducta:
 assumes "\<turnstile> empty  \<longrightarrow> f " 
         "\<turnstile> \<circle> f \<longrightarrow> f " 
 shows "\<turnstile> finite \<longrightarrow> f "
proof -
 have  1: "\<turnstile> empty  \<longrightarrow> f " using assms by auto
 have  2: "\<turnstile> \<circle> f \<longrightarrow> f " using assms by blast
 have  3: "\<turnstile> (empty \<or> \<circle> f) \<longrightarrow> f  " using "1" "2"  by fastforce
 have  4: "\<turnstile> wnext f = (empty \<or> \<circle> f)" by  (rule  WnextEqvEmptyOrNext)
 hence 5: "\<turnstile> wnext f \<longrightarrow> f" using "3" by fastforce
 hence 6: "\<turnstile> \<not>f \<longrightarrow> \<not> (wnext f)" by auto
 hence 7: "\<turnstile> \<not>f \<longrightarrow> \<circle>(\<not> f) " by (auto simp: wnext_d_def)
 hence 8: "\<turnstile> finite \<longrightarrow> \<not> \<not> f " by (rule NextLoop)
 from 8 show ?thesis by auto
qed

lemma EmptyNextInductb:
 assumes "\<turnstile> empty  \<and> f \<longrightarrow> g " 
         "\<turnstile> \<circle>(f\<longrightarrow> g) \<and> f \<longrightarrow> g "
 shows   "\<turnstile> f \<and> finite \<longrightarrow> g "
proof -
 have  1: "\<turnstile> empty  \<and> f \<longrightarrow> g " using assms by auto
 have  2: "\<turnstile> \<circle>(f\<longrightarrow> g) \<and> f \<longrightarrow> g " using assms by blast
 have  3: "\<turnstile> (empty \<or> \<circle>(f\<longrightarrow> g)) \<and> f \<longrightarrow> g " using "1" "2" by fastforce  
 hence 4: "\<turnstile> wnext (f\<longrightarrow> g)  \<and> f \<longrightarrow> g "   using WnextEqvEmptyOrNext by fastforce 
 hence 5: "\<turnstile> wnext (f\<longrightarrow> g) \<longrightarrow> ( f \<longrightarrow> g) "  by fastforce 
 hence 6: "\<turnstile> \<not> ( f \<longrightarrow> g) \<longrightarrow> \<not> ( wnext (f\<longrightarrow> g)) " by fastforce
 hence 7: "\<turnstile> \<not> ( f \<longrightarrow> g) \<longrightarrow> \<circle> ( \<not>(f\<longrightarrow> g)) " by (simp add: wnext_d_def)
 hence 8: "\<turnstile> finite \<longrightarrow> \<not> \<not> ( f \<longrightarrow> g)" by (rule NextLoop)
 from 8 show ?thesis by auto
qed


lemma FinImpFin:
 assumes "\<turnstile> f \<longrightarrow> g "
 shows   "\<turnstile>  fin  f \<longrightarrow>  fin  g "
using ImpBoxRule[of "LIFT (empty \<longrightarrow> f)" "LIFT (empty \<longrightarrow> g)"] assms 
      fin_d_def[of "f"] fin_d_def[of "g"] by fastforce 

lemma FinEqvFin:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile>  fin  f =  fin  g "
using assms by (simp add: FinImpFin Prop11) 

lemma FinAndFinImpFinRule:
 assumes "\<turnstile> f \<and> g \<longrightarrow> h "
 shows   "\<turnstile>  fin  f \<and>  fin  g \<longrightarrow>  fin  h "
using assms by (auto simp add: itl_defs Valid_def)  

lemma FinAndFinEqvFinRule:
 assumes "\<turnstile> (f \<and> g) = h "
 shows   "\<turnstile>  (fin  f \<and>  fin  g) =  fin  h "
using assms 
by (simp add: FinAndFinImpFinRule FinImpFin Prop11 Prop12)

lemma HaltEqvHalt:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile>  halt f =  halt g "
proof -
 have  1: "\<turnstile> f = g" using assms by auto
 hence 2: "\<turnstile> (empty = f)  = (empty = g) " by auto
 hence 3: "\<turnstile> \<box>(empty = f)  = \<box> (empty = g)" by (rule BoxEqvBox)
 from 3 show ?thesis by (simp add: halt_d_def)
qed

lemma BiImpDiImpDi:
  "\<turnstile> bi (f \<longrightarrow> g) \<longrightarrow>  di  f \<longrightarrow>  di  g "
proof -
 have 1: "\<turnstile> bi (f\<longrightarrow> g)  \<longrightarrow> (f; #True) \<longrightarrow> (g; #True) "  by (rule BiChopImpChop)
 from 1 show ?thesis   by (simp add: di_d_def)
qed

lemma DiImpDi:
 assumes "\<turnstile> f \<longrightarrow> g "
 shows   "\<turnstile>  di  f \<longrightarrow>  di  g "
proof -
 have  1: "\<turnstile> f\<longrightarrow> g " using assms by auto
 hence 2: "\<turnstile> f; #True \<longrightarrow> g; #True " by (rule LeftChopImpChop)
 from 2 show ?thesis by (simp add: di_d_def)
qed

lemma BiImpBiRule:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows   "\<turnstile> bi f \<longrightarrow> bi g "
proof -
 have  1: "\<turnstile> f\<longrightarrow> g" using assms by auto
 hence 2: "\<turnstile> \<not>  g \<longrightarrow> \<not>  f " by auto
 hence 3: "\<turnstile>  di (\<not>  g) \<longrightarrow>  di (\<not>  f) " by (rule DiImpDi)
 hence 4: "\<turnstile> \<not> ( di (\<not>  f)) \<longrightarrow> \<not> ( di (\<not>  g)) " by auto
 from 4 show ?thesis by (simp add: bi_d_def)
qed

lemma DiEqvDi:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile>  di f =  di g "
proof -
 have  1: "\<turnstile> f = g " using assms by auto
 hence 2: "\<turnstile> f; #True = g; #True " by (rule LeftChopEqvChop)
 from 2 show ?thesis by (simp add: di_d_def)
qed

lemma BiEqvBi:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile>  bi f =  bi  g "
proof -
 have  1: "\<turnstile> f = g " using assms by auto
 hence 2: "\<turnstile> (\<not>  f) = (\<not>  g) " by auto
 hence 3: "\<turnstile>  di (\<not>  f) =  di (\<not>  g) " by (rule DiEqvDi)
 hence 4: "\<turnstile> (\<not>  (di (\<not>  f))) = (\<not> ( di (\<not>  g))) " by auto
 from 4 show ?thesis by (simp add: bi_d_def)
qed

lemma LeftChopChopImpChopRule:
 assumes "\<turnstile> (f; g) \<longrightarrow> g "
 shows   "\<turnstile> (f; g); h \<longrightarrow> (g; h) "
proof -
 have  1: "\<turnstile> (f; g) \<longrightarrow> g " using assms by blast
 hence 2: "\<turnstile> (f; g); h \<longrightarrow> g; h " by (rule LeftChopImpChop)
 have  3: "\<turnstile> f; (g; h) = (f; g); h   " by (rule ChopAssoc) 
 from 2 3 show ?thesis by auto
qed

lemma AndChopCommute :
 "\<turnstile> (f \<and> f1); g = (f1 \<and> f); g "
proof -
 have 1: "\<turnstile> (f \<and> f1) = (f1 \<and> f)  " by auto
 from 1 show ?thesis by (rule LeftChopEqvChop)
qed

lemma BiAndChopImport:
 "\<turnstile>  bi f \<and> (f1; g) \<longrightarrow> (f \<and> f1);g "
proof -
 have  1: "\<turnstile> f \<longrightarrow> (f1 \<longrightarrow> f\<and> f1) " by auto
 hence 2: "\<turnstile> bi  f \<longrightarrow> bi  (f1 \<longrightarrow> f\<and> f1) " by (rule BiImpBiRule)
 have  3: "\<turnstile> bi  (f1 \<longrightarrow> (f \<and> f1)) \<longrightarrow>   f1; g \<longrightarrow> (f \<and> f1); g " by (rule BiChopImpChop)
 from 2 3 show ?thesis using MP by fastforce
qed

lemma StateAndChopImport:
 "\<turnstile> (init w) \<and> (f;g) \<longrightarrow> ((init w) \<and> f);g "
proof -
 have  1: "\<turnstile> (init w)\<longrightarrow> bi  (init w) " by (rule StateImpBi)
 hence 2: "\<turnstile> (init w) \<and> (f; g) \<longrightarrow> bi (init w) \<and> (f; g) " by auto
 have  3: "\<turnstile> bi  (init w) \<and> (f; g) \<longrightarrow> ((init w) \<and> f); g "  by (rule BiAndChopImport)
 from 2 3 show ?thesis using MP by fastforce
qed

subsection \<open>Further Properties Di and Bi\<close>

lemma ImpDi:
 "\<turnstile> f \<longrightarrow>  di  f "
proof -
 have  1: "\<turnstile> f;  empty  = f " by (rule ChopEmpty)
 have  2: "\<turnstile> empty  \<longrightarrow> #True " by auto
 hence 3: "\<turnstile> f;  empty  \<longrightarrow> f; #True " by (rule RightChopImpChop)
 have 4 : "\<turnstile> f \<longrightarrow> f; #True " using "1" "3" by fastforce
 from 4 show ?thesis by (simp add: di_d_def)
qed

lemma DiState:
 "\<turnstile> di (init w) = (init w) "
proof -
 have  0: "\<turnstile> (init (\<not>w)) \<longrightarrow> bi   (init (\<not>w)) " using StateImpBi by fastforce
 hence 1: "\<turnstile> \<not>(init w) \<longrightarrow> bi ( \<not> (init w))" using Initprop(2) by (metis inteq_reflection)
 hence 2: "\<turnstile> (\<not>  (init w)) \<longrightarrow> \<not> ( di (\<not> \<not>  (init w))) " by (simp add: bi_d_def)
 have  3: "\<turnstile> (\<not>  (init w) \<longrightarrow>\<not>  (di (\<not> \<not>  (init w)))) \<longrightarrow> 
             ( di (\<not> \<not>  (init w)) \<longrightarrow> (init w)) " by auto
 have  4: "\<turnstile>  di (\<not> \<not>  (init w)) \<longrightarrow> (init w) " using "2" "3" MP by blast
 have  5: "\<turnstile> (init w) \<longrightarrow> \<not> \<not>  (init w) " by auto
 hence 6: "\<turnstile>  di  (init w) \<longrightarrow>  di (\<not> \<not>  (init w)) " by (rule DiImpDi) 
 have  7: "\<turnstile>  di  (init w) \<longrightarrow> (init w)" using "6" "4" using lift_imp_trans by metis
 have  8: "\<turnstile> (init w) \<longrightarrow>  di  (init w) " by (rule ImpDi)
 from 7 8 show ?thesis by fastforce
qed

lemma StateChop:
 "\<turnstile> (init w); f \<longrightarrow> (init w) "
 by (metis ChopAssoc Prop12 RightChopImpChop TrueW init_d_def int_simps(13) int_simps(17)
     inteq_reflection)

lemma StateChopExportA:
 "\<turnstile> ((init w) \<and> f); g \<longrightarrow> (init w) "
using DiState AndChopA StateChop by fastforce 

lemma StateAndChop:
 "\<turnstile> ((init w) \<and> f); g = ((init w) \<and> (f; g)) "
by (simp add: AndChopB StateAndChopImport StateChopExportA Prop11 Prop12)
 
lemma StateAndChopImpChopRule:
 assumes "\<turnstile> (init w) \<and> f \<longrightarrow> f1 " 
 shows   "\<turnstile> (init w) \<and> (f; g) \<longrightarrow> (f1; g) "
proof -
 have  1: "\<turnstile> (init w) \<and> f \<longrightarrow> f1 " using assms by auto
 hence 2: "\<turnstile> ((init w) \<and> f); g \<longrightarrow> f1; g " by (rule LeftChopImpChop)
 have  3: "\<turnstile> ((init w) \<and> f); g = ((init w) \<and> (f; g)) " by (rule StateAndChop)
 from 2 3 show ?thesis by fastforce
qed

lemma StateImpChopEqvChop :
 assumes "\<turnstile> (init w) \<longrightarrow> (f = f1)" 
 shows   "\<turnstile> (init w) \<longrightarrow> ((f; g) = (f1; g)) "
proof -
 have  1: "\<turnstile> (init w) \<longrightarrow> (f= f1)" using assms by auto
 hence 2: "\<turnstile> (init w) \<and> f \<longrightarrow> f1 " by auto
 hence 3: "\<turnstile> (init w) \<and> (f; g) \<longrightarrow> (f1; g) " by (rule StateAndChopImpChopRule)
 have  4: "\<turnstile> (init w) \<and> f1 \<longrightarrow> f " using "1" by auto
 hence 5: "\<turnstile> (init w) \<and> (f1; g) \<longrightarrow> (f; g)" by (rule StateAndChopImpChopRule)
 from 3 5 show ?thesis by fastforce
qed

lemma ChopEqvStateAndChop:
  assumes "\<turnstile> f = (init w) \<and> f1 "
  shows   "\<turnstile> (f; g) = ((init w) \<and> (f1; g)) "
proof -
 have  1: "\<turnstile> f = ((init w) \<and> f1) " using assms by auto
 hence 2: "\<turnstile> f; g = (((init w) \<and> f1); g) " by (rule LeftChopEqvChop)
 have  3: "\<turnstile> ((init w) \<and> f1); g = ((init w) \<and> (f1; g)) " by (rule StateAndChop)
 from 2 3 show ?thesis by fastforce
qed

lemma DiIntro:
 "\<turnstile> f \<longrightarrow>  di  f "
proof -
  have  1: "\<turnstile> f;empty  = f " by (rule ChopEmpty)
  have  2: "\<turnstile>  empty  \<longrightarrow> #True " by auto
  hence 3: "\<turnstile> \<box>( empty  \<longrightarrow> #True)" by (rule BoxGen)
  have  4: "\<turnstile> \<box>( empty  \<longrightarrow> #True) \<longrightarrow>  (f;  empty  \<longrightarrow> f; #True) " by (rule BoxChopImpChop)
  have  5: "\<turnstile> f;empty  \<longrightarrow> f;#True " using "3" "4" MP by fastforce
  hence 6: "\<turnstile> f;empty  \<longrightarrow>  di  f"  by (simp add: di_d_def)
  from 1 6 show ?thesis by fastforce
qed

lemma BiElim:
 "\<turnstile>   bi  f \<longrightarrow> f "
proof - 
  have  1: "\<turnstile> \<not>f \<longrightarrow>  di (\<not>f) " by (rule DiIntro)
  have  2: "\<turnstile> (\<not>f \<longrightarrow>  di (\<not>f)) \<longrightarrow> (\<not>(di (\<not>f)) \<longrightarrow> f) " by auto
  have  3: "\<turnstile> \<not>  (di (\<not>f)) \<longrightarrow> f " using 1 2 MP by blast
  from 3 show ?thesis by (metis bi_d_def)
qed

lemma BiContraPosImpDist:
  "\<turnstile> bi (\<not>g \<longrightarrow> \<not>f) \<longrightarrow> (bi f) \<longrightarrow> (bi g) "
proof -
  have  1: "\<turnstile> bi (\<not>g \<longrightarrow> \<not>f) \<longrightarrow> (di (\<not>g)) \<longrightarrow> (di (\<not>f)) " by (rule BiImpDiImpDi)
  hence 2: "\<turnstile> bi (\<not>g \<longrightarrow> \<not>f) \<longrightarrow> (\<not>( di (\<not>f))) \<longrightarrow> (\<not>(di (\<not>g))) " by auto
  from 2 show ?thesis by (metis bi_d_def)
qed

lemma BiImpDist:
 "\<turnstile> bi (f\<longrightarrow> g) \<longrightarrow> (bi f) \<longrightarrow> (bi g) "
proof -
 have  1: "\<turnstile> (f \<longrightarrow> g) \<longrightarrow> (\<not>  g \<longrightarrow> \<not>  f) " by auto
 hence 2: "\<turnstile> \<not> (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> \<not> (f \<longrightarrow> g) " by auto
 hence 3: "\<turnstile> bi (\<not> (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> \<not> (f \<longrightarrow> g)) " by (rule BiGen)
 have  4: "\<turnstile> bi (\<not> (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> \<not> (f \<longrightarrow> g))
             \<longrightarrow>  
             bi (f \<longrightarrow> g) \<longrightarrow> bi (\<not>  g \<longrightarrow> \<not>  f) " by (rule BiContraPosImpDist)
 have  5: "\<turnstile> bi (f \<longrightarrow> g) \<longrightarrow> bi (\<not>  g \<longrightarrow> \<not>  f) " using 3 4 MP by blast
 have  6: "\<turnstile> bi (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> (bi  f) \<longrightarrow> (bi  g) " by (rule BiContraPosImpDist)
 from 5 6 show ?thesis  using lift_imp_trans by blast 
qed

lemma IfChopEqvRule:
 assumes "\<turnstile> f =  if\<^sub>i  (init w)  then  f1  else  f2 "
 shows   "\<turnstile> f; g =  if\<^sub>i  (init w)  then  (f1; g)  else  (f2; g) "
proof -
 have  1: "\<turnstile> f =  if\<^sub>i  (init w)  then  f1  else  f2" 
   using assms by auto
 hence 2: "\<turnstile> f = (((init w) \<and> f1) \<or>  ( (init (\<not> w)) \<and> f2)) "
   by (metis Initprop(2) ifthenelse_d_def inteq_reflection) 
 hence 3: "\<turnstile> f; g = (((init w) \<and> f1); g \<or>  (  (init (\<not> w)) \<and> f2); g) " 
   by (rule OrChopEqvRule)
 have  4: "\<turnstile> ((init w) \<and> f1); g = ((init w) \<and> (f1; g)) " 
   by (rule StateAndChop)
 have  5: "\<turnstile> (  (init (\<not> w)) \<and> f2); g =   ((init (\<not> w)) \<and> (f2; g)) " 
   by (rule StateAndChop)
 have  6: "\<turnstile> f; g = (((init w) \<and> f1; g)  \<or>  (  (init (\<not> w)) \<and> f2; g)) " 
   using "3" "4" "5" by fastforce
 from 6 show ?thesis
   by (metis Initprop(2) ifthenelse_d_def inteq_reflection) 
qed

lemma ChopOrEqvRule:
 assumes "\<turnstile> g = (g1 \<or>  g2) "
 shows   "\<turnstile> f;g = ((f;g1) \<or>  (f;g2)) "
proof -
 have  1: "\<turnstile> g = (g1 \<or>  g2) " using assms by auto
 hence 2: "\<turnstile> f; g = (f; (g1 \<or>  g2)) " by (rule RightChopEqvChop)
 have  3: "\<turnstile> f; (g1 \<or>  g2) = (f; g1 \<or>  f; g2) " by (rule ChopOrEqv)
 from 2 3 show ?thesis by fastforce
qed

lemma EmptyOrChopEqv: 
 "\<turnstile> ( empty  \<or>  f); g = (g \<or>  (f; g)) "
proof -
 have 1: "\<turnstile> ( empty  \<or>  f); g = (( empty ; g) \<or>  (f; g)) " by (rule OrChopEqv)
 have 2: "\<turnstile>  empty ; g = g " by (rule EmptyChop)
 from 1 2 show ?thesis by fastforce
qed

lemma EmptyOrNextChopEqv:
  "\<turnstile> ( empty  \<or>  \<circle> f); g = (g \<or>  \<circle>(f; g)) "
proof -
 have 1: "\<turnstile> ( empty  \<or>  \<circle> f); g = (g \<or>  ((\<circle> f); g)) " by (rule EmptyOrChopEqv)
 have 2: "\<turnstile> (\<circle> f); g = \<circle>(f; g) " by (rule NextChop)
 from 1 2 show ?thesis by fastforce
qed

lemma EmptyOrChopImpRule:
 assumes "\<turnstile>   f \<longrightarrow>  empty  \<or>  f1 " 
 shows   "\<turnstile> f; g \<longrightarrow> g \<or>  (f1; g) " 
proof -
 have  1: "\<turnstile> f \<longrightarrow>  empty  \<or>  f1 " using assms by auto
 hence 2: "\<turnstile> f; g \<longrightarrow> ( empty  \<or>  f1); g " by (rule LeftChopImpChop)
 have  3: "\<turnstile> ( empty  \<or>  f1); g = (g \<or>  (f1; g)) " by (rule EmptyOrChopEqv)
 from 2 3 show ?thesis by fastforce
qed

lemma EmptyOrChopEqvRule:
 assumes "\<turnstile>   f =  (empty  \<or>  f1) "
 shows   "\<turnstile> f; g = (g \<or>  (f1; g)) "
proof -
 have  1: "\<turnstile> f =  (empty  \<or>  f1)  " using assms by auto
 hence 2: "\<turnstile> f; g = (( empty  \<or>  f1); g) " by (rule LeftChopEqvChop)
 have  3: "\<turnstile> ( empty  \<or>  f1); g = (g \<or>  (f1; g))" by (rule EmptyOrChopEqv)
 from 2 3 show ?thesis by fastforce
qed

lemma EmptyOrNextChopImpRule:
 assumes "\<turnstile> f \<longrightarrow>  empty  \<or>  \<circle> f1 "
 shows   "\<turnstile> f; g \<longrightarrow> g \<or>  \<circle>(f1; g) "
proof -
 have  1: "\<turnstile> f \<longrightarrow>  empty  \<or>  \<circle> f1 " using assms by auto
 hence 2: "\<turnstile> f; g \<longrightarrow> ( empty  \<or>  \<circle> f1); g " by (rule LeftChopImpChop)
 have  3: "\<turnstile> ( empty  \<or>  \<circle> f1); g = (g \<or>  \<circle>(f1; g)) " by (rule EmptyOrNextChopEqv)
 from 2 3 show ?thesis by fastforce
qed

lemma EmptyOrNextChopEqvRule:
 assumes "\<turnstile> f =  (empty  \<or>  \<circle> f1) "
 shows   "\<turnstile> f; g = (g \<or>  \<circle>(f1; g)) "
proof -
 have  1: "\<turnstile> f =  (empty  \<or>  \<circle> f1) " using assms by auto
 hence 2: "\<turnstile> f; g = (( empty  \<or>  \<circle> f1); g) " by (rule LeftChopEqvChop)
 have  3: "\<turnstile> ( empty  \<or>  \<circle> f1); g = (g \<or>  \<circle>(f1; g)) " by (rule EmptyOrNextChopEqv)
 from 2 3 show ?thesis by fastforce
qed

lemma ChopEmptyOrImpRule:
 assumes "\<turnstile>  g \<longrightarrow>  empty  \<or>  g1 "
 shows "\<turnstile> f; g \<longrightarrow> f \<or>  (f; g1) "
proof -
 have  1: "\<turnstile> g \<longrightarrow>  empty  \<or>  g1 " using assms by auto
 hence 2: "\<turnstile> f; g \<longrightarrow> (f;  empty ) \<or>  (f; g1) " by (rule ChopOrImpRule)
 have  3: "\<turnstile> f;  empty  = f " by (rule ChopEmpty)
 from 2 3 show ?thesis by fastforce
qed

lemma StateAndEmptyImpBoxState:
 "\<turnstile> (init w) \<and> empty \<longrightarrow> \<box> (init w)"
 using BoxEqvAndEmptyOrNextBox by fastforce

lemma BoxEqvAndBox:
 "\<turnstile> \<box> f = (f \<and> \<box> f)"
 using BoxElim by fastforce

lemma NotBoxImpNotOrNotNextBox:
 "\<turnstile> \<not>( \<box> f) \<longrightarrow> \<not>f \<or>  \<not>( \<circle> (\<box> f) )"
proof -
  have 1: "\<turnstile> f \<and> (\<circle> (\<box> f)) \<longrightarrow> \<box> f" using BoxEqvAndEmptyOrNextBox by fastforce
  hence 2: "\<turnstile> \<not>( \<box> f) \<longrightarrow> \<not>(f \<and> (\<circle> (\<box> f)) )  " by fastforce
  have 3: "\<turnstile> (\<not>(f \<and> (\<circle> (\<box> f)) )) = (\<not>f \<or> \<not>( \<circle> (\<box> f) )   ) " by auto
  from 2 3 show ?thesis by auto
qed


lemma BoxStateChopBoxAndInfImpBox:
 "\<turnstile> \<box> (init w); \<box> (init w) \<and> inf \<longrightarrow> \<box> (init w)"
by (auto simp add: Valid_def itl_defs nfirst_eq_nnth_zero)
 (metis enat_ile le_add_diff_inverse linorder_le_cases min_def ndropn_nlength nfinite_ndropn_b 
    nlength_eq_enat_nfiniteD ntaken_nnth) 
  

lemma BoxStateChopBoxEqvBox:
 "\<turnstile>   \<box> (init w); \<box> (init w) = \<box> (init w) "
proof -
 have   1: "\<turnstile> (\<box> (init w)) = ((init w) \<and> ( empty  \<or>  \<circle>(\<box> (init w)))) " 
   by (rule BoxEqvAndEmptyOrNextBox)
 hence  2: "\<turnstile> (\<box> (init w); \<box> (init w)) =
               ((init w) \<and> (( empty  \<or>  \<circle>(\<box> (init w))); \<box> (init w))) " 
   by (metis StateAndChop inteq_reflection)
 have   3: "\<turnstile> (( empty  \<or>  \<circle>(\<box> (init w))); \<box> (init w)) =
              (\<box> (init w) \<or>  \<circle>(\<box> (init w); \<box> (init w))) " 
   by (rule EmptyOrNextChopEqv)
 have   4: "\<turnstile> (\<box> (init w); \<box> (init w)) = 
              ((init w) \<and> (\<box> (init w) \<or>  \<circle>(\<box> (init w); \<box> (init w)))) " 
   using "2" "3" by fastforce
 have   5: "\<turnstile> \<not> (\<box> (init w)) \<longrightarrow> \<not>  (init w) \<or>  \<not>( \<circle>(\<box> (init w))) " 
   by (rule NotBoxImpNotOrNotNextBox)
 have   6: "\<turnstile> (\<box> (init w); \<box> (init w)) \<and> \<not>( \<box> (init w)) \<longrightarrow>
              \<circle>(\<box> (init w); \<box> (init w)) \<and> \<not>( \<circle>(\<box> (init w)))  " 
   using "4" "5" by fastforce
 hence  7: "\<turnstile> \<box> (init w); \<box> (init w) \<and> finite \<longrightarrow> \<box> (init w) " 
   by (rule NextContra) 
 have   8: "\<turnstile> \<box> (init w); \<box> (init w) \<and> inf \<longrightarrow> \<box> (init w)"
   by (rule BoxStateChopBoxAndInfImpBox)
 have   9: "\<turnstile> \<box> (init w); \<box> (init w) \<and> (finite \<or> inf) \<longrightarrow> \<box> (init w)"
   using "7" "8"  by fastforce
 hence  10: "\<turnstile>  \<box> (init w); \<box> (init w) \<longrightarrow> \<box> (init w)"
   using FiniteOrInfinite by fastforce  
 have  11: "\<turnstile> \<box> (init w) = ((init w) \<and> \<box> (init w)) " 
   by (rule BoxEqvAndBox)
 have  12: "\<turnstile> empty ; \<box> (init w) = \<box> (init w)" 
   by (rule EmptyChop)
 have  13: "\<turnstile> ((init w) \<and>  empty ); \<box> (init w) = ((init w) \<and> ( empty ; \<box> (init w))) " 
   by (rule StateAndChop)
 have  14: "\<turnstile> \<box> (init w) = ((init w) \<and>  empty ); \<box> (init w) " 
   using "11" "12" "13" by fastforce
 have  15: "\<turnstile> (init w) \<and>  empty  \<longrightarrow> \<box> (init w) " 
   by (rule StateAndEmptyImpBoxState)
 hence 16: "\<turnstile> ((init w) \<and>  empty ); \<box> (init w) \<longrightarrow> \<box> (init w); \<box> (init w) " 
   by (rule LeftChopImpChop)
 have  17: "\<turnstile> \<box> (init w) \<longrightarrow> \<box> (init w); \<box> (init w)" 
   using "14" "16" by fastforce
 from 10 17 show ?thesis by fastforce
qed

lemma NotBoxStateImpBoxYieldsNotBox:
 "\<turnstile>   \<not>( \<box> (init w)) \<longrightarrow> (\<box> (init w)) yields (\<not>( \<box> (init w)))"
proof -
 have  1: "\<turnstile> \<box> (init w); \<box> (init w) = \<box> (init w) " by (rule BoxStateChopBoxEqvBox)
 have  2: "\<turnstile> \<box> (init w) = (\<not> \<not>( \<box> (init w))) " by auto
 hence 3: "\<turnstile> \<box> (init w); \<box> (init w) = \<box> (init w); (\<not> \<not>( \<box> (init w))) " by (rule RightChopEqvChop)
 have  4: "\<turnstile> \<not>( \<box> (init w)) \<longrightarrow> \<not> (\<box> (init w); (\<not> \<not> (\<box> (init w)))) " using "1" "3" by auto
 from 4 show ?thesis by (simp add: yields_d_def)
qed

lemma StateEqvBi:
 "\<turnstile>   (init w) = bi  (init w) "
proof -
  have 1: "\<turnstile> (init w) \<longrightarrow> bi  (init w) " by (rule StateImpBi)
  have 2: "\<turnstile> bi  (init w) \<longrightarrow> (init w) " by (rule BiElim)
  from 1 2 show ?thesis by fastforce
qed

lemma FiniteChopEqvDiamond:
 "\<turnstile>   finite; f = \<diamond> f"
by (simp add: sometimes_d_def)


subsection \<open>Properties of Da and Ba\<close>

lemma DaEqvDtDi:
 "\<turnstile>    da  f = \<diamond> (di  f) "
proof -
 have  1: "\<turnstile> finite; (f; #True) = finite; (f; #True) " by auto
 hence 2: "\<turnstile> finite; (f; #True) = finite;  di  f " by (simp add: di_d_def)
 have  3: "\<turnstile> finite;  di  f = \<diamond>( di  f)" by (rule FiniteChopEqvDiamond)
 have  4: "\<turnstile> finite; (f; #True) = \<diamond>( di  f) " using "2" "3" by fastforce
 from 4 show ?thesis by (simp add:da_d_def)
qed

lemma DaEqvDiDt:
 "\<turnstile>    da  f =  di (\<diamond> f) "
proof -
 have  1: "\<turnstile> finite; f = \<diamond> f " by (rule FiniteChopEqvDiamond)
 hence 2: "\<turnstile> (finite; f); #True = (\<diamond> f); #True " by (rule LeftChopEqvChop)
 hence 3: "\<turnstile> (finite; f); #True =  di( \<diamond> f) " by (simp add: di_d_def)
 have  4: "\<turnstile> finite; (f; #True) = (finite; f); #True   " by (rule ChopAssoc)
 have  5: "\<turnstile> finite; (f; #True) =  di (\<diamond> f) " using "3" "4" by fastforce
 from 5 show ?thesis by (simp add: da_d_def)
qed

lemma DtDiEqvDiDt:
 "\<turnstile>   \<diamond> (di  f) =  di (\<diamond> f) "
by (metis ChopAssoc di_d_def sometimes_d_def)

lemma DiamondNotEqvNotBox:
 "\<turnstile> \<diamond> (\<not>  f) = (\<not> (\<box> f))"
by (simp add: always_d_def)

lemma BaEqvBiBt:
 "\<turnstile>    ba f = bi( \<box> f) "
proof -
 have  1: "\<turnstile>  da (\<not>  f) =  di( \<diamond> (\<not>  f)) " by (rule DaEqvDiDt)
 have  2: "\<turnstile> \<diamond> (\<not>  f) = (\<not>( \<box> f)) " by (rule DiamondNotEqvNotBox)
 hence 3: "\<turnstile>  di (\<diamond>(\<not>  f)) =  di (\<not> (\<box> f)) " by (rule DiEqvDi)
 have  4: "\<turnstile>  da (\<not>  f) =  di (\<not>( \<box> f)) " using "1" "3" by fastforce
 hence 5: "\<turnstile> (\<not>  (da (\<not>  f))) = (\<not> ( di (\<not>( \<box> f)))) " by auto
 hence 6: "\<turnstile> (\<not> ( da (\<not>  f))) = bi( \<box> f) " by (simp add: bi_d_def)
 from 6 show ?thesis by (simp add: ba_d_def)
qed
 
lemma DiNotEqvNotBi:
 "\<turnstile>    di (\<not>  f) = (\<not>( bi  f)) "
proof -
 have 1: "\<turnstile> bi  f = (\<not> ( di (\<not>  f))) " by (simp add: bi_d_def)
 from 1 show ?thesis by auto
qed

lemma NotDiamondNotEqvBox:
 "\<turnstile> (\<not> (\<diamond>(\<not> f))) = \<box> f "
by (simp add: always_d_def)

lemma BaEqvBtBi:
 "\<turnstile>    ba  f = \<box> (bi  f) "
proof -
 have  1: "\<turnstile>  da (\<not>  f) = \<diamond> (di (\<not>  f)) " by (rule DaEqvDtDi)
 have  2: "\<turnstile>  di (\<not>  f) = (\<not> (bi  f)) " by (rule DiNotEqvNotBi)
 hence 3: "\<turnstile> \<diamond> (di (\<not>  f)) = \<diamond>(\<not> (bi  f)) " by (rule DiamondEqvDiamond)
 have  4: "\<turnstile> (\<not> (\<diamond>(\<not> (bi  f)))) = \<box>(bi  f)  " by (rule NotDiamondNotEqvBox)
 have  5: "\<turnstile> (\<not> ( da (\<not>  f))) = \<box>(bi  f) " using "1" "2" "3" "4" by fastforce
 from 5 show ?thesis by (simp add: ba_d_def)
qed

lemma BtBiEqvBiBt:
 "\<turnstile>   \<box> (bi  f) = bi( \<box> f) "
proof -
 have 1: "\<turnstile>    ba  f = \<box> (bi  f)  " by (rule BaEqvBtBi)
 have 2: "\<turnstile>    ba  f = bi( \<box> f) "  by (rule BaEqvBiBt)
 from 1 2 show ?thesis by fastforce
qed

lemma BoxStateEqvBaBoxState:
 "\<turnstile>   \<box> (init w) =  ba (\<box> (init w)) "
proof -
 have  1: "\<turnstile> (init w) = bi  (init w) " by (rule StateEqvBi)
 hence 2: "\<turnstile> \<box> (init w) = \<box> (bi  (init w)) " by (rule BoxEqvBox)
 have  3: "\<turnstile> \<box> (bi  (init w)) = bi( \<box> (init w))  " by (rule BtBiEqvBiBt)
 have  4: "\<turnstile> \<box> (init w) = \<box>(\<box> (init w)) " by (rule BoxEqvBoxBox)
 hence 5: "\<turnstile> bi( \<box> (init w)) = bi (\<box>(\<box> (init w))) " by (rule BiEqvBi)
 have  6: " \<turnstile>  ba( \<box> (init w)) = bi( \<box>(\<box> (init w))) "  by (rule BaEqvBiBt)
 from 2 3 5 6 show ?thesis  by fastforce 
qed

lemma BaImpBi:
 "\<turnstile>    ba  f \<longrightarrow> bi  f "
proof -
 have 1: "\<turnstile>  ba  f = \<box>(bi  f) " by (rule BaEqvBtBi)
 have 2: "\<turnstile> \<box>(bi  f) \<longrightarrow> bi  f " by (rule BoxElim)
 from 1 2 show ?thesis using lift_imp_trans by fastforce
qed

lemma BaImpBt:
 "\<turnstile>    ba  f \<longrightarrow> \<box> f "
proof -
 have 1: "\<turnstile>  ba  f = bi( \<box> f) " by (rule BaEqvBiBt)
 have 2: "\<turnstile> bi( \<box> f) \<longrightarrow> \<box> f " by (rule BiElim)
 from 1 2 show ?thesis using lift_imp_trans by fastforce
qed

lemma DiamondImpDa:
 "\<turnstile>   \<diamond> f \<longrightarrow>  da  f "
by (metis DiIntro DiamondImpDiamond da_d_def di_d_def sometimes_d_def)

lemma DiImpDa:
 "\<turnstile>    di  f \<longrightarrow>  da  f"
by (metis NowImpDiamond da_d_def di_d_def sometimes_d_def)

lemma BoxAndChopImport:
 "\<turnstile>   \<box> h \<and> f; g \<longrightarrow> f; (h \<and> g) " 
proof -
 have  1: "\<turnstile> h \<longrightarrow> g \<longrightarrow> (h\<and> g) " by auto
 hence 2: "\<turnstile> \<box> h \<longrightarrow> \<box>(g \<longrightarrow> (h\<and> g)) " by (rule ImpBoxRule)
 have  3: "\<turnstile> \<box>(g \<longrightarrow> (h\<and> g)) \<longrightarrow> f; g \<longrightarrow> f; (h\<and> g) " by (rule BoxChopImpChop)
 from 2 3 show ?thesis by fastforce
qed

lemma BaAndChopImport:
 " \<turnstile>    ba  f \<and> (g; g1) \<longrightarrow> (f \<and> g); (f \<and> g1) "
proof -
 have  1: "\<turnstile> ba  f \<longrightarrow> bi  f " by (rule BaImpBi)
 have  2: "\<turnstile> bi  f \<and> (g; g1) \<longrightarrow> (f \<and> g); g1 " by (rule BiAndChopImport)
 have  3: "\<turnstile> ba  f \<longrightarrow> \<box> f " by (rule BaImpBt)
 have  4: "\<turnstile> \<box> f \<and> (f \<and> g); g1 \<longrightarrow> (f \<and> g); (f \<and> g1) " by (rule BoxAndChopImport)
 from 1 2 3 4 show ?thesis by fastforce
qed

lemma ChopAndCommute:
 "\<turnstile>   f; (g \<and> g1) = f; (g1 \<and> g) "
proof -
 have 1: "\<turnstile> (g \<and> g1) = (g1 \<and> g)" by auto
 from 1 show ?thesis by (rule RightChopEqvChop)
qed

lemma ChopAndA: 
 "\<turnstile>   f; (g \<and> g1) \<longrightarrow> f; g "
proof -
 have 1: "\<turnstile> (g \<and> g1) \<longrightarrow>  g " by auto
 from 1 show ?thesis by (rule RightChopImpChop)
qed

lemma ChopAndB:
 "\<turnstile>   f; (g \<and> g1) \<longrightarrow> f; g1 "
proof -
 have 1: "\<turnstile> (g \<and> g1) \<longrightarrow>  g1 " by auto
from 1 show ?thesis by (rule RightChopImpChop)
qed
 
lemma BoxStateAndChopEqvChop:
 "\<turnstile>  (\<box> (init w) \<and> (f; g)) = ((\<box> (init w) \<and> f); (\<box> (init w) \<and> g)) "
proof -
 have  1: "\<turnstile> \<box> (init w) =  ba( \<box> (init w)) " 
   by (rule BoxStateEqvBaBoxState)
 have  2: "\<turnstile> ba( \<box> (init w)) \<and> (f; g) \<longrightarrow> (\<box> (init w) \<and> f); (\<box> (init w) \<and> g) " 
   by (rule BaAndChopImport)
 have  3: "\<turnstile> \<box> (init w) \<and> (f; g) \<longrightarrow> (\<box> (init w) \<and> f); (\<box> (init w) \<and> g) " 
   using "1" "2"  by fastforce 
 have 11: "\<turnstile> (\<box> (init w) \<and> f); (\<box> (init w) \<and> g) \<longrightarrow> (\<box> (init w)); (\<box> (init w) \<and> g) " 
   by (rule AndChopA)
 have 12: "\<turnstile> (\<box> (init w)); (\<box> (init w) \<and> g) \<longrightarrow> (\<box> (init w)); (\<box> (init w)) " 
   by (rule ChopAndA)
 have 13: "\<turnstile> (\<box> (init w)); (\<box> (init w)) = \<box> (init w) " 
   by (rule BoxStateChopBoxEqvBox)
 have 14: "\<turnstile> (\<box> (init w) \<and> f); (\<box> (init w) \<and> g) \<longrightarrow> f; (\<box> (init w) \<and> g) " 
   by (rule AndChopB)
 have 15: "\<turnstile> f; (\<box> (init w) \<and> g) \<longrightarrow> f; g " 
   by (rule ChopAndB)
 have 16: "\<turnstile> (\<box> (init w) \<and> f); (\<box> (init w) \<and> g) \<longrightarrow> \<box> (init w) \<and> (f; g) " 
   using  "11" "12" "13" "14" "15"  by fastforce
 from 3 16 show ?thesis by fastforce
qed

lemma DiEqvNotBiNot:
 " \<turnstile>    di  f = (\<not>( bi (\<not>  f))) "
proof -
 have  1: "\<turnstile> bi (\<not>  f) = (\<not> ( di (\<not> \<not>  f))) " by (simp add: bi_d_def)
 hence 2: "\<turnstile>  di (\<not> \<not>  f) = (\<not>( bi (\<not>  f))) " by auto
 have  3: "\<turnstile> f = (\<not> \<not>  f) " by auto
 hence 4: "\<turnstile>  di  f =  di  (\<not> \<not>  f) " by (rule DiEqvDi)
 from 2 4 show ?thesis by auto
qed

lemma ChopAndBoxImport:
 "\<turnstile>   f; g \<and> \<box> h \<longrightarrow> f; (g \<and> h) "
proof -
 have 1: "\<turnstile> \<box> h \<and> f; g \<longrightarrow> f; (h \<and> g) " by (rule BoxAndChopImport)
 have 2: "\<turnstile> f; (h \<and> g) = f; (g \<and> h) " by (rule ChopAndCommute)
 from 1 2 show ?thesis by fastforce
qed

lemma AndChopAndCommute:
 "\<turnstile>   (f \<and> g); (f1 \<and> g1) = (g \<and> f); (g1 \<and> f1) "
proof -
 have 1: "\<turnstile> (f \<and> g); (f1 \<and> g1) = (g \<and> f); (f1 \<and> g1) " by (rule AndChopCommute)
 have 2: "\<turnstile> (g \<and> f); (f1 \<and> g1) = (g \<and> f); (g1 \<and> f1) "  by (rule ChopAndCommute)
 from 1 2 show ?thesis by fastforce
qed

lemma ChopImpChop:
 assumes "\<turnstile> f \<longrightarrow> f1 " 
         "\<turnstile> g \<longrightarrow> g1 "
 shows   "\<turnstile> f;g \<longrightarrow> f1;g1 "
proof -
 have  1: "\<turnstile> f \<longrightarrow> f1 " using assms by auto
 hence 2: "\<turnstile> f; g \<longrightarrow> f1; g " by (rule LeftChopImpChop)
 have  3: "\<turnstile> g \<longrightarrow> g1 " using assms by auto
 hence 4: "\<turnstile> f1; g \<longrightarrow> f1; g1 " by (rule RightChopImpChop)
 from 2 4 show ?thesis by fastforce
qed

lemma ChopEqvChop:
 assumes "\<turnstile> f = f1" 
         "\<turnstile> g = g1 "
 shows   "\<turnstile> f;g = f1;g1 "
proof -
 have  1: "\<turnstile> f = f1 " using assms by auto
 hence 2: "\<turnstile> f; g = f1; g " by (rule LeftChopEqvChop)
 have  3: "\<turnstile> g = g1 " using assms by auto
 hence 4: "\<turnstile> f1; g = f1; g1 " by (rule RightChopEqvChop)
 from 2 4 show ?thesis by fastforce
qed

lemma BoxImpBoxImpBox:
 "\<turnstile> \<box> h \<longrightarrow> \<box>(g \<longrightarrow> \<box> h \<and> g ) "
proof -
 have 1: "\<turnstile> \<box> h \<longrightarrow> (g \<longrightarrow> \<box> h \<and> g ) " by auto
 hence 2: "\<turnstile> \<box>(\<box> h) \<longrightarrow> \<box>(g \<longrightarrow> \<box> h \<and> g ) " by (rule ImpBoxRule)
 have 3: "\<turnstile> \<box> h = \<box>(\<box>h) " by (rule BoxEqvBoxBox)
 from 2 3 show ?thesis by fastforce
qed

lemma BoxChopImpChopBox:
 "\<turnstile>   \<box> h \<longrightarrow> f; g \<longrightarrow> f; (\<box> h \<and> g) "
proof -
 have 1: "\<turnstile> \<box> h \<longrightarrow> \<box>(g \<longrightarrow> \<box> h \<and> g ) " by (rule BoxImpBoxImpBox)
 have 2: "\<turnstile> \<box>(g \<longrightarrow> \<box> h \<and> g ) \<longrightarrow> f; g \<longrightarrow> f; (\<box> h \<and> g) " by (rule BoxChopImpChop)
 from 1 2 show ?thesis by fastforce
qed

lemma NotChopEqvYieldsNot:
"\<turnstile>   (\<not> (f; g)) = f yields  (\<not>  g) "
proof -
 have  1: "\<turnstile> g = (\<not> \<not>  g) " by auto
 hence 2: "\<turnstile> f; g = f; (\<not> \<not>  g) " by (rule RightChopEqvChop)
 hence 3: "\<turnstile> (\<not> (f; g)) = (\<not> (f; (\<not> \<not>  g))) " by auto
 from 3 show ?thesis by (simp add: yields_d_def)
qed

lemma NotDiFalse:
 "\<turnstile>   \<not> ( di  #False) "
proof -
 have  1: "\<turnstile> (init #True) \<longrightarrow> bi (init #True) " by (rule StateImpBi)
 hence 2: "\<turnstile> #True \<longrightarrow> bi #True " by (simp add: BiGen) 
 have  3: "\<turnstile> #True" by auto
 have  4: "\<turnstile> bi #True" using "2" "3" MP by auto
 hence 5: "\<turnstile> \<not> ( di (\<not> #True)) " by (simp add: bi_d_def)
 have  6: "\<turnstile> (\<not> #True) =  #False " by auto
 hence 7: "\<turnstile>  di (\<not> #True) =  di  #False " by (rule DiEqvDi)
 from 5 7 show ?thesis by auto
qed

lemma StateAndEmptyChop:
 "\<turnstile>   ((init w) \<and>  empty ); f = ((init w) \<and> f) "
proof - 
 have 1: "\<turnstile> ((init w) \<and>  empty ); f = ((init w) \<and>  empty ; f) " by (rule StateAndChop)
 have 2: "\<turnstile>  empty ; f = f " by (rule EmptyChop)
 from 1 2 show ?thesis by fastforce
qed

lemma StateAndNextChop:
 "\<turnstile>   ((init w) \<and> \<circle> f); g = ((init w) \<and> \<circle>(f; g)) "
proof -
 have 1: "\<turnstile> ((init w) \<and> \<circle> f); g = ((init w) \<and> (\<circle> f); g) " by (rule StateAndChop)
 have 2: "\<turnstile> (\<circle> f); g = \<circle>(f; g) " by (rule NextChop)
 from 1 2 show ?thesis by fastforce
qed

lemma NextAndEqvNextAndNext:
 "\<turnstile> \<circle> (f \<and> g) = (\<circle> f \<and> \<circle> g)"
by (metis NextAndNextEqvNextRule int_eq lift_and_com)

lemma NextStateAndChop:
 "\<turnstile>   \<circle>(((init w) \<and> f); g) = (\<circle> (init w) \<and> \<circle>(f; g)) "
proof -
 have  1: "\<turnstile> ((init w) \<and> f); g = ((init w) \<and> f; g)  " by (rule StateAndChop)
 hence 2: "\<turnstile> \<circle>(((init w) \<and> f); g) = \<circle>((init w) \<and> f; g) " by (rule NextEqvNext)
 have  3: "\<turnstile> \<circle>((init w) \<and> f; g) = (\<circle> (init w) \<and> \<circle>(f; g)) " by (rule NextAndEqvNextAndNext)
 from 2 3 show ?thesis by fastforce
qed

lemma StateYieldsEqv:
 "\<turnstile>    ((init w) \<longrightarrow> (f yields  g)) = ((init w) \<and> f) yields  g " 
proof -
 have  1: "\<turnstile> ((init w) \<and> f); (\<not>  g) = ((init w) \<and> f; (\<not>  g))  " by (rule StateAndChop)
 hence 2: "\<turnstile> ((init w) \<longrightarrow> \<not> (f; (\<not>  g))) = (\<not> (((init w) \<and> f); (\<not>  g) )) " by auto
 from 2 show ?thesis by (simp add: yields_d_def)
qed

lemma StateAndDi:
 "\<turnstile>   ((init w) \<and>  di  f) =  di ((init w) \<and> f) "
proof - 
 have 1: "\<turnstile> ((init w) \<and> f); #True=  ((init w) \<and> f; #True)  " by (rule StateAndChop)
 from 1 show ?thesis by (metis di_d_def inteq_reflection)
qed

lemma DiNext: 
 "\<turnstile>    di( \<circle> f) = \<circle> (di  f) "
proof -
 have 1: "\<turnstile> (\<circle> f); #True = \<circle>(f; #True) " by (rule NextChop)
 from 1 show ?thesis by (simp add: di_d_def)
qed

lemma DiNextState:
 "\<turnstile>    di( \<circle> (init w)) = \<circle> (init w) "
proof -
 have  1: "\<turnstile>  di( \<circle> (init w)) = \<circle>( di  (init w)) " by (rule DiNext)
 have  2: "\<turnstile>  di  (init w) = (init w) " by (rule DiState)
 hence 3: "\<turnstile> \<circle>( di  (init w)) = \<circle> (init w) " by (rule NextEqvNext)
 from 1 3 show ?thesis by fastforce
qed

lemma StateImpBiGen:
 assumes "\<turnstile> (init w) \<longrightarrow> f "
 shows   "\<turnstile> (init w) \<longrightarrow> bi  f "
proof -
 have  1: "\<turnstile> (init w) \<longrightarrow> f " using assms by auto
 hence 2: "\<turnstile> \<not>  f \<longrightarrow> \<not>  (init w) " by auto
 hence 3: "\<turnstile>  di (\<not>  f) \<longrightarrow>  di (\<not>  (init w)) " by (rule DiImpDi)
 hence 4: "\<turnstile>  di (\<not>  f) \<longrightarrow>  di  (init (\<not>w)) " by (metis Initprop(2) inteq_reflection)
 have  5: "\<turnstile>  di (init (\<not> w)) =   (init (\<not> w)) " by (rule DiState)
 have  6: "\<turnstile>  di (\<not>  f) \<longrightarrow> \<not>  (init w) " using 4 5 using Initprop(2) by fastforce
 hence 7: "\<turnstile> (init w) \<longrightarrow> \<not> ( di (\<not>  f)) " by auto
 from 7 show ?thesis by (simp add: bi_d_def)
qed

lemma ChopAndNotChopImp:
 "\<turnstile>   f; g \<and> \<not> (f; g1) \<longrightarrow> f; (g \<and> \<not>  g1) "
proof -
 have  1: "\<turnstile> g \<longrightarrow> (g\<and> \<not>  g1) \<or>  g1 " by auto
 hence 2: "\<turnstile> f; g \<longrightarrow> f; ((g\<and> \<not>  g1) \<or>  g1) " by (rule RightChopImpChop)
 have  3: "\<turnstile> f; ((g\<and> \<not>  g1) \<or>  g1) \<longrightarrow> (f; (g\<and> \<not>  g1)) \<or>  (f; g1) " by (rule ChopOrImp)
 have  4: "\<turnstile> f; g \<longrightarrow> f; (g\<and> \<not>  g1) \<or>  f; g1 " using "2" "3" MP by fastforce
 from 4 show ?thesis by auto
qed

lemma ChopAndYieldsImp:
 "\<turnstile>   f; g \<and> f yields  g1 \<longrightarrow> f; (g \<and> g1) "
proof -
 have  1: "\<turnstile> g \<longrightarrow> (g\<and> g1) \<or>  \<not>  g1 " by auto
 hence 2: "\<turnstile> f; g \<longrightarrow> f; ((g\<and>   g1) \<or>  \<not> g1) "  by (rule RightChopImpChop)
 have  3: "\<turnstile> f; ((g\<and>   g1) \<or>  \<not> g1) \<longrightarrow> (f; (g\<and>   g1)) \<or>  (f; (\<not> g1)) " by (rule ChopOrImp)
 have  4: "\<turnstile> f; g \<longrightarrow> f; (g\<and>   g1) \<or>  f; (\<not> g1) " using "2" "3" MP by fastforce
 hence 5: "\<turnstile> f; g \<and> \<not>  (f; (\<not>  g1)) \<longrightarrow> f; (g \<and> g1) " by auto
 from 5 show ?thesis by (simp add: yields_d_def)
qed

lemma ChopAndYieldsMP:
 "\<turnstile>   f; g \<and> f yields (g\<longrightarrow> g1) \<longrightarrow> f; g1 "
proof -
 have  1: "\<turnstile> f; g \<and> f yields (g\<longrightarrow> g1) \<longrightarrow> f; (g \<and> (g \<longrightarrow> g1)) " by (rule ChopAndYieldsImp)
 have  2: "\<turnstile> g \<and> (g \<longrightarrow> g1) \<longrightarrow> g1 " by auto
 hence 3: "\<turnstile> f; (g \<and> (g \<longrightarrow> g1)) \<longrightarrow> f; g1 " by (rule RightChopImpChop)
 from 1 3 show ?thesis by fastforce
qed

lemma OrYieldsImp:
  "\<turnstile>   (f \<or>  f1) yields  g = ((f yields  g) \<and> (f1 yields  g)) "
proof -
 have  1: "\<turnstile> ((f\<or>  f1); (\<not>  g)) = ((f; (\<not>  g)) \<or>  (f1; (\<not>  g))) " by (rule OrChopEqv)
 hence 2: "\<turnstile> (\<not> ((f\<or>  f1); (\<not>  g))) = (\<not> (f; (\<not>  g)) \<and>  \<not>(f1; (\<not>  g))) " by auto
 from 2 show ?thesis by (simp add: yields_d_def)
qed

lemma LeftYieldsImpYields:
 assumes "\<turnstile> f \<longrightarrow> f1 "
 shows   "\<turnstile> (f1 yields  g) \<longrightarrow> (f yields  g) "
proof -
 have  1: "\<turnstile> f \<longrightarrow> f1 " using assms by auto
 hence 2: "\<turnstile> f; (\<not>  g) \<longrightarrow> f1; (\<not>  g)  " by (rule LeftChopImpChop)
 hence 3: "\<turnstile> \<not> (f1; (\<not>  g)) \<longrightarrow> \<not> (f; (\<not>  g)) " by auto
 from 3 show ?thesis by (simp add: yields_d_def)
qed

lemma LeftYieldsEqvYields:
 assumes "\<turnstile>  f = f1 "
 shows "\<turnstile> (f yields  g) = (f1 yields  g) "
proof -
 have  1: "\<turnstile> f = f1 " using assms by auto
 hence 2: "\<turnstile> f; (\<not>  g) = f1; (\<not>  g) " by (rule LeftChopEqvChop)
 hence 3: "\<turnstile> (\<not> (f; (\<not>  g))) = (\<not> (f1; (\<not>  g))) " by auto
 from 3 show ?thesis by (simp add: yields_d_def)
qed


subsection \<open>Properties of Fin\<close>

lemma FinEqvTrueChopAndEmpty:
  " \<turnstile>  fin f = #True;(f \<and> empty)"
proof -
 have 1: "\<turnstile> fin f = \<box>(empty \<longrightarrow> f)" 
   by (simp add: fin_d_def)
 have 2: "\<turnstile> \<box>(empty \<longrightarrow> f) = (\<not>(\<diamond>(\<not>(empty \<longrightarrow> f ) ) ))" 
   by (simp add: always_d_def)
 have 3: "\<turnstile> (\<not>(empty \<longrightarrow> f )) = (\<not> f \<and> empty)  " 
   by auto
 hence 4: "\<turnstile> \<diamond>(\<not>(empty \<longrightarrow> f )) = \<diamond>(\<not> f \<and> empty) " 
   using DiamondEqvDiamond by blast
 hence 5: "\<turnstile> \<not>(\<diamond>(\<not>(empty \<longrightarrow> f ))) = (\<not>(\<diamond>(\<not> f \<and> empty)))" 
   by auto
 have 51: "\<turnstile> finite;((\<not> f \<and> empty) \<and> finite)  \<longrightarrow> finite ;  (\<not> f \<and> empty)"
   by (simp add: ChopAndA) 
 have 52: "\<turnstile> (#True;(\<not> f \<and> empty) \<and> finite) \<longrightarrow> finite ;  (\<not> f \<and> empty)"
   by (metis "51" TrueChopAndFiniteEqvAndFiniteChopFinite int_eq) 
 have 53: "\<turnstile> \<not> (#True;(f \<and> empty))  \<longrightarrow> finite ;  (\<not> f \<and> empty)"
   by (metis "52" Finprop(5) int_eq) 
 have 54: " \<turnstile> \<not> finite;(\<not> f \<and> empty) \<longrightarrow> #True;(f \<and> empty)" 
   using 53 by auto
 have 6: "\<turnstile> (\<not>(\<diamond>(\<not> f \<and> empty))) \<longrightarrow> #True;(f \<and> empty)"
   unfolding sometimes_d_def using 54 by auto
 have 61:  "\<turnstile> \<not> f \<and> empty \<longrightarrow> finite"
   by (metis ChopAndB DiamondEmptyEqvFinite NowImpDiamond inteq_reflection lift_imp_trans sometimes_d_def)
 have 62: "\<turnstile>(\<not> #True;(f \<and> empty)) = finite;(\<not> f \<and> empty)"
   using 61 
   by (metis (no_types) Finprop(5) Prop10 TrueChopAndFiniteEqvAndFiniteChopFinite inteq_reflection)
 have 7: "\<turnstile> #True;(f \<and> empty) \<longrightarrow> (\<not>(\<diamond>(\<not> f \<and> empty)))"
   unfolding sometimes_d_def using TrueChopAndFiniteEqvAndFiniteChopFinite[of "LIFT(f \<and> empty)"]
   using "62" by auto
 have 8: "\<turnstile> (\<not>(\<diamond>(\<not> f \<and> empty))) = #True;(f \<and> empty)" 
   by (simp add: "6" "7" int_iffI)
 from 1 2 5 8 show ?thesis by fastforce
qed


lemma DiamondFin:
 "\<turnstile> \<diamond>(fin w) = fin w"
by (metis (no_types, lifting) ChopAssoc ChopOrEqv FinEqvTrueChopAndEmpty FiniteChopFiniteEqvFinite
           FiniteChopInfEqvInf FiniteOrInfinite int_eq_true inteq_reflection sometimes_d_def)

lemma FiniteChopFinExportA:
 "\<turnstile> (f \<and> finite);(g \<and> fin w) \<longrightarrow> fin w"
using DiamondFin 
by (metis ChopAndB FiniteChopImpDiamond inteq_reflection lift_imp_trans)

lemma FinImpBox:
 "\<turnstile> fin w \<longrightarrow> \<box>(fin w)" 
by (metis BoxImpBoxBox fin_d_def)

lemma FinAndChopImport:
 "\<turnstile> (fin w) \<and> (f;g) \<longrightarrow> f;((fin w) \<and> g)"
proof -
 have 1: "\<turnstile> fin w \<longrightarrow> \<box>(fin w)" by (rule FinImpBox)
 hence 2: "\<turnstile> fin w \<and> f;g \<longrightarrow> \<box>(fin w) \<and> (f;g)" by auto
 have 3: "\<turnstile> \<box>(fin w) \<and> (f;g) \<longrightarrow> f;((fin w) \<and> g)" using BoxAndChopImport by blast
 from 2 3 show ?thesis using MP by fastforce
qed

lemma FinAndChop:
 "\<turnstile> ((f \<and> finite);(g \<and> fin w)) = (fin w \<and> (f \<and> finite);g)  "
using FinAndChopImport FiniteChopFinExportA ChopAndA ChopAndCommute 
by fastforce

lemma ChopAndEmptyEqvEmptyChopEmpty:
 "\<turnstile> ((f;g) \<and> empty) = (f \<and> empty);(g \<and> empty) "
by (auto simp: itl_defs min_absorb1)

lemma FinAndEmpty:
 "\<turnstile> ((fin w) \<and> empty) = (w \<and> empty)"
proof -
 have 1: "\<turnstile> ((fin w) \<and> empty) = (#True;(w \<and> empty) \<and> empty)" 
   using FinEqvTrueChopAndEmpty by fastforce
 have 2: "\<turnstile> (#True;(w \<and> empty) \<and> empty) = ((#True \<and> empty);(w \<and> empty))" 
   using ChopAndEmptyEqvEmptyChopEmpty[of "LIFT(#True)" "LIFT(w \<and> empty)"] 
   by (metis AndChopA ChopAndA ChopAndEmptyEqvEmptyChopEmpty Prop11 Prop12 int_eq) 
 have 3: "\<turnstile> (#True \<and> empty);(w \<and> empty) = (empty;(w \<and> empty))" 
   using LeftChopEqvChop by fastforce
 have 4: "\<turnstile> (empty;(w \<and> empty)) = (w \<and> empty)" 
   using EmptyChop by blast
 from 1 2 3 4 show ?thesis by fastforce
qed

lemma AndFinEqvChopAndEmpty:
 "\<turnstile>   ((f \<and> finite) \<and> fin g) = (f \<and> finite); (g \<and>  empty ) "
proof -
 have 1: "\<turnstile> ((f \<and> finite) \<and> fin g) = ((f \<and> finite) ;empty \<and> fin g)" 
   using ChopEmpty by (metis inteq_reflection) 
 have 2: "\<turnstile> (fin g \<and> (f \<and> finite);empty) = ((f \<and> finite);(empty \<and> fin g))  " 
   using FinAndChop by fastforce
 have 3: "\<turnstile> (empty \<and> fin g) =  (fin g \<and> empty)" 
   by auto
 have 4: "\<turnstile> (fin g \<and> empty) = (g \<and> empty)" 
   using FinAndEmpty by metis
 have 5: "\<turnstile> (empty \<and> fin g) = (g \<and> empty)" 
   using "3" "4" by auto
 hence 6: "\<turnstile> (f \<and> finite);(empty \<and> fin g) = (f \<and> finite);(g \<and> empty)" 
   using RightChopEqvChop by blast
 from 1 2 5 show ?thesis by (metis inteq_reflection lift_and_com)
qed

lemma AndFinEqvChopStateAndEmpty:
 "\<turnstile>   ((f \<and> finite) \<and>  fin  (init w)) = (f \<and> finite); ((init w) \<and>  empty ) "
using AndFinEqvChopAndEmpty by blast

lemma FinStateEqvStateAndEmptyOrNextFinState:
 "\<turnstile>  fin  (init w) = (((init w) \<and>  empty ) \<or>  \<circle>( fin  (init w))) "
proof -
 have 1: "\<turnstile> fin (init w) = \<box>( empty \<longrightarrow> init w)" 
   by (simp add: fin_d_def)
 have 2 : "\<turnstile> \<box>(empty \<longrightarrow> init w) = 
               ((empty \<longrightarrow> init w) \<and> wnext (\<box> (empty \<longrightarrow> init w))) " 
   by (rule BoxEqvAndWnextBox)
 have 3: "\<turnstile> fin (init w) =  ((empty \<longrightarrow> init w) \<and> wnext (fin (init w))) " 
   using "1" "2" by (simp add: fin_d_def) 
 have 4: "\<turnstile> wnext (fin (init w)) = (empty \<or> \<circle> (fin (init w))) " 
   by (rule WnextEqvEmptyOrNext)
 have 5: "\<turnstile> fin (init w) =  ((empty \<longrightarrow> init w) \<and> (empty \<or> \<circle> (fin (init w))))" 
   using "3" "4" by fastforce
 have 6: "\<turnstile> ((empty \<longrightarrow> init w) \<and> (empty \<or> \<circle> (fin (init w)))) =
            (((empty \<longrightarrow> init w) \<and> empty) \<or> ((empty \<longrightarrow> init w) \<and> \<circle> (fin (init w))))" 
   by auto
 have 7: "\<turnstile> ((empty \<longrightarrow> init w) \<and> empty) = ((init w) \<and> empty)" 
   by auto
 have 8: "\<turnstile> ((empty \<longrightarrow> init w) \<and> \<circle> (fin (init w))) =  \<circle> (fin (init w))"
   by (metis (no_types, lifting) "5" DiamondFin NextDiamondImpDiamond Prop10 Prop12 int_eq 
       lift_and_com) 
 have 9: "\<turnstile> (((empty \<longrightarrow> init w) \<and> empty) \<or> ((empty \<longrightarrow> init w) \<and> \<circle> (fin (init w)))) =
            ((init w) \<and>  empty ) \<or>  \<circle>( fin  (init w))" 
   using "7" "8" by auto
 from 5 6 8 9 show ?thesis  by fastforce
qed

lemma FinChopEqvOr:
 " \<turnstile>   ( fin  (init w)); f = (((init w) \<and> f) \<or>  \<circle>(( fin  (init w)); f)) "
proof -
 have  1: "\<turnstile>  fin  (init w) = (((init w) \<and>  empty ) \<or>  \<circle>( fin  (init w))) " 
   by (rule FinStateEqvStateAndEmptyOrNextFinState)
 hence 2: "\<turnstile> ( fin  (init w)); f = (((init w) \<and>  empty )\<or>  \<circle>( fin  (init w))); f" 
   by (rule LeftChopEqvChop)
 have  3: "\<turnstile> (((init w) \<and>  empty )\<or>  \<circle> (fin  (init w))); f
            = (((init w) \<and>  empty ); f \<or>  (\<circle> (fin  (init w))); f) " 
   by (rule OrChopEqv)
 have  4: "\<turnstile> ((init w) \<and>  empty ); f = ((init w) \<and> f) " 
   by (rule StateAndEmptyChop)
 have  5: "\<turnstile> (\<circle> (fin  (init w))); f = \<circle>(( fin  (init w)); f) " 
   by (rule NextChop)
 from 2 3 4 5 show ?thesis by fastforce
qed

lemma FinChopEqvDiamond:
 "\<turnstile>   ( fin  (init w) \<and> finite); f = \<diamond> ((init w) \<and> f) "
proof -
 have  1: "\<turnstile> ( fin  (init w) \<and> finite) = (finite;((init w) \<and> empty))"
   by (metis AndFinEqvChopAndEmpty int_simps(17) inteq_reflection lift_and_com) 
 hence 2: "\<turnstile> (fin  (init w) \<and> finite);f = (finite;((init w) \<and> empty));f" 
   by (rule LeftChopEqvChop)
 have  3: "\<turnstile> finite;(( (init w) \<and> empty);f)  = (finite;((init w) \<and> empty));f " 
   by (rule ChopAssoc)
 have  4: "\<turnstile> finite;(( (init w) \<and> empty);f)= \<diamond> ( ( (init w) \<and> empty);f) " 
   by (simp add: sometimes_d_def)
 have  5: "\<turnstile> ( (init w) \<and> empty);f = ((init w) \<and> f) " 
   using StateAndEmptyChop by blast 
 hence 6: "\<turnstile> \<diamond> ( ( (init w) \<and> empty);f) = \<diamond> ( (init w) \<and> f) " 
   by (rule DiamondEqvDiamond)
 from 2 3 4 6 show ?thesis by fastforce
qed

lemma NotDiamondAndNot:
 "\<turnstile> \<not>( \<diamond> ( f \<and> \<not> f))"
proof -
 have 1: "\<turnstile> (\<not>( \<diamond> ( f \<and> \<not> f))) = \<box>(\<not>(f \<and> \<not>f)) " using NotDiamondNotEqvBox by fastforce
 have 2: "\<turnstile> \<not>(f \<and> \<not>f)" by simp
 have 3: "\<turnstile> \<box>(\<not>(f \<and> \<not>f))" using "2" by (simp add: BoxGen)
 from 1 3 show ?thesis by fastforce
qed

lemma FinYields:
 "\<turnstile>    ( fin  (init w) \<and> finite)  yields  (init w) "
proof -
 have 1: "\<turnstile> (fin (init w) \<and> finite); (\<not>(init w)) = \<diamond>((init w) \<and> \<not>(init w)) " 
   by (rule FinChopEqvDiamond)
 have 2: "\<turnstile>  \<not>( \<diamond>((init w) \<and> \<not>  (init w))) " 
   by (rule NotDiamondAndNot)
 have 3: "\<turnstile> \<not> (( fin  (init w) \<and> finite); (\<not>  (init w))) " 
   using "1" "2" by fastforce
 from 3 show ?thesis by (simp add: yields_d_def)
qed

lemma ImpAndFinStateOrFinNotState:
 "\<turnstile> f \<longrightarrow> (f \<and> fin (init w)) \<or> (f \<and> fin (\<not> (init w))) "
by (simp add: itl_defs Valid_def)

lemma AndFinChopEqvStateAndChop:
 "\<turnstile>   ((f \<and> finite) \<and> fin  (init w)); g = (f \<and> finite); ((init w) \<and> g) "
proof -
 have   1: "\<turnstile> ( fin  (init w) \<and> finite) yields  (init w) " 
   by (rule FinYields)
 have   2: "\<turnstile> (f \<and> finite) \<and>  fin  (init w) \<longrightarrow>  fin  (init w) " 
   by auto
 hence  3: "\<turnstile> ( fin  (init w) \<and> finite) yields  (init w) \<longrightarrow> 
              ((f \<and> finite) \<and>  fin  (init w)) yields  (init w) " 
   using LeftYieldsImpYields 
   by (metis AndFinEqvChopAndEmpty Prop11 Prop12 inteq_reflection)
 have   4: "\<turnstile> ((f \<and> finite) \<and>  fin  (init w)) yields  (init w) " 
   using "1" "3" MP by fastforce
 have   5: "\<turnstile> ((f \<and> finite) \<and>  fin  (init w)); g \<and> ((f \<and> finite) \<and>  fin  (init w)) yields (init w)
              \<longrightarrow> ((f \<and> finite) \<and>  fin  (init w)); (g \<and> (init w)) " 
   by (rule ChopAndYieldsImp)
 have   6: "\<turnstile> ((f \<and> finite) \<and>  fin  (init w)); g \<longrightarrow> 
               ((f \<and> finite) \<and>  fin  (init w)); (g \<and> (init w)) " 
   using "4" "5" by fastforce
 have   7: "\<turnstile> ((f \<and> finite) \<and>  fin  (init w)); (g \<and> (init w)) \<longrightarrow> (f \<and> finite); (g \<and> (init w)) " 
   by (rule AndChopA)
 have   8: "\<turnstile> g \<and> (init w) \<longrightarrow> (init w) \<and> g " 
   by auto
 hence  9: "\<turnstile> (f \<and> finite); (g \<and> (init w)) \<longrightarrow> (f \<and> finite); ((init w) \<and> g) " 
   by (rule RightChopImpChop)
 have  10: "\<turnstile> ((f \<and> finite) \<and>  fin  (init w)); g \<longrightarrow> (f \<and> finite); ((init w) \<and> g) " 
   using "6" "7" "9" by fastforce
 have  11: "\<turnstile> (f \<and> finite) \<longrightarrow> ((f \<and> finite) \<and> fin (init w)) \<or> ((f \<and> finite) \<and> fin (\<not> (init w))) " 
   using ImpAndFinStateOrFinNotState by blast
 hence 12: "\<turnstile> (f \<and> finite); ((init w) \<and> g) \<longrightarrow>
              (((f \<and> finite) \<and>  fin  (init w)) \<or> 
                ((finite \<and> f) \<and> fin (\<not> (init w)) )); ((init w) \<and> g) " 
   using LeftChopImpChop 
   by (metis inteq_reflection lift_and_com)
 have  13: "\<turnstile> (((f \<and> finite) \<and>  fin  (init w)) \<or> ((f \<and> finite) \<and> fin (\<not> (init w)))); ((init w) \<and> g)
              =  
              (((f \<and> finite) \<and>  fin  (init w)); 
               ((init w) \<and> g) \<or>  ((f \<and> finite) \<and> fin (\<not>  (init w))); ((init w) \<and> g)) " 
   by (rule OrChopEqv)
 have  14: "\<turnstile> ( (f \<and> finite) \<and> fin   (init (\<not> w))); ((init w) \<and> g) \<longrightarrow> 
               \<diamond>(  (init (\<not> w)) \<and> ((init w) \<and> g)) " 
   using FinChopEqvDiamond
   by (metis AndFinEqvChopAndEmpty ChopEmpty FiniteChopImpDiamond LeftChopImpChop int_eq) 
 have 141: "\<turnstile> \<not>( \<diamond>(  (init (\<not> w)) \<and> ((init w) \<and> g))) \<longrightarrow> 
              \<not> ( ( (f \<and> finite) \<and> fin   (init (\<not> w))); ((init w) \<and> g))" 
   using "14" by fastforce
 have 142: "\<turnstile> ((init (\<not> w)) \<and> ((init w) \<and> g)) = #False"
   using Initprop(2) by fastforce
 have  15: "\<turnstile> \<not>( \<diamond>(  (init (\<not> w)) \<and> ((init w) \<and> g))) "
   by (metis "142" NotDiamondAndNot int_simps(21) inteq_reflection) 
 have 151: "\<turnstile> \<not> ( ( (f \<and> finite) \<and> fin   (init (\<not> w))); ((init w) \<and> g))" 
   using "15" "141" by fastforce
 have 1511: "\<turnstile> ( (f \<and> finite) \<and> fin (\<not>  (init w))); ((init w) \<and> g) \<longrightarrow> #False" 
   using "151"  by (metis Initprop(2) int_simps(14) inteq_reflection) 
 have 152: "\<turnstile> ((f \<and> finite) \<and>  fin  (init w)); 
                ((init w) \<and> g) \<or>  ( (f \<and> finite) \<and> fin (\<not>  (init w))); ((init w) \<and> g) \<longrightarrow>
              ((f \<and> finite)\<and>  fin  (init w)); ((init w) \<and> g)" 
   using "1511" by fastforce
 have  16: "\<turnstile> (f \<and> finite); ((init w) \<and> g) \<longrightarrow> ((f \<and> finite)\<and>  fin  (init w)); ((init w) \<and> g) " 
   using "12" "13" "152"
   proof -
    have "\<turnstile> (f \<and> finite);(init w \<and> g) \<longrightarrow> 
                ((f \<and> finite) \<and> fin (init w) \<or> (f \<and> finite) \<and> fin (\<not> init w));(init w \<and> g)"
      by (metis "12" inteq_reflection lift_and_com)
    then show ?thesis
      using "13" "152" by fastforce
   qed 
 have  17: "\<turnstile> ((f \<and> finite)\<and>  fin  (init w)); ((init w) \<and> g) \<longrightarrow> ((f \<and> finite) \<and> fin (init w)); g " 
   by (rule ChopAndB)
 have  18: "\<turnstile> (f \<and> finite); ((init w) \<and> g) \<longrightarrow> ((f \<and> finite) \<and>  fin  (init w)); g " 
   using "16" "17" by fastforce
 from 10 18 show ?thesis by fastforce
qed

lemma DiAndFinEqvChopState:
 "\<turnstile>    di ((f \<and> finite) \<and>  fin  (init w)) = (f \<and> finite); (init w) "
proof -
 have  1: "\<turnstile> ((f \<and> finite) \<and> fin(init w)); #True = (f \<and> finite);((init w) \<and> #True) " 
   by (rule AndFinChopEqvStateAndChop)
 have  2: "\<turnstile> ((init w) \<and> #True) = (init w) " 
   by auto
 hence 3: "\<turnstile> ((f \<and> finite); ((init w) \<and> #True)) = ((f \<and> finite); (init w))" 
   by (rule RightChopEqvChop)
 have  4: "\<turnstile> ((f \<and> finite) \<and>  fin  (init w)); #True = (f \<and> finite); (init w) " 
   using "1" "3" by auto
 from 4 show ?thesis by (simp add: di_d_def)
qed

lemma FinNotStateEqvNotFinState:
 "\<turnstile>   (\<not>( fin   (init w)) \<and> finite ) = (fin   (init (\<not> w)) \<and> finite  )"
using FinEqvTrueChopAndEmpty Finprop(4) Initprop(2) FiniteImpAnd by (metis inteq_reflection)

lemma BiImpFinEqvYieldsState:
 "\<turnstile>   bi (f \<and> finite \<longrightarrow>  fin  (init w)) = (f  \<and> finite)yields  (init w) "
proof -
 have  1: "\<turnstile>  di ((f \<and> finite) \<and> fin (init (\<not> w))) = (f \<and> finite);   (init (\<not> w)) " 
   by (rule DiAndFinEqvChopState)
 have  2: "\<turnstile> ((f \<and> finite) \<and> fin(init (\<not> w)))  = ((f \<and> finite) \<and> \<not>(fin(init w))) " 
   using FinNotStateEqvNotFinState by fastforce
 have  3: "\<turnstile> ((f \<and> finite) \<and> \<not> (fin(init w))) = (\<not> (f \<and> finite \<longrightarrow>  fin  (init w))) " 
   by auto
 have  4: "\<turnstile> ((f \<and> finite) \<and> fin(init  (\<not> w))) = (\<not> (f \<and> finite\<longrightarrow>  fin(init w)))" 
   using "2" "3" by fastforce
 hence 5: "\<turnstile>  di ((f \<and> finite) \<and>  fin   (init (\<not> w))) =  di (\<not> (f \<and> finite\<longrightarrow> fin(init w))) " 
   by (rule DiEqvDi)
 have  6: "\<turnstile>  di (\<not> (f \<and> finite \<longrightarrow>  fin  (init w))) = (\<not>( bi (f \<and> finite\<longrightarrow> fin(init w))))" 
   by (rule DiNotEqvNotBi)
 have  7: "\<turnstile> \<not> (bi (f \<and> finite \<longrightarrow>  fin  (init w))) = (f \<and> finite);(init (\<not> w)) " 
   using "1"  "5" "6" Initprop by fastforce
 hence 8: "\<turnstile> bi (f \<and> finite \<longrightarrow>  fin  (init w)) = (\<not> ((f \<and> finite); (\<not>  (init w)))) " 
   by (metis Initprop(2) int_eq int_simps(7))
 from 8 show ?thesis by (simp add: yields_d_def)
qed

lemma StateImpYields:
 assumes " \<turnstile>   (init w) \<and> f \<and> finite \<longrightarrow>  fin  (init w1) "
 shows   "\<turnstile> (init w) \<longrightarrow> ((f \<and> finite) yields  (init w1)) "
proof -
 have  1: "\<turnstile> (init w) \<and> f \<and> finite \<longrightarrow>  fin  (init w1) " 
   using assms by auto
 hence 2: "\<turnstile> (init w) \<longrightarrow> (f \<and> finite \<longrightarrow>  fin  (init w1)) " 
   by auto
 hence 3: "\<turnstile> (init w) \<longrightarrow> bi (f \<and> finite \<longrightarrow>  fin  (init w1)) " 
   by (rule StateImpBiGen)
 have  4: "\<turnstile> bi (f \<and> finite \<longrightarrow>  fin  (init w1)) = (f \<and> finite)yields  (init w1) " 
   by (rule BiImpFinEqvYieldsState)
 from 3 4 show ?thesis by fastforce
qed

lemma StateAndYieldsImpYields:
 assumes "\<turnstile>   (init w) \<and> f \<longrightarrow> f1 "
 shows   "\<turnstile> (init w) \<and> (f1 yields  g) \<longrightarrow> (f yields  g) "
proof -
 have  1: "\<turnstile> (init w) \<and> f \<longrightarrow> f1 " using assms by auto
 hence 2: "\<turnstile> (init w) \<and> (f; (\<not>  g)) \<longrightarrow> f1; (\<not>  g) " by (rule StateAndChopImpChopRule)
 hence 3: "\<turnstile> (init w) \<and> \<not> (f1; (\<not>  g)) \<longrightarrow> \<not> (f; (\<not>  g)) " by auto
 from 3 show ?thesis by (simp add: yields_d_def)
qed

lemma AndYieldsA:
 "\<turnstile>   f yields  g \<longrightarrow> (f \<and> f1) yields  g "
proof -
 have 1: "\<turnstile> f \<and> f1 \<longrightarrow> f " by auto
 from 1 show ?thesis by (rule LeftYieldsImpYields)
qed

lemma AndYieldsB:
 " \<turnstile>   f1 yields  g \<longrightarrow> (f \<and> f1) yields  g "
proof -
 have 1: "\<turnstile> f \<and> f1 \<longrightarrow> f1 " by auto
 from 1 show ?thesis by (rule LeftYieldsImpYields)
qed

lemma RightYieldsImpYields:
 assumes "\<turnstile>   g\<longrightarrow> g1 "
 shows   "\<turnstile> (f yields  g) \<longrightarrow> (f yields  g1) "
proof -
 have  1: "\<turnstile> g \<longrightarrow> g1 " using assms by auto
 hence 2: " \<turnstile> \<not>  g1 \<longrightarrow> \<not>  g" by auto
 hence 3: "\<turnstile> f; (\<not>  g1) \<longrightarrow> f; (\<not>  g) " by (rule RightChopImpChop)
 hence 4: "\<turnstile> \<not> (f; (\<not>  g)) \<longrightarrow> \<not> (f; (\<not>  g1)) " by auto
 from 4 show ?thesis by (simp add: yields_d_def)
qed

lemma RightYieldsEqvYields:
 assumes "\<turnstile>   g = g1 "
 shows   "\<turnstile> (f yields  g) = (f yields  g1) "
proof -
 have  1: "\<turnstile> g = g1 " using assms by auto
 hence 2: "\<turnstile> (\<not>  g) = (\<not>  g1) " by auto
 hence 3: "\<turnstile> f; (\<not>  g) = f; (\<not>  g1) " by (rule RightChopEqvChop)
 hence 4: "\<turnstile> (\<not> (f; (\<not>  g))) = (\<not> (f; (\<not>  g1))) " by auto
 from 4 show ?thesis by (simp add: yields_d_def)
qed

lemma BoxImpYields:
 "\<turnstile>   \<box> g \<longrightarrow> (f \<and> finite) yields  g "
proof -
 have  1: "\<turnstile> (f \<and> finite); (\<not>  g) \<longrightarrow> \<diamond>(\<not>  g) " by (rule FiniteChopImpDiamond)
 hence 2: "\<turnstile> \<not> (\<diamond>(\<not>  g)) \<longrightarrow> \<not> ((f \<and> finite); (\<not>  g)) " by auto
 from 2 show ?thesis by (simp add: yields_d_def always_d_def)
qed

lemma BoxEqvFiniteYields:
 "\<turnstile>   \<box> f = finite yields  f "
proof -
 have  1: "\<turnstile> finite; (\<not>  f) = \<diamond> (\<not>  f) " by (rule FiniteChopEqvDiamond)
 hence 2: "\<turnstile> (\<not> (finite; (\<not>  f))) = (\<not>( \<diamond> (\<not>  f))) " by auto
 have  3: "\<turnstile> \<box> f = (\<not> ( \<diamond> (\<not>  f))) " by (simp add: always_d_def)
 have  4: "\<turnstile> \<box> f = (\<not> (finite; (\<not>  f))) " using "2" "3" by fastforce
 from 4 show ?thesis by (simp add: yields_d_def)
qed

lemma YieldsGen:
 assumes "\<turnstile>   g "
 shows   "\<turnstile> (f \<and> finite) yields  g "
proof -
  have  1: "\<turnstile> g " using assms by auto
  hence 2: "\<turnstile> \<box> g " by (rule BoxGen)
  have  3: "\<turnstile> \<box> g \<longrightarrow> (f \<and> finite) yields  g " by (rule BoxImpYields)
  from 2 3 show ?thesis using MP by fastforce
qed

lemma YieldsAndYieldsEqvYieldsAnd:
  "\<turnstile>   ((f yields  g) \<and> (f  yields  g1)) = f yields (g \<and> g1) "
proof -
 have  1: "\<turnstile> f; (\<not>  g \<or>  \<not>  g1) = ((f; (\<not>  g)) \<or>  (f; (\<not>  g1))) " 
   by (rule ChopOrEqv)
 hence 2: "\<turnstile> ((f; (\<not>  g)) \<or>  (f; (\<not>  g1))) = f; (\<not>  g \<or>  \<not>  g1) " 
   by auto
 have  3: "\<turnstile> (\<not>  g \<or>  \<not>  g1)  =  (\<not> (g \<and> g1)) " 
   by auto
 hence 4: "\<turnstile> f; (\<not>  g \<or>  \<not>  g1)  =  f; (\<not> (g \<and> g1)) " 
   by (rule RightChopEqvChop)
 have  5: "\<turnstile> (f; (\<not>  g)) \<or>  (f; (\<not>  g1)) = f; (\<not> (g \<and> g1)) " 
   using "2" "4" by fastforce
 hence 6: "\<turnstile> (\<not> (f; (\<not>  g)) \<and> \<not> (f; (\<not>  g1))) = (\<not> (f; (\<not> (g \<and> g1)))) "
    by (metis "1" "3" int_simps(14) int_simps(33) inteq_reflection) 
 from 6 show ?thesis by (simp add: yields_d_def)
qed

lemma YieldsAndYieldsImpAndYieldsAnd:
 "\<turnstile>   (f yields  g) \<and> (f1 yields  g1) \<longrightarrow> (f\<and> f1) yields (g\<and> g1) "
proof -
 have 1: "\<turnstile> f yields  g \<longrightarrow> (f \<and> f1) yields  g" 
   by (rule AndYieldsA)
 have 2: "\<turnstile> f1 yields  g1 \<longrightarrow> (f \<and> f1) yields  g1 " 
   by (rule AndYieldsB)
 have 3: "\<turnstile> ((f \<and> f1) yields  g \<and> (f \<and> f1) yields  g1) = (f \<and> f1) yields  (g \<and> g1) " 
   by (rule YieldsAndYieldsEqvYieldsAnd)
 from 1 2 3 show ?thesis by fastforce
qed

lemma YieldsYieldsEqvChopYields:
 "\<turnstile>   f yields (g yields  h) = (f; g) yields h  " 
proof -
 have  1: "\<turnstile> f; (g; (\<not>  h)) = (f; g); (\<not>  h)   " by (rule ChopAssoc)
 hence 2: "\<turnstile> f; (g; (\<not>  h)) = (f; g); (\<not>  h) " by auto
 have  3: "\<turnstile> g; (\<not>  h) = (\<not> \<not> (g; (\<not>  h))) " by auto
 hence 4: "\<turnstile> f; (g; (\<not>  h)) = f; (\<not> \<not> (g; (\<not>  h))) " by (rule RightChopEqvChop)
 have  5: "\<turnstile> f; (\<not> \<not> (g; (\<not>  h))) = (f; g); (\<not>  h) " using "2" "4" by auto
 hence 6: "\<turnstile> f; (\<not> (g yields  h)) = (f; g); (\<not>  h) " by (simp add: yields_d_def)
 hence 7: "\<turnstile> (\<not> (f; (\<not> (g yields  h)))) = (\<not> ((f; g); (\<not>  h))) " by auto
 from 7 show ?thesis by (simp add: yields_d_def)
qed

lemma EmptyYields:
 "\<turnstile>    empty   yields  f = f "
proof -
 have  1: "\<turnstile>  empty ; (\<not>  f) = (\<not>  f) " by (rule EmptyChop)
 hence 2: "\<turnstile> (\<not> ( empty ; (\<not>  f))) = f " by auto
 from 2 show ?thesis by (simp add: yields_d_def)
qed

lemma NextYields:
 "\<turnstile>   (\<circle> f) yields  g = wnext (f yields  g) "
proof -
 have  1: "\<turnstile> (\<circle> f); (\<not>  g) = \<circle>(f; (\<not>  g)) " by (rule NextChop)
 hence 2: "\<turnstile> (\<not> ((\<circle> f); (\<not>  g))) = (\<not> (\<circle>(f; (\<not>  g)))) " by auto
 hence 3: "\<turnstile> (\<circle> f) yields  g = (\<not> (\<circle>(f; (\<not>  g)))) " by (simp add: yields_d_def)
 have  4: "\<turnstile> (\<not>( \<circle>(f; (\<not>  g)))) = wnext (\<not> (f; (\<not>  g))) " by (auto simp: wnext_d_def)
 have  5: "\<turnstile> (\<circle> f) yields  g = wnext (\<not> (f; (\<not>  g))) " using "3" "4" by fastforce
 from 5 show ?thesis by (simp add: yields_d_def)
qed

lemma SkipChopEqvNext:
 " \<turnstile>    skip ; f = \<circle> f "
by (simp add: next_d_def)

lemma SkipYieldsEqvWeakNext:
 "\<turnstile>    skip  yields  f = wnext  f "
proof -
 have  1: "\<turnstile>  skip ; (\<not>  f) = \<circle>(\<not>  f) " by (rule SkipChopEqvNext)
 hence 2: "\<turnstile> (\<not> ( skip ; (\<not>  f))) = (\<not>( \<circle>(\<not>  f))) " by auto
 have  3: "\<turnstile> (\<not> (\<circle>(\<not>  f))) = wnext  f " by (auto simp: wnext_d_def)
 have  4: "\<turnstile> (\<not> ( skip ; (\<not>  f))) = wnext  f " using "2" "3" by fastforce
 from 4 show ?thesis by (simp add: yields_d_def)
qed

lemma NextImpSkipYields:
 "\<turnstile>   \<circle> f \<longrightarrow>  skip  yields  f "
proof -
 have 1: "\<turnstile> \<circle> f \<longrightarrow> wnext  f " using WnextEqvEmptyOrNext by fastforce
 have 2: "\<turnstile>  skip  yields  f = wnext  f " by (rule SkipYieldsEqvWeakNext)
 from 1 2 show ?thesis by fastforce
qed

lemma MoreEqvSkipChopTrue:
 "\<turnstile>    more  =  skip ; #True"
proof -
 have  1: "\<turnstile>  skip ; #True = \<circle>#True " by (rule SkipChopEqvNext)
 hence 2: "\<turnstile> \<circle>#True =  skip ; #True " by auto
 from 2 show ?thesis by (simp add: more_d_def)
qed

lemma MoreChopImpMore:
 "\<turnstile>    more ; f \<longrightarrow>  more "
proof -
 have 1: "\<turnstile> (\<circle>#True); f = \<circle>(#True; f)" by (rule NextChop)
 have 2: "\<turnstile> \<circle>(#True; f) \<longrightarrow>  more " by (simp add: NextImpNext more_d_def) 
 have 3: "\<turnstile> (\<circle>#True; f) \<longrightarrow>  more " using "1" "2" by fastforce
 from 3 show ?thesis by (metis more_d_def)
qed

lemma FmoreChopImpFmore:
 "\<turnstile>    fmore ; (f \<and> finite) \<longrightarrow>  fmore "
proof -
 have 1: "\<turnstile> fmore; (f \<and> finite)= \<circle>(finite; (f \<and> finite))" 
   using FmoreEqvSkipChopFinite by (metis NextChop inteq_reflection next_d_def)
 have 2: "\<turnstile> \<circle>(finite; (f \<and> finite)) \<longrightarrow>  fmore "
   by (metis ChopAndB FiniteChopFiniteEqvFinite FmoreEqvSkipChopFinite RightChopImpChop 
       inteq_reflection next_d_def) 
 have 3: "\<turnstile> (\<circle>finite; (f \<and> finite)) \<longrightarrow>  fmore " using "1" "2" 
   by (metis FmoreEqvSkipChopFinite inteq_reflection next_d_def)
 from 1 2 3 show ?thesis by fastforce
qed

lemma ChopMoreImpMore:
 "\<turnstile>   f;  more  \<longrightarrow>  more  "
proof -
 have 1: "\<turnstile> (f \<and> finite) ;  more  \<longrightarrow> \<diamond> more " 
   by (rule FiniteChopImpDiamond)
 have 11: "\<turnstile> (f \<and> inf) ;  more  \<longrightarrow> more"
   by (metis AndInfChopEqvAndInf MoreAndInfEqvInf Prop11 Prop12 lift_imp_trans)
 have 2: "\<turnstile> \<diamond> more  \<longrightarrow>  more "
   by (metis FiniteChopMoreEqvMore NowImpDiamond inteq_reflection sometimes_d_def) 
 have 3: "\<turnstile> (f \<and> finite) ;  more  \<longrightarrow> more " 
   using 1 2 by fastforce
 have 4: "\<turnstile> f  = ( (f \<and> finite)  \<or> (f \<and> inf))  "
   by (simp add: OrFiniteInf)
 hence 5: "\<turnstile> f;more  = ( (f \<and> finite);more  \<or> (f \<and> inf);more) " 
   by (simp add: OrChopEqvRule) 
 from  11 3 5 show ?thesis  by fastforce
qed

lemma MoreChopEqvNextDiamond:
 "\<turnstile>    fmore ; f = \<circle>(\<diamond> f) "
proof -
 have 1: "\<turnstile>  fmore ; f = (\<circle> finite); f "
   by (simp add: FmoreEqvSkipChopFinite LeftChopEqvChop next_d_def)
 have 2: "\<turnstile> (\<circle> finite); f = \<circle>(finite; f) " 
   by (rule NextChop)
 have 3: "\<turnstile>  fmore ; f = \<circle>(finite; f) " 
   using "1" "2" by fastforce
 from 3 show ?thesis by (simp add: sometimes_d_def)
qed

lemma WeakNextBoxImpMoreYields:
 "\<turnstile>    fmore  yields  f = wnext( \<box> f) "
proof - 
 have 1: "\<turnstile> fmore ; (\<not> f) = \<circle>(\<diamond> (\<not>f)) " by (rule MoreChopEqvNextDiamond)
 have 2: "\<turnstile> \<circle>(\<diamond> (\<not>f)) = \<circle>(\<not>(\<box>f))" by (auto simp: always_d_def)
 have 3: "\<turnstile> \<circle>(\<not>(\<box>f)) = (\<not> ( wnext( \<box> f) ))" by (auto simp: wnext_d_def)
 have 4: "\<turnstile> fmore ; (\<not> f)  = (\<not>(fmore  yields  f))" by (simp add: yields_d_def) 
 from 1 2 3 4 show ?thesis by fastforce
qed

lemma NotEqvYieldsMore:
 "\<turnstile>   (\<not>  f) = f yields  more  "
proof -
 have  1: "\<turnstile> f;  empty  = f " by (rule ChopEmpty)
 hence 2: "\<turnstile> (\<not> (f;  empty )) = (\<not>  f) " by auto
 have  3: "\<turnstile>  empty  = (\<not>  more) " by (auto simp: empty_d_def)
 hence 4: "\<turnstile> f;  empty  = f; (\<not>  more) " by (rule RightChopEqvChop)
 hence 5: "\<turnstile> (\<not> (f;  empty )) = (\<not> (f; (\<not>  more ))) " by auto
 have  6: "\<turnstile> (\<not>  f) = (\<not> (f; (\<not>  more) )) " using "2" "5" by fastforce
 from 6 show ?thesis by (metis yields_d_def)
qed

lemma LeftChopImpMoreRule:
 assumes "\<turnstile> f \<longrightarrow>  more "
 shows   "\<turnstile> f; g \<longrightarrow>  more "
proof -
 have  1: "\<turnstile> f \<longrightarrow>  more " using assms by auto
 hence 2: "\<turnstile> f; g \<longrightarrow>  more ; g " by (rule LeftChopImpChop)
 have 3: " \<turnstile>  more ; g \<longrightarrow>  more " by (rule MoreChopImpMore)
 from 2 3 show ?thesis using lift_imp_trans by blast
qed

lemma LeftChopImpFMoreRule:
 assumes "\<turnstile> f \<longrightarrow>  fmore "
 shows   "\<turnstile> f; (g \<and> finite) \<longrightarrow>  fmore "
proof -
 have  1: "\<turnstile> f \<longrightarrow>  fmore " using assms by auto
 hence 2: "\<turnstile> f; (g \<and> finite) \<longrightarrow>  fmore ; (g \<and> finite) " by (rule LeftChopImpChop)
 have 3: " \<turnstile>  fmore ; (g \<and> finite) \<longrightarrow>  fmore " using FmoreChopImpFmore by fastforce
 from 2 3 show ?thesis using lift_imp_trans by blast
qed

lemma RightChopImpMoreRule:
 assumes "\<turnstile>   g \<longrightarrow>  more "
 shows   "\<turnstile> f; g \<longrightarrow>  more "
proof -
 have  1: "\<turnstile> g \<longrightarrow>  more " using assms by auto
 hence 2: "\<turnstile> f; g \<longrightarrow> f;  more " by (rule RightChopImpChop)
 have  3: "\<turnstile> f;  more  \<longrightarrow>  more " by (rule ChopMoreImpMore)
 from 2 3 show ?thesis using lift_imp_trans by blast
qed

lemma NotDiEqvBiNot:
 "\<turnstile>   (\<not> ( di  f)) = bi (\<not>  f) "
proof -
 have  1: "\<turnstile> f = (\<not> \<not>  f) " by auto
 hence 2: "\<turnstile>  di  f =  di (\<not> \<not>  f)" by (rule DiEqvDi)
 hence 3: "\<turnstile> (\<not>  ( di  f)) = (\<not> ( di (\<not> \<not>  f)))" by auto
 from 3 show ?thesis by (simp add: bi_d_def)
qed

lemma ChopImpDi:
 "\<turnstile>   f; g \<longrightarrow>  di  f "
proof -
 have  1: "\<turnstile> g \<longrightarrow> #True " by auto
 hence 2: "\<turnstile> f; g \<longrightarrow> f; #True " by (rule RightChopImpChop)
 from 2 show ?thesis by (simp add: di_d_def)
qed

lemma TrueEqvTrueChopTrue:
 "\<turnstile>   #True = #True; #True "
proof -
 have  1: "\<turnstile> #True; #True \<longrightarrow> #True" by auto
 have  2: "\<turnstile> #True \<longrightarrow>  di #True " by (rule DiIntro)
 hence 3: "\<turnstile> #True \<longrightarrow> #True; #True " by (simp add: di_d_def)
 from 1 3 show ?thesis by auto
qed

lemma DiEqvDiDi:
 "\<turnstile>    di  f =  di (  di  f) "
proof -
 have  1: "\<turnstile> #True = #True; #True " by (rule TrueEqvTrueChopTrue)
 hence 2: "\<turnstile> f; #True = f; (#True; #True) " by (rule RightChopEqvChop)
 have  3: "\<turnstile> f; (#True; #True)= (f; #True); #True   " by (rule ChopAssoc)
 have  4: "\<turnstile> f; #True = (f; #True); #True " using "2" "3"  by fastforce 
 from 4 show ?thesis by (metis di_d_def)
qed

lemma BiEqvBiBi:
 "\<turnstile>   bi  f = bi( bi  f) "
proof -
 have  1: "\<turnstile>  di (\<not>  f) =  di(  di (\<not>  f)) " by (rule DiEqvDiDi)
 have  2: "\<turnstile>  di (\<not>  f) = (\<not> ( bi  f)) " by (rule DiNotEqvNotBi)
 hence 3: "\<turnstile>  di ( di (\<not>  f)) =  di (\<not> ( bi  f)) " by (rule DiEqvDi)
 have  4: "\<turnstile>  di (\<not>  f) =  di (\<not>(  bi  f)) " using "1" "3"  by fastforce 
 hence 5: "\<turnstile> (\<not> ( di (\<not>  f))) = (\<not> ( di (\<not> ( bi  f)))) " by fastforce
 from 5 show ?thesis by (metis bi_d_def)
qed

lemma DiOrEqv:
 "\<turnstile>    di (f \<or>  g) =  (di  f \<or>   di  g) " 
proof -
 have 1: "\<turnstile> (f\<or>  g); #True = (f; #True \<or>  g; #True) " by (rule OrChopEqv)
 from 1 show ?thesis by (simp add: di_d_def)
qed

lemma DiAndA:
 "\<turnstile>    di (f \<and> g) \<longrightarrow>  di  f "
proof -
 have 1: "\<turnstile> (f \<and> g); #True \<longrightarrow> f; #True " by (rule AndChopA)
 from 1 show ?thesis by (simp add: di_d_def)
qed

lemma DiAndB:
 "\<turnstile>    di (f \<and> g) \<longrightarrow>  di  g "
proof -
 have 1: "\<turnstile> (f \<and> g); #True \<longrightarrow> g; #True " by (rule AndChopB)
 from 1 show ?thesis by (simp add: di_d_def)
qed

lemma DiAndImpAnd:
 "\<turnstile>    di (f \<and> g) \<longrightarrow>  di  f \<and>  di  g "  
proof -
 have 1: "\<turnstile>  di (f \<and> g) \<longrightarrow>  di  f " by (rule DiAndA)
 have 2: "\<turnstile>  di (f \<and> g) \<longrightarrow>  di  g " by (rule DiAndB)
 from 1 2 show ?thesis by fastforce
qed

lemma DiSkipEqvMore:
 "\<turnstile>    di  skip  =  more "
proof -
 have 1: "\<turnstile>  skip ; #True = \<circle>#True " by (rule SkipChopEqvNext)
 have 2: "\<turnstile> \<circle>#True =  more " by (auto simp: more_d_def)
 have 3: "\<turnstile>  skip ; #True =  more " using "1" "2" by fastforce
 from 3 show ?thesis by (simp add: di_d_def)
qed

lemma DiMoreEqvMore:
 "\<turnstile>    di  more  =  more "
proof -
 have  1: "\<turnstile>  di (\<circle> #True ) = \<circle>( di #True) "
   by (rule DiNext)
 have  2: "\<turnstile> \<circle>( di  #True)  \<longrightarrow>  more "
   by (metis "1" ChopImpDi TrueEqvTrueChopTrue di_d_def int_eq more_d_def) 
 have  3: "\<turnstile>  di( \<circle> #True)  \<longrightarrow>  more " 
   using "1" "2" by fastforce
 hence 4: "\<turnstile>  di  more  \<longrightarrow>  more " 
   by (simp add: more_d_def)
 have  5: "\<turnstile>  more  \<longrightarrow>  di  more " 
   by (rule ImpDi)
 from 4 5 show ?thesis by fastforce
qed

lemma DiIfEqvRule:
 assumes "\<turnstile>   f =  if\<^sub>i  (init w)  then  g  else  h "
 shows   "\<turnstile>  di  f =  if\<^sub>i  (init w)  then  ( di  g)  else   (di  h) "
proof -
 have  1: "\<turnstile> f =  if\<^sub>i  (init w)  then  g  else  h " using assms by auto
 hence 2: "\<turnstile> f; #True =  if\<^sub>i (init w) then (g; #True) else (h; #True) " by (rule IfChopEqvRule)
 from 2 show ?thesis by (simp add: di_d_def)
qed

lemma DiEmpty:
 "\<turnstile>    di  empty "
proof -
 have 1: "\<turnstile> #True  " by auto
 have 2: "\<turnstile>  empty ; #True = #True " by (rule EmptyChop) 
 have 3: "\<turnstile>  empty ; #True " using "1" "2" by auto
 from 3 show ?thesis by (simp add: di_d_def)
qed

lemma DaNotEqvNotBa:
 "\<turnstile>    da (\<not>  f) = (\<not> ( ba  f)) "
proof -
 have 1: "\<turnstile>  ba  f = (\<not> ( da (\<not>  f))) " by (simp add: ba_d_def)
 from 1 show ?thesis by fastforce
qed

lemma DaEqvDa:
 assumes "\<turnstile>   f = g "
 shows "\<turnstile> da  f =  da  g "
using assms using int_eq by force

lemma DaEqvNotBaNot:
 "\<turnstile>    da  f = (\<not> ( ba (\<not>  f))) "
proof -
 have  1: "\<turnstile>  ba (\<not>  f) = (\<not> ( da (\<not> \<not>  f)))" by (simp add: ba_d_def)
 hence 2: "\<turnstile>  da (\<not> \<not>  f) = (\<not>(  ba (\<not>  f))) " by fastforce
 have  3: "\<turnstile> f = (\<not> \<not>  f) " by simp
 hence 4: "\<turnstile>  da  f =  da  (\<not> \<not>  f) " by (rule DaEqvDa)
 from 2 4 show ?thesis by simp
qed

lemma BaElim:
 "\<turnstile>    ba  f \<longrightarrow> f "
proof -
 have  1: "\<turnstile>  ba  f = \<box>(bi  f) " by (rule BaEqvBtBi)
 have  2: "\<turnstile> bi  f \<longrightarrow> f " by (rule BiElim)
 hence 3: "\<turnstile> \<box>(bi  f \<longrightarrow> f) " by (rule BoxGen)
 have  4: "\<turnstile> \<box>(bi  f \<longrightarrow> f) \<longrightarrow> \<box>(bi  f) \<longrightarrow> \<box> f " by (rule BoxImpDist)
 have  5: "\<turnstile> \<box>(bi  f) \<longrightarrow> \<box> f " using "3" "4" MP by fastforce
 have  6: "\<turnstile> \<box> f \<longrightarrow> f " by (rule BoxElim)
 from 1 5 6 show ?thesis  using BaImpBt lift_imp_trans by metis
qed

lemma DaIntro:
 "\<turnstile>   f \<longrightarrow>  da  f "
proof -
 have  1: "\<turnstile>  ba  (\<not>  f) \<longrightarrow> (\<not>  f) "  by (rule BaElim)
 hence 2: "\<turnstile> \<not> \<not>  f \<longrightarrow> \<not> (  ba  (\<not>  f)) " by fastforce
 have  3: "\<turnstile> f = (\<not> \<not>  f) " by simp
 have  4: "\<turnstile>  da  f = (\<not>  ( ba  (\<not>  f))) " by (rule DaEqvNotBaNot)
 from 2 3 4 show ?thesis by fastforce
qed

lemma BaGen:
 assumes "\<turnstile>   f "
 shows   "\<turnstile>  ba  f "
proof -
 have  1: "\<turnstile>   f " using assms by auto
 hence 2: "\<turnstile> \<box> f " by (rule BoxGen)
 hence 3: "\<turnstile> bi( \<box> f) " by (rule BiGen)
 have  4: "\<turnstile>  ba  f = bi (\<box> f) " by (rule BaEqvBiBt)
 from 3 4 show ?thesis by fastforce
qed

lemma BaImpDist:
 "\<turnstile>  ba (f \<longrightarrow> g) \<longrightarrow>  ba  f \<longrightarrow>  ba  g "
proof -
 have  1: "\<turnstile> bi (f \<longrightarrow> g) \<longrightarrow> (bi  f \<longrightarrow> bi  g) " 
   by (rule BiImpDist)
 hence 2: "\<turnstile> \<box>(bi (f \<longrightarrow> g) \<longrightarrow> (bi  f \<longrightarrow> bi  g)) " 
   by (rule BoxGen)
 have  3: "\<turnstile> \<box>(bi (f \<longrightarrow> g) \<longrightarrow> (bi  f \<longrightarrow> bi  g))
            \<longrightarrow>  
            (\<box> (bi (f \<longrightarrow> g)) \<longrightarrow> (\<box>(bi  f) \<longrightarrow> \<box>(bi  g))) " 
   by (meson "2" BoxImpDist MP lift_imp_trans Prop01 Prop05 Prop09)
 have  4: "\<turnstile> \<box>(bi (f \<longrightarrow> g)) \<longrightarrow> (\<box>(bi  f) \<longrightarrow> \<box>(bi  g)) " 
   using "2" "3" MP by fastforce
 have  5: "\<turnstile>  ba (f\<longrightarrow> g) = \<box>(bi (f \<longrightarrow> g)) " 
   by (rule BaEqvBtBi)
 have  6: "\<turnstile>  ba  f = \<box>(bi  f) "
   by (rule BaEqvBtBi)
 have  7: "\<turnstile>  ba  g = \<box>(bi  g) " 
   by (rule BaEqvBtBi)
 from 4 5 6 7 show ?thesis by fastforce
qed

lemma BiAndEqv:
 "\<turnstile> bi (f \<and> g) = (bi f \<and> bi g)" 
proof -
 have 1: "\<turnstile> di (\<not> f \<or> \<not> g) = (di (\<not> f) \<or> di (\<not> g))"
   by (simp add: DiOrEqv)
 have 2: "\<turnstile> (\<not> (di (\<not> f \<or> \<not> g))) = (\<not> di (\<not> f) \<and> \<not> di (\<not> g)) "
   using "1" by auto
 have 3: "\<turnstile> (f \<and> g) = (\<not> (\<not> f \<or> \<not> g))"
   by fastforce
 have 4: "\<turnstile> bi (f \<and> g) = (\<not> (di (\<not> f \<or> \<not> g)))"
   unfolding bi_d_def using 3 by (metis int_simps(4) inteq_reflection)
 from 2 4 show ?thesis unfolding bi_d_def by (metis inteq_reflection)
qed  

lemma BaAndEqv:
 "\<turnstile>    ba (f \<and> g) =  (ba  f \<and>  ba  g) "
proof -
 have  1: "\<turnstile>    ba (f \<and> g) =   \<box>(bi (f \<and> g)) " 
   by (rule BaEqvBtBi)
 have  2: "\<turnstile>  bi (f \<and> g) = (bi f \<and> bi g) "
   by (simp add: BiAndEqv)
 hence 3: "\<turnstile> \<box>(bi (f \<and> g)) = \<box>(bi f \<and> bi g) " 
   using BoxEqvBox by blast
 have  4: "\<turnstile> \<box>(bi f \<and> bi g)= (\<box>(bi f)  \<and> \<box>(bi g)) " 
   by (metis "2" BoxAndBoxEqvBoxRule inteq_reflection)
 have  5: "\<turnstile>ba  f = \<box>(bi  f) " 
   by (rule BaEqvBtBi)
 have  6: "\<turnstile>  ba  g = \<box>(bi  g) " 
   by  (rule BaEqvBtBi)
 from 1 3 4 5 6 show ?thesis by  fastforce
qed

lemma BaImpBaEqvBa:
 "\<turnstile>   ba (f = g) \<longrightarrow>  (ba  f =  ba  g) "
proof -
 have  1: "\<turnstile>  ba (f \<longrightarrow> g) \<longrightarrow>  ba  f \<longrightarrow>  ba  g " by (rule BaImpDist)
 have  2: "\<turnstile>  ba (g \<longrightarrow> f) \<longrightarrow>  ba  g \<longrightarrow>  ba  f " by (rule BaImpDist)
 have  25: "\<turnstile> (f = g) = ((f \<longrightarrow> g) \<and> (g \<longrightarrow> f))" by fastforce
 have  3: "\<turnstile> ba (f = g) = ba ((f \<longrightarrow> g) \<and> (g \<longrightarrow> f)) " by (metis "25" BaAndEqv inteq_reflection) 
 have  4: "\<turnstile> ba ((f \<longrightarrow> g) \<and> (g \<longrightarrow> f)) = (ba((f \<longrightarrow> g)) \<and> ba((g \<longrightarrow> f))) " by (rule BaAndEqv) 
 have  5: "\<turnstile> ((ba  f \<longrightarrow>  ba  g) \<and> (ba  g \<longrightarrow>  ba  f)) = (ba  f =  ba  g) " by auto
 from 1 2 3 4 5 show ?thesis  by fastforce
qed

lemma BaImpBa:
 assumes "\<turnstile>   f \<longrightarrow> g"
 shows   "\<turnstile>ba  f \<longrightarrow>  ba  g"
using BaGen BaImpDist MP assms by metis

lemma BaEqvBa: 
 assumes "\<turnstile>   f = g"
 shows   "\<turnstile> ba  f =  ba  g"
using BaGen BaImpBaEqvBa MP assms by metis

lemma DaImpDa:
 assumes "\<turnstile>   f \<longrightarrow> g"
 shows   "\<turnstile> da  f \<longrightarrow>  da  g"
using assms by (metis DaEqvDtDi DiAndB DiamondImpDiamond inteq_reflection Prop10)

lemma DiamondEqvDiamondDiamond:
 "\<turnstile> \<diamond> f = \<diamond> (\<diamond> f)"
proof -
 have 1: "\<turnstile> \<diamond> (\<diamond> f) = finite;(finite;f)" 
   by (simp add: sometimes_d_def)
 have 2: "\<turnstile> finite;(finite;f)  = (finite;finite);f" 
   by (rule ChopAssoc)
 have 3: "\<turnstile> (finite;finite);f = finite;f" 
   by (simp add: LeftChopEqvChop FiniteChopFiniteEqvFinite)
 have 4: "\<turnstile> finite;f  = \<diamond>f" 
   by (simp add: sometimes_d_def)
 from 1 2 3 4 show ?thesis by fastforce
qed

lemma DaEqvDaDa:
  "\<turnstile>   da  f =  da (  da  f)"
proof -
 have  1: "\<turnstile>  da  f = \<diamond>( di  f)" 
   by (rule DaEqvDtDi)
 have  2: "\<turnstile>  di  f =  (di ( di  f))"  
   by (rule DiEqvDiDi)
 hence 3: "\<turnstile> \<diamond> ( di  f) = \<diamond> (di  (di  f))" 
   by (rule DiamondEqvDiamond)
 have  4: "\<turnstile> \<diamond>  (di  f) = \<diamond>(\<diamond> (di  (di  f)))" 
   using DiamondEqvDiamondDiamond DiEqvDiDi using "3" by fastforce
 have  5: "\<turnstile> \<diamond>  (di  (di  f)) =  di (\<diamond>  (di  f))"  
   by (rule DtDiEqvDiDt)
 hence 6: "\<turnstile> \<diamond>(\<diamond>  (di  (di  f))) = \<diamond> (di (\<diamond>  (di  f)))" 
   by (rule DiamondEqvDiamond)
 have  7: "\<turnstile>  da  f = \<diamond> (di( \<diamond> ( di  f)))" 
   using "1" "3" "4" "6" by fastforce
 have  8: "\<turnstile>  da (\<diamond> ( di  f)) = \<diamond>( di (\<diamond>  (di  f)))" 
   by (rule DaEqvDtDi)  
 have  9: "\<turnstile>  da ( da  f) =  da (\<diamond>  (di  f))" 
   using 1 by (rule DaEqvDa)
 from 7 8 9 show ?thesis by fastforce
qed

lemma BaEqvBaBa:
 "\<turnstile>    ba  f =  ba  (ba  f)"
proof -
 have 1: "\<turnstile>  da (\<not>  f) =  da  (da (\<not>  f))" by (rule DaEqvDaDa)  
 have 2: "\<turnstile>  da  (da (\<not>  f)) = (\<not>  (ba (\<not>  (da (\<not>  f)))))"  by (rule DaEqvNotBaNot)  
 have 3: "\<turnstile> (\<not>  (da  (da (\<not>  f)))) =  ba (\<not>  (da (\<not>  f)))" by (auto simp: ba_d_def)  
 have 4: "\<turnstile> (\<not>  (da (\<not>  f))) =  ba (\<not>  (da (\<not>  f)))" using "1" "2" "3" by fastforce
 from 4 show ?thesis by (metis ba_d_def)
qed

lemma BaLeftChopImpChop:
 "\<turnstile>    ba (f\<longrightarrow> f1) \<longrightarrow> f; g \<longrightarrow> f1; g"
proof -
 have 1: "\<turnstile>  ba (f \<longrightarrow> f1) \<longrightarrow> bi (f \<longrightarrow> f1)" by (rule BaImpBi)  
 have 2: "\<turnstile> bi (f \<longrightarrow> f1) \<longrightarrow> f; g \<longrightarrow> f1; g" by (rule BiChopImpChop)  
 from 1 2 show ?thesis by fastforce
qed

lemma BaRightChopImpChop: 
 "\<turnstile>    ba (g\<longrightarrow> g1) \<longrightarrow> f; g \<longrightarrow> f; g1"
proof -
 have 1: "\<turnstile>  ba (g \<longrightarrow> g1) \<longrightarrow> \<box>(g \<longrightarrow> g1)" by (rule BaImpBt)  
 have 2: "\<turnstile> \<box>(g \<longrightarrow> g1) \<longrightarrow> f; g \<longrightarrow> f; g1" by (rule BoxChopImpChop)  
 from 1 2 show ?thesis by fastforce
qed

lemma ChopAndBaImport:
  "\<turnstile>   (f; f1) \<and>  ba  g \<longrightarrow> (f \<and> g); (f1 \<and> g)"
proof -
 have 1: "\<turnstile>  ba  g \<and> (f; f1) \<longrightarrow> (g \<and> f); (g \<and> f1)" by (rule BaAndChopImport)  
 have 2: "\<turnstile> (g \<and> f); (g \<and> f1) = (f \<and> g); (f1 \<and> g)" by (rule AndChopAndCommute) 
 from 1 2 show ?thesis by fastforce
qed

lemma BaAndChopImportA: 
 "\<turnstile> ba f \<and> g;g1 \<longrightarrow> (f \<and> g);g1"
by (meson BaAndChopImport ChopAndB lift_imp_trans)

lemma BaAndChopImportB:
 "\<turnstile> ba f \<and> g;g1 \<longrightarrow> (f \<and> g);(ba f \<and> g1)" 
proof -
 have 1: "\<turnstile> ba f = ba (ba f)" 
   by (simp add: BaEqvBaBa)
 have 2: "\<turnstile> ba (ba f) \<and> g;g1 \<longrightarrow> g;(ba f \<and> g1)"
   by (metis  AndChopB BaAndChopImport lift_imp_trans)
 have 3: "\<turnstile> ba f \<and> g;(ba f \<and> g1) \<longrightarrow> (f \<and> g);(ba f \<and> g1)"
   by (simp add: BaAndChopImportA)
 from 1 2 3 show ?thesis by fastforce
qed
 
lemma BaImpBaImpBaAnd:
 "\<turnstile> ba h \<longrightarrow> ba(g \<longrightarrow> ba h \<and> g ) "
proof -
 have  1: "\<turnstile> ba h \<longrightarrow> (g \<longrightarrow> ba h \<and> g ) " by fastforce
 hence 2: "\<turnstile> ba(ba h) \<longrightarrow> ba(g \<longrightarrow> ba h \<and> g ) " by (rule BaImpBa)
 have  3: "\<turnstile> ba h = ba(ba h) " by (rule BaEqvBaBa)
 from 2 3 show ?thesis by fastforce
qed

lemma BaChopImpChopBa:
 "\<turnstile>    ba  f \<longrightarrow> g; g1 \<longrightarrow> g; (( ba  f) \<and> g1)"
proof - 
 have 1: "\<turnstile>  ba  f \<longrightarrow>  ba (g1 \<longrightarrow> (ba f) \<and> g1 )" by (rule BaImpBaImpBaAnd)
 have 2: "\<turnstile>  ba (g1 \<longrightarrow>  ba  f \<and> g1 ) \<longrightarrow> g; g1 \<longrightarrow> g; ( ba  f \<and> g1)" by (rule BaRightChopImpChop)  
 from 1 2 show ?thesis by fastforce
qed

lemma DiNotBaImpNotBa:
  "\<turnstile>   di (\<not> ( ba  f)) \<longrightarrow> \<not> ( ba  f)" 
proof -
 have  1: "\<turnstile>  ba  f =  ba(  ba  f)" by (rule BaEqvBaBa)  
 have  2: "\<turnstile>  ba ( ba  f) \<longrightarrow> bi ( ba  f)" by (rule BaImpBi)  
 have  3: "\<turnstile>  ba  f \<longrightarrow> bi ( ba  f)" using "1" "2" by fastforce
 hence 4: "\<turnstile>  ba  f \<longrightarrow> \<not> ( di (\<not> ( ba  f)))" by (simp add: bi_d_def)    
 from 4 show ?thesis by fastforce
qed

lemma NotBaChopImpNotBa:
 "\<turnstile>   (\<not> ( ba  f)); g \<longrightarrow> \<not> ( ba  f)"
proof -
 have 1: "\<turnstile> (\<not> ( ba  f)); g \<longrightarrow>  di (\<not> ( ba  f))" by (rule ChopImpDi)  
 have 2: "\<turnstile>  di (\<not> ( ba  f)) \<longrightarrow> \<not> ( ba  f)"  by (rule DiNotBaImpNotBa) 
 from 1 2 show ?thesis using lift_imp_trans by blast  
qed

lemma DiamondFinImpFin:
 " \<turnstile> \<diamond> (fin f) \<longrightarrow> fin f"
proof -
 have  1: "\<turnstile> fin f = #True;(f \<and> empty)" 
   by (rule FinEqvTrueChopAndEmpty)
 hence 2: "\<turnstile> \<diamond> (fin f) = finite;(#True;(f \<and> empty)) " 
   by (metis FiniteChopFiniteEqvFinite LeftChopEqvChop inteq_reflection sometimes_d_def)      
 have  3: "\<turnstile> finite;(#True;(f \<and> empty)) = (finite;#True);(f \<and> empty)" 
   by (rule ChopAssoc)
 have  4: "\<turnstile> (finite;#True);(f \<and> empty) \<longrightarrow> #True;(f \<and> empty)" 
   using "1" "2" "3" DiamondFin by fastforce
 from 1 2 3 4 show ?thesis by fastforce
qed

lemma ChopFinImpFin:
 " \<turnstile>   (f \<and> finite);  fin  (init w) \<longrightarrow>  fin  (init w)"
proof -
 have 1: "\<turnstile> (f \<and> finite);  fin  (init w) \<longrightarrow> \<diamond> ( fin  (init w))"  by (rule  FiniteChopImpDiamond) 
 have 2: "\<turnstile> \<diamond>  (fin  (init w)) \<longrightarrow>  fin  (init w)" by (rule DiamondFinImpFin)
 from  1 2  show ?thesis using lift_imp_trans by blast   
qed

lemma FiniteRightChopEqvChop:
 assumes "\<turnstile> finite \<longrightarrow> g = g1"
 shows   "\<turnstile> finite \<longrightarrow> f;g = f;g1"
using assms by (auto simp add: Valid_def itl_defs)

lemma FinImpYieldsFin:
 "\<turnstile>    fin  (init w) \<and> finite \<longrightarrow> (f \<and> finite) yields (fin  (init w) \<and> finite)"
proof -
 have  1: "\<turnstile> (f \<and> finite);  (fin   (init (\<not> w)) \<and> finite) \<longrightarrow>  (fin   (init (\<not> w)) \<and> finite)" 
   by (metis (no_types, lifting)  ChopAndB FiniteChopEqvDiamond FiniteChopFinExportA 
       FiniteChopFiniteEqvFinite FiniteChopImpDiamond Prop12 inteq_reflection
       lift_and_com lift_imp_trans) 
 have  2: "\<turnstile> finite \<longrightarrow> (\<not> ( fin (init w) \<and> finite)) = (fin (init (\<not> w)) \<and> finite)     " 
   using FinNotStateEqvNotFinState by fastforce
 hence 3: "\<turnstile>  finite \<longrightarrow> (f \<and> finite); (\<not> ( fin (init w) \<and> finite)) = 
              (f \<and> finite); ( fin (init (\<not> w)) \<and> finite)"  
   using FiniteRightChopEqvChop[of "LIFT(\<not> ( fin (init w) \<and> finite))" 
                                  "LIFT(fin (init (\<not> w)) \<and> finite)" "LIFT(f \<and> finite)"]
   by blast
 have  4: "\<turnstile> (f \<and> finite); (\<not> ( fin  (init w) \<and> finite)) \<longrightarrow> (\<not> ( fin  (init w) \<and> finite))" 
   using "1" "2" "3" by fastforce
 hence 5: "\<turnstile>  fin  (init w) \<and> finite \<longrightarrow> \<not> ((f \<and> finite); (\<not> ( fin  (init w) \<and> finite)))" 
   by fastforce     
 from 5 show ?thesis by (simp add: yields_d_def)
qed

lemma ChopAndFin:
 " \<turnstile>  ((f; g) \<and>  (fin  (init w) \<and> finite)) = (f \<and> finite); (g \<and>  (fin  (init w) \<and> finite))"
proof -
 have  1: "\<turnstile>  fin  (init w) \<and> finite \<longrightarrow> (f \<and> finite) yields ( fin  (init w) \<and> finite)" 
   by (rule FinImpYieldsFin)  
 have 10: "\<turnstile> ((f; g) \<and>  (fin  (init w) \<and> finite)) = 
             (((f \<and> finite); (g \<and> finite)) \<and>  fin  (init w) \<and> finite)"
   using ChopAndFiniteDist[of "f" "g"] by auto 
 have 2: "\<turnstile> (f; g) \<and>  (fin  (init w) \<and> finite) \<longrightarrow> 
            ((f \<and> finite); (g \<and> finite)) \<and> (f \<and> finite) yields  ( fin  (init w) \<and> finite)" 
   using 1 10 by fastforce  
 have  3: "\<turnstile> ((f \<and> finite); (g \<and> finite)) \<and> (f \<and> finite) yields  ( fin  (init w) \<and> finite) \<longrightarrow>
              (f \<and> finite); ((g \<and> finite) \<and>  (fin  (init w) \<and> finite))" 
   using ChopAndYieldsImp by blast
 have 30: "\<turnstile> ((g \<and> finite) \<and>  (fin  (init w) \<and> finite)) = (g \<and> fin  (init w) \<and> finite)"
   by auto 
 have  4: "\<turnstile> (f; g) \<and>  (fin  (init w) \<and> finite)\<longrightarrow> (f \<and> finite); (g \<and>  fin  (init w) \<and> finite)" 
   using "2" "3" "30"  
   by (metis (mono_tags, lifting) inteq_reflection lift_imp_trans)
 have 11: "\<turnstile> (f \<and> finite); (g \<and>  fin  (init w) \<and> finite) \<longrightarrow> (f \<and> finite); (g \<and> finite)"  
   using ChopAndA by (metis "30" inteq_reflection)  
 have 12: "\<turnstile> (f \<and> finite); (g \<and>  fin  (init w) \<and> finite) \<longrightarrow> 
             (f \<and> finite);  (fin  (init w) \<and> finite)" 
   by (rule ChopAndB)  
 have 13: "\<turnstile> (f \<and> finite);  (fin  (init w) \<and> finite)\<longrightarrow> \<diamond> ( fin  (init w) \<and> finite)" 
   using FiniteChopImpDiamond  by blast
 have 14: "\<turnstile> \<diamond>( fin  (init w) \<and> finite) \<longrightarrow>  fin  (init w)"   
   by (metis ChopAndA DiamondFin inteq_reflection sometimes_d_def)
 have 15: "\<turnstile> (f \<and> finite); (g \<and>  fin  (init w) \<and> finite) \<longrightarrow> 
              ((f \<and> finite); (g \<and> finite)) \<and>  fin  (init w)" 
   using "11" "12" "13" "14" by fastforce
 from 4 15 show ?thesis by (metis ChopAndFiniteDist Prop12 int_iffI inteq_reflection)
qed

lemma ChopAndNotFin:
  " \<turnstile>   (f; g \<and> \<not> ( fin  (init w)) \<and> finite) = (f \<and> finite); (g \<and> \<not> ( fin  (init w)) \<and> finite)"
proof -
 have  1: "\<turnstile> (f; g \<and>  fin   (init (\<not> w)) \<and> finite) = 
             (f \<and> finite); (g \<and>  fin   (init (\<not> w)) \<and> finite)" 
   by (rule ChopAndFin)  
 have  2: "\<turnstile>  (fin   (init (\<not> w)) \<and> finite)= ( (\<not>  ( fin  (init w) )) \<and> finite)"   
   using FinNotStateEqvNotFinState by fastforce
 show ?thesis by (metis "1" "2" int_eq) 
qed

lemma FinChopChain:
 "\<turnstile>   (((init w )  \<and> finite \<longrightarrow>  fin  (init w1))); 
      (((init w1)  \<and> finite \<longrightarrow>  fin  (init w2)))
     \<and> finite
  \<longrightarrow> (((init w )  \<and> finite \<longrightarrow>  fin  (init w2)))"
proof -
 have  1: "\<turnstile>  (init w) \<and> finite \<and>
              ((init w) \<and> finite \<longrightarrow>  fin  (init w1)); ((init w1) \<and> finite \<longrightarrow>  fin  (init w2))
           \<longrightarrow>   
              ((init w) \<and> finite \<and>  ((init w) \<and> finite \<longrightarrow> fin  (init w1))); 
              (((init w1) \<and> finite\<longrightarrow>  fin  (init w2)) \<and> finite)"
   using ChopAndFiniteDist StateAndChopImport
   by (metis (no_types, opaque_lifting) inteq_reflection lift_and_com)          
 have  2: "\<turnstile> (init w) \<and> finite \<and> ((init w) \<and> finite \<longrightarrow>  fin(init w1)) \<longrightarrow> fin (init w1) \<and> finite" 
   by auto
 have  3: "\<turnstile> ((init w) \<and> finite \<and> ((init w) \<and> finite \<longrightarrow>  fin  (init w1))); 
             (((init w1) \<and> finite \<longrightarrow>  fin  (init w2)) \<and> finite)
           \<longrightarrow>  
           ( fin  (init w1) \<and> finite); (((init w1) \<and> finite \<longrightarrow>  fin  (init w2)) \<and> finite)" 
   using  "2" LeftChopImpChop by blast
 have  4: "\<turnstile>  ( fin  (init w1) \<and> finite); (((init w1) \<and> finite \<longrightarrow>  fin  (init w2)) \<and> finite) =
            \<diamond>((init w1)  \<and> ((init w1) \<and> finite \<longrightarrow>  fin  (init w2)) \<and> finite)" 
   using FinChopEqvDiamond by blast
 have 41: "\<turnstile> ((init w1) \<and> finite \<and> ((init w1) \<and> finite \<longrightarrow>  fin (init w2))) \<longrightarrow> fin (init w2)" 
   by auto
 have 42: "\<turnstile> \<diamond>((init w1) \<and> finite \<and> ((init w1) \<and> finite \<longrightarrow>  fin  (init w2))) \<longrightarrow> \<diamond>(fin (init w2))" 
   using "41" DiamondImpDiamond   by blast
 have  5: "\<turnstile> \<diamond>( fin( init w2)) \<longrightarrow>  fin  (init w2)" 
   using DiamondFinImpFin by blast
 have  6: "\<turnstile> (init w) \<and> finite \<and> ((init w) \<and> finite \<longrightarrow>  fin  (init w1)); 
             ((init w1) \<and> finite \<longrightarrow>  fin  (init w2))
               \<longrightarrow>  fin  (init w2)" 
   using   "1" "3" "4" "5" "42"
   using ChopAndCommute FinChopEqvDiamond by fastforce 
 from 6 show ?thesis by fastforce 
qed

lemma ChopRule:
 assumes "\<turnstile>   (init w) \<and> f \<and> finite \<longrightarrow>  fin  (init w1)"  
         "\<turnstile>   (init w1)\<and> f1 \<and> finite \<longrightarrow>  fin  (init w2)"  
 shows   "\<turnstile>   (init w) \<and> (f; f1) \<and> finite \<longrightarrow>  fin  (init w2)"
proof -
 have  1: "\<turnstile> (init w) \<and> (f; f1) \<and> finite \<longrightarrow> ((init w) \<and> f \<and> finite); (f1 \<and> finite)" 
   using  StateAndChopImport 
   by (metis ChopAndFiniteDist inteq_reflection)
 have  2: "\<turnstile> (init w) \<and> f \<and> finite \<longrightarrow>  fin  (init w1) \<and> finite" 
   using assms by auto
 hence 3: "\<turnstile> ((init w) \<and> f \<and> finite); (f1 \<and> finite) \<longrightarrow> ( fin  (init w1) \<and> finite); (f1 \<and> finite)"
   by (rule LeftChopImpChop)  
 have  4: "\<turnstile> ( fin  (init w1) \<and> finite); (f1 \<and> finite) =  \<diamond>((init w1) \<and> f1 \<and> finite)" 
   by (rule FinChopEqvDiamond)  
 have  5: "\<turnstile> (init w1) \<and> f1 \<and> finite \<longrightarrow>  fin  (init w2)" 
   using assms by auto  
 hence 6: "\<turnstile> \<diamond>((init w1) \<and> f1 \<and> finite) \<longrightarrow> \<diamond> (fin  (init w2))" 
   by (rule DiamondImpDiamond)  
 have  7: "\<turnstile> \<diamond>( fin  (init w2)) \<longrightarrow>  fin  (init w2)"  
   using DiamondFinImpFin by blast
 from 1 3 4 6 7 show ?thesis by fastforce
qed

lemma ChopRep:
 assumes "\<turnstile>  (init w) \<and> f \<and> finite \<longrightarrow> f1 \<and>  fin  (init w1)"  
         "\<turnstile>   (init w1) \<and> g \<and> finite \<longrightarrow> g1"  
 shows   "\<turnstile>   (init w) \<and> (f; g) \<and> finite \<longrightarrow> ((f1 \<and> finite); g1)"
proof -
 have  1: "\<turnstile> (init w) \<and> f \<and> finite \<longrightarrow> ((f1 \<and> finite) \<and>  fin  (init w1))" 
   using assms by auto  
 hence 2: "\<turnstile> (init w) \<and> ((f \<and> finite); (g \<and> finite))  \<longrightarrow> 
            ((f1 \<and> finite)\<and>  fin  (init w1)); (g \<and> finite)" 
   using StateAndChopImpChopRule by blast
 have  3: "\<turnstile> ((f1 \<and> finite) \<and>  fin  (init w1)); (g \<and> finite) = 
            (f1 \<and> finite); ((init w1) \<and> (g \<and> finite))" 
   using AndFinChopEqvStateAndChop   by blast
 have  4: "\<turnstile> (init w1)\<and> g \<and> finite \<longrightarrow> g1" 
   using assms by auto   
 hence 5: "\<turnstile> (f1 \<and> finite); ((init w1) \<and> g \<and> finite) \<longrightarrow> (f1 \<and> finite); g1" 
   using RightChopImpChop by blast
 from 2 3 5 show ?thesis using ChopAndFiniteDist by fastforce
qed

lemma ChopRepAndFin:
 assumes "\<turnstile>   (init w) \<and> f \<and> finite \<longrightarrow> f1 \<and>  fin  (init w1)"  
         "\<turnstile>   (init w1) \<and> g \<and> finite \<longrightarrow> g1 \<and>  fin  (init w2)"  
 shows   "\<turnstile>   (init w) \<and> (f; g) \<and> finite \<longrightarrow> ((f1 \<and> finite); g1) \<and>  fin  (init w2)"
proof -
 have  1: "\<turnstile> (init w) \<and> f \<and> finite \<longrightarrow> f1 \<and>  fin  (init w1)" 
   using assms by auto  
 have  2: "\<turnstile> (init w1) \<and> g \<and> finite \<longrightarrow> g1 \<and>  fin  (init w2)" 
   using assms by auto   
 have  3: "\<turnstile> (init w) \<and> (f; g) \<and> finite \<longrightarrow> (f1 \<and> finite); (g1 \<and>  fin  (init w2))" 
   using "1" "2" by (rule ChopRep)  
 have  4: "\<turnstile> (f1 \<and> finite); (g1 \<and>  fin  (init w2)) \<longrightarrow> (f1 \<and> finite); g1" 
   by (rule ChopAndA)  
 have  5: "\<turnstile> (f1 \<and> finite); (g1 \<and>  fin  (init w2)) \<longrightarrow> (f1 \<and> finite);  fin  (init w2)" 
   by (rule ChopAndB)  
 have  6: "\<turnstile> (f1 \<and> finite);  fin  (init w2) \<longrightarrow>  fin  (init w2)" 
   by (rule ChopFinImpFin)  
 from 1 2 3 4 5 6 show ?thesis by (meson Prop12 lift_imp_trans)
qed

lemma TrueChopMoreEqvMore:
 "\<turnstile> #True ; more = more"
by (metis ChopMoreImpMore EmptyImpFinite FiniteAndEmptyEqvEmpty FiniteChopMoreEqvMore 
          LeftChopImpChop Prop09 int_eq_true int_iffI inteq_reflection)

lemma FiniteChopFmoreEqvFmore:
 "\<turnstile> finite;fmore = fmore"
by (metis TrueChopAndFiniteEqvAndFiniteChopFinite TrueChopMoreEqvMore fmore_d_def inteq_reflection)

lemma MoreChopLoop:
 assumes "\<turnstile>   f \<longrightarrow>  fmore ; f"
 shows   "\<turnstile> finite \<longrightarrow> \<not>  f"
proof -
 have   1: "\<turnstile> f \<longrightarrow>  fmore ; f" 
   using assms by auto 
 hence 11: "\<turnstile> \<diamond> (f) \<longrightarrow> \<diamond> (fmore;f)" 
   using DiamondImpDiamond by blast
 have  12: "\<turnstile> \<diamond> (fmore;f) = finite;(fmore;f)" 
   by (simp add: sometimes_d_def)
 have  13: "\<turnstile> finite;(fmore;f) = (finite;fmore);f" 
   by (rule ChopAssoc)
 have  14: "\<turnstile>  \<diamond> (fmore;f) = fmore;f" 
   using FiniteChopFmoreEqvFmore "12" "13"  by (metis int_eq)
 have   2: "\<turnstile>  fmore ; f = \<circle>(\<diamond> f)" 
   using MoreChopEqvNextDiamond by blast  
 have   3: "\<turnstile>  \<diamond> (f) \<longrightarrow> \<circle>(\<diamond> f)"  
   using "11" "14" "2" by fastforce
 hence  4: "\<turnstile> finite \<longrightarrow> \<not> (\<diamond> f)" 
   using NextLoop by blast
 have   5: "\<turnstile> \<not> (\<diamond> f) \<longrightarrow> \<not> f" 
   using NowImpDiamond by fastforce
 from 4 5 show ?thesis using lift_imp_trans by blast
qed

lemma MoreChopContra:
 assumes "\<turnstile>   f \<and> \<not>  g \<longrightarrow> ( fmore ; (f \<and> \<not>  g))"
 shows   "\<turnstile> f  \<and> finite \<longrightarrow> g "
proof -
 have  1: "\<turnstile> f \<and> \<not>  g \<longrightarrow> ( fmore ; (f \<and> \<not>  g))" using assms by auto
 hence 2: "\<turnstile> finite \<longrightarrow> \<not> (f \<and> \<not>  g)" by (rule MoreChopLoop)  
 from 2 show ?thesis by auto 
qed

lemma MoreChopLoopFinite:
 assumes "\<turnstile>   f \<and> finite \<longrightarrow>  fmore ; f"
 shows   "\<turnstile> finite \<longrightarrow> \<not>  f"
proof -
 have   1: "\<turnstile> f \<and> finite \<longrightarrow>  fmore ; f" 
   using assms by auto 
 hence 11: "\<turnstile> \<diamond> (f \<and> finite) \<longrightarrow> \<diamond> (fmore;f)" 
   using DiamondImpDiamond by blast
 have  12: "\<turnstile> \<diamond> (fmore;f) = finite;(fmore;f)" 
   by (simp add: sometimes_d_def)
 have  13: "\<turnstile> finite;(fmore;f) = (finite;fmore);f" 
   by (rule ChopAssoc)
 have  14: "\<turnstile>  \<diamond> (fmore;f) = fmore;f" 
   using FiniteChopFmoreEqvFmore "12" "13"  by (metis int_eq)
 have   2: "\<turnstile>  fmore ; f  = \<circle>(\<diamond> f)" 
   using MoreChopEqvNextDiamond by blast  
 have   3: "\<turnstile>  \<diamond> (f \<and> finite)  \<longrightarrow> \<circle>(\<diamond> f)"  
   using "11" "14" "2" by fastforce
 have 31: "\<turnstile> \<diamond> (f \<and> finite) = ((\<diamond> f) \<and> finite)"
   by (metis (no_types, lifting) "3" ChopAndB ChopAndNotChopImp DiamondDiamondEqvDiamond 
       DiamondIntroC FiniteChopFiniteEqvFinite FiniteChopInfEqvInf Prop11 Prop12 finite_d_def 
       inteq_reflection sometimes_d_def)
 have 32: "\<turnstile> (\<diamond> f) \<and> finite \<longrightarrow> \<circle>(\<diamond> f)"
   using "3" "31" by fastforce
 hence  4: "\<turnstile> finite \<longrightarrow> \<not> (\<diamond> f)" 
   by (metis (no_types, lifting) DiamondIntro FiniteChopInfEqvInf InfEqvNotFinite Prop09 
      finite_d_def int_simps(15) int_simps(32) inteq_reflection sometimes_d_def)
 have   5: "\<turnstile> \<not> (\<diamond> f) \<longrightarrow> \<not> f" 
   by (simp add: NowImpDiamond)
 from 4 5 show ?thesis using lift_imp_trans by fastforce
qed

lemma MoreChopEqvFmoreOrInf:
 "\<turnstile> more ; f = ( (fmore;f) \<or> inf)"
 by (metis AndInfChopEqvAndInf MoreAndInfEqvInf OrChopEqv OrFiniteInf fmore_d_def int_eq)

lemma MoreChopLoopFiniteB:
 assumes "\<turnstile>   f  \<longrightarrow>  more ; f"
 shows   "\<turnstile> finite \<longrightarrow> \<not>  f"
proof -
 have   1: "\<turnstile> f \<longrightarrow>  more ; f" 
   using assms by auto 
 have  10: "\<turnstile> f  \<longrightarrow> (fmore;f) \<or> inf" 
   using MoreChopEqvFmoreOrInf  assms by fastforce
 hence 100: "\<turnstile> f  \<and> finite \<longrightarrow> (fmore;f)"
   by (simp add: Prop13 finite_d_def)
 hence 11: "\<turnstile> \<diamond> (f \<and> finite) \<longrightarrow> \<diamond> (fmore;f)" 
   using DiamondImpDiamond by blast
 have  12: "\<turnstile> \<diamond> (fmore;f) = finite;(fmore;f)" 
   by (simp add: sometimes_d_def)
 have  13: "\<turnstile> finite;(fmore;f) = (finite;fmore);f" 
   by (rule ChopAssoc)
 have  14: "\<turnstile>  \<diamond> (fmore;f) = fmore;f" 
   using FiniteChopFmoreEqvFmore "12" "13"  by (metis int_eq)
 have   2: "\<turnstile>  fmore ; f  = \<circle>(\<diamond> f)" 
   using MoreChopEqvNextDiamond by blast  
 have   3: "\<turnstile>  \<diamond> (f \<and> finite)  \<longrightarrow> \<circle>(\<diamond> f)"  
   using "11" "14" "2" by fastforce
 have 31: "\<turnstile> \<diamond> (f \<and> finite) = ((\<diamond> f) \<and> finite)"
   by (metis (no_types, opaque_lifting) ChopAndA ChopAndB ChopAndNotChopImp FiniteChopFiniteEqvFinite
       FiniteChopInfEqvInf Prop11 Prop12 finite_d_def inteq_reflection sometimes_d_def)
 have 32: "\<turnstile> (\<diamond> f) \<and> finite \<longrightarrow> \<circle>(\<diamond> f)"
   using "3" "31" by fastforce
 hence  4: "\<turnstile> finite \<longrightarrow> \<not> (\<diamond> f)" 
   by (metis (no_types, lifting) DiamondIntro FiniteChopInfEqvInf InfEqvNotFinite Prop09 
       finite_d_def int_simps(15) int_simps(32) inteq_reflection sometimes_d_def)
 have   5: "\<turnstile> \<not> (\<diamond> f) \<longrightarrow> \<not> f" 
   by (simp add: NowImpDiamond)
 from 4 5 show ?thesis using lift_imp_trans by fastforce
qed

lemma MoreChopContraFinite:
 assumes "\<turnstile>   (f \<and> \<not>  g) \<and> finite \<longrightarrow> ( fmore ; (f \<and> \<not>  g))"
 shows   "\<turnstile> f \<and> finite \<longrightarrow> g "
proof -
 have  1: "\<turnstile> (f \<and> \<not>  g) \<and> finite \<longrightarrow> ( fmore ; (f \<and> \<not>  g))" using assms by auto
 hence 2: "\<turnstile> finite \<longrightarrow> \<not> (f \<and> \<not>  g)" using MoreChopLoopFinite by (simp add: MoreChopLoopFinite)
 from 2 show ?thesis by (simp add: Valid_def)
qed

lemma MoreChopContraFiniteB:
 assumes "\<turnstile>   (f \<and> \<not>  g)  \<longrightarrow> ( more ; (f \<and> \<not>  g))"
 shows   "\<turnstile> f \<and> finite \<longrightarrow> g "
proof -
 have  1: "\<turnstile> (f \<and> \<not>  g)  \<longrightarrow> ( more ; (f \<and> \<not>  g))" using assms by auto
 hence 2: "\<turnstile> finite \<longrightarrow> \<not> (f \<and> \<not>  g)" using MoreChopLoopFinite by (simp add: MoreChopLoopFiniteB)
 from 2 show ?thesis by (simp add: Valid_def )
qed

lemma ChopLoop:
 assumes "\<turnstile>   f \<longrightarrow> g;f" 
         "\<turnstile> g \<longrightarrow>  fmore"
 shows   "\<turnstile> finite \<longrightarrow> \<not>  f"
proof -
have  1: "\<turnstile> f \<longrightarrow> g; f" using assms by auto  
have  2: "\<turnstile> g \<longrightarrow>  fmore" using assms by auto  
hence 3: "\<turnstile> g; f \<longrightarrow>  fmore ; f" by (rule LeftChopImpChop)  
have  4: "\<turnstile> f \<longrightarrow>  fmore ; f" using "1" "3" by fastforce 
from 4 show ?thesis using MoreChopLoop by auto
qed

lemma ChopLoopB:
 assumes "\<turnstile>   f \<longrightarrow> g;f" 
         "\<turnstile> g \<longrightarrow>  more"
 shows   "\<turnstile> finite \<longrightarrow> \<not>  f"
proof -
have  1: "\<turnstile> f \<longrightarrow> g; f" using assms by auto  
have  2: "\<turnstile> g \<longrightarrow>  more" using assms by auto  
hence 3: "\<turnstile> g; f \<longrightarrow>  more ; f" by (rule LeftChopImpChop)  
have  4: "\<turnstile> f \<longrightarrow>  more ; f" using "1" "3" by fastforce 
from 4 show ?thesis using MoreChopLoopFiniteB by auto
qed

lemma ChopContra:
 assumes  "\<turnstile>   f \<and> \<not>  g \<longrightarrow> h; f \<and> \<not> (h; g)"
          "\<turnstile> h \<longrightarrow>  fmore" 
 shows    "\<turnstile> f  \<and> finite \<longrightarrow> g "
proof -
have  1: "\<turnstile> f \<and> \<not>  g \<longrightarrow> h; f \<and> \<not> (h; g)" using assms by auto  
have  2: "\<turnstile> h \<longrightarrow>  fmore" using assms by auto   
have  3: "\<turnstile> h; f \<and> \<not> (h; g) \<longrightarrow> h; (f \<and> \<not>  g)" by (rule ChopAndNotChopImp)  
have  4: "\<turnstile> h; (f \<and> \<not>  g) \<longrightarrow>  fmore ; (f \<and> \<not>  g)" using "2" by (rule LeftChopImpChop)  
have  5: "\<turnstile> f \<and> \<not>  g \<longrightarrow>  fmore ; (f \<and> \<not>  g)" using "1" "3" "4" by fastforce  
from 5 show ?thesis using MoreChopContra by auto
qed

lemma ChopContraB:
 assumes  "\<turnstile> f \<and> \<not>  g \<longrightarrow> h; f \<and> \<not> (h; g)"
          "\<turnstile> h \<longrightarrow>  more" 
 shows    "\<turnstile> f \<and> finite \<longrightarrow> g "
proof -
have  1: "\<turnstile> f \<and> \<not>  g \<longrightarrow> h; f \<and> \<not> (h; g)" using assms by auto  
have  2: "\<turnstile> h \<longrightarrow>  more" using assms by auto   
have  3: "\<turnstile> h; f \<and> \<not> (h; g) \<longrightarrow> h; (f \<and> \<not>  g)" by (rule ChopAndNotChopImp)  
have  4: "\<turnstile> h; (f \<and> \<not>  g) \<longrightarrow>  more ; (f \<and> \<not>  g)" using "2" by (rule LeftChopImpChop)  
have  5: "\<turnstile> f \<and> \<not>  g \<longrightarrow>  more ; (f \<and> \<not>  g)" using "1" "3" "4" by fastforce  
from 5 show ?thesis using MoreChopContraFiniteB by auto
qed


subsection \<open>Properties of Halt\<close>

lemma WnextAndMoreEqvNext:
 "\<turnstile> (wnext f \<and> more) = \<circle> f" 
proof -
 have 1: "\<turnstile> wnext f = (empty \<or> \<circle> f)" 
   by (simp add: WnextEqvEmptyOrNext)
 have 2: "\<turnstile> \<circle> f \<longrightarrow> more" 
   by (metis DiIntro LeftChopImpMoreRule di_d_def more_d_def next_d_def)
 have 3: "\<turnstile> ((empty \<or> \<circle> f) \<and> more) = \<circle> f " 
   unfolding empty_d_def using "2" by auto
 show ?thesis by (metis "1" "3" int_eq)
qed

lemma BoxStateAndEmptyEqvStateAndEmpty:
 "\<turnstile> (\<box>(empty = (init w)) \<and> empty) = ((init w) \<and> empty) "
proof -
 have 1: "\<turnstile> ((empty = (init w)) \<and> empty) = ((init w) \<and> empty)"
   by force
 have 2: "\<turnstile> (\<box>(empty = (init w)) \<and> empty) \<longrightarrow> ((init w) \<and> empty)"
   using BoxElim by fastforce
 have 3: "\<turnstile> ((init w) \<and> empty) \<longrightarrow> (\<box>(empty = (init w)) \<and> empty)"
   using BoxEqvAndEmptyOrNextBox by fastforce 
 show ?thesis
   by (simp add: "2" "3" int_iffI)
qed

lemma BoxEmptyEqvIStateqvEmptyAndStateOrNotStateNext:
 "\<turnstile> \<box>(empty = (init w)) = ((empty \<and> init w) \<or> (\<not>(init w) \<and> \<circle>(\<box>(empty = (init w))))) "
proof -
 have  1: "\<turnstile> \<box>(empty = (init w)) = 
            ((\<box>(empty = (init w)) \<and> empty) \<or> (\<box>(empty = (init w)) \<and> more))" 
   by (auto simp: empty_d_def)
 have  2: "\<turnstile> (\<box>(empty = (init w)) \<and> empty) = ((init w) \<and> empty) " 
   using BoxStateAndEmptyEqvStateAndEmpty by blast
 have  3: "\<turnstile> \<box>(empty = (init w))  = ((empty = (init w)) \<and> wnext(\<box>(empty = (init w)))) " 
   using BoxEqvAndWnextBox by blast
 hence 4: "\<turnstile> (\<box>(empty = (init w)) \<and> more)  = 
               (((empty = (init w)) \<and> more) \<and> (wnext(\<box>(empty = (init w))) \<and> more)) " 
   by auto
 have  5: "\<turnstile> ((empty = (init w)) \<and> more) = (\<not>(init w) \<and> more) " 
   by (auto simp: empty_d_def)
 have  6: "\<turnstile> (wnext(\<box>(empty = (init w))) \<and> more) = \<circle>(\<box>(empty = (init w)))  " 
   using WnextAndMoreEqvNext by metis
 have  7: "\<turnstile> (\<box>(empty = (init w)) \<and> more) = 
             ((\<not> (init w) \<and> more) \<and> (wnext(\<box>(empty = (init w))) \<and> more) )" 
   using "4" "5" by fastforce
 have  8: "\<turnstile> ((\<not> (init w) \<and> more) \<and> (wnext(\<box>(empty = (init w))) \<and> more) ) =
             ((\<not> (init w)) \<and> (wnext(\<box>(empty = (init w))) \<and> more) )" 
   by auto
 have  9: "\<turnstile> ((\<not> (init w)) \<and> (wnext(\<box>(empty = (init w))) \<and> more) ) =
              ((\<not> (init w)) \<and> \<circle>(\<box>(empty = (init w)))) " 
   using "8" "6" by auto 
 have 10: "\<turnstile> \<box>(empty = (init w)) = (((init w) \<and> empty) \<or> (\<box>(empty = (init w)) \<and> more) ) "
   using "1" "2" by fastforce
 show ?thesis 
   using "10" "7" "9" by fastforce
qed

lemma HaltStateEqvIfStateThenEmptyElseNext:
 "\<turnstile>  halt(  init w) =  if\<^sub>i  (init w)  then   empty   else ( \<circle>( halt ( init w))) "
by (metis BoxEmptyEqvIStateqvEmptyAndStateOrNotStateNext halt_d_def ifthenelse_d_def inteq_reflection 
    lift_and_com) 

lemma HaltChopEqv:
 "\<turnstile>  ((halt ( init w)) ; f) =  (if\<^sub>i  (init w)  then ( f ) else (\<circle>( (halt ( init w)); f))) "
proof -
 have  1: "\<turnstile>  halt(init w) =  
             (if\<^sub>i  (init w)  then   empty   else ( \<circle>( halt ( init w)))) " 
   by (rule HaltStateEqvIfStateThenEmptyElseNext)
 hence 2: "\<turnstile>  ((halt(init w));f) =  
             (if\<^sub>i  (init w)  then   (empty;f)   else ( \<circle>( halt ( init w));f)) " 
   by (rule IfChopEqvRule)
 have  3: "\<turnstile>  empty ; f = f " 
   by (rule EmptyChop)
 have  4: "\<turnstile> (\<circle> (halt ( init w))); f = \<circle>( halt ( init w); f)" 
   by (rule NextChop)
 from 2 3 4 show ?thesis by (metis inteq_reflection)
qed

lemma AndHaltChopImp:
 "\<turnstile>   init w \<and> ( halt ( init w); f) \<longrightarrow> f "
proof -
 have 1: "\<turnstile> halt ( init w); f =  if\<^sub>i  (init w)  then  f  else ( \<circle>( halt ( init w); f)) " 
  by (rule HaltChopEqv)
 have 2: "\<turnstile> init w \<and> if\<^sub>i  (init w)  then  f  else ( \<circle>( halt ( init w); f)) \<longrightarrow> f" 
   by (auto simp: ifthenelse_d_def)
 from 1 2 show ?thesis by fastforce
qed
 
lemma NotAndHaltChopImpNext:
 "\<turnstile>  \<not> ( init w) \<and> ( halt  (init w); f) \<longrightarrow> \<circle>( halt ( init w); f) "
proof -
 have 1: "\<turnstile>  halt ( init w); f =  if\<^sub>i  (init w)  then  f  else ( \<circle>( halt ( init w); f)) " 
   by (rule HaltChopEqv)
 have 2: "\<turnstile> \<not> ( init w) \<and> if\<^sub>i  (init w)  then  f  else ( \<circle>( halt ( init w); f)) \<longrightarrow> 
            \<circle>( halt ( init w); f)"  
   by (auto simp: ifthenelse_d_def) 
 from 1 2 show ?thesis by fastforce
qed

lemma NotAndHaltChopImpSkipYields:
 "\<turnstile>  \<not> ( init w) \<and> ( halt ( init w); f) \<longrightarrow>  skip  yields ( halt  (init w); f) "
proof -
 have 1: "\<turnstile> \<not> ( init w) \<and> ( halt ( init w); f) \<longrightarrow> \<circle>( halt ( init w); f) " 
   by (rule NotAndHaltChopImpNext)
 have 2: "\<turnstile> \<circle>( halt ( init w); f) \<longrightarrow>  skip  yields ( halt ( init w); f) " 
   by (rule NextImpSkipYields)
 from 1 2 show ?thesis by fastforce
qed

lemma FiniteChopAndEmptyEqvChopAndEmpty:
 " \<turnstile> ((finite;(f \<and> empty)) \<and> g) = ((g \<and> finite);(f\<and> empty))  "
proof -
 have 1: "\<turnstile> g \<and> finite;(f \<and> empty) \<longrightarrow> fin f"
   by (metis ChopAndA DiamondFin FinAndEmpty Prop01 Prop05 inteq_reflection sometimes_d_def)
 have 2: "\<turnstile> g \<and> finite;(f \<and> empty) \<longrightarrow> (finite \<and> g) \<and> fin f"
   using "1"  by (metis (no_types, lifting) ChopAndB ChopEmpty Prop10 Prop12 int_iffD1 
                  inteq_reflection) 
 have 3: "\<turnstile> ((finite;(f \<and> empty)) \<and> g) \<longrightarrow> ((g \<and> finite);(f\<and> empty))" 
   using "2" using AndFinEqvChopAndEmpty by fastforce
 have 4: "\<turnstile> ((g \<and> finite);(f\<and> empty)) \<longrightarrow> ((finite;(f \<and> empty)) \<and> g)" 
   by (metis AndChopB ChopAndB ChopEmpty Prop12 inteq_reflection)
 from 3 4 show ?thesis by fastforce
qed

lemma WprevEqvEmptyOrPrev:
 "\<turnstile> wprev f = (empty \<or> prev f)"
using nlength_eq_enat_nfiniteD by (auto simp add: Valid_def itl_defs zero_enat_def)

lemma NotChopSkipEqvMoreAndNotChopSkip:
 "\<turnstile> (\<not> f);skip = (more \<and> \<not>(f;skip))"
proof -
 have 1: "\<turnstile> wprev f = (empty \<or> prev f)" using WprevEqvEmptyOrPrev by auto
 hence 2: "\<turnstile> (\<not>(wprev f)) = (\<not>(empty \<or> prev f)) " by auto
 have 3: "\<turnstile> \<not>(wprev f) = ((\<not> f);skip) " by (simp add: wprev_d_def prev_d_def)
 have 31: "\<turnstile> (empty \<or> prev f) = (empty \<or> (f;skip))" by (simp add: prev_d_def)
 have 32: "\<turnstile> (empty \<or> (f;skip)) = (\<not>more \<or> \<not>\<not>(f;skip))" by (simp add: empty_d_def)
 have 33: "\<turnstile> (\<not>more \<or> \<not>\<not>(f;skip)) = (\<not>(more \<and> \<not>(f;skip)))" by fastforce
 have 34: "\<turnstile> (empty \<or> prev f) = (\<not>(more \<and> \<not>(f;skip)))" using "31" "32" "33" by (metis int_eq)
 have 4: "\<turnstile> \<not>(empty \<or> prev f) = (more \<and> \<not>(f;skip))" using "34" by fastforce
 from 2 3 4 show ?thesis by fastforce
qed

lemma HaltChopImpNotHaltChopNot:
 "\<turnstile>    halt ( init w); f \<and> finite \<longrightarrow> \<not> ( halt ( init w); (\<not>  f)) "
proof -
 have   1: "\<turnstile>  halt  (init w); f =  if\<^sub>i  (init w)  then  f  else ( \<circle>( halt ( init w); f)) " 
   by (rule HaltChopEqv)
 have   2: "\<turnstile> if\<^sub>i  (init w)  then  f  else ( \<circle>( halt ( init w); f)) \<longrightarrow> 
              ( ((init w) \<longrightarrow> f) \<and> ( \<not>(init w) \<longrightarrow> ( \<circle>( halt ( init w); f))))"  
   by (rule IfThenElseImp) 
 have   3: "\<turnstile>  halt  (init w); (\<not>f) =  
               if\<^sub>i  (init w)  then  (\<not>f)  else ( \<circle>( halt ( init w); (\<not>f))) " 
   by (rule HaltChopEqv)
 have   4: "\<turnstile> if\<^sub>i  (init w)  then  (\<not>f)  else ( \<circle>( halt ( init w); (\<not>f))) \<longrightarrow> 
              ( ((init w) \<longrightarrow> \<not>f) \<and> ( \<not>(init w) \<longrightarrow> ( \<circle>( halt ( init w); (\<not>f)))))"  
   by (rule IfThenElseImp)
 have   5: "\<turnstile> halt  (init w); f \<and> halt  (init w); (\<not>f) \<longrightarrow>
            ( ((init w) \<longrightarrow> f) \<and> ( \<not>(init w) \<longrightarrow> ( \<circle>( halt ( init w); f)))) \<and>
            ( ((init w) \<longrightarrow> \<not>f) \<and> ( \<not>(init w) \<longrightarrow> ( \<circle>( halt ( init w); (\<not>f))))) " 
   using "1" "2" "3" "4" by fastforce
 have   6: "\<turnstile> ( ((init w) \<longrightarrow> f) \<and> ( \<not>(init w) \<longrightarrow> ( \<circle>( halt ( init w); f)))) \<and>
            ( ((init w) \<longrightarrow> \<not>f) \<and> ( \<not>(init w) \<longrightarrow> ( \<circle>( halt ( init w); (\<not>f))))) \<longrightarrow>
            ( \<circle>( halt ( init w); f)) \<and> ( \<circle>( halt ( init w); (\<not>f))) " 
   by auto
 have   7: "\<turnstile> halt  (init w); f \<and> halt  (init w); (\<not>f) \<longrightarrow>
            ( \<circle>( halt ( init w); f)) \<and> ( \<circle>( halt ( init w); (\<not>f))) " 
   using "5" "6" lift_imp_trans by blast
 have   8: "\<turnstile> (( \<circle>( halt ( init w); f)) \<and> ( \<circle>( halt ( init w); (\<not>f)))) =
             \<circle> (halt ( init w); f \<and> halt ( init w); (\<not>f)) " 
   using NextAndEqvNextAndNext by fastforce
 have   9: "\<turnstile> halt  (init w); f \<and> halt  (init w); (\<not>f) \<longrightarrow> 
              \<circle> (halt ( init w); f \<and> halt ( init w); (\<not>f)) " 
   using "7" "8" by fastforce
 hence 10: "\<turnstile> finite \<longrightarrow> \<not>(halt  (init w); f \<and> halt  (init w); (\<not>f)) " 
   using NextLoop by blast
 from 10 show ?thesis by auto
qed

lemma HaltChopImpHaltYields:
 "\<turnstile>    halt ( init w); f \<and> finite \<longrightarrow> ( halt ( init w)) yields  f "
proof -
 have 1: "\<turnstile>  halt ( init w); f \<and> finite \<longrightarrow> \<not> ( halt ( init w); (\<not>  f)) " 
   by (rule HaltChopImpNotHaltChopNot)
 from 1 show ?thesis by (simp add: yields_d_def)
qed

lemma HaltChopAnd:
 "\<turnstile> ( halt (init w)); f \<and> ( halt  (init w)); g \<and> finite\<longrightarrow> ( halt ( init w)); (f \<and> g) "
proof -
 have  1: "\<turnstile> ( halt (init w)); g \<and> finite \<longrightarrow> ( halt (init w))  yields  g " 
   by (rule HaltChopImpHaltYields)
 hence 2: "\<turnstile> ( halt  (init w)); f \<and> ( halt  (init w)); g \<and> finite \<longrightarrow>
             ( halt  (init w)); f \<and> ( halt  (init w)) yields  g "
   by auto
 have  3: "\<turnstile> ( halt  (init w)); f \<and> ( halt  (init w)) yields  g\<longrightarrow>
             ( halt  (init w)); (f \<and> g) " 
   by (rule ChopAndYieldsImp)
 from 2 3 show ?thesis by fastforce
qed

lemma HaltAndChopAndHaltChopImpHaltAndChopAnd:
 "\<turnstile> (halt (init w) \<and> f);f1 \<and> (halt (init w); g) \<and> finite \<longrightarrow> ( halt ( init w) \<and> f); (f1 \<and> g) "
proof -
 have   1: "\<turnstile> f1 \<longrightarrow> \<not>  g \<or>  (f1 \<and> g) " 
   by auto
 hence  2: "\<turnstile> ( halt  (init w) \<and> f); f1 \<longrightarrow>  
             ( halt  (init w) \<and> f); (\<not>  g) \<or>  (( halt  (init w) \<and> f); (f1 \<and> g)) " 
   by (rule ChopOrImpRule)
 have   3: "\<turnstile> ( halt  (init w) \<and> f); (\<not>  g) \<longrightarrow>  halt  (init w); (\<not>  g) " 
   by (rule AndChopA)
 have  31: "\<turnstile> ( halt  (init w) \<and> f); f1 \<longrightarrow> 
             halt  (init w); (\<not>  g) \<or>  (( halt  (init w) \<and> f); (f1 \<and> g)) " 
   using "2""3" by fastforce
 have   4: "\<turnstile>  halt  (init w); g \<and> finite \<longrightarrow> \<not> ( halt  (init w); (\<not>  g)) " 
   by (rule HaltChopImpNotHaltChopNot)
 hence 41: "\<turnstile> ( halt  (init w); (\<not>  g)) \<and> finite \<longrightarrow> \<not>(halt  (init w); g)" 
   by auto
 have  42: "\<turnstile> ( halt  (init w) \<and> f); f1 \<and> finite \<longrightarrow> 
             \<not>( halt  (init w); g) \<or>  (( halt  (init w) \<and> f); (f1 \<and> g)) " 
   using "31" "41" by fastforce
 from 42 show ?thesis by auto
qed

lemma HaltImpBoxYields:
 "\<turnstile>   ( halt  (init w)); f \<and> finite \<longrightarrow> (\<box>(\<not> ( init w)))  yields  (( halt  (init w)); f)"
proof -
 have   1: "\<turnstile> (\<box> (\<not>  (init w))); (\<not> ( halt  (init w); f)) \<longrightarrow>  di  (\<box> (\<not>  (init w))) " 
   by (rule ChopImpDi)
 have   2: "\<turnstile> \<box> (\<not>  (init w)) \<longrightarrow> \<not>  (init w) " 
   by (rule BoxElim)
 hence  3: "\<turnstile>  di (\<box> (\<not>  (init w))) \<longrightarrow>  di (\<not>  (init w)) " 
   by (rule DiImpDi)
 have   4: "\<turnstile>  di   (init (\<not> w)) =   (init (\<not>w)) " 
   by (rule DiState)
 have  41: "\<turnstile>  (init (\<not> w)) =  (\<not> (init w))" 
   using Initprop(2) by fastforce
 have  42: "\<turnstile>  di  (\<not> (init  w)) =   (\<not>(init w)) " 
   using "4" "41" by (metis inteq_reflection)
 have   5: "\<turnstile> ((\<box>(\<not>  (init w))); (\<not> ( halt  (init w); f))) \<longrightarrow> \<not> ( init w) " 
   using "1" "2" "42" using "3" by fastforce
 hence 51: "\<turnstile> ( halt  (init w); f) \<and> ((\<box>(\<not>  (init w))); (\<not> ( halt  (init w); f))) \<longrightarrow>
            ( halt  (init w); f) \<and> \<not> ( init w) " 
   by fastforce
 have   6: "\<turnstile>  halt  (init w); f = if\<^sub>i (init w)  then  f  else  (\<circle>( halt  (init w); f)) " 
   by (rule HaltChopEqv)
 hence 61: "\<turnstile> (halt  (init w); f \<and> \<not> ( init w)) = 
              ((if\<^sub>i (init w)  then  f  else  (\<circle>( halt  (init w); f))) \<and> \<not> ( init w)) " 
   using "6" by auto
 have  62: "\<turnstile> (if\<^sub>i (init w)  then  f  else  (\<circle>( halt  (init w); f))) \<and> 
              \<not> ( init w) \<longrightarrow> (\<circle>( halt  (init w); f)) "  
   by (auto simp: ifthenelse_d_def) 
 have  63: "\<turnstile> halt  (init w); f \<and> \<not> ( init w) \<longrightarrow> (\<circle>( halt  (init w); f))  " 
   using "61" "62" by fastforce
 have   7: "\<turnstile> ( halt  (init w); f) \<and> (\<box>(\<not>  (init w))); (\<not> ( halt  (init w); f)) \<longrightarrow>
             \<circle>(( halt  (init w)); f) " 
   using "51" "63" using lift_imp_trans by blast
 have   8: "\<turnstile> \<box> (\<not> (init w)) \<longrightarrow>  empty  \<or>  \<circle>(\<box>(\<not>(  init w))) "
   by (metis BoxImpYields WeakNextBoxImpMoreYields WnextEqvEmptyOrNext fmore_d_def int_eq) 
 hence  9: "\<turnstile> ((\<box> (\<not> ( init w))); (\<not> ( halt  (init w); f))) \<longrightarrow>  
            \<not> ( halt  (init w); f) \<or>  \<circle>((\<box>(\<not>  (init w))); (\<not> ( halt  (init w); f))) " 
   by (rule EmptyOrNextChopImpRule)
 hence 10: "\<turnstile> (( halt  (init w)); f) \<and> (\<box> (\<not>  (init w))); (\<not> ( halt  (init w); f)) \<longrightarrow>  
            \<circle>((\<box>(\<not>  (init w))); (\<not> ( halt  (init w); f))) " 
   by fastforce
 have  11: "\<turnstile> ( halt  (init w)); f \<and> (\<box> (\<not>  (init w))); (\<not> ( halt  (init w); f)) \<longrightarrow> 
           \<circle>(( halt  (init w)); f) \<and> \<circle>((\<box>(\<not>  (init w))); (\<not> ( halt  (init w); f))) " 
   using "7" "10" by fastforce
 have  12: "\<turnstile> \<circle>(( halt  (init w)); f) \<and> \<circle>((\<box>(\<not>  (init w))); (\<not> ( halt  (init w); f)))
           \<longrightarrow>  \<circle>((( halt (init w)); f) \<and> ((\<box>(\<not>  (init w))); (\<not> ( halt  (init w); f)))) " 
   using NextAndEqvNextAndNext by fastforce
 have  13: "\<turnstile> ( halt  (init w)); f \<and> (\<box> (\<not>  (init w))); (\<not> ( halt  (init w); f)) \<longrightarrow>  
             \<circle>((( halt  (init w)); f) \<and> ((\<box>(\<not>  (init w))); (\<not> ( halt  (init w); f)))) " 
   using "11" "12" by fastforce
 hence 14: "\<turnstile> finite \<longrightarrow> \<not> (( halt  (init w)); f \<and> (\<box> (\<not>  (init w))); (\<not> ( halt  (init w); f))) " 
   using NextLoop by blast
 hence 15: "\<turnstile> ( halt  (init w)); f \<and> finite \<longrightarrow> \<not> ((\<box> (\<not>  (init w))); (\<not> ( halt  (init w); f))) " 
   by auto
 from 15 show ?thesis by (simp add: yields_d_def)
qed

subsection \<open>Properties of Groups of chops\<close>

lemma NestedChopImpChop:
 assumes "\<turnstile>   init w \<and> f \<longrightarrow> g; (init w1 \<and> f1) "
         "\<turnstile> init w1 \<and> f1 \<longrightarrow> g1; (init w2 \<and> f2) "
 shows   "\<turnstile> init w \<and> f \<longrightarrow> g; (g1; (init w2 \<and> f2)) "
proof -
 have  1: "\<turnstile> init w \<and> f \<longrightarrow> g; (init w1 \<and> f1) " using assms(1) by auto
 have  2: "\<turnstile> init w1 \<and> f1 \<longrightarrow> g1; (init w2 \<and> f2) " using assms(2) by auto
 hence 3: "\<turnstile> g; (init w1 \<and> f1) \<longrightarrow> g; (g1; (init w2 \<and> f2)) " by (rule RightChopImpChop)
 from 1 3 show ?thesis by fastforce
qed
 

end


