(*  Title     : A Shallow Encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

chapter \<open>Shallow embedding of Finite and Infinite ITL\<close>

section \<open>Semantics\<close>   

theory Semantics
imports  NELList_Extras "HOL-TLA.Intensional" 
begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                     isar_proofs=false,
                    provers="
   cvc4 z3 e spass vampire 
"]
*)

(*
declare [[show_types]]

declare [[show_consts]]
*)

text \<open> 
  This theory mechanises a \emph{shallow} embedding of Finite and Infinite ITL using the
  \<open>NELList\<close> and \<open>Intensional\<close> theories. 
\<close>

subsection "Types of Formulas"

text \<open>
  To mechanise the Finite and Infinite ITL semantics, the following type abbreviations are used:
\<close>

type_synonym 'a intervals = "'a nellist"

type_synonym ('a,'b) formfun    = "'a intervals \<Rightarrow> 'b"
type_synonym 'a formula         = "('a,bool) formfun"
type_synonym ('a,'b) stfun      = "'a \<Rightarrow> 'b"
type_synonym 'a stpred          = "('a,bool) stfun"

instance 
 "fun" :: (type,type) world ..

instance
 "prod" :: (type,type) world ..

instance
 "sum" :: (type,type) world ..

instance
 "nellist" :: (type) world ..

text \<open>
  Pair, function, sum, and interval are instantiated to be of type class world.
  This allows use of the lifted Intensional logic for formulas, and 
  standard logical connectives can therefore be used.
\<close>

subsection \<open>Semantics of ITL\<close>

text \<open> The semantics of ITL is defined. \<close>

definition skip_d :: "('a ::world) formula "
where " skip_d \<equiv> (\<lambda>s. nlength s = (enat (Suc 0)))"

definition chop_d :: " ('a ::world) formula \<Rightarrow> ('a ::world) formula \<Rightarrow> ('a ::world) formula"
where "chop_d F1 F2 \<equiv>
        (\<lambda>s. 
            (\<exists>n\<le>nlength s. ( (ntaken n s)  \<Turnstile> F1)  \<and> ((ndropn n s) \<Turnstile> F2))
            \<or> (\<not> nfinite s \<and> (s \<Turnstile> F1) )
       )" 

definition current_val_d :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun"
where "current_val_d f = (\<lambda>s. ( (nfirst s) \<Turnstile> f) )"

definition next_val_d :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun"
where "next_val_d f \<equiv> 
   (\<lambda> s. ( if nlength s \<noteq> (enat 0) then ((nnth s 1) \<Turnstile> f) else (\<some> (x::'b ). x=x) )
   )" 

definition fin_val_d :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun"
where "fin_val_d f \<equiv> \<lambda> s. (( if nfinite s then  ( (nlast s)  \<Turnstile> f) else (\<some> (x::'b ). x=x)))"

definition penult_val_d :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun"
where "penult_val_d f \<equiv> 
    (\<lambda> s. 
     (if nfinite s 
      then (if nlength s \<noteq> (enat 0) 
            then ((nnth s (the_enat (epred(nlength s)))) \<Turnstile> f) 
            else (\<some> (x::'b ). x=x)) 
      else (\<some> (x::'b ). x=x)
     )
    )" 


syntax
 "_skip_d"         :: "lift"                ("(skip)")
 "_chop_d"         :: "[lift,lift] \<Rightarrow> lift" ("(_;_)" [84,84] 83) 
 "_current_val_d"  :: "lift \<Rightarrow> lift"        ("($_)" [100] 99)
 "_next_val_d"     :: "lift \<Rightarrow> lift"        ("(_$)" [100] 99)
 "_fin_val_d"      :: "lift \<Rightarrow> lift"        ("(!_)" [100] 99)
 "_penult_val_d"   :: "lift \<Rightarrow> lift"        ("(_!)" [100] 99)
 "TEMP"            :: "lift \<Rightarrow> 'b"          ("(TEMP _)")

syntax (ASCII)
 "_skip_d"         :: "lift"                ("(skip)")
 "_chop_d"         :: "[lift,lift] \<Rightarrow> lift" ("(_;_)" [84,84] 83)
 "_current_val_d"  :: "lift \<Rightarrow> lift"        ("($_)" [100] 99)
 "_next_val_d"     :: "lift \<Rightarrow> lift"        ("(_$)" [100] 99)
 "_fin_val_d"      :: "lift \<Rightarrow> lift"        ("(!_)" [100] 99)
 "_penult_val_d"   :: "lift \<Rightarrow> lift"        ("(_!)" [100] 99)

translations
 "_skip_d"        \<rightleftharpoons> "CONST skip_d"
 "_chop_d"        \<rightleftharpoons> "CONST chop_d"
 "_current_val_d" \<rightleftharpoons> "CONST current_val_d"
 "_next_val_d"    \<rightleftharpoons> "CONST next_val_d"
 "_fin_val_d"     \<rightleftharpoons> "CONST fin_val_d"
 "_penult_val_d"  \<rightleftharpoons> "CONST penult_val_d" 
 "TEMP F"         \<rightharpoonup> "(F:: (_ intervals) \<Rightarrow> _)"


subsection "Abbreviations"

text \<open> Some standard temporal abbreviations, with their concrete syntax. \<close>

definition infinite_d :: "('a ::world) formula"
where "infinite_d \<equiv> LIFT(#True;#False) "

syntax
  "_infinite_d"  :: "lift"         ("inf")

syntax (ASCII)
  "_infinite_d"  :: "lift"         ("inf")

translations 
  "_infinite_d"  \<rightleftharpoons> "CONST infinite_d"

definition finite_d :: "('a ::world) formula"
where "finite_d \<equiv> LIFT(\<not>(inf)) "

syntax
 "_finite_d"    :: "lift"         ("finite")

syntax (ASCII)
  "_finite_d"    :: "lift"         ("finite")

translations 
  "_finite_d"    \<rightleftharpoons> "CONST finite_d"


definition schop_d :: "('a::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula" 
where "schop_d F1 F2 \<equiv> LIFT((F1 \<and> finite);F2)"

definition sometimes_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "sometimes_d F \<equiv> LIFT(finite;F)"

definition di_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "di_d F \<equiv> LIFT(F;#True)" 

definition da_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "da_d F \<equiv> LIFT(finite;(F;#True))" 

definition next_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "next_d F \<equiv> LIFT(skip;F)" 

definition prev_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "prev_d F \<equiv> LIFT(F;skip)"


syntax
  "_schop_d"     :: "[lift,lift] \<Rightarrow> lift" ("(_ \<frown> _)" [84,84] 83)
  "_sometimes_d" :: "lift \<Rightarrow> lift" ("(\<diamond>_)" [88] 87)
  "_di_d"        :: "lift \<Rightarrow> lift" ("(di _)" [88] 87)
  "_da_d"        :: "lift \<Rightarrow> lift" ("(da _)" [88] 87)
  "_next_d"      :: "lift \<Rightarrow> lift" ("(\<circle> _)" [88] 87)
  "_prev_d"      :: "lift \<Rightarrow> lift" ("(prev _)" [88] 87)

syntax (ASCII)
  "_schop_d"     :: "[lift,lift] \<Rightarrow> lift" ("(_ schop _)" [84,84] 83)
  "_sometimes_d" :: "lift \<Rightarrow> lift" ("(<>_)" [88] 87)
  "_di_d"        :: "lift \<Rightarrow> lift" ("(di _)" [88] 87) 
  "_da_d"        :: "lift \<Rightarrow> lift" ("(da _)" [88] 87)
  "_next_d"      :: "lift \<Rightarrow> lift" ("(next _)" [88] 87)
  "_prev_d"      :: "lift \<Rightarrow> lift" ("(prev _)" [88] 87)

translations 
  "_schop_d"     \<rightleftharpoons> "CONST schop_d"
  "_sometimes_d" \<rightleftharpoons> "CONST sometimes_d"
  "_di_d"        \<rightleftharpoons> "CONST di_d" 
  "_da_d"        \<rightleftharpoons> "CONST da_d"
  "_next_d"      \<rightleftharpoons> "CONST next_d" 
  "_prev_d"      \<rightleftharpoons> "CONST prev_d"


definition df_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "df_d F \<equiv> LIFT(F\<frown>#True)"

definition sda_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "sda_d F \<equiv> LIFT(#True\<frown>(F\<frown>#True))"

definition always_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "always_d F \<equiv> LIFT(\<not>(\<diamond>(\<not>F)))"

definition bi_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "bi_d F \<equiv> LIFT(\<not>(di(\<not>F)))"

definition ba_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "ba_d F \<equiv> LIFT(\<not>(da(\<not>F)))"

definition wnext_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "wnext_d F \<equiv> LIFT(\<not>(\<circle>(\<not>F)))"

definition wprev_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "wprev_d F \<equiv> LIFT(\<not>(prev(\<not>F)))"

definition more_d :: "('a::world) formula "
where "more_d \<equiv> LIFT(\<circle>(#True))" 


syntax
 "_df_d"       :: "lift \<Rightarrow> lift" ("(df _)" [88] 87)
 "_sda_d"      :: "lift \<Rightarrow> lift" ("(sda _)" [88] 87)
 "_always_d"   :: "lift \<Rightarrow> lift" ("(\<box> _)" [88] 87)
 "_bi_d"       :: "lift \<Rightarrow> lift" ("(bi _)" [88] 87)
 "_ba_d"       :: "lift \<Rightarrow> lift" ("(ba _)" [88] 87)
 "_wnext_d"    :: "lift \<Rightarrow> lift" ("(wnext _)" [88] 87)
 "_wprev_d"    :: "lift \<Rightarrow> lift" ("(wprev _)" [88] 87)
 "_more_d"     :: "lift"         ("(more)")
 

syntax (ASCII)
 "_df_d"       :: "lift \<Rightarrow> lift" ("(df _)" [88] 87)
 "_sda_d"      :: "lift \<Rightarrow> lift" ("(sda _)" [88] 87)
 "_always_d"   :: "lift \<Rightarrow> lift" ("([] _)" [88] 87)
 "_bi_d"       :: "lift \<Rightarrow> lift" ("(bi _)" [88] 87)
 "_ba_d"       :: "lift \<Rightarrow> lift" ("(ba _)" [88] 87)
 "_wnext_d"    :: "lift \<Rightarrow> lift" ("(wnext _)" [88] 87)
 "_wprev_d"    :: "lift \<Rightarrow> lift" ("(wprev _)" [88] 87)
 "_more_d"     :: "lift"         ("(more)")
 

translations
 "_df_d"     \<rightleftharpoons> "CONST df_d"
 "_sda_d"    \<rightleftharpoons> "CONST sda_d"
 "_always_d" \<rightleftharpoons> "CONST always_d"
 "_bi_d"     \<rightleftharpoons> "CONST bi_d" 
 "_ba_d"     \<rightleftharpoons> "CONST ba_d"
 "_wnext_d"  \<rightleftharpoons> "CONST wnext_d"
 "_wprev_d"  \<rightleftharpoons> "CONST wprev_d"
 "_more_d"   \<rightleftharpoons> "CONST more_d" 


definition bf_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "bf_d F \<equiv> LIFT(\<not>(df(\<not>F)))"

definition sba_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "sba_d F \<equiv> LIFT(\<not>(sda(\<not>F)))"

definition empty_d :: "('a::world) formula "
where "empty_d \<equiv> LIFT(\<not>(more))" 

definition fmore_d :: "('a::world) formula "
where "fmore_d \<equiv> LIFT(more \<and> finite)" 

definition dm_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "dm_d F \<equiv> LIFT(#True;(more \<and> F))" 

syntax
 "_bf_d"       :: "lift \<Rightarrow> lift" ("(bf _)" [88] 87)
 "_sba_d"      :: "lift \<Rightarrow> lift" ("(sba _)" [88] 87)
 "_empty_d"    :: "lift"         ("(empty)")
 "_fmore_d"    :: "lift"         ("(fmore)")
 "_dm_d"       :: "lift \<Rightarrow> lift" ("(dm _)" [88] 87)

syntax (ASCII)
 "_bf_d"       :: "lift \<Rightarrow> lift" ("(bf _)" [88] 87)
 "_sba_d"      :: "lift \<Rightarrow> lift" ("(sba _)" [88] 87)
 "_empty_d"    :: "lift"         ("(empty)")
 "_fmore_d"    :: "lift"         ("(fmore)")
 "_dm_d"       :: "lift \<Rightarrow> lift" ("(dm _)" [88] 87)

translations
 "_bf_d"    \<rightleftharpoons> "CONST bf_d"
 "_sba_d"   \<rightleftharpoons> "CONST sba_d"
 "_empty_d" \<rightleftharpoons> "CONST empty_d"
 "_fmore_d" \<rightleftharpoons> "CONST fmore_d"
 "_dm_d"    \<rightleftharpoons> "CONST dm_d"


definition bm_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "bm_d F \<equiv> LIFT(\<not>(dm(\<not>F)))" 

definition init_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "init_d F \<equiv> LIFT((empty \<and> F);#True)" 

definition fin_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "fin_d F \<equiv> LIFT(\<box>(empty \<longrightarrow> F))" 

definition halt_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "halt_d F \<equiv> LIFT(\<box>(empty = F))" 

definition initonly_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "initonly_d F \<equiv> LIFT(bi(empty = F))" 

definition keep_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "keep_d F \<equiv> LIFT(ba(skip \<longrightarrow> F))" 

definition yields_d :: "('a::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula "
where "yields_d F1 F2 \<equiv> LIFT(\<not>(F1;(\<not>F2)))" 

definition syields_d :: "('a::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula "
where "syields_d F1 F2 \<equiv> LIFT(\<not>(F1\<frown>(\<not>F2)))" 

definition ifthenelse_d :: "('a::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula "
where "ifthenelse_d F G H \<equiv> LIFT((F \<and> G) \<or> (\<not>F \<and> H) ) "

definition revyields_d :: "('a ::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
 where " revyields_d F1 F2 \<equiv> LIFT(\<not>((\<not> F1);F2)) "   

definition revsyields_d :: "('a ::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
 where " revsyields_d F1 F2 \<equiv> LIFT(\<not>((\<not> F1)\<frown>F2)) "   


syntax
 "_bm_d"         :: "lift \<Rightarrow> lift"             ("(bm _)" [88] 87)
 "_init_d"       :: "lift \<Rightarrow> lift"             ("(init _)" [88] 87)
 "_fin_d"        :: "lift \<Rightarrow> lift"             ("(fin _)" [88] 87)
 "_halt_d"       :: "lift \<Rightarrow> lift"             ("(halt _)" [88] 87)
 "_initonly_d"   :: "lift \<Rightarrow> lift"             ("(initonly _)" [88] 87)
 "_keep_d"       :: "lift \<Rightarrow> lift"             ("(keep _)" [88] 87)
 "_yields_d"     :: "[lift,lift] \<Rightarrow> lift"      ("(_ |\<sim>> _)" [88,88] 87)
 "_syields_d"    :: "[lift,lift] \<Rightarrow> lift"      ("(_ |\<simeq>> _)" [88,88] 87)
 "_ifthenelse_d" :: "[lift,lift,lift] \<Rightarrow> lift" ("(if\<^sub>i _ then _ else _ )"  [88,88,88] 87)
 "_revyields_d"  :: "[lift,lift] \<Rightarrow> lift"      ("(_ <\<sim>| _)" [84,84] 83)
 "_revsyields_d" :: "[lift,lift] \<Rightarrow> lift"      ("(_ <\<simeq>| _)" [84,84] 83)


syntax (ASCII)
 "_bm_d"         :: "lift \<Rightarrow> lift"             ("(bm _)" [88] 87)
 "_init_d"       :: "lift \<Rightarrow> lift"             ("(init _)" [88] 87)
 "_fin_d"        :: "lift \<Rightarrow> lift"             ("(fin _)" [88] 87)
 "_halt_d"       :: "lift \<Rightarrow> lift"             ("(halt _)" [88] 87)
 "_initonly_d"   :: "lift \<Rightarrow> lift"             ("(initonly _)" [88] 87)
 "_keep_d"       :: "lift \<Rightarrow> lift"             ("(keep _)" [88] 87)
 "_yields_d"     :: "[lift,lift] \<Rightarrow> lift"      ("(_ yields _)" [88,88] 87)
 "_syields_d"    :: "[lift,lift] \<Rightarrow> lift"      ("(_ syields _)" [88,88] 87)
 "_ifthenelse_d" :: "[lift,lift,lift] \<Rightarrow> lift" ("(if\<^sub>i _ then _ else _ )"  [88,88,88] 87)
 "_revyields_d"  :: "[lift,lift] \<Rightarrow> lift"      ("(_ revyields _)" [84,84] 83)
 "_revsyields_d" :: "[lift,lift] \<Rightarrow> lift"      ("(_ revsyields _)" [84,84] 83)


translations
 "_bm_d"          \<rightleftharpoons> "CONST bm_d"
 "_init_d"        \<rightleftharpoons> "CONST init_d"
 "_fin_d"         \<rightleftharpoons> "CONST fin_d"
 "_halt_d"        \<rightleftharpoons> "CONST halt_d"
 "_initonly_d"    \<rightleftharpoons> "CONST initonly_d" 
 "_keep_d"        \<rightleftharpoons> "CONST keep_d"
 "_yields_d"      \<rightleftharpoons> "CONST yields_d"
 "_syields_d"     \<rightleftharpoons> "CONST syields_d"
 "_ifthenelse_d"  \<rightleftharpoons> "CONST ifthenelse_d" 
 "_revyields_d"   \<rightleftharpoons> "CONST revyields_d" 
 "_revsyields_d"  \<rightleftharpoons> "CONST revsyields_d" 
 

definition sfin_d :: "('a::world) formula \<Rightarrow> 'a formula "
where "sfin_d F \<equiv> LIFT(\<not> (fin (\<not> F)))" 

definition ifthen_d :: "('a::world) formula  \<Rightarrow> 'a formula \<Rightarrow> 'a formula "
where "ifthen_d F G \<equiv> LIFT(if\<^sub>i F then G else #True ) "


syntax
 "_ifthen_d" :: "[lift,lift] \<Rightarrow> lift" ("(if\<^sub>i _ then _  )"  [88,88] 87)
 "_sfin_d"        :: "lift \<Rightarrow> lift"   ("(sfin _)" [88] 87)

syntax (ASCII)
 "_ifthen_d" :: "[lift,lift] \<Rightarrow> lift" ("(if\<^sub>i _ then _  )"  [88,88] 87)

 "_sfin_d"        :: "lift \<Rightarrow> lift"   ("(sfin _)" [88] 87)

translations
 "_ifthen_d" \<rightleftharpoons> "CONST ifthen_d"
 "_sfin_d"   \<rightleftharpoons> "CONST sfin_d"


definition afb_d :: "('a ::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
 where "afb_d F1 F2 \<equiv> LIFT(bf(F1 \<longrightarrow> fin F2) )" 

definition safb_d :: "('a ::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
 where "safb_d F1 F2 \<equiv> LIFT(bf(F1 = fin F2) )"

syntax
 "_afb_d"    :: "[lift,lift] \<Rightarrow> lift"  ("(_ \<mapsto> _)" [84,84] 83)
 "_safb_d"    :: "[lift,lift] \<Rightarrow> lift" ("(_ \<longleftrightarrow> _)" [84,84] 83)


syntax (ASCII)
 "_afb_d"     :: "[lift,lift] \<Rightarrow> lift" ("(_ afb _)" [84,84] 83)
 "_safb_d"    :: "[lift,lift] \<Rightarrow> lift" ("(_ safb _)" [84,84] 83)

translations
 "_afb_d"  \<rightleftharpoons> "CONST afb_d" 
 "_safb_d" \<rightleftharpoons> "CONST safb_d" 

definition next_assign_d :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun \<Rightarrow> 'a formula "
where "next_assign_d v e \<equiv> LIFT( v$ = e)" 

definition prev_assign_d :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun \<Rightarrow> 'a formula "
where "prev_assign_d v e \<equiv> LIFT( finite \<longrightarrow> v! = e)"  

definition always_eq_d  :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun \<Rightarrow> 'a formula "
where "always_eq_d v e \<equiv> \<lambda> s. s \<Turnstile> \<box>($v = e)"  

definition temporal_assign_d  :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun \<Rightarrow> 'a formula "
where "temporal_assign_d v e \<equiv> \<lambda> s. s \<Turnstile> finite \<longrightarrow> !v = e" 

definition gets_d :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun \<Rightarrow> 'a formula "
where "gets_d v e \<equiv> \<lambda> s. s \<Turnstile> keep( temporal_assign_d v e)" 

definition stable_d :: "('a::world,'b) stfun \<Rightarrow> 'a formula "
where "stable_d v \<equiv> \<lambda> s. s \<Turnstile> gets_d v (current_val_d v) "

definition padded_d :: "('a::world,'b) stfun \<Rightarrow> 'a formula "
where "padded_d v \<equiv> \<lambda> s. s \<Turnstile> (stable_d v);skip \<or> empty" 

definition padded_temp_assign_d :: "('a::world,'b) stfun \<Rightarrow> ('a,'b) formfun \<Rightarrow> 'a formula "
where "padded_temp_assign_d v e \<equiv> \<lambda> s. s \<Turnstile> (temporal_assign_d v e) \<and> (padded_d v)" 


syntax
 "_next_assign_d"        :: "[lift,lift] \<Rightarrow> lift" ("(_ := _)" [50,51] 50)
 "_prev_assign_d"        :: "[lift,lift] \<Rightarrow> lift" ("(_ =: _)" [50,51] 50)
 "_always_eq_d"          :: "[lift,lift] \<Rightarrow> lift" ("(_ \<approx> _)" [50,51] 50)
 "_temporal_assign_d"    :: "[lift,lift] \<Rightarrow> lift" ("(_ \<leftarrow> _)" [50,51] 50)
 "_gets_d"               :: "[lift,lift] \<Rightarrow> lift" ("(_ gets _)" [50,51] 50)
 "_stable_d"             :: "lift \<Rightarrow> lift"        ("(stable _)" [51] 50)
 "_padded_d"             :: "lift \<Rightarrow> lift"        ("(padded _)" [51] 50)
 "_padded_temp_assign_d" :: "[lift,lift] \<Rightarrow> lift" ("(_ <\<sim> _)" [50,51] 50)

syntax (ASCII)
 "_next_assign_d"        :: "[lift,lift] \<Rightarrow> lift" ("(_ := _)" [50,51] 50)
 "_prev_assign_d"        :: "[lift,lift] \<Rightarrow> lift" ("(_ =: _)" [50,51] 50)
 "_always_eq_d"          :: "[lift,lift] \<Rightarrow> lift" ("(_ alweqv _)" [50,51] 50)
 "_temporal_assign_d"    :: "[lift,lift] \<Rightarrow> lift" ("(_ <-- _)" [50,51] 50)
 "_gets_d"               :: "[lift,lift] \<Rightarrow> lift" ("(_ gets _)" [50,51] 50)
 "_stable_d"             :: "lift \<Rightarrow> lift"        ("(stable _)" [51] 50)
 "_padded_d"             :: "lift \<Rightarrow> lift"        ("(padded _)" [51] 50)
 "_padded_temp_assign_d" :: "[lift,lift] \<Rightarrow> lift" ("(_ <~ _)" [50,51] 50)

translations
 "_next_assign_d"        \<rightleftharpoons> "CONST next_assign_d"
 "_prev_assign_d"        \<rightleftharpoons> "CONST prev_assign_d"
 "_always_eq_d"          \<rightleftharpoons> "CONST always_eq_d"
 "_temporal_assign_d"    \<rightleftharpoons> "CONST temporal_assign_d"
 "_gets_d"               \<rightleftharpoons> "CONST gets_d"
 "_stable_d"             \<rightleftharpoons> "CONST stable_d"
 "_padded_d"             \<rightleftharpoons> "CONST padded_d"
 "_padded_temp_assign_d" \<rightleftharpoons> "CONST padded_temp_assign_d"

lemmas itl_def = skip_d_def chop_d_def current_val_d_def next_val_d_def fin_val_d_def penult_val_d_def
       infinite_d_def finite_d_def schop_d_def sometimes_d_def di_d_def da_d_def next_d_def prev_d_def 
       df_d_def sda_d_def always_d_def bi_d_def ba_d_def wnext_d_def wprev_d_def more_d_def bf_d_def 
       sba_d_def empty_d_def fmore_d_def dm_d_def bm_d_def init_d_def fin_d_def halt_d_def initonly_d_def
       keep_d_def yields_d_def syields_d_def ifthenelse_d_def  sfin_d_def ifthen_d_def next_assign_d_def 
       prev_assign_d_def always_eq_d_def temporal_assign_d_def gets_d_def stable_d_def padded_d_def 
       padded_temp_assign_d_def revyields_d_def revsyields_d_def afb_d_def safb_d_def

subsection \<open>Properties of Operators\<close>

text \<open> The following lemmas show that above operators have the expected semantics. \<close>

lemma skip_defs :
 "(w \<Turnstile> skip) =  (nlength w = (enat 1))  "
by (simp add: itl_def)

lemma chop_defs :
 "(w \<Turnstile> F1 ; F2) = 
  (
       (\<exists>n\<le>nlength w.  ( (ntaken n w) \<Turnstile> F1 )  \<and>  ((ndropn n w) \<Turnstile> F2 ) )
       \<or> (\<not> nfinite w \<and> ( w \<Turnstile> F1))
  )"
by (simp add: itl_def)

lemma yields_defs :
 "(w \<Turnstile> F1 yields F2) = 
  ((\<forall> n. ((ntaken n w) \<Turnstile> F1)  \<longrightarrow> enat n \<le> nlength w \<longrightarrow>  (ndropn n w \<Turnstile>  F2)) \<and> (nfinite w \<or> (w \<Turnstile> \<not>F1) ))
  "
by (simp add: itl_def)

lemma revyields_defs :
 "(w \<Turnstile> F1 revyields F2) = 
  ((\<forall> n. ((ndropn n w) \<Turnstile> F2)  \<longrightarrow> enat n \<le> nlength w \<longrightarrow>  ((ntaken n w) \<Turnstile>  F1)) \<and> (nfinite w \<or> (w \<Turnstile> F1) ))
  "
by (simp add: itl_def)
blast

lemma revsyields_defs :
 "(w \<Turnstile> F1 revsyields F2) = 
  (\<forall> n. ( (ndropn n w) \<Turnstile> F2)  \<longrightarrow> enat n \<le> nlength w \<longrightarrow>  ((ntaken n w \<Turnstile>  F1))) 
 "
by (simp add:  itl_def)
blast



lemma infinite_defs:
 " (w \<Turnstile> inf) = (\<not> nfinite w)"
by (simp add: itl_def)

lemma finite_defs :
 " (w \<Turnstile> finite) = (nfinite w)"
by (simp add: itl_def)

lemma schop_defs :
 "(w \<Turnstile> F1 \<frown> F2) = 
  (
       (\<exists>n\<le>nlength w.  ( (ntaken n w) \<Turnstile> F1 )  \<and> ( (ndropn n w) \<Turnstile> F2 ) )           
  )
 "
by (auto simp add: itl_def chop_defs finite_defs )

lemma syields_defs :
 "(w \<Turnstile> F1 syields F2) = 
  (\<forall> n. ( (ntaken n w) \<Turnstile> F1)  \<longrightarrow> enat n \<le> nlength w \<longrightarrow>  ((ndropn n w \<Turnstile>  F2))) 
 "
by (simp add:  itl_def)

lemma sometimes_defs : 
  "(w \<Turnstile> \<diamond> F) = (\<exists> n\<le>nlength w.  ( (ndropn n w)) \<Turnstile> F)"
by (simp add: itl_def finite_defs chop_defs)

lemma always_defs : 
 " (w \<Turnstile> \<box> F) = 
    (\<forall> n\<le>nlength w.  ( (ndropn n w) \<Turnstile> F)) "
by (simp add: itl_def sometimes_defs)

lemma di_defs : 
  "(w \<Turnstile> di F) = 
  ((\<exists> n\<le>nlength w.  ((ntaken n w) \<Turnstile> F)) \<or> (\<not>nfinite w \<and> (w \<Turnstile> F)))"
by (simp add: itl_def )

lemma df_defs : 
  "(w \<Turnstile> df F) = 
   (\<exists> n\<le>nlength w.  ((ntaken n w) \<Turnstile> F))"
by (simp add: df_d_def schop_defs)

lemma bi_defs : 
  "(w \<Turnstile> bi F) = 
   ((\<forall> n\<le>nlength w . ((ntaken n w) \<Turnstile> F)) \<and>(nfinite w \<or> (w \<Turnstile> F))) "
by (simp add: itl_def di_defs )

lemma bf_defs : 
  "(w \<Turnstile> bf F) = 
    (\<forall> n\<le>nlength w. ( (ntaken n w) \<Turnstile> F))"
by (simp add: bf_d_def df_defs )

lemma da_defs : 
  "(w \<Turnstile> da F) = 
   ( (\<exists> n na.  ( n+na \<le> nlength w \<and> ( (nsubn w n (n+ na)) \<Turnstile> F)) \<or> (\<not>nfinite w \<and> ( (ndropn n w) \<Turnstile> F)))
   )"
proof 
 (auto simp add: itl_def chop_defs nsubn_def1)
 show "\<And>n na.
       enat n \<le> nlength w \<Longrightarrow>
       enat na \<le> nlength w - enat n \<Longrightarrow>
       F (ntaken na (ndropn n w)) \<Longrightarrow> \<exists>n. (\<exists>na. enat (n + na) \<le> nlength w \<and> F (ntaken na (ndropn n w)))
         \<or> \<not> nfinite w \<and> F (ndropn n w) "
       by (metis add_left_mono enat.simps(3) enat_add_sub_same le_iff_add plus_enat_simps(1))  
 next
  fix n :: nat and na :: nat
  assume a1: "enat (n + na) \<le> nlength w"
  assume a2: "F (ntaken na (ndropn n w))"
  have "enat n \<le> nlength w"
  using a1 by (meson enat_ord_simps(1) le_iff_add order_subst2)
  then show "\<exists>n. enat n \<le> nlength w \<and>
                   ((\<exists>na. enat na \<le> nlength w - enat n \<and> F (ntaken na (ndropn n w)))
                     \<or> \<not> nfinite w \<and> F (ndropn n w))"
  using a2 a1 by (metis (no_types) add_diff_cancel_left' enat_minus_mono1 idiff_enat_enat)
 next      
 show "\<And>n. \<not> nfinite w \<Longrightarrow>
         F (ndropn n w) \<Longrightarrow>
         \<exists>n. enat n \<le> nlength w \<and> 
             ((\<exists>na. enat na \<le> nlength w - enat n \<and> F (ntaken na (ndropn n w))) \<or> F (ndropn n w))"
  by (meson enat_ile le_cases nfinite_conv_nlength_enat)
qed

lemma ba_defs :
 "(w \<Turnstile> ba F) = 
  (\<forall> n na. (enat (n + na) \<le> nlength w \<longrightarrow> ( (nsubn w n (n+na) ) \<Turnstile> F)) 
                         \<and> (nfinite w \<or> ( (ndropn n w) \<Turnstile> F)))"
by (simp add: ba_d_def da_defs )

lemma sda_defs : 
  "(w \<Turnstile> sda F) = 
    (\<exists> n na.  ( n+na \<le> nlength w \<and> ( (nsubn w n (n+ na)) \<Turnstile> F)))"
proof 
 (auto simp add: sda_d_def schop_defs nsubn_def1)
 show "\<And>n na.
       enat n \<le> nlength w \<Longrightarrow>
       enat na \<le> nlength w - enat n \<Longrightarrow> F (ntaken na (ndropn n w)) \<Longrightarrow> 
       \<exists>n na. enat (n + na) \<le> nlength w \<and> F (ntaken na (ndropn n w))"
       by (metis add_le_cancel_left enat_ord_simps(1) idiff_enat_enat le_add_diff_inverse le_cases 
           nfinite_nlength_enat nfinite_ntaken ntaken_all)
next
 fix n :: nat and na :: nat
 assume a1: "F (ntaken na (ndropn n w))"
 assume a2: "enat (n + na) \<le> nlength w"
 have "enat na \<le> nlength w - enat n"
 by (metis a2 add_diff_cancel_left' enat_minus_mono1 idiff_enat_enat)
 then show "\<exists>n. enat n \<le> nlength w \<and> 
                   (\<exists>na. enat na \<le> nlength w - enat n \<and> F (ntaken na (ndropn n w)))"
 using  a2 a1
 by (metis add.right_neutral le_cases le_zero_eq ndropn_all ndropn_nlength nlength_NNil 
     the_enat.simps the_enat_0)    
qed

lemma sba_defs :
 "(w \<Turnstile> sba F) = 
   (\<forall> n na.  n+na \<le> nlength w \<longrightarrow> ( (nsubn w n (n+na) ) \<Turnstile> F))"
by (simp add: sba_d_def sda_defs)

lemma next_defs :
 " (w \<Turnstile> \<circle> F) = 
  (nlength w \<noteq> (enat 0) \<and> ((ndropn 1 w)\<Turnstile> F) )"
by (simp add: itl_def chop_defs)
   (metis One_nat_def Suc_ile_eq dual_order.order_iff_strict enat_le_plus_same(1) gen_nlength_def 
    min.orderE min_enat_simps(2) nlength_code nlength_eq_enat_nfiniteD one_enat_def 
    the_enat.simps zero_enat_def zero_one_enat_neq(1))

lemma wnext_defs :
 "(w \<Turnstile> wnext F) = 
  (nlength w = (enat 0) \<or> ((ndropn 1 w)\<Turnstile> F) )"
by (simp add: wnext_d_def next_defs)

lemma prev_defs :
 " (w \<Turnstile> prev F) = 
    ((nlength w \<noteq> (enat 0) \<and> nfinite w \<and> 
     ( (ntaken (the_enat((epred(nlength w))) ) w) \<Turnstile> F) ) \<or> (\<not>nfinite w \<and> (w \<Turnstile> F)))"
proof (cases "nfinite w")
case True
then show ?thesis 
 by (auto simp add: prev_d_def chop_defs skip_defs )
    (metis One_nat_def diff_diff_cancel enat_ord_simps(1) epred_enat idiff_enat_enat nfinite_nlength_enat 
     the_enat.simps,
     metis One_nat_def diff_le_self eSuc_epred enat.simps(3) enat_add_sub_same enat_ord_simps(1) 
     epred_enat nfinite_nlength_enat one_enat_def plus_1_eSuc(2) the_enat.simps zero_enat_def)
next
case False
then show ?thesis 
  by (auto simp add: prev_d_def chop_defs skip_defs )
     (metis ndropn_nlength nfinite_ndropn_b nlength_eq_enat_nfiniteD)
qed
   
lemma wprev_defs :
 " (w \<Turnstile> wprev F) = 
  ( (nfinite w \<longrightarrow> (nlength w = (enat 0) \<or> ( (ntaken (the_enat(epred(nlength w))) w) \<Turnstile> F) ) )
    \<and> ( nfinite w \<or> (w \<Turnstile> F))) "
by (simp add: wprev_d_def prev_defs )

lemma more_defs :
 " (w \<Turnstile> more) = ( 0 < nlength w)"
using zero_enat_def by (auto simp add: more_d_def next_defs ) 

lemma fmore_defs :
 " (w \<Turnstile> fmore) = (nfinite w \<and> (0 < nlength w))"
by (auto simp add: fmore_d_def more_defs finite_defs ) 

lemma empty_defs :
 " (w \<Turnstile> empty) = (nlength w = 0)"
by (simp add: empty_d_def more_defs)

lemma init_defs :
 " (w \<Turnstile> init F) = ( (ntaken 0 w) \<Turnstile> F )"
by (simp add: init_d_def chop_defs empty_defs min_def) 
 (metis ntaken_0 ntaken_all the_enat.simps zero_enat_def zero_le)

lemma initalt_defs :
 " (w \<Turnstile> bi( empty \<longrightarrow> F)) = ( (ntaken 0 w) \<Turnstile> F )" 
by (simp add: bi_defs empty_defs  min_def)
   (metis enat_0_iff(2) nlength_eq_enat_nfiniteD ntaken_0 zero_le)  


lemma fin_defs :
 " (w \<Turnstile> fin F) = 
   (  (nfinite w \<and> ( (ndropn (the_enat(nlength w)) w) \<Turnstile> F) ) \<or> (\<not>nfinite w) )"
by (simp add: fin_d_def empty_defs always_defs )
   (metis add.right_neutral enat.distinct(2) enat_add_sub_same le_iff_add nfinite_nlength_enat 
    nlength_eq_enat_nfiniteD the_enat.simps)

lemma finalt_defs :
 " (w \<Turnstile> #True;(F \<and> empty)) = 
   (  (nfinite w \<and> ( (ndropn (the_enat(nlength w)) w) \<Turnstile> F) ) \<or> (\<not>nfinite w) )"
by (simp add: chop_defs empty_defs )
   (metis add.right_neutral enat.distinct(2) enat_add_sub_same le_iff_add nfinite_nlength_enat 
    the_enat.simps)

lemma sfin_defs :
 " (w \<Turnstile> sfin F) = (nfinite w \<and> ( (ndropn (the_enat(nlength w)) w) \<Turnstile> F) )"
by (auto simp add: sfin_d_def fin_defs )

lemma afb_defs:
 "(w \<Turnstile> f1 afb f2) = 
  (\<forall> n. ( (ntaken n w) \<Turnstile> f1) \<and> enat n \<le> nlength w  \<longrightarrow> ((NNil (nnth w n)) \<Turnstile> f2))  "
unfolding afb_d_def 
using bf_defs[of "LIFT(f1 \<longrightarrow> fin f2)" ]  fin_defs[of f2] 
by simp
 (metis min.absorb1 ndropn_all ntaken_nlast ntaken_nlength order_refl)

lemma safb_defs:
 "(w \<Turnstile> f1 safb f2) = 
  (\<forall> n.  enat n \<le> nlength w  \<longrightarrow> ((ntaken n w) \<Turnstile> f1) = ((NNil (nnth w n)) \<Turnstile> f2))  "
unfolding safb_d_def 
using bf_defs[of "LIFT(f1 = fin f2)" ]  fin_defs[of f2] 
by (simp add: ndropn_all ntaken_nlast)


lemma halt_defs :
 " (w \<Turnstile> halt(F)) = (\<forall>n. n\<le>nlength w\<longrightarrow> (nlength w = n) = (  (ndropn n w) \<Turnstile> F))
"
by (simp add: halt_d_def empty_defs always_defs )
   (metis add.right_neutral dual_order.strict_iff_order enat_add_sub_same enat_ord_code(4) 
    le_iff_add)

lemma initonly_defs :
 " (w \<Turnstile> initonly(F)) = 
   ( (\<forall>n. n\<le>nlength w \<longrightarrow> (n = 0) = ( (ntaken  n w) \<Turnstile> F ))\<and>
     (nfinite w \<or> (nlength w = 0) = F w)        
   )"
  by (simp add: initonly_d_def bi_defs empty_defs zero_enat_def)

lemma ifthenelse_defs:
 " (w \<Turnstile> if\<^sub>i F then G else H) = 
   ( ((w \<Turnstile> F) \<and> (w \<Turnstile> G)) \<or> (( \<not>(w \<Turnstile> F) \<and> (w \<Turnstile> H))) )"
by (simp add: itl_def)



lemma currentval_defs :
 " (s \<Turnstile> $v) = (v (nfirst s))"
by (simp add: itl_def)

lemma nextval_defs :
 "(s \<Turnstile> v$) = 
   (if nlength s \<noteq> (enat 0) then (v (nnth s 1)) else (\<some> x. x=x))"
by (simp add: itl_def)

lemma finval_defs :
 " (s \<Turnstile> !v) = 
   ( (if nfinite s then (v (nlast s)) else (\<some> x. x=x))
   )"
by (simp add: itl_def)

lemma penultval_defs :
 " (s \<Turnstile> v!) = 
   (if nfinite s then 
     (if nlength s \<noteq> (enat 0) then (v (nnth s (the_enat(epred(nlength s))))) else (\<some> x. x=x))
    else (\<some> x. x=x)
   )"
by (simp add: itl_def) 

lemma next_assign_defs :
 " (s \<Turnstile> v := e) = ((if nlength s \<noteq> (enat 0) then v (nnth s 1) else (\<some> x. x=x)) = e s)"
by (auto simp: itl_def)

lemma prev_assign_defs :
 " (s \<Turnstile> v =: e) = 
  (if nfinite s then 
    (if nlength s \<noteq> (enat 0) then (v (nnth s (the_enat(epred(nlength s))))) = e s  
                             else ((\<some> x. x=x) = e s))
   else  True
   )  "
by (simp add: itl_def finite_defs)

lemma always_eqv_defs :
 " (s \<Turnstile> v \<approx> e) = 
   ( (\<forall> i. i\<le> nlength s \<longrightarrow> v (nnth s i) = e (ndropn i s))
   ) "
by (simp add: always_eq_d_def always_defs current_val_d_def ndropn_nfirst) 

lemma temporal_assign_defs :
 " (s \<Turnstile> v \<leftarrow> e) = 
   (if nfinite s then (v (nlast s)) = e s
                 else True
   ) "
by (simp add: itl_def finite_defs)

lemma gets_defs :
 " (s \<Turnstile> v gets e) = 
  ( (\<forall> i. i < nlength s \<longrightarrow> v (nnth s (Suc i)) = e ((nsubn s i (Suc i))) )
           
  )"
proof (auto simp add: min_def finite_defs gets_d_def keep_d_def ba_defs skip_defs 
     temporal_assign_d_def fin_val_d_def nsubn_nlength Suc_ile_eq nsubn_def1 ntaken_ndropn_nlast)
show "\<And>i. \<forall>n na. (enat na \<le> nlength s - enat n \<longrightarrow> 
                   enat (n + na) \<le> nlength s \<longrightarrow> 
                   na = Suc 0 \<longrightarrow> v (nnth s (Suc n)) = e (ntaken (Suc 0) (ndropn n s))) \<and>
                (\<not> enat na \<le> nlength s - enat n \<longrightarrow>
                 enat (n + na) \<le> nlength s \<longrightarrow> 
                nlength s - enat n = enat (Suc 0) \<longrightarrow> 
                v (nnth s (na + n)) = e (ntaken na (ndropn n s))) \<Longrightarrow>
         enat i < nlength s \<Longrightarrow> 
         v (nnth s (Suc i)) = e (ntaken (Suc 0) (ndropn i s))"
   by (metis One_nat_def add.commute eSuc_enat i0_less ileI1 iless_Suc_eq ndropn_Suc_conv_ndropn 
       ndropn_nlength nlength_NCons one_eSuc one_enat_def plus_1_eq_Suc zero_le) 
show "\<And>n na.
       \<forall>i. enat i < nlength s \<longrightarrow> v (nnth s (Suc i)) = e (ntaken (Suc 0) (ndropn i s)) \<Longrightarrow>
       \<not> na \<le> Suc 0 \<Longrightarrow> enat (n + na) \<le> nlength s \<Longrightarrow> nlength s - enat n = enat (Suc 0) \<Longrightarrow> 
        v (nnth s (na + n)) = e (ntaken na (ndropn n s))"
   by (metis (no_types) add_diff_cancel_left' enat_minus_mono1 enat_ord_simps(1) idiff_enat_enat)  
qed     

lemma stable_defs_helpa:
assumes "(\<forall>i. i<nlength s \<longrightarrow> v (nnth s (Suc i)) = v (nnth s i))"  
        " i \<le> nlength s" 
 shows  "(v (nnth s i) = v (nfirst s))" 
using assms
proof (induct i arbitrary:s)
case 0
then show ?case by (metis ndropn_0 ndropn_nfirst)
next
case (Suc i)
then show ?case 
by (simp add: Suc_ile_eq)
qed

lemma stable_defs_helpb:
assumes "(\<forall>i. i\<le>nlength s \<longrightarrow> v (nnth s i) = v (nfirst s))"  
        " i < nlength s" 
 shows  "v (nnth s (Suc i)) = v (nnth s i)" 
using assms
proof (induct i arbitrary:s)
case 0
then show ?case using Suc_ile_eq by auto
next
case (Suc i)
then show ?case by (metis eSuc_enat ileI1 less_imp_le)
qed

lemma stable_defs_help:
 "(\<forall>i. i<nlength s \<longrightarrow> v (nnth s (Suc i)) = v (nnth s i)) = 
  (\<forall>i. i\<le>nlength s \<longrightarrow> v (nnth s i) = v (nfirst s))"
using stable_defs_helpa[of s v] stable_defs_helpb[of s v]
by blast 

lemma stable_defs:
 "(s \<Turnstile> stable v) =
  (\<forall> i. i\<le>nlength s \<longrightarrow> (v (nnth s i)) = (v (nfirst s)))"
proof (simp add: stable_d_def gets_defs current_val_d_def)
 have 1: "\<And>i. i < nlength s \<longrightarrow> v (nfirst  (nsubn s i (Suc i))) = v (nnth s i)" 
   by (simp add: nsubn_def1 ntaken_ndropn_nfirst)
 have 2: "(\<forall>i. enat i < nlength s \<longrightarrow> v (nnth s (Suc i)) = v (nfirst (nsubn s i (Suc i)))) =
          (\<forall>i. enat i < nlength s \<longrightarrow> v (nnth s (Suc i)) = v (nnth s i))"
   by (simp add: "1") 
 have 3: "(\<forall>i. enat i < nlength s \<longrightarrow> v (nnth s (Suc i)) = v (nnth s i)) = 
          (\<forall> i. i\<le>nlength s \<longrightarrow> (v (nnth s i)) = (v (nfirst s)))" 
   using stable_defs_help[of s v] by blast
 show "(\<forall>i. enat i < nlength s \<longrightarrow> v (nnth s (Suc i)) = v (nfirst (nsubn s i (Suc i)))) =
          (\<forall>i. enat i \<le> nlength s \<longrightarrow> v (nnth s i) = v (nfirst s))" 
   by (simp add: "2" "3")
qed

lemma padded_defs :
 " (s \<Turnstile> padded v) = 
  ( ((\<forall>i.  i< nlength s \<longrightarrow> (v (nnth s i)) = (v (nfirst s)))) \<or> nlength s = (enat 0))"
proof (cases s)
case (NNil x1)
then show ?thesis 
 by (auto simp add: padded_d_def stable_defs chop_d_def skip_defs empty_defs ntaken_nfirst
      ntaken_nnth zero_enat_def)
next
case (NCons x21 x22)
then show ?thesis 
by (auto simp add: padded_d_def stable_defs chop_d_def skip_defs empty_defs ntaken_nfirst
      ntaken_nnth zero_enat_def)
   (metis One_nat_def enat.simps(3) enat_add_sub_same enat_ord_simps(1) iless_Suc_eq le_iff_add 
    less_imp_le one_enat_def plus_1_eSuc(2),
    meson iless_Suc_eq less_imp_le,
    metis One_nat_def enat.simps(3) enat_add_sub_same enat_ord_simps(1) ile_eSuc nfinite_nlength_enat 
    one_enat_def plus_1_eSuc(2),
    metis One_nat_def antisym_conv2 co.enat.sel(2) diff_le_self enat_add_sub_same enat_ord_code(4) 
    enat_ord_simps(1) epred_enat iless_Suc_eq one_enat_def plus_1_eSuc(2))
qed

lemma padded_temporal_assign_defs :
 " (s \<Turnstile> v <\<sim> e) = 
  ((s \<Turnstile> padded v) \<and>
    (if nfinite s then (v (nlast s )) = e s else True)
   )"
by (auto simp add: padded_temp_assign_d_def padded_defs temporal_assign_defs)

lemma chop_nfuse_1 : 
  " (\<exists> \<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>
       (\<sigma>1  \<Turnstile> f) \<and> (\<sigma>2  \<Turnstile> g) \<and> 
       (nlast \<sigma>1 = nfirst \<sigma>2)) =
      ((\<exists> i. 0 \<le> i \<and> i\<le> nlength \<sigma> \<and> ( ntaken i \<sigma>  \<Turnstile> f) \<and> (ndropn i \<sigma>  \<Turnstile> g))) "
by auto
 (metis enat_le_plus_same(1) nfuse_nlength ndropn_nfuse nfinite_conv_nlength_enat ntaken_nfuse 
  the_enat.simps,
  metis nfuse_ntaken_ndropn ndropn_nfirst nfinite_ntaken ntaken_nlast)

lemma chop_nfuse_2 : 
  " (\<exists> \<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>
       (\<sigma>1 \<in> X) \<and> (\<sigma>2 \<in> Y) \<and> 
       (nlast \<sigma>1 = nfirst \<sigma>2)) =
      (\<exists> i. i\<le> nlength \<sigma> \<and> (ntaken i \<sigma>)\<in>X \<and> (ndropn i \<sigma>)\<in>Y) "
by auto
   (metis enat_le_plus_same(1) ndropn_nfuse nfinite_nlength_enat nfuse_nlength ntaken_nfuse 
    the_enat.simps,
    metis ndropn_nfirst nfinite_ntaken nfuse_ntaken_ndropn ntaken_nlast)
 
lemma chop_nfuse:
  " (\<sigma>  \<Turnstile> f;g) = (
        (\<exists> \<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>
         (\<sigma>1  \<Turnstile> f) \<and> ( \<sigma>2  \<Turnstile> g)  \<and> (nlast \<sigma>1 = nfirst \<sigma>2)) 
        \<or> (\<not> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f))
     )  
       "
by (simp add: chop_defs chop_nfuse_1)

lemmas itl_defs = skip_defs chop_defs yields_defs infinite_defs finite_defs schop_defs syields_defs 
       sometimes_defs always_defs di_defs df_defs bi_defs bf_defs da_defs ba_defs sda_defs sba_defs
       next_defs wnext_defs prev_defs wprev_defs more_defs fmore_defs empty_defs init_defs 
       initalt_defs fin_defs finalt_defs sfin_defs halt_defs initonly_defs ifthenelse_defs  
       currentval_defs nextval_defs finval_defs penultval_defs next_assign_defs prev_assign_defs
       always_eqv_defs temporal_assign_defs gets_defs stable_defs padded_defs 
       padded_temporal_assign_defs  revyields_defs revsyields_defs afb_defs safb_defs

subsection \<open>Soundness Axioms\<close>

subsubsection \<open>ChopAssoc\<close>

lemma ChopAssocSemHelpa:
assumes "((\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))) \<or>
     \<not> nfinite \<sigma> \<and> f \<sigma>)"
shows "((\<exists>n. enat n \<le> nlength \<sigma> \<and> (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> 
             f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>))) \<and> h (ndropn n \<sigma>)) \<or>
     \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>))" 
proof -
 have 1: "((\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))) \<or>
     \<not> nfinite \<sigma> \<and> f \<sigma>)" 
   using assms by auto
 have 2: "\<not> nfinite \<sigma> \<and> f \<sigma> \<Longrightarrow> 
          ((\<exists>n. enat n \<le> nlength \<sigma> \<and> 
             (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
              \<and> h (ndropn n \<sigma>)) \<or>
     \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>))"
   by simp 
 have 3: "(\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))) \<Longrightarrow>
           ((\<exists>n. enat n \<le> nlength \<sigma> \<and> 
               (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
                \<and> h (ndropn n \<sigma>)) \<or>
     \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>)) "
  proof -
     assume 4: "(\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>)))" 
     obtain n where 5: "enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))" 
       using 4 by auto
     have 6: "enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>)" 
       using 5 by auto
     have 7: "\<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>) \<Longrightarrow> 
             ((\<exists>n. enat n \<le> nlength \<sigma> \<and> 
                (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
                 \<and> h (ndropn n \<sigma>)) \<or>
            \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>))"
       using "6" nfinite_ndropn_a by blast
     have 8: "(\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<Longrightarrow>
               ((\<exists>n. enat n \<le> nlength \<sigma> \<and> 
                   (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
                 \<and> h (ndropn n \<sigma>)) \<or>
     \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>))"
       proof -
         assume 9: "(\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>)))"
         show   "((\<exists>n. enat n \<le> nlength \<sigma> \<and> 
                   (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
                      \<and> h (ndropn n \<sigma>)) \<or>
     \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>))" 
         proof -
          obtain na where 10: "enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and>
               h (ndropn na (ndropn n \<sigma>))"
            using 9 by auto
          have 11: "h (ndropn (na+n) \<sigma>)"
            by (metis "10" add.commute ndropn_ndropn)    
          have 12: "na+n \<le> nlength \<sigma>"
            by (metis "10" "6" Groups.add_ac(2) dual_order.strict_implies_order enat.simps(3) 
                 enat_add_sub_same enat_less_enat_plusI2 le_iff_add not_le_imp_less 
                  order.not_eq_order_implies_strict plus_enat_simps(1))
          have 13: "g (ndropn n (ntaken (n+na) \<sigma>))"
            by (metis "10" "12" add.commute ntaken_ndropn_swap plus_enat_simps(1)) 
          have 14: "f (ntaken (min n (n+na)) \<sigma>)" 
            using "6" by linarith
          show ?thesis 
          by (metis "10" "12" "13" "14" "6" add.commute le_add1 ndropn_ndropn)
        qed
       qed
     show ?thesis 
     using "5" "7" "8" by blast
    qed
   show ?thesis 
   using "3" assms by blast
qed

lemma ChopAssocSemHelpb:
assumes " ((\<exists>n. enat n \<le> nlength \<sigma> \<and> 
             (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
              \<and> h (ndropn n \<sigma>)) \<or>
     \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>))" 
shows "((\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))) \<or>
     \<not> nfinite \<sigma> \<and> f \<sigma>)" 
proof -
 have 1: "((\<exists>n. enat n \<le> nlength \<sigma> \<and> 
            (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
              \<and> h (ndropn n \<sigma>)) \<or>
     \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>))"
   using assms by auto
 have 2: "\<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>) \<Longrightarrow> 
          ((\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))) \<or>
     \<not> nfinite \<sigma> \<and> f \<sigma>)"
   using nfinite_ndropn by blast
 have 3: "(\<exists>n. enat n \<le> nlength \<sigma> \<and> 
            (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
              \<and> h (ndropn n \<sigma>)) \<Longrightarrow>
          ((\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))) \<or>
     \<not> nfinite \<sigma> \<and> f \<sigma>)" 
  proof -
   assume 4: "(\<exists>n. enat n \<le> nlength \<sigma> \<and> 
                (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
                 \<and> h (ndropn n \<sigma>))"
   show "((\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))) \<or>
     \<not> nfinite \<sigma> \<and> f \<sigma>)"  
    proof -
     obtain n where 5: "enat n \<le> nlength \<sigma> \<and> 
             (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
               \<and> h (ndropn n \<sigma>)" 
       using 4 by auto
     have 6: "(\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))"
       using 5 by auto
     obtain na where 7: "na\<le>n \<and> enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>))"
       using 6 by auto
     have 8: "na \<le> nlength \<sigma>"
       by (simp add: "7")  
     have 9: "n-na \<le> nlength \<sigma> - na"
       by (metis "5" enat_minus_mono1 idiff_enat_enat)
     have 10: "f (ntaken na \<sigma>)" 
       using "7" by linarith
     have 11: "g (ntaken (n-na) (ndropn na \<sigma>))"
       by (simp add: "5" "7" ntaken_ndropn_swap) 
     have 12: "h (ndropn ((n-na)+na) \<sigma>)" 
       by (simp add: "5" "7")
     have 13: "h (ndropn (n-na) (ndropn na \<sigma>))"
       by (metis "12" add.commute ndropn_ndropn) 
     show ?thesis 
       using "10" "11" "13" "7" "9" by auto
  qed
 qed
 show ?thesis 
 using "2" "3" assms by blast
qed

lemma ChopAssocSemHelp:
 "((\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))) \<or>
     \<not> nfinite \<sigma> \<and> f \<sigma>) =
  ((\<exists>n. enat n \<le> nlength \<sigma> \<and> 
       (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
           \<and> h (ndropn n \<sigma>)) \<or>
     \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>))"
 using ChopAssocSemHelpa[of \<sigma> f g h] ChopAssocSemHelpb[of \<sigma> f g h] by blast

lemma ChopAssocSemHelp1:
 " (( \<sigma>) \<Turnstile>  f ; (g ; h)) = (( \<sigma>) \<Turnstile> (f;g);h)"
proof -
 have " (\<sigma> \<Turnstile>  f ; (g ; h)) = ((\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          ((\<exists>na. enat na \<le> nlength \<sigma> - enat n \<and> g (ntaken na (ndropn n \<sigma>)) \<and> h (ndropn na (ndropn n \<sigma>))) \<or>
           \<not> nfinite (ndropn n \<sigma>) \<and> g (ndropn n \<sigma>))) \<or>
     \<not> nfinite \<sigma> \<and> f \<sigma>)" 
 by (simp add: chop_defs)
 also have " ... = 
          ((\<exists>n. enat n \<le> nlength \<sigma> \<and> 
             (\<exists>na\<le>n. enat na \<le> nlength \<sigma> \<and> f (ntaken (min na n) \<sigma>) \<and> g (ndropn na (ntaken n \<sigma>)))
              \<and> h (ndropn n \<sigma>)) \<or>
     \<not> nfinite \<sigma> \<and> ((\<exists>n. enat n \<le> nlength \<sigma> \<and> f (ntaken n \<sigma>) \<and> g (ndropn n \<sigma>)) \<or> \<not> nfinite \<sigma> \<and> f \<sigma>))" 
   using ChopAssocSemHelp[of \<sigma> f g h] by blast
 also have " ... =
             ( \<sigma> \<Turnstile> (f;g);h)" by (simp add: chop_defs)
 finally show "( \<sigma> \<Turnstile>  f ; (g ; h)) = ( \<sigma> \<Turnstile> (f;g);h) " .
qed

lemma ChopAssocSem:
  " (\<sigma> \<Turnstile>  f ; (g ; h) =  (f;g);h)"
using ChopAssocSemHelp1[of f g h \<sigma>] by auto


subsubsection \<open>OrChopImp\<close>

lemma OrChopImpSem:
  " (\<sigma> \<Turnstile>  ( f \<or> g);h   \<longrightarrow>  f;h \<or> g;h ) "
by (auto simp add: chop_defs )
 

subsubsection \<open>ChopOrImp\<close>

lemma ChopOrImpSem:
  " (\<sigma> \<Turnstile> f;(g \<or> h) \<longrightarrow>   f;g \<or>  f;h ) "
by (auto simp add: chop_defs )

subsubsection \<open>EmptyChop\<close>

lemma EmptyChopSem:
  " (\<sigma> \<Turnstile> empty ; f = f ) " 
by (simp add: chop_defs empty_defs min_def)
   (metis enat_0_iff(1) ndropn_0 nlength_eq_enat_nfiniteD zero_le)


subsubsection \<open>ChopEmpty\<close>

lemma ChopEmptySem:
  " (\<sigma> \<Turnstile> f;empty = f ) "
by (simp add: chop_defs empty_defs min_def)
   (metis cancel_comm_monoid_add_class.diff_cancel enat_diff_cancel_left idiff_enat_enat 
    nfinite_nlength_enat ntaken_all order_refl zero_enat_def)


subsubsection \<open>StateImpBi\<close>

lemma StateImpBiSem:
  " (\<sigma> \<Turnstile> init f \<longrightarrow>   bi (init f) ) "
by (simp add: init_defs bi_defs  )
   

subsubsection \<open>NextImpNotNextNot\<close>

lemma NextImpNotNextNotSem:
  " (\<sigma> \<Turnstile> \<circle> f \<longrightarrow>  \<not> (\<circle> (\<not> f)) ) "
by (simp add: next_defs) 

subsubsection \<open>BiBoxChopImpChop\<close>

lemma BiBoxChopImpChopSem:
  " (\<sigma> \<Turnstile> bi ( f \<longrightarrow> f1) \<and> \<box>(g \<longrightarrow> g1) \<longrightarrow> f;g \<longrightarrow> f1;g1 ) "
by (simp add: bi_defs always_defs chop_defs)
   fastforce

subsubsection \<open>BoxInduct\<close>

lemma box_induct_help_1 :
  "\<And>j. \<forall>n. enat n \<le> nlength \<sigma> \<longrightarrow> f (ndropn n \<sigma>) \<longrightarrow> 
           nlength \<sigma> - enat n = (enat 0) \<or> f (ndropn (Suc 0) (ndropn n \<sigma>)) \<Longrightarrow>
         f \<sigma> \<Longrightarrow> 
         enat j \<le> nlength \<sigma> \<Longrightarrow> 
         f (ndropn j \<sigma>) "
proof -
    fix j
    show " \<forall>n. enat n \<le> nlength \<sigma> \<longrightarrow> f (ndropn n \<sigma>) \<longrightarrow> 
            nlength \<sigma> - enat n = (enat 0) \<or> f (ndropn (Suc 0) (ndropn n \<sigma>)) \<Longrightarrow>
         f \<sigma> \<Longrightarrow> 
         enat j \<le> nlength \<sigma> \<Longrightarrow> 
         f (ndropn j \<sigma>)" 
     proof (induct j arbitrary: \<sigma>)
     case 0
     then show ?case by simp
     next
     case (Suc j)
     then show ?case by (simp add: ndropn_ndropn)
      (metis Suc_ile_eq add.right_neutral enat.simps(3) enat_add_sub_same le_cases 
       le_iff_add not_less zero_enat_def)
     qed
qed

lemma BoxInductSem:
  " (\<sigma> \<Turnstile> \<box> (f \<longrightarrow> wnext f) \<and> f \<longrightarrow> \<box> f)"
proof 
 (auto simp add: always_defs wnext_defs )
 show "\<And>n. \<forall>n. enat n \<le> nlength \<sigma> \<longrightarrow> f (ndropn n \<sigma>) \<longrightarrow> 
             nlength \<sigma> - enat n = enat 0 \<or> f (ndropn (Suc 0) (ndropn n \<sigma>)) \<Longrightarrow>
         f \<sigma> \<Longrightarrow> 
         enat n \<le> nlength \<sigma> \<Longrightarrow> 
         f (ndropn n \<sigma>)" 
 using box_induct_help_1 by blast
qed


subsection \<open>Quantification over State (Flexible) Variables\<close>

text \<open>
  Quantification in Infinite ITL is done similarly as in Finite ITL.
\<close>

typedecl state

instance state :: world ..

type_synonym 'a statefun = "(state,'a) stfun"
type_synonym statepred   = "bool statefun"
type_synonym 'a tempfun  = "(state,'a) formfun"
type_synonym temporal    = "state formula"




subsection \<open>Temporal Quantifiers\<close>

definition exist_state_d :: " ('a statefun \<Rightarrow>  temporal )\<Rightarrow> temporal" (binder "Eex " 10)
where "exist_state_d  F  \<equiv> (\<lambda>s.  (\<exists> x.  s \<Turnstile> F x ))"

syntax
 "_Eex" :: "[idts, lift] \<Rightarrow> lift"      ("(3\<exists>\<exists> _./ _)" [0,10] 10)

translations
 "_Eex v A"  == "Eex v. A" 

definition forall_state_d :: " ('a statefun \<Rightarrow>  temporal )\<Rightarrow> temporal" (binder "Aall " 10)
where "forall_state_d F \<equiv> LIFT(\<not>(\<exists>\<exists> x. \<not>(F x)))"

syntax 
 "_Aall" :: "[idts, lift] \<Rightarrow> lift"      ("(3\<forall>\<forall> _./ _)" [0,10] 10)

translations
 "_Aall v A" == "Aall v. A"

 

end
