(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                Dimitar Guelev
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

chapter \<open>Deep embedding of QPITL\<close>

theory QPITL_Deep imports
 NELList_Extras

begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="cvc4 z3 e spass vampire " ]
*)
text \<open>
This theory is a deep embedding of Quantified Propositional Interval Temporal Logic (QPITL).
It provides the syntax and semantics. 
 
\<close>

section \<open>Syntax\<close>

subsection \<open>Primitive formulae\<close>

datatype  qpitl =
    false_d                         ("ifalse")
  | atom_d "nat"                    ("($_)") 
  | iimp_d " qpitl" " qpitl"        ("( _ iimp _ )" [26,25] 25)
  | next_d " qpitl"                 ("(next _)"  [88] 87)
  | chop_d " qpitl" " qpitl"        ("( _ schop _ )" [84,84] 83)
  | schopstar_d " qpitl"            ("(schopstar _ )" [85] 85)
  | omega_d " qpitl"                ("(omega _ )" [85] 85)  
  | exists_d "nat" "qpitl"          ( "Ex _. _" 10) 

subsection \<open>state formula\<close>

fun state_qpitl :: " qpitl \<Rightarrow> bool" 
where
  "state_qpitl (ifalse)      = True"
| "state_qpitl ($n)          = True "
| "state_qpitl (f iimp g)    = ((state_qpitl f) \<and> (state_qpitl g))"
| "state_qpitl (next f)      = False "
| "state_qpitl (f schop g)   = False "
| "state_qpitl (schopstar f) = False "
| "state_qpitl (omega f)     = False  " 
| "state_qpitl (Ex n. f)     = state_qpitl f" 

subsection \<open>Derived Boolean Operators\<close>

definition inot_d ("(inot _)" [90] 90) 
where 
  "inot f \<equiv> f iimp ifalse"

definition itrue_d ("itrue") 
where 
  "itrue \<equiv> inot  ifalse"

definition ior_d ("(_ ior _)" [31,30] 30) 
where 
  "f ior g \<equiv> (inot f) iimp g"

definition iand_d ("(_ iand _)" [36,35] 35)
where 
  "f iand g \<equiv> inot (inot f ior inot g)"

definition ieqv_d ( "(_ ieqv _) " [21,20] 20)
where 
  "f ieqv g \<equiv> ( ( f iimp g) iand  (g iimp f))" 
 

subsection \<open>more, empty, skip and wnext \<close>

definition more_d ("more") 
where 
  "more \<equiv> next itrue"

definition empty_d ("empty")
where
 "empty \<equiv> inot more"

definition skip_d ("skip") 
where 
  "skip \<equiv> next empty"

definition wnext_d ("(wnext _)" [88] 87)
where
 "wnext f \<equiv> inot (next (inot f))" 


subsection \<open>ifinite and inf\<close>

definition finite_d ("ifinite") 
where 
  "ifinite \<equiv> itrue schop empty"

definition inf_d ("inf") 
where 
  "inf \<equiv> inot ifinite"


subsection \<open>weak chop\<close>

definition chop_d ("( _ ; _ )" [84,84] 83)
 where " f ; g \<equiv> (f schop g) ior (f iand inf)" 

subsection \<open>Box and Diamond Operators\<close>

definition sometimes_d ("(diamond _)" [88] 87)
where 
  "diamond f \<equiv> itrue schop f"

definition always_d ("(box _)" [88] 87)
where 
  "box f \<equiv> inot (diamond (inot f))"

definition di_d ("(di _)" [88] 87)
where
 "di f \<equiv> f ; itrue " 

definition bi_d ("(bi _)" [88] 87)
where 
  "bi f \<equiv> inot (di (inot f))"

definition df_d ("(df _)" [88] 87)
where
 "df f \<equiv> f schop itrue " 

definition bf_d ("(bf _)" [88] 87)
where 
  "bf f \<equiv> inot (df (inot f))"

definition da_d ("(da _)" [88] 87)
where
 "da f \<equiv> itrue schop (f ; itrue) " 

definition ba_d ("(ba _)" [88] 87)
where 
  "ba f \<equiv> inot (da (inot f))"

definition sda_d ("(sda _)" [88] 87)
where
 "sda f \<equiv> itrue schop (f schop itrue) " 

definition sba_d ("(sba _)" [88] 87)
where 
  "sba f \<equiv> inot (sda (inot f))"


subsection \<open>Initial and Final Operators\<close>

definition init_d ("(init _)" [88] 87)
where
 " init f \<equiv> ((empty iand f) schop itrue)"

definition fin_d ("(fin _)" [88] 87)
where
  "fin f \<equiv> diamond ( empty iand f)"

subsection \<open>Forall\<close>

definition forall_d ("Fa _. _" 10)
where
 " Fa v. f \<equiv> inot (Ex v. inot f) " 


subsection \<open> Big operations\<close>


definition big_ior :: "  qpitl list \<Rightarrow>  qpitl "
 where " big_ior F = foldr (\<lambda> x y. x ior y) F ifalse " 

definition big_iand :: " qpitl list \<Rightarrow>  qpitl "
 where " big_iand F = foldr (\<lambda> x y. x iand y) F itrue " 


definition state_qpitl_list :: "qpitl list \<Rightarrow> bool"
 where " state_qpitl_list L = foldr (\<lambda>x y. state_qpitl x \<and> y) L True " 


section \<open>Semantics\<close>

subsection \<open>Intervals\<close>

type_synonym interval = " nat set nellist " 

subsection \<open>Semantics Primitive Operators\<close>

definition similar :: " interval \<Rightarrow> nat \<Rightarrow> interval \<Rightarrow> bool " ("(_ \<sim>_ _)" )   
where "similar \<sigma>0 n \<sigma>1 = (nlength \<sigma>0 = nlength \<sigma>1 \<and>
                          (\<forall> k p . k\<le> nlength \<sigma>0 \<longrightarrow> 
                             (p \<in> (nnth \<sigma>0 k) \<and> p \<noteq> n) \<longleftrightarrow> (p \<in> (nnth \<sigma>1 k) \<and> p \<noteq> n) ) ) " 

definition similars :: " interval \<Rightarrow> nat set \<Rightarrow> interval \<Rightarrow> bool "  ("(_ \<approx>\<^sub>_ _)" )  
where "similars \<sigma>0 A \<sigma>1 = (nlength \<sigma>0 = nlength \<sigma>1 \<and>
                           (\<forall> k p. k\<le> nlength \<sigma>0 \<longrightarrow>  p \<notin> A \<longrightarrow> (p \<in> (nnth \<sigma>0 k) \<longleftrightarrow> p \<in> (nnth \<sigma>1 k)))) " 

primrec fvars :: "qpitl \<Rightarrow> nat set"
where 
  "fvars ifalse        = {} "
| "fvars ($n)          = {n} "
| "fvars (f iimp g)    = (fvars f) \<union> (fvars g) "
| "fvars (next f)      = fvars f "
| "fvars (f schop g)   = (fvars f) \<union> (fvars g) "
| "fvars (schopstar f) = fvars f"
| "fvars (omega f)     = fvars f"  
| "fvars (Ex n. f)     = (fvars f) - {n} " 

primrec bvars :: "qpitl \<Rightarrow> nat set"
where 
  "bvars ifalse        = {} "
| "bvars ($n)          = {} "
| "bvars (f iimp g)    = (bvars f) \<union> (bvars g) "
| "bvars (next f)      = bvars f "
| "bvars (f schop g)   = (bvars f) \<union> (bvars g) "
| "bvars (schopstar f) = bvars f"
| "bvars (omega f)     = bvars f"  
| "bvars (Ex n. f)     = (insert n (bvars f)) " 

primrec suform :: "qpitl \<Rightarrow> nat \<Rightarrow> qpitl \<Rightarrow> qpitl "
where 
  " suform ifalse       n h = ifalse " 
| " suform ($k)         n h = (if k = n then h else ($k)) "
| " suform (f iimp g)   n h = ((suform f n h) iimp (suform g n h)) " 
| " suform (next f)     n h = (next (suform f n h)) " 
| " suform (f schop g)  n h = ((suform f n h) schop (suform g n h)) " 
| " suform (schopstar f) n h = (schopstar (suform f n h)) " 
| " suform (omega f) n h     = (omega (suform f n h)) " 
| " suform (Ex k. f)    n h = (Ex k. (suform f n h)) " 

primrec rename ::  "qpitl \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> qpitl" 
where 
   " rename ifalse n k       = ifalse "
|  " rename ($m) n k         = (if m = n then $k else $m) "
|  " rename (f iimp g) n k   = ((rename f n k) iimp (rename g n k)) " 
|  " rename (next f) n k     = (next (rename f n k)) "
|  " rename (f schop g) n k  = ((rename f n k) schop (rename g n k)) "
|  " rename (schopstar f) n k = (schopstar (rename f n k)) " 
|  " rename (omega f) n k     = (omega (rename f n k)) " 
|  " rename (Ex m. f) n k    = (Ex (if m = n then k else m). (rename f n k)) "

 
primrec max_var :: "qpitl \<Rightarrow> nat \<Rightarrow> nat" 
where 
   " max_var ifalse m       = m  "
|  " max_var ($k) m         = (if m < k then k else m) " 
|  " max_var (f iimp g) m   = (if (max_var f m) < (max_var g m) then (max_var g m) else (max_var f m)) " 
|  " max_var (next f) m     = max_var f m "
|  " max_var (f schop g) m  = (if (max_var f m) < (max_var g m) then (max_var g m) else (max_var f m)) "
|  " max_var (schopstar f) m = max_var f m "
|  " max_var (omega f) m     = max_var f m "  
|  " max_var (Ex k. f) m    = (if (max_var f m) < k then k else (max_var f m)) " 


primcorec upd :: " interval \<Rightarrow> nat \<Rightarrow> (bool nellist) \<Rightarrow> interval " 
where 
" upd \<sigma> n l = (case \<sigma> of (NNil b) \<Rightarrow> 
                      (NNil (if (nfirst l) then  insert n b else b - {n})) |
                      (NCons x \<sigma>') \<Rightarrow> 
                        (case l of (NNil a) \<Rightarrow> 
                            (NNil (if a then insert n x else x - {n})) |
                          (NCons y l') \<Rightarrow>
                          (NCons (if y then (insert n x) else (x - {n}))
                                 (upd \<sigma>' n l')))) "  

simps_of_case upd_simps [simp, code, nitpick_simp]: upd.code



fun semantics_qpitl :: "[interval, qpitl] \<Rightarrow> bool" ("(_ \<Turnstile> _)" [80,10] 10)
where
  "(\<sigma> \<Turnstile> ifalse) = False"
| "(\<sigma> \<Turnstile>  $p) = (p \<in> (nfirst \<sigma>))"
| "(\<sigma> \<Turnstile> f iimp g) = ((\<sigma> \<Turnstile> f) \<longrightarrow> (\<sigma> \<Turnstile> g))"
| "(\<sigma> \<Turnstile> (next f))  = (1 \<le> nlength \<sigma>  \<and> ((ndropn 1 \<sigma>) \<Turnstile> f))"
| "(\<sigma> \<Turnstile> f schop g) =
   (\<exists>i. ((0 \<le> i) \<and> (i \<le> (nlength \<sigma>)) \<and> ((ntaken i \<sigma>) \<Turnstile> f) \<and> ((ndropn i \<sigma>) \<Turnstile> g)) )"
| "(\<sigma> \<Turnstile> (schopstar f)) = 
    (\<exists> (l:: nat nellist). 
                   (nnth l 0) = 0 \<and> nfinite l \<and> nidx l  \<and> 
                   (enat (nlast l)) = (nlength \<sigma>) \<and> nfinite \<sigma> \<and> 
                   (\<forall> (i::nat) . ( (enat i)< (nlength l)) \<longrightarrow> 
                               ((nsubn \<sigma> (nnth l i) (nnth l (Suc i)) ) \<Turnstile> f) )
             )
    "
| "(\<sigma> \<Turnstile> (omega f)) = 
    (\<exists>(l:: nat nellist). 
              \<not>nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
              (\<forall>i. (nnth l i) \<le> nlength \<sigma>) \<and>
              (\<forall>i. ( (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> f) ) 
           ) " 
| " (\<sigma> \<Turnstile> (Ex n. f) ) = (\<exists> l. nlength \<sigma> = nlength l \<and> ((upd \<sigma> n l)  \<Turnstile> f) ) " 



lemma similar_defs [simp]: 
 "  similar \<sigma>0 n \<sigma>1  \<longleftrightarrow>  (nlength \<sigma>0 = nlength \<sigma>1 \<and> 
                           (\<forall> k p . k\<le> nlength \<sigma>0 \<longrightarrow>  (p \<in> (nnth \<sigma>0 k) \<and> p \<noteq> n) \<longleftrightarrow> (p \<in> (nnth \<sigma>1 k) \<and> p \<noteq> n)) )  "
unfolding similar_def by auto

lemma similar_defs_alt: 
 " (nlength \<sigma>0 = nlength \<sigma>1 \<and> (\<forall> k . k\<le> nlength \<sigma>0 \<longrightarrow>  (nnth \<sigma>0 k) - {n} = (nnth \<sigma>1 k) - {n})) \<longleftrightarrow> similar \<sigma>0 n \<sigma>1   "
unfolding similar_def by auto

lemma similars_defs [simp]:
 "similars \<sigma>0 A \<sigma>1  \<longleftrightarrow>  (nlength \<sigma>0 = nlength \<sigma>1 \<and>
                           (\<forall> k p. k\<le> nlength \<sigma>0 \<longrightarrow>  p \<notin> A \<longrightarrow> ( p \<in> (nnth \<sigma>0 k) \<longleftrightarrow> p \<in> (nnth \<sigma>1 k)))) " 
unfolding similars_def by auto

lemma similars_defs_alt:
 "similars \<sigma>0 A \<sigma>1 \<longleftrightarrow>  (nlength \<sigma>0 = nlength \<sigma>1 \<and> (\<forall> k. k\<le> nlength \<sigma>0 \<longrightarrow>  (nnth \<sigma>0 k) - A = (nnth \<sigma>1 k) - A))  " 
unfolding similars_def by auto

lemma similar_similars: 
 " similar \<sigma>0 n \<sigma>1 = similars \<sigma>0 {n} \<sigma>1" 
using similar_defs_alt similars_defs_alt by auto

lemma similars_mono: 
 assumes "A \<subseteq> B "
         "similars \<sigma>0 A \<sigma>1"
 shows " similars \<sigma>0 B \<sigma>1" 
using assms
by fastforce 


lemma similar_refl:
 "similar \<sigma> n \<sigma> " 
by simp

lemma similar_commute: 
 "similar \<sigma> n \<sigma>'  = similar \<sigma>' n \<sigma>  "
by fastforce 


lemma similar_trans:
 assumes " similar \<sigma> n \<sigma>'  "
         " similar \<sigma>' n \<sigma>'' "
 shows   " similar \<sigma> n \<sigma>'' "
using assms by simp 


lemma similars_refl:
 "similars \<sigma> A \<sigma> " 
by simp

lemma similars_trans:
 assumes " similars \<sigma> A \<sigma>'  "
         " similars \<sigma>' A \<sigma>'' "
 shows   " similars \<sigma> A \<sigma>'' "
using assms by simp  


lemma upd_NCons: 
 "upd (NCons x xs) n (NCons y ys) = 
  (if y then NCons (insert n x) (upd xs n ys) else  NCons (x - {n}) (upd xs n ys))" 
by simp

lemma upd_NNil: 
 "upd (NNil a) n (NNil b) = 
  (if b then  (NNil (insert n a)) else (NNil (a - {n}))) " 
by (simp add: nfirst_eq_nnth_zero nnth_NNil)


lemma nnth_upd: 
assumes "nlength \<sigma> = nlength l"
        " (enat k) \<le> nlength \<sigma>"
shows  " (nnth (upd \<sigma> n l) k) = (if (nnth l k) then (insert n (nnth \<sigma> k)) else (nnth \<sigma> k) - {n}) " 
using assms 
proof (induction k arbitrary: \<sigma> l)
case 0
then show ?case 
 proof (cases \<sigma>)
 case (NNil x1)
 then show ?thesis
 by (simp add: nfirst_eq_nnth_zero nnth_NNil) 
 next
 case (NCons x21 x22)
 then show ?thesis 
   proof -
    have 1: "\<exists> b bs. l = (NCons b bs)" 
     using 0
      by (metis NCons nellist_split_2_first nlength_NCons order_le_less zero_enat_def zero_ne_eSuc)
    obtain b bs where 2: "l = (NCons b bs)" 
      using 1 by blast
    show ?thesis by (simp add: "2" NCons)
 qed
 qed
next
case (Suc k)
then show ?case 
   proof (cases \<sigma>)
   case (NNil x1)
   then show ?thesis using Suc enat_0_iff(1) by auto
   next
   case (NCons x21 x22)
   then show ?thesis 
     proof -
      have 1: "\<exists> b bs. l = (NCons b bs)" 
       using Suc 
       by (metis enat_0_iff(1) linorder_not_less nat.simps(3) nellist_split_2_first not_gr_zero)
      obtain b bs where 2: "l = (NCons b bs)"
        using 1 by blast
      show ?thesis using 2 Suc 
      by (metis "2" NCons eSuc_enat eSuc_ile_mono eSuc_inject nlength_NCons nnth_Suc_NCons upd_simps(3)) 
   qed
  qed
qed


lemma nlength_upd: 
 assumes "nlength \<sigma> = nlength l"
 shows "nlength ((upd \<sigma> n l)) = nlength \<sigma>" 
using assms
proof (coinduction arbitrary: \<sigma> l rule: enat_coinduct)
case Eq_enat
then show ?case 
  proof (cases "nlength \<sigma> = 0")
  case True
  then show ?thesis 
     by (metis True nlength_NNil ntaken_0 ntaken_all order_le_less upd_simps(1) zero_enat_def)
    
  next
  case False
  then show ?thesis 
    proof -
     have 2: "nlength l > 0 "
       using False by (simp add: Eq_enat)
     show ?thesis using False 2 by simp 
   (metis "2" Eq_enat co.enat.sel(2) nellist_split_2_first nlength_NCons upd_simps(3) zero_ne_eSuc)
  qed
 qed
qed

lemma upd_nmap_nzip: 
assumes " nlength \<sigma> = nlength l"
 shows " upd \<sigma> n l = nmap (\<lambda> (x,y). (if y then insert n x else x - {n})) (nzip \<sigma> l) " 
proof -
 have 1: "nlength (upd \<sigma> n l) = nlength (nmap (\<lambda> (x,y). (if y then insert n x else x - {n})) (nzip \<sigma> l))"
 using assms nlength_upd by auto 
 have 2: "\<And>i. i\<le> nlength (upd \<sigma> n l) \<Longrightarrow>
             (nnth (upd \<sigma> n l) i) =
             (nnth (nmap (\<lambda> (x,y). (if y then insert n x else x - {n})) (nzip \<sigma> l)) i) "
             by (simp add: assms nlength_upd nnth_nzip nnth_upd) 
 show ?thesis
 using "1" "2" nellist_eq_nnth_eq by blast 
qed
   
   
lemma upd_swap: 
 assumes "nlength \<sigma> = nlength l1"
         "nlength \<sigma> = nlength l2" 
         " x1 \<noteq> x2 " 
 shows " upd (upd \<sigma> x1 l1) x2 l2 = (upd (upd \<sigma> x2 l2) x1 l1) " 
proof -
 have 1: "nlength (upd (upd \<sigma> x1 l1) x2 l2) = nlength (upd (upd \<sigma> x2 l2) x1 l1)"
 using assms nlength_upd by auto
 have 2: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow>
           (nnth (upd (upd \<sigma> x1 l1) x2 l2) i) = 
         (if (nnth l2 i) then insert x2 (nnth (upd \<sigma> x1 l1) i) else (nnth (upd \<sigma> x1 l1) i) - {x2})" 
   using assms by (simp add: nlength_upd nnth_upd) 
 have 3: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow>
           (nnth (upd (upd \<sigma> x2 l2) x1 l1) i) = 
          (if (nnth l1 i) then insert x1 (nnth (upd \<sigma> x2 l2) i) else (nnth (upd \<sigma> x2 l2) i) - {x1})" 
   using assms by (simp add: nlength_upd nnth_upd) 
 have 4: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow>
           (if (nnth l2 i) then insert x2 (nnth (upd \<sigma> x1 l1) i) else (nnth (upd \<sigma> x1 l1) i) - {x2}) =
           (if (nnth l1 i) then insert x1 (nnth (upd \<sigma> x2 l2) i) else (nnth (upd \<sigma> x2 l2) i) - {x1}) " 
    using assms nnth_upd nlength_upd by auto
 show ?thesis
 by (simp add: "1" "2" "3" "4" nellist_eq_nnth_eq) 
qed

lemma upd_absorb: 
 assumes "nlength \<sigma> = nlength l1"
         "nlength \<sigma> = nlength l2" 
         " x1 = x2 " 
 shows " upd (upd \<sigma> x1 l1) x2 l2 = (upd \<sigma> x2 l2) "
proof -
 have 1: "nlength (upd (upd \<sigma> x1 l1) x2 l2) = nlength (upd \<sigma> x2 l2)"
 using assms nlength_upd by auto  
 have 2: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow>
           (nnth (upd (upd \<sigma> x1 l1) x2 l2) i) = 
         (if (nnth l2 i) then insert x2 (nnth (upd \<sigma> x1 l1) i) else (nnth (upd \<sigma> x1 l1) i) - {x2})" 
   using assms by (simp add: nlength_upd nnth_upd)
 have 3: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow> 
           (nnth (upd \<sigma> x2 l2) i) =
           (if (nnth l2 i) then insert x2 (nnth \<sigma> i) else (nnth \<sigma> i) - {x2})"
    using assms  by (simp add: nlength_upd nnth_upd)
 have 4: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow> 
            (if (nnth l2 i) then insert x2 (nnth (upd \<sigma> x1 l1) i) else (nnth (upd \<sigma> x1 l1) i) - {x2}) =
            (if (nnth l2 i) then insert x2 (nnth \<sigma> i) else (nnth \<sigma> i) - {x2})"
   using assms  nnth_upd nlength_upd by auto
 show ?thesis
 by (simp add: "1" "2" "3" "4" nellist_eq_nnth_eq) 
qed
   
lemma upd_nappend: 
 assumes "nlength \<sigma>1 = nlength l1"
         "nlength \<sigma>2 = nlength l2"
 shows " upd (nappend \<sigma>1 \<sigma>2) x (nappend l1 l2) = 
         (nappend (upd \<sigma>1 x l1) (upd \<sigma>2 x l2)) " 
using assms
by (simp add: nmap_nappend_distrib nzip_nappend upd_nmap_nzip)

 
lemma ntaken_upd:
 assumes "nlength \<sigma> = nlength l"
         "j \<le> nlength \<sigma> "
 shows "ntaken j (upd \<sigma> x l) = (upd (ntaken j \<sigma>) x (ntaken j l)) "
using assms 
by (simp add: ntaken_nzip upd_nmap_nzip)


lemma ndropn_upd:
 assumes "nlength \<sigma> = nlength l"
         "j \<le> nlength \<sigma> "
 shows "ndropn j (upd \<sigma> x l) = (upd (ndropn j \<sigma>) x (ndropn j l)) "
using assms ndropn_nzip upd_nmap_nzip by (simp add: ndropn_nmap)

 
lemma nsubn_upd:
 assumes "nlength \<sigma> = nlength l"
         " (enat j) \<le> enat k" 
         " (enat k) \<le> nlength \<sigma> "
 shows " nsubn (upd \<sigma> x l) j k  = (upd (nsubn  \<sigma> j k) x (nsubn  l j k))" 
proof -
 have 1: "nsubn (upd \<sigma> x l) j k = ntaken (k - j) (ndropn j (upd \<sigma> x l))" 
   using assms unfolding nsubn_def1 by simp
 have 2: " (ndropn j (upd \<sigma> x l)) = (upd (ndropn j \<sigma>) x (ndropn j l))"  
   using assms by (meson dual_order.trans ndropn_upd)  
 have 3: " ntaken (k - j) (upd (ndropn j \<sigma>) x (ndropn j l)) =
           (upd (ntaken (k-j) (ndropn j \<sigma>)) x (ntaken (k-j) (ndropn j l)))"
    using assms  ntaken_upd[of "(ndropn j \<sigma>)" "(ndropn j l)" "(k - j)" x]
    by (metis ndropn_nlength nle_le nlength_upd ntaken_all)
 have 4: "(upd (ntaken (k-j) (ndropn j \<sigma>)) x (ntaken (k-j) (ndropn j l))) =
          (upd (nsubn  \<sigma> j k) x (nsubn  l j k))" 
    unfolding nsubn_def1 by simp
 show ?thesis using "1" "2" "3" "4" by presburger 
qed       
 
lemma nfirst_upd: 
 assumes " nlength \<sigma> = nlength l" 
 shows "nfirst (upd \<sigma> x l) = (if nfirst l then insert x (nfirst \<sigma>) else (nfirst \<sigma>) - {x} )"
 by (metis assms nfirst_eq_nnth_zero nnth_upd zero_enat_def zero_le) 

lemma nlast_upd: 
 assumes " nlength \<sigma> = nlength l" 
         "nfinite \<sigma> "
 shows " nlast (upd \<sigma> x l) = (if nlast l then insert x (nlast \<sigma>) else (nlast \<sigma>) - {x} ) " 
using assms 
by (metis dual_order.order_iff_strict nfinite_conv_nlength_enat nlength_upd nnth_nlast nnth_upd the_enat.simps)


lemma upd_nfuse: 
 assumes "nlength \<sigma>1 = nlength l1"
         "nlength \<sigma>2 = nlength l2"
         "nfirst \<sigma>2 = nlast \<sigma>1" 
         " nfinite \<sigma>1"
         " nfirst l2 = nlast l1" 
 shows " upd (nfuse \<sigma>1 \<sigma>2) x (nfuse l1 l2) = 
         (nfuse (upd \<sigma>1 x l1) (upd \<sigma>2 x l2)) " 
using assms by (simp add: nfuse_nlength nmap_nfuse nzip_nfuse upd_nmap_nzip)


lemma similar_upd: 
assumes "nlength \<sigma> = nlength l"
shows " similar \<sigma> n (upd \<sigma> n l) " 
using assms 
using nlength_upd nnth_upd by auto




lemma exist_sigma_exist_value: 
" (\<exists> \<sigma>'. (\<sigma>' \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>')) \<longleftrightarrow> (\<exists> l. nlength \<sigma> = nlength l \<and> ((upd \<sigma> n l)  \<Turnstile> f) ) "
proof - 
 have 2: "(\<exists>\<sigma>'. (\<sigma>' \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>')) \<Longrightarrow> 
          (\<exists> l. nlength \<sigma> = nlength l \<and> ((upd \<sigma> n l)  \<Turnstile> f) )" 
  proof -
   assume a0: "(\<exists>\<sigma>'. (\<sigma>' \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>'))"
   show "(\<exists> l. nlength \<sigma> = nlength l \<and> ((upd \<sigma> n l)  \<Turnstile> f) )"
   proof -
    obtain \<sigma>' where 3: "(\<sigma>' \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>')"
      using a0 by blast
    have 30: "nlength \<sigma>' = nlength (upd \<sigma>' n (nmap (\<lambda>x. n \<in> x) \<sigma>'))"
      by (simp add: nlength_upd) 
    have 31: "\<And>k . k \<le> nlength \<sigma>' \<Longrightarrow> nnth \<sigma>' k = (nnth (upd \<sigma>' n (nmap (\<lambda>x. n \<in> x) \<sigma>')) k)"
       by (simp add: insert_absorb nnth_upd) 
    have 4: "\<sigma>' = (upd \<sigma>' n (nmap (\<lambda>x. n \<in> x) \<sigma>'))"
        by (simp add: "30" "31" nellist_eq_nnth_eq)
    have 40: "nlength \<sigma> = nlength (upd \<sigma> n (nmap (\<lambda>x. n \<in> x) \<sigma>))"
      by (simp add: nlength_upd) 
    have 41: "\<And>k. k\<le> nlength \<sigma> \<Longrightarrow> nnth \<sigma> k = (nnth (upd \<sigma> n (nmap (\<lambda>x. n \<in> x) \<sigma>)) k)"
       by (simp add: insert_absorb nnth_upd) 
    have 8: "\<sigma> = (upd \<sigma> n (nmap (\<lambda>x. n \<in> x) \<sigma>))" 
       by (simp add: "40" "41" nellist_eq_nnth_eq)  
    have 12: "(similar \<sigma> n \<sigma>') " 
        using "3" by auto
    have 13: "nlength \<sigma> = nlength \<sigma>'" 
         using "12" by auto
    have 14: "\<And> k . k \<le> nlength \<sigma> \<Longrightarrow> (nnth \<sigma>' k) = (nnth (upd \<sigma> n (nmap (\<lambda>x. n \<in> x) \<sigma>')) k) " 
       using 13 nnth_upd 12 by auto
    have 15: "nlength \<sigma>' = nlength (upd \<sigma> n (nmap (\<lambda>x. n \<in> x) \<sigma>'))" 
       by (simp add: "13" nlength_upd)
    have 16: "\<sigma>' = (upd \<sigma> n (nmap (\<lambda>x. n \<in> x) \<sigma>'))"
        using nellist_eq_nnth_eq[of \<sigma>' "(upd \<sigma> n (nmap (\<lambda>x. n \<in> x) \<sigma>'))"] 
      using "13" "14" "15" by presburger
    show ?thesis 
     by (metis "13" "16" "3" nlength_nmap)
    qed
   qed
  have 17: "(\<exists> l. nlength \<sigma> = nlength l \<and> ((upd \<sigma> n l)  \<Turnstile> f) ) \<Longrightarrow> 
            (\<exists>\<sigma>'. (\<sigma>' \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>'))"  
   proof -
    assume a1: "(\<exists> l. nlength \<sigma> = nlength l \<and> ((upd \<sigma> n l)  \<Turnstile> f) )"
    show "(\<exists>\<sigma>'. (\<sigma>' \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>'))"
    proof -
     obtain l where 18: " nlength \<sigma> = nlength l \<and> ((upd \<sigma> n l)  \<Turnstile> f)"
       using a1 by blast
     have 19: "similar \<sigma> n (upd \<sigma> n l)"
       using "18" nlength_upd nnth_upd by auto     
     show ?thesis 
     using "18" "19" by blast
    qed
   qed
  show ?thesis
  using "17" "2" by blast 
qed 

 
lemma not_fvar_upd: 
 assumes " n \<notin> fvars f"
         " nlength l = nlength \<sigma>" 
 shows " (\<sigma> \<Turnstile> f) = ((upd \<sigma> n l) \<Turnstile> f) " 
using assms
proof (induction f arbitrary: \<sigma> l )
case false_d
then show ?case 
using semantics_qpitl.simps(1) by presburger
next
case (atom_d x)
then show ?case by simp 
 (metis Diff_iff empty_iff insert_iff nfirst_eq_nnth_zero nnth_upd zero_enat_def zero_le)
next
case (iimp_d f1 f2)
then show ?case by simp 
next
case (next_d f)
then show ?case using nlength_upd by simp
 (metis One_nat_def ndropn_nlength ndropn_upd one_enat_def) 
next
case (chop_d f1 f2)
then show ?case 
  proof -
   have c1: "(\<sigma> \<Turnstile>  f1 schop f2 ) = 
            (\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> f1) \<and> (ndropn i \<sigma> \<Turnstile> f2)) " by simp 
   have c2: "(upd \<sigma> n l \<Turnstile>  f1 schop f2 ) = 
             (\<exists>i. enat i \<le> nlength (upd \<sigma> n l) \<and> (ntaken i (upd \<sigma> n l) \<Turnstile> f1) \<and>
                   (ndropn i (upd \<sigma> n l) \<Turnstile> f2))" 
     by simp
   have c3: "(\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> f1) \<and> (ndropn i \<sigma> \<Turnstile> f2)) \<Longrightarrow>
             (\<exists>i. enat i \<le> nlength (upd \<sigma> n l) \<and> (ntaken i (upd \<sigma> n l) \<Turnstile> f1) \<and>
                   (ndropn i (upd \<sigma> n l) \<Turnstile> f2))"
    proof -
     assume a0: "(\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> f1) \<and> (ndropn i \<sigma> \<Turnstile> f2))"
     show "(\<exists>i. enat i \<le> nlength (upd \<sigma> n l) \<and> (ntaken i (upd \<sigma> n l) \<Turnstile> f1) \<and>
                   (ndropn i (upd \<sigma> n l) \<Turnstile> f2))"
     proof -
     obtain i where c31: "enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> f1) \<and> (ndropn i \<sigma> \<Turnstile> f2) " 
       using a0 by blast
     have c32: "n \<notin> fvars f1"
       using chop_d.prems(1) by auto 
     have c33: "nlength (ntaken i l) = nlength (ntaken i \<sigma>)" 
       by (simp add: chop_d.prems(2))
     have c34: "(upd (ntaken i \<sigma>) n (ntaken i l)  \<Turnstile> f1)"
         using chop_d.IH(1)[of "(ntaken i l)" "(ntaken i \<sigma>)" ]  c31 c32 c33 by blast
     have c35: "n \<notin> fvars f2"
       using chop_d.prems(1) by auto 
     have c36: "nlength (ndropn i l) = nlength (ndropn i \<sigma>)"
       by (simp add: chop_d.prems(2))
     have c37: "(upd (ndropn i \<sigma>) n (ndropn i l) \<Turnstile> f2)"
      using chop_d.IH(2)[of "(ndropn i l)" "(ndropn i \<sigma>)" ] c31 c35 c36 by blast
     have c38: "(upd (ntaken i \<sigma>) n (ntaken i l)) = ntaken i (upd \<sigma> n l)"
       by (simp add: c31 chop_d.prems(2) ntaken_upd)
     have c39: "(upd (ndropn i \<sigma>) n (ndropn i l)) = ndropn i (upd \<sigma> n l)"
       by (simp add: c31 chop_d.prems(2) ndropn_upd)
     have c40: " enat i \<le> nlength (upd \<sigma> n l)"
     by (simp add: c31 chop_d.prems(2) nlength_upd)
     have c41: "(ntaken i (upd \<sigma> n l) \<Turnstile> f1)"
     using c34 c38 by auto 
     have c42: "(ndropn i (upd \<sigma> n l) \<Turnstile> f2)"
     using c37 c39 by auto 
     show ?thesis
     using c40 c41 c42 by auto
    qed
   qed
  have c5: "(\<exists>i. enat i \<le> nlength (upd \<sigma> n l) \<and> (ntaken i (upd \<sigma> n l) \<Turnstile> f1) \<and>
                   (ndropn i (upd \<sigma> n l) \<Turnstile> f2)) \<Longrightarrow>
            (\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> f1) \<and> (ndropn i \<sigma> \<Turnstile> f2)) " 
  proof -
   assume a1: "(\<exists>i. enat i \<le> nlength (upd \<sigma> n l) \<and> (ntaken i (upd \<sigma> n l) \<Turnstile> f1) \<and>
                   (ndropn i (upd \<sigma> n l) \<Turnstile> f2))"
   show "(\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> f1) \<and> (ndropn i \<sigma> \<Turnstile> f2))"
   proof -
    obtain i where c51: "enat i \<le> nlength (upd \<sigma> n l) \<and> (ntaken i (upd \<sigma> n l) \<Turnstile> f1) \<and>
                   (ndropn i (upd \<sigma> n l) \<Turnstile> f2)"
    using a1 by blast
    have c52: "nlength (upd \<sigma> n l) = nlength \<sigma>"
    by (simp add: chop_d.prems(2) nlength_upd)    
    have c53: "(ntaken i (upd \<sigma> n l)) = upd (ntaken i \<sigma>) n (ntaken i l)"
    using c51 c52 chop_d.prems(2) ntaken_upd by auto  
    have c54: "n \<notin> fvars f1"
    using chop_d.prems(1) by auto 
    have c55: "(ntaken i \<sigma> \<Turnstile> f1)"
    by (metis c51 c53 c54 chop_d.IH(1) chop_d.prems(2) ntaken_nlength)
    have c56: "n \<notin> fvars f2"
    using chop_d.prems(1) by auto 
    have c57: "(ndropn i (upd \<sigma> n l)) = upd (ndropn i \<sigma>) n (ndropn i l)"
    using c51 c52 chop_d.prems(2) ndropn_upd by auto 
    have c58: "(ndropn i \<sigma> \<Turnstile> f2)"
    by (metis c51 c56 c57 chop_d.IH(2) chop_d.prems(2) ndropn_nlength) 
    show ?thesis
    using c51 c52 c55 c58 by auto
   qed
  qed
  show ?thesis
  using c1 c2 c3 c5 by blast
 qed         
next
case (schopstar_d f)
then show ?case 
  proof -
   have s1: "(\<sigma> \<Turnstile> schopstar f ) = (\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength \<sigma> \<and>
         nfinite \<sigma> \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn \<sigma> (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f)))" by simp 
   have s2: "((upd \<sigma> n l) \<Turnstile> schopstar f ) =  (\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength (upd \<sigma> n l) \<and>
         nfinite (upd \<sigma> n l) \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn (upd \<sigma> n l) (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f)))" by simp
   have s3: "(\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength \<sigma> \<and>
         nfinite \<sigma> \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn \<sigma> (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f))) \<Longrightarrow> 
          (\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength (upd \<sigma> n l) \<and>
         nfinite (upd \<sigma> n l) \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn (upd \<sigma> n l) (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f)))"
    proof -
     assume a2: "(\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength \<sigma> \<and>
         nfinite \<sigma> \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn \<sigma> (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f)))"
     show "(\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength (upd \<sigma> n l) \<and>
         nfinite (upd \<sigma> n l) \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn (upd \<sigma> n l) (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f)))"
     proof -
      obtain l1 where s31: "nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength \<sigma> \<and>
         nfinite \<sigma> \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn \<sigma> (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f))"
       using a2 by blast
     have s32: "nlength (upd \<sigma> n l) = nlength \<sigma>"
     by (simp add: nlength_upd schopstar_d.prems(2))  
     have s33: "\<And>i. enat i < nlength l1 \<longrightarrow> 
                 nlength (nsubn l (nnth l1 i) (nnth l1 (Suc i))) =
                 nlength(nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)))"
                 by (simp add: nsubn_nlength schopstar_d.prems(2)) 
     have s33: "\<And>i.   enat i < nlength l1 \<longrightarrow> 
                 (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f) = 
                  (upd (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) n 
                       (nsubn l (nnth l1 i) (nnth l1 (Suc i))) \<Turnstile> f) "
     using fvars.simps(6) s33 schopstar_d.IH schopstar_d.prems(1) by blast 
     have s34: "\<And>i. enat i < nlength l1 \<longrightarrow> 
                 (upd (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) n 
                       (nsubn l (nnth l1 i) (nnth l1 (Suc i)))) =
                 (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i))) "
     by (metis eSuc_enat enat_ord_simps(1) ileI1 nidx_all_le_nlast nidx_expand nsubn_upd
          order_less_imp_le s31 schopstar_d.prems(2))     
     have s35: "(\<forall>i. enat i < nlength l1 \<longrightarrow>
                 (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)) "
     using s31 s33 s34 by auto 
     have s36: "nfinite (upd \<sigma> n l) "
     by (metis nlength_eq_enat_nfiniteD s31 s32) 
     show ?thesis
     using s31 s32 s35 s36 by auto
    qed   
   qed
  have s4: "(\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength (upd \<sigma> n l) \<and>
         nfinite (upd \<sigma> n l) \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn (upd \<sigma> n l) (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f))) \<Longrightarrow>
          (\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength \<sigma> \<and>
         nfinite \<sigma> \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn \<sigma> (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f))) " 
   proof -
    assume a3: "(\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength (upd \<sigma> n l) \<and>
         nfinite (upd \<sigma> n l) \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn (upd \<sigma> n l) (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f)))"
    show "(\<exists>l1. nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength \<sigma> \<and>
         nfinite \<sigma> \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn \<sigma> (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f)))"
    proof -
     obtain l1 where s41: "nnth l1 0 = 0 \<and>
         nfinite l1 \<and>
         nidx l1 \<and>
         enat (nlast l1) = nlength (upd \<sigma> n l) \<and>
         nfinite (upd \<sigma> n l) \<and>
         (\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn (upd \<sigma> n l) (nnth l1 i)
                (nnth l1 (Suc i)) \<Turnstile> f))"
      using a3 by blast  
    have s42: "nlength (upd \<sigma> n l) = nlength \<sigma>"
    by (simp add: nlength_upd schopstar_d.prems(2))  
    have s43: "\<And>i.  enat i < nlength l1 \<longrightarrow>
               nlength (nsubn l (nnth l1 i) (nnth l1 (Suc i))) =
                 nlength(nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) "
    by (simp add: nsubn_nlength schopstar_d.prems(2)) 
    have s44: "\<And>i.  enat i < nlength l1 \<longrightarrow> 
               (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i))) = 
               upd (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) n 
                   (nsubn l (nnth l1 i) (nnth l1 (Suc i)))"
    by (metis eSuc_enat enat_ord_simps(1) ileI1 nidx_all_le_nlast nidx_expand nsubn_upd 
        order_less_imp_le s41 s42 schopstar_d.prems(2)) 
    have s45: "\<And>i. enat i < nlength l1 \<longrightarrow>
                  (upd (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) n 
                       (nsubn l (nnth l1 i) (nnth l1 (Suc i))) \<Turnstile> f) =
                  (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)"
    using fvars.simps(6) s43 schopstar_d.IH schopstar_d.prems(1) by blast 
    have s46: "(\<forall>i. enat i < nlength l1 \<longrightarrow>
              (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f))"
              using s41 s44 s45 by auto 
    have s47: "nfinite \<sigma>"
    by (metis nlength_eq_enat_nfiniteD s41 s42) 
    show ?thesis
    using s41 s42 s46 s47 by auto
   qed
  qed
  show ?thesis
  using s4 s1 s2 s3 by blast
 qed
next
case (omega_d f)
then show ?case 
  proof -
   have o1: "(\<sigma> \<Turnstile> omega f ) = (\<exists>l1. \<not> nfinite l1 \<and>
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength \<sigma>) \<and>
         (\<forall>i. nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)) " 
   by simp 
   have o2: "((upd \<sigma> n l) \<Turnstile> omega f ) =  (\<exists>l1. \<not>nfinite l1 \<and> 
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength (upd \<sigma> n l)) \<and>
         (\<forall>i. (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)))" 
   by simp
   have o3: "(\<exists>l1. \<not> nfinite l1 \<and>
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength \<sigma>) \<and>
         (\<forall>i. nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)) \<Longrightarrow> 
         (\<exists>l1. \<not>nfinite l1 \<and> 
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength (upd \<sigma> n l)) \<and>
         (\<forall>i. (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)))"
    proof -
     assume a5: "(\<exists>l1. \<not> nfinite l1 \<and>
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength \<sigma>) \<and>
         (\<forall>i. nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f))"
     show "(\<exists>l1. \<not>nfinite l1 \<and> 
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength (upd \<sigma> n l)) \<and>
         (\<forall>i. (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)))"
     proof -
      obtain l1 where o31: "\<not> nfinite l1 \<and>
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength \<sigma>) \<and>
         (\<forall>i. nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)"
       using a5 by blast
     have o32: "(\<forall>i. enat (nnth l1 i) \<le> nlength (upd \<sigma> n l))"
     by (simp add: nlength_upd o31 omega_d.prems(2))
     have o33: "\<And>i. 
                 nlength (nsubn l (nnth l1 i) (nnth l1 (Suc i))) =
                 nlength(nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)))"
     by (simp add: nsubn_nlength omega_d.prems(2))
     have o33: "\<And>i.   
                 (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f) = 
                  (upd (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) n 
                       (nsubn l (nnth l1 i) (nnth l1 (Suc i))) \<Turnstile> f) "
     using fvars.simps(7) omega_d.IH omega_d.prems(1) o33 by blast
     have o34: "\<And>i. 
                 (upd (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) n 
                       (nsubn l (nnth l1 i) (nnth l1 (Suc i)))) =
                 (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i))) "
      by (metis Suc_n_not_le_n enat_ord_simps(1) linorder_le_cases nfinite_ntaken nidx_less_eq 
          nsubn_upd ntaken_all o31 omega_d.prems(2))
    have o35: "(\<forall>i. 
                 (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)) "
      using o31 o33 o34 by auto
     show ?thesis
     using o31 o32 o35 by blast
    qed   
   qed
  have o4: "(\<exists>l1. \<not>nfinite l1 \<and> 
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength (upd \<sigma> n l)) \<and>
         (\<forall>i. (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f))) \<Longrightarrow>
       (\<exists>l1. \<not> nfinite l1 \<and>
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength \<sigma>) \<and>
         (\<forall>i. nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f))" 
   proof -
    assume a6: "(\<exists>l1. \<not>nfinite l1 \<and> 
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength (upd \<sigma> n l)) \<and>
         (\<forall>i. (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)))"
    show "(\<exists>l1. \<not> nfinite l1 \<and>
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength \<sigma>) \<and>
         (\<forall>i. nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f))"
    proof -
     obtain l1 where o41: "\<not>nfinite l1 \<and> 
         nnth l1 0 = 0 \<and>
         nidx l1 \<and>
         (\<forall>i. enat (nnth l1 i) \<le> nlength (upd \<sigma> n l)) \<and>
         (\<forall>i. (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f))"
      using a6 by blast  
    have o42: "(\<forall>i. enat (nnth l1 i) \<le> nlength \<sigma>) "
    using nlength_upd o41 omega_d.prems(2) by auto
    have o43: "\<And>i.  
               nlength (nsubn l (nnth l1 i) (nnth l1 (Suc i))) =
                 nlength(nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) "
                 by (simp add: nsubn_nlength omega_d.prems(2))
    have o44: "\<And>i.  
               (nsubn (upd \<sigma> n l) (nnth l1 i) (nnth l1 (Suc i))) = 
               upd (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) n 
                   (nsubn l (nnth l1 i) (nnth l1 (Suc i)))"
      by (metis enat_ord_simps(1) less_Suc_eq less_le_not_le linorder_not_less nfinite_ntaken 
          nidx_less_eq nsubn_upd ntaken_all o41 o42 omega_d.prems(2))
    have o45: "\<And>i. 
                  (upd (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i))) n 
                       (nsubn l (nnth l1 i) (nnth l1 (Suc i))) \<Turnstile> f) =
                  (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f)"
    using fvars.simps(7) o43 omega_d.IH omega_d.prems(1) by blast
    have o46: "(\<forall>i. 
              (nsubn \<sigma> (nnth l1 i) (nnth l1 (Suc i)) \<Turnstile> f))"
    using o41 o44 o45 by auto         
    show ?thesis
    using o41 o42 o46 by blast 
   qed
  qed
  show ?thesis
  using o1 o2 o3 o4 by blast
 qed
next
case (exists_d x1 f)
then show ?case 
  proof -
   have e1: "(\<sigma> \<Turnstile> Ex x1. f) = (\<exists> l. nlength \<sigma> = nlength l \<and> ((upd \<sigma> x1 l)  \<Turnstile> f) )" 
     by simp
   have e2: "(upd \<sigma> n l \<Turnstile> Ex x1. f) = 
             (\<exists> l1. nlength (upd \<sigma> n l) = nlength l1 \<and> ((upd (upd \<sigma> n l) x1 l1)  \<Turnstile> f) )" 
     by simp
   have e3: "(\<exists> l. nlength \<sigma> = nlength l \<and> ((upd \<sigma> x1 l)  \<Turnstile> f) ) \<Longrightarrow>
             (\<exists> l1. nlength (upd \<sigma> n l) = nlength l1 \<and> ((upd (upd \<sigma> n l) x1 l1)  \<Turnstile> f) )"
     by (metis Diff_iff empty_iff exists_d.IH exists_d.prems(1) exists_d.prems(2) 
             fvars.simps(8) insert_iff nlength_upd upd_absorb upd_swap)
   have e4: "(\<exists> l1. nlength (upd \<sigma> n l) = nlength l1 \<and> ((upd (upd \<sigma> n l) x1 l1)  \<Turnstile> f) ) \<Longrightarrow>
             (\<exists> l. nlength \<sigma> = nlength l \<and> ((upd \<sigma> x1 l)  \<Turnstile> f) )"
     by (metis Diff_iff empty_iff exists_d.IH exists_d.prems(1) exists_d.prems(2) fvars.simps(8)
          insert_iff nlength_upd upd_absorb upd_swap)
   show ?thesis
   using e1 e2 e3 e4 by blast      
   qed  
qed 

lemma inot_defs [simp]:
  " (\<sigma> \<Turnstile> inot f ) = Not ( \<sigma> \<Turnstile> f) " 
by (simp add: inot_d_def)

lemma ior_defs [simp]: 
  " (\<sigma> \<Turnstile> f1 ior f2) = ( (\<sigma> \<Turnstile> f1) \<or> (\<sigma> \<Turnstile> f2))"
by (metis inot_defs ior_d_def semantics_qpitl.simps(3))

lemma iand_defs [simp]:
  " (\<sigma> \<Turnstile> f1 iand f2) = ( (\<sigma> \<Turnstile> f1) \<and> (\<sigma> \<Turnstile> f2))"
by (simp add: iand_d_def)

lemma ieqv_defs [simp]: 
  " (\<sigma> \<Turnstile> f1 ieqv f2) = ( (\<sigma> \<Turnstile> f1) = (\<sigma> \<Turnstile> f2))"
by (metis iand_defs ieqv_d_def semantics_qpitl.simps(3))

lemma itrue_defs [simp]:
  " (\<sigma> \<Turnstile> itrue)"
by (simp add: itrue_d_def)

lemma empty_defs [simp]: 
  "(\<sigma> \<Turnstile> empty) = ( (nlength \<sigma>) = 0) "
by (auto simp add: empty_d_def more_d_def)
   (metis co.enat.exhaust_sel enat_le_plus_same(2) plus_1_eSuc(2))


lemma forall_defs [simp]: 
 " (\<sigma> \<Turnstile> Fa v. f) = (\<forall>l. nlength l = nlength \<sigma> \<longrightarrow> (upd \<sigma> v l \<Turnstile> f)) "
 using forall_d_def by auto

lemma sometimes_defs:
 " (\<sigma> \<Turnstile> diamond f) = (\<exists>i. i\<le> nlength \<sigma> \<and> ((ndropn i \<sigma>) \<Turnstile> f)) " 
by (auto simp add: sometimes_d_def) 

lemma always_defs:
 " (\<sigma> \<Turnstile> box f) = (\<forall>i. i \<le> nlength \<sigma> \<longrightarrow> ((ndropn i \<sigma> \<Turnstile> f))) " 
by (auto simp add: always_d_def sometimes_defs) 

lemma fin_defs:
 " (\<sigma> \<Turnstile> (fin f)) \<longleftrightarrow> nfinite \<sigma> \<and> ( (ndropn (the_enat(nlength \<sigma>)) \<sigma>) \<Turnstile> f) " 
apply (auto simp add: sometimes_defs fin_d_def)
apply (metis ndropn_nlength nfinite_ndropn nlength_eq_enat_nfiniteD zero_enat_def)
apply (metis enat_diff_cancel_left idiff_self not_enat_eq order_refl the_enat.simps)
by (metis enat_the_enat idiff_self infinity_ileE is_NNil_def is_NNil_ndropn ndropn_nlast)

lemma ifinite_defs:
 "(\<sigma> \<Turnstile> ifinite) \<longleftrightarrow> nfinite \<sigma> "  
apply (auto simp add: finite_d_def)
using nfinite_conv_nlength_enat apply fastforce
by (metis enat_ord_simps(3) enat_the_enat ndropn_nlast ndropn_nlength nlength_NNil
  order_refl)

lemma inf_defs:
 "(\<sigma> \<Turnstile> inf) \<longleftrightarrow> \<not> nfinite \<sigma> " 
by (auto simp add: inf_d_def ifinite_defs)  

lemma chop_defs:
 "(\<sigma> \<Turnstile> f;g) \<longleftrightarrow> (\<exists>i. i \<le> nlength \<sigma> \<and> ((ntaken i \<sigma>) \<Turnstile> f) \<and> ((ndropn i \<sigma>) \<Turnstile> g)) \<or> (\<not>nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f)) " 
by (auto simp add: chop_d_def inf_defs)

lemma di_defs:
 "(\<sigma> \<Turnstile> di f) \<longleftrightarrow> (\<exists>i. i\<le> nlength \<sigma> \<and> ((ntaken i \<sigma>) \<Turnstile> f)) \<or> (\<not> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) )  "
by (auto simp add: di_d_def chop_defs)  

lemma bi_defs: 
 " (\<sigma> \<Turnstile> bi f) \<longleftrightarrow> (\<forall>i. i\<le>nlength \<sigma> \<longrightarrow> ((ntaken i \<sigma>) \<Turnstile> f)) \<and> (\<not>nfinite \<sigma> \<longrightarrow> (\<sigma> \<Turnstile> f)) " 
by (auto simp add: di_defs bi_d_def)

lemma df_defs:
 "(\<sigma> \<Turnstile> df f) \<longleftrightarrow> (\<exists>i. i\<le> nlength \<sigma> \<and> ((ntaken i \<sigma>) \<Turnstile> f))    "
by (auto simp add: df_d_def)  

lemma bf_defs: 
 " (\<sigma> \<Turnstile> bf f) \<longleftrightarrow> (\<forall>i. i\<le>nlength \<sigma> \<longrightarrow> ((ntaken i \<sigma>) \<Turnstile> f)) " 
by (auto simp add: df_defs bf_d_def)

lemma da_defs : 
  "(\<sigma> \<Turnstile> da f) = 
   ( (\<exists> n k .  ( n \<le> k \<and> k \<le> nlength \<sigma> \<and> ( (nsubn \<sigma> n k) \<Turnstile> f))) \<or> 
   (\<not>nfinite \<sigma> \<and>(\<exists>n. ( (ndropn n \<sigma>) \<Turnstile> f)))
   )"
apply (auto simp add: da_d_def chop_defs  )
apply (metis enat_min_eq le_add2 ntaken_ndropn plus_enat_simps(1))
apply (metis enat_le_plus_same(2) enat_min_eq enat_ord_simps(1) ntaken_ndropn
  plus_enat_simps(1))
apply (metis enat_minus_mono1 enat_ord_simps(1) idiff_enat_enat nsubn_def1 order_subst2)
using is_NNil_imp_nfinite is_NNil_ndropn nfinite_ndropn nle_le by blast


lemma ba_defs:
 "(\<sigma> \<Turnstile> ba f) \<longleftrightarrow> ( (\<forall> n k.  ( n \<le>k \<and> k \<le> nlength \<sigma> \<longrightarrow> ( (nsubn \<sigma> n k) \<Turnstile> f)) \<and> (\<not>nfinite \<sigma> \<longrightarrow> ( (ndropn n \<sigma>) \<Turnstile> f)))
   ) " 
by (auto simp add: ba_d_def da_defs) 

lemma sda_defs : 
  "(\<sigma> \<Turnstile> sda f) = 
   ( (\<exists> n k .  ( n \<le> k \<and> k \<le> nlength \<sigma> \<and> ( (nsubn \<sigma> n k) \<Turnstile> f))) 
   )"
apply (auto simp add: sda_d_def chop_defs )
apply (metis add_diff_cancel_right' enat_min_eq le_add2 nsubn_def1 plus_enat_simps(1))
by (metis enat_minus_mono1 enat_ord_simps(1) idiff_enat_enat nsubn_def1 order.trans)

lemma sba_defs:
 "(\<sigma> \<Turnstile> sba f) \<longleftrightarrow> ( (\<forall> n k.  ( n \<le>k \<and> k \<le> nlength \<sigma> \<longrightarrow> ( (nsubn \<sigma> n k) \<Turnstile> f)) )
   ) " 
by (auto simp add: sba_d_def sda_defs) 



lemma state_qpitl_defs: 
assumes "state_qpitl w"
shows " (\<sigma> \<Turnstile> w) \<longleftrightarrow> ((NNil (nfirst \<sigma>)) \<Turnstile> w) "
using assms
proof (induct w arbitrary: \<sigma>)
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case 
by (metis nfirst_nfuse nfuse_leftneutral nlast_NNil semantics_qpitl.simps(2))
next
case (iimp_d w1 w2)
then show ?case by simp
next
case (next_d w)
then show ?case by auto
next
case (chop_d w1 w2)
then show ?case by auto
next
case (schopstar_d w)
then show ?case by auto
next
case (omega_d w)
then show ?case by auto
next
case (exists_d x1 w)
then show ?case 
    proof -
     have 1: "(\<sigma> \<Turnstile> Ex x1. w) \<longleftrightarrow> (\<exists>\<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar \<sigma> x1 \<sigma>'') "
     using exist_sigma_exist_value semantics_qpitl.simps(8) by presburger 
     have 2: "(NNil (nfirst \<sigma>) \<Turnstile> Ex x1. w) \<longleftrightarrow> (\<exists> \<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar (NNil (nfirst \<sigma>)) x1 \<sigma>'')"
     using exist_sigma_exist_value semantics_qpitl.simps(8) by presburger       
     have 3: "(\<exists>\<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar \<sigma> x1 \<sigma>'') \<Longrightarrow> 
              (\<exists> \<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar (NNil (nfirst \<sigma>)) x1 \<sigma>'') " 
       proof -
        assume a0: "(\<exists>\<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar \<sigma> x1 \<sigma>'')"
        show "(\<exists> \<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar (NNil (nfirst \<sigma>)) x1 \<sigma>'')"
        proof -
         obtain \<sigma>'' where 4: "(\<sigma>'' \<Turnstile> w) \<and> similar \<sigma> x1 \<sigma>''"
          using a0 by blast
         have 5: "state_qpitl w"  
          using exists_d.prems by auto
         have 6: "(NNil (nfirst \<sigma>'') \<Turnstile> w)"
          using "4" "5" exists_d.hyps by blast 
         have 7: "similar (NNil (nfirst \<sigma>)) x1 (NNil (nfirst \<sigma>''))"
           by (metis (no_types, lifting) "4" enat_defs(1) nfirst_eq_nnth_zero nlength_NNil nnth_NNil 
               similar_defs_alt zero_order(1))
         show ?thesis 
         using "6" "7" by blast
        qed
      qed
    have 8: "(\<exists> \<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar (NNil (nfirst \<sigma>)) x1 \<sigma>'') \<Longrightarrow>
             (\<exists>\<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar \<sigma> x1 \<sigma>'')" 
      proof -
       assume a1: "(\<exists> \<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar (NNil (nfirst \<sigma>)) x1 \<sigma>'')"
       show "(\<exists>\<sigma>''. (\<sigma>'' \<Turnstile> w) \<and> similar \<sigma> x1 \<sigma>'')"
       proof -
        obtain \<sigma>'' where 9: "(\<sigma>'' \<Turnstile> w) \<and> similar (NNil (nfirst \<sigma>)) x1 \<sigma>''"
         using a1 by blast
        have 10: "state_qpitl w"  
          using exists_d.prems by auto 
        have 11: "(NNil (nfirst \<sigma>'') \<Turnstile> w)"
         using 9 10  exists_d.hyps by blast 
        have 12: "nlength \<sigma> = 0 \<Longrightarrow> similar \<sigma> x1 \<sigma>''"   
         by (metis "9" ndropn_0 ndropn_nlast nlength_eq_enat_nfiniteD ntake_0 ntake_NNil 
          the_enat.simps zero_enat_def)
        have 13: "nlength \<sigma> > 0 \<Longrightarrow> similar \<sigma> x1 (nappend (NNil (nfirst \<sigma>)) (ndropn 1 \<sigma>))  "
          by (simp add: enat_defs(1) ndropn_Suc_conv_ndropn nfirst_eq_nnth_zero)  
        have 14: "nlength \<sigma> = 0 \<Longrightarrow> (\<sigma>'' \<Turnstile> w)"
           by (simp add: "9") 
        have 15: "nlength \<sigma> > 0 \<Longrightarrow> (NNil (nfirst (nappend (NNil (nfirst \<sigma>)) (ndropn 1 \<sigma>)))) = (NNil (nfirst \<sigma>)) " 
           by (simp add: nfirst_eq_nnth_zero)
        have 16: "nlength \<sigma> > 0 \<Longrightarrow> similar (NNil (nfirst \<sigma>)) x1 (NNil (nfirst \<sigma>''))"
           by (metis "9" le_numeral_extra(3) nlength_NNil ntaken_0 ntaken_all similar_defs_alt zero_enat_def) 
        have 17: "nlength \<sigma> > 0 \<Longrightarrow> \<sigma> = nappend (NNil (nfirst \<sigma>)) (ndropn 1 \<sigma>)" 
          by (simp add: ndropn_Suc_conv_ndropn nfirst_eq_nnth_zero zero_enat_def)
        have 18: "nlength \<sigma> > 0 \<Longrightarrow> similar (nappend (NNil (nfirst \<sigma>)) (ndropn 1 \<sigma>)) x1 (nappend (NNil (nfirst \<sigma>'')) (ndropn 1 \<sigma>))" 
         by simp
           (metis (no_types, lifting) "9" le_zero_eq nappend_NNil nfirst_eq_nnth_zero nlength_NNil nnth_0 
            nnth_nappend similar_defs the_enat.simps)
              have 19: "nlength \<sigma> > 0 \<Longrightarrow> (nappend (NNil (nfirst \<sigma>'')) (ndropn 1 \<sigma>)) \<Turnstile> w"
            by (metis "10" "9" exists_d.hyps nfirst_eq_nnth_zero nnth_nappend zero_enat_def zero_le)
        show ?thesis 
        by (metis "12" "17" "18" "19" "9" i0_less)   
     qed
   qed
   show ?thesis 
   using "1" "2" "3" "8" by blast   
 qed        
qed

lemma not_fvar_upd_state: 
 assumes " n \<notin> fvars w"
         " nlength l = nlength \<sigma>" 
         "state_qpitl w" 
 shows " (NNil (nnth \<sigma> 0) \<Turnstile> w) = (NNil (nnth (upd \<sigma> n l) 0) \<Turnstile> w) "
using assms by (metis  ndropn_0 ndropn_nfirst not_fvar_upd state_qpitl_defs)

lemma not_fvar_upd_nmap: 
assumes "n \<notin> fvars w" 
        "nlength l = nlength \<sigma> " 
        "state_qpitl w" 
shows "(nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> n l)) = (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)" 
proof -
 have 1: "nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> n l)) = nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)"
 by (simp add: assms(2) nlength_upd)
 have 2: "\<And> i. enat i \<le> nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> n l)) \<Longrightarrow>
          (nnth (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> n l)) i) =
          (nnth (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) i) " 
   using assms nnth_nmap[of _ "(upd \<sigma> n l)" "(\<lambda>c. NNil c \<Turnstile> w)"  ]
   by (metis (mono_tags, lifting) "1" ndropn_nfirst ndropn_nlength ndropn_upd nlength_nmap
        nnth_nmap not_fvar_upd state_qpitl_defs)
 show ?thesis
 using "1" "2" nellist_eq_nnth_eq by blast   
qed


lemma subst_suform: 
 assumes "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_qpitl w" 
 shows   " (\<sigma> \<Turnstile> suform f n w) = ((upd \<sigma> n (nmap (\<lambda> c. (NNil c) \<Turnstile> w) \<sigma>)) \<Turnstile> f)" 
using assms
proof (induction f arbitrary: \<sigma> w n)
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case 
  proof -
   have a1: "(\<sigma> \<Turnstile> (suform ($x) n w)) = (if x = n then (\<sigma> \<Turnstile> w) else (x \<in> nfirst \<sigma>))"
    by simp 
   have a2: " nlength \<sigma> = nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)" 
     by auto
   have a3: "(upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) \<Turnstile> $x) = (if x = n then (\<sigma> \<Turnstile> w) else (x \<in> nfirst \<sigma>))" 
    using a2 nfirst_upd by simp
     (metis atom_d.prems(3) ndropn_0 ndropn_nfirst nnth_nmap state_qpitl_defs zero_enat_def zero_le)
   show ?thesis 
   using a1 a3 by blast
  qed
next
case (iimp_d f1 f2)
then show ?case by auto
next
case (next_d f)
then show ?case by (auto simp add: nlength_upd ndropn_nmap ndropn_upd one_enat_def ) 
next
case (chop_d f1 f2)
then show ?case 
 proof -
  have c1: " (\<sigma> \<Turnstile> suform ( f1 schop f2 ) n w) = 
             (\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> suform f1 n w) \<and> (ndropn i \<sigma> \<Turnstile> suform f2 n w))"  
   by simp
  have c2: "(upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) \<Turnstile>  f1 schop f2 ) = 
            (\<exists>i. enat i \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
             (ntaken i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f1) \<and> 
             (ndropn i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f2))" 
    by simp
  have c3: "(\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> suform f1 n w) \<and> (ndropn i \<sigma> \<Turnstile> suform f2 n w))
            \<Longrightarrow> 
            (\<exists>i. enat i \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
             (ntaken i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f1) \<and> 
             (ndropn i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f2))"
   proof -
    assume c4: "(\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> suform f1 n w) \<and> (ndropn i \<sigma> \<Turnstile> suform f2 n w))"  
    show "(\<exists>i. enat i \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
             (ntaken i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f1) \<and> 
             (ndropn i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f2))" 
    proof -
     obtain i where c5: "enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> suform f1 n w) \<and> (ndropn i \<sigma> \<Turnstile> suform f2 n w)" 
     using c4 by blast
     have c6: "nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) = nlength \<sigma>" 
      by (simp add: nlength_upd)
     have c7: "n \<notin> bvars f1" 
       using chop_d.prems(1) by auto
     have c8: "(ntaken i \<sigma> \<Turnstile> suform f1 n w) = (ntaken i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f1)" 
       using chop_d.IH(1)[of n w "(ntaken i \<sigma>)" ] 
       using c5 c7 chop_d.prems(2) chop_d.prems(3) ntaken_upd by auto
     have c9: "n \<notin> bvars f2"
       using chop_d.prems(1) by auto 
     have c10: "(ndropn i \<sigma> \<Turnstile> suform f2 n w) = (ndropn i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f2)"
       using chop_d.IH(2)[of n w "(ndropn i \<sigma>)"] 
       by (metis Un_iff bvars.simps(5) c5 c9 chop_d.prems(2) chop_d.prems(3) ndropn_nmap ndropn_upd nlength_nmap)
     show ?thesis 
     using c10 c5 c6 c8 by auto
    qed
   qed
  have c11: "(\<exists>i. enat i \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
             (ntaken i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f1) \<and> 
             (ndropn i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f2))
              \<Longrightarrow>
             (\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> suform f1 n w) \<and> (ndropn i \<sigma> \<Turnstile> suform f2 n w))" 
  proof -
   assume c12: "(\<exists>i. enat i \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
             (ntaken i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f1) \<and> 
             (ndropn i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f2))"
   show "(\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> suform f1 n w) \<and> (ndropn i \<sigma> \<Turnstile> suform f2 n w))"  
   proof -
    obtain i where c13: "enat i \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
             (ntaken i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f1) \<and> 
             (ndropn i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f2)"
     using c12 by blast
    have c14: "nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) = nlength \<sigma> "  
     by (simp add: nlength_upd)
    have c15: "n \<notin> bvars f1" 
       using chop_d.prems(1) by auto
    have c16: "(ntaken i \<sigma> \<Turnstile> suform f1 n w) = (ntaken i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f1)" 
     using chop_d.IH(1)[of n w "(ntaken i \<sigma>)" ] 
     using c13 c14 c15 chop_d.prems(2) chop_d.prems(3) ntaken_upd by auto
    have c17: "n \<notin> bvars f2" 
       using chop_d.prems(1) by auto 
    have c18: "(ndropn i \<sigma> \<Turnstile> suform f2 n w) = (ndropn i (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<Turnstile> f2)"
       using chop_d.IH(2)[of n w "(ndropn i \<sigma>)"] 
       by (metis Un_iff bvars.simps(5) c13 c14 c17 chop_d.prems(2) chop_d.prems(3) ndropn_nmap ndropn_upd nlength_nmap)
    show ?thesis 
    using c13 c14 c16 c18 by auto 
   qed
  qed
  show ?thesis 
  using c1 c11 c2 c3 by blast
 qed
next
case (schopstar_d f)
then show ?case 
  proof -
   have s1: "(\<sigma> \<Turnstile> suform (schopstar f ) n w) =
             (\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength \<sigma> \<and> nfinite \<sigma> \<and> 
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)))" 
   by simp
   have s2: "(upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) \<Turnstile> schopstar f ) =
             (\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         nfinite (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)))" 
   by simp
   have s3: "(\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength \<sigma> \<and> nfinite \<sigma> \<and> 
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w))) \<Longrightarrow>
         (\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         nfinite (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)))"
   proof -
    assume sa0: "(\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength \<sigma> \<and> nfinite \<sigma> \<and> 
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)))"
    show "(\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         nfinite (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)))"
    proof -
     obtain l where s4: "nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength \<sigma> \<and> nfinite \<sigma> \<and> 
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w))"
     using sa0 by blast
     have s5: "enat (nlast l) = nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>))"
     by (simp add: nlength_upd s4)     
     have s6: "nfinite (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>))"
     by (metis nlength_eq_enat_nfiniteD s5) 
     have s7: "\<And>i.  enat i < nlength l \<Longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)"
     using s4 by blast
     have s8: "\<And>i.  enat i < nlength l \<Longrightarrow> 
                (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)" 
      proof -
       fix i 
       assume sa1: "enat i < nlength l" 
       show "(nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)" 
       proof -
        have s9: "(upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f)"
       using schopstar_d.IH[of n w "(nsubn \<sigma> (nnth l i) (nnth l (Suc i)))" ]
       using bvars.simps(6) s4 sa1 schopstar_d.prems(1) schopstar_d.prems(2) schopstar_d.prems(3) by blast 
       have s10: "(nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i))) = 
                  upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nsubn (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) (nnth l i) (nnth l (Suc i)))"
           using nsubn_upd[of \<sigma> "(nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)" "(nnth l i)" "(nnth l (Suc i))" n]
           by (metis Suc_n_not_le_n eSuc_enat enat_ord_simps(1) ileI1 nidx_all_le_nlast nidx_less_eq nle_le nlength_nmap s4 sa1)
       have s11: "(nsubn (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) (nnth l i) (nnth l (Suc i))) =
                   (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))))"
       by (simp add: ndropn_nmap nsubn_def1)       
       have s12: "(upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))))) =
                  (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)))"
       by (simp add: s10 s11)  
       show ?thesis
       using s12 s9 by auto 
      qed
     qed
     show ?thesis
     using s4 s5 s6 s8 by blast
     qed
    qed
    have s13: "(\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         nfinite (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f))) \<Longrightarrow>
         (\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength \<sigma> \<and> nfinite \<sigma> \<and> 
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)))"
     proof -
      assume sa2: "(\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         nfinite (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)))"
      show "(\<exists>l. nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength \<sigma> \<and> nfinite \<sigma> \<and> 
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)))"
      proof -
       obtain l where s14: "nnth l 0 = 0 \<and>
         nfinite l \<and>
         nidx l \<and>
         enat (nlast l) = nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         nfinite (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) \<and>
         (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f))"
       using sa2 by blast
       have s15: "enat (nlast l) = nlength \<sigma>"
       by (simp add: nlength_upd s14)
       have s16: "nfinite \<sigma>"
       by (metis nlength_eq_enat_nfiniteD s15)
       have s17: "\<And>i.  enat i < nlength l \<Longrightarrow>
                   (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)"
       using s14 by blast 
       have s18: "\<And>i.  enat i < nlength l \<Longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)"
       proof -
        fix i
        assume sa3: "enat i < nlength l"
        show "(nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)"
        proof -
         have s19: "(nsubn (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) (nnth l i) (nnth l (Suc i))) =
                   (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))))"
         by (simp add: ndropn_nmap nsubn_def1)
         have s20: "(nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i))) = 
                  upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nsubn (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) (nnth l i) (nnth l (Suc i)))"
         using nsubn_upd[of \<sigma> "(nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)" "(nnth l i)" "(nnth l (Suc i))" n]
         by (metis Suc_n_not_le_n eSuc_enat enat_ord_simps(1) ileI1 linorder_le_cases nidx_all_le_nlast 
             nidx_less_eq nlength_nmap s14 s15 sa3)
         have s21: "(upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))))) =
                  (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)))"
         using s19 s20 by presburger
         have s22: "(nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)"
         using s14 sa3 by blast  
         have s23: "(nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w) "
         using bvars.simps(6) s21 s22 schopstar_d.IH schopstar_d.prems(1) schopstar_d.prems(2)
          schopstar_d.prems(3) by presburger
         show ?thesis
         by (simp add: s23) 
        qed
       qed
      show ?thesis
      using s14 s15 s16 s18 by blast
     qed
    qed
   show ?thesis
   using s1 s13 s2 s3 by blast
  qed  
next
case (omega_d f)
then show ?case 
 proof -
   have s1: "(\<sigma> \<Turnstile> suform (omega f ) n w) = 
             (\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and>
              (\<forall>i. nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w))" 
    by simp
   have s2: "(upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) \<Turnstile> omega f ) = 
             (\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>))) \<and>
              (\<forall>i. nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f))" 
   by simp
   have s3: "(\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and>
              (\<forall>i. nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)) \<Longrightarrow>
             (\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>))) \<and>
              (\<forall>i. nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f))"
   proof -
    assume sa0: "(\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and>
              (\<forall>i. nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w))"
    show "(\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>))) \<and>
              (\<forall>i. nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f))"
    proof -
     obtain l where s4: "\<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and>
              (\<forall>i. nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)"
     using sa0 by blast
     have s5: "(\<forall>i. enat (nnth l i) \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)))"
     by (simp add: nlength_upd s4)   
     have s7: "\<And>i.  enat i < nlength l \<Longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)"
     using s4 by blast
     have s8: "\<And>i.  enat i < nlength l \<Longrightarrow> 
                (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)" 
      proof -
       fix i 
       assume sa1: "enat i < nlength l" 
       show "(nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)" 
       proof -
        have s9: "(upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f)"
       using omega_d.IH[of n w "(nsubn \<sigma> (nnth l i) (nnth l (Suc i)))" ]
       using omega_d.prems(1) omega_d.prems(2) omega_d.prems(3) s7 sa1 by force
       have s10: "(nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i))) = 
                  upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nsubn (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) (nnth l i) (nnth l (Suc i)))"
           using nsubn_upd[of \<sigma> "(nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)" "(nnth l i)" "(nnth l (Suc i))" n]
       by (metis Suc_n_not_le_n eSuc_enat enat_ord_simps(1) ileI1 nidx_less_eq nle_le nlength_nmap s4 sa1)
       have s11: "(nsubn (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) (nnth l i) (nnth l (Suc i))) =
                   (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))))"
       by (simp add: ndropn_nmap nsubn_def1)       
       have s12: "(upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))))) =
                  (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)))"
       by (simp add: s10 s11)  
       show ?thesis
       using s12 s9 by auto 
      qed
     qed
     show ?thesis
     by (metis enat_ile linorder_not_less nlength_eq_enat_nfiniteD s4 s5 s8)
     qed
    qed
    have s13: "(\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>))) \<and>
              (\<forall>i. nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)) \<Longrightarrow>
         (\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and>
              (\<forall>i. nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w))"
     proof -
      assume sa2: "(\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>))) \<and>
              (\<forall>i. nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f))"
      show "(\<exists>l. \<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and>
              (\<forall>i. nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w))"
      proof -
       obtain l where s14: "\<not> nfinite l \<and>
              nnth l 0 = 0 \<and>
              nidx l \<and>
              (\<forall>i. enat (nnth l i) \<le> nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>))) \<and>
              (\<forall>i. nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)"
       using sa2 by blast
       have s15: "(\<forall>i. enat (nnth l i) \<le> nlength \<sigma>)"
       using nlength_upd s14 by auto
       have s17: "\<And>i.  (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)"
       using s14 by blast 
       have s18: "\<And>i.   (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)"
       proof -
        fix i
        show "(nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w)"
        proof -
         have s19: "(nsubn (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) (nnth l i) (nnth l (Suc i))) =
                   (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))))"
         by (simp add: ndropn_nmap nsubn_def1)
         have s20: "(nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i))) = 
                  upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nsubn (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) (nnth l i) (nnth l (Suc i)))"
         using nsubn_upd[of \<sigma> "(nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)" "(nnth l i)" "(nnth l (Suc i))" n]
          unfolding nsubn_def1
          by (metis Suc_ile_eq enat_ord_simps(1) le_SucI linorder_not_le nfinite_ntaken nidx_less_eq nlength_nmap ntaken_all s14 s15)
         have s21: "(upd (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))))) =
                  (nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)))"
         using s19 s20 by presburger
         have s22: "(nsubn (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> f)"
         by (simp add: s17) 
         have s23: "(nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> suform f n w) "
         using bvars.simps(7) omega_d.IH omega_d.prems(1) omega_d.prems(2) omega_d.prems(3) s21 s22 by presburger
         show ?thesis
         by (simp add: s23) 
        qed
       qed
      show ?thesis
      using s14 s15 s18 by blast
     qed
    qed
   show ?thesis
   using s1 s13 s2 s3 by blast
  qed  
next
case (exists_d x1 f)
then show ?case 
 proof -
  have e1: "(\<sigma> \<Turnstile> suform (Ex x1. f) n w) = (\<sigma> \<Turnstile> Ex x1. suform f n w)"
  using suform.simps(8) by presburger  
  have e2: "n = x1 \<Longrightarrow> ?thesis"
  using bvars.simps(8) exists_d.prems(1) by blast 
  
  have e3: "n \<noteq> x1 \<Longrightarrow> ?thesis"
    proof -
     assume ae0: "n \<noteq> x1"
     show ?thesis
     proof -
      have e4: "n \<notin> bvars f"
      using exists_d.prems(1) by auto 
      have e5: "(upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) \<Turnstile> Ex x1. f) =
                (\<exists> l. nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) = nlength l \<and>
                      (upd (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) x1 l \<Turnstile> f)) "
      by simp
      have e6: "(\<And>z. z \<in> fvars w \<Longrightarrow> z \<notin> bvars f) "
      using exists_d.prems(2) by auto 
      have e60: " x1 \<notin> fvars w"
      using exists_d.prems(2) by auto
      have e61: "\<And> \<sigma> l. nlength l = nlength \<sigma> \<Longrightarrow> (\<sigma> \<Turnstile> w) = ((upd \<sigma> x1 l) \<Turnstile> w)  "
      using e60 not_fvar_upd by blast      
      have e7: "(\<sigma> \<Turnstile> Ex x1. suform f n w) = (\<exists>l. nlength \<sigma> = nlength l \<and> (upd \<sigma> x1 l \<Turnstile> suform f n w))"
      by simp
      have e8: "(\<exists>l. nlength \<sigma> = nlength l \<and> (upd \<sigma> x1 l \<Turnstile> suform f n w)) \<Longrightarrow>
                (\<exists> l. nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) = nlength l \<and>
                      (upd (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) x1 l \<Turnstile> f))" 
      proof -
       assume ae1: "(\<exists>l. nlength \<sigma> = nlength l \<and> (upd \<sigma> x1 l \<Turnstile> suform f n w))"
       show "(\<exists> l. nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) = nlength l \<and>
                      (upd (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) x1 l \<Turnstile> f))" 
       proof -
        obtain l where e9: " nlength \<sigma> = nlength l \<and> (upd \<sigma> x1 l \<Turnstile> suform f n w)"
          using ae1 by blast
        have e10: "(upd (upd \<sigma> x1 l) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) \<Turnstile> f)"
        using e4 e6 e9 exists_d.IH exists_d.prems(3) by auto  
        have e11: "(upd (upd \<sigma> x1 l) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l))) =
                   (upd (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l))) x1 l)"
                   by (simp add: ae0 e9 nlength_upd upd_swap) 
        have e12: " nlength \<sigma> = nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) "
        by (simp add: e9 nlength_upd) 
        have e120: "nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) = nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)"
        by (simp add: e12)  
        have e121: "\<And> i. i \<le> nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) \<Longrightarrow>
                     (nnth (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) i) =
                     (nnth (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) i) "
        by (simp add: e60 e9 exists_d.prems(3) not_fvar_upd_nmap) 
        have e122: "(nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) = (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) "
        using e120 e121 nellist_eq_nnth_eq by blast 
        show ?thesis
        using e10 e11 e122 e9 nlength_upd by auto
       qed
      qed
     have e13: "(\<exists> l. nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) = nlength l \<and>
                      (upd (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) x1 l \<Turnstile> f)) \<Longrightarrow>
                (\<exists>l. nlength \<sigma> = nlength l \<and> (upd \<sigma> x1 l \<Turnstile> suform f n w)) "
     proof -
      assume ae1: "(\<exists> l. nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) = nlength l \<and>
                      (upd (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) x1 l \<Turnstile> f))"
      show "(\<exists>l. nlength \<sigma> = nlength l \<and> (upd \<sigma> x1 l \<Turnstile> suform f n w))"
      proof -
       obtain l where e14: "nlength (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) = nlength l \<and>
                      (upd (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)) x1 l \<Turnstile> f)"
       using ae1 by blast
       have e15: "nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) = nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)"
       using e14 nlength_upd by auto   
       have e16: "\<And> i. i \<le> nlength (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) \<Longrightarrow>
                     (nnth (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) i) =
                     (nnth (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>) i)"
       using e14 e60 exists_d.prems(3) nlength_upd not_fvar_upd_nmap by force  
       have e17: "(nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) = (nmap (\<lambda>c. NNil c \<Turnstile> w) \<sigma>)"
       using e15 e16 nellist_eq_nnth_eq by blast
       have e18: "(upd (upd \<sigma> x1 l) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l))) =
                   (upd (upd \<sigma> n (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l))) x1 l)"
       using ae0 e14 nlength_upd upd_swap by auto 
       have e19: "(upd (upd \<sigma> x1 l) n (nmap (\<lambda>c. NNil c \<Turnstile> w) (upd \<sigma> x1 l)) \<Turnstile> f)"
       using e14 e17 e18 by presburger 
       have e20: "(upd \<sigma> x1 l \<Turnstile> suform f n w)"
       by (simp add: e19 e4 e6 exists_d.IH exists_d.prems(3)) 
       show ?thesis
       using e14 e20 nlength_upd by auto
      qed
     qed
    show ?thesis
    using e1 e13 e5 e7 e8 by blast    
   qed
  qed   
  show ?thesis
  using e2 e3 by blast
  qed            
qed

lemma rename_qpitl: 
 assumes "v1 \<notin> bvars f"
         "v2 \<notin> bvars f"
         "v1 \<noteq> v2"
         "v2 \<notin> fvars f" 
 shows " (\<sigma> \<Turnstile> Ex v1. f) = (\<sigma> \<Turnstile> Ex v2. suform f v1 ($v2)) "
proof -
  have 1: " (\<sigma> \<Turnstile> Ex v1. f) = (\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v1 l) \<Turnstile> f) )"
  by force
  have 2: " (\<sigma> \<Turnstile> Ex v2. suform f v1 ($v2)) = 
            (\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v2 l) \<Turnstile> suform f v1 ($v2))) "
  by force
  have 3: " (\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v2 l) \<Turnstile> suform f v1 ($v2))) \<Longrightarrow>
            (\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v1 l) \<Turnstile> f) )"
   proof -
    assume a0: "(\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v2 l) \<Turnstile> suform f v1 ($v2)))"
    show "(\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v1 l) \<Turnstile> f) )"
    proof -
     obtain l where 4: "nlength l = nlength \<sigma> \<and> ((upd \<sigma> v2 l) \<Turnstile> suform f v1 ($v2))"
     using a0 by blast
     have 5: "((upd \<sigma> v2 l) \<Turnstile> suform f v1 ($v2)) =
              (upd (upd \<sigma> v2 l) v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l)) \<Turnstile> f)"
      using  subst_suform[of v1 f "$v2" "(upd \<sigma> v2 l)"]
      by (simp add: assms(1) assms(2))     
     have 6: "(upd (upd \<sigma> v2 l) v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))) =
              (upd (upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))) v2 l)"
     by (simp add: "4" assms(3) nlength_upd upd_swap)    
     have 8: "((upd (upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))) v2 l) \<Turnstile> f) =
               ((upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))) \<Turnstile> f)"
     by (metis "4" assms(4) nlength_nmap nlength_upd not_fvar_upd)   
     show ?thesis
     by (metis "4" "5" "6" "8" nlength_nmap nlength_upd)
    qed
   qed
   have 9: "(\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v1 l) \<Turnstile> f) ) \<Longrightarrow>
            (\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v2 l) \<Turnstile> suform f v1 ($v2)))"
    proof -
     assume a1: "(\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v1 l) \<Turnstile> f) )"
     show "(\<exists> l. nlength l = nlength \<sigma> \<and> ((upd \<sigma> v2 l) \<Turnstile> suform f v1 ($v2)))"
     proof -
      obtain l where 10: "nlength l = nlength \<sigma> \<and> ((upd \<sigma> v1 l) \<Turnstile> f)"
      using a1 by blast
      have 11: "nlength l = nlength (upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l)))"
      by (simp add: "10" nlength_upd) 
      have 12: "v2 \<notin> fvars f"
      by (simp add: assms(4)) 
      have 13: "(upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l)) \<Turnstile> f) =
                ((upd (upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))) v2 l) \<Turnstile> f)"
      using not_fvar_upd[of v2 f l "(upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l)))" ]
      "11" "12" by blast 
      have 14: "nlength (upd \<sigma> v1 l) = 
                nlength (upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l)) )"
      by (simp add: "10" nlength_upd)  
      have 15: "\<And> i . i \<le> nlength (upd \<sigma> v1 l) \<Longrightarrow>
                 (nnth (upd \<sigma> v1 l) i) = 
                 (nnth (upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l)) ) i)"
        using nnth_upd[of \<sigma> "(nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))" _ v1]by auto 
         (metis "10" Diff_iff insertCI nfirst_eq_nnth_zero nlength_upd nnth_NNil nnth_upd)+
      have 16: "(upd \<sigma> v1 l) = 
                 (upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l)) )"
      by (simp add: "14" "15" nellist_eq_nnth_eq) 
      have 17: "((upd \<sigma> v2 l) \<Turnstile> suform f v1 ($v2)) =
              (upd (upd \<sigma> v2 l) v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l)) \<Turnstile> f)"
      using assms(1) assms(2) subst_suform by fastforce  
      have 18: "(upd (upd \<sigma> v2 l) v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))) =
              (upd (upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))) v2 l)"
      by (simp add: "10" assms(3) nlength_upd upd_swap) 
      have 19: "((upd (upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))) v2 l) \<Turnstile> f) =
               ((upd \<sigma> v1 (nmap (\<lambda>c. NNil c \<Turnstile> $v2) (upd \<sigma> v2 l))) \<Turnstile> f)"
      using "13" by auto 
      show ?thesis
      using "10" "13" "16" "17" "18" by auto
     qed
    qed
  show ?thesis
  using "1" "2" "3" "9" by blast 
qed


lemma state_qpitl_list_defs: 
 "state_qpitl_list L \<longleftrightarrow> (\<forall>i<length L. state_qpitl (L!i)) " 
proof (induct L)
case Nil
then show ?case unfolding state_qpitl_list_def by simp
next
case (Cons a L)
then show ?case 
  proof -
   have 1: "state_qpitl_list (a # L) \<longleftrightarrow> state_qpitl a \<and> state_qpitl_list L"
      unfolding  state_qpitl_list_def by simp
   have 2: "(\<forall>i<length (a#L). state_qpitl ((a#L)!i)) \<longleftrightarrow>
            state_qpitl ((a#L)!0) \<and>
            (\<forall>i. 0<i \<and> i<length (a#L)\<longrightarrow> state_qpitl ((a#L)!i))" 
        by (metis length_greater_0_conv list.discI zero_less_iff_neq_zero)
   have 3: " state_qpitl ((a#L)!0) \<longleftrightarrow> state_qpitl a " 
      by simp
   have 4: "(\<forall>i. 0<i \<and> i<length (a#L)\<longrightarrow> state_qpitl ((a#L)!i)) \<longleftrightarrow>
            (\<forall>i<length (L). state_qpitl (L!i)) " 
     by auto
   show ?thesis 
   using "1" "2" "3" "4" local.Cons by presburger
 qed
qed



lemma big_ior_defs : 
 " (\<sigma> \<Turnstile> (big_ior F)) \<longleftrightarrow> (\<exists> i< length F. (\<sigma> \<Turnstile> F ! i))"
proof (induct F)
case Nil
then show ?case unfolding big_ior_def by simp
next
case (Cons a F)
then show ?case 
   proof -
    have 1: "(\<sigma> \<Turnstile> (big_ior (a#F))) \<longleftrightarrow> (\<sigma> \<Turnstile> a ior (big_ior F))"
        unfolding big_ior_def by simp
    have 2: "(\<exists>i<length (a # F). \<sigma> \<Turnstile> (a # F) ! i) \<longleftrightarrow>
               (\<sigma> \<Turnstile> (a # F) ! 0) \<or>
               (\<exists>i. 0 < i \<and> i<length (a # F) \<and> (\<sigma> \<Turnstile> (a # F) ! i))" 
      by (metis length_greater_0_conv neq_Nil_conv zero_less_iff_neq_zero)  
    have 3: "(\<sigma> \<Turnstile> (a # F) ! 0) = (\<sigma> \<Turnstile> a)"
       by auto
    have 4: "(\<exists>i. 0 < i \<and> i<length (a # F) \<and> (\<sigma> \<Turnstile> (a # F) ! i)) \<longleftrightarrow>
             (\<exists>i<length ( F). (\<sigma> \<Turnstile> ( F) ! i))"  
       by auto
         (metis Suc_le_eq diff_Suc_Suc diff_zero gr0_conv_Suc less_Suc_eq_le)
     show ?thesis 
     using "1" "2" "4" local.Cons by force
   qed
qed


lemma big_iand_defs : 
 " (\<sigma> \<Turnstile> (big_iand F)) \<longleftrightarrow> (\<forall> i< length F. (\<sigma> \<Turnstile> F ! i))"
proof (induct F)
case Nil
then show ?case unfolding big_iand_def by simp
next
case (Cons a F)
then show ?case 
   proof -
    have 1: "(\<sigma> \<Turnstile> (big_iand (a#F))) \<longleftrightarrow> (\<sigma> \<Turnstile> a iand (big_iand F))"
        unfolding big_iand_def by simp
    have 2: "(\<forall>i<length (a # F). \<sigma> \<Turnstile> (a # F) ! i) \<longleftrightarrow>
               (\<sigma> \<Turnstile> (a # F) ! 0) \<and>
               (\<forall>i. 0 < i \<and> i<length (a # F) \<longrightarrow> (\<sigma> \<Turnstile> (a # F) ! i))" 
      by (metis length_greater_0_conv neq_Nil_conv zero_less_iff_neq_zero)  
    have 3: "(\<sigma> \<Turnstile> (a # F) ! 0) = (\<sigma> \<Turnstile> a)"
       by auto
    have 4: "(\<forall>i. 0 < i \<and> i<length (a # F) \<longrightarrow> (\<sigma> \<Turnstile> (a # F) ! i)) \<longleftrightarrow>
             (\<forall>i<length ( F). (\<sigma> \<Turnstile> ( F) ! i))"  
       by auto
     show ?thesis 
     using "1" "2" "4" local.Cons by force
   qed
qed


lemma big_ior_map_iand_zip : 
assumes "length F =length G" 
shows "(\<sigma> \<Turnstile>  big_ior (map2 iand_d  F G)) \<longleftrightarrow>
       (\<exists>i < length F. (\<sigma> \<Turnstile> F ! i iand G ! i)) "
using assms
proof (induct F arbitrary: G)
case Nil
then show ?case by (simp add: big_ior_defs)
next
case (Cons a F)
then show ?case 
   proof (cases "G=[]")
   case True
   then show ?thesis using Cons.prems by auto
   next
   case False
   then show ?thesis using Cons by (auto simp add: big_ior_defs)
   qed
qed

lemma big_iand_map_iimp_zip : 
assumes "length F =length G" 
shows "(\<sigma> \<Turnstile>  big_iand (map2 iimp_d F G)) \<longleftrightarrow>
       (\<forall>i < length F. (\<sigma> \<Turnstile> F ! i iimp G ! i)) "
using assms
proof (induct F arbitrary: G)
case Nil
then show ?case by (simp add: big_iand_defs)
next
case (Cons a F)
then show ?case 
   proof (cases "G=[]")
   case True
   then show ?thesis using Cons.prems by auto
   next
   case False
   then show ?thesis using Cons by (auto simp add: big_iand_defs)
   qed
qed


lemma map2_iand_defs : 
 assumes "length F = length G"
 shows "(\<exists>i<length G. \<sigma> \<Turnstile> map2 iand_d F G ! i) \<longleftrightarrow>
        (\<exists>i<length G. (\<sigma> \<Turnstile>  F !i) \<and> (\<sigma> \<Turnstile> G ! i)) "
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
  proof (cases "F=[]")
  case True
  then show ?thesis using Cons.prems by auto
  next
  case False
  then show ?thesis using Cons by auto 
  qed
qed 

lemma map2_iand_inot_defs : 
 assumes "length F = length G"
 shows "(\<exists>i<length G. \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand inot y) F G ! i) \<longleftrightarrow>
        (\<exists>i<length G. (\<sigma> \<Turnstile> F! i) \<and> \<not>(\<sigma> \<Turnstile>  G!i))" 
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
  proof (cases "F=[]")
  case True
  then show ?thesis using Cons.prems by auto
  next
  case False
  then show ?thesis using Cons by auto 
  qed
qed 

subsection \<open>Validity\<close>

definition valid :: " qpitl \<Rightarrow> bool" ("(\<turnstile> _)" 5)
where "(\<turnstile> f) = (\<forall> \<sigma>. (\<sigma> \<Turnstile> f))"

lemma itl_valid [simp] :
  " (\<turnstile> f) = (\<forall> \<sigma>. (\<sigma> \<Turnstile> f))"
by (simp add: valid_def)

lemma itl_ieq:
  " (\<turnstile> f ieqv g) = (\<forall> \<sigma>. (\<sigma> \<Turnstile> f) = (\<sigma> \<Turnstile> g)) "
by (auto simp add: ieqv_d_def iand_d_def ior_d_def inot_d_def) 

lemma big_ior_iimp_subset_eq: 
 assumes "set L1 \<subseteq> set L2"
 shows " \<turnstile> big_ior L1 iimp big_ior L2" 
using assms 
proof (induct  rule: list_induct2')
case 1
then show ?case by simp
next
case (2 x xs)
then show ?case by simp
next
case (3 y ys)
then show ?case by (simp add: big_ior_defs)
next
case (4 x xs y ys)
then show ?case 
  proof (cases "x=y")
  case True
  then show ?thesis using 4 by (auto simp add: big_ior_defs)
  (metis diff_Suc_1 in_set_conv_nth length_Cons less_Suc_eq_0_disj list.simps(15) nth_Cons' subsetD)
  next
  case False
  then show ?thesis using 4 by (auto simp add: big_ior_defs)
    (metis "4.prems" in_set_conv_nth length_Cons subsetD)
  qed
qed

lemma big_ior_ieqv_set_eq: 
 assumes "set L1 = set L2"
 shows " \<turnstile> big_ior L1 ieqv big_ior L2" 
using assms 
by (metis big_ior_iimp_subset_eq dual_order.refl iand_defs ieqv_d_def valid_def)

lemma big_ior_ieqv_remdups: 
 shows " \<turnstile> big_ior L ieqv big_ior (remdups L)"
by (metis big_ior_ieqv_set_eq set_remdups)
 

lemma big_iand_iimp_subset_eq: 
 assumes "set L1 \<subseteq> set L2"
 shows " \<turnstile> big_iand L2 iimp big_iand L1" 
using assms 
proof (induct  rule: list_induct2')
case 1
then show ?case by simp
next
case (2 x xs)
then show ?case by (simp add: big_iand_defs)
next
case (3 y ys)
then show ?case by simp
next
case (4 x xs y ys)
then show ?case 
  proof (cases "x=y")
  case True
  then show ?thesis using 4 by (auto simp add: big_iand_defs)
  (metis (mono_tags, lifting) "4.prems" all_nth_imp_all_set length_Cons list_ball_nth subsetD)
  next
  case False
  then show ?thesis using 4 by (auto simp add: big_iand_defs) 
   (metis (no_types, opaque_lifting) "4.prems" all_nth_imp_all_set insert_absorb insert_subset length_Cons nth_mem)
  qed
qed

lemma big_iand_ieqv_set_eq: 
 assumes "set L1 = set L2"
 shows " \<turnstile> big_iand L1 ieqv big_iand L2" 
using assms 
by (metis big_iand_iimp_subset_eq dual_order.refl iand_defs ieqv_d_def valid_def)

lemma big_iand_ieqv_remdups: 
 " \<turnstile> big_iand L ieqv big_iand (remdups L)"
 by (metis big_iand_ieqv_set_eq set_remdups)



lemma big_ior_empty:
 "(\<turnstile> big_ior []) \<longleftrightarrow> (\<turnstile> ifalse) " 
by (auto simp add: big_ior_defs)

lemma big_ior_Cons: 
 "(\<turnstile> big_ior (f # L)) \<longleftrightarrow> (\<turnstile> f ior (big_ior L)) "
proof -
 have 1: "(\<turnstile> big_ior (f # L)) \<Longrightarrow> (\<turnstile> f ior (big_ior L))" 
  by (auto simp add: big_ior_defs)
    (metis diff_Suc_1 gr0_conv_Suc less_Suc_eq less_imp_diff_less nth_Cons')
  have 2: "(\<turnstile> f ior (big_ior L)) \<Longrightarrow> (\<turnstile> big_ior (f # L))"
    by  (auto simp add: big_ior_defs) fastforce 
 show ?thesis using 1 2 by auto
qed

lemma big_ior_Cons_alt: 
 "\<turnstile> big_ior (f # L) ieqv f ior big_ior  L"
unfolding big_ior_def
by auto



lemma big_ior_1:
 assumes "\<And>i. i< length F \<Longrightarrow>  (\<turnstile> F ! i ieqv G ! i)"
         "length F = length G"
 shows " \<turnstile> (big_ior F) ieqv (big_ior G) " 
using assms 
by (auto simp add: big_ior_defs)


lemma big_iand_1:
 assumes "\<And>i. i< length F \<Longrightarrow>  (\<turnstile> F ! i ieqv G ! i)"
         "length F = length G"
 shows " \<turnstile> (big_iand F) ieqv (big_iand G) " 
using assms 
by (auto simp add: big_iand_defs)


lemma big_ior_big_iand_1: 
 " \<turnstile> (inot ( big_ior F)) ieqv (big_iand (map (\<lambda>x. inot x) F)) " 
by (auto simp add: big_ior_defs big_iand_defs)

lemma big_ior_big_iand_2: 
 " \<turnstile> (inot ( big_iand F)) ieqv (big_ior (map (\<lambda>x. inot x) F)) " 
by (auto simp add: big_ior_defs big_iand_defs)


lemma big_ior_append: 
 " (\<turnstile> (big_ior (L1@L2)) ieqv (big_ior L1) ior (big_ior L2)) " 
proof (induct L1 arbitrary: L2)
case Nil
then show ?case by (simp add: big_ior_defs)
next
case (Cons a L1a)
then show ?case 
   proof -
    have 1: "\<turnstile> big_ior ((a # L1a) @ L2) ieqv a ior ((big_ior (L1a @ L2)))" 
     using big_ior_Cons_alt by auto   
    have 2: "\<turnstile> big_ior (L1a @ L2) ieqv big_ior L1a ior big_ior L2 "
       using local.Cons by blast 
    have 3: "\<turnstile> a ior ((big_ior (L1a @ L2))) ieqv a ior (big_ior L1a ior big_ior L2)"
      using 1 2 by auto 
    have 4: "\<turnstile> big_ior (a # L1a) ieqv a ior (big_ior L1a)"
     using big_ior_Cons_alt by blast
    have 5: "\<turnstile>  a ior (big_ior L1a ior big_ior L2) ieqv  big_ior (a # L1a) ior big_ior L2  " 
          using 4 by auto
    show ?thesis using "1" "3" "5" by force
  qed
qed

lemma state_qpitl_list_append:  
 shows   "state_qpitl_list (L@L1) \<longleftrightarrow> state_qpitl_list L \<and> state_qpitl_list L1 " 
proof (induct L arbitrary: L1)
case Nil
then show ?case 
by (simp add: state_qpitl_list_defs)
next
case (Cons a L)
then show ?case 
   proof -
    have 1: "state_qpitl_list ((a # L) @ L1) \<longleftrightarrow> state_qpitl a \<and> state_qpitl_list (L @ L1)" 
      unfolding state_qpitl_list_def by auto
    have 2: "state_qpitl_list (L @ L1) \<longleftrightarrow> state_qpitl_list L \<and> state_qpitl_list L1"
      by (simp add: local.Cons) 
    have 3: "state_qpitl a \<and> state_qpitl_list L \<and> state_qpitl_list L1 \<longleftrightarrow>
             (state_qpitl_list (a # L) \<and> state_qpitl_list L1)" 
      by (simp add: state_qpitl_list_def)
    show ?thesis using "1" "3" local.Cons by auto
   qed
qed


lemma map2_map:
 assumes "length L1 = length L2"
 shows   " map2 f (map g L1) (map h L2) = map2 (\<lambda>x y. f (g x) (h y)) L1 L2"
using assms 
proof (induct L1 L2 rule:list_induct2)
case Nil
then show ?case by simp
next
case (Cons x xs y ys)
then show ?case by auto
qed
 

lemma map2_map_a: 
assumes "length L1 = length L2"
shows "(map2 (\<lambda>x y. x iand next y) (map (\<lambda>x1. x iand x1) L1) (map (\<lambda>y1. y ior y1) L2)) = 
       (map2 (\<lambda>x1 y1. (x iand x1) iand next ( y ior y1)) L1 L2)"
using assms map2_map by blast

lemma map2_map_a_a: 
assumes "length L1 = length L2"
shows "(map2 (\<lambda>x y. x iand next y) (map (\<lambda>x1. x iand x1) L1) (map (\<lambda>y1. (if y = ifalse then y1 else y ior y1)) L2)) = 
       (map2 (\<lambda>x1 y1. (x iand x1) iand next ( (if y = ifalse then y1 else y ior y1))) L1 L2)"
using assms map2_map by blast


lemma map2_map_b: 
assumes "length L1 = length L2"
shows "(map2 (\<lambda>x y. x iand next y) (map (\<lambda>x1. (inot x) iand x1) L1) L2) = 
       (map2 (\<lambda>x1 y1. ((inot x) iand x1) iand next  y1) L1 L2)"
using assms map2_map[of L1 L2 "(\<lambda>x y. x iand next y)" " (\<lambda>x1. (inot x) iand x1)" id] 
by auto


lemma big_ior_big_ior_map2: 
assumes "length L1 = length L2" 
        "\<turnstile> big_ior L1" 
shows   "\<turnstile> big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 )" 
using assms 
proof (induct L1 L2 rule:list_induct2)
case Nil
then show ?case by simp
next
case (Cons x xs y ys)
then show ?case 
by (metis fst_def list.size(4) map_eq_conv map_fst_zip)   
qed


lemma big_ior_split_off: 
assumes "length L1 = length L2" 
       "(\<turnstile> big_ior L1)" 
shows "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next (y ior  y1))) L1 L2 )) ior
         (big_ior  (map2 (\<lambda>x1 y1. (inot x iand  x1) iand next  y1)      L1 L2 )) ieqv
         (x iand next y) ior 
         (big_ior  (map2 (\<lambda>x y.  x iand next  y)  L1 L2 )) " 
proof -
 have 1: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next (y ior y1))) L1 L2 )) ieqv
            (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next (y ior y1))) L1 L2 )))" 
        by (auto simp add: big_ior_defs)
 have 2: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next (y ior y1))) L1 L2 )) ieqv
            ((next y iand (big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 ))) 
            ior 
            (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))"
       by (auto simp add: big_ior_defs) 
 have 3: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next (y ior y1))) L1 L2 )) ieqv 
            (x iand  (next y iand big_ior  (map2 (\<lambda>x1 y1.  x1) L1 L2 ) 
             ior 
            (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))))" 
       by (auto simp add: big_ior_defs) 
  have 4: "\<turnstile> big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 )"
         using assms big_ior_big_ior_map2 by blast
  have 5: "\<turnstile> (x iand  (next y iand big_ior  (map2 (\<lambda>x1 y1.  x1 )  L1 L2 ) 
             ior 
             (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))) ieqv
            (x iand next y) ior (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))" 
       using 4 by auto
  have 6: "\<turnstile> (big_ior (map2 (\<lambda>x1 y1. (inot x iand  x1) iand next  y1) L1 L2 )) ieqv
             (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next ( y1))) L1 L2 )))"
     by (auto simp add: big_ior_defs) 
  have 7: "\<turnstile> (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ior 
              (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ieqv
              (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))" 
       by (auto simp add: big_ior_defs) 
  have 8 : "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next ( y ior y1))) L1 L2 )) ior
              (big_ior (map2 (\<lambda>x1 y1. (inot x iand x1) iand next  y1) L1 L2 )) ieqv
              (x iand next y) ior 
              (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ior
              (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) "
          using 5 6 3 by auto
   show ?thesis using 7 8 by auto
qed

lemma big_ior_split_off_a: 
assumes "length L1 = length L2" 
       "(\<turnstile> big_ior L1)" 
shows "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next ((if y = ifalse then y1 else y ior y1)))) L1 L2 )) ior
         (big_ior  (map2 (\<lambda>x1 y1. (inot x iand  x1) iand next  y1)      L1 L2 )) ieqv
         (x iand next y) ior 
         (big_ior  (map2 (\<lambda>x y.  x iand next  y)  L1 L2 )) " 
proof -
 have 1: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next ((if y = ifalse then y1 else y ior y1)))) L1 L2 )) ieqv
            (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next ((if y = ifalse then y1 else y ior y1)))) L1 L2 )))" 
        by (auto simp add: big_ior_defs)
 have 2: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next ((if y = ifalse then y1 else y ior y1)))) L1 L2 )) ieqv
            ((next y iand (big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 ))) 
            ior 
            (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))"
    proof (cases "y=ifalse")
    case True
    then show ?thesis by simp
    next
    case False
    then show ?thesis by (auto simp add: big_ior_defs) 
    qed
 have 3: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next (if y = ifalse then y1 else y ior y1))) L1 L2 )) ieqv 
            (x iand  (next y iand big_ior  (map2 (\<lambda>x1 y1.  x1) L1 L2 ) 
             ior 
            (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))))" 
     proof (cases "y=ifalse")
     case True
     then show ?thesis by (auto simp add: big_ior_defs) 
     next
     case False
     then show ?thesis by (auto simp add: big_ior_defs) 
     qed
  have 4: "\<turnstile> big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 )"
         using assms big_ior_big_ior_map2 by blast
  have 5: "\<turnstile> (x iand  (next y iand big_ior  (map2 (\<lambda>x1 y1.  x1 )  L1 L2 ) 
             ior 
             (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))) ieqv
            (x iand next y) ior (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))" 
       using 4 by auto
  have 6: "\<turnstile> (big_ior (map2 (\<lambda>x1 y1. (inot x iand  x1) iand next  y1) L1 L2 )) ieqv
             (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next ( y1))) L1 L2 )))"
     by (auto simp add: big_ior_defs) 
  have 7: "\<turnstile> (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ior 
              (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ieqv
              (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))" 
       by (auto simp add: big_ior_defs) 
  have 8 : "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next ( if y = ifalse then y1 else y ior y1))) L1 L2 )) ior
              (big_ior (map2 (\<lambda>x1 y1. (inot x iand x1) iand next  y1) L1 L2 )) ieqv
              (x iand next y) ior 
              (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ior
              (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) "
          using 5 6 3 by auto
   show ?thesis using 7 8 by auto
qed


lemma state_qpitl_list_map:
 assumes "state_qpitl f"
         "state_qpitl_list L"
 shows " state_qpitl_list (map (\<lambda> x. f iand x) L) " 
using assms 
proof (induct L)
case Nil
then show ?case by simp
next
case (Cons a L)
then show ?case 
  proof -
   have 1: "state_qpitl_list (map (\<lambda>x. f iand x ) (a # L))  \<longleftrightarrow>
            state_qpitl ( f iand a ) \<and>  state_qpitl_list ((map (\<lambda>x. f iand x) L)) " 
         unfolding state_qpitl_list_def by auto
   have 2: "state_qpitl_list (a # L) \<longleftrightarrow> state_qpitl a \<and> state_qpitl_list L"
      unfolding state_qpitl_list_def by auto
   have 3: " state_qpitl ( f iand a)" 
      unfolding iand_d_def inot_d_def ior_d_def 
      using "2" Cons.prems(2) assms(1) by auto
   show ?thesis
   using "1" "2" "3" Cons.hyps Cons.prems(2) assms(1) by blast
 qed          
qed

lemma state_qpitl_list_map_alt:
 assumes "state_qpitl f"
         "state_qpitl_list L"
 shows " state_qpitl_list (map (\<lambda> x. x iand f) L) " 
using assms 
proof (induct L)
case Nil
then show ?case by simp
next
case (Cons a L)
then show ?case 
  proof -
   have 1: "state_qpitl_list (map (\<lambda>x. x iand f ) (a # L))  \<longleftrightarrow>
            state_qpitl ( a iand f ) \<and>  state_qpitl_list ((map (\<lambda>x. x iand f) L)) " 
         unfolding state_qpitl_list_def by auto
   have 2: "state_qpitl_list (a # L) \<longleftrightarrow> state_qpitl a \<and> state_qpitl_list L"
      unfolding state_qpitl_list_def by auto
   have 3: " state_qpitl ( a iand f)" 
      unfolding iand_d_def inot_d_def ior_d_def 
      using "2" Cons.prems(2) assms(1) by auto
   show ?thesis
   using "1" "2" "3" Cons.hyps Cons.prems(2) assms(1) by blast
 qed          
qed

lemma map_fst_filter_sat: 
assumes "length L1 = length L2" 
shows " map fst (filter (\<lambda> (x,y). (\<exists> \<sigma>. \<sigma> \<Turnstile> x))  (zip L1 L2)) = 
        filter (\<lambda>x. (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) L1"
using assms 
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case by simp
next
case (Cons x xs y ys)
then show ?case 
  proof (cases "xs = [] \<and> ys = [] ")
  case True
  then show ?thesis by simp
  next
  case False
  then show ?thesis 
     using Cons by auto
  qed
qed


lemma big_ior_ieqv_filter_sat: 
 "\<turnstile> big_ior L ieqv big_ior ( filter (\<lambda>x. (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) L)"
proof (induct L)
case Nil
then show ?case by simp
next
case (Cons a L)
then show ?case 
using big_ior_Cons_alt by auto
qed


lemma total_filter_sat: 
 assumes "\<turnstile> big_ior L"
 shows "\<turnstile> big_ior ( filter (\<lambda>x. (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) L)"
using assms 
using big_ior_ieqv_filter_sat by auto


lemma  big_ior_filter_sat_eqv: 
assumes "length L1 = length L2"      
shows "\<turnstile> big_ior ( map (\<lambda> (x,y) . x iand (next  y)) 
                  (filter (\<lambda> (x,y). (\<exists> \<sigma>. \<sigma> \<Turnstile> x))  (zip L1 L2))) 
         ieqv 
         big_ior (map2 (\<lambda> x y. x iand (next y)) L1 L2) "
using assms
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case by (simp add: big_ior_def)
next
case (Cons x xs y ys)
then show ?case 
  proof (cases "xs = [] \<and> ys = [] ")
  case True
  then show ?thesis using Cons 
    unfolding big_ior_def big_iand_def
      by simp
  next
  case False
  then show ?thesis 
    proof -
     have 1: "length xs = length ys"
           by (simp add: Cons.hyps(1))
     have 2: "\<turnstile> big_ior (map (\<lambda>(x, y). x iand next y)
                  (filter (\<lambda>(x, y).   \<exists>\<sigma>. \<sigma> \<Turnstile> x)
                   (zip xs ys))) 
                  ieqv big_ior (map2 (\<lambda>x y. x iand next y) xs ys) " 
         using  Cons.hyps(2) False by fastforce
      show ?thesis 
      using "2" big_ior_Cons_alt by auto  
    qed    
  qed
qed


lemma big_ior_rev_ieqv: 
 "\<turnstile> big_ior L ieqv big_ior (rev L) " 
by (auto simp add: big_ior_defs)
   (metis in_set_conv_nth length_rev set_rev,
    metis diff_less length_greater_0_conv less_nat_zero_code list.size(3) rev_nth zero_less_Suc)

lemma big_ior_map2_append: 
assumes "length L1= length N1"
        "length L2= length N2" 
shows " \<turnstile> (big_ior (map2 (\<lambda> x y . x iand (next y)) (L1@L2) (N1@N2))) ieqv 
          (big_ior (map2 (\<lambda> x y . x iand (next y)) L1 N1)) ior 
          (big_ior (map2 (\<lambda> x y . x iand (next y))  L2 N2)) " 
using assms
proof (induct L1 N1 arbitrary: L2 N2 rule:list_induct2)
case Nil
then show ?case by (simp add: big_ior_defs)
next
case (Cons x xs y ys)
then show ?case 
   by (metis Cons.hyps(1) big_ior_append length_Cons map_append zip_append)       
         
qed

lemma map2_iand_next_defs : 
 assumes "length F = length G"
 shows "(\<exists>i<length G. \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand (next y)) F G ! i) \<longleftrightarrow>
        (\<exists>i<length G. (\<sigma> \<Turnstile>  F !i) \<and> (\<sigma> \<Turnstile> (next (G ! i)))) "
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
  proof (cases "F=[]")
  case True
  then show ?thesis using Cons.prems by auto
  next
  case False
  then show ?thesis using Cons by auto 
  qed
qed 

lemma map2_iand_inot_next_defs : 
 assumes "length F = length G"
         "0<nlength \<sigma>" 
 shows "(\<exists>i<length G. \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand (next (inot y))) F G ! i) \<longleftrightarrow>
        (\<exists>i<length G. (\<sigma> \<Turnstile> F! i) \<and> \<not>(\<sigma> \<Turnstile>  (next (G!i))))" 
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
   proof (cases "F=[]")
   case True
   then show ?thesis using Cons by simp
   next
   case False
   then show ?thesis 
     proof -
      obtain b F1 where 1: "F = b#F1" 
        using False by (meson neq_Nil_conv) 
     have 2: " length F1 = length G" 
          using "1" Cons.prems by auto
     have 3: "(\<exists>i<length G. \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) F1 G ! i) = 
              (\<exists>i<length G. (\<sigma> \<Turnstile> F1 ! i) \<and> \<not> (\<sigma> \<Turnstile> next G ! i))" 
        using "2" Cons.hyps assms(2) by presburger
     have 4: "(\<exists>i<length (a # G). \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) F (a # G) ! i) \<longleftrightarrow>
              (\<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (b#F1) (a # G) ! 0) \<or>
              (\<exists>i. 0 < i \<and> i<length (a # G) \<and> ( \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (b#F1) (a # G) ! i))"
         using 1  by (metis gr_zeroI length_greater_0_conv list.discI)
     have 5: "(\<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (b#F1) (a # G) ! 0) \<longleftrightarrow>
              (\<sigma> \<Turnstile> b iand next inot a)"
       by auto
     have 6: "(\<exists>i. 0 < i \<and> i<length (a # G) \<and> ( \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (b#F1) (a # G) ! i)) \<longleftrightarrow>
              (\<exists>i<length ( G).  ( \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (F1) (G) ! i)) " 
        by auto
           (metis Suc_less_eq Suc_pred)
     have 7: "(\<exists>i<length (a # G). (\<sigma> \<Turnstile> (b#F1) ! i) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! i))  \<longleftrightarrow>
              ((\<sigma> \<Turnstile> (b#F1) ! 0) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! 0)) \<or>
              (\<exists>i. 0< i \<and> i<length (a # G) \<and> (\<sigma> \<Turnstile> (b#F1) ! i) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! i)) " 
        by (metis gr_zeroI length_greater_0_conv list.discI)
     have 8: "((\<sigma> \<Turnstile> (b#F1) ! 0) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! 0)) \<longleftrightarrow>
              (\<sigma> \<Turnstile> b iand next inot a)" 
        by simp
         (metis One_nat_def Suc_ile_eq assms(2) enat_defs(1) enat_defs(2))
     have 9: "(\<exists>i. 0< i \<and> i<length (a # G) \<and> (\<sigma> \<Turnstile> (b#F1) ! i) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! i)) \<longleftrightarrow>
              (\<exists>i<length ( G). (\<sigma> \<Turnstile> (F1) ! i) \<and> \<not> (\<sigma> \<Turnstile> next ( G) ! i))" 
         by auto
          (metis Suc_diff_Suc Suc_less_eq diff_zero, metis Suc_less_eq Suc_pred) 
     show ?thesis 
     using "1" "3" "4" "5" "6" "7" "8" "9" by presburger
    qed    
   qed
qed

lemma iand_big_ior_dist: 
assumes "length G_now = length G_next"
shows "\<turnstile> F_e iand big_ior (map2 (\<lambda> x y. x iand next y) G_now G_next) ieqv
         big_ior (map2 (\<lambda> x y. x iand next y) 
                        (map (\<lambda>x. x iand F_e) G_now) G_next)"
using assms by (auto simp add: big_ior_defs)


lemma big_ior_map2_defs: 
 assumes "length L1 = length L2"
 shows " (\<sigma> \<Turnstile> (big_ior (map2 (\<lambda>x y. f x y) L1 L2))) \<longleftrightarrow>
         (\<exists>i<length L1. (\<sigma> \<Turnstile> f (L1!i) (L2!i)))"
using assms
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case unfolding big_ior_def by auto
next
case (Cons x xs y ys)
then show ?case using big_ior_def less_Suc_eq_0_disj by simp auto
qed

lemma big_ior_map2_schop_dis: 
 assumes "length L1 = length L2"
 shows " \<turnstile> (big_ior (map2 (\<lambda>x y. f x y) L1 L2)) schop g ieqv
           (big_ior (map2 (\<lambda>x y. (f x y) schop g) L1 L2)) " 
using assms 
by (auto simp add: big_ior_map2_defs)

subsection \<open>Quantifier lemmas\<close>

lemma ExistSubst: 
 assumes  "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_qpitl w" 
 shows "\<turnstile> suform f n w iimp (Ex n. f)"
 using assms subst_suform valid_def by force


(* Spec *)
lemma ForallSubst: 
 assumes  "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_qpitl w" 
 shows "\<turnstile> (Fa n. f) iimp suform f n w "
using assms  by (simp add:  subst_suform)


lemma ExistsImp: 
 assumes " v1 \<notin> fvars f1"
 shows " \<turnstile> (Ex v1. (f1 iimp f2)) ieqv (f1 iimp (Ex v1. f2)) " 
proof -
 have 1: "\<And> \<sigma>. (\<sigma> \<Turnstile> (Ex v1. (f1 iimp f2))) = (\<sigma> \<Turnstile> (f1 iimp (Ex v1. f2))) " 
  by simp
     (metis assms nlength_nmap not_fvar_upd) 
 show ?thesis
 using "1" itl_ieq by blast  
qed

lemma ExistsSChopRight:
 assumes " v1 \<notin> fvars f2"
 shows " \<turnstile> (Ex v1. f1 schop f2) iimp (Ex v1. f1) schop f2 " 
proof -
 have 1: "\<And> \<sigma>. (\<sigma> \<Turnstile> (Ex v1. f1 schop f2) iimp (Ex v1. f1) schop f2)" 
 proof -
  fix \<sigma>
  show "(\<sigma> \<Turnstile> (Ex v1. f1 schop f2) iimp (Ex v1. f1) schop f2)"
  proof -
   have 2: " (\<sigma> \<Turnstile> (Ex v1. f1 schop f2)) =
             (\<exists>l k. nlength \<sigma> = nlength l \<and> k\<le> nlength \<sigma> \<and>  
              (ntaken k (upd \<sigma> v1 l)  \<Turnstile>  f1) \<and>
              (ndropn k  (upd \<sigma> v1 l)  \<Turnstile>  f2 )) " 
   by simp (metis nlength_upd) 
   have 3: "(\<sigma> \<Turnstile> (Ex v1. f1) schop f2) =
            (\<exists> k l. k \<le> nlength \<sigma> \<and> nlength l = nlength (ntaken k \<sigma>) 
              \<and> (upd (ntaken k \<sigma>) v1 l \<Turnstile>  f1) \<and> (ndropn k \<sigma> \<Turnstile> f2))"
   by (metis less_eq_nat.simps(1) semantics_qpitl.simps(5) semantics_qpitl.simps(8))
   have 4: "(\<exists>l k. nlength \<sigma> = nlength l \<and> k\<le> nlength \<sigma> \<and>  
              (ntaken k (upd \<sigma> v1 l)  \<Turnstile>  f1) \<and>
              (ndropn k  (upd \<sigma> v1 l)  \<Turnstile>  f2 )) \<Longrightarrow>
            (\<exists> k l. k \<le> nlength \<sigma> \<and> nlength l = nlength (ntaken k \<sigma>) 
              \<and> (upd (ntaken k \<sigma>) v1 l \<Turnstile>  f1) \<and> (ndropn k \<sigma> \<Turnstile> f2))"
   by (metis assms ndropn_nlength ndropn_upd not_fvar_upd ntaken_nlength ntaken_upd)
   show ?thesis
   using "2" "3" "4" semantics_qpitl.simps(3) by presburger   
  qed
 qed
 show ?thesis
 using "1" valid_def by blast  
qed

lemma ExistsSChopLeft:
 assumes " v1 \<notin> fvars f1"
 shows " \<turnstile> (Ex v1. f1 schop f2) iimp f1 schop (Ex v1. f2)  " 
proof - 
 have 1: "\<And> \<sigma>. \<sigma> \<Turnstile> (Ex v1. f1 schop f2) iimp f1 schop (Ex v1. f2)" 
  proof -
   fix \<sigma>
   show "\<sigma> \<Turnstile> (Ex v1. f1 schop f2) iimp f1 schop (Ex v1. f2)"
   proof -
    have 2: "(\<sigma> \<Turnstile> (Ex v1. f1 schop f2)) =
             (\<exists>l k. nlength \<sigma> = nlength l \<and> k\<le> nlength \<sigma> \<and>  
              (ntaken k (upd \<sigma> v1 l)  \<Turnstile>  f1) \<and>
              (ndropn k  (upd \<sigma> v1 l)  \<Turnstile>  f2 )) " 
    by simp (metis nlength_upd) 
    have 3: "(\<sigma> \<Turnstile> f1 schop (Ex v1. f2)) =
             (\<exists> k l. k\<le> nlength \<sigma> \<and> nlength l = nlength (ndropn k \<sigma>) \<and>
               (ntaken k \<sigma> \<Turnstile> f1) \<and> (upd (ndropn k \<sigma>) v1 l \<Turnstile> f2))"
    by force
    have 4: "(\<exists>l k. nlength \<sigma> = nlength l \<and> k\<le> nlength \<sigma> \<and>  
              (ntaken k (upd \<sigma> v1 l)  \<Turnstile>  f1) \<and>
              (ndropn k  (upd \<sigma> v1 l)  \<Turnstile>  f2 )) \<Longrightarrow>
             (\<exists> k l. k\<le> nlength \<sigma> \<and> nlength l = nlength (ndropn k \<sigma>) \<and>
               (ntaken k \<sigma> \<Turnstile> f1) \<and> (upd (ndropn k \<sigma>) v1 l \<Turnstile> f2))"
    by (metis assms ndropn_nlength ndropn_upd not_fvar_upd ntaken_nlength ntaken_upd) 
   show ?thesis
   using "2" "3" "4" semantics_qpitl.simps(3) by presburger
  qed
 qed
 show ?thesis
 using "1" valid_def by blast 
qed

(* ForallI *) 
lemma ForallGen: 
assumes "\<turnstile> f "
shows   "\<turnstile> Fa v. f" 
using assms by simp
 
lemma ExistsI:
assumes " (\<sigma> \<Turnstile> f) " 
shows   " (\<sigma> \<Turnstile> Ex v. f) "
using assms exist_sigma_exist_value semantics_qpitl.simps(8) similar_refl by blast
 
lemma ExistsNoDep:
 assumes " v \<notin> fvars f" 
 shows   " \<turnstile> (Ex v . f) ieqv f"
using ExistsI assms not_fvar_upd by auto 

lemma ForaAllNoDep: 
 assumes " v \<notin> fvars f" 
 shows   " \<turnstile> (Fa v. f) ieqv f" 
using assms 
using nlength_nmap not_fvar_upd 
by (metis (mono_tags, lifting) forall_defs itl_ieq)

lemma ExistsEqvRule: 
 assumes " \<turnstile> f ieqv g" 
 shows   " \<turnstile> (Ex v. f) ieqv (Ex v. g) "
using assms by auto 

lemma ForallImpExists: 
 "\<turnstile> (Fa v. f) iimp (Ex v. f) " 
 by simp
(metis exist_sigma_exist_value itrue_defs similar_refl)

lemma ExistsImpRuleDist:
 assumes " \<turnstile> f iimp g" 
 shows " \<turnstile> (Fa v. f) iimp (Ex v. g) " 
using assms by simp
 (metis nlength_nmap)

lemma ExistsImpNoDepDist:
 assumes "v \<notin> fvars f"
         "\<turnstile> f iimp g"
 shows   "\<turnstile> f iimp (Ex v. g) " 
using assms
using ExistsI by force 

lemma ExistsOrDist_1: 
 "\<turnstile> (Ex v. h) iimp (Ex v. f ior h) "
by auto 

lemma ExistsOrDist_2: 
 "\<turnstile> (Ex v. h) iimp (Ex v. h ior f) "
by auto

lemma ExistsOrDist_3: 
" \<turnstile> (Ex v. f) ior (Ex v. h) iimp (Ex v. f ior h) " 
by auto

lemma ExistsOrDist:
 " \<turnstile> (Ex v. f) ior (Ex v. h) ieqv (Ex v. f ior h)"
by auto

lemma ExistsOrImport_1:
assumes " v \<notin> fvars g" 
shows " \<turnstile> g iimp (Ex v.  g ior f) " 
using assms
using ExistsI ior_defs semantics_qpitl.simps(3) valid_def by presburger 
  
lemma ExistsOrImport_2:
assumes " v \<notin> fvars g" 
shows " \<turnstile> (Ex v. f) iimp (Ex v.  g ior f) "      
using assms
using ExistsOrDist_1 by blast 

lemma ExistsOrImport_3:
assumes " v \<notin> fvars g" 
shows " \<turnstile> g ior (Ex v. f) iimp  (Ex v. g ior   f) "    
using assms
using ExistsI by fastforce

lemma ExistsOrImport_4:
assumes " v \<notin> fvars g" 
shows " \<turnstile>  (Ex v. g ior f) iimp  g ior (Ex v. f) "  
using assms
using ExistsNoDep by auto 

lemma ExistsOrImport:
assumes " v \<notin> fvars g" 
shows " \<turnstile>  g ior (Ex v. g ior f) ieqv (Ex v. g ior f) "  
using assms
using ExistsI by fastforce 

lemma ExistsAndImport_1:
 assumes " v \<notin> fvars g" 
  shows " \<turnstile> g iand (Ex v. f) iimp (Ex v. g iand f) " 
using assms
using not_fvar_upd by auto 

lemma ExistsAndImport_2:
 assumes " v \<notin> fvars g" 
  shows " \<turnstile>  (Ex v. g iand f) iimp g iand (Ex v.  f) " 
using assms
using not_fvar_upd by auto 

lemma ExistsAndImport:
 assumes " v \<notin> fvars g" 
  shows " \<turnstile>  g iand (Ex v.  f) ieqv (Ex v. g iand f) " 
using assms 
using not_fvar_upd by auto

lemma ExistsAndDist: 
 assumes "\<turnstile> f iand g" 
 shows "\<turnstile> (Ex v. f) iand (Ex v. g) " 
using assms
using ExistsI by fastforce 

lemma ExistsAndNoDepDist: 
 assumes "\<turnstile> f iand g"
         " v \<notin> fvars f"  
 shows "\<turnstile> f iand (Ex v. g) " 
using assms
by (meson ExistsI iand_defs valid_def) 

lemma ForallE:
 assumes "\<turnstile> Fa v. f" 
         "\<turnstile> f iimp g"
 shows   "\<turnstile> g" 
using assms
by (metis ExistsI forall_d_def inot_defs semantics_qpitl.simps(3) valid_def) 

lemma ForallEqvRule:
 assumes "\<turnstile> f ieqv g "
 shows   "\<turnstile> (Fa v. f) ieqv (Fa v. g) " 
using assms by auto

lemma ForallAndDist:
 "\<turnstile> (Fa v. f iand g) ieqv (Fa v. f) iand (Fa v. g) " 
by auto 

lemma ForallAndImport:
assumes "v \<notin> fvars g" 
shows "\<turnstile> g iand (Fa v. f) ieqv (Fa v. g iand f) " 
using assms
using ForaAllNoDep ForallAndDist by force 

lemma ForallOrImport:
assumes "v \<notin> fvars g" 
shows "\<turnstile> g ior (Fa v. f) ieqv (Fa v. g ior f) " 
using assms
using not_fvar_upd by auto 
 
lemma RightChopImpChop: 
 assumes "\<turnstile> f iimp g" 
 shows   "\<turnstile> h schop f iimp h schop g" 
using assms by auto


lemma ExistsImpChopRule:
 assumes "\<turnstile> f iimp g"
         "v \<notin> fvars h" 
 shows "\<turnstile> (Ex v. (h schop f) iimp (h schop g)) " 
 using assms
by (meson ExistsI RightChopImpChop valid_def) 


lemma SChop_nfuse: 
shows " (\<sigma> \<Turnstile> f schop g) = 
        (\<exists> \<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nlast \<sigma>1 = nfirst \<sigma>2 \<and> nfinite \<sigma>1 \<and>
          (\<sigma>1 \<Turnstile> f) \<and> (\<sigma>2 \<Turnstile> g)) " 
by auto
(metis ndropn_nfirst nfinite_ntaken nfuse_ntaken_ndropn ntaken_nlast,
 metis linorder_le_cases ndropn_nfuse nfinite_conv_nlength_enat ntaken_all ntaken_nfuse the_enat.simps)

lemma ExistsSChopRightNoDep: 
 assumes "v \<notin> fvars g"
 shows " \<turnstile> (Ex v. f schop g) ieqv (Ex v. f) schop g" 
proof -
 have 1: " \<turnstile> (Ex v. f schop g) iimp (Ex v. f) schop g"
 using ExistsSChopRight assms by blast 
 have 2: "\<turnstile> (Ex v. f) schop g iimp (Ex v. f schop g)" 
   proof -
    have 3: "\<And> \<sigma>. (\<sigma> \<Turnstile> (Ex v. f) schop g) \<Longrightarrow> (\<sigma> \<Turnstile> (Ex v. f schop g))"
     proof -
      fix \<sigma> 
      assume a0: "(\<sigma> \<Turnstile> (Ex v. f) schop g)"
      show "(\<sigma> \<Turnstile> (Ex v. f schop g))"
      proof -
       have 4: "(\<exists> k l. k \<le> nlength \<sigma> \<and> nlength l = nlength(ntaken k \<sigma>) 
                  \<and> ((upd (ntaken k \<sigma>) v l) \<Turnstile> f) \<and> (ndropn k \<sigma> \<Turnstile> g) )"
        using a0 by fastforce    
       obtain k l where 5: "k \<le> nlength \<sigma> \<and> nlength l = nlength(ntaken k \<sigma>) 
                  \<and> ((upd (ntaken k \<sigma>) v l) \<Turnstile> f) \<and> (ndropn k \<sigma> \<Turnstile> g)"
       using 4 by blast
       have 6: "\<And> l1. nlength l1 = nlength (ndropn k \<sigma>) \<Longrightarrow>
                  ( (upd (ndropn k \<sigma>) v l1)  \<Turnstile> g)"
       using "5" assms not_fvar_upd by blast 
       obtain l1 where 7: "nlength l1 = nlength (ndropn k \<sigma>) \<and>
                           nfirst l1 = nlast l \<and>
                            ( (upd (ndropn k \<sigma>) v l1)  \<Turnstile> g) "
       using "6" nlength_nmap
       by (metis ndropn_0 ndropn_nfirst nellist.exhaust nlength_NCons nlength_NNil nnth_0 nnth_NNil) 
       have 70: "enat k \<le> nlength (nfuse l l1)"
       by (simp add: "5" nfuse_nlength) 
       have 8: "ntaken k (nfuse l l1) = l"
       by (metis "5" enat_le_plus_same(2) gen_nlength_def min.absorb1 nfuse_def1 nlength_code 
           ntaken_all ntaken_nappend1 ntaken_nlength) 
       have 9: " ndropn k (nfuse l l1) = l1"
         using ndropn_nfuse[of l l1] 
         by (metis "7" "70" "8" min.absorb1 nfinite_ntaken ntaken_nlength the_enat.simps) 
       have 10: "(ntaken k (upd \<sigma> v (nfuse l l1)) \<Turnstile> f)"
       by (metis "5" "7" "8" nfuse_nlength nfuse_ntaken_ndropn ntaken_upd)  
       have 11: "(ndropn k (upd \<sigma> v (nfuse l l1)) \<Turnstile> g)"
       by (metis "5" "7" "9" ndropn_upd nfuse_nlength nfuse_ntaken_ndropn) 
       have 12: "nlength \<sigma> = nlength (nfuse l l1) "
       by (metis "5" "7" nfuse_nlength nfuse_ntaken_ndropn_nlength) 
       show ?thesis using 70 10 11 12
       using nlength_upd by fastforce 
      qed
     qed
     show ?thesis
     using "3" semantics_qpitl.simps(3) valid_def by presburger 
    qed
  show ?thesis
  using "1" "2" ieqv_d_def by auto 
qed

lemma ExistsSChopLeftNoDep: 
 assumes "v \<notin> fvars g"
 shows " \<turnstile> (Ex v. g schop f) ieqv g schop (Ex v. f) " 
proof -
 have 1: "\<turnstile> (Ex v. g schop f) iimp g schop (Ex v. f)"
 using ExistsSChopLeft assms by blast 
 have 2: "\<turnstile>  g schop (Ex v. f) iimp (Ex v. g schop f)" 
  proof -
   have 3: "\<And> \<sigma>. (\<sigma> \<Turnstile>  g schop (Ex v. f)) \<Longrightarrow> (\<sigma> \<Turnstile> (Ex v. g schop f))"
    proof -
     fix \<sigma>
     assume a0: "(\<sigma> \<Turnstile>  g schop (Ex v. f))" 
     show "(\<sigma> \<Turnstile> (Ex v. g schop f))"
     proof -
      have 4: "(\<exists> k l. k \<le> nlength \<sigma> \<and> nlength l = nlength (ndropn k \<sigma>) \<and>
                (ntaken k \<sigma> \<Turnstile> g) \<and> (upd (ndropn k \<sigma>) v l \<Turnstile> f))"
      using a0 by fastforce
      obtain k l where 5: "k \<le> nlength \<sigma> \<and> nlength l = nlength (ndropn k \<sigma>) \<and>
                (ntaken k \<sigma> \<Turnstile> g) \<and> (upd (ndropn k \<sigma>) v l \<Turnstile> f)" 
      using 4 by blast
      have 6: "\<And>l1. nlength l1 = nlength (ntaken k \<sigma>)  \<Longrightarrow>
                   (upd (ntaken k \<sigma>) v l1 \<Turnstile> g)  " 
         using "5" assms nfinite_conv_nlength_enat not_fvar_upd by auto
            
      have 60: "nlength (ntaken k \<sigma>) = k " 
      by (simp add: "5")
      have 61: "nlength \<sigma> = 0 \<Longrightarrow>  \<exists> l1. nlength l1 = nlength (ntaken k \<sigma>) \<and> (upd (ntaken k \<sigma>) v l1 \<Turnstile> g) \<and> nfinite l1 
                \<and> nlast l1 = nfirst l" 
        by (metis "5" "6" le_zero_eq ndropn_0 nfirst_eq_nnth_zero nlength_eq_enat_nfiniteD nnth_nlast 
            ntaken_all the_enat.simps the_enat_0)
      have 62: "nlength \<sigma> >0 \<and> k> 0 \<Longrightarrow> \<exists> l1. nlength (nappend l1 (NNil (nnth l 0)))= nlength (ntaken k \<sigma>) \<and> 
         (upd (ntaken k \<sigma>) v (nappend l1 (NNil (nnth l 0))) \<Turnstile> g) \<and> nfinite l1 " 
        using 6 60 5 by auto 
         (metis One_nat_def Suc_pred add.right_neutral add_Suc_right idiff_enat_enat min.orderE 
          ndropn_nlength nfinite_ntaken nle_le nlength_NNil nlength_nappend nlength_nmap ntaken_all
           ntaken_nlength one_enat_def plus_enat_simps(1))
      have 63: "nlength \<sigma> > 0 \<and> k =0 \<Longrightarrow>  \<exists> l1. nlength l1 = nlength (ntaken k \<sigma>) \<and> (upd (ntaken k \<sigma>) v l1 \<Turnstile> g) \<and> nfinite l1 
                \<and> nlast l1 = nfirst l " 
      by (metis "6" nfinite_NNil nlast_NNil nlength_NNil ntaken_0)
      have 64: "nlength \<sigma> = 0 \<Longrightarrow> ?thesis"
        by (metis "5" "6" ndropn_0 nle_le nlength_upd ntaken_all semantics_qpitl.simps(5) 
            semantics_qpitl.simps(8) the_enat.simps zero_enat_def zero_le) 
      have 65: "nlength \<sigma> > 0 \<and> k =0 \<Longrightarrow> ?thesis"
       by (metis "5" "6" ndropn_0 nlength_upd ntaken_nlength ntaken_upd semantics_qpitl.simps(5) 
           semantics_qpitl.simps(8) zero_le) 
      have 66: "nlength \<sigma> > 0 \<and> k >0 \<Longrightarrow> ?thesis"
       proof -
        assume a1: "nlength \<sigma> > 0 \<and> k >0" 
        show ?thesis
        proof -
         have 70: "\<exists> l1. nlength (nappend l1 (NNil (nnth l 0)))= nlength (ntaken k \<sigma>) \<and> 
         (upd (ntaken k \<sigma>) v (nappend l1 (NNil (nnth l 0))) \<Turnstile> g) \<and> nfinite l1" 
          using a1 62 by auto
         obtain l1 where 71: "nlength (nappend l1 (NNil (nnth l 0)))= nlength (ntaken k \<sigma>) \<and> 
         (upd (ntaken k \<sigma>) v (nappend l1 (NNil (nnth l 0))) \<Turnstile> g) \<and> nfinite l1"
          using 70 by blast
         have 72: "nlast  (nappend l1 (NNil (nnth l 0))) = nfirst l " 
          by (simp add: "71" nfirst_eq_nnth_zero nlast_nappend)
         have 73: "enat k \<le> nlength (nfuse (nappend l1 (NNil (nnth l 0))) l)"
          by (metis "60" "71" enat_le_plus_same(1) nfuse_nlength) 
         have 74: "ntaken k (nfuse (nappend l1 (NNil (nnth l 0))) l) = (nappend l1 (NNil (nnth l 0)))"         
          using "60" "71" "72" ntaken_nfuse[of "(nappend l1 (NNil (nnth l 0)))" l]  
          nlength_eq_enat_nfiniteD  by (metis the_enat.simps)
         have 75: "ndropn k (nfuse (nappend l1 (NNil (nnth l 0))) l) = l"
          by (metis "60" "71" "74" ndropn_nfuse nfinite_ntaken nfirst_eq_nnth_zero nlast_NNil 
              nlast_nappend the_enat.simps) 
         have 76: "(ntaken k (upd \<sigma> v (nfuse (nappend l1 (NNil (nnth l 0))) l)) \<Turnstile> g)"
          by (metis "5" "71" "74" nfuse_nlength nfuse_ntaken_ndropn ntaken_upd) 
         have 77: "(ndropn k (upd \<sigma> v (nfuse (nappend l1 (NNil (nnth l 0))) l)) \<Turnstile> f)"
          by (metis "5" "71" "75" ndropn_upd nfuse_nlength nfuse_ntaken_ndropn)
         have 78: "nlength \<sigma> = nlength (nfuse (nappend l1 (NNil (nnth l 0))) l)" 
          by (metis "5" "71" nfuse_nlength nfuse_ntaken_ndropn_nlength)
         show ?thesis by simp 
          (metis "5" "76" "77" "78" nlength_upd)
        qed
       qed
     show ?thesis 
     by (meson "64" "65" "66" i0_less less_nat_zero_code linorder_neqE_nat)
    qed
   qed
  show ?thesis
  using "3" semantics_qpitl.simps(3) valid_def by presburger
  qed
 show ?thesis 
 using "1" "2" ieqv_d_def by auto 
qed
  
lemma ExistsExistsSChop: 
assumes " v \<notin> fvars g" 
        " y \<notin> fvars f"  
shows "\<turnstile> (Ex v. (Ex y. f schop g)) ieqv (Ex v. f) schop (Ex y. g) " 
proof -
 have 1: "\<turnstile> ( (Ex y. f schop g)) ieqv  ( f schop (Ex y.  g) ) " 
  using assms ExistsSChopLeftNoDep[of y f g] by blast
 have 2: "\<turnstile> (Ex v. (Ex y. f schop g)) ieqv  (Ex v.  f schop (Ex y.  g) )" 
  using "1" ExistsEqvRule by blast
 have 3: "\<turnstile>  (Ex v.  f schop (Ex y.  g) ) ieqv (Ex v. f) schop (Ex y.  g) " 
  by (metis Diff_iff ExistsSChopRightNoDep assms(1) fvars.simps(8))
 show ?thesis 
 by (meson "2" "3" itl_ieq)
qed
 



lemma exists_elim_state_qpitl: 
 assumes "state_qpitl w"
         "n \<notin> bvars w" 
 shows " \<turnstile> (Ex n. w) ieqv (suform w n itrue) ior (suform w n ifalse) " 
proof -
 have 1: "state_qpitl itrue" 
    by (simp add: inot_d_def itrue_d_def)
 have 2: "(\<And>z. z \<in> fvars itrue \<Longrightarrow> z \<notin> bvars w)" 
    by (simp add: inot_d_def itrue_d_def)
 have 3: "\<And> \<sigma>. (\<sigma> \<Turnstile> (suform w n itrue)) = ((upd \<sigma> n (nmap (\<lambda>c. (NNil c) \<Turnstile> itrue) \<sigma>)) \<Turnstile> w)" 
    using subst_suform[of n w itrue ] "1" "2" assms(2) by blast
 have 4: "state_qpitl ifalse"
    by simp
 have 5: "(\<And>z. z \<in> fvars ifalse \<Longrightarrow> z \<notin> bvars w)"  
   by simp
 have 6: "\<And> \<sigma>. (\<sigma> \<Turnstile> (suform w n ifalse)) = ((upd \<sigma> n (nmap (\<lambda>c. (NNil c) \<Turnstile> ifalse) \<sigma>)) \<Turnstile> w)" 
   using subst_suform[of n w ifalse ] "4" "5" assms(2) by blast
 have 7: "\<And> \<sigma>. (\<sigma> \<Turnstile> (Ex n. w)) = (\<exists>l. nlength \<sigma> = nlength l \<and> (upd \<sigma> n l \<Turnstile> w))"  
    by simp 
 have 8: "\<And> \<sigma>. (\<exists>l. nlength \<sigma> = nlength l \<and> (upd \<sigma> n l \<Turnstile> w)) = 
                (\<exists>l. nlength \<sigma> = nlength l \<and> ((NNil (nfirst (upd \<sigma> n l)))  \<Turnstile> w))" 
   using assms(1) state_qpitl_defs by blast
 have 9: "\<And> \<sigma>. ((upd \<sigma> n (nmap (\<lambda>c. (NNil c) \<Turnstile> itrue) \<sigma>)) \<Turnstile> w) =
                (NNil (nfirst (upd \<sigma> n (nmap (\<lambda>c. (NNil c) \<Turnstile> itrue) \<sigma>))) \<Turnstile> w)"
     using assms(1) state_qpitl_defs by blast 
 have 10: "\<And> \<sigma>.  (nfirst (upd \<sigma> n (nmap (\<lambda>c. (NNil c) \<Turnstile> itrue) \<sigma>))) =
                  insert n (nnth \<sigma> 0)"
      by (metis itrue_defs nfirst_eq_nnth_zero nfirst_upd nlength_nmap nnth_nmap zero_enat_def zero_le) 
 have 11: "\<And> \<sigma>. ((upd \<sigma> n (nmap (\<lambda>c. (NNil c) \<Turnstile> ifalse) \<sigma>)) \<Turnstile> w) =
                (NNil (nfirst (upd \<sigma> n (nmap (\<lambda>c. (NNil c) \<Turnstile> ifalse) \<sigma>))) \<Turnstile> w)"
     using assms(1) state_qpitl_defs by blast 
 have 12: "\<And> \<sigma>.  (nfirst (upd \<sigma> n (nmap (\<lambda>c. (NNil c) \<Turnstile> ifalse) \<sigma>))) =
                   (nnth \<sigma> 0) - {n}"
     by (metis i0_less nfirst_eq_nnth_zero nfirst_upd nle_le nlength_nmap nnth_nmap order_less_imp_le 
        semantics_qpitl.simps(1) zero_enat_def) 
 have 13: "\<And> \<sigma> l. nlength \<sigma> = nlength l \<Longrightarrow> (nfirst (upd \<sigma> n l)) = (if (nnth l 0) then insert n (nnth \<sigma> 0) else (nnth \<sigma> 0) - {n})"
    by (metis ndropn_0 ndropn_nfirst nfirst_upd)
 have 14: "\<And> \<sigma>. (\<exists>l. nlength \<sigma> = nlength l \<and> ((NNil (nfirst (upd \<sigma> n l)))  \<Turnstile> w)) \<longleftrightarrow>
                (NNil (insert n (nnth \<sigma> 0)) \<Turnstile> w) \<or> (NNil ((nnth \<sigma> 0) -{n}) \<Turnstile> w) " 
      by (metis "10" "12" ndropn_0 ndropn_nfirst nfirst_upd nlength_nmap) 
 have 15: "\<And> \<sigma>. (\<sigma> \<Turnstile> (Ex n. w)) \<longleftrightarrow> (\<sigma> \<Turnstile> (suform w n itrue)) \<or> (\<sigma> \<Turnstile> (suform w n ifalse)) " 
   by (metis "10" "12" "14" "3" "6" "7" assms(1) state_qpitl_defs)
 show ?thesis 
 using "15" ior_defs itl_ieq by presburger
qed
    

end

