(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                Dimitar Guelev
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

chapter \<open>Deep embedding of PITL\<close>

theory PITL_Deep imports
 NELList_Extras

begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="cvc4 z3 e spass vampire " ]
*)
text \<open>
This theory is a deep embedding of Propositional Interval Temporal Logic (PITL).
It provides the syntax and semantics. 
 
\<close>
section \<open>Syntax\<close>

subsection \<open>Primitive formulae\<close>

datatype  pitl =
    false_d                          ("ifalse")
  | atom_d "nat "                    ("(atom _ )") 
  | iimp_d " pitl" " pitl"           ("( _ iimp _ )" [26,25] 25)
  | next_d " pitl"                   ("(next _)"  [88] 87)
  | chop_d " pitl" " pitl"           ("( _ schop _ )" [84,84] 83)
  | schopstar_d "pitl"              ("(schopstar _ )" [85] 85)
  | omega_d " pitl"                  ("(omega _ )" [85] 85)  


subsection \<open>state formula\<close>

fun state_pitl :: " pitl \<Rightarrow> bool" 
where
  "state_pitl (ifalse) = True"
| "state_pitl (atom p) = True "
| "state_pitl (f iimp g) = ((state_pitl f) \<and> (state_pitl g))"
| "state_pitl (next f)  = False "
| "state_pitl (f schop g) = False "
| "state_pitl (schopstar f) = False "
| "state_pitl (omega f) = False  " 

subsection \<open>Derived Boolean Operators\<close>

definition inot_d ("(inot _)" [90] 90) 
where 
  "inot f \<equiv> f iimp ifalse"

definition itrue_d ("itrue") 
where 
  "itrue \<equiv> inot  ifalse"

definition ior_d ("(_ ior _)" [31,30] 30) 
where 
  "f ior g \<equiv> (inot f) iimp g"

definition iand_d ("(_ iand _)" [36,35] 35)
where 
  "f iand g \<equiv> inot (inot f ior inot g)"

definition ieqv_d ( "(_ ieqv _) " [21,20] 20)
where 
  "f ieqv g \<equiv> ( ( f iimp g) iand  (g iimp f))" 
 

subsection \<open>more, empty, skip and wnext \<close>

definition more_d ("more") 
where 
  "more \<equiv> next itrue"

definition empty_d ("empty")
where
 "empty \<equiv> inot more"

definition skip_d ("skip") 
where 
  "skip \<equiv> next empty"

definition wnext_d ("(wnext _)" [88] 87)
where
 "wnext f \<equiv> inot (next (inot f))" 


subsection \<open>ifinite and inf\<close>

definition finite_d ("ifinite") 
where 
  "ifinite \<equiv> itrue schop empty"

definition inf_d ("inf") 
where 
  "inf \<equiv> inot ifinite"


subsection \<open>weak chop\<close>

definition chop_d ("( _ ; _ )" [84,84] 83)
 where " f ; g \<equiv> (f schop g) ior (f iand inf)" 

subsection \<open>Box and Diamond Operators\<close>

definition sometimes_d ("(diamond _)" [88] 87)
where 
  "diamond f \<equiv> itrue schop f"

definition always_d ("(box _)" [88] 87)
where 
  "box f \<equiv> inot (diamond (inot f))"


subsection \<open>Initial and Final Operators\<close>

definition init_d ("(init _)" [88] 87)
where
 " init f \<equiv> ((empty iand f) schop itrue)"

definition fin_d ("(fin _)" [88] 87)
where
  "fin f \<equiv> diamond ( empty iand f)"


subsection \<open> Big operations\<close>


definition big_ior :: "  pitl list \<Rightarrow>  pitl "
 where " big_ior F = foldr (\<lambda> x y. x ior y) F ifalse " 

definition big_iand :: "  pitl list \<Rightarrow>  pitl "
 where " big_iand F = foldr (\<lambda> x y. x iand y) F itrue " 


definition state_pitl_list :: " pitl list \<Rightarrow> bool"
 where " state_pitl_list L = foldr (\<lambda>x y. state_pitl x \<and> y) L True " 



section \<open>Semantics\<close>

subsection \<open>Intervals\<close>

type_synonym  interval = " nat set nellist " 


subsection \<open>Semantics Primitive Operators\<close>

fun semantics_pitl :: "[ interval,  pitl] \<Rightarrow> bool" ("(_ \<Turnstile> _)" [80,10] 10)
where
  "(\<sigma> \<Turnstile> ifalse) = False"
| "(\<sigma> \<Turnstile> atom p) = (p \<in> (nfirst \<sigma>))"
| "(\<sigma> \<Turnstile> f iimp g) = ((\<sigma> \<Turnstile> f) \<longrightarrow> (\<sigma> \<Turnstile> g))"
| "(\<sigma> \<Turnstile> (next f))  = (1 \<le> nlength \<sigma>  \<and> ((ndropn 1 \<sigma>) \<Turnstile> f))"
| "(\<sigma> \<Turnstile> f schop g) =
   (\<exists>i. ((0 \<le> i) \<and> (i \<le> (nlength \<sigma>)) \<and> ((ntaken i \<sigma>) \<Turnstile> f) \<and> ((ndropn i \<sigma>) \<Turnstile> g)) )"
| "(\<sigma> \<Turnstile> (schopstar f)) = 
    (\<exists> (l:: nat nellist). 
                   (nnth l 0) = 0 \<and> nfinite l \<and> nidx l  \<and> 
                   (enat (nlast l)) = (nlength \<sigma>) \<and> nfinite \<sigma> \<and> 
                   (\<forall> (i::nat) . ( (enat i)< (nlength l)) \<longrightarrow> 
                               ((nsubn \<sigma> (nnth l i) (nnth l (Suc i)) ) \<Turnstile> f) )
             )
    "
| "(\<sigma> \<Turnstile> (omega f)) = 
    (\<exists>(l:: nat nellist). 
              \<not>nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
              (\<forall>i. (nnth l i) \<le> nlength \<sigma>) \<and>
              (\<forall>i. ( (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> f) ) 
           ) " 


lemma inot_defs [simp]:
  " (\<sigma> \<Turnstile> inot f ) = Not ( \<sigma> \<Turnstile> f) " 
by (simp add: inot_d_def)

lemma ior_defs [simp]: 
  " (\<sigma> \<Turnstile> f1 ior f2) = ( (\<sigma> \<Turnstile> f1) \<or> (\<sigma> \<Turnstile> f2))"
by (metis inot_defs ior_d_def semantics_pitl.simps(3))

lemma iand_defs [simp]:
  " (\<sigma> \<Turnstile> f1 iand f2) = ( (\<sigma> \<Turnstile> f1) \<and> (\<sigma> \<Turnstile> f2))"
by (simp add: iand_d_def)

lemma ieqv_defs [simp]: 
  " (\<sigma> \<Turnstile> f1 ieqv f2) = ( (\<sigma> \<Turnstile> f1) = (\<sigma> \<Turnstile> f2))"
by (metis iand_defs ieqv_d_def semantics_pitl.simps(3))

lemma itrue_defs [simp]:
  " (\<sigma> \<Turnstile> itrue)"
by (simp add: itrue_d_def)

lemma empty_defs [simp]: 
  "(\<sigma> \<Turnstile> empty) = ( (nlength \<sigma>) = 0) "
by (auto simp add: empty_d_def more_d_def)
   (metis co.enat.exhaust_sel enat_le_plus_same(2) plus_1_eSuc(2))

lemma state_pitl_defs: 
assumes "state_pitl w"
shows " (\<sigma> \<Turnstile> w) \<longleftrightarrow> ((NNil (nfirst \<sigma>)) \<Turnstile> w) "
using assms
proof (induct w)
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case 
by (metis nfirst_nfuse nfuse_leftneutral nlast_NNil semantics_pitl.simps(2))
next
case (iimp_d w1 w2)
then show ?case by simp
next
case (next_d w)
then show ?case by auto
next
case (chop_d w1 w2)
then show ?case by auto
next
case (schopstar_d w)
then show ?case by auto
next
case (omega_d w)
then show ?case by auto
qed

lemma state_pitl_list_defs: 
 "state_pitl_list L \<longleftrightarrow> (\<forall>i<length L. state_pitl (L!i)) " 
proof (induct L)
case Nil
then show ?case unfolding state_pitl_list_def by simp
next
case (Cons a L)
then show ?case 
  proof -
   have 1: "state_pitl_list (a # L) \<longleftrightarrow> state_pitl a \<and> state_pitl_list L"
      unfolding  state_pitl_list_def by simp
   have 2: "(\<forall>i<length (a#L). state_pitl ((a#L)!i)) \<longleftrightarrow>
            state_pitl ((a#L)!0) \<and>
            (\<forall>i. 0<i \<and> i<length (a#L)\<longrightarrow> state_pitl ((a#L)!i))" 
        by (metis length_greater_0_conv list.discI zero_less_iff_neq_zero)
   have 3: " state_pitl ((a#L)!0) \<longleftrightarrow> state_pitl a " 
      by simp
   have 4: "(\<forall>i. 0<i \<and> i<length (a#L)\<longrightarrow> state_pitl ((a#L)!i)) \<longleftrightarrow>
            (\<forall>i<length (L). state_pitl (L!i)) " 
     by auto
   show ?thesis 
   using "1" "2" "3" "4" local.Cons by presburger
 qed
qed



lemma big_ior_defs : 
 " (\<sigma> \<Turnstile> (big_ior F)) \<longleftrightarrow> (\<exists> i< length F. (\<sigma> \<Turnstile> F ! i))"
proof (induct F)
case Nil
then show ?case unfolding big_ior_def by simp
next
case (Cons a F)
then show ?case 
   proof -
    have 1: "(\<sigma> \<Turnstile> (big_ior (a#F))) \<longleftrightarrow> (\<sigma> \<Turnstile> a ior (big_ior F))"
        unfolding big_ior_def by simp
    have 2: "(\<exists>i<length (a # F). \<sigma> \<Turnstile> (a # F) ! i) \<longleftrightarrow>
               (\<sigma> \<Turnstile> (a # F) ! 0) \<or>
               (\<exists>i. 0 < i \<and> i<length (a # F) \<and> (\<sigma> \<Turnstile> (a # F) ! i))" 
      by (metis length_greater_0_conv neq_Nil_conv zero_less_iff_neq_zero)  
    have 3: "(\<sigma> \<Turnstile> (a # F) ! 0) = (\<sigma> \<Turnstile> a)"
       by auto
    have 4: "(\<exists>i. 0 < i \<and> i<length (a # F) \<and> (\<sigma> \<Turnstile> (a # F) ! i)) \<longleftrightarrow>
             (\<exists>i<length ( F). (\<sigma> \<Turnstile> ( F) ! i))"  
       by auto
         (metis Suc_le_eq diff_Suc_Suc diff_zero gr0_conv_Suc less_Suc_eq_le)
     show ?thesis 
     using "1" "2" "4" local.Cons by force
   qed
qed


lemma big_iand_defs : 
 " (\<sigma> \<Turnstile> (big_iand F)) \<longleftrightarrow> (\<forall> i< length F. (\<sigma> \<Turnstile> F ! i))"
proof (induct F)
case Nil
then show ?case unfolding big_iand_def by simp
next
case (Cons a F)
then show ?case 
   proof -
    have 1: "(\<sigma> \<Turnstile> (big_iand (a#F))) \<longleftrightarrow> (\<sigma> \<Turnstile> a iand (big_iand F))"
        unfolding big_iand_def by simp
    have 2: "(\<forall>i<length (a # F). \<sigma> \<Turnstile> (a # F) ! i) \<longleftrightarrow>
               (\<sigma> \<Turnstile> (a # F) ! 0) \<and>
               (\<forall>i. 0 < i \<and> i<length (a # F) \<longrightarrow> (\<sigma> \<Turnstile> (a # F) ! i))" 
      by (metis length_greater_0_conv neq_Nil_conv zero_less_iff_neq_zero)  
    have 3: "(\<sigma> \<Turnstile> (a # F) ! 0) = (\<sigma> \<Turnstile> a)"
       by auto
    have 4: "(\<forall>i. 0 < i \<and> i<length (a # F) \<longrightarrow> (\<sigma> \<Turnstile> (a # F) ! i)) \<longleftrightarrow>
             (\<forall>i<length ( F). (\<sigma> \<Turnstile> ( F) ! i))"  
       by auto
     show ?thesis 
     using "1" "2" "4" local.Cons by force
   qed
qed


lemma big_ior_map_iand_zip : 
assumes "length F =length G" 
shows "(\<sigma> \<Turnstile>  big_ior (map2 iand_d  F G)) \<longleftrightarrow>
       (\<exists>i < length F. (\<sigma> \<Turnstile> F ! i iand G ! i)) "
using assms
proof (induct F arbitrary: G)
case Nil
then show ?case by (simp add: big_ior_defs)
next
case (Cons a F)
then show ?case 
   proof (cases "G=[]")
   case True
   then show ?thesis using Cons.prems by auto
   next
   case False
   then show ?thesis using Cons by (auto simp add: big_ior_defs)
   qed
qed

lemma big_iand_map_iimp_zip : 
assumes "length F =length G" 
shows "(\<sigma> \<Turnstile>  big_iand (map2 iimp_d F G)) \<longleftrightarrow>
       (\<forall>i < length F. (\<sigma> \<Turnstile> F ! i iimp G ! i)) "
using assms
proof (induct F arbitrary: G)
case Nil
then show ?case by (simp add: big_iand_defs)
next
case (Cons a F)
then show ?case 
   proof (cases "G=[]")
   case True
   then show ?thesis using Cons.prems by auto
   next
   case False
   then show ?thesis using Cons by (auto simp add: big_iand_defs)
   qed
qed


lemma map2_iand_defs : 
 assumes "length F = length G"
 shows "(\<exists>i<length G. \<sigma> \<Turnstile> map2 iand_d F G ! i) \<longleftrightarrow>
        (\<exists>i<length G. (\<sigma> \<Turnstile>  F !i) \<and> (\<sigma> \<Turnstile> G ! i)) "
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
  proof (cases "F=[]")
  case True
  then show ?thesis using Cons.prems by auto
  next
  case False
  then show ?thesis using Cons by auto 
  qed
qed 

lemma map2_iand_inot_defs : 
 assumes "length F = length G"
 shows "(\<exists>i<length G. \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand inot y) F G ! i) \<longleftrightarrow>
        (\<exists>i<length G. (\<sigma> \<Turnstile> F! i) \<and> \<not>(\<sigma> \<Turnstile>  G!i))" 
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
  proof (cases "F=[]")
  case True
  then show ?thesis using Cons.prems by auto
  next
  case False
  then show ?thesis using Cons by auto 
  qed
qed 

subsection \<open>Validity\<close>

definition valid :: " pitl \<Rightarrow> bool" ("(\<turnstile> _)" 5)
where "(\<turnstile> f) = (\<forall> \<sigma>. (\<sigma> \<Turnstile> f))"

lemma itl_valid [simp] :
  " (\<turnstile> f) = (\<forall> \<sigma>. (\<sigma> \<Turnstile> f))"
by (simp add: valid_def)

lemma itl_ieq:
  " (\<turnstile> f ieqv g) = (\<forall> \<sigma>. (\<sigma> \<Turnstile> f) = (\<sigma> \<Turnstile> g)) "
by (auto simp add: ieqv_d_def iand_d_def ior_d_def inot_d_def) 

lemma big_ior_iimp_subset_eq: 
 assumes "set L1 \<subseteq> set L2"
 shows " \<turnstile> big_ior L1 iimp big_ior L2" 
using assms 
proof (induct  rule: list_induct2')
case 1
then show ?case by simp
next
case (2 x xs)
then show ?case by simp
next
case (3 y ys)
then show ?case by (simp add: big_ior_defs)
next
case (4 x xs y ys)
then show ?case 
  proof (cases "x=y")
  case True
  then show ?thesis using 4 by (auto simp add: big_ior_defs)
  (metis diff_Suc_1 in_set_conv_nth length_Cons less_Suc_eq_0_disj list.simps(15) nth_Cons' subsetD)
  next
  case False
  then show ?thesis using 4 by (auto simp add: big_ior_defs)
    (metis "4.prems" in_set_conv_nth length_Cons subsetD)
  qed
qed

lemma big_ior_ieqv_set_eq: 
 assumes "set L1 = set L2"
 shows " \<turnstile> big_ior L1 ieqv big_ior L2" 
using assms 
by (metis big_ior_iimp_subset_eq dual_order.refl iand_defs ieqv_d_def valid_def)

lemma big_ior_ieqv_remdups: 
 shows " \<turnstile> big_ior L ieqv big_ior (remdups L)"
by (metis big_ior_ieqv_set_eq set_remdups)
 

lemma big_iand_iimp_subset_eq: 
 assumes "set L1 \<subseteq> set L2"
 shows " \<turnstile> big_iand L2 iimp big_iand L1" 
using assms 
proof (induct  rule: list_induct2')
case 1
then show ?case by simp
next
case (2 x xs)
then show ?case by (simp add: big_iand_defs)
next
case (3 y ys)
then show ?case by simp
next
case (4 x xs y ys)
then show ?case 
  proof (cases "x=y")
  case True
  then show ?thesis using 4 by (auto simp add: big_iand_defs)
  (metis (mono_tags, lifting) "4.prems" all_nth_imp_all_set length_Cons list_ball_nth subsetD)
  next
  case False
  then show ?thesis using 4 by (auto simp add: big_iand_defs)
  (metis "4.prems" add.right_neutral add_Suc_right in_set_conv_nth list.size(4) subsetD)    
  qed
qed

lemma big_iand_ieqv_set_eq: 
 assumes "set L1 = set L2"
 shows " \<turnstile> big_iand L1 ieqv big_iand L2" 
using assms 
by (metis big_iand_iimp_subset_eq dual_order.refl iand_defs ieqv_d_def valid_def)

lemma big_iand_ieqv_remdups: 
 " \<turnstile> big_iand L ieqv big_iand (remdups L)"
 by (metis big_iand_ieqv_set_eq set_remdups)



lemma big_ior_empty:
 "(\<turnstile> big_ior []) \<longleftrightarrow> (\<turnstile> ifalse) " 
by (auto simp add: big_ior_defs)

lemma big_ior_Cons: 
 "(\<turnstile> big_ior (f # L)) \<longleftrightarrow> (\<turnstile> f ior (big_ior L)) "
proof -
 have 1: "(\<turnstile> big_ior (f # L)) \<Longrightarrow> (\<turnstile> f ior (big_ior L))" 
  by (auto simp add: big_ior_defs)
    (metis diff_Suc_1 gr0_conv_Suc less_Suc_eq less_imp_diff_less nth_Cons')
  have 2: "(\<turnstile> f ior (big_ior L)) \<Longrightarrow> (\<turnstile> big_ior (f # L))"
    by  (auto simp add: big_ior_defs) fastforce 
 show ?thesis using 1 2 by auto
qed

lemma big_ior_Cons_alt: 
 "\<turnstile> big_ior (f # L) ieqv f ior big_ior  L"
unfolding big_ior_def
by auto



lemma big_ior_1:
 assumes "\<And>i. i< length F \<Longrightarrow>  (\<turnstile> F ! i ieqv G ! i)"
         "length F = length G"
 shows " \<turnstile> (big_ior F) ieqv (big_ior G) " 
using assms 
by (auto simp add: big_ior_defs)


lemma big_iand_1:
 assumes "\<And>i. i< length F \<Longrightarrow>  (\<turnstile> F ! i ieqv G ! i)"
         "length F = length G"
 shows " \<turnstile> (big_iand F) ieqv (big_iand G) " 
using assms 
by (auto simp add: big_iand_defs)


lemma big_ior_big_iand_1: 
 " \<turnstile> (inot ( big_ior F)) ieqv (big_iand (map (\<lambda>x. inot x) F)) " 
by (auto simp add: big_ior_defs big_iand_defs)

lemma big_ior_big_iand_2: 
 " \<turnstile> (inot ( big_iand F)) ieqv (big_ior (map (\<lambda>x. inot x) F)) " 
by (auto simp add: big_ior_defs big_iand_defs)


lemma big_ior_append: 
 " (\<turnstile> (big_ior (L1@L2)) ieqv (big_ior L1) ior (big_ior L2)) " 
proof (induct L1 arbitrary: L2)
case Nil
then show ?case by (simp add: big_ior_defs)
next
case (Cons a L1a)
then show ?case 
   proof -
    have 1: "\<turnstile> big_ior ((a # L1a) @ L2) ieqv a ior ((big_ior (L1a @ L2)))"
    using big_ior_Cons_alt by auto 
    have 2: "\<turnstile> big_ior (L1a @ L2) ieqv big_ior L1a ior big_ior L2 "
       using local.Cons by blast 
    have 3: "\<turnstile> a ior ((big_ior (L1a @ L2))) ieqv a ior (big_ior L1a ior big_ior L2)"
      using 1 2 by auto 
    have 4: "\<turnstile> big_ior (a # L1a) ieqv a ior (big_ior L1a)"
    using big_ior_Cons_alt by blast
    have 5: "\<turnstile>  a ior (big_ior L1a ior big_ior L2) ieqv  big_ior (a # L1a) ior big_ior L2  " 
          using 4 by auto
    show ?thesis using "1" "3" "5" by force
  qed
qed

lemma state_pitl_list_append:  
 shows   "state_pitl_list (L@L1) \<longleftrightarrow> state_pitl_list L \<and> state_pitl_list L1 " 
proof (induct L arbitrary: L1)
case Nil
then show ?case 
by (simp add: state_pitl_list_defs)
next
case (Cons a L)
then show ?case 
   proof -
    have 1: "state_pitl_list ((a # L) @ L1) \<longleftrightarrow> state_pitl a \<and> state_pitl_list (L @ L1)" 
      unfolding state_pitl_list_def by auto
    have 2: "state_pitl_list (L @ L1) \<longleftrightarrow> state_pitl_list L \<and> state_pitl_list L1"
      by (simp add: local.Cons) 
    have 3: "state_pitl a \<and> state_pitl_list L \<and> state_pitl_list L1 \<longleftrightarrow>
             (state_pitl_list (a # L) \<and> state_pitl_list L1)" 
      by (simp add: state_pitl_list_def)
    show ?thesis using "1" "3" local.Cons by auto
   qed
qed


lemma map2_map:
 assumes "length L1 = length L2"
 shows   " map2 f (map g L1) (map h L2) = map2 (\<lambda>x y. f (g x) (h y)) L1 L2"
using assms 
proof (induct L1 L2 rule:list_induct2)
case Nil
then show ?case by simp
next
case (Cons x xs y ys)
then show ?case by auto
qed
 

lemma map2_map_a: 
assumes "length L1 = length L2"
shows "(map2 (\<lambda>x y. x iand next y) (map (\<lambda>x1. x iand x1) L1) (map (\<lambda>y1. y ior y1) L2)) = 
       (map2 (\<lambda>x1 y1. (x iand x1) iand next ( y ior y1)) L1 L2)"
using assms map2_map by blast

lemma map2_map_a_a: 
assumes "length L1 = length L2"
shows "(map2 (\<lambda>x y. x iand next y) 
             (map (\<lambda>x1. x iand x1) L1) 
             (map (\<lambda>y1. (if y = ifalse then y1 else y ior y1)) L2)) = 
       (map2 (\<lambda>x1 y1. (x iand x1) iand next ( (if y = ifalse then y1 else y ior y1))) L1 L2)"
using assms map2_map by blast


lemma map2_map_b: 
assumes "length L1 = length L2"
shows "(map2 (\<lambda>x y. x iand next y) (map (\<lambda>x1. (inot x) iand x1) L1) L2) = 
       (map2 (\<lambda>x1 y1. ((inot x) iand x1) iand next  y1) L1 L2)"
using assms map2_map[of L1 L2 "(\<lambda>x y. x iand next y)" " (\<lambda>x1. (inot x) iand x1)" id] 
by auto


lemma big_ior_big_ior_map2: 
assumes "length L1 = length L2" 
        "\<turnstile> big_ior L1" 
shows   "\<turnstile> big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 )" 
using assms 
proof (induct L1 L2 rule:list_induct2)
case Nil
then show ?case by simp
next
case (Cons x xs y ys)
then show ?case
by (metis fst_def list.size(4) map_eq_conv map_fst_zip) 
qed


lemma big_ior_split_off: 
assumes "length L1 = length L2" 
       "(\<turnstile> big_ior L1)" 
shows "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next (y ior  y1))) L1 L2 )) ior
         (big_ior  (map2 (\<lambda>x1 y1. (inot x iand  x1) iand next  y1)      L1 L2 )) ieqv
         (x iand next y) ior 
         (big_ior  (map2 (\<lambda>x y.  x iand next  y)  L1 L2 )) " 
proof -
 have 1: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next (y ior y1))) L1 L2 )) ieqv
            (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next (y ior y1))) L1 L2 )))" 
        by (auto simp add: big_ior_defs)
 have 2: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next (y ior y1))) L1 L2 )) ieqv
            ((next y iand (big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 ))) 
            ior 
            (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))"
       by (auto simp add: big_ior_defs) 
 have 3: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next (y ior y1))) L1 L2 )) ieqv 
            (x iand  (next y iand big_ior  (map2 (\<lambda>x1 y1.  x1) L1 L2 ) 
             ior 
            (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))))" 
       by (auto simp add: big_ior_defs) 
  have 4: "\<turnstile> big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 )"
         using assms big_ior_big_ior_map2 by blast
  have 5: "\<turnstile> (x iand  (next y iand big_ior  (map2 (\<lambda>x1 y1.  x1 )  L1 L2 ) 
             ior 
             (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))) ieqv
            (x iand next y) ior (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))" 
       using 4 by auto
  have 6: "\<turnstile> (big_ior (map2 (\<lambda>x1 y1. (inot x iand  x1) iand next  y1) L1 L2 )) ieqv
             (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next ( y1))) L1 L2 )))"
     by (auto simp add: big_ior_defs) 
  have 7: "\<turnstile> (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ior 
              (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ieqv
              (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))" 
       by (auto simp add: big_ior_defs) 
  have 8 : "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next ( y ior y1))) L1 L2 )) ior
              (big_ior (map2 (\<lambda>x1 y1. (inot x iand x1) iand next  y1) L1 L2 )) ieqv
              (x iand next y) ior 
              (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ior
              (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) "
          using 5 6 3 by auto
   show ?thesis using 7 8 by auto
qed

lemma big_ior_split_off_a: 
assumes "length L1 = length L2" 
       "(\<turnstile> big_ior L1)" 
shows "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next ((if y = ifalse then y1 else y ior y1)))) L1 L2 )) ior
         (big_ior  (map2 (\<lambda>x1 y1. (inot x iand  x1) iand next  y1)      L1 L2 )) ieqv
         (x iand next y) ior 
         (big_ior  (map2 (\<lambda>x y.  x iand next  y)  L1 L2 )) " 
proof -
 have 1: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next ((if y = ifalse then y1 else y ior y1)))) L1 L2 )) ieqv
            (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next ((if y = ifalse then y1 else y ior y1)))) L1 L2 )))" 
        by (auto simp add: big_ior_defs)
 have 2: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next ((if y = ifalse then y1 else y ior y1)))) L1 L2 )) ieqv
            ((next y iand (big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 ))) 
            ior 
            (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))"
    proof (cases "y=ifalse")
    case True
    then show ?thesis by simp
    next
    case False
    then show ?thesis by (auto simp add: big_ior_defs) 
    qed
 have 3: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next (if y = ifalse then y1 else y ior y1))) L1 L2 )) ieqv 
            (x iand  (next y iand big_ior  (map2 (\<lambda>x1 y1.  x1) L1 L2 ) 
             ior 
            (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))))" 
     proof (cases "y=ifalse")
     case True
     then show ?thesis by (auto simp add: big_ior_defs) 
     next
     case False
     then show ?thesis by (auto simp add: big_ior_defs) 
     qed
  have 4: "\<turnstile> big_ior  (map2 (\<lambda>x1 y1.  x1 ) L1 L2 )"
         using assms big_ior_big_ior_map2 by blast
  have 5: "\<turnstile> (x iand  (next y iand big_ior  (map2 (\<lambda>x1 y1.  x1 )  L1 L2 ) 
             ior 
             (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))) ieqv
            (x iand next y) ior (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 )))" 
       using 4 by auto
  have 6: "\<turnstile> (big_ior (map2 (\<lambda>x1 y1. (inot x iand  x1) iand next  y1) L1 L2 )) ieqv
             (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand ( next ( y1))) L1 L2 )))"
     by (auto simp add: big_ior_defs) 
  have 7: "\<turnstile> (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ior 
              (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ieqv
              (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))" 
       by (auto simp add: big_ior_defs) 
  have 8 : "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand ( next ( if y = ifalse then y1 else y ior y1))) L1 L2 )) ior
              (big_ior (map2 (\<lambda>x1 y1. (inot x iand x1) iand next  y1) L1 L2 )) ieqv
              (x iand next y) ior 
              (x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) ior
              (inot x iand (big_ior  (map2 (\<lambda>x1 y1.  x1 iand next ( y1)) L1 L2 ))) "
          using 5 6 3 by auto
   show ?thesis using 7 8 by auto
qed


lemma state_pitl_list_map:
 assumes "state_pitl f"
         "state_pitl_list L"
 shows " state_pitl_list (map (\<lambda> x. f iand x) L) " 
using assms 
proof (induct L)
case Nil
then show ?case by simp
next
case (Cons a L)
then show ?case 
  proof -
   have 1: "state_pitl_list (map (\<lambda>x. f iand x ) (a # L))  \<longleftrightarrow>
            state_pitl ( f iand a ) \<and>  state_pitl_list ((map (\<lambda>x. f iand x) L)) " 
         unfolding state_pitl_list_def by auto
   have 2: "state_pitl_list (a # L) \<longleftrightarrow> state_pitl a \<and> state_pitl_list L"
      unfolding state_pitl_list_def by auto
   have 3: " state_pitl ( f iand a)" 
      unfolding iand_d_def inot_d_def ior_d_def 
      using "2" Cons.prems(2) assms(1) by auto
   show ?thesis
   using "1" "2" "3" Cons.hyps Cons.prems(2) assms(1) by blast
 qed          
qed

lemma state_pitl_list_map_alt:
 assumes "state_pitl f"
         "state_pitl_list L"
 shows " state_pitl_list (map (\<lambda> x. x iand f) L) " 
using assms 
proof (induct L)
case Nil
then show ?case by simp
next
case (Cons a L)
then show ?case 
  proof -
   have 1: "state_pitl_list (map (\<lambda>x. x iand f ) (a # L))  \<longleftrightarrow>
            state_pitl ( a iand f ) \<and>  state_pitl_list ((map (\<lambda>x. x iand f) L)) " 
         unfolding state_pitl_list_def by auto
   have 2: "state_pitl_list (a # L) \<longleftrightarrow> state_pitl a \<and> state_pitl_list L"
      unfolding state_pitl_list_def by auto
   have 3: " state_pitl ( a iand f)" 
      unfolding iand_d_def inot_d_def ior_d_def 
      using "2" Cons.prems(2) assms(1) by auto
   show ?thesis
   using "1" "2" "3" Cons.hyps Cons.prems(2) assms(1) by blast
 qed          
qed

lemma map_fst_filter_sat: 
assumes "length L1 = length L2" 
shows " map fst (filter (\<lambda> (x,y). (\<exists> \<sigma>. \<sigma> \<Turnstile> x))  (zip L1 L2)) = 
        filter (\<lambda>x. (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) L1"
using assms 
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case by simp
next
case (Cons x xs y ys)
then show ?case 
  proof (cases "xs = [] \<and> ys = [] ")
  case True
  then show ?thesis by simp
  next
  case False
  then show ?thesis 
     using Cons by auto
  qed
qed


lemma big_ior_ieqv_filter_sat: 
 "\<turnstile> big_ior L ieqv big_ior ( filter (\<lambda>x. (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) L)"
proof (induct L)
case Nil
then show ?case by simp
next
case (Cons a L)
then show ?case 
using big_ior_Cons_alt by auto
qed


lemma total_filter_sat: 
 assumes "\<turnstile> big_ior L"
 shows "\<turnstile> big_ior ( filter (\<lambda>x. (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) L)"
using assms 
using big_ior_ieqv_filter_sat by auto


lemma  big_ior_filter_sat_eqv: 
assumes "length L1 = length L2"      
shows "\<turnstile> big_ior ( map (\<lambda> (x,y) . x iand (next  y)) 
                  (filter (\<lambda> (x,y). (\<exists> \<sigma>. \<sigma> \<Turnstile> x))  (zip L1 L2))) 
         ieqv 
         big_ior (map2 (\<lambda> x y. x iand (next y)) L1 L2) "
using assms
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case by (simp add: big_ior_def)
next
case (Cons x xs y ys)
then show ?case 
  proof (cases "xs = [] \<and> ys = [] ")
  case True
  then show ?thesis using Cons 
    unfolding big_ior_def big_iand_def
      by simp
  next
  case False
  then show ?thesis 
    proof -
     have 1: "length xs = length ys"
           by (simp add: Cons.hyps(1))
     have 2: "\<turnstile> big_ior (map (\<lambda>(x, y). x iand next y)
                  (filter (\<lambda>(x, y).   \<exists>\<sigma>. \<sigma> \<Turnstile> x)
                   (zip xs ys))) 
                  ieqv big_ior (map2 (\<lambda>x y. x iand next y) xs ys) " 
         using  Cons.hyps(2) False by fastforce
      show ?thesis 
      using "2" big_ior_Cons_alt by auto  
    qed    
  qed
qed


lemma big_ior_rev_ieqv: 
 "\<turnstile> big_ior L ieqv big_ior (rev L) " 
by (auto simp add: big_ior_defs)
   (metis in_set_conv_nth length_rev set_rev,
    metis diff_less length_greater_0_conv less_nat_zero_code list.size(3) rev_nth zero_less_Suc)

lemma big_ior_map2_append: 
assumes "length L1= length N1"
        "length L2= length N2" 
shows " \<turnstile> (big_ior (map2 (\<lambda> x y . x iand (next y)) (L1@L2) (N1@N2))) ieqv 
          (big_ior (map2 (\<lambda> x y . x iand (next y)) L1 N1)) ior 
          (big_ior (map2 (\<lambda> x y . x iand (next y))  L2 N2)) " 
using assms
proof (induct L1 N1 arbitrary: L2 N2 rule:list_induct2)
case Nil
then show ?case by (simp add: big_ior_defs)
next
case (Cons x xs y ys)
then show ?case 
   by (metis Cons.hyps(1) big_ior_append length_Cons map_append zip_append)       
         
qed

lemma map2_iand_next_defs : 
 assumes "length F = length G"
 shows "(\<exists>i<length G. \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand (next y)) F G ! i) \<longleftrightarrow>
        (\<exists>i<length G. (\<sigma> \<Turnstile>  F !i) \<and> (\<sigma> \<Turnstile> (next (G ! i)))) "
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
  proof (cases "F=[]")
  case True
  then show ?thesis using Cons.prems by auto
  next
  case False
  then show ?thesis using Cons by auto 
  qed
qed 

lemma map2_iand_inot_next_defs : 
 assumes "length F = length G"
         "0<nlength \<sigma>" 
 shows "(\<exists>i<length G. \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand (next (inot y))) F G ! i) \<longleftrightarrow>
        (\<exists>i<length G. (\<sigma> \<Turnstile> F! i) \<and> \<not>(\<sigma> \<Turnstile>  (next (G!i))))" 
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
   proof (cases "F=[]")
   case True
   then show ?thesis using Cons by simp
   next
   case False
   then show ?thesis 
     proof -
      obtain b F1 where 1: "F = b#F1" 
        using False by (meson neq_Nil_conv) 
     have 2: " length F1 = length G" 
          using "1" Cons.prems by auto
     have 3: "(\<exists>i<length G. \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) F1 G ! i) = 
              (\<exists>i<length G. (\<sigma> \<Turnstile> F1 ! i) \<and> \<not> (\<sigma> \<Turnstile> next G ! i))" 
        using "2" Cons.hyps assms(2) by presburger
     have 4: "(\<exists>i<length (a # G). \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) F (a # G) ! i) \<longleftrightarrow>
              (\<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (b#F1) (a # G) ! 0) \<or>
              (\<exists>i. 0 < i \<and> i<length (a # G) \<and> ( \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (b#F1) (a # G) ! i))"
         using 1  by (metis gr_zeroI length_greater_0_conv list.discI)
     have 5: "(\<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (b#F1) (a # G) ! 0) \<longleftrightarrow>
              (\<sigma> \<Turnstile> b iand next inot a)"
       by auto
     have 6: "(\<exists>i. 0 < i \<and> i<length (a # G) \<and> ( \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (b#F1) (a # G) ! i)) \<longleftrightarrow>
              (\<exists>i<length ( G).  ( \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next inot y) (F1) (G) ! i)) " 
        by auto
           (metis Suc_less_eq Suc_pred)
     have 7: "(\<exists>i<length (a # G). (\<sigma> \<Turnstile> (b#F1) ! i) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! i))  \<longleftrightarrow>
              ((\<sigma> \<Turnstile> (b#F1) ! 0) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! 0)) \<or>
              (\<exists>i. 0< i \<and> i<length (a # G) \<and> (\<sigma> \<Turnstile> (b#F1) ! i) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! i)) " 
        by (metis gr_zeroI length_greater_0_conv list.discI)
     have 8: "((\<sigma> \<Turnstile> (b#F1) ! 0) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! 0)) \<longleftrightarrow>
              (\<sigma> \<Turnstile> b iand next inot a)" 
        by simp
         (metis One_nat_def Suc_ile_eq assms(2) enat_defs(1) enat_defs(2))
     have 9: "(\<exists>i. 0< i \<and> i<length (a # G) \<and> (\<sigma> \<Turnstile> (b#F1) ! i) \<and> \<not> (\<sigma> \<Turnstile> next (a # G) ! i)) \<longleftrightarrow>
              (\<exists>i<length ( G). (\<sigma> \<Turnstile> (F1) ! i) \<and> \<not> (\<sigma> \<Turnstile> next ( G) ! i))" 
         by auto
          (metis Suc_diff_Suc Suc_less_eq diff_zero, metis Suc_less_eq Suc_pred) 
     show ?thesis 
     using "1" "3" "4" "5" "6" "7" "8" "9" by presburger
    qed    
   qed
qed

lemma iand_big_ior_dist: 
assumes "length G_now = length G_next"
shows "\<turnstile> F_e iand big_ior (map2 (\<lambda> x y. x iand next y) G_now G_next) ieqv
         big_ior (map2 (\<lambda> x y. x iand next y) 
                        (map (\<lambda>x. x iand F_e) G_now) G_next)"
using assms by (auto simp add: big_ior_defs)


lemma big_ior_map2_defs: 
 assumes "length L1 = length L2"
 shows " (\<sigma> \<Turnstile> (big_ior (map2 (\<lambda>x y. f x y) L1 L2))) \<longleftrightarrow>
         (\<exists>i<length L1. (\<sigma> \<Turnstile> f (L1!i) (L2!i)))"
using assms
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case unfolding big_ior_def by auto
next
case (Cons x xs y ys)
then show ?case unfolding big_ior_def using less_Suc_eq_0_disj by auto
qed

lemma big_ior_map2_schop_dis: 
 assumes "length L1 = length L2"
 shows " \<turnstile> (big_ior (map2 (\<lambda>x y. f x y) L1 L2)) schop g ieqv
           (big_ior (map2 (\<lambda>x y. (f x y) schop g) L1 L2)) " 
using assms 
by (auto simp add: big_ior_map2_defs)





end
