(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                Dimitar P Guelev
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

chapter \<open>NL-ITL\<close>

theory NL_ITL_Deep imports
 Extended_Int NELList_Extras 

begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true, max_proofs = 1,
                    provers="cvc4 z3 e spass vampire " ]
*)
(*
declare [[show_types]]
*)

text \<open>
This theory contains the syntax and semantics of ITL with neighbourhood operators, i.e., 
NL-ITL\cite{GuelevMoszkowski2022}. Note, we have both infinite past and infinite future, 
therefore we need the theory of extended integers. 
\<close>

section \<open>Syntax\<close>

subsection \<open>Primitive formulae\<close>

datatype  nl_itl =
    false_d                            ("ifalse")
  | atom_d "nat "                      ("($_)" [100] 99) 
  | iimp_d " nl_itl" " nl_itl"         ("( _ iimp _ )" [26,25] 25)
  | next_d " nl_itl"                   ("(next _)"  [88] 87)
  | chop_d " nl_itl" " nl_itl"         ("( _ schop _ )" [84,84] 83)
  | ldiamond_d " nl_itl"               ("(ldiamond _)" [88] 87)
  | rdiamond_d " nl_itl"               ("(rdiamond _)" [88] 87) 
  | chopstar_d " nl_itl"               ("(chopstar _ )" [85] 85)
  | exists_d "nat" "nl_itl"            ( "Ex _. _" 10)

(*
print_theorems
*)

datatype  introspective_nl_itl =
    cfalse_d                                                   ("cifalse")
  | catom_d "nat"                                              ("($$_)" [100] 99) 
  | ciimp_d "introspective_nl_itl" "introspective_nl_itl"      ("( _ ciimp _ )" [26,25] 25)
  | cnext_d "introspective_nl_itl"                             ("(cnext _)"  [88] 87)
  | cchop_d "introspective_nl_itl" "introspective_nl_itl"      ("( _ cschop _ )" [84,84] 83)
  | cchopstar_d "introspective_nl_itl"                         ("(cchopstar _ )" [85] 85)
  | cexists_d "nat" "introspective_nl_itl"                     ( "cEx _. _" 10)

datatype future_nl_itl =
  fintro_d "introspective_nl_itl"          ("(fintro _)" [88] 87) 
| finot_d "future_nl_itl"                  ("(finot _)" [90] 90)
| fior_d  "future_nl_itl" "future_nl_itl"  ("(_ fior _)"  [31,30] 30)
| frdiamond_d "future_nl_itl"              ("(frdiamond _)" [88] 87) 

datatype stricly_future_nl_itl =
 sfrdiamond_d "future_nl_itl"              ("(sfrdiamond _)" [88] 87) 

datatype past_nl_itl =
  pintro_d "introspective_nl_itl"           ("(pintro _)" [88] 87) 
| pinot_d "past_nl_itl"                     ("(pinot _)" [90] 90)
| pior_d  "past_nl_itl" "past_nl_itl"       ("(_ pior _)"  [31,30] 30)
| pldiamond_d "past_nl_itl"                 ("(pldiamond _)" [88] 87) 

datatype strictly_past_nl_itl =
 spldiamond_d "past_nl_itl"                 ("(spldiamond _)" [88] 87) 

datatype separated_nl_itl =
  past_d "past_nl_itl"                             ("(past _)" [88] 87) 
| future_d "future_nl_itl"                         ("(future _)" [88] 87) 
| sinot_d "separated_nl_itl"                       ("(sinot _)" [90] 90)
| sior_d  "separated_nl_itl" "separated_nl_itl"    ("(_ sior _)"  [31,30] 30)            



subsection \<open>state, introspective, future and past formula\<close>

fun  state_nl_itl :: "nl_itl \<Rightarrow> bool" 
where
  "state_nl_itl (ifalse)     = True"
| "state_nl_itl ($n)         = True "
| "state_nl_itl (f iimp g)   = ((state_nl_itl f) \<and> (state_nl_itl g))"
| "state_nl_itl (next f)     = False "
| "state_nl_itl (f schop g)  = False "
| "state_nl_itl (ldiamond f) = False " 
| "state_nl_itl (rdiamond f) = False " 
| "state_nl_itl (chopstar f) = False "
| "state_nl_itl (Ex n. f)    =  state_nl_itl f"  


fun  introspective_nl_itl_prop :: " nl_itl \<Rightarrow> bool" 
where
  "introspective_nl_itl_prop (ifalse)     = True"
| "introspective_nl_itl_prop ($n)         = True "
| "introspective_nl_itl_prop (f iimp g)   = ((introspective_nl_itl_prop f) \<and> (introspective_nl_itl_prop g))"
| "introspective_nl_itl_prop (next f)     = introspective_nl_itl_prop f "
| "introspective_nl_itl_prop (f schop g)  = ((introspective_nl_itl_prop f) \<and> (introspective_nl_itl_prop g)) "
| "introspective_nl_itl_prop (ldiamond f) = False " 
| "introspective_nl_itl_prop (rdiamond f) = False " 
| "introspective_nl_itl_prop (chopstar f) = introspective_nl_itl_prop f "
| "introspective_nl_itl_prop (Ex n. f)    = introspective_nl_itl_prop f" 



fun  future_nl_itl_prop :: " nl_itl \<Rightarrow> bool" 
where
  "future_nl_itl_prop (ifalse)     = True"
| "future_nl_itl_prop ($n)         = True "
| "future_nl_itl_prop (f iimp g)   = ((future_nl_itl_prop f) \<and> (future_nl_itl_prop g))"
| "future_nl_itl_prop (next f)     = future_nl_itl_prop f "
| "future_nl_itl_prop (f schop g)  = ((future_nl_itl_prop f) \<and> (future_nl_itl_prop g)) "
| "future_nl_itl_prop (ldiamond f) = False " 
| "future_nl_itl_prop (rdiamond f) = future_nl_itl_prop f " 
| "future_nl_itl_prop (chopstar f) = future_nl_itl_prop f "
| "future_nl_itl_prop (Ex n. f)    = future_nl_itl_prop f" 



fun  past_nl_itl_prop :: " nl_itl \<Rightarrow> bool" 
where
  "past_nl_itl_prop (ifalse)     = True"
| "past_nl_itl_prop ($n)         = True "
| "past_nl_itl_prop (f iimp g)   = ((past_nl_itl_prop f) \<and> (past_nl_itl_prop g))"
| "past_nl_itl_prop (next f)     = past_nl_itl_prop f "
| "past_nl_itl_prop (f schop g)  = ((past_nl_itl_prop f) \<and> (past_nl_itl_prop g)) "
| "past_nl_itl_prop (ldiamond f) = (past_nl_itl_prop f) " 
| "past_nl_itl_prop (rdiamond f) = False " 
| "past_nl_itl_prop (chopstar f) = past_nl_itl_prop f "
| "past_nl_itl_prop (Ex n. f)    =  past_nl_itl_prop f" 



subsection \<open>Derived  Operators\<close>

definition inot_d ("(inot _)" [90] 90) 
where 
  "inot f \<equiv> f iimp ifalse"

definition cinot_d ("(cinot _)" [90] 90) 
where 
  "cinot f \<equiv> f ciimp cifalse"

definition itrue_d ("itrue") 
where 
  "itrue \<equiv> inot  ifalse"

definition citrue_d ("citrue") 
where 
  "citrue \<equiv> cinot  cifalse"

definition ior_d ("(_ ior _)" [31,30] 30) 
where 
  "f ior g \<equiv> (inot f) iimp g"

definition cior_d ("(_ cior _)" [31,30] 30) 
where 
  "f cior g \<equiv> (cinot f) ciimp g"

definition iand_d ("(_ iand _)" [36,35] 35)
where 
  "f iand g \<equiv> inot (inot f ior inot g)"

definition ciand_d ("(_ ciand _)" [36,35] 35)
where 
  "f ciand g \<equiv> cinot (cinot f cior cinot g)"


definition ieqv_d ( "(_ ieqv _) " [21,20] 20)
where 
  "f ieqv g \<equiv> ( ( f iimp g) iand  (g iimp f))" 

definition cieqv_d ( "(_ cieqv _) " [21,20] 20)
where 
  "f cieqv g \<equiv> ( ( f ciimp g) ciand  (g ciimp f))" 
 


subsection \<open>more, empty, skip and wnext \<close>

definition more_d ("more") 
where 
  "more \<equiv> itrue schop (next itrue)"

definition cmore_d ("cmore") 
where 
  "cmore \<equiv> citrue cschop (cnext citrue)"

definition empty_d ("empty")
where
 "empty \<equiv> inot more"

definition cempty_d ("cempty")
where
 "cempty \<equiv> cinot cmore"

definition skip_d ("skip") 
where 
  "skip \<equiv> next empty"

definition cskip_d ("cskip") 
where 
  "cskip \<equiv> cnext cempty"


definition wnext_d ("(wnext _)" [88] 87)
where
 "wnext f \<equiv> inot (next (inot f))" 

definition cwnext_d ("(cwnext _)" [88] 87)
where
 "cwnext f \<equiv> cinot (cnext (cinot f))" 


definition prev_d ("(prev _)" [88] 87)
where 
 "prev f \<equiv> f schop skip" 

definition cprev_d ("(cprev _)" [88] 87)
where 
 "cprev f \<equiv> f cschop cskip" 

definition wprev_d ("(wprev _)" [88] 87)
where 
 "wprev f \<equiv> inot( prev (inot f))" 

definition cwprev_d ("(cwprev _)" [88] 87)
where 
 "cwprev f \<equiv> cinot( cprev (cinot f)) " 

subsection \<open>rfinite, lfinite, rinf and linf\<close>

definition rfinite_d ("rfinite") 
where 
  "rfinite \<equiv> itrue schop empty"

definition crfinite_d ("crfinite") 
where 
  "crfinite \<equiv> citrue cschop cempty"

definition rinf_d ("rinf") 
where 
  "rinf \<equiv> inot rfinite"

definition crinf_d ("crinf") 
where 
  "crinf \<equiv> cinot crfinite"

definition lfinite_d ("lfinite") 
where 
  "lfinite \<equiv> (next itrue) ior empty"

definition clfinite_d ("clfinite") 
where 
  "clfinite \<equiv> (cnext citrue) cior cempty"

definition linf_d ("linf") 
where 
  "linf \<equiv> inot lfinite"

definition clinf_d ("clinf") 
where 
  "clinf \<equiv> cinot clfinite"


subsection \<open>weak chop\<close>

definition wchop_d ("( _ wchop _ )" [84,84] 83)
 where " f wchop g \<equiv> (f schop g) ior (f iand rinf)" 

definition cwchop_d ("( _ cwchop _ )" [84,84] 83)
 where " f cwchop g \<equiv> (f cschop g) cior (f ciand crinf)" 

subsection \<open>Box and Diamond Operators\<close>

definition sometimes_d ("(diamond _)" [88] 87)
where 
  "diamond f \<equiv> itrue schop f"

definition csometimes_d ("(cdiamond _)" [88] 87)
where 
  "cdiamond f \<equiv> citrue cschop f"


definition always_d ("(box _)" [88] 87)
where 
  "box f \<equiv> inot (diamond (inot f))"

definition calways_d ("(cbox _)" [88] 87)
where 
  "cbox f \<equiv> cinot (cdiamond (cinot f))"


definition di_d ("(di _)" [88] 87)
where 
  "di f \<equiv> f schop itrue"

definition cdi_d ("(cdi _)" [88] 87)
where 
  "cdi f \<equiv> f cschop citrue"

definition bi_d ("(bi _)" [88] 87)
where 
  "bi f \<equiv> inot (di (inot f))"

definition cbi_d ("(cbi _)" [88] 87)
where 
  "cbi f \<equiv> cinot (cdi (cinot f))"


subsection \<open>Initial and Final Operators\<close>

definition init_d ("(init _)" [88] 87)
where
 " init f \<equiv> (empty iand f) schop itrue"

definition cinit_d ("(cinit _)" [88] 87)
where
 " cinit f \<equiv> (cempty ciand f) cschop citrue"


definition fin_d ("(fin _)" [88] 87)
where
  "fin f \<equiv> itrue schop (empty iand f)"


definition cfin_d ("(cfin _)" [88] 87)
where
  "cfin f \<equiv> citrue cschop (cempty ciand f)"

subsection \<open>Forall\<close>

definition forall_d ("Fa _. _" 10)
where
 " Fa v. f \<equiv> inot (Ex v. inot f) " 


definition cforall_d ("cFa _. _" 10)
where
 " cFa v. f \<equiv> cinot (cEx v. cinot f) " 


subsection \<open> Big operations\<close>

definition big_ior :: "  nl_itl list \<Rightarrow>  nl_itl "
 where " big_ior F = foldr (\<lambda> x y. x ior y) F ifalse " 

definition big_cior :: "  introspective_nl_itl list \<Rightarrow>  introspective_nl_itl "
 where " big_cior F = foldr (\<lambda> x y. x cior y) F cifalse " 

definition big_iand :: "  nl_itl list \<Rightarrow>  nl_itl "
 where " big_iand F = foldr (\<lambda> x y. x iand y) F itrue " 

definition big_ciand :: "  introspective_nl_itl list \<Rightarrow>  introspective_nl_itl "
 where " big_ciand F = foldr (\<lambda> x y. x ciand y) F citrue " 


definition state_nl_itl_list :: " nl_itl list \<Rightarrow> bool"
 where " state_nl_itl_list L = foldr (\<lambda>x y. state_nl_itl x \<and> y) L True " 

section \<open>Semantics\<close>


subsection \<open>Intervals\<close>

type_synonym  interval = " int \<Rightarrow> nat set " 

subsection \<open>Semantics Primitive Operators\<close>

definition similar :: " interval \<Rightarrow> nat \<Rightarrow> interval \<Rightarrow> bool " ("(_ \<sim>\<^sub>_ _)" )   
where "similar \<sigma>0 n \<sigma>1 = (\<forall> k p . (p \<in> (\<sigma>0 k) \<and> p \<noteq> n) \<longleftrightarrow> (p \<in> (\<sigma>1 k) \<and> p \<noteq> n) ) " 

definition csimilar :: " interval \<Rightarrow> nat \<Rightarrow>  eint \<Rightarrow> eint \<Rightarrow> interval \<Rightarrow> bool "    
where "csimilar \<sigma>0 n i j \<sigma>1 = 
       (\<forall> k . if i \<le> k \<and> k \<le> j then (\<forall>p. (p \<in> (\<sigma>0 k) \<and> p \<noteq> n) \<longleftrightarrow> (p \<in> (\<sigma>1 k) \<and> p \<noteq> n))
                else (\<forall> p. (p \<in> (\<sigma>0 k)) \<longleftrightarrow> (p \<in> (\<sigma>1 k))) ) " 

definition similars :: " interval \<Rightarrow> nat set \<Rightarrow> interval \<Rightarrow> bool "  ("(_ \<approx>\<^sub>_ _)" )  
where "similars \<sigma>0 A \<sigma>1 = (\<forall> k p. p \<notin> A \<longrightarrow> (p \<in> (\<sigma>0 k) \<longleftrightarrow> p \<in> (\<sigma>1 k))) " 

definition upd_add_sigma :: "interval \<Rightarrow> int \<Rightarrow> nat \<Rightarrow> interval "
where " upd_add_sigma \<sigma> k n = \<sigma>(k := insert n (\<sigma> k)) " 

definition upd_remove_sigma :: "interval \<Rightarrow> int \<Rightarrow> nat set \<Rightarrow> interval "
where " upd_remove_sigma \<sigma> k A = \<sigma>(k :=  (\<sigma> k) - A ) " 

definition upd_remove_all_sigma :: "interval \<Rightarrow> nat set \<Rightarrow> interval "
where "upd_remove_all_sigma \<sigma> A = (\<lambda> i. (upd_remove_sigma \<sigma> i A) i) "  

primrec fvars :: "nl_itl \<Rightarrow> nat set"
where 
  "fvars ifalse       = {} "
| "fvars ($n)         = {n} "
| "fvars (f iimp g)   = (fvars f) \<union> (fvars g) "
| "fvars (next f)     = fvars f "
| "fvars (f schop g)  = (fvars f) \<union> (fvars g) "
| "fvars (ldiamond f) = fvars f "
| "fvars (rdiamond f) = fvars f" 
| "fvars (chopstar f) = fvars f"
| "fvars (Ex n. f)    = (fvars f) - {n} " 

primrec cfvars :: "introspective_nl_itl \<Rightarrow> nat set"
where 
  "cfvars cifalse       = {} "
| "cfvars ($$n)         = {n} "
| "cfvars (f ciimp g)   = (cfvars f) \<union> (cfvars g) "
| "cfvars (cnext f)     = cfvars f "
| "cfvars (f cschop g)  = (cfvars f) \<union> (cfvars g) "
| "cfvars (cchopstar f) = cfvars f"
| "cfvars (cEx n. f)    = (cfvars f) - {n} " 

primrec bvars :: "nl_itl \<Rightarrow> nat set"
where 
  "bvars ifalse       = {} "
| "bvars ($n)         = {} "
| "bvars (f iimp g)   = (bvars f) \<union> (bvars g) "
| "bvars (next f)     = bvars f "
| "bvars (f schop g)  = (bvars f) \<union> (bvars g) "
| "bvars (ldiamond f) = bvars f "
| "bvars (rdiamond f) = bvars f" 
| "bvars (chopstar f) = bvars f"
| "bvars (Ex n. f)    = (insert n (bvars f)) " 

primrec cbvars :: "introspective_nl_itl \<Rightarrow> nat set"
where 
  "cbvars cifalse       = {} "
| "cbvars ($$n)         = {} "
| "cbvars (f ciimp g)   = (cbvars f) \<union> (cbvars g) "
| "cbvars (cnext f)     = cbvars f "
| "cbvars (f cschop g)  = (cbvars f) \<union> (cbvars g) "
| "cbvars (cchopstar f) = cbvars f"
| "cbvars (cEx n. f)    = (insert n (cbvars f)) " 

primrec suform :: "nl_itl \<Rightarrow> nat \<Rightarrow> nl_itl \<Rightarrow> nl_itl "
where 
  " suform ifalse       n h = ifalse " 
| " suform ($k)         n h = (if k = n then h else ($k)) "
| " suform (f iimp g)   n h = ((suform f n h) iimp (suform g n h)) " 
| " suform (next f)     n h = (next (suform f n h)) " 
| " suform (f schop g)  n h = ((suform f n h) schop (suform g n h)) " 
| " suform (ldiamond f) n h = (ldiamond (suform f n h)) " 
| " suform (rdiamond f) n h = (rdiamond (suform f n h)) " 
| " suform (chopstar f) n h = (chopstar (suform f n h)) " 
| " suform (Ex k. f)    n h = (Ex k. (suform f n h)) " 
  

primrec csuform :: "introspective_nl_itl \<Rightarrow> nat \<Rightarrow> introspective_nl_itl \<Rightarrow> introspective_nl_itl "
where 
  " csuform cifalse       n h = cifalse " 
| " csuform ($$k)         n h = (if k = n then h else ($$k)) "
| " csuform (f ciimp g)   n h = ((csuform f n h) ciimp (csuform g n h)) " 
| " csuform (cnext f)     n h = (cnext (csuform f n h)) " 
| " csuform (f cschop g)  n h = ((csuform f n h) cschop (csuform g n h)) " 
| " csuform (cchopstar f) n h = (cchopstar (csuform f n h)) " 
| " csuform (cEx k. f)    n h = (cEx k. (csuform f n h)) " 
  

primrec crename ::  "introspective_nl_itl \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> introspective_nl_itl" 
where 
   " crename cifalse n k       = cifalse "
|  " crename ($$m) n k         = (if m = n then $$k else $$m) "
|  " crename (f ciimp g) n k   = ((crename f n k) ciimp (crename g n k)) " 
|  " crename (cnext f) n k     = (cnext (crename f n k)) "
|  " crename (f cschop g) n k  = ((crename f n k) cschop (crename g n k)) "
|  " crename (cchopstar f) n k = (cchopstar (crename f n k)) " 
|  " crename (cEx m. f) n k    = (cEx (if m = n then k else m). (crename f n k)) "

primrec rename ::  "nl_itl \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> nl_itl" 
where 
   " rename ifalse n k       = ifalse "
|  " rename ($m) n k         = (if m = n then $k else $m) "
|  " rename (f iimp g) n k   = ((rename f n k) iimp (rename g n k)) " 
|  " rename (next f) n k     = (next (rename f n k)) "
|  " rename (f schop g) n k  = ((rename f n k) schop (rename g n k)) "
|  " rename (ldiamond f) n k = (ldiamond (rename f n k)) "
|  " rename (rdiamond f) n k = (rdiamond (rename f n k)) "
|  " rename (chopstar f) n k = (chopstar (rename f n k)) " 
|  " rename (Ex m. f) n k    = (Ex (if m = n then k else m). (rename f n k)) "

primrec max_var :: "nl_itl \<Rightarrow> nat \<Rightarrow> nat" 
where 
   " max_var ifalse m       = m  "
|  " max_var ($k) m         = (if m < k then k else m) " 
|  " max_var (f iimp g) m   = (if (max_var f m) < (max_var g m) then (max_var g m) else (max_var f m)) " 
|  " max_var (next f) m     = max_var f m "
|  " max_var (f schop g) m  = (if (max_var f m) < (max_var g m) then (max_var g m) else (max_var f m)) "
|  " max_var (ldiamond f) m = max_var f m "
|  " max_var (rdiamond f) m = max_var f m "
|  " max_var (chopstar f) m = max_var f m "
|  " max_var (Ex k. f) m    = (if (max_var f m) < k then k else (max_var f m)) "  


primrec cmax_var :: "introspective_nl_itl \<Rightarrow> nat \<Rightarrow> nat" 
where 
   " cmax_var cifalse m       = m  "
|  " cmax_var ($$k) m         = (if m < k then k else m) " 
|  " cmax_var (f ciimp g) m   = (if (cmax_var f m) < (cmax_var g m) then (cmax_var g m) else (cmax_var f m)) " 
|  " cmax_var (cnext f) m     = cmax_var f m "
|  " cmax_var (f cschop g) m  = (if (cmax_var f m) < (cmax_var g m) then (cmax_var g m) else (cmax_var f m)) "
|  " cmax_var (cchopstar f) m = cmax_var f m "
|  " cmax_var (cEx k. f) m    = (if (cmax_var f m) < k then k else (cmax_var f m)) "  



definition upd :: " interval \<Rightarrow> nat \<Rightarrow> (int \<Rightarrow> bool) \<Rightarrow> interval" 
where " upd \<sigma> n l = (\<lambda> k. if (l k) then insert n (\<sigma> k) else ((\<sigma> k) - {n})) " 





fun semantics_nl_itl :: " interval \<Rightarrow> eint \<Rightarrow> eint \<Rightarrow>  nl_itl \<Rightarrow> bool" 
  ("(_ _ _ \<Turnstile> _)" [80,80,80,10] 10)
where
  "( \<sigma> i j \<Turnstile> ifalse)      = False"
| "( \<sigma> i j \<Turnstile> ($p))        = (i \<noteq> -\<infinity> \<and> (p \<in> (\<sigma> (int_of_eint i))))"
| "( \<sigma> i j \<Turnstile> (f iimp g))  = ( ( \<sigma> i j \<Turnstile>  f) \<longrightarrow> ( \<sigma> i j \<Turnstile> g) )"
| "( \<sigma> i j \<Turnstile> (next f))    = ( i \<noteq> -\<infinity> \<and> i < j \<and> ( \<sigma> (i+ (eint 1) ) j \<Turnstile> f))"
| "( \<sigma> i j \<Turnstile> (f schop g)) = (\<exists>k.  i \<le> eint k \<and> eint k \<le> j \<and>  ( \<sigma> i k \<Turnstile>  f) \<and> ( \<sigma> k j \<Turnstile> g)) "
| "( \<sigma> i j \<Turnstile> ldiamond f)  = ( i > -\<infinity> \<and> (\<exists>k. k \<le> i \<and> (\<sigma> k i \<Turnstile> f))) "
| "( \<sigma> i j \<Turnstile> rdiamond f)  = ( j <  \<infinity> \<and> (\<exists>k.  k \<ge> j \<and> (\<sigma> j k \<Turnstile> f))) "  
| "( \<sigma> i j \<Turnstile> (chopstar f)) = 
   (i = j \<or> 
    ( i\<noteq>j \<and> i \<noteq> -\<infinity> \<and> j \<noteq> \<infinity> \<and> 
      (\<exists> (lrf:: eint nellist). 
       (nnth lrf 0) = i \<and> nfinite lrf \<and> nlast lrf = j \<and>  
        nridx (\<lambda> a b. a < b \<and>(\<sigma> a b \<Turnstile> f)) lrf
      )
    ) \<or>
    ( i\<noteq>j \<and> i = -\<infinity> \<and> j \<noteq> \<infinity> \<and> 
      (\<exists> (li :: eint nellist). 
       (nnth li 0) = j \<and> \<not> nfinite li  \<and>  
        nridx (\<lambda> a b. b < a \<and>(\<sigma> b a \<Turnstile> f)) li 
      )
    ) \<or>
    ( i\<noteq>j \<and> i \<noteq> -\<infinity> \<and> j = \<infinity> \<and> 
      (\<exists> (ri :: eint nellist). 
       (nnth ri 0) = i \<and> \<not> nfinite ri  \<and>  
       nridx (\<lambda> a b. a < b \<and> (\<sigma> a b \<Turnstile> f)) ri
      )
    ) \<or>
    ( i\<noteq>j \<and> i = -\<infinity> \<and> j = \<infinity> \<and> 
      (\<exists> (li :: int nellist) (ri :: int nellist) . 
         (nnth li 0) = (nnth ri 0) \<and> 
        \<not> nfinite li  \<and>  nridx (\<lambda> a b. b < a \<and> (\<sigma> b a \<Turnstile> f)) li \<and>
        \<not> nfinite ri  \<and>  nridx (\<lambda> a b. a < b \<and> (\<sigma> a b \<Turnstile> f)) ri 
      )
    )
   ) 
  "
| " (\<sigma> i j \<Turnstile> Ex n. f) = (\<exists> l. ((upd \<sigma> n l) i j \<Turnstile> f)  ) "



fun semantics_introspective_nl_itl :: " interval \<Rightarrow> eint \<Rightarrow> eint \<Rightarrow>  introspective_nl_itl \<Rightarrow> bool" 
  ("(_ _ _ c\<Turnstile> _)" [80,80,80,10] 10)
where
  "( \<sigma> i j c\<Turnstile> cifalse)      = False"
| "( \<sigma> i j c\<Turnstile> ($$p))        = (i \<noteq> -\<infinity> \<and> (p \<in> (\<sigma> (int_of_eint i))))"
| "( \<sigma> i j c\<Turnstile> (f ciimp g))  = ( ( \<sigma> i j c\<Turnstile>  f) \<longrightarrow> ( \<sigma> i j c\<Turnstile> g) )"
| "( \<sigma> i j c\<Turnstile> (cnext f))    = ( i \<noteq> -\<infinity> \<and> i < j \<and> ( \<sigma> (i+ (eint 1) ) j c\<Turnstile> f))"
| "( \<sigma> i j c\<Turnstile> (f cschop g)) = (\<exists>k.  i \<le> eint k \<and> eint k \<le> j \<and>  ( \<sigma> i k c\<Turnstile>  f) \<and> ( \<sigma> k j c\<Turnstile> g)) "
| "( \<sigma> i j c\<Turnstile> (cchopstar f)) = 
   (i = j \<or> 
    ( i\<noteq>j \<and> i \<noteq> -\<infinity> \<and> j \<noteq> \<infinity> \<and> 
      (\<exists> (lrf:: eint nellist). 
       (nnth lrf 0) = i \<and> nfinite lrf \<and> nlast lrf = j \<and>  
        nridx (\<lambda> a b. a < b \<and> (\<sigma> a b c\<Turnstile> f)) lrf
      )
    ) \<or>
    ( i\<noteq>j \<and> i = -\<infinity> \<and> j \<noteq> \<infinity> \<and> 
      (\<exists> (li :: eint nellist). 
       (nnth li 0) = j \<and> \<not> nfinite li  \<and>  
        nridx (\<lambda> a b. b < a \<and> (\<sigma> b a c\<Turnstile> f)) li
      )
    ) \<or>
    ( i\<noteq>j \<and> i \<noteq> -\<infinity> \<and> j = \<infinity> \<and> 
      (\<exists> (ri :: eint nellist). 
       (nnth ri 0) = i \<and> \<not> nfinite ri  \<and>  
        nridx (\<lambda> a b. a < b \<and> (\<sigma> a b c\<Turnstile> f)) ri
      )
    ) \<or>
    ( i\<noteq>j \<and> i = -\<infinity> \<and> j = \<infinity> \<and> 
      (\<exists> (li :: int nellist) (ri :: int nellist) . 
         (nnth li 0) = (nnth ri 0) \<and> 
        \<not> nfinite li  \<and> nridx (\<lambda> a b. b < a \<and> (\<sigma> b a c\<Turnstile> f)) li \<and>
        \<not> nfinite ri  \<and> nridx (\<lambda> a b. a < b \<and> (\<sigma> a b c\<Turnstile> f)) ri 
      )
    )
   ) 
  "
| " (\<sigma> i j c\<Turnstile> cEx n. f) = (\<exists> l. ((upd \<sigma> n l) i j c\<Turnstile> f)  ) "


fun semantics_future_nl_itl :: " interval \<Rightarrow> eint \<Rightarrow> eint \<Rightarrow>  future_nl_itl \<Rightarrow> bool" 
  ("(_ _ _ f\<Turnstile> _)" [80,80,80,10] 10)
where
  "( \<sigma> i j f\<Turnstile> fintro f)     =  (\<sigma> i j c\<Turnstile> f)"
| "( \<sigma> i j f\<Turnstile> finot f)      = (\<not> (\<sigma> i j f\<Turnstile> f) )"
| "( \<sigma> i j f\<Turnstile> (f fior g))   = ( ( \<sigma> i j f\<Turnstile>  f) \<or> ( \<sigma> i j f\<Turnstile> g) )"
| "( \<sigma> i j f\<Turnstile> frdiamond f)  = ( j <  \<infinity> \<and> (\<exists>k.  k \<ge> j \<and> (\<sigma> j k f\<Turnstile> f))) "  


fun semantics_strictly_future_nl_itl :: " interval \<Rightarrow> eint \<Rightarrow> eint \<Rightarrow>  stricly_future_nl_itl \<Rightarrow> bool" 
  ("(_ _ _ sf\<Turnstile> _)" [80,80,80,10] 10)
where
 "(\<sigma> i j sf\<Turnstile> sfrdiamond f) = ( j <  \<infinity> \<and> (\<exists>k. k = j \<or> (k > j \<and> (\<sigma> j k f\<Turnstile> f)))) " 


fun semantics_past_nl_itl :: " interval \<Rightarrow> eint \<Rightarrow> eint \<Rightarrow>  past_nl_itl \<Rightarrow> bool" 
  ("(_ _ _ p\<Turnstile> _)" [80,80,80,10] 10)
where
  "( \<sigma> i j p\<Turnstile> pintro f)     =  (\<sigma> i j c\<Turnstile> f)"
| "( \<sigma> i j p\<Turnstile> pinot f)      = (\<not> (\<sigma> i j p\<Turnstile> f) )"
| "( \<sigma> i j p\<Turnstile> (f pior g))   = ( ( \<sigma> i j p\<Turnstile>  f) \<or> ( \<sigma> i j p\<Turnstile> g) )"
| "( \<sigma> i j p\<Turnstile> pldiamond f)  = ( i > - \<infinity> \<and> (\<exists>k.  k \<le> i \<and> (\<sigma> k i p\<Turnstile> f))) "  

fun semantics_stricly_past_nl_itl :: " interval \<Rightarrow> eint \<Rightarrow> eint \<Rightarrow>  strictly_past_nl_itl \<Rightarrow> bool" 
  ("(_ _ _ sp\<Turnstile> _)" [80,80,80,10] 10)
where
 "(\<sigma> i j sp\<Turnstile> spldiamond f) = ( i > -\<infinity> \<and> (\<exists>k.  k = i \<or> (k < i \<and> (\<sigma> k i p\<Turnstile> f)))) " 


fun semantics_separated_nl_itl :: " interval \<Rightarrow> eint \<Rightarrow> eint \<Rightarrow>  separated_nl_itl \<Rightarrow> bool" 
  ("(_ _ _ s\<Turnstile> _)" [80,80,80,10] 10)
where
  "( \<sigma> i j s\<Turnstile> past f)       =  (\<sigma> i j p\<Turnstile> f)"
| "( \<sigma> i j s\<Turnstile> future f)     =  (\<sigma> i j f\<Turnstile> f)"
| "( \<sigma> i j s\<Turnstile> sinot f)      = (\<not> (\<sigma> i j s\<Turnstile> f) )"
| "( \<sigma> i j s\<Turnstile> (f sior g))   = ( ( \<sigma> i j s\<Turnstile>  f) \<or> ( \<sigma> i j s\<Turnstile> g) )"



lemma similar_defs : 
 "  similar \<sigma>0 n \<sigma>1  \<longleftrightarrow>  (\<forall> k p . (p \<in> (\<sigma>0 k) \<and> p \<noteq> n) \<longleftrightarrow> (p \<in> (\<sigma>1 k) \<and> p \<noteq> n))   "
unfolding similar_def by auto

lemma csimilar_defs [simp]: 
 "  csimilar \<sigma>0 n i j \<sigma>1  \<longleftrightarrow>  
   (\<forall> k p. if i\<le>k \<and> k\<le>j then (p \<in> (\<sigma>0 k) \<and> p \<noteq> n) \<longleftrightarrow> (p \<in> (\<sigma>1 k) \<and> p \<noteq> n)
           else (p \<in> (\<sigma>0 k)) \<longleftrightarrow> (p \<in> (\<sigma>1 k)) )   "
unfolding csimilar_def by auto

lemma similar_defs_alt: 
 " (\<forall> k . (\<sigma>0 k) - {n} = (\<sigma>1 k) - {n}) \<longleftrightarrow> similar \<sigma>0 n \<sigma>1   "
unfolding similar_def by auto

lemma similars_defs [simp]:
 "similars \<sigma>0 A \<sigma>1  \<longleftrightarrow>  (\<forall> k p. p \<notin> A \<longrightarrow> ( p \<in> (\<sigma>0 k) \<longleftrightarrow> p \<in> (\<sigma>1 k))) " 
unfolding similars_def by auto


lemma similars_defs_alt:
 "similars \<sigma>0 A \<sigma>1 \<longleftrightarrow>  (\<forall> k. (\<sigma>0 k) - A = (\<sigma>1 k) - A)  " 
unfolding similars_def by auto

lemma similar_similars: 
 " similar \<sigma>0 n \<sigma>1 = similars \<sigma>0 {n} \<sigma>1" 
using similar_defs_alt similars_defs_alt by presburger


lemma similars_mono: 
 assumes "A \<subseteq> B "
         "similars \<sigma>0 A \<sigma>1"
 shows " similars \<sigma>0 B \<sigma>1" 
using assms by auto


lemma similar_refl:
 "similar \<sigma> n \<sigma> " 
using similar_defs_alt by blast


lemma csimilar_refl:
 "csimilar \<sigma> n i j \<sigma> " 
by simp

lemma similar_commute: 
 "similar \<sigma> n \<sigma>'  = similar \<sigma>' n \<sigma>  " 
by (metis similar_defs_alt)

lemma csimilar_commute: 
 "csimilar \<sigma> n i j \<sigma>'  = csimilar \<sigma>' n i j \<sigma>  " 
by simp blast

lemma similar_trans:
 assumes " similar \<sigma> n \<sigma>'  "
         " similar \<sigma>' n \<sigma>'' "
 shows   " similar \<sigma> n \<sigma>'' "
using assms
by (metis similar_defs_alt)

lemma csimilar_trans:
 assumes " csimilar \<sigma> n i j \<sigma>'  "
         " csimilar \<sigma>' n i j \<sigma>'' "
 shows   " csimilar \<sigma> n i j \<sigma>'' "
using assms by simp  



lemma similars_refl:
 "similars \<sigma> A \<sigma> " 
by simp

lemma similars_commute: 
 "similars \<sigma> A \<sigma>'  = similars \<sigma>' A \<sigma>  " 
by simp blast

lemma similars_trans:
 assumes " similars \<sigma> A \<sigma>'  "
         " similars \<sigma>' A \<sigma>'' "
 shows   " similars \<sigma> A \<sigma>'' "
using assms by simp  



lemma upd_add_sigma_defs [simp]:
" upd_add_sigma \<sigma> k n = \<sigma>(k := insert n (\<sigma> k))"
unfolding upd_add_sigma_def by simp

lemma upd_remove_sigma_defs [simp]:
" upd_remove_sigma \<sigma> k A = \<sigma>(k :=  (\<sigma> k) - A)"
unfolding upd_remove_sigma_def by simp

lemma upd_remove_all_sigma_defs [simp]: 
 " upd_remove_all_sigma \<sigma> A = (\<lambda>i. (upd_remove_sigma \<sigma> i A) i) "
unfolding upd_remove_all_sigma_def by simp

lemma similars_upd_remove_all: 
 "similars \<sigma> A (upd_remove_all_sigma \<sigma> A) "
by simp





lemma finite_bvars: 
 " finite (bvars f) " 
by (induct f) simp_all

lemma finite_fvars: 
 " finite (fvars f) " 
by (induct f) simp_all


lemma similar_upd:
 shows " similar \<sigma> n (upd \<sigma> n l) " 
using similar_defs_alt[of \<sigma> n "(upd \<sigma> n l)"] 
using upd_def by force




lemma exist_sigma_exist_value: 
" (\<exists> \<sigma>'. (\<sigma>' i j \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') ) \<longleftrightarrow> (\<exists> l. (upd \<sigma> n l) i j \<Turnstile> f ) "  
proof -
 have 2: "(\<exists> \<sigma>'. (\<sigma>' i j \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') ) \<Longrightarrow> 
          (\<exists> l . ((upd \<sigma> n l) i j \<Turnstile> f))" 
   proof -
    assume a0: "(\<exists> \<sigma>'. (\<sigma>' i j \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') )" 
    show "(\<exists> l . ((upd \<sigma> n l) i j \<Turnstile> f))" 
     proof -
      obtain \<sigma>' where 4: "(\<sigma>' i j \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>')"  
        using a0 by blast
      have 6: "\<sigma>' = (upd \<sigma>' n (\<lambda> k. n \<in> (\<sigma>' k) )) " 
        unfolding upd_def 
        by (metis Diff_empty Diff_insert0 insert_absorb)
      have 7: "\<sigma> = (upd \<sigma> n (\<lambda> k. n \<in> (\<sigma> k) ))" 
         unfolding upd_def 
        by (metis Diff_empty Diff_insert0 insert_absorb)
      have 8: "\<And>x . (\<sigma>' x) = (upd \<sigma> n (\<lambda> k. n \<in> (\<sigma>' k) )) x" 
       unfolding upd_def using 4 
       by (metis "6" insert_Diff_single similar_defs_alt upd_def)
      have 9: "\<sigma>' = (upd \<sigma> n (\<lambda> k. n \<in> (\<sigma>' k) ))" 
         using 8 by blast
      have 10: "(upd \<sigma> n (\<lambda> k. n \<in> (\<sigma>' k) )) i j \<Turnstile> f   " 
        using "4" "9" by auto
       show ?thesis using "10" by blast 
     qed
    qed
  have 11: "(\<exists> l . ((upd \<sigma> n l) i j \<Turnstile> f)) \<Longrightarrow>
            (\<exists> \<sigma>'. (\<sigma>' i j \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') )" 
  proof -
   assume a1: "(\<exists> l . ((upd \<sigma> n l) i j \<Turnstile> f))" 
   show "(\<exists> \<sigma>'. (\<sigma>' i j \<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') )" 
   proof -
    obtain l where 12: "((upd \<sigma> n l) i j \<Turnstile> f)" 
     using a1 by blast
    have 13: " similar \<sigma> n (upd \<sigma> n l) " 
     by (simp add: similar_upd)
    show ?thesis 
    using "12" "13" by blast
   qed
  qed
 show ?thesis 
 using  "11" "2" by blast
qed


lemma cexist_sigma_cexist_value: 
" (\<exists> \<sigma>'. (\<sigma>' i j c\<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') ) \<longleftrightarrow> (\<exists> l. (upd \<sigma> n l) i j c\<Turnstile> f ) "  
proof -
 have 2: "(\<exists> \<sigma>'. (\<sigma>' i j c\<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') ) \<Longrightarrow> 
          (\<exists> l . ((upd \<sigma> n l) i j c\<Turnstile> f))" 
   proof -
    assume a0: "(\<exists> \<sigma>'. (\<sigma>' i j c\<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') )" 
    show "(\<exists> l . ((upd \<sigma> n l) i j c\<Turnstile> f))" 
     proof -
      obtain \<sigma>' where 4: "(\<sigma>' i j c\<Turnstile> f) \<and> (similar \<sigma> n \<sigma>')"  
        using a0 by blast
      have 6: "\<sigma>' = (upd \<sigma>' n (\<lambda> k. n \<in> (\<sigma>' k) )) " 
        unfolding upd_def 
        by (metis Diff_empty Diff_insert0 insert_absorb)
      have 7: "\<sigma> = (upd \<sigma> n (\<lambda> k. n \<in> (\<sigma> k) ))" 
         unfolding upd_def 
        by (metis Diff_empty Diff_insert0 insert_absorb)
      have 8: "\<And>x . (\<sigma>' x) = (upd \<sigma> n (\<lambda> k. n \<in> (\<sigma>' k) )) x" 
       unfolding upd_def using 4
       by (metis "6" insert_Diff_single similar_defs_alt upd_def) 
      have 9: "\<sigma>' = (upd \<sigma> n (\<lambda> k. n \<in> (\<sigma>' k) ))" 
         using 8 by blast
      have 10: "(upd \<sigma> n (\<lambda> k. n \<in> (\<sigma>' k) )) i j c\<Turnstile> f   " 
        using "4" "9" by auto
       show ?thesis using "10" by blast 
     qed
    qed
  have 11: "(\<exists> l . ((upd \<sigma> n l) i j c\<Turnstile> f)) \<Longrightarrow>
            (\<exists> \<sigma>'. (\<sigma>' i j c\<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') )" 
  proof -
   assume a1: "(\<exists> l . ((upd \<sigma> n l) i j c\<Turnstile> f))" 
   show "(\<exists> \<sigma>'. (\<sigma>' i j c\<Turnstile> f) \<and> (similar \<sigma> n \<sigma>') )" 
   proof -
    obtain l where 12: "((upd \<sigma> n l) i j c\<Turnstile> f)" 
     using a1 by blast
    have 13: " similar \<sigma> n (upd \<sigma> n l) "
    by (simp add: similar_upd) 
    show ?thesis 
    using "12" "13" by blast
   qed
  qed
 show ?thesis 
 using  "11" "2" by blast
qed

lemma upd_absorb: 
 assumes " x1 = x2 " 
 shows " upd (upd \<sigma> x1 l1) x2 l2 = (upd \<sigma> x2 l2) "
proof -
 have 1: "\<And> i. (upd (upd \<sigma> x1 l1) x2 l2 ) i = (upd \<sigma> x2 l2) i " 
  using assms unfolding upd_def by auto
 show ?thesis using 1 by auto
qed

lemma upd_swap: 
 assumes " x1 \<noteq> x2 " 
 shows " upd (upd \<sigma> x1 l1) x2 l2 = (upd (upd \<sigma> x2 l2) x1 l1) " 
proof -
 have 1: "\<And> i. (upd (upd \<sigma> x1 l1) x2 l2) i = (upd (upd \<sigma> x2 l2) x1 l1) i " 
   using assms unfolding upd_def by auto
 show ?thesis using 1 by auto
qed


lemma not_fvar_upd: 
 assumes " n \<notin> fvars f"
 shows " (\<sigma> i j  \<Turnstile> f) = ((upd \<sigma> n l) i j \<Turnstile> f) " 
using assms 
proof (induction f arbitrary: \<sigma> l i j)
case false_d
then show ?case by simp 
next
case (atom_d x)
then show ?case
by (metis fvars.simps(2) insertCI semantics_nl_itl.simps(2) similar_defs similar_upd) 
next
case (iimp_d f1 f2)
then show ?case by simp 
next
case (next_d f)
then show ?case by auto 
next
case (chop_d f1 f2)
then show ?case by simp 
next
case (ldiamond_d f)
then show ?case by simp 
next
case (rdiamond_d f)
then show ?case by simp
next
case (chopstar_d f)
then show ?case
by force 
next
case (exists_d x1 f)
then show ?case 
  proof -
   have e1: "(\<sigma> i j \<Turnstile> Ex x1. f) = (\<exists> l. ((upd \<sigma> x1 l) i j \<Turnstile> f) )" 
     using exist_sigma_exist_value by auto
   have e2: "((upd \<sigma> n l) i j \<Turnstile> Ex x1. f) = 
             (\<exists> l1.  ((upd (upd \<sigma> n l) x1 l1) i j \<Turnstile> f) )" 
    using exist_sigma_exist_value by force
   have e3: "(\<exists> l.  ((upd \<sigma> x1 l)  i j \<Turnstile> f) ) \<Longrightarrow>
             (\<exists> l1.  ((upd (upd \<sigma> n l) x1 l1)  i j \<Turnstile> f) )"
       by (metis Diff_iff exists_d.IH exists_d.prems fvars.simps(9) singletonD upd_absorb upd_swap)
   have e4: "(\<exists> l1.  ((upd (upd \<sigma> n l) x1 l1)  i j \<Turnstile> f) ) \<Longrightarrow>
             (\<exists> l.  ((upd \<sigma> x1 l) i j  \<Turnstile> f) )"
    by (metis Diff_iff emptyE exists_d.IH exists_d.prems fvars.simps(9) insertE upd_absorb upd_swap)
   show ?thesis
   using e1 e2 e3 e4 by blast      
   qed  
qed 

lemma not_fvar_upd_introspective: 
 assumes " n \<notin> cfvars f"
 shows " (\<sigma> i j  c\<Turnstile> f) = ((upd \<sigma> n l) i j c\<Turnstile> f) " 
using assms 
proof (induction f arbitrary: \<sigma> l i j)
case cfalse_d
then show ?case by simp
next
case (catom_d x)
then show ?case 
by (metis cfvars.simps(2) insertCI semantics_introspective_nl_itl.simps(2) similar_defs similar_upd)
next
case (ciimp_d f1 f2)
then show ?case by simp
next
case (cnext_d f)
then show ?case by auto
next
case (cchop_d f1 f2)
then show ?case by simp
next
case (cchopstar_d f)
then show ?case by force
next
case (cexists_d x1 f)
then show ?case 
  proof -
   have e1: "(\<sigma> i j c\<Turnstile> cEx x1. f) = (\<exists> l. ((upd \<sigma> x1 l) i j c\<Turnstile> f) )" 
     using cexist_sigma_cexist_value by auto
   have e2: "((upd \<sigma> n l) i j c\<Turnstile> cEx x1. f) = 
             (\<exists> l1.  ((upd (upd \<sigma> n l) x1 l1) i j c\<Turnstile> f) )" 
    using cexist_sigma_cexist_value by force
   have e3: "(\<exists> l.  ((upd \<sigma> x1 l)  i j c\<Turnstile> f) ) \<Longrightarrow>
             (\<exists> l1.  ((upd (upd \<sigma> n l) x1 l1)  i j c\<Turnstile> f) )"
       by (metis Diff_iff cexists_d.IH cexists_d.prems cfvars.simps(7) singletonD upd_absorb upd_swap)
   have e4: "(\<exists> l1.  ((upd (upd \<sigma> n l) x1 l1)  i j c\<Turnstile> f) ) \<Longrightarrow>
             (\<exists> l.  ((upd \<sigma> x1 l) i j  c\<Turnstile> f) )"
    by (metis Diff_iff emptyE cexists_d.IH cexists_d.prems cfvars.simps(7) insertE upd_absorb upd_swap)
   show ?thesis
   using e1 e2 e3 e4 by blast      
   qed  
qed




lemma inot_defs [simp]:
  "  (\<sigma> i j \<Turnstile> inot f ) =  (\<not>(\<sigma> i j \<Turnstile> f)) " 
by (simp add: inot_d_def) 

lemma cinot_defs [simp]:
  "  (\<sigma> i j c\<Turnstile> cinot f ) =  (\<not>(\<sigma> i j c\<Turnstile> f)) " 
by (simp add: cinot_d_def) 

lemma ior_defs [simp]: 
  " (\<sigma> i j \<Turnstile> f1 ior f2) = ( (\<sigma> i j \<Turnstile> f1) \<or> (\<sigma> i j \<Turnstile> f2))"
by (metis inot_defs ior_d_def semantics_nl_itl.simps(3))

lemma cior_defs [simp]: 
  " (\<sigma> i j c\<Turnstile> f1 cior f2) = ( (\<sigma> i j c\<Turnstile> f1) \<or> (\<sigma> i j c\<Turnstile> f2))"
by (metis cinot_defs cior_d_def semantics_introspective_nl_itl.simps(3))

lemma iand_defs [simp]:
  " (\<sigma> i j \<Turnstile> f1 iand f2) = ( (\<sigma> i j \<Turnstile> f1) \<and> (\<sigma> i j \<Turnstile> f2))"
by (simp add: iand_d_def)

lemma ciand_defs [simp]:
  " (\<sigma> i j c\<Turnstile> f1 ciand f2) = ( (\<sigma> i j c\<Turnstile> f1) \<and> (\<sigma> i j c\<Turnstile> f2))"
by (simp add: ciand_d_def)

lemma ieqv_defs [simp]: 
  " (\<sigma> i j \<Turnstile> f1 ieqv f2) = ( (\<sigma> i j \<Turnstile> f1) = (\<sigma> i j \<Turnstile> f2))"
by (metis iand_defs ieqv_d_def semantics_nl_itl.simps(3))

lemma cieqv_defs [simp]: 
  " (\<sigma> i j c\<Turnstile> f1 cieqv f2) = ( (\<sigma> i j c\<Turnstile> f1) = (\<sigma> i j c\<Turnstile> f2))"
by (metis ciand_defs cieqv_d_def semantics_introspective_nl_itl.simps(3))

lemma itrue_defs [simp]:
  " (\<sigma> i j \<Turnstile> itrue)"
by (simp add: itrue_d_def)

lemma citrue_defs [simp]:
  " (\<sigma> i j c\<Turnstile> citrue)"
by (simp add: citrue_d_def)

lemma empty_defs [simp]: 
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
shows  "(\<sigma> i j \<Turnstile> empty) = ( i = j ) "
using assms
by (auto simp add: empty_d_def more_d_def)
   (metis eint_bot eint_cases eint_less_eq(2) nle_le)

lemma cempty_defs [simp]: 
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
shows  "(\<sigma> i j c\<Turnstile> cempty) = ( i = j ) "
using assms
by (auto simp add: cempty_d_def cmore_d_def)
   (metis eint_bot eint_cases eint_less_eq(2) nle_le)

lemma skip_defs [simp]:
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
shows " (\<sigma> i j \<Turnstile> skip) = ( i+(eint 1) = j) " 
proof -
 have 1: " (\<sigma> i j \<Turnstile> skip) = (\<sigma> i j  \<Turnstile> next empty)" 
  unfolding skip_d_def by simp
 have 2: " (\<sigma> i j  \<Turnstile> next empty) = ( i \<noteq> -\<infinity> \<and> i<j \<and> ( \<sigma> (i + (eint 1) ) j \<Turnstile> empty)) " 
    by simp
 have 3: "i \<noteq> -\<infinity> \<and> i<j \<longrightarrow> (i + (eint 1)) \<le> j "
    by (metis abs_eint.cases assms(2) eint_less_le_alt plus_eint.simps(1))
 have 4: "i + eint (1::int) \<noteq> \<infinity>" 
 by (simp add: assms(2))
 have 5: "( i \<noteq> -\<infinity> \<and> i<j \<and> ( \<sigma> (i+ (eint 1) ) j \<Turnstile> empty)) \<longrightarrow> ((i + (eint 1) ) = j)"
    using  assms empty_defs[of "i+(eint 1)" j ] 3 4 by blast
 have 6: "( i+(eint 1) = j) \<longrightarrow> ( i \<noteq> -\<infinity> \<and> i<j \<and> ( \<sigma> (i + (eint 1) ) j \<Turnstile> empty)) "  
    using assms empty_defs[of "i+(eint 1)" j]
    by (metis "4" eint_0_less_1 eint_between(2) eint_infinity_cases 
        less_eq_eint_def one_eint_def plus_eint.simps(5))
 show ?thesis using "1" "2" "5" "6" by blast
qed

lemma cskip_defs [simp]:
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
shows " (\<sigma> i j c\<Turnstile> cskip) = ( i+(eint 1) = j) " 
proof -
 have 1: " (\<sigma> i j c\<Turnstile> cskip) = (\<sigma> i j  c\<Turnstile> cnext cempty)" 
  unfolding cskip_d_def by simp
 have 2: " (\<sigma> i j  c\<Turnstile> cnext cempty) = ( i \<noteq> -\<infinity> \<and> i<j \<and> ( \<sigma> (i + (eint 1) ) j c\<Turnstile> cempty)) " 
    by simp
 have 3: "i \<noteq> -\<infinity> \<and> i<j \<longrightarrow> (i + (eint 1)) \<le> j "
    by (metis abs_eint.cases assms(2) eint_less_le_alt plus_eint.simps(1))
 have 4: "i + eint (1::int) \<noteq> \<infinity>" 
 by (simp add: assms(2))
 have 5: "( i \<noteq> -\<infinity> \<and> i<j \<and> ( \<sigma> (i+ (eint 1) ) j c\<Turnstile> cempty)) \<longrightarrow> ((i + (eint 1) ) = j)"
    using  assms cempty_defs[of "i+(eint 1)" j ] 3 4 by blast
 have 6: "( i+(eint 1) = j) \<longrightarrow> ( i \<noteq> -\<infinity> \<and> i<j \<and> ( \<sigma> (i + (eint 1) ) j c\<Turnstile> cempty)) "  
    using assms cempty_defs[of "i+(eint 1)" j]
    by (metis "4" eint_0_less_1 eint_between(2) eint_infinity_cases 
        less_eq_eint_def one_eint_def plus_eint.simps(5))
 show ?thesis using "1" "2" "5" "6" by blast
qed


lemma lfinite_defs[simp]: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
shows " (\<sigma> i j \<Turnstile> lfinite) = (i \<noteq> -\<infinity>) " 
using assms 
by (auto simp add: empty_d_def more_d_def lfinite_d_def)
   (metis eint_bot nle_le)

lemma clfinite_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
shows " (\<sigma> i j c\<Turnstile> clfinite) = (i \<noteq> -\<infinity>) " 
using assms 
by (auto simp add: cempty_d_def cmore_d_def clfinite_d_def)
   (metis eint_bot nle_le)
 
lemma rfinite_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
shows " (\<sigma> i j \<Turnstile> rfinite) = (j \<noteq> \<infinity>) " 
using assms
by (auto simp add: empty_d_def more_d_def rfinite_d_def)
   (metis eint_cases eint_less_eq(3) nle_le)

lemma crfinite_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
shows " (\<sigma> i j c\<Turnstile> crfinite) = (j \<noteq> \<infinity>) " 
using assms
by (auto simp add: cempty_d_def cmore_d_def crfinite_d_def)
   (metis eint_cases eint_less_eq(3) nle_le)

lemma prev_defs:
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j \<Turnstile> prev f) =  (j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) \<Turnstile> f)) " 
proof -
 have 1: "(\<sigma> i j \<Turnstile> prev f) = 
         (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) \<Turnstile> f) \<and> (\<sigma> (eint k) j \<Turnstile> skip))  "
   unfolding prev_d_def by simp
 have 2: "(\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) \<Turnstile> f) \<and> (\<sigma> (eint k) j \<Turnstile> skip)) \<longrightarrow>
          (j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) \<Turnstile> f))" 
    using assms(3) le_eint_less by auto
 have 3: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) \<Turnstile> f)) \<longrightarrow> (\<exists>k. j = (eint k))" 
    by (metis less_eint.elims(2))
 have 4: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) \<Turnstile> f)) \<longrightarrow>
          i \<le> eint (int_of_eint (j-1)) " 
   using assms 3 using int_less_eint_iff int_of_eint_minus by fastforce
 have 5: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) \<Turnstile> f)) \<longrightarrow>
           eint (int_of_eint (j-1)) \<le> j \<and>
          (\<sigma> i (eint (int_of_eint (j-1))) \<Turnstile> f) \<and> (\<sigma> (eint (int_of_eint (j-1))) j \<Turnstile> skip)
           "
           using assms 3  by (auto simp add: one_eint_def) 
 have 6: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) \<Turnstile> f)) \<longrightarrow>
          (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) \<Turnstile> f) \<and> (\<sigma> (eint k) j \<Turnstile> skip))"
    using 4 5 by fastforce
 show ?thesis using 2 6 1 by blast
qed

lemma cprev_defs:
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j c\<Turnstile> cprev f) =  (j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) c\<Turnstile> f)) " 
proof -
 have 1: "(\<sigma> i j c\<Turnstile> cprev f) = 
         (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> f) \<and> (\<sigma> (eint k) j c\<Turnstile> cskip))  "
   unfolding cprev_d_def by simp
 have 2: "(\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> f) \<and> (\<sigma> (eint k) j c\<Turnstile> cskip)) \<longrightarrow>
          (j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) c\<Turnstile> f))" 
    using assms(3) le_eint_less by auto
 have 3: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) c\<Turnstile> f)) \<longrightarrow> (\<exists>k. j = (eint k))" 
    by (metis less_eint.elims(2))
 have 4: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) c\<Turnstile> f)) \<longrightarrow>
          i \<le> eint (int_of_eint (j-1)) " 
   using assms 3 using int_less_eint_iff int_of_eint_minus by fastforce
 have 5: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) c\<Turnstile> f)) \<longrightarrow>
           eint (int_of_eint (j-1)) \<le> j \<and>
          (\<sigma> i (eint (int_of_eint (j-1))) c\<Turnstile> f) \<and> (\<sigma> (eint (int_of_eint (j-1))) j c\<Turnstile> cskip)
           "
           using assms 3  by (auto simp add: one_eint_def) 
 have 6: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - (eint 1)) c\<Turnstile> f)) \<longrightarrow>
          (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> f) \<and> (\<sigma> (eint k) j c\<Turnstile> cskip))"
    using 4 5 by fastforce
 show ?thesis using 2 6 1 by blast
qed

lemma wnext_defs :
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j \<Turnstile> wnext f) = ( i = -\<infinity> \<or> i = j \<or> (i < j \<and> (\<sigma> (i + eint 1) j \<Turnstile> f))) " 
proof -
 have 2: "(\<sigma> i j \<Turnstile> inot (next inot f)) = (\<not>(\<sigma> i j \<Turnstile>  (next inot f)))"
   using inot_defs by blast
 have 3: "(\<sigma> i j \<Turnstile>  (next inot f)) = (i \<noteq> - \<infinity> \<and> i < j \<and> \<not> (\<sigma> (i + eint 1) j \<Turnstile> f))"
   by simp 
 have 4: "(\<not>(\<sigma> i j \<Turnstile>  (next inot f))) =
          (\<not>(i \<noteq> - \<infinity> \<and> i < j \<and> \<not> (\<sigma> (i + eint 1) j \<Turnstile> f)))"
      using "3" by blast 
 have 5: "(\<not>(i \<noteq> - \<infinity> \<and> i < j \<and> \<not> (\<sigma> (i + eint 1) j \<Turnstile> f))) =
          ( i = -\<infinity> \<or> i = j \<or> (i < j \<and> (\<sigma> (i + eint 1) j \<Turnstile> f)))"
   using assms(1) by auto 
 show ?thesis using "2" "3" "5" wnext_d_def by presburger
qed

lemma cwnext_defs :
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j c\<Turnstile> cwnext f) = ( i = -\<infinity> \<or> i = j \<or> (i < j \<and> (\<sigma> (i + eint 1) j c\<Turnstile> f))) " 
proof -
 have 2: "(\<sigma> i j c\<Turnstile> cinot (cnext cinot f)) = (\<not>(\<sigma> i j c\<Turnstile>  (cnext cinot f)))"
   using cinot_defs by blast
 have 3: "(\<sigma> i j c\<Turnstile>  (cnext cinot f)) = (i \<noteq> - \<infinity> \<and> i < j \<and> \<not> (\<sigma> (i + eint 1) j c\<Turnstile> f))"
   by simp 
 have 4: "(\<not>(\<sigma> i j c\<Turnstile>  (cnext cinot f))) =
          (\<not>(i \<noteq> - \<infinity> \<and> i < j \<and> \<not> (\<sigma> (i + eint 1) j c\<Turnstile> f)))"
      using "3" by blast 
 have 5: "(\<not>(i \<noteq> - \<infinity> \<and> i < j \<and> \<not> (\<sigma> (i + eint 1) j c\<Turnstile> f))) =
          ( i = -\<infinity> \<or> i = j \<or> (i < j \<and> (\<sigma> (i + eint 1) j c\<Turnstile> f)))"
   using assms(1) by auto 
 show ?thesis using "2" "3" "5" cwnext_d_def by presburger
qed

lemma wprev_defs:
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j \<Turnstile> wprev f) = (j = \<infinity> \<or> i = j \<or> (i < j \<and> (\<sigma> i (j - (eint 1)) \<Turnstile> f)))"
proof -
 have 1: "(\<sigma> i j \<Turnstile> wprev f) = (\<sigma> i j \<Turnstile> inot (prev inot f))" 
  unfolding wprev_d_def by blast
 have 2: "(\<sigma> i j \<Turnstile> inot (prev inot f)) = (\<not> (\<sigma> i j \<Turnstile> prev inot f))"
  using assms  inot_defs[of \<sigma> i j "prev inot f"] by blast
 have 3: "(\<sigma> i j \<Turnstile> prev inot f) = (j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - eint 1) \<Turnstile> inot f))" 
  using assms  prev_defs[of i j \<sigma> "inot f" ] by blast
 have 4: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - eint 1) \<Turnstile> inot f)) =
          (j \<noteq> \<infinity> \<and> i < j \<and> \<not>(\<sigma> i (j - eint 1) \<Turnstile>  f))" 
   by simp
 have 5: " (\<not> (\<sigma> i j \<Turnstile> prev inot f)) =
           (\<not>(j \<noteq> \<infinity> \<and> i < j \<and> \<not>(\<sigma> i (j - eint 1) \<Turnstile>  f))) "
   using "3" "4" by presburger
 have 6: "(\<not>(j \<noteq> \<infinity> \<and> i < j \<and> \<not>(\<sigma> i (j - eint 1) \<Turnstile>  f))) =
         (j = \<infinity> \<or> i = j \<or> (i < j \<and> (\<sigma> i (j - (eint 1)) \<Turnstile> f))) " 
    using assms(1) order_less_le by blast
 from 1 2 5 6 show  ?thesis 
 by presburger
qed

lemma cwprev_defs:
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j c\<Turnstile> cwprev f) = (j = \<infinity> \<or> i = j \<or> (i < j \<and> (\<sigma> i (j - (eint 1)) c\<Turnstile> f)))"
proof -
 have 1: "(\<sigma> i j c\<Turnstile> cwprev f) = (\<sigma> i j c\<Turnstile> cinot (cprev cinot f))" 
  unfolding cwprev_d_def by blast
 have 2: "(\<sigma> i j c\<Turnstile> cinot (cprev cinot f)) = (\<not> (\<sigma> i j c\<Turnstile> cprev cinot f))"
  using assms  cinot_defs[of \<sigma> i j "cprev cinot f"] by blast
 have 3: "(\<sigma> i j c\<Turnstile> cprev cinot f) = (j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - eint 1) c\<Turnstile> cinot f))" 
  using assms  cprev_defs[of i j \<sigma> "cinot f" ] by blast
 have 4: "(j \<noteq> \<infinity> \<and> i < j \<and> (\<sigma> i (j - eint 1) c\<Turnstile> cinot f)) =
          (j \<noteq> \<infinity> \<and> i < j \<and> \<not>(\<sigma> i (j - eint 1) c\<Turnstile>  f))" 
   by simp
 have 5: " (\<not> (\<sigma> i j c\<Turnstile> cprev cinot f)) =
           (\<not>(j \<noteq> \<infinity> \<and> i < j \<and> \<not>(\<sigma> i (j - eint 1) c\<Turnstile>  f))) "
   using "3" "4" by presburger
 have 6: "(\<not>(j \<noteq> \<infinity> \<and> i < j \<and> \<not>(\<sigma> i (j - eint 1) c\<Turnstile>  f))) =
         (j = \<infinity> \<or> i = j \<or> (i < j \<and> (\<sigma> i (j - (eint 1)) c\<Turnstile> f))) " 
    using assms(1) order_less_le by blast
 from 1 2 5 6 show  ?thesis 
 by presburger
qed


lemma not_atom_defs:
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j \<Turnstile> inot ($p)) = (i = -\<infinity> \<or>  (p \<notin>(\<sigma> (int_of_eint i)))) "
using assms by simp

lemma not_catom_defs:
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j c\<Turnstile> cinot ($$p)) = (i = -\<infinity> \<or>  (p \<notin>(\<sigma> (int_of_eint i)))) "
using assms by simp

lemma diamond_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j \<Turnstile> diamond f) = (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> f) ) " 
using assms unfolding sometimes_d_def by simp

lemma cdiamond_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j c\<Turnstile> cdiamond f) = (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j c\<Turnstile> f) ) " 
using assms unfolding csometimes_d_def by simp

lemma box_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j \<Turnstile> box f) = (\<forall>k. i \<le> eint k \<and> eint k \<le> j \<longrightarrow> (\<sigma> (eint k) j \<Turnstile> f) ) " 
unfolding always_d_def sometimes_d_def using assms by auto

lemma cbox_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j c\<Turnstile> cbox f) = (\<forall>k. i \<le> eint k \<and> eint k \<le> j \<longrightarrow> (\<sigma> (eint k) j c\<Turnstile> f) ) " 
unfolding calways_d_def csometimes_d_def using assms by auto

lemma di_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j \<Turnstile> di f) = (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) \<Turnstile> f) ) " 
using assms unfolding di_d_def by simp

lemma cdi_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j c\<Turnstile> cdi f) = (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> f) ) " 
using assms unfolding cdi_d_def by simp

lemma bi_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j \<Turnstile> bi f) = (\<forall>k. i \<le> eint k \<and> eint k \<le> j \<longrightarrow> (\<sigma> i (eint k)  \<Turnstile> f) ) " 
unfolding bi_d_def di_d_def using assms by auto

lemma cbi_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j c\<Turnstile> cbi f) = (\<forall>k. i \<le> eint k \<and> eint k \<le> j \<longrightarrow> (\<sigma> i (eint k)  c\<Turnstile> f) ) " 
unfolding cbi_d_def cdi_d_def using assms by auto

lemma init_defs:
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j \<Turnstile> init f) = (i \<noteq> - \<infinity> \<and> (\<sigma> i i \<Turnstile> f)) " 
proof -
 have 1: "(\<sigma> i j \<Turnstile> init f) =
          (\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) \<Turnstile> empty) \<and> (\<sigma> i (eint k) \<Turnstile> f)) "
  unfolding init_d_def by simp
 have 2: "(\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) \<Turnstile> empty) \<and> (\<sigma> i (eint k) \<Turnstile> f)) 
          \<longrightarrow> (i \<noteq> \<infinity> \<and> (\<sigma> i i \<Turnstile> f))"
      using assms(2) by force 
 have 3: "i \<noteq> -\<infinity> \<and> (\<sigma> i i \<Turnstile> f) \<longrightarrow> (\<exists> k. i = (eint k))"
    by (meson assms(2) int_of_eint.elims)
 have 4: "i \<noteq> -\<infinity> \<and> (\<sigma> i i \<Turnstile> f) \<longrightarrow> 
          i \<le> eint (int_of_eint i) \<and> eint (int_of_eint i) \<le> j \<and>
          (\<sigma> i (eint (int_of_eint i)) \<Turnstile> empty) \<and> (\<sigma> i (eint (int_of_eint i)) \<Turnstile> f)"
    using assms "3" by fastforce 
 have 5: "i \<noteq> -\<infinity> \<and> (\<sigma> i i \<Turnstile> f) \<longrightarrow> 
          (\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) \<Turnstile> empty) \<and> (\<sigma> i (eint k) \<Turnstile> f))  " 
   using 4 by fastforce
 show ?thesis using "1" "2" "4" by auto
qed 

lemma cinit_defs:
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j c\<Turnstile> cinit f) = (i \<noteq> - \<infinity> \<and> (\<sigma> i i c\<Turnstile> f)) " 
proof -
 have 1: "(\<sigma> i j c\<Turnstile> cinit f) =
          (\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> cempty) \<and> (\<sigma> i (eint k) c\<Turnstile> f)) "
  unfolding cinit_d_def by simp
 have 2: "(\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> cempty) \<and> (\<sigma> i (eint k) c\<Turnstile> f)) 
          \<longrightarrow> (i \<noteq> \<infinity> \<and> (\<sigma> i i c\<Turnstile> f))"
      using assms(2) by force 
 have 3: "i \<noteq> -\<infinity> \<and> (\<sigma> i i c\<Turnstile> f) \<longrightarrow> (\<exists> k. i = (eint k))"
    by (meson assms(2) int_of_eint.elims)
 have 4: "i \<noteq> -\<infinity> \<and> (\<sigma> i i c\<Turnstile> f) \<longrightarrow> 
          i \<le> eint (int_of_eint i) \<and> eint (int_of_eint i) \<le> j \<and>
          (\<sigma> i (eint (int_of_eint i)) c\<Turnstile> cempty) \<and> (\<sigma> i (eint (int_of_eint i)) c\<Turnstile> f)"
    using assms "3" by fastforce 
 have 5: "i \<noteq> -\<infinity> \<and> (\<sigma> i i c\<Turnstile> f) \<longrightarrow> 
          (\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> cempty) \<and> (\<sigma> i (eint k) c\<Turnstile> f))  " 
   using 4 by fastforce
 show ?thesis using "1" "2" "4" by auto
qed 

 
lemma fin_defs:
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j \<Turnstile> fin f) = (j \<noteq> \<infinity> \<and> (\<sigma> j j \<Turnstile> f)) "
proof -
 have 1: "(\<sigma> i j \<Turnstile> fin f) =
          (\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> empty) \<and> (\<sigma> (eint k) j \<Turnstile> f)) " 
  unfolding fin_d_def by simp 
 have 2: "(\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> empty) \<and> (\<sigma> (eint k) j \<Turnstile> f)) \<longrightarrow>
          (j \<noteq>  \<infinity> \<and> (\<sigma> j j \<Turnstile> f))"
   using PInfty_neq_eint(1) assms(3) empty_defs by blast 
 have 3: "(j \<noteq>  \<infinity> \<and> (\<sigma> j j \<Turnstile> f)) \<longrightarrow> (\<exists>k. j = eint k)"
    using assms by (metis MInfty_eq_minfinity PInfty_eq_infinity eint.exhaust) 
 have 4: "(j \<noteq>  \<infinity> \<and> (\<sigma> j j \<Turnstile> f)) \<longrightarrow>
          i \<le> eint (int_of_eint j) \<and> eint (int_of_eint j) \<le> j \<and> 
          (\<sigma> (eint (int_of_eint j)) j \<Turnstile> empty) \<and> (\<sigma> (eint (int_of_eint j)) j \<Turnstile> f)" 
   using assms 3 by fastforce
 show ?thesis using 1 2 4 by auto
qed

lemma cfin_defs:
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j c\<Turnstile> cfin f) = (j \<noteq> \<infinity> \<and> (\<sigma> j j c\<Turnstile> f)) "
proof -
 have 1: "(\<sigma> i j c\<Turnstile> cfin f) =
          (\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j c\<Turnstile> cempty) \<and> (\<sigma> (eint k) j c\<Turnstile> f)) " 
  unfolding cfin_d_def by simp 
 have 2: "(\<exists>k::int. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j c\<Turnstile> cempty) \<and> (\<sigma> (eint k) j c\<Turnstile> f)) \<longrightarrow>
          (j \<noteq>  \<infinity> \<and> (\<sigma> j j c\<Turnstile> f))"
   using PInfty_neq_eint(1) assms(3) cempty_defs by blast 
 have 3: "(j \<noteq>  \<infinity> \<and> (\<sigma> j j c\<Turnstile> f)) \<longrightarrow> (\<exists>k. j = eint k)"
    using assms by (metis MInfty_eq_minfinity PInfty_eq_infinity eint.exhaust) 
 have 4: "(j \<noteq>  \<infinity> \<and> (\<sigma> j j c\<Turnstile> f)) \<longrightarrow>
          i \<le> eint (int_of_eint j) \<and> eint (int_of_eint j) \<le> j \<and> 
          (\<sigma> (eint (int_of_eint j)) j c\<Turnstile> cempty) \<and> (\<sigma> (eint (int_of_eint j)) j c\<Turnstile> f)" 
   using assms 3 by fastforce
 show ?thesis using 1 2 4 by auto
qed

lemma state_nl_itl_defs: 
assumes "state_nl_itl w"
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
shows " (\<sigma> i j \<Turnstile> w) \<longleftrightarrow> (\<sigma> i i \<Turnstile> w) "
using assms
by (induct w arbitrary: \<sigma> ) simp_all



lemma next_diamond: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows "(\<sigma> i j \<Turnstile> next (diamond f)) = (\<sigma> i j \<Turnstile> (lfinite iand diamond (next f))) "  
proof -
 have 1: "(\<sigma> i j \<Turnstile> next (diamond f)) = (i \<noteq> - \<infinity> \<and> i < j \<and> (\<sigma> (i + eint 1) j \<Turnstile> diamond f))" 
   by simp
 have 2: "(i \<noteq> - \<infinity> \<and> i < j \<and> (\<sigma> (i + eint 1) j \<Turnstile> diamond f)) = 
          ( i \<noteq> - \<infinity> \<and> i < j \<and> (\<exists>k. i + eint 1 \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> f)))" 
   using diamond_defs[of "(i + eint 1)" j \<sigma> f ] 
   using assms(3) less_eint.elims(2) by fastforce
 have 30: "( i \<noteq> - \<infinity> \<and> i < j \<and> (\<exists>k. i + eint 1 \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> f))) \<Longrightarrow>
         (i\<noteq>-\<infinity> \<and>  (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> eint k < j \<and> (\<sigma> (eint (k + 1)) j \<Turnstile> f)))" 
       proof -
        assume a0: "( i \<noteq> - \<infinity> \<and> i < j \<and> (\<exists>k. i + eint 1 \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> f)))" 
        show "( i\<noteq>-\<infinity> \<and> (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> eint k < j \<and> (\<sigma> (eint (k + 1)) j \<Turnstile> f)))" 
        proof -
         obtain k where 31: "i + eint 1 \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> f)" 
           using a0 by blast
         have 32: "i \<le> eint (k-1)"
            using "31" assms(2) less_eint.elims(2) by fastforce 
         have 33: " eint (k-1) \<le> j \<and> eint (k-1) < j"
           by (metis "31" diff_add_cancel eint_le_less less_add_one order_less_imp_le) 
         have 34: "(\<sigma> (eint ((k-1) + 1)) j \<Turnstile> f)" 
           by (simp add: "31")
         show ?thesis 
         using "32" "33" "34" a0 by blast
        qed
      qed
 have 35: "(i\<noteq>-\<infinity> \<and>  (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> eint k < j \<and> (\<sigma> (eint (k + 1)) j \<Turnstile> f))) \<Longrightarrow>
           ( i \<noteq> - \<infinity> \<and> i < j \<and> (\<exists>k. i + eint 1 \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> f)))" 
    proof -
      assume a1: "(i\<noteq>-\<infinity> \<and>  (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> eint k < j \<and> (\<sigma> (eint (k + 1)) j \<Turnstile> f)))" 
      show "( i \<noteq> - \<infinity> \<and> i < j \<and> (\<exists>k. i + eint 1 \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> f)))" 
      proof -
       obtain k where 36: "i \<le> eint k \<and> eint k \<le> j \<and> eint k < j \<and> (\<sigma> (eint (k + 1)) j \<Turnstile> f)" 
        using a1 by blast
       have 37: "i + eint 1 \<le> eint (k+1)" 
         by (metis "36" add.commute add_mono eint_eq_1(1) eint_plus_1(1) order_refl)
       have 38: "eint (k+1) \<le> j" 
           by (simp add: "36" eint_less_le_alt)
       have 39: "(\<sigma> (eint (k+1)) j \<Turnstile> f)" 
         by (simp add: "36")
       have 40: "(\<exists>k. i + eint 1 \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> f))" 
         using "36" "37" "38" by auto
       have 41: "i < j"
         using a1 by fastforce 
       have 42: "i \<noteq> -\<infinity>" 
         by (simp add: a1)
      show ?thesis 
      by (simp add: "40" "41" "42")
     qed
   qed
 have 4: "(i\<noteq>-\<infinity> \<and> (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> next f))) =
         (i\<noteq>-\<infinity> \<and> (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> eint k < j \<and> (\<sigma> (eint (k + 1)) j \<Turnstile> f)))"
    by simp 
 have 5: "(\<sigma> i j \<Turnstile> (lfinite iand diamond (next f))) = 
           (i\<noteq>-\<infinity> \<and>(\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> (eint k) j \<Turnstile> next f))) "
       using diamond_defs[of i j \<sigma> "next f"]  lfinite_defs assms by fastforce
 show ?thesis 
 by (metis "1" "2" "30" "35" "4" "5")
qed



lemma linf_defs[simp]: 
 assumes " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        "i \<le> j" 
shows " (\<sigma> i j \<Turnstile> linf) = (i = -\<infinity>)" 
using assms unfolding linf_d_def by (simp)

lemma rinf_defs[simp]:
assumes " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        "i \<le> j" 
shows " (\<sigma> i j \<Turnstile> rinf) = (j = \<infinity>)" 
using assms unfolding rinf_d_def by (simp add: rfinite_defs)

lemma wchop_defs[simp]: 
assumes       
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        "i \<le> j" 
shows " (\<sigma> i j \<Turnstile> f wchop g) = 
        ((\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) \<Turnstile> f) \<and> (\<sigma> (eint k) j \<Turnstile> g)) \<or> 
         ((\<sigma> i j \<Turnstile> f) \<and> j = \<infinity>) 
        )" 
using assms unfolding wchop_d_def by simp


lemma clinf_defs[simp]: 
 assumes " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        "i \<le> j" 
shows " (\<sigma> i j c\<Turnstile> clinf) = (i = -\<infinity>)" 
using assms unfolding clinf_d_def by (simp add: clfinite_defs)

lemma crinf_defs[simp]:
assumes " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        "i \<le> j" 
shows " (\<sigma> i j c\<Turnstile> crinf) = (j = \<infinity>)" 
using assms unfolding crinf_d_def by (simp add: crfinite_defs)

lemma cwchop_defs[simp]: 
assumes       
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        "i \<le> j" 
shows " (\<sigma> i j c\<Turnstile> f cwchop g) = 
        ((\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> f) \<and> (\<sigma> (eint k) j c\<Turnstile> g)) \<or> 
         ((\<sigma> i j c\<Turnstile> f) \<and> j = \<infinity>) 
        )" 
using assms unfolding cwchop_d_def by simp


definition agree_on :: " interval \<Rightarrow> interval \<Rightarrow> eint \<Rightarrow> eint \<Rightarrow> bool" 
where " agree_on \<sigma> \<sigma>' i j = (\<forall>k. i \<le> k \<and> k \<le> j \<longrightarrow> (\<sigma> k) = (\<sigma>' k)) "

lemma agree_on_defs[simp]:
 " agree_on \<sigma> \<sigma>' i j = (\<forall>k. i \<le> k \<and> k \<le> j \<longrightarrow> (\<sigma> k) = (\<sigma>' k))" 
unfolding agree_on_def by simp

lemma agree_on_sub: 
 assumes " i \<le>j " 
         "agree_on \<sigma> \<sigma>' i j"
         " i \<le> l0" 
         " l0 \<le> l1"
         " l1 \<le> j" 
  shows "agree_on \<sigma> \<sigma>' l0 l1" 
using assms 
using order_trans by auto


lemma nridx_eint_gr_first: 
fixes i :: eint
assumes "nfinite l"
        " i \<noteq> -\<infinity> " 
        " i \<noteq> \<infinity> " 
        "(nnth l 0) = i"  
        "nridx (<) l " 
        " (enat k) \<le> nlength l" 
shows   " i \<le> (nnth l k) " 
using assms 
nridx_gr_first[of "\<lambda> (a :: eint) b. a < b" l k ]
  using order_le_less by auto


lemma nridx_eint_less_last: 
fixes j :: eint
assumes "nfinite l"
        " j \<noteq> -\<infinity> " 
        " j \<noteq> \<infinity> " 
        "nlast l = j"  
        "nridx (<) l " 
        " (enat k) < nlength l" 
shows   "  (nnth l (Suc k)) \<le> j" 
using assms 
nridx_less_last[of "\<lambda> (a :: eint) b. a < b" l k "the_enat (nlength l)" ]
apply simp
by (metis Suc_lessI enat_ord_simps(2) less_eq_eint_def less_imp_Suc_add nfinite_conv_nlength_enat
  nle_le nnth_nlast nridx_less the_enat.simps transp_on_less)

 
lemma
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        "(\<sigma> i j c\<Turnstile> f)"
        " agree_on \<sigma> \<sigma>' i j"
 shows " (\<sigma>' i j c\<Turnstile> f)"  
using assms 
proof (induction f arbitrary: \<sigma> \<sigma>' i j)
case cfalse_d
then show ?case by simp
next
case (catom_d x)
then show ?case apply simp 
by (metis eint_infty_less(2) int_of_eint.simps(1) less_eint.elims(2) order_refl)
next
case (ciimp_d f1 f2)
then show ?case apply simp by metis
next
case (cnext_d f)
then show ?case
  proof -
   have c1: "( \<sigma>' i j c\<Turnstile> cnext f) \<longleftrightarrow> (i \<noteq> - \<infinity> \<and> i < j \<and> (\<sigma>' (i + eint 1) j c\<Turnstile> f))" 
    by simp 
   have c2: "  (i \<noteq> - \<infinity> \<and> i < j \<and> (\<sigma> (i + eint 1) j c\<Turnstile> f)) " 
      using cnext_d by simp 
   have c3: "\<forall>k. i \<le> eint k \<and> eint k \<le> j \<longrightarrow> \<sigma> k = \<sigma>' k "
    using cnext_d by auto
   have c4: "agree_on \<sigma> \<sigma>' (i + eint 1) j  " 
     unfolding agree_on_def 
     by (metis MInfty_neq_eint(2) PInfty_neq_eint(1) add.right_neutral add_increasing2 c3 eint_0_plus
       eint_add_le_add_iff2 eint_less_eq(3) zero_less_one_class.zero_le_one)
   have c5: "(i \<noteq> - \<infinity> \<and> i < j \<and> (\<sigma> (i + eint 1) j c\<Turnstile> f))"
    using cnext_d.IH using c2 cnext_d.prems(4) by blast
   have c6: "i + eint 1 \<le> j" 
     by (metis c5 cnext_d.prems(2) eint_less_le_alt int_of_eint.cases plus_eint.simps(1))
   have c7: "i + eint 1 \<noteq> \<infinity> " 
     by (simp add: cnext_d.prems(2))
   have c8: " (\<sigma>' (i + eint 1) j c\<Turnstile> f)"
     using cnext_d.IH[of "(i + eint 1)"  j \<sigma> \<sigma>' ] 
     using c4 c5 c6 c7 cnext_d.prems(3) by blast
   show ?thesis    by (simp add: c5 c8) 
  qed
next
case (cchop_d f1 f2)
then show ?case 
   proof -
    have 1: "  (\<exists>k. i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> f1) \<and> (\<sigma> (eint k) j c\<Turnstile> f2))"
       using cchop_d.prems(4) by auto 
    obtain k where 2: "i \<le> eint k \<and> eint k \<le> j \<and> (\<sigma> i (eint k) c\<Turnstile> f1) \<and> (\<sigma> (eint k) j c\<Turnstile> f2)" 
     using 1 by auto
    have 3: "agree_on \<sigma> \<sigma>' i k" 
      by (meson "2" agree_on_defs cchop_d.prems(5) order_trans)
    have 4: "agree_on \<sigma> \<sigma>' k j" 
      by (meson "2" agree_on_defs cchop_d.prems(5) order_trans) 
    have 5: "\<sigma>' i k c\<Turnstile> f1" 
        by (metis "2" "3" MInfty_neq_eint(2) cchop_d.IH(1) cchop_d.prems(2))
    have 6: "\<sigma>' k j c\<Turnstile> f2"
      using "2" "4" PInfty_neq_eint(1) cchop_d.IH(2) cchop_d.prems(3) by blast 
    have 7: "\<sigma>' i j c\<Turnstile> f1 cschop f2"
     using 5 6 "2" semantics_introspective_nl_itl.simps(5) by blast 
    show ?thesis 
    using "7" by blast
  qed
next
case (cchopstar_d f)
then show ?case 
   proof (cases "i=j")
   case True
   then show ?thesis 
   using semantics_introspective_nl_itl.simps(6) by blast
   next
   case False
   then show ?thesis 
     proof (cases "i\<noteq>-\<infinity> \<and> j\<noteq> \<infinity>")
     case True
     then show ?thesis 
        proof -
        have 1: "\<sigma> i j c\<Turnstile> cchopstar f " 
          using cchopstar_d.prems by blast
        have 2: "(\<exists> (lrf:: eint nellist). 
       (nnth lrf 0) = i \<and> nfinite lrf \<and> nlast lrf = j \<and>  
        nridx (\<lambda> a b. a < b \<and> (\<sigma> a b c\<Turnstile> f)) lrf
      ) " 
          using \<open>i\<noteq>-\<infinity> \<and> j\<noteq> \<infinity>\<close> \<open>i \<noteq> j\<close> cchopstar_d.prems by simp
        obtain lrf where 3: "(nnth lrf 0) = i \<and> nfinite lrf \<and> nlast lrf = j \<and>  
        nridx (\<lambda> a b. a < b \<and> (\<sigma> a b c\<Turnstile> f)) lrf" 
         using 2 by blast
        have 4: "(\<forall> (k::nat) . ( (enat k)< (nlength lrf)) \<longrightarrow> 
                               ( \<sigma> (nnth lrf k) (nnth lrf (Suc k))   c\<Turnstile> f) ) "   
          by (metis (no_types, lifting) "3" eSuc_enat ileI1 nridx_expand) 
        have 5: "\<And>k. ( (enat k)< (nlength lrf)) \<longrightarrow> (nnth lrf k) \<le> (nnth lrf (Suc k)) "
           by (metis (no_types, lifting) "3" eSuc_enat ileI1 nridx_expand order_less_imp_le)
        have 51: "nridx (<) lrf"
          by (metis (no_types, lifting) "3" nridx_expand)       
        have 6: "\<And>k. ( (enat k)< (nlength lrf)) \<longrightarrow> i \<le> (nnth lrf k)" 
            using 51 nridx_eint_gr_first 
            by (simp add: "3" True cchopstar_d.prems(2))
        have 7: "\<And>k. ( (enat k)< (nlength lrf)) \<longrightarrow> (nnth lrf (Suc k)) \<le> j" 
             using nridx_eint_less_last 
             by (simp add: "3" "51" True cchopstar_d.prems(3))
        have 8: "\<And>k. ( (enat k)< (nlength lrf)) \<longrightarrow>  
                     agree_on \<sigma> \<sigma>' (nnth lrf k) (nnth lrf (Suc k))" 
           using agree_on_sub[of i j \<sigma> \<sigma>'] 
           by (meson "5" "6" "7" cchopstar_d.prems(1) cchopstar_d.prems(5))
        have 9: "\<And>k. ( (enat k)< (nlength lrf)) \<longrightarrow>
                     ( \<sigma>' (nnth lrf k) (nnth lrf (Suc k))   c\<Turnstile> f) " 
          by (metis "4" "5" "6" "7" "8" True cchopstar_d.IH eint_infty_less_eq(2) eint_less_eq(1) nle_le)
        have 10: "nridx (\<lambda> a b. a < b \<and> (\<sigma>' a b c\<Turnstile> f)) lrf"  
          using 9 
          by (metis (no_types, lifting) "3" Suc_ile_eq nridx_expand)   
         show ?thesis  
         using "10" "3" True by auto
        qed
     next
     case False
     then show ?thesis 
        proof (cases "i = -\<infinity> \<and> j\<noteq> \<infinity>")
        case True
        then show ?thesis 
           proof -
            have b1: "(\<exists> (li :: eint nellist). 
                      (nnth li 0) = j \<and> \<not> nfinite li  \<and> nridx (\<lambda> a b. b < a \<and> (\<sigma> b a c\<Turnstile> f)) li
                      )" 
            using \<open>i = -\<infinity> \<and> j\<noteq> \<infinity>\<close> cchopstar_d.prems by simp
            obtain li where b2: "(nnth li 0) = j \<and> \<not> nfinite li  \<and>  
                                 nridx (\<lambda> a b. b < a \<and> (\<sigma> b a c\<Turnstile> f)) li" 
            using b1 by blast
            have b3: "(\<forall> (k::nat) .  
                               ( \<sigma>  (nnth li (Suc k)) (nnth li k)   c\<Turnstile> f) )" 
              by (metis (no_types, lifting) b2 enat_ile linorder_le_cases nfinite_conv_nlength_enat nridx_expand)
             have b5: "\<And>k.  (nnth li (Suc k)) \<le>(nnth li k)  "
                by (metis (no_types, lifting) b2 enat_ile linorder_le_cases nfinite_conv_nlength_enat nridx_expand order_less_imp_le)
             have b51: "nridx (>) li"
               by (metis (no_types, lifting) b2 nridx_expand)
        have b8: "\<And>k.  agree_on \<sigma> \<sigma>' (nnth li (Suc k)) (nnth li k) " 
                by (metis (mono_tags, lifting) "b5" True agree_on_sub b2 cchopstar_d.prems(5) eint_infty_less_eq(2) lift_Suc_antimono_le nle_le zero_le)
        have b9: "\<And>k. 
                     ( \<sigma>' (nnth li (Suc k)) (nnth li k)    c\<Turnstile> f) " 
         by (metis (no_types, lifting) "b5" "b8" MInfty_neq_eint(1) PInfty_neq_eint(1) True b2 cchopstar_d.IH cchopstar_d.prems(2) enat_ile less_eint.elims(2) linorder_le_cases nfinite_conv_nlength_enat nridx_expand)
          have b10: "nridx (\<lambda> a b. a > b \<and> (\<sigma>' b a  c\<Turnstile> f)) li"  
          using b9
          by (metis (no_types, lifting) b2 nridx_expand)
        show ?thesis 
        using "b10" True b2 by auto
        qed
        next
        case False
        then show ?thesis 
          proof (cases "i \<noteq> -\<infinity> \<and> j = \<infinity> " )
          case True
          then show ?thesis 
            proof -
             have c1: "(\<exists> (ri :: eint nellist). 
                       (nnth ri 0) = i \<and> \<not> nfinite ri  \<and> nridx (\<lambda> a b. a < b \<and> (\<sigma> a b c\<Turnstile> f)) ri
                      )" 
            using \<open>i \<noteq> -\<infinity> \<and> j = \<infinity>\<close> cchopstar_d.prems by simp
            obtain ri where c2: "(nnth ri 0) = i \<and> \<not> nfinite ri  \<and>  nridx (\<lambda> a b. a < b \<and> (\<sigma> a b c\<Turnstile> f)) ri" 
            using c1 by blast
            have c3: "(\<forall> (k::nat) .  
                               ( \<sigma>   (nnth ri k) (nnth ri (Suc k))  c\<Turnstile> f) )" 
               by (metis (no_types, lifting) c2 enat_ile linorder_le_cases nfinite_conv_nlength_enat nridx_expand)
            have c5: "\<And>k.  (nnth ri k) \<le> (nnth ri (Suc k))  "
                by (metis (no_types, lifting) c2 enat_ile linorder_le_cases nfinite_conv_nlength_enat nridx_expand order_less_imp_le)
             have c51: "nridx (<) ri"
               by (metis (no_types, lifting) c2 nridx_expand)
           
        have c6: "\<And>k.  i \<le> (nnth ri k)" 
            using c51 nridx_eint_gr_first 
            using c2 c5 lift_Suc_mono_le by blast
        have c8: "\<And>k.  agree_on \<sigma> \<sigma>'  (nnth ri k) (nnth ri (Suc k))" 
         using True agree_on_sub c2 c5 cchopstar_d.prems(5) eint_less_eq(1) lift_Suc_mono_le by blast
         have c9: "\<And>k. 
                     ( \<sigma>' (nnth ri k)  (nnth ri (Suc k))    c\<Turnstile> f) " 
           by (metis (no_types, lifting) PInfty_neq_eint(1) True c2 c5 c6 c8 cchopstar_d.IH eint_infty_less_eq(2) enat_ile less_eint.elims(2) linorder_le_cases nfinite_conv_nlength_enat nridx_expand)
          have c10: "nridx (\<lambda> a b. a < b \<and> (\<sigma>' a b   c\<Turnstile> f)) ri"  
          using c9
          by (metis (no_types, lifting) c2 nridx_expand)
           show ?thesis 
           using True c10 c2 by auto
          qed
          next
          case False
          then show ?thesis 
            proof -
             have d0: "i = -\<infinity> \<and> j = \<infinity>" 
                using \<open>i \<noteq>j\<close> \<open>\<not>(i \<noteq> -\<infinity> \<and> j = \<infinity> )\<close> \<open>\<not>(i = -\<infinity> \<and> j\<noteq> \<infinity> )\<close> \<open>\<not>(i\<noteq>-\<infinity> \<and> j\<noteq> \<infinity> )\<close>
                  by blast
             have d1: "(\<exists> (li :: int nellist) (ri :: int nellist) . 
                         (nnth li 0) = (nnth ri 0) \<and> 
                         \<not> nfinite li  \<and> nridx (\<lambda> a b. b < a \<and> (\<sigma> b a c\<Turnstile> f)) li \<and>
                         \<not> nfinite ri  \<and> nridx (\<lambda> a b. a < b \<and> (\<sigma> a b c\<Turnstile> f)) ri 
                          )" 
            using d0 cchopstar_d.prems by simp
            obtain ri li  where d2: "(nnth (li :: int nellist) 0) = (nnth (ri :: int nellist) 0) \<and> 
                \<not> nfinite li  \<and> nridx (\<lambda> a b. b < a \<and> (\<sigma> b a c\<Turnstile> f)) li \<and>
                \<not> nfinite ri  \<and> nridx (\<lambda> a b. a < b \<and> (\<sigma> a b c\<Turnstile> f)) ri " 
                using d1 by blast
            have d3: "(\<forall> (k::nat) .  
                               ( \<sigma>   (nnth ri k) (nnth ri (Suc k))  c\<Turnstile> f) )" 
               by (metis (no_types, lifting) d2 enat_ile linorder_le_cases nfinite_conv_nlength_enat nridx_expand)
            have d4: "(\<forall> (k::nat) .  
                               ( \<sigma>  (nnth li (Suc k)) (nnth li k)   c\<Turnstile> f) )" 
              by (metis (no_types, lifting) d2 enat_ile linorder_le_cases nfinite_conv_nlength_enat nridx_expand)
            have d5: "\<And>k.  (nnth ri k) \<le> (nnth ri (Suc k))  "
                by (metis (no_types, lifting) d2 enat_ile linorder_le_cases nfinite_conv_nlength_enat nridx_expand order_less_imp_le)
            have d50: "\<And>k.  (nnth li (Suc k)) \<le> (nnth li k) "  
               by (metis (no_types, lifting) d2 enat_ile linorder_le_cases nfinite_conv_nlength_enat nridx_expand order_less_imp_le)
            have d51: "nridx (<) ri"
               by (metis (no_types, lifting) d2 nridx_expand)
            have d52: "nridx (>) li"
                  by (metis (no_types, lifting) d2 nridx_expand)
            have d8: "\<And>k.  agree_on \<sigma> \<sigma>'  (nnth ri k) (nnth ri (Suc k))" 
              using cchopstar_d.prems(5) d0 by auto
            have d80: "\<And>k.  agree_on \<sigma> \<sigma>' (nnth li (Suc k)) (nnth li k) "   
              using cchopstar_d.prems(5) d0 by auto
            have d9: "\<And>k. 
                     ( \<sigma>' (nnth ri k)  (nnth ri (Suc k))    c\<Turnstile> f) " 
              by (meson MInfty_neq_eint(1) PInfty_neq_eint(1) cchopstar_d.IH d3 d5 d8 eint_less_eq(3))
            have d90: "\<And>k. 
                     ( \<sigma>' (nnth li (Suc k)) (nnth li k)      c\<Turnstile> f) "    
             by (metis MInfty_neq_eint(1) PInfty_neq_eint(2) cchopstar_d.IH d4 d50 d80 eint_less_eq(3))
            have d10: "nridx (\<lambda> a b. a < b \<and> (\<sigma>' a b   c\<Turnstile> f)) ri"  
             using d9 by (metis (no_types, lifting) d2 nridx_expand)
            have d11: "nridx (\<lambda> a b. b < a \<and> (\<sigma>' b a   c\<Turnstile> f)) li"  
             using d90 by (metis (no_types, lifting) d2 nridx_expand)
           show ?thesis  
           using  d10 d2 d11 d0 by auto
           qed
          qed 
        qed
     qed
   qed
next
case (cexists_d x1 f)
then show ?case 
    proof -
     have 1: "(\<exists> l. (upd \<sigma> x1 l) i j c\<Turnstile> f )"
     using cexist_sigma_cexist_value cexists_d.prems(4) by auto 
     obtain l where 2: "(upd \<sigma> x1 l) i j c\<Turnstile> f" 
      using 1 by blast
     have 3: "(\<And>i j \<sigma> \<sigma>'. i \<le> j \<Longrightarrow> i \<noteq> \<infinity> \<Longrightarrow> j \<noteq> - \<infinity> \<Longrightarrow> 
                 \<sigma> i j c\<Turnstile> f \<Longrightarrow> agree_on \<sigma> \<sigma>' i j \<Longrightarrow> \<sigma>' i j c\<Turnstile> f) " 
       using cexists_d.IH by blast
     have 4: " \<sigma> i j c\<Turnstile> cEx x1. f"
          using cexists_d.prems by blast
     have 5: "agree_on \<sigma> \<sigma>' i j" 
         using cexists_d.prems by blast
     have 6: "agree_on (upd \<sigma> x1 l) (upd \<sigma>' x1 l) i j "
          using "5" upd_def by force 
     have 7: " (upd \<sigma>' x1 l) i j c\<Turnstile> f" 
        using "2" "6" cexists_d.IH cexists_d.prems(1) cexists_d.prems(2) cexists_d.prems(3) by blast
     show ?thesis using "7" cexist_sigma_cexist_value by auto
   qed
qed           


lemma schop_dist_ior: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
 shows " (\<sigma> i j \<Turnstile> ((f1 ior f2) schop f3) ieqv (f1 schop f3 ior f2 schop f3)) " 
using assms by auto
 
lemma cschop_dist_ior: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
 shows " (\<sigma> i j c\<Turnstile> ((f1 cior f2) cschop f3) cieqv (f1 cschop f3 cior f2 cschop f3)) " 
using assms by auto


lemma exists_atom: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
  shows "(\<sigma> i j \<Turnstile> Ex n. $n) = (i \<noteq> -\<infinity>)  " 
proof -
  have 1: "(\<sigma> i j \<Turnstile> Ex n. $n) \<longrightarrow> (i \<noteq> -\<infinity>)"
    by simp
  have 2: "(i \<noteq> -\<infinity>) \<longrightarrow> 
           n \<in> (upd_add_sigma \<sigma> (int_of_eint i) n) (int_of_eint i)
           \<and> (similar \<sigma> n (upd_add_sigma \<sigma> (int_of_eint i) n)) "apply simp
           by (metis fun_upd_apply insert_Diff_if similar_defs_alt singletonI)
  have 3: "(i \<noteq> -\<infinity>) \<longrightarrow> (\<exists>\<sigma>'. n \<in> \<sigma>' (int_of_eint i) \<and> similar \<sigma> n \<sigma>')" 
     using "2" by blast
  have 4: "(i \<noteq> -\<infinity>) \<longrightarrow> (\<sigma> i j \<Turnstile> Ex n.  $n)" 
  using 3 
  by (meson exist_sigma_exist_value semantics_nl_itl.simps(2) semantics_nl_itl.simps(9))
  show ?thesis using "1" "4" by blast
qed

lemma cexists_catom: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
  shows "(\<sigma> i j c\<Turnstile> cEx n. $$n) = (i \<noteq> -\<infinity>)  " 
proof -
  have 1: "(\<sigma> i j c\<Turnstile> cEx n. $$n) \<longrightarrow> (i \<noteq> -\<infinity>)"
    by simp
  have 2: "(i \<noteq> -\<infinity>) \<longrightarrow> 
           n \<in> (upd_add_sigma \<sigma> (int_of_eint i) n) (int_of_eint i)
           \<and> (similar \<sigma> n (upd_add_sigma \<sigma> (int_of_eint i) n)) " apply simp 
           by (metis fun_upd_apply insert_Diff1 similar_defs_alt singletonI)
  have 3: "(i \<noteq> -\<infinity>) \<longrightarrow> (\<exists>\<sigma>'. n \<in> \<sigma>' (int_of_eint i) \<and> similar \<sigma> n \<sigma>')" 
     using "2" by blast
  have 4: "(i \<noteq> -\<infinity>) \<longrightarrow> (\<sigma> i j c\<Turnstile> cEx n.  $$n)" 
  using 3 
  by (meson cexist_sigma_cexist_value semantics_introspective_nl_itl.simps(2) semantics_introspective_nl_itl.simps(7))
  show ?thesis using "1" "4" by blast
qed


lemma exists_not_atom: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
  shows "(\<sigma> i j \<Turnstile> Ex n. inot $n)   " 
using assms apply simp unfolding  upd_def 
 by auto


lemma cexists_not_catom: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
  shows "(\<sigma> i j c\<Turnstile> cEx n. cinot $$n)   " 
using assms apply simp unfolding  upd_def 
 by auto



lemma exists_exists: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
  shows "(\<sigma> i j \<Turnstile> Ex n.  (Ex n. f)) = (\<sigma> i j \<Turnstile> Ex n.  f) " 
using assms
apply simp 
using similar_refl similar_trans 
using upd_absorb by presburger

lemma cexists_cexists: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
  shows "(\<sigma> i j c\<Turnstile> cEx n.  (cEx n. f)) = (\<sigma> i j c\<Turnstile> cEx n.  f) " 
using assms 
apply simp 
using upd_absorb similar_refl similar_trans by presburger


lemma exists_ior_dist: 
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
shows " (\<sigma> i j \<Turnstile> (Ex n. f1 ior f2)) = (\<sigma> i j \<Turnstile> (Ex n. f1) ior (Ex n. f2)) " 
using assms by auto

lemma cexists_cior_dist: 
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
shows " (\<sigma> i j c\<Turnstile> (cEx n. f1 cior f2)) = (\<sigma> i j c\<Turnstile> (cEx n. f1) cior (cEx n. f2)) " 
using assms by auto


lemma exists_schop_imp: 
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
shows " (\<sigma> i j \<Turnstile> (Ex n. (f1 schop f2))) \<Longrightarrow> (\<sigma> i j \<Turnstile> (Ex n. f1) schop (Ex n. f2)) " 
using assms by auto

lemma cexists_cschop_imp: 
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
shows " (\<sigma> i j c\<Turnstile> (cEx n. (f1 cschop f2))) \<Longrightarrow> (\<sigma> i j c\<Turnstile> (cEx n. f1) cschop (cEx n. f2)) " 
using assms by auto

lemma sfrdiamond_defs: 
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
shows " (\<sigma> i j \<Turnstile> rdiamond (inot(skip schop (inot f)))) = 
        ( j <  \<infinity> \<and> (\<exists>k. k = j \<or> (k > j \<and> (\<sigma> j k \<Turnstile> f))))" 
using assms apply auto
apply (metis dual_order.refl less_le_not_le semantics_nl_itl.simps(4) skip_d_def)
by (metis MInfty_neq_eint(1) not_less_iff_gr_or_eq order_le_less skip_defs)

lemma spldiamond_defs: 
 assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
 shows " (\<sigma> i j \<Turnstile> ldiamond (inot ((inot f) schop skip))) = 
         ( i > -\<infinity> \<and> (\<exists>k.  k = i \<or> (k < i \<and> (\<sigma> k i \<Turnstile> f))))"
using assms apply auto
apply fastforce
using add.commute by auto


lemma  strictly_future_eqv_future_sem: 
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
shows "\<exists> f.  (\<sigma> i j sf\<Turnstile> sfrdiamond g) = (\<sigma> i j f\<Turnstile> f)" 
using assms
proof (induction g arbitrary:   j )
case (fintro_d x)
then show ?case apply auto
by (metis semantics_future_nl_itl.simps(2))       
next
case (finot_d g)
then show ?case by auto 
next
case (fior_d g1 g2)
then show ?case apply auto
by metis
next
case (frdiamond_d g)
then show ?case by auto 
qed

lemma  strictly_past_eqv_past_sem: 
assumes " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> "
shows "\<exists> f.  (\<sigma> i j sp\<Turnstile> spldiamond g) = (\<sigma> i j p\<Turnstile> f)" 
using assms
proof (induction g arbitrary:  i  )
case (pintro_d x)
then show ?case apply auto
by (metis semantics_past_nl_itl.simps(2))
next
case (pinot_d g)
then show ?case by auto
next
case (pior_d g1 g2)
then show ?case apply auto by metis
next
case (pldiamond_d g)
then show ?case by auto
qed


lemma state_nl_itl_list_defs: 
 "state_nl_itl_list L \<longleftrightarrow> (\<forall>i<length L. state_nl_itl (L!i)) " 
proof (induct L)
case Nil
then show ?case unfolding state_nl_itl_list_def by simp
next
case (Cons a L)
then show ?case 
  proof -
   have 1: "state_nl_itl_list (a # L) \<longleftrightarrow> state_nl_itl a \<and> state_nl_itl_list L"
      unfolding  state_nl_itl_list_def by simp
   have 2: "(\<forall>i<length (a#L). state_nl_itl ((a#L)!i)) \<longleftrightarrow>
            state_nl_itl ((a#L)!0) \<and>
            (\<forall>i. 0<i \<and> i<length (a#L)\<longrightarrow> state_nl_itl ((a#L)!i))" 
        by (metis length_greater_0_conv list.discI zero_less_iff_neq_zero)
   have 3: " state_nl_itl ((a#L)!0) \<longleftrightarrow> state_nl_itl a " 
      by simp
   have 4: "(\<forall>i. 0<i \<and> i<length (a#L)\<longrightarrow> state_nl_itl ((a#L)!i)) \<longleftrightarrow>
            (\<forall>i<length (L). state_nl_itl (L!i)) " 
     by auto
   show ?thesis 
   using "1" "2" "3" "4" local.Cons by presburger
 qed
qed



lemma big_ior_defs : 
 " (\<sigma> i j \<Turnstile> (big_ior F)) \<longleftrightarrow> (\<exists> k< length F. (\<sigma>  i j \<Turnstile> F ! k))"
proof (induct F)
case Nil
then show ?case unfolding big_ior_def by simp
next
case (Cons a F)
then show ?case 
   proof -
    have 1: "(\<sigma> i j \<Turnstile> (big_ior (a#F))) \<longleftrightarrow> (\<sigma> i j \<Turnstile> a ior (big_ior F))"
        unfolding big_ior_def by simp
    have 2: "(\<exists>k<length (a # F). \<sigma> i j \<Turnstile> (a # F) ! k) \<longleftrightarrow>
               (\<sigma> i j \<Turnstile> (a # F) ! 0) \<or>
               (\<exists>k. 0 < k \<and> k<length (a # F) \<and> (\<sigma> i j \<Turnstile> (a # F) ! k))" 
      by (metis length_greater_0_conv neq_Nil_conv zero_less_iff_neq_zero)  
    have 3: "(\<sigma> i j \<Turnstile> (a # F) ! 0) = (\<sigma> i j\<Turnstile> a)"
       by auto
    have 4: "(\<exists>k. 0 < k \<and> k<length (a # F) \<and> (\<sigma> i j \<Turnstile> (a # F) ! k)) \<longleftrightarrow>
             (\<exists>k<length ( F). (\<sigma> i j \<Turnstile> ( F) ! k))"  
       by auto
         (metis Suc_le_eq diff_Suc_Suc diff_zero gr0_conv_Suc less_Suc_eq_le)
     show ?thesis 
     using "1" "2" "4" local.Cons by force
   qed
qed


lemma big_iand_defs : 
 " (\<sigma> i j \<Turnstile> (big_iand F)) \<longleftrightarrow> (\<forall> k< length F. (\<sigma> i j \<Turnstile> F ! k))"
proof (induct F)
case Nil
then show ?case unfolding big_iand_def by simp
next
case (Cons a F)
then show ?case 
   proof -
    have 1: "(\<sigma> i j \<Turnstile> (big_iand (a#F))) \<longleftrightarrow> (\<sigma> i j \<Turnstile> a iand (big_iand F))"
        unfolding big_iand_def by simp
    have 2: "(\<forall>k<length (a # F). \<sigma> i j \<Turnstile> (a # F) ! k) \<longleftrightarrow>
               (\<sigma> i j \<Turnstile> (a # F) ! 0) \<and>
               (\<forall>k. 0 < k \<and> k<length (a # F) \<longrightarrow> (\<sigma> i j \<Turnstile> (a # F) ! k))" 
      by (metis length_greater_0_conv neq_Nil_conv zero_less_iff_neq_zero)  
    have 3: "(\<sigma> i j \<Turnstile> (a # F) ! 0) = (\<sigma> i j \<Turnstile> a)"
       by auto
    have 4: "(\<forall>k. 0 < k \<and> k<length (a # F) \<longrightarrow> (\<sigma> i j \<Turnstile> (a # F) ! k)) \<longleftrightarrow>
             (\<forall>k<length ( F). (\<sigma> i j \<Turnstile> ( F) ! k))"  
       by auto
     show ?thesis 
     using "1" "2" "4" local.Cons by force
   qed
qed


lemma big_ior_map_iand_zip : 
assumes "length F =length G" 
shows "(\<sigma> i j \<Turnstile>  big_ior (map2 iand_d  F G)) \<longleftrightarrow>
       (\<exists>k < length F. (\<sigma> i j \<Turnstile> F ! k iand G ! k)) "
using assms
proof (induct F arbitrary: G)
case Nil
then show ?case by (simp add: big_ior_defs)
next
case (Cons a F)
then show ?case 
   proof (cases "G=[]")
   case True
   then show ?thesis using Cons.prems by auto
   next
   case False
   then show ?thesis using Cons by (auto simp add: big_ior_defs)
   qed
qed


lemma big_iand_map_iimp_zip : 
assumes "length F =length G" 
shows "(\<sigma> i j \<Turnstile>  big_iand (map2 iimp_d F G)) \<longleftrightarrow>
       (\<forall>k < length F. (\<sigma> i j \<Turnstile> F ! k iimp G ! k)) "
using assms
proof (induct F arbitrary: G)
case Nil
then show ?case by (simp add: big_iand_defs)
next
case (Cons a F)
then show ?case 
   proof (cases "G=[]")
   case True
   then show ?thesis using Cons.prems by auto
   next
   case False
   then show ?thesis using Cons by (auto simp add: big_iand_defs)
   qed
qed



lemma map2_iand_defs : 
 assumes "length F = length G"
 shows "(\<exists>k<length G. \<sigma> i j \<Turnstile> map2 iand_d F G ! k) \<longleftrightarrow>
        (\<exists>k<length G. (\<sigma> i j \<Turnstile>  F !k) \<and> (\<sigma> i j \<Turnstile> G ! k)) "
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
  proof (cases "F=[]")
  case True
  then show ?thesis using Cons.prems by auto
  next
  case False
  then show ?thesis using Cons by auto 
  qed
qed 



lemma map2_iand_inot_defs : 
 assumes "length F = length G"
 shows "(\<exists>k<length G. \<sigma> i j \<Turnstile> map2 (\<lambda>x y. x iand inot y) F G ! k) \<longleftrightarrow>
        (\<exists>k<length G. (\<sigma> i j \<Turnstile> F! k) \<and> \<not>(\<sigma> i j \<Turnstile>  G!k))" 
using assms 
proof (induct G arbitrary: F)
case Nil
then show ?case by simp
next
case (Cons a G)
then show ?case 
  proof (cases "F=[]")
  case True
  then show ?thesis using Cons.prems by auto
  next
  case False
  then show ?thesis using Cons by auto 
  qed
qed 

section \<open>Validity\<close>

definition valid :: " nl_itl \<Rightarrow> bool" ("(\<turnstile> _)" 5)
where "(\<turnstile> f) = (\<forall> \<sigma> i j. i \<le> j \<and> i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity> \<longrightarrow>(\<sigma> i j \<Turnstile> f))"

lemma itl_valid [simp] :
  " (\<turnstile> f) = (\<forall> \<sigma> i j. i \<le> j \<and> i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity> \<longrightarrow> (\<sigma> i j \<Turnstile> f))"
by (simp add: valid_def)

lemma itl_ieq:
  " (\<turnstile> f ieqv g) = (\<forall> \<sigma> i j. i \<le> j \<and> i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity> \<longrightarrow> (\<sigma> i j \<Turnstile> f) = (\<sigma> i j \<Turnstile> g)) "
by (auto simp add: ieqv_d_def iand_d_def ior_d_def inot_d_def) 


lemma big_ior_iimp_subset_eq: 
 assumes "set L1 \<subseteq> set L2"
 shows " \<turnstile> big_ior L1 iimp big_ior L2" 
using assms 
proof (induct  rule: list_induct2')
case 1
then show ?case by simp
next
case (2 x xs)
then show ?case by simp
next
case (3 y ys)
then show ?case by (simp add: big_ior_defs)
next
case (4 x xs y ys)
then show ?case 
  proof (cases "x=y")
  case True
  then show ?thesis using 4 apply (auto simp add: big_ior_defs)
   by (metis diff_Suc_1 in_set_conv_nth length_Cons less_Suc_eq_0_disj list.simps(15) nth_Cons' subsetD)
  next
  case False
  then show ?thesis using 4 apply (auto simp add: big_ior_defs)
   by (metis "4.prems" in_set_conv_nth length_Cons subsetD)
  qed
qed


lemma big_ior_ieqv_set_eq: 
 assumes "set L1 = set L2"
 shows " \<turnstile> big_ior L1 ieqv big_ior L2" 
using assms 
by (metis big_ior_iimp_subset_eq dual_order.refl iand_defs ieqv_d_def valid_def)


lemma big_ior_ieqv_remdups: 
 shows " \<turnstile> big_ior L ieqv big_ior (remdups L)"
by (metis big_ior_ieqv_set_eq set_remdups)
 


lemma big_iand_iimp_subset_eq: 
 assumes "set L1 \<subseteq> set L2"
 shows " \<turnstile> big_iand L2 iimp big_iand L1" 
using assms 
proof (induct  rule: list_induct2')
case 1
then show ?case by simp
next
case (2 x xs)
then show ?case by (simp add: big_iand_defs)
next
case (3 y ys)
then show ?case by simp
next
case (4 x xs y ys)
then show ?case 
  proof (cases "x=y")
  case True
  then show ?thesis using 4 apply (auto simp add: big_iand_defs ) 
    using 4 all_nth_imp_all_set length_Cons  subsetD 
    by (metis nth_mem)
  next
  case False
  then show ?thesis using 4 apply (auto simp add: big_iand_defs)
   by (metis "4.prems" all_nth_imp_all_set length_Cons nth_mem subsetD)
  qed
qed



lemma big_iand_ieqv_set_eq: 
 assumes "set L1 = set L2"
 shows " \<turnstile> big_iand L1 ieqv big_iand L2" 
using assms 
by (metis big_iand_iimp_subset_eq dual_order.refl iand_defs ieqv_d_def valid_def)

lemma big_iand_ieqv_remdups: 
 " \<turnstile> big_iand L ieqv big_iand (remdups L)"
 by (metis big_iand_ieqv_set_eq set_remdups)


lemma linf_subst: 
 assumes "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_nl_itl w" 
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j \<Turnstile>  suform linf n w) = ((upd \<sigma> n (\<lambda> c. (\<sigma> c c \<Turnstile> w) ) )  i j \<Turnstile> linf )" 
using assms
unfolding linf_d_def iand_d_def lfinite_d_def itrue_d_def empty_d_def more_d_def ior_d_def inot_d_def
by simp

lemma linf_subst_1: 
 assumes "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_nl_itl w" 
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j \<Turnstile>  suform linf n w) =  (\<sigma> i j \<Turnstile> linf)" 
using assms
unfolding linf_d_def iand_d_def lfinite_d_def itrue_d_def empty_d_def more_d_def ior_d_def inot_d_def
by simp

lemma state_nl: 
assumes "state_nl_itl w" 
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        " (\<sigma> i j \<Turnstile> $x) " 
shows   " i \<noteq>- \<infinity> "
using assms by simp


lemma lfinite_subst_2: 
 assumes "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_nl_itl w" 
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j \<Turnstile>  suform ($x) n (w iand lfinite)) =  ((upd \<sigma> n (\<lambda> c. (\<sigma> c c \<Turnstile> w) ) )  i j \<Turnstile> $x )"
proof -
 have 1: "(\<sigma> i j \<Turnstile> suform ($x) n (w iand lfinite)) = 
          (if x =n then (\<sigma> i j \<Turnstile> (w iand lfinite)) else (i \<noteq> - \<infinity> \<and> x \<in> \<sigma> (int_of_eint i)))" 
   by simp    
 have 15: "(if x =n then (\<sigma> i j \<Turnstile> (w iand lfinite)) else (i \<noteq> - \<infinity> \<and> x \<in> \<sigma> (int_of_eint i))) =
           ( i\<noteq> -\<infinity> \<and> (if x =n then (\<sigma> i j \<Turnstile> (w )) else ( x \<in> \<sigma> (int_of_eint i))))"
     using assms(4) assms(5) assms(6) lfinite_defs by force 
 have 2: "((upd \<sigma> n (\<lambda> c. (\<sigma> c c \<Turnstile> w) ) )  i j \<Turnstile> $x ) = 
          (i \<noteq> - \<infinity> \<and> x \<in> (upd \<sigma> n (\<lambda>x. \<sigma> (eint x) (eint x) \<Turnstile> w)) (int_of_eint i))"
   by simp
 have 3: "i\<noteq> - \<infinity> \<Longrightarrow>
           x \<in> (upd \<sigma> n (\<lambda>x. \<sigma> (eint x) (eint x) \<Turnstile> w)) (int_of_eint i) = 
           (if x =n then (\<sigma> i j \<Turnstile> (w )) else ( x \<in> \<sigma> (int_of_eint i)))
     "
   unfolding  upd_def  apply simp 
   by (metis PInfty_neq_eint(1) abs_eint.elims assms(3) assms(5) assms(6) eint_int state_nl_itl_defs)
 show ?thesis 
  using "1" "15" "2" "3" by blast
qed


lemma subst_suform: 
 assumes "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_nl_itl w" 
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows   " (\<sigma> i j \<Turnstile>  suform f n (w iand lfinite)) = ((upd \<sigma> n (\<lambda> c. (\<sigma> c c \<Turnstile> w) ) )  i j \<Turnstile> f )" 
using assms
proof (induction f arbitrary: \<sigma> w i j n)
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case 
using lfinite_subst_2 by blast
next
case (iimp_d f1 f2)
then show ?case 
by simp
next
case (next_d f)
then show ?case 
 proof -
  have n1: "(\<sigma> i j \<Turnstile> suform (next f) n (w iand lfinite)) = 
            (i \<noteq> - \<infinity> \<and> i < j \<and> (\<sigma> (i + eint 1) j \<Turnstile> suform f n (w iand lfinite)))"
     by simp 
  have n2: "((upd \<sigma> n (\<lambda>x. \<sigma> (eint x) (eint x) \<Turnstile> w)) i j \<Turnstile> next f) = 
            (i \<noteq> - \<infinity> \<and> i < j \<and> ((upd \<sigma> n (\<lambda>x. \<sigma> (eint x) (eint x) \<Turnstile> w)) (i + eint 1) j \<Turnstile> f))"
    by simp
  have n3: "i \<noteq> - \<infinity> \<and> i < j  \<Longrightarrow> i + eint 1 \<le> j" 
     by (metis eint_less_le_alt int_of_eint.cases next_d.prems(5) plus_eint.simps(1))
  have n4: "i \<noteq> - \<infinity> \<and> i < j  \<Longrightarrow>
             (\<sigma> (i + eint 1) j \<Turnstile> suform f n (w iand lfinite)) =
             ((upd \<sigma> n (\<lambda>x. \<sigma> (eint x) (eint x) \<Turnstile> w)) (i + eint 1) j \<Turnstile> f)" 
    using next_d.IH[of n w "(i + eint 1)" j \<sigma>]  
    using n3 next_d.prems(1) next_d.prems(2) next_d.prems(3) next_d.prems(5) next_d.prems(6) by auto
  show ?thesis 
  using n1 n2 n4 by blast   
 qed  
next
case (chop_d f1 f2)
then show ?case by auto
next
case (ldiamond_d f)
then show ?case
  proof - 
    have l1: "(\<sigma> i j \<Turnstile> suform (ldiamond f) n (w iand lfinite)) = 
              (i \<noteq> - \<infinity> \<and> (\<exists>k\<le>i. \<sigma> k i \<Turnstile> suform f n (w iand lfinite)))" 
        by simp
    have l2: "((upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) i j \<Turnstile> ldiamond f) = 
              (i \<noteq> - \<infinity> \<and> (\<exists>k\<le>i. (upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) k i \<Turnstile> f))
             " 
       by simp
    have l3: " i \<noteq> - \<infinity> \<Longrightarrow> 
                 (\<exists>k\<le>i. (\<sigma> k i \<Turnstile> suform f n (w iand lfinite))) =
                 (\<exists>k\<le>i. ((upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) k i \<Turnstile> f))" 
           using ldiamond_d.IH[of n w _ i \<sigma>] 
           using bvars.simps(6) eint_infty_less_eq(1) ldiamond_d.prems(1) ldiamond_d.prems(2) ldiamond_d.prems(3) ldiamond_d.prems(5) by blast
    show ?thesis 
    using l1 l2 l3 by blast
   qed
next
case (rdiamond_d f)
then show ?case 
   proof -
    have r1: "(\<sigma> i j \<Turnstile> suform (rdiamond f) n (w iand lfinite)) = 
              (j \<noteq> \<infinity> \<and> (\<exists>k\<ge>j. \<sigma> j k \<Turnstile> suform f n (w iand lfinite)))"
    by simp 
    have r2: "((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) i j \<Turnstile> rdiamond f) = 
              (j \<noteq> \<infinity> \<and> (\<exists>k\<ge>j. (upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) j k \<Turnstile> f)) " 
     by simp
    have r3: "j \<noteq> \<infinity> \<Longrightarrow>
               (\<exists>k\<ge>j. \<sigma> j k \<Turnstile> suform f n (w iand lfinite)) =
                (\<exists>k\<ge>j. (upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) j k \<Turnstile> f)"
     using rdiamond_d.IH[of n w j _ \<sigma>] 
     using bvars.simps(7) eint_infty_less_eq(2) rdiamond_d.prems(1) rdiamond_d.prems(2) rdiamond_d.prems(3) rdiamond_d.prems(6) by blast
   show ?thesis   
   using r1 r2 r3 by blast
  qed
next
case (chopstar_d f)
then show ?case 
 proof -
  have c1: "(\<sigma> i j \<Turnstile> suform (chopstar f ) n (w iand lfinite)) = 
            (i = j \<or>
     i \<noteq> j \<and> i \<noteq> - \<infinity> \<and> j \<noteq> \<infinity> \<and> 
     (\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
      nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) lrf) \<or>
     i \<noteq> j \<and> i = - \<infinity> \<and> j \<noteq> \<infinity> \<and> 
     (\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and> 
       nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform f n (w iand lfinite))) li) \<or>
     i \<noteq> j \<and> i \<noteq> - \<infinity> \<and> j = \<infinity> \<and> 
     (\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
       nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) ri) \<or>
     i \<noteq> j \<and> i = - \<infinity> \<and> j = \<infinity> \<and>
     (\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and>
         \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform f n (w iand lfinite))) li \<and>
         \<not> nfinite ri \<and> nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform f n (w iand lfinite))) ri))" 
    by simp
  have c2: "((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) i j \<Turnstile> chopstar f ) = 
            (i = j \<or>
     i \<noteq> j \<and> i \<noteq> - \<infinity> \<and> j \<noteq> \<infinity> \<and>
     (\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
        nridx (\<lambda>(a::eint) b::eint. a < b \<and> 
               ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) lrf) \<or>
     i \<noteq> j \<and> i = - \<infinity> \<and> j \<noteq> \<infinity> \<and> 
      (\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and> 
        nridx (\<lambda>(a::eint) b::eint. b < a \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) b a \<Turnstile> f)) li) \<or>
     i \<noteq> j \<and> i \<noteq> - \<infinity> \<and> j = \<infinity> \<and> 
      (\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
        nridx (\<lambda>(a::eint) b::eint. a < b \<and> 
          ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) ri) \<or>
     i \<noteq> j \<and> i = - \<infinity> \<and> j = \<infinity> \<and>
     (\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and> 
         \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint b) (eint a) \<Turnstile> f)) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint a) (eint b) \<Turnstile> f)) ri))"
     by simp
 have c3: "i = j \<Longrightarrow> ?thesis" 
    using c1 c2 by presburger
 have c4: "i\<noteq>j \<Longrightarrow> i \<noteq> - \<infinity> \<and> j \<noteq> \<infinity> \<Longrightarrow> 
           (\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) lrf) \<Longrightarrow>
           (\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) lrf)" 
   proof -
    assume c4a0: "i\<noteq>j"
    assume c4a1: "i \<noteq> - \<infinity> \<and> j \<noteq> \<infinity>"
    assume c4a2: "(\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) lrf)"
    show "(\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) lrf)"
    proof -
     obtain lrf where c41: "nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) lrf"  
     using c4a2 by blast
     have c42: "nridx (\<lambda>(a::eint) b::eint. a < b \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) lrf"   
      using chopstar_d.IH[of n w _ _ \<sigma>]  c41 chopstar_d.prems
      by (metis (no_types, lifting) bvars.simps(8) eint_minus_less_minus eint_uminus_eq_reorder less_eint.simps(2) nridx_expand order_less_imp_le)
     show ?thesis 
     using c41 c42 by blast
    qed
   qed
 have c5: "i\<noteq>j \<Longrightarrow> i \<noteq> - \<infinity> \<and> j \<noteq> \<infinity> \<Longrightarrow> 
           (\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) lrf) \<Longrightarrow>
           (\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) lrf) "
  proof -
   assume c5a0: "i\<noteq>j"
   assume c5a1: "i \<noteq> - \<infinity> \<and> j \<noteq> \<infinity>"
   assume c5a2: "(\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) lrf)"
   show "(\<exists>lrf::eint nellist. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) lrf)"
   proof -
    obtain lrf where c51: "nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> 
             nridx (\<lambda>(a::eint) b::eint. a < b \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) lrf"
     using c5a2 by blast
    have c52: "nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) lrf"
       using chopstar_d.IH[of n w _ _ \<sigma>] c51  chopstar_d.prems 
       by (metis (no_types, lifting) bvars.simps(8) less_eint.simps(2) less_eint.simps(3) nridx_expand order_less_imp_le)     
    show ?thesis using c51 c52 by blast
   qed
  qed
 have c6: "i\<noteq>j \<and> i \<noteq> - \<infinity> \<and> j \<noteq> \<infinity> \<Longrightarrow> ?thesis "
    using c4 c5 apply simp by argo 
 have c7: "i\<noteq>j \<Longrightarrow> i = - \<infinity> \<and> j \<noteq> \<infinity> \<Longrightarrow>
           (\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and>
             nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform f n (w iand lfinite))) li) \<Longrightarrow>
           (\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and> 
              nridx (\<lambda>(a::eint) b::eint. b < a \<and> 
               ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) b a \<Turnstile> f)) li)" 
  proof -
   assume c7a0: "i\<noteq>j" 
   assume c7a1: "i = - \<infinity> \<and> j \<noteq> \<infinity>"
   assume c7a2: "(\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and>
             nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform f n (w iand lfinite))) li)"
   show "(\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and> 
              nridx (\<lambda>(a::eint) b::eint. b < a \<and> 
               ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) b a \<Turnstile> f)) li)"
   proof -
    obtain li where c71: "nnth li (0::nat) = j \<and> \<not> nfinite li \<and>
             nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform f n (w iand lfinite))) li"
     using c7a2 by blast
    have c72: "nridx (\<lambda>(a::eint) b::eint. b < a \<and> 
               ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) b a \<Turnstile> f)) li"
           using chopstar_d.IH[of n w _ _ \<sigma>] c71  chopstar_d.prems 
           by (metis (no_types, lifting) bvars.simps(8) dual_order.strict_iff_order eint_minus_less_minus eint_uminus_uminus less_eint.simps(2) nridx_expand)
    show ?thesis using c71 c72 by blast
   qed
  qed
 have c8: "i\<noteq>j \<Longrightarrow> i = - \<infinity> \<and> j \<noteq> \<infinity> \<Longrightarrow>
           (\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and> 
              nridx (\<lambda>(a::eint) b::eint. b < a \<and> 
               ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) b a \<Turnstile> f)) li) \<Longrightarrow>
           (\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and>
             nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform f n (w iand lfinite))) li) "
  proof -
   assume c8a0: "i\<noteq>j"
   assume c8a1: "i = - \<infinity> \<and> j \<noteq> \<infinity>"
   assume c8a2: "(\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and> 
              nridx (\<lambda>(a::eint) b::eint. b < a \<and> 
               ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) b a \<Turnstile> f)) li)" 
   show "(\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and>
             nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform f n (w iand lfinite))) li)"
   proof -
    obtain li where c81: "nnth li (0::nat) = j \<and> \<not> nfinite li \<and> 
              nridx (\<lambda>(a::eint) b::eint. b < a \<and> 
               ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) b a \<Turnstile> f)) li"  
     using c8a2 by blast
    have c82: "nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform f n (w iand lfinite))) li"
      using chopstar_d.IH[of n w _ _ \<sigma>] c81  chopstar_d.prems
      by (metis (no_types, lifting) bvars.simps(8) less_eint.simps(2) less_eint.simps(3) nridx_expand order_less_imp_le)
    show ?thesis using c81 c82 by blast
   qed
  qed
 have c9: "i\<noteq>j \<Longrightarrow> i = - \<infinity> \<and> j \<noteq> \<infinity> \<Longrightarrow> ?thesis" 
   using c7 c8  apply simp by argo
 have c10: "i\<noteq>j \<Longrightarrow> i \<noteq> - \<infinity> \<and> j = \<infinity> \<Longrightarrow> 
           (\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
       nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) ri) \<Longrightarrow>
         (\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
        nridx (\<lambda>(a::eint) b::eint. a < b \<and> 
          ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) ri)"
  proof -
   assume c10a0: "i\<noteq>j"
   assume c10a1: "i \<noteq> - \<infinity> \<and> j = \<infinity>"
   assume c10a2: "(\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
       nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) ri)"  
   show "(\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
        nridx (\<lambda>(a::eint) b::eint. a < b \<and> 
          ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) ri)"
   proof -
    obtain ri where c101: "nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
       nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) ri"
     using c10a2 by blast
    have c102: "nridx (\<lambda>(a::eint) b::eint. a < b \<and> 
          ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) ri"   
      using chopstar_d.IH[of n w _ _ \<sigma>] c101  chopstar_d.prems 
      by (metis (no_types, lifting) bvars.simps(8) less_eint.simps(2) less_eint.simps(3) nridx_expand order_less_le)
    show ?thesis using c101 c102 by blast
   qed
  qed
 have c11: "i\<noteq>j \<Longrightarrow> i \<noteq> - \<infinity> \<and> j = \<infinity> \<Longrightarrow> 
           (\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
        nridx (\<lambda>(a::eint) b::eint. a < b \<and> 
          ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) ri) \<Longrightarrow>
           (\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
       nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) ri) "
  proof -
  assume c11a0: "i\<noteq>j "
  assume c11a1: "i \<noteq> - \<infinity> \<and> j = \<infinity>"
  assume c11a2: "(\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
        nridx (\<lambda>(a::eint) b::eint. a < b \<and> 
          ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) ri)"
  show "(\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
       nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) ri)"
  proof -
   obtain ri where c111: "nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> 
        nridx (\<lambda>(a::eint) b::eint. a < b \<and> 
          ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) a b \<Turnstile> f)) ri"
   using c11a2 by blast
   have c112: "nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform f n (w iand lfinite))) ri"
    using  chopstar_d.IH[of n w _ _ \<sigma>] c111  chopstar_d.prems  
    by (metis (no_types, lifting) bvars.simps(8) less_eint.simps(2) less_eint.simps(3) nridx_expand order_less_imp_le) 
   show ?thesis using c111 c112 by blast   
  qed
 qed
 have c12: "i\<noteq>j \<and> i \<noteq> - \<infinity> \<and> j = \<infinity> \<Longrightarrow> ?thesis"
   using c1 c2 apply simp 
   using c10 c11 by argo
 have c13: "i \<noteq>j \<Longrightarrow> i = - \<infinity> \<and> j = \<infinity> \<Longrightarrow>
            (\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and> \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform f n (w iand lfinite))) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform f n (w iand lfinite))) ri) \<Longrightarrow>
         (\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and> 
         \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint b) (eint a) \<Turnstile> f)) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint a) (eint b) \<Turnstile> f)) ri)"
  proof -
   assume c13a0: "i \<noteq>j"
   assume c13a1: "i = - \<infinity> \<and> j = \<infinity>"
   assume c13a2: "(\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and> \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform f n (w iand lfinite))) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform f n (w iand lfinite))) ri)"
   show "(\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and> 
         \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint b) (eint a) \<Turnstile> f)) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint a) (eint b) \<Turnstile> f)) ri)"
   proof -
    obtain li ri where c131: "nnth li (0::nat) = nnth ri (0::nat) \<and> \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform f n (w iand lfinite))) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform f n (w iand lfinite))) ri"
    using c13a2 by blast
   have c132: "nridx (\<lambda>(a::int) b::int. b < a \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint b) (eint a) \<Turnstile> f)) li"
    using  chopstar_d.IH[of n w _ _ \<sigma>] c131  chopstar_d.prems unfolding nridx_expand 
      by simp  auto
   have c133: "nridx (\<lambda>(a::int) b::int. a < b \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint a) (eint b) \<Turnstile> f)) ri"
     using  chopstar_d.IH[of n w _ _ \<sigma>] c131  chopstar_d.prems unfolding nridx_expand by auto
   show ?thesis using c131 c132 c133 by blast
  qed  
 qed
 have c14: "i \<noteq>j \<Longrightarrow> i = - \<infinity> \<and> j = \<infinity> \<Longrightarrow>
            (\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and> 
         \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint b) (eint a) \<Turnstile> f)) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint a) (eint b) \<Turnstile> f)) ri) \<Longrightarrow>
            (\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and> \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform f n (w iand lfinite))) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform f n (w iand lfinite))) ri) "
   proof -
    assume c14a0: "i \<noteq>j"
    assume c14a1: "i = - \<infinity> \<and> j = \<infinity>"
    assume c14a2: "(\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and> 
         \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint b) (eint a) \<Turnstile> f)) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint a) (eint b) \<Turnstile> f)) ri)"
    show "(\<exists>(li::int nellist) ri::int nellist.
         nnth li (0::nat) = nnth ri (0::nat) \<and> \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform f n (w iand lfinite))) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform f n (w iand lfinite))) ri)"
    proof -
     obtain li ri where c141: "nnth li (0::nat) = nnth ri (0::nat) \<and> 
         \<not> nfinite li \<and>
         nridx (\<lambda>(a::int) b::int. b < a \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint b) (eint a) \<Turnstile> f)) li \<and>
         \<not> nfinite ri \<and> 
         nridx (\<lambda>(a::int) b::int. a < b \<and> 
         ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> w)) (eint a) (eint b) \<Turnstile> f)) ri"
      using c14a2 by blast
     have c142: "nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform f n (w iand lfinite))) li"
      using   chopstar_d.IH[of n w _ _ \<sigma>] c141  chopstar_d.prems  unfolding nridx_expand by auto
     have c143: "nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform f n (w iand lfinite))) ri"
      using   chopstar_d.IH[of n w _ _ \<sigma>] c141  chopstar_d.prems  unfolding nridx_expand by auto
     show ?thesis using c141 c142 c143 by blast
    qed
   qed
  have c15: "i \<noteq>j \<Longrightarrow> i = - \<infinity> \<and> j = \<infinity> \<Longrightarrow> ?thesis" 
    using c13 c14 apply simp by argo
  show ?thesis 
  using c12 c15 c3 c6 c9 by argo
 qed      
next
case (exists_d x1 f)
then show ?case 
    proof (cases "n= x1")
    case True
    then show ?thesis using exists_d by simp
    next
    case False
    then show ?thesis 
      proof -
       have e4: "n \<notin> bvars f"
         using exists_d.prems(1) by auto 
       have e5: " ((upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) i j \<Turnstile> Ex x1. f)  =  
         (\<exists>l.  (upd (upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) x1 l) i j \<Turnstile> f)"
         using exist_sigma_exist_value by auto  
       have e6: "(\<And>z. z \<in> fvars w \<Longrightarrow> z \<notin> bvars f)"
         using exists_d.prems(2) by auto 
       have e60: " x1 \<notin> fvars w"
         using exists_d.prems(2) by auto
       have e7: "(\<sigma> i j \<Turnstile> Ex x1. suform f n (w iand lfinite)) =
                 (\<exists> l.  (upd \<sigma> x1 l) i j \<Turnstile> suform f n (w iand lfinite))"
        using exist_sigma_exist_value by auto 
       have e8: "(\<exists> l.  (upd \<sigma> x1 l) i j \<Turnstile> suform f n (w iand lfinite)) \<Longrightarrow>
                (\<exists>l.  (upd (upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) x1 l) i j \<Turnstile> f) "
       proof -
        assume ae0: "(\<exists> l.  (upd \<sigma> x1 l) i j \<Turnstile> suform f n (w iand lfinite)) "  
        show "(\<exists>l.  (upd (upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) x1 l) i j \<Turnstile> f)"
       proof -
        obtain l where e9: "(upd \<sigma> x1 l) i j \<Turnstile> suform f n (w iand lfinite)"  
          using ae0 by blast
        have e10: "((upd (upd \<sigma> x1 l) n (\<lambda>c. (upd \<sigma> x1 l) (eint c) (eint c) \<Turnstile> w)) i j \<Turnstile> f) " 
          using exists_d.IH[of n w i j "(upd \<sigma> x1 l)"] 
          using e4 e6 e9 exists_d.prems(3) exists_d.prems(4) exists_d.prems(5) exists_d.prems(6) by fastforce  
        have e11: "(upd (upd \<sigma> x1 l) n (\<lambda>c. (upd \<sigma> x1 l) (eint c) (eint c) \<Turnstile> w)) =
                   upd (upd \<sigma> n (\<lambda>c. (upd \<sigma> x1 l) (eint c) (eint c) \<Turnstile> w)) x1 l" 
        using upd_swap[of x1 n \<sigma> l "(\<lambda>c. (upd \<sigma> x1 l) (eint c) (eint c) \<Turnstile> w)"] 
        using False by blast
        have e12: "( upd (upd \<sigma> n (\<lambda>c. (upd \<sigma> x1 l) (eint c) (eint c) \<Turnstile> w)) x1 l) =
                   (upd (upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) x1 l)"
          using e60 not_fvar_upd by auto  
        show ?thesis 
        using e10 e11 e12 by auto
       qed
      qed
     have e13: "(\<exists>l.  (upd (upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) x1 l) i j \<Turnstile> f) \<Longrightarrow>
                (\<exists> l.  (upd \<sigma> x1 l) i j \<Turnstile> suform f n (w iand lfinite)) "
     proof -
      assume ae1: "(\<exists>l.  (upd (upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) x1 l) i j \<Turnstile> f)"
      show "(\<exists> l.  (upd \<sigma> x1 l) i j \<Turnstile> suform f n (w iand lfinite))"
      proof -
       obtain l where e14: "(upd (upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) x1 l) i j \<Turnstile> f"
        using ae1 by blast
       have e15: "( upd (upd \<sigma> n (\<lambda>c. (upd \<sigma> x1 l) (eint c) (eint c) \<Turnstile> w)) x1 l) =
                   (upd (upd \<sigma> n (\<lambda>c. \<sigma> (eint c) (eint c) \<Turnstile> w)) x1 l)"
        using e60 not_fvar_upd by auto 
       have e16: "(upd (upd \<sigma> n (\<lambda>c. (upd \<sigma> x1 l) (eint c) (eint c) \<Turnstile> w)) x1 l) =
                  (upd (upd \<sigma> x1 l) n (\<lambda>c. (upd \<sigma> x1 l) (eint c) (eint c) \<Turnstile> w)) "
         using False upd_swap by auto 
       have e17: "((upd (upd \<sigma> x1 l) n (\<lambda>c. (upd \<sigma> x1 l) (eint c) (eint c) \<Turnstile> w)) i j \<Turnstile> f)"
          using e14 e15 e16 by auto 
       have e18: "(upd \<sigma> x1 l) i j \<Turnstile> suform f n (w iand lfinite)"
         by (simp add: e17 e4 e6 exists_d.IH exists_d.prems(3) exists_d.prems(4) exists_d.prems(5) exists_d.prems(6))
        show ?thesis 
       using e18 by auto 
      qed
     qed
     show ?thesis 
     using e13 e5 e7 e8 by auto     
    qed   
    qed
qed

lemma state_nl_itl_point_interval: 
fixes i :: int
assumes "state_nl_itl w" 
        " \<sigma>' i = \<sigma> i " 
shows " (\<sigma>' i i \<Turnstile> w) \<longleftrightarrow>(\<sigma> i i \<Turnstile> w) " 
using assms 
proof (induction w arbitrary: \<sigma> \<sigma>' i)
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case by simp
next
case (iimp_d w1 w2)
then show ?case  
using semantics_nl_itl.simps(3) state_nl_itl.simps(3) by blast
next
case (next_d w)
then show ?case by simp
next
case (chop_d w1 w2)
then show ?case by simp
next
case (ldiamond_d w)
then show ?case by simp
next
case (rdiamond_d w)
then show ?case by simp
next
case (chopstar_d w)
then show ?case by simp
next
case (exists_d x1 w)
then show ?case 
  proof -
   have 1: "(\<sigma> (eint i) (eint i) \<Turnstile> Ex x1. w) = (\<exists>l. ((upd \<sigma> x1 l) (eint i) (eint i) \<Turnstile> w))" (is " _ = ?lhs")
    by simp
   have 2: "(\<sigma>' (eint i) (eint i) \<Turnstile> Ex x1. w) = (\<exists>l. ((upd \<sigma>' x1 l) (eint i) (eint i) \<Turnstile> w))" (is " _ = ?rhs")
      by simp
   have 3: "?lhs \<Longrightarrow> ?rhs"     
    proof -
     assume a: ?lhs
     show ?rhs 
     proof -
      obtain l where 4: "((upd \<sigma> x1 l) (eint i) (eint i) \<Turnstile> w)" 
        using a by blast
      have 5: "(upd \<sigma> x1 l) i = (upd \<sigma>' x1 l) i"
         using exists_d.prems(2) upd_def by presburger
       show ?thesis 
       using "4" "5" exists_d.IH exists_d.prems(1) state_nl_itl.simps(9) by blast 
     qed
    qed
   have 4: "?rhs \<Longrightarrow> ?lhs" 
    proof -
     assume b: ?rhs
     show ?lhs
     proof -
      obtain l where 6: "((upd \<sigma>' x1 l) (eint i) (eint i) \<Turnstile> w)" 
        using b by blast
      have 7: "(upd \<sigma>' x1 l) i = (upd \<sigma> x1 l) i"
       using exists_d.prems(2) upd_def by presburger 
      show ?thesis  
      using "6" "7" exists_d.IH exists_d.prems(1) state_nl_itl.simps(9) by blast 
    qed
   qed
   show ?thesis 
   using "1" "2" "3" "4" by blast
  qed
qed 


lemma exists_elim_state_nl_itl_help:
assumes 
        "  i \<noteq> \<infinity> "
       "state_nl_itl w"
         "n \<notin> bvars w" 
       " i\<noteq> -\<infinity>" 
shows " (\<exists>l. ((upd \<sigma> n l) i i \<Turnstile> w)) \<longleftrightarrow>
                 (((upd \<sigma> n (\<lambda>x::int. True)) i i \<Turnstile> w) \<or>  ((upd \<sigma> n (\<lambda>x::int. False)) i i \<Turnstile> w))" 
using assms 
proof (induction w arbitrary: \<sigma> n i)
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case using upd_def by auto
next
case (iimp_d w1 w2)
then show ?case 
  proof -
   have 1: "state_nl_itl  w1"
     using iimp_d.prems(2) by auto 
   have 2: "state_nl_itl  w2"
     using iimp_d.prems(2) by auto 
   have 3: " n \<notin> bvars  w1 " 
     using iimp_d.prems(3) by simp
   have 4: "n \<notin> bvars  w2"
     using iimp_d.prems(3) by simp
   have 5: "(\<exists>l. (upd \<sigma> n l) i i \<Turnstile>  w1 iimp w2 ) = 
            (\<exists>l. ((upd \<sigma> n l) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n l) i i \<Turnstile> w2))" 
     by simp
   have 6: "((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile>  w1 iimp w2 ) = 
           (((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w2))" 
    by simp
   have 7: "((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile>  w1 iimp w2 ) = 
           (((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w2))" 
    by simp
   have 8: "(((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w2)) \<Longrightarrow>
             (\<exists>l. ((upd \<sigma> n l) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n l) i i \<Turnstile> w2))"
      by blast
   have 9: "(((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w2)) \<Longrightarrow>
             (\<exists>l. ((upd \<sigma> n l) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n l) i i \<Turnstile> w2))"
       by blast
   have 10: "(\<exists>l. ((upd \<sigma> n l) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n l) i i \<Turnstile> w2)) \<Longrightarrow>
             (((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w2)) \<or>
             (((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w2))"
    proof -
     assume a1: "(\<exists>l. ((upd \<sigma> n l) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n l) i i \<Turnstile> w2))"
     show "(((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w2)) \<or>
             (((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w2))"
     proof -
      obtain l where 11: "((upd \<sigma> n l) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n l) i i \<Turnstile> w2)"
       using a1 by blast
      have 111: "\<exists> i1 . eint i1 = i"
      by (metis MInfty_eq_minfinity PInfty_eq_infinity eint.exhaust iimp_d.prems(1,4))
      obtain i1 where 112: "eint i1 = i"
       using 111 by blast
      have 12 : "((upd \<sigma> n l) i i \<Turnstile> w1) \<Longrightarrow> 
                  (((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w1) \<or> ((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w1))"  
         using "1" "3" iimp_d.IH(1) iimp_d.prems(1)  
         using iimp_d.prems(4) by blast
      have 13: "((upd \<sigma> n l) i i \<Turnstile> w2) \<Longrightarrow> 
                  (((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w2) \<or> ((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w2))"
         using "2" "4" iimp_d.IH(2) iimp_d.prems(1)  using iimp_d.prems(4) by blast
      have 14: "(((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w2) \<or> ((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w2)) \<Longrightarrow> 
                  (((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w2) \<or> ((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w2))"
        by (simp add: "11" "13") 
      have 141: " (upd \<sigma> n (\<lambda>x. True)) i1 \<noteq> (upd \<sigma> n (\<lambda>x. False)) i1" 
        unfolding upd_def by auto 
     have 15: "(((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. True)) i i \<Turnstile> w2)) \<or>
             (((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w1) \<longrightarrow> ((upd \<sigma> n (\<lambda>x. False)) i i \<Turnstile> w2))"  
      using 1 11 12 13 112 141 iimp_d.prems state_nl_itl_point_interval[of w1 _ i1]
        by (metis upd_def)
        show ?thesis 
        using "15" by blast   
     qed     
     qed
   show ?thesis 
   using "10" "5" "6" "7" by blast
  qed
next
case (next_d w)
then show ?case by simp
next
case (chop_d w1 w2)
then show ?case by simp
next
case (ldiamond_d w)
then show ?case by simp
next
case (rdiamond_d w)
then show ?case by simp
next
case (chopstar_d w)
then show ?case by simp
next
case (exists_d x1 w)
then show ?case apply simp
by (metis upd_swap)
qed


lemma suform_lfinite_export: 
 " \<turnstile> suform (f iand lfinite) n g ieqv (lfinite iand (suform f n g)) " 
unfolding valid_def unfolding iand_d_def lfinite_d_def inot_d_def ior_d_def itrue_d_def empty_d_def more_d_def
apply simp 
by meson


lemma subst_suform_state: 
 assumes "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_nl_itl w" 
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        "state_nl_itl f" 
 shows   " (\<sigma> i j \<Turnstile>  suform (f iand lfinite) n (w)) = (\<sigma> i j \<Turnstile>  suform (f iand lfinite) n (w iand lfinite)) " 
using assms
proof (induction f arbitrary: \<sigma> )
case false_d
then show ?case 
using itl_ieq suform.simps(1) suform_lfinite_export by presburger
next
case (atom_d x)
then show ?case 
using suform_lfinite_export by auto
next
case (iimp_d f1 f2)
then show ?case 
 proof -
  have i1: "(\<sigma> i j \<Turnstile> suform ((inot f1) iand lfinite) n w) = (\<sigma> i j \<Turnstile>  suform ((inot f1) iand lfinite) n (w iand lfinite))" 
    using iimp_d inot_d_def suform_lfinite_export by force
  have i2: "(\<sigma> i j \<Turnstile> suform (f2 iand lfinite) n w) = (\<sigma> i j \<Turnstile>  suform (f2 iand lfinite) n (w iand lfinite))" 
    using iimp_d by force
  show ?thesis 
  using i1 i2 iimp_d.prems(4) iimp_d.prems(5) iimp_d.prems(6) inot_d_def suform_lfinite_export by auto
 qed
next
case (next_d f)
then show ?case by simp
next
case (chop_d f1 f2)
then show ?case by simp
next
case (ldiamond_d f)
then show ?case by simp
next
case (rdiamond_d f)
then show ?case by simp
next
case (chopstar_d f)
then show ?case by simp
next
case (exists_d x1 f)
then show ?case 
 proof -
   have e1: "(\<sigma> i j \<Turnstile> suform ((Ex x1. f) iand lfinite) n w) =
             (\<sigma> i j \<Turnstile> suform ((Ex x1. f iand lfinite)) n w)"
      using exists_d.prems(4) exists_d.prems(5) exists_d.prems(6) suform_lfinite_export by auto 
   have e2: "(\<sigma> i j \<Turnstile> suform ((Ex x1. f) iand lfinite) n (w iand lfinite)) =
             (\<sigma> i j \<Turnstile> suform ((Ex x1. f iand lfinite)) n (w iand lfinite))" 
     using exists_d.prems(4) exists_d.prems(5) exists_d.prems(6) suform_lfinite_export by force
   have e3: "(\<sigma> i j \<Turnstile> suform (f iand lfinite) n w) = (\<sigma> i j \<Turnstile> suform (f iand lfinite) n (w iand lfinite))"
        using exists_d by auto
    show ?thesis 
    using e1 e2 exists_d.IH exists_d.prems(1) exists_d.prems(2) exists_d.prems(3) exists_d.prems(4) exists_d.prems(5) exists_d.prems(6) exists_d.prems(7) by force
   qed
qed
 



lemma subst_suform_state_1: 
 assumes "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_nl_itl w" 
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
        "state_nl_itl f" 
 shows   " (\<sigma> i j \<Turnstile>  suform (f iand lfinite) n (w)) = ((upd \<sigma> n (\<lambda> c. (\<sigma> c c \<Turnstile> w ) ) )  i j \<Turnstile> f iand lfinite)" 
proof -
 have 1: "(n::nat) \<notin> bvars ((f::nl_itl) iand lfinite)" 
    unfolding iand_d_def inot_d_def lfinite_d_def itrue_d_def empty_d_def more_d_def 
    by (simp add: assms(1) inot_d_def ior_d_def)
 have 2: "(\<And>z::nat. z \<in> fvars (w::nl_itl) \<Longrightarrow> z \<notin> bvars (f iand lfinite))" 
   unfolding iand_d_def inot_d_def lfinite_d_def itrue_d_def empty_d_def more_d_def ior_d_def
    by (simp add: assms(2))
 have 3: " (\<sigma> i j \<Turnstile>  suform (f iand lfinite) n (w iand lfinite)) = ((upd \<sigma> n (\<lambda> c. (\<sigma> c c \<Turnstile> w ) ) )  i j \<Turnstile> f iand lfinite)" 
    using subst_suform[of n "f iand lfinite" w i j \<sigma>] assms 
    using "1" "2" by blast
 have 4: "(\<sigma> i j \<Turnstile>  suform (f iand lfinite) n (w iand lfinite)) =
          (\<sigma> i j \<Turnstile>  suform (f iand lfinite) n (w ))"
   by (simp add: assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) assms(7) subst_suform_state)
 show ?thesis 
 using "3" "4" by blast
qed


lemma rename_nl_pitl: 
 assumes "v1 \<notin> bvars f"
         "v2 \<notin> bvars f"
         "v1 \<noteq> v2"
         "v2 \<notin> fvars f" 
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j \<Turnstile> Ex v1. f) = (\<sigma> i j\<Turnstile> Ex v2. suform f v1 ($v2 iand lfinite)) "
proof -
 have 1: " (\<sigma> i j \<Turnstile> Ex v1. f) = (\<exists> l. ((upd \<sigma> v1 l) i j \<Turnstile> f))" 
   using exist_sigma_exist_value by auto
 have 2: "(\<sigma> i j\<Turnstile> Ex v2. suform f v1 ($v2 iand lfinite)) = 
          (\<exists>l. ((upd \<sigma> v2 l) i j \<Turnstile> suform f v1 ($v2 iand lfinite) )) "
    using exist_sigma_exist_value by auto
 have 3: "(\<exists>l. ((upd \<sigma> v2 l) i j \<Turnstile> suform f v1 ($v2 iand lfinite) )) \<Longrightarrow>
           (\<exists> l. ((upd \<sigma> v1 l) i j \<Turnstile> f))   " 
  proof -
   assume a0: "(\<exists>l. ((upd \<sigma> v2 l) i j \<Turnstile> suform f v1 ($v2 iand lfinite) ))"
   show "(\<exists> l. ((upd \<sigma> v1 l) i j \<Turnstile> f))"
   proof -
    obtain l where 4: "((upd \<sigma> v2 l) i j \<Turnstile> suform f v1 ($v2 iand lfinite) )"
     using a0 by blast
     have 5: "((upd \<sigma> v2 l) i j \<Turnstile> suform f v1 ($v2 iand lfinite) ) = 
               ((upd (upd \<sigma> v2 l) v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) i j \<Turnstile> f)"
      using subst_suform[of v1 f "$v2" i j  "(upd \<sigma> v2 l)"   ] 
     by (simp add: assms(1) assms(2) assms(5) assms(6) assms(7))
    have 6: "(upd (upd \<sigma> v2 l) v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) = 
             upd (upd \<sigma> v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) v2 l"
      using upd_swap[of v2 v1 \<sigma> l "(\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)"]
    using assms(3) by blast
    have 8: "((upd (upd \<sigma> v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) v2 l) i j \<Turnstile> f) =
             ( (upd \<sigma> v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) i j \<Turnstile> f)"
      using not_fvar_upd[of v2 f ] using assms(4) by blast
    show ?thesis using "4" "5" "6" "8" by auto
   qed
  qed
 have 9: "(\<exists> l. ((upd \<sigma> v1 l) i j \<Turnstile> f)) \<Longrightarrow>
          (\<exists>l. ((upd \<sigma> v2 l) i j \<Turnstile> suform f v1 ($v2 iand lfinite) ))"
 proof -
  assume a1: "(\<exists> l. ((upd \<sigma> v1 l) i j \<Turnstile> f))"
  show "(\<exists>l. ((upd \<sigma> v2 l) i j \<Turnstile> suform f v1 ($v2 iand lfinite) ))"
  proof -
   obtain l where 10: "((upd \<sigma> v1 l) i j \<Turnstile> f)" 
    using a1 by blast
   have 11: "( (upd \<sigma> v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) i j \<Turnstile> f) =
             ((upd (upd \<sigma> v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) v2 l) i j \<Turnstile> f) 
             "
    using not_fvar_upd[of v2 f ] using assms(4) by blast
   have 12: "\<And>k. (upd \<sigma> v1 l) k = (upd \<sigma> v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) k"
      using upd_def by fastforce
   have 13: "(upd \<sigma> v1 l) = (upd \<sigma> v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2))"
    using 12 by blast   
   have 14: "((upd \<sigma> v2 l) i j \<Turnstile> suform f v1 ($v2 iand lfinite) ) =
             ((upd (upd \<sigma> v2 l) v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) i j \<Turnstile> f)"
      using subst_suform[of v1 f "$v2" i j  "(upd \<sigma> v2 l)" ]
        by (simp add: assms(1) assms(2) assms(5) assms(6) assms(7))
   have 15: "(upd (upd \<sigma> v2 l) v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) =
             (upd (upd \<sigma> v1 (\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)) v2 l)"
     using upd_swap[of v2 v1 \<sigma> l "(\<lambda>x::int. (upd \<sigma> v2 l) (eint x) (eint x) \<Turnstile> $v2)"]
    using assms(3) by blast
   show ?thesis 
   using "10" "11" "13" "14" "15" by auto
  qed
 qed
 show ?thesis
 using "1" "2" "3" "9" by blast 
qed


lemma atom_lfinite_absorb: 
 "\<turnstile> ($n) ieqv $n iand lfinite" 
unfolding valid_def apply simp by meson 

lemma suform_ieqv_sem: 
 assumes "\<And> \<sigma> i j . i \<le> j \<and>  i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity> \<Longrightarrow> (\<sigma> i j \<Turnstile>  f) = (\<sigma> i j \<Turnstile>  g)"
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows   "(\<sigma> i j \<Turnstile> (suform h n f)) = (\<sigma> i j \<Turnstile>  (suform h n g))" 
using assms 
proof (induction h arbitrary: \<sigma> i j )
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case by simp
next
case (iimp_d h1 h2)
then show ?case 
 proof -
  have i1: "(\<sigma> i j \<Turnstile>  suform (inot h1) n f ieqv suform (inot h1) n g) " 
      using assms iimp_d.IH(1) unfolding inot_d_def using iimp_d.prems by auto
  have i2: "(\<sigma> i j \<Turnstile>  suform (h2) n f ieqv suform (h2) n g) " 
     using assms iimp_d.IH(1) iimp_d.prems using iimp_d.IH(2) ieqv_defs by presburger
  show ?thesis using i1 i2 inot_d_def by auto
 qed
next
case (next_d h)
then show ?case 
   proof -
    have n1: "(\<sigma> i j \<Turnstile> suform (next h) n f) = (i \<noteq> - \<infinity> \<and> i < j \<and> (\<sigma> (i + eint (1)) j \<Turnstile> suform h n f))" 
       by simp
    have n2: "(\<sigma> i j \<Turnstile> suform (next h) n g) = (i \<noteq> - \<infinity> \<and> i < j \<and> (\<sigma> (i + eint (1)) j \<Turnstile> suform h n g))"
       by simp 
    have n3: "i \<noteq> - \<infinity> \<and> i < j \<Longrightarrow> (\<sigma> (i + eint (1)) j \<Turnstile> suform h n f) = (\<sigma> (i + eint (1)) j \<Turnstile> suform h n g)"
      using next_d.IH[of "(i + eint (1))" j \<sigma>] next_d.prems apply auto 
      apply (metis eint_less_le_alt int_of_eint.cases plus_eint.simps(1))
      by (metis eint_less_le_alt int_of_eint.cases plus_eint.simps(1))
    show ?thesis
    using n1 n2 n3 by blast
  qed
next
case (chop_d h1 h2)
then show ?case by auto 
next
case (ldiamond_d h)
then show ?case apply auto
apply (metis eint_less_eq(1) less_eq_eint_def linorder_not_le)
by (metis eint_less_eq(1) linorder_not_le order_less_le)
next
case (rdiamond_d h)
then show ?case apply auto
apply (metis less_eint.simps(3) linorder_not_le not_less_iff_gr_or_eq)
by (metis eint_less_eq(2) linorder_not_less order_less_le)
next
case (chopstar_d h)
then show ?case 
   proof -
    have c1: "i = j \<Longrightarrow> ?thesis" 
     using semantics_nl_itl.simps(8) suform.simps(8) by presburger
    have c10: "i\<noteq>j \<and> i \<noteq> -\<infinity> \<and> j \<noteq> \<infinity> \<Longrightarrow> ?thesis"
       proof -
        assume c10a: "i\<noteq>j \<and> i \<noteq> -\<infinity> \<and> j \<noteq> \<infinity>"
        show "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f) = (\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"
        proof -
         have c11: "\<sigma> i j \<Turnstile> suform (chopstar h ) n f \<Longrightarrow> \<sigma> i j \<Turnstile> suform (chopstar h ) n g"
          proof -
           assume c11a: "\<sigma> i j \<Turnstile> suform (chopstar h ) n f"
           show "\<sigma> i j \<Turnstile> suform (chopstar h ) n g"
           proof -
            have c12: "\<exists>lrf. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n f)) lrf"
              using c10a c11a by simp 
            obtain lrf where c13: "nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n f)) lrf" 
            using c12 by blast
            have c14: "nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n g)) lrf"  
              using chopstar_d.IH c13 nridx_expand 
              by (metis (no_types, lifting) chopstar_d.prems(1) eint_less_eq(1) less_eint.simps(3) linorder_not_le order_less_le)
            have c15: "\<exists>lrf. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n g)) lrf"
              using c14 c13 by blast
           show ?thesis using c15 c10a by simp  
          qed
         qed
        have c16: " \<sigma> i j \<Turnstile> suform (chopstar h ) n g \<Longrightarrow> \<sigma> i j \<Turnstile> suform (chopstar h ) n f"
         proof -
          assume c16a: "\<sigma> i j \<Turnstile> suform (chopstar h ) n g"
          show "\<sigma> i j \<Turnstile> suform (chopstar h ) n f"
          proof -
           have c17: "\<exists>lrf. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n g)) lrf"
           using c10a c16a by simp
           obtain lrf where c18: "nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n g)) lrf"
           using c17 by blast
           have c19: "nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n f)) lrf" 
             using chopstar_d.IH c18 nridx_expand 
              by (metis (no_types, lifting) chopstar_d.prems(1) eint_less_eq(1) less_eint.simps(3) linorder_not_le order_less_le)
           have c19a: "\<exists>lrf. nnth lrf (0::nat) = i \<and> nfinite lrf \<and> nlast lrf = j \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n f)) lrf"
             using c18 c19 by blast
           show ?thesis using c19a c10a by simp
          qed  
         qed
       show ?thesis 
       using c11 c16 by blast
        qed
       qed      
    have c20: "i\<noteq>j \<and> i = -\<infinity> \<and> j \<noteq> \<infinity> \<Longrightarrow> ?thesis" 
      proof -
       assume c20a: "i\<noteq>j \<and> i = -\<infinity> \<and> j \<noteq> \<infinity>" 
       show "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f) = (\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"
       proof -
        have c21: "\<sigma> i j \<Turnstile> suform (chopstar h ) n f \<Longrightarrow> \<sigma> i j \<Turnstile> suform (chopstar h ) n g" 
         proof -
          assume c21a: "\<sigma> i j \<Turnstile> suform (chopstar h ) n f" 
          show "\<sigma> i j \<Turnstile> suform (chopstar h ) n g" 
          proof -
           have c22: "(\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and> nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform h n f)) li)" 
           using c20a c21a by fastforce
           obtain li where c23: "nnth li (0::nat) = j \<and> \<not> nfinite li \<and> nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform h n f)) li"
             using c22 by blast
           have c24: "nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform h n g)) li"
            by (metis (no_types, lifting) c23 chopstar_d.IH chopstar_d.prems(1) eint_less_eq(1) less_eint.simps(3) linorder_not_le nridx_expand order_less_le)  
           show ?thesis using c20a c23 c24 by auto
          qed
         qed
        have c25: "\<sigma> i j \<Turnstile> suform (chopstar h ) n g \<Longrightarrow> \<sigma> i j \<Turnstile> suform (chopstar h ) n f"
         proof -
          assume c25a: "\<sigma> i j \<Turnstile> suform (chopstar h ) n g"
          show "\<sigma> i j \<Turnstile> suform (chopstar h ) n f"
          proof -
           have c26: "(\<exists>li::eint nellist. nnth li (0::nat) = j \<and> \<not> nfinite li \<and> nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform h n g)) li)"    
            using c25a c20a by fastforce
           obtain li where c27: "nnth li (0::nat) = j \<and> \<not> nfinite li \<and> nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform h n g)) li"
            using c26 by blast
           have c28: "nridx (\<lambda>(a::eint) b::eint. b < a \<and> (\<sigma> b a \<Turnstile> suform h n f)) li"
            by (metis (no_types, lifting) c27 chopstar_d.IH chopstar_d.prems(1) eint_less_eq(1) less_eint.simps(3) linorder_not_le nridx_expand order_less_le)  
           show ?thesis using c20a c27 c28 by auto
          qed 
         qed
        show ?thesis 
        using c21 c25 by blast
       qed
      qed
    have c30: " i\<noteq>j \<and> i \<noteq> -\<infinity> \<and> j = \<infinity> \<Longrightarrow> ?thesis"
      proof -
       assume c30a: "i\<noteq>j \<and> i \<noteq> -\<infinity> \<and> j = \<infinity>"
       show "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f) = (\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"
       proof -
        have c31: "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f) \<Longrightarrow> (\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"  
        proof -
         assume c31a: "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f)"
         show "(\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"
         proof -
          have c32: "(\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n f)) ri) "
           using c30a c31a by auto  
          obtain ri where c33: "nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n f)) ri"
           using c32 by blast
          have c34: "nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n g)) ri"
           by (metis (no_types, lifting) c33 chopstar_d.IH chopstar_d.prems(1) eint_less_eq(1) less_eint.simps(3) linorder_not_le nridx_expand order_less_le)    
          show ?thesis using c33 c34 c30a by auto
         qed
        qed
       have c35: "(\<sigma> i j \<Turnstile> suform (chopstar h ) n g) \<Longrightarrow> (\<sigma> i j \<Turnstile> suform (chopstar h ) n f)"
        proof -
         assume c35a: "(\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"
         show "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f)"
         proof -
          have c36: "(\<exists>ri::eint nellist. nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n g)) ri)"
           using c30a c35a by auto
          obtain ri where c37: "nnth ri (0::nat) = i \<and> \<not> nfinite ri \<and> nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n g)) ri"
            using c36 by blast
          have c38: "nridx (\<lambda>(a::eint) b::eint. a < b \<and> (\<sigma> a b \<Turnstile> suform h n f)) ri"
           by (metis (no_types, lifting) c37 chopstar_d.IH chopstar_d.prems(1) eint_less_PInfty less_eint.simps(3) not_less_iff_gr_or_eq nridx_expand order_less_le)     
          show ?thesis using c37 c38 c30a by auto
         qed
        qed
       show ?thesis using c35 c31 by blast
      qed
     qed 
    have c40: " i\<noteq>j \<and> i = -\<infinity> \<and> j = \<infinity> \<Longrightarrow> ?thesis"
     proof -
      assume c40a: "i\<noteq>j \<and> i = -\<infinity> \<and> j = \<infinity>"
      show "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f) = (\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"
      proof -
       have c41: "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f) \<Longrightarrow> (\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"  
        proof -
         assume c41a: "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f)"
         show "(\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"
         proof -
          have c42: "\<exists>(li::int nellist) ri::int nellist.
                       nnth li (0::nat) = nnth ri (0::nat) \<and> \<not> nfinite li \<and>
                      nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform h n f)) li \<and>
                     \<not> nfinite ri \<and> 
                      nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform h n f)) ri"
           using c40a c41a by simp  
         obtain li and ri where c43: "nnth li (0::nat) = nnth ri (0::nat) \<and> \<not> nfinite li \<and>
                      nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform h n f)) li \<and>
                     \<not> nfinite ri \<and> 
                      nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform h n f)) ri"
          using c42 by blast
          have c44a: "nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform h n g)) li"  
          by (metis (no_types, lifting) PInfty_neq_eint(2) c43 chopstar_d.IH chopstar_d.prems(1) less_eint.simps(1) less_eint.simps(3) less_eq_eint_def nridx_expand)  
         have c44b: "nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform h n g)) ri"
          by (metis (no_types, lifting) PInfty_neq_eint(2) c43 chopstar_d.IH chopstar_d.prems(1) less_eint.simps(1) less_eint.simps(3) less_eq_eint_def nridx_expand) 
         show ?thesis using c40a c44a c44b c43 by auto
        qed
       qed
      have c45: "(\<sigma> i j \<Turnstile> suform (chopstar h ) n g) \<Longrightarrow> (\<sigma> i j \<Turnstile> suform (chopstar h ) n f)"
       proof -
        assume c45a: "(\<sigma> i j \<Turnstile> suform (chopstar h ) n g)"
        show "(\<sigma> i j \<Turnstile> suform (chopstar h ) n f)"
        proof -
         have c46: "\<exists>(li::int nellist) ri::int nellist.
                       nnth li (0::nat) = nnth ri (0::nat) \<and> \<not> nfinite li \<and>
                      nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform h n g)) li \<and>
                     \<not> nfinite ri \<and> 
                      nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform h n g)) ri"
          using c40a c45a by simp
         obtain li and ri where c47: "nnth li (0::nat) = nnth ri (0::nat) \<and> \<not> nfinite li \<and>
                      nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform h n g)) li \<and>
                     \<not> nfinite ri \<and> 
                      nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform h n g)) ri"
          using c46 by blast
         have c48a: "nridx (\<lambda>(a::int) b::int. b < a \<and> (\<sigma> (eint b) (eint a) \<Turnstile> suform h n f)) li"
          by (metis (no_types, lifting) PInfty_neq_eint(2) c47 chopstar_d.IH chopstar_d.prems(1) less_eint.simps(1) less_eint.simps(3) less_eq_eint_def nridx_expand)           
         have c48b: "nridx (\<lambda>(a::int) b::int. a < b \<and> (\<sigma> (eint a) (eint b) \<Turnstile> suform h n f)) ri"
          by (metis (no_types, lifting) PInfty_neq_eint(2) c47 chopstar_d.IH chopstar_d.prems(1) less_eint.simps(1) less_eint.simps(3) less_eq_eint_def nridx_expand)  
         show ?thesis using c40a c48a c48b c47 by auto
        qed
       qed
      show ?thesis using c45 c41 by blast
     qed
    qed
   show ?thesis 
    using c1 c10 c20 c30 c40 by argo
   qed
next
case (exists_d x1 h)
then show ?case 
  by auto
qed 

lemma suform_ieqv: 
 assumes "\<turnstile> f ieqv g"
 shows   "\<turnstile> (suform h n f)  ieqv  (suform h n g)"
using assms unfolding valid_def using suform_ieqv_sem 
by fastforce
 

lemma rename_nl_pitl_1: 
 assumes "v1 \<notin> bvars f"
         "v2 \<notin> bvars f"
         "v1 \<noteq> v2"
         "v2 \<notin> fvars f" 
         " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
 shows " (\<sigma> i j \<Turnstile> Ex v1. f) = (\<sigma> i j\<Turnstile> Ex v2. suform f v1 ($v2)) "
proof -
 have 1: "(\<sigma> i j \<Turnstile> Ex v1. f) = (\<sigma> i j\<Turnstile> Ex v2. suform f v1 ($v2 iand lfinite)) " 
   using assms rename_nl_pitl by blast
 have 2: "\<turnstile> $v2 ieqv $v2 iand lfinite"
   using atom_lfinite_absorb by blast 
 have 3: "\<turnstile> suform f v1 ($v2 iand lfinite) ieqv suform f v1 ($v2)" 
   using "2" suform_ieqv_sem by fastforce
 have 4: "(\<sigma> i j\<Turnstile> Ex v2. suform f v1 ($v2 iand lfinite)) =
          (\<sigma> i j\<Turnstile> Ex v2. suform f v1 ($v2))" 
    using "3" assms(5) assms(6) assms(7) by auto
 show ?thesis 
 using "1" "4" by blast
qed


lemma ExistSubst: 
 assumes  "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_nl_itl w" 
 shows "\<turnstile> suform f n (w iand lfinite) iimp (Ex n. f)"
using assms unfolding valid_def
using  similar_upd[of _ n ] subst_suform[of n f w ]
using semantics_nl_itl.simps(3) semantics_nl_itl.simps(9) by blast 


lemma ForallSubst: 
 assumes  "n \<notin> bvars f "
         "\<And>z . z \<in> fvars w \<Longrightarrow> z \<notin> bvars f" 
         "state_nl_itl w" 
 shows "\<turnstile> (Fa n. f) iimp suform f n (w iand lfinite) "
using assms
using forall_d_def similar_upd subst_suform by auto

lemma ExistsImp: 
 assumes " v1 \<notin> fvars f1"
 shows " \<turnstile> (Ex v1. (f1 iimp f2)) ieqv (f1 iimp (Ex v1. f2)) " 
 unfolding valid_def apply simp 
 by (meson assms exist_sigma_exist_value not_fvar_upd similar_commute similar_refl)

lemma ExistsSChopRight:
 assumes " v1 \<notin> fvars f2"
 shows " \<turnstile> (Ex v1. f1 schop f2) iimp (Ex v1. f1) schop f2 " 
using assms 
unfolding valid_def apply auto
using not_fvar_upd by blast

lemma ExistsSChopLeft:
 assumes " v1 \<notin> fvars f1"
 shows " \<turnstile> (Ex v1. f1 schop f2) iimp f1 schop (Ex v1. f2)  " 
using assms 
unfolding valid_def apply auto
using not_fvar_upd by blast

lemma ForallGen: 
assumes "\<turnstile> f "
shows   "\<turnstile> Fa v. f" 
using assms 
by (simp add: forall_d_def)

lemma ExistsI:
assumes  " i \<le> j "
        " i \<noteq> \<infinity> "
        " j \<noteq> -\<infinity> " 
         " (\<sigma> i j \<Turnstile> f) " 
shows   " (\<sigma> i j \<Turnstile> Ex v. f) "
using assms apply simp
using exist_sigma_exist_value similar_refl by blast

lemma init_state_nl_itl: 
 assumes "state_nl_itl w"
 shows " \<turnstile> init w ieqv lfinite iand w"
using assms
unfolding valid_def 
by (simp add: init_defs  state_nl_itl_defs)



lemma exists_elim_state_nl_itl: 
 assumes "state_nl_itl w"
         "n \<notin> bvars w" 
 shows " \<turnstile> (Ex n. w iand lfinite) ieqv 
             (suform (w iand lfinite) n (itrue iand lfinite)) ior 
             (suform (w iand lfinite) n (ifalse iand lfinite)) " 
proof -
  have 1: "state_nl_itl itrue" 
    by (simp add: inot_d_def itrue_d_def)
  have 2: "(\<And>z. z \<in> fvars itrue \<Longrightarrow> z \<notin> bvars w)" 
    by (simp add: inot_d_def itrue_d_def)
  have 3: "\<And> \<sigma> i j.  i \<le> j \<and> i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity>  \<Longrightarrow> 
            (\<sigma> i j \<Turnstile> (suform (w iand lfinite)  n (itrue iand lfinite))) = 
            (i \<noteq> -\<infinity> \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> itrue)) i j \<Turnstile> w))" 
   using subst_suform[of n w itrue ] 1 2 assms suform_lfinite_export[of w n "itrue iand lfinite"]   
   suform_lfinite_export[of w n "ifalse iand lfinite"] by simp 
  have 4: "state_nl_itl ifalse"
    by simp
  have 5: "(\<And>z. z \<in> fvars ifalse \<Longrightarrow> z \<notin> bvars w)"  
   by simp
  have 6: "\<And> \<sigma> i j.  i \<le> j \<and> i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity>  \<Longrightarrow> 
            (\<sigma> i j \<Turnstile> (suform (w iand lfinite) n (ifalse iand lfinite))) = 
             (i \<noteq> -\<infinity> \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> ifalse)) i j \<Turnstile> w))" 
   using subst_suform[of n w ifalse ] 4 5 assms 
   suform_lfinite_export[of w n "itrue iand lfinite"]   
   suform_lfinite_export[of w n "ifalse iand lfinite"]
  by simp 
  have 7: "\<And> \<sigma> i j . i \<le> j \<and> i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity>  \<Longrightarrow>
             (\<sigma> i j \<Turnstile> (Ex n. w iand lfinite)) =
             (\<exists>l. i\<noteq> -\<infinity> \<and> ((upd \<sigma> n l) i i \<Turnstile> w))" 
     using assms(1) lfinite_defs state_nl_itl_defs by auto
  have 8: "\<And> \<sigma> i j.  i \<le> j \<and> i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity>  \<Longrightarrow> 
            (i \<noteq> -\<infinity> \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> itrue)) i j \<Turnstile> w)) = 
               (i \<noteq> -\<infinity> \<and> ((upd \<sigma> n (\<lambda>x::int. True)) i i \<Turnstile> w))" 
     using state_nl_itl_defs[of w] assms  by simp 
  have 9: "\<And> \<sigma> i j.  i \<le> j \<and> i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity>  \<Longrightarrow> 
            (i \<noteq> -\<infinity> \<and> ((upd \<sigma> n (\<lambda>x::int. \<sigma> (eint x) (eint x) \<Turnstile> ifalse)) i j \<Turnstile> w)) = 
               (i \<noteq> -\<infinity> \<and> ((upd \<sigma> n (\<lambda>x::int. False)) i i \<Turnstile> w))" 
    using state_nl_itl_defs[of w] assms   by simp
  have 10: "\<And> \<sigma> i j.  i \<le> j \<and> i \<noteq> \<infinity> \<and> j \<noteq> -\<infinity>  \<Longrightarrow>
                (\<exists>l. i\<noteq> -\<infinity> \<and> ((upd \<sigma> n l) i i \<Turnstile> w)) \<longleftrightarrow>
                 (i \<noteq> -\<infinity> \<and> ((upd \<sigma> n (\<lambda>x::int. True)) i i \<Turnstile> w) \<or>  i \<noteq> -\<infinity> \<and> ((upd \<sigma> n (\<lambda>x::int. False)) i i \<Turnstile> w))"
    using assms exists_elim_state_nl_itl_help by blast
  show ?thesis unfolding valid_def using lfinite_defs apply simp 
  by (meson "3" "6" "8" "9" assms(1) assms(2) exists_elim_state_nl_itl_help state_nl_itl_defs)
qed

lemma exists_elim_state_nl_itl_1: 
 assumes "state_nl_itl w"
         "n \<notin> bvars w" 
 shows " \<turnstile> (Ex n. w iand lfinite) ieqv 
             (suform (w iand lfinite) n (itrue)) ior 
             (suform (w iand lfinite) n (ifalse)) " 
proof -
 have 1: "(\<And>z::nat. z \<in> fvars itrue \<Longrightarrow> z \<notin> bvars w)" 
    by (simp add: inot_d_def itrue_d_def)
 have 2: "state_nl_itl itrue"
    by (simp add: inot_d_def itrue_d_def) 
 have 3: "\<turnstile> suform (w iand lfinite) n (itrue iand lfinite) ieqv
            suform (w iand lfinite) n (itrue)" 
    using subst_suform_state[of n w itrue ] assms 
    by (simp add: "1" "2")
 have 4: "\<turnstile> suform (w iand lfinite) n (ifalse iand lfinite) ieqv
            suform (w iand lfinite) n (ifalse)" 
    using subst_suform_state[of n w ifalse ] assms by (simp)
 have 5: "\<turnstile> (suform (w iand lfinite) n (itrue iand lfinite)) ior 
             (suform (w iand lfinite) n (ifalse iand lfinite)) 
             ieqv
            (suform (w iand lfinite) n (itrue)) ior 
             (suform (w iand lfinite) n (ifalse))" 
    using "3" "4" by auto
 show ?thesis 
 using "5" assms exists_elim_state_nl_itl by force
qed

end

