(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

section \<open>Extra operations on non-empty conductive lists\<close>

text \<open>
The operations ndropns, nkfilter, nleast, nidx, nfuse, nbutlast, nsubn, nfuse, nridx,
	nlastnfirst, nfusecat and nrev are defined for nellists together with a library
of lemmas. 
\<close>

theory NELList_Extras 
 imports   NELList
begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="cvc4 z3 e spass vampire " ]
*)

subsection \<open>ndropns\<close>

primcorec ndropns :: " 'a nellist \<Rightarrow> 'a nellist nellist "
 where "ndropns nell = 
        (case nell of (NNil b) \<Rightarrow> (NNil (NNil b)) |
          (NCons x nell') \<Rightarrow> (NCons (NCons x nell') (ndropns nell'))) "

simps_of_case ndropns_code [code, simp, nitpick_simp]: ndropns.code

lemma ndropns_simps [simp]:
 shows nhd_ndropns: " \<not> is_NNil nell \<Longrightarrow> nhd (ndropns nell) = nell "
  and  ndropns_NCons: " ntl (ndropns nell) = (case nell of (NNil b) \<Rightarrow> (NNil (NNil b)) |
                                                (NCons x nell') \<Rightarrow> ndropns nell')  "
by (auto simp add: nellist.case_eq_if)   
   (metis ndropns_code(1) nellist.collapse(1) nellist.sel(4))

lemma ndropns_nnth:
 assumes " i \<le> nlength nell" 
  shows " nnth (ndropns nell) i = ndropn i nell"
using assms 
proof (induction i arbitrary: nell)
case 0
then show ?case 
 proof (cases nell)
 case (NNil x1)
 then show ?thesis by (simp add: nnth_NNil)
 next
 case (NCons x21 x22)
 then show ?thesis by simp
 qed
next
case (Suc i)
then show ?case 
 proof (cases nell)
 case (NNil x1)
 then show ?thesis by (simp add: nnth_NNil)
 next
 case (NCons x21 x22)
 then show ?thesis using Suc by (simp add: Suc_ile_eq)
 qed
qed

lemma ndropns_nlength:
 " nlength (ndropns nell) = (nlength nell)"
by (coinduction arbitrary: nell rule: enat_coinduct)
   (case_tac nell, auto)

lemma in_nset_ndropns:
 " nell \<in> nset(ndropns nellx) \<longleftrightarrow> (\<exists> i. i \<le> nlength nellx \<and> nell = ndropn i nellx)"
by (metis in_nset_conv_nnth ndropns_nlength ndropns_nnth)

lemma nset_ndropns:
 " nset (ndropns nell) = { ndropn i nell | i. i \<le> nlength nell}"
using in_nset_conv_nnth[of _ "ndropns nell"] nset_conv_nnth[of "ndropns nell"]
using in_nset_ndropns[of  _ nell] by auto 

lemma nmap_first_ndropns:
 " nmap (\<lambda> nell. nnth nell 0) (ndropns nell) = nell"
by (coinduction arbitrary: nell)
   (case_tac nell, auto simp add: nnth_NNil)

lemma ndropn_ndropns:
 assumes " i \<le> nlength(ndropns nell)"
 shows " ndropn i (ndropns nell) = ndropns (ndropn i nell)"
using assms
proof (coinduction arbitrary: nell i)
case (Eq_nellist ia nellx)
then show ?case 
   proof -
    have 1: "enat nellx \<le> nlength (ndropns ia) \<Longrightarrow>
             is_NNil (ndropn nellx (ndropns ia)) = is_NNil (ndropns (ndropn nellx ia))"
      by (simp add: is_NNil_ndropn ndropns_nlength) 
    have 2: "enat nellx \<le> nlength (ndropns ia) \<Longrightarrow>
             (is_NNil (ndropn nellx (ndropns ia)) \<longrightarrow>
              is_NNil (ndropns (ndropn nellx ia)) \<longrightarrow> 
              nlast (ndropn nellx (ndropns ia)) = nlast (ndropns (ndropn nellx ia)))"
      by (metis dual_order.antisym ndropn_eq_NNil ndropns.disc(2) ndropns.simps(3) ndropns_nlength
          ndropns_nnth nellist.case_eq_if nellist.collapse(1) the_enat.simps) 
    have 3: "enat nellx \<le> nlength (ndropns ia) \<Longrightarrow>
             (\<not> is_NNil (ndropn nellx (ndropns ia)) \<longrightarrow>
              \<not> is_NNil (ndropns (ndropn nellx ia)) \<longrightarrow>
              nhd (ndropn nellx (ndropns ia)) = nhd (ndropns (ndropn nellx ia)))"
      by (metis Nat.add_0_right ndropn_nnth ndropns.disc(1) ndropns_nlength ndropns_nnth nhd_conv_nnth
          nhd_ndropns) 
    have 4: "enat nellx \<le> nlength (ndropns ia) \<Longrightarrow>
             (\<not> is_NNil (ndropn nellx (ndropns ia)) \<longrightarrow>
              \<not> is_NNil (ndropns (ndropn nellx ia)) \<longrightarrow>
              (\<exists>nell i.
                ntl (ndropn nellx (ndropns ia)) = ndropn i (ndropns nell) \<and>
                ntl (ndropns (ndropn nellx ia)) = ndropns (ndropn i nell) \<and> 
                enat i \<le> nlength (ndropns nell))) "
      by (metis Suc_ile_eq is_NNil_ndropn ndropn_Suc_conv_ndropn ndropns_code(2) ndropns_nlength
          nellist.sel(5) order.order_iff_strict) 
    show ?thesis 
     using "1" "2" "3" "4" Eq_nellist by blast
  qed
qed

lemma ndropns_nfilter_nnth:
 assumes " i \<le> nlength (nfilter P (ndropns nell))"
         " \<exists> nelly \<in> nset(ndropns nell). P nelly"
 shows   " P (nnth (nfilter P (ndropns nell)) i) "
using assms using nset_nfilter[of "ndropns nell" P] 
by (metis (full_types) Int_iff in_nset_conv_nnth mem_Collect_eq ) 

lemma nnth_zero_ndropn:
 " nnth (ndropn n nell) 0 = nnth nell n"
by simp

lemma in_nset_ndropns_nhd:
 "x \<in> nset nell \<longleftrightarrow>( \<exists> ys. x=(nnth ys 0) \<and> ys \<in> nset(ndropns nell))"
by auto
   (metis in_nset_conv_nnth ndropns_nlength nmap_first_ndropns nnth_nmap,
    metis Nat.add_0_right in_nset_conv_nnth in_nset_ndropns ndropn_nnth)

lemma nset_ndropns_nhd:
" nset nell = {(nnth nelly 0) | nelly. nelly \<in> nset(ndropns nell) }"
by auto
   (meson in_nset_ndropns_nhd,
    metis in_nset_conv_nnth in_nset_ndropns nnth_zero_ndropn)

lemma nellist_all2_ndropnsI:
 " nellist_all2 A nellx nelly   \<Longrightarrow> nellist_all2 (nellist_all2 A) (ndropns nellx) (ndropns nelly)"
by (coinduction arbitrary: nellx nelly)
   (auto simp add: nellist.case_eq_if dest: nellist_all2_nhdD nellist_all2_is_NNilD 
    intro: nellist_all2_ntlI)

subsection \<open>Definitions\<close>

context
includes nellist.lifting
begin

lift_definition nkfilter :: " ('a \<Rightarrow> bool) \<Rightarrow> nat \<Rightarrow>'a nellist \<Rightarrow> nat nellist"
is "\<lambda> P n xs.   (if lnull(kfilter P n xs) then (iterates Suc n) else kfilter P n xs)"
by simp

lift_definition nleast :: " ('a \<Rightarrow> bool) \<Rightarrow> 'a nellist \<Rightarrow> nat "
is "\<lambda> P xs. lleast P xs"
by auto

lift_definition nridx :: "('a \<Rightarrow> 'a \<Rightarrow> bool) \<Rightarrow> 'a nellist \<Rightarrow> bool "
 is "\<lambda> R xs. ridx R xs" 
by auto

lift_definition nidx :: " nat nellist \<Rightarrow> bool "
 is "\<lambda> xs. lidx xs"
by auto

lift_definition nbutlast :: "'a nellist \<Rightarrow> 'a nellist"
is "\<lambda> xs. (if lnull (lbutlast xs) then (LCons (llast xs) LNil) else lbutlast xs) " 
by auto

lift_definition nfuse :: "'a nellist \<Rightarrow> 'a nellist \<Rightarrow> 'a nellist"
is "\<lambda> xs ys. lfuse xs ys"
using lfuse_conv_lnull by blast

lift_definition nsubn :: "'a nellist \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> 'a nellist"
 is "(\<lambda> xs n1 n2. lsubc n1 n2 xs)" 
unfolding lsubc_def
by auto
   (metis co.enat.exhaust_sel dual_order.refl enat_ile iless_Suc_eq leD llength_eq_0)

lift_definition nlastnfirst :: "'a nellist nellist \<Rightarrow> bool" 
 is "\<lambda> xss. llastlfirst xss " 
apply ( simp add: pcr_nellist_def cr_nellist_def OO_def rel_fun_def llist.rel_eq  )
unfolding llastlfirst_def 
using llist_all2_llist_of_nellist_1 by blast

lemma nfusecat_help: 
assumes "\<exists>y. llist_all2 (\<lambda>x z. llist_of_nellist z = x) llist1 y \<and> y \<noteq> LNil \<and> 
             llist_all2 (\<lambda>x z. llist_of_nellist z = x) llist2 y "
shows "lfusecat llist1 \<noteq> LNil \<and> lfusecat llist1 = lfusecat llist2" 
proof -
 have 1: "lfusecat llist1 = lfusecat llist2"
    using assms llist_all2_llist_of_nellist_1 lnull_def by blast
 have 2: "lfusecat llist1 \<noteq> LNil"
    using assms apply auto 
    using lfusecat_not_lnull_var llist_all2_lnullD llist_all2_lsetD1 by fastforce
 show ?thesis using 1 2 by auto
qed

lift_definition nfusecat :: "'a nellist nellist \<Rightarrow> 'a nellist " 
 is "\<lambda> xss. lfusecat xss " 
using nfusecat_help 
by (simp add:  pcr_nellist_def OO_def cr_nellist_def nellist.pcr_cr_eq rel_fun_def lnull_def 
    llist_all2_cases llist_all2_rsp llist.rel_eq not_lnull_conv_llist_of_nellist) blast

lift_definition nrev :: " 'a nellist \<Rightarrow> 'a nellist "
 is "\<lambda> xs. if lfinite xs then lrev xs else xs" 
by auto



subsection \<open>nbutlast\<close>

lemma nbutlast_NNil[simp]: 
 "nbutlast (NNil x) = (NNil x) " 
apply transfer
by auto

lemma nbutlast_snoc [simp]:
 "nbutlast (nappend nell (NNil x)) = nell " 
apply transfer
by auto

lemma nbutlast_def1: 
 "nbutlast nell =
   (case nell of (NNil x) \<Rightarrow> (NNil x) |
     (NCons x nell1) \<Rightarrow> 
       (case nell1 of (NNil y) \<Rightarrow> (NNil x) | 
          (NCons z nell2) \<Rightarrow> (NCons x (nbutlast nell1)))) " 
proof (cases nell)
case (NNil x1)
then show ?thesis by simp
next
case (NCons x21 x22)
then show ?thesis 
   proof -
    have 1: "is_NNil x22 \<Longrightarrow> nbutlast (NCons x21 x22) = (NNil x21)"
      by (metis nappend_NNil nbutlast_snoc nellist.collapse(1))    
    have 2: "\<not>is_NNil x22 \<Longrightarrow>  nbutlast (NCons x21 x22) = (NCons x21 (nbutlast x22)) " 
     apply transfer
     by auto
         (metis lhd_LCons_ltl llist.collapse(1),
          metis lbutlast_simps(3) lhd_LCons_ltl)
    show ?thesis 
    by (simp add: "1" "2" NCons nellist.case_eq_if)
   qed
qed 

lemma ntl_nbutlast: 
 "  ntl (nbutlast nell) = 
     (if is_NNil nell then  ntl nell  else 
       (if is_NNil (ntl nell) then NNil (nhd nell) else nbutlast (ntl nell)))  " 
by (auto simp add:  nbutlast_def1 nellist.case_eq_if) 
        

lemma nbutlast_not_nfinite:
 assumes "\<not> nfinite nell"
 shows   "nbutlast nell = nell"
 using assms 
apply transfer
by simp
   (metis lbutlast.disc_iff(2) lbutlast_not_lfinite)

lemma nbutlast_nfinite:
 "nfinite (nbutlast nell) \<longleftrightarrow> nfinite nell"
apply transfer
by (metis (full_types) lbutlast_lfinite lbutlast_not_lfinite lfinite_LNil lfinite_code(2))

lemma nlength_nbutlast [simp]:
 " nlength (nbutlast nell) = epred (nlength nell)"
apply transfer
by (simp, simp add: epred_llength)

lemma nbutlast_nappend:
 " nbutlast (nappend nellx nelly) = 
    (if is_NNil nelly then nellx else nappend nellx (nbutlast nelly))"
apply transfer
by simp
   (metis lbutlast_lappend lbutlast_snoc lhd_LCons_ltl lnull_def)

lemma nappend_nbutlast_nlast_id_nfinite:
 assumes "nfinite nellx"
         "\<not>is_NNil nellx" 
 shows   "(nappend (nbutlast nellx) (NNil (nlast nellx))) = nellx"
using assms
apply transfer
by simp
   (metis lhd_LCons_ltl llist.collapse(1))

lemma nappend_nbutlast_nlast_id_not_nfinite:
 assumes "\<not>nfinite nellx"
         " \<not> is_NNil nellx"
 shows   "(nappend (nbutlast nellx) (NNil (nlast nellx))) = nellx"
using assms
apply transfer
by simp
   (metis lappend_inf lbutlast.disc_iff(2) lbutlast_snoc)

lemma nappend_nbutlast_nlast_id [simp]:
shows " \<not> is_NNil nell \<Longrightarrow> (nappend (nbutlast nell) (NNil (nlast nell))) = nell"
using nappend_nbutlast_nlast_id_nfinite nappend_nbutlast_nlast_id_not_nfinite by blast

lemma nbutlast_eq_NNil_conv: 
 " nbutlast nell = (NNil (nfirst nell)) \<longleftrightarrow> 
    nell= (NNil (nfirst nell)) \<or> (\<exists>x. nell = (NCons x (NNil (nlast nell)))) "
apply transfer
by (auto simp add: llist.expand lbutlast_not_lfinite )
   (metis lhd_LCons_ltl llast_LCons, 
    metis eq_LConsD lbutlast_eq_LNil_conv lbutlast_ltl lhd_LCons_ltl llast_LCons2 llast_singleton,
    simp add: eq_LConsD,
    simp add: eq_LConsD lbutlast_eq_LCons_conv, 
    metis lfinite_LNil lfinite_ltl llist.collapse(1))

lemma nbutlast_eq_NCons_conv: 
 " nbutlast nell = (NCons x ys) \<longleftrightarrow>  
    nell = (NCons x (nappend ys (NNil (nlast nell)))) "
apply transfer
by (auto simp add: eq_LConsD lbutlast_eq_LCons_conv llast_linfinite)

lemma nbutlast_conv_ntake:
 " nbutlast nell = ntake (epred (nlength nell)) nell"
apply transfer
by simp
 (metis co.enat.exhaust_sel ileI1 iless_eSuc0 lappend_lbutlast_llast_id lappend_lnull1 
  lbutlast.disc_iff(2) lbutlast_conv_ltake llength_eq_0 llength_lbutlast ltake_all)

lemma nmap_nbutlast:
 " nmap f (nbutlast nell) = nbutlast (nmap f nell)"
apply transfer
by simp
   (metis lfinite_ltl llast_lmap lmap_lbutlast lnull_imp_lfinite)

lemma snocs_eq_iff_nbutlast:
 " nappend nell (NNil x) = nell1 \<longleftrightarrow> 
   (( nfinite nell1 \<and> \<not> is_NNil nell1 \<and> nbutlast nell1 = nell \<and> nlast nell1 = x) 
   \<or> (\<not> nfinite nell1 \<and> nbutlast nell = nell1))"
by (metis is_NNil_nappend nappend_inf nappend_nbutlast_nlast_id nbutlast_nfinite nbutlast_snoc
    nlast_NNil nlast_nappend)

lemma in_nset_nbutlastD:
 " x \<in> nset(nbutlast nell) \<Longrightarrow> x \<in> nset nell"
by (metis in_nset_snocn_iff nappend_nbutlast_nlast_id_nfinite nappend_snocn nbutlast_NNil 
    nbutlast_not_nfinite nellist.collapse(1))


lemma in_nset_nbutlast_nappendI:
 " x \<in> nset (nbutlast nell) \<or> (nfinite nell \<and> \<not> is_NNil nell1 \<and> x \<in> nset(nbutlast nell1)) \<Longrightarrow>
   x \<in> nset (nbutlast (nappend nell nell1))"
unfolding nbutlast_nappend
by (metis (full_types) Un_iff in_nset_nbutlastD nset_nappend)

lemma nnth_nbutlast:
 assumes "n \<le> nlength(nbutlast nell)"
 shows   " nnth (nbutlast nell) n = nnth nell n"
by (metis assms nappend_nbutlast_nlast_id_nfinite nbutlast_eq_NNil_conv nbutlast_not_nfinite 
    ndropn_is_NNil ndropn_nfirst nellist.collapse(1) nnth_nappend1 nnth_nlast)

subsection \<open>nsubn\<close>

lemma nsubn_def1: 
  "nsubn nell i j = ntaken (j-i) (ndropn i nell)"
apply transfer
unfolding lsubc_def
by auto
   (metis co.enat.exhaust_sel dual_order.refl enat_ile iless_Suc_eq leD llength_eq_0 , 
    metis eSuc_enat enat_the_enat infinity_ileE ldrop_enat min.cobounded1)

lemma nsubn_same: 
 shows   "nsubn nell k k  = (NNil (nnth nell k )) " 
unfolding nsubn_def1
by (simp add: ndropn_nfirst) 


lemma nsubn_nlength:
 " nlength(nsubn nell i j) =  min (j-i) (nlength nell - i)"
by (simp add: nsubn_def1) 

lemma nsubn_nlength_gr_one:
 assumes "k<n"
         "n \<le> nlength nell " 
 shows "0< nlength (nsubn nell k n) "
using assms
unfolding nsubn_nlength
by (metis enat_minus_mono1 enat_ord_simps(2) idiff_enat_enat min_def zero_enat_def zero_less_diff)
 
lemma nsubn_nfinite:       
 shows "nfinite (nsubn nell k n)"
by (simp add: nsubn_def1)


lemma nsubn_nnth:
shows " nnth (nsubn nell i j) k = nnth nell (i + min k (j - i))"
unfolding nsubn_def1
using ntaken_nnth[of "j-i" "(ndropn i nell)" k] ndropn_nnth[of i nell "(min k (j - i))"]
by simp 


lemma ntaken_ndropn:
 " ntaken n (ndropn k nell) = nsubn nell k (n+k)"
by (simp add: nsubn_def1)

lemma ntaken_ndropn_nfirst:
 " nfirst (ntaken n (ndropn k nell)) = nnth nell k"
by (metis min_0L ndropn_nfirst ntaken_0 ntaken_ntaken nlast_NNil)

lemma ntaken_ndropn_nfirst_a:
 " nfirst (ntaken n (ndropn k nell)) = nfirst(ndropn k nell)"
by (simp add: ndropn_nfirst ntaken_ndropn_nfirst)

lemma ntaken_ndropn_nlast:
 " nlast(ntaken n (ndropn k nell)) = nnth nell (n+k)"
by (simp add: add.commute ntaken_nlast)

lemma nsubn_nfirst:
 " nfirst (nsubn nell i j) = nnth nell i" 
by (simp add: nsubn_def1 ntaken_ndropn_nfirst)

lemma nsubn_nlast:
 " nlast (nsubn nell i j) = nnth nell (j - i + i) " 
by ( simp add: nsubn_def1 ntaken_ndropn_nlast)

lemma nsubn_ndropn:
 assumes "i<j"
 shows " nsubn nell (i+k) (j+k) = nsubn (ndropn k nell) i j "
using assms
by (simp add: add.commute ndropn_ndropn nsubn_def1)


lemma pref_ntaken_3:
 " (ntaken i (ntaken (i+k) nell)) = (ntaken i nell)"
by (metis le_add1 min.orderE ntaken_ntaken)

lemma ntaken_ndropn_swap_nlength:
 assumes " ia+ i \<le> nlength nell"
 shows " nlength (ntaken ia (ndropn i nell)) = nlength (ndropn i (ntaken (ia+i) nell))"
using assms ndropn_nlength[of i "(ntaken (ia+i) nell)"] ntaken_nlength[of ia "(ndropn i nell)" ]
by auto
   (metis add_diff_cancel_right' enat_minus_mono1 idiff_enat_enat min.orderE)
 
lemma ntaken_ndropn_swap_nnth:
 assumes "m \<le> ia"
         " ia+ i \<le> nlength nell"
 shows " nnth (ntaken ia (ndropn i nell)) m = nnth (ndropn i (ntaken (ia+i) nell)) m"
using assms
by (simp add: ntaken_nnth)

lemma nellist_eq_nnth_eq:
 " (nellx = nelly) \<longleftrightarrow> nlength nellx = nlength nelly \<and> (\<forall> i \<le> nlength nellx. nnth nellx i = nnth nelly i)"
by transfer
   (metis co.enat.exhaust_sel iless_Suc_eq llength_eq_0 llist_eq_lnth_eq min_absorb1 the_enat.simps)

lemma ntaken_ndropn_swap:
 assumes " ia+ i \<le> nlength nell"
 shows "  (ntaken ia (ndropn i nell)) =  (ndropn i (ntaken (ia+i) nell))"
using assms nellist_eq_nnth_eq[of "(ntaken ia (ndropn i nell))"  " (ndropn i (ntaken (ia+i) nell))"]
using ntaken_ndropn_swap_nlength
using ntaken_ndropn_swap_nnth by fastforce


lemma ntaken_nsubn: 
 assumes "n \<le> nlength nell " 
         "m + k \<le>  n " 
 shows " ntaken m (nsubn nell k n) = nsubn nell k (m+k) "
using assms 
unfolding nsubn_def1 
by simp

lemma ndropn_nsubn: 
assumes "n \<le> nlength nell " 
        "m + k \<le>  n " 
 shows " ndropn m (nsubn nell k n) = nsubn nell (m+k) n "
proof -
 have 1: "ntaken (n-k) (ndropn k nell) = ndropn k (ntaken n nell)"
     by (metis add_leD2 assms(1) assms(2) diff_add ntaken_ndropn_swap plus_enat_simps(1))
 have 2: "ndropn m (ndropn k (ntaken n nell)) =
          ndropn (m +k) (ntaken n nell)" 
     by (simp add: add.commute ndropn_ndropn)
 show ?thesis unfolding nsubn_def1 using 1 2 
 by (simp add: assms(1) assms(2) ntaken_ndropn_swap)
qed

lemma ntl_nsubn: 
 assumes " n \<le> nlength nell"
         " k \<le> n "
 shows   "ntl(nsubn nell k n) = (if n=k then (NNil (nnth nell k)) else  nsubn (ntl nell) k (n-1))"
using assms unfolding nsubn_def1 
using ntl_ntaken[of "n-k" "ndropn k nell" ] ntl_ndropn[of k nell] 
  by (metis diff_diff_cancel diff_right_commute diff_zero nellist.sel(4) nsubn_def1 nsubn_same)

lemma nsubn_nsubn:
 assumes "n1 \<le> n2"
         "n0 \<le> n4"
         "n2 \<le> n4-n0"
         "n4 \<le> n3"
         "n3 \<le> nlength nell"
 shows   "  (nsubn (nsubn nell n0 n3) n1 n2 ) = (nsubn (nsubn nell n0 n4) n1 n2 )"
proof -
 have 1: " nlength(nsubn nell n0 n3 ) = n3 -n0"
   using assms by (metis enat_minus_mono1 idiff_enat_enat min.orderE nsubn_nlength)
 have 2: " nlength (nsubn (nsubn nell n0 n3 ) n1 n2 ) = n2 -n1 "
    using assms 
    by (metis Nat.le_diff_conv2 enat_minus_mono1 enat_ord_simps(1) idiff_enat_enat le_trans 
        min.orderE nsubn_nlength)
 have 3: " nlength(nsubn nell n0 n4) = n4 -n0"
     using assms nsubn_nlength[of nell n0 n4] 
      unfolding min_def 
      by (metis enat_minus_mono1 idiff_enat_enat min.coboundedI1 min.left_commute min_absorb1
          min_enat_simps(1))
 have 4: " nlength (nsubn (nsubn nell n0 n4 ) n1 n2 ) = n2 -n1"  
    using assms 
    by (metis "3" enat_minus_mono1 enat_ord_simps(1) idiff_enat_enat min.orderE nsubn_nlength)
 have 5: " \<And>i. i\<le> (n3 -n0) \<longrightarrow> (nnth (nsubn nell n0 n3 ) i ) = (nnth nell (n0+i))"
  using assms by (simp add: nsubn_def1 ntaken_nnth)
 have 6: " \<And>i . i\<le> (n2-n1) \<longrightarrow> 
          (nnth (nsubn (nsubn nell n0 n3 ) n1 n2 ) i) = (nnth (nsubn nell n0 n3 ) (n1+i))"
    using assms by (simp add: Nat.le_diff_conv2 nsubn_nnth) 
 have 7: "\<And>i . i\<le> (n2-n1) \<longrightarrow> 
          (nnth (nsubn (nsubn nell n0 n3 ) n1 n2 ) i) = (nnth nell (n0+(n1+i))) "
    using "5" "6" assms by auto
 have 8: " n0\<le> n4 \<and> n4 \<le> nlength nell"
   using assms by (simp add: order_subst2)
 have 9: " \<And>i. i\<le> (n4 -n0) \<longrightarrow> (nnth (nsubn nell n0 n4) i ) = (nnth nell (n0+i))"
   using "8" by (simp add: nsubn_def1 ntaken_nnth)
 have 10: "\<And>i. i\<le> (n2-n1) \<longrightarrow> 
           (nnth (nsubn (nsubn nell n0 n4) n1 n2 ) i) = (nnth (nsubn nell n0 n4) (n1+i))"
        by (simp add: nsubn_def1 ntaken_nnth)
 have 11: "\<And>i. i\<le> (n2-n1) \<longrightarrow> 
           (nnth (nsubn (nsubn nell n0 n4)  n1 n2 ) i) = (nnth nell (n0 +(n1+i))) "
      by (metis "10" "9" Nat.le_diff_conv2 add.commute assms(1) assms(3) le_trans)
 have 12: "\<And>i. i\<le> (n2-n1) \<longrightarrow> 
           (nnth (nsubn (nsubn nell n0 n3) n1 n2 ) i) = (nnth (nsubn (nsubn nell n0 n4)  n1 n2 ) i) "
   by (simp add: "11" "7")
 from 12 2 4 show ?thesis 
 using nellist_eq_nnth_eq[of "(nsubn (nsubn nell n0 n3) n1 n2 )" "(nsubn (nsubn nell n0 n4)  n1 n2 )"]
 enat_ord_simps(1) by presburger
qed

lemma nsubn_nsubn_1:
 assumes "n1 \<le> n2"
         "n0 \<le> n3"
         "n2 \<le> n3-n0"
         "n3 \<le> nlength nell"
 shows   "(nsubn (nsubn nell n0 n3) n1 n2 ) = (nsubn nell (n0+n1) (n0+n2))"
proof -
 have 0: "nlength(nsubn (nsubn nell n0 n3) n1 n2 ) = n2 - n1"
     using assms 
     by (metis enat_minus_mono1 idiff_enat_enat min.orderE nsubn_nlength of_nat_eq_enat of_nat_mono)
 have 1: " nlength(nsubn nell (n1+n0) (n2+n0)) = (n2+n0) -(n1+n0)"
   using assms nsubn_nlength[of nell "(n1+n0)" "(n2+n0)"]  
    unfolding min_def
    by (metis Nat.le_diff_conv2 dual_order.trans enat_minus_mono1 enat_ord_simps(1) idiff_enat_enat)
 have 10: "  (n2+n0) -(n1+n0) = n2 -n1"
     using diff_cancel2 by blast 
 have 2: "\<And>i. i\<le> (n2-n1) \<longrightarrow> 
            (nnth (nsubn (nsubn nell n0 n3) n1 n2 ) i) = (nnth nell (n0+(n1+i))) "
   using assms by (simp add: nsubn_nnth)
 have 3: "\<And>i. i\<le> (n2-n1) \<longrightarrow> 
           (nnth (nsubn nell (n0+n1) (n0+n2)) i) = (nnth nell (n0+(n1+i))) "
    using assms by (metis "10" add.assoc add.commute min.orderE nsubn_nnth)
 have 4: "\<And>i. i\<le> (n2-n1) \<longrightarrow> 
          (nnth (nsubn (nsubn nell n0 n3) n1 n2 ) i) = (nnth (nsubn nell (n0+n1) (n0+n2)) i)"
   by (simp add: "2" "3")
 show ?thesis using nellist_eq_nnth_eq[of "(nsubn (nsubn nell n0 n3) n1 n2 )" 
 "(nsubn nell (n0+n1) (n0+n2))"]
     0 10 4 1 
    by (metis add.commute enat_ord_simps(1))  
qed

lemma nsubn_eq:
 assumes "n\<le>m"
         "m\<le>nlength nellx"
         "m\<le>nlength nelly"
         "\<forall>j. n \<le>j \<and> j \<le>m \<longrightarrow> nnth nellx j = nnth nelly j" 
 shows   "nsubn nellx n m = nsubn nelly n m" 
proof -
 have 1: "nlength (nsubn nellx n m) = nlength (nsubn nelly n m)"
   using assms by (metis  enat_minus_mono1 min_def nsubn_nlength) 
 have 2: "\<And>j. j\<le>nlength (nsubn nellx n m) \<longrightarrow> nnth (nsubn nellx n m) j = nnth (nsubn nelly n m) j"
   using assms by (simp add: Nat.le_diff_conv2 nsubn_nlength nsubn_nnth)
 show ?thesis 
 by (simp add: "1" "2" nellist_eq_nnth_eq)
qed


subsection \<open>nfuse\<close>

lemma nfuse_def1: 
 "nfuse nellx nelly = (if is_NNil nelly then  nellx else nappend nellx (ntl nelly))"
apply transfer
unfolding lfuse_def by simp force

lemma nfuse_NCons_a:
 "nfuse (NCons x nellx) nelly = (NCons x (nfuse nellx nelly)) " 
by (simp add: nfuse_def1)

lemma nfuse_NCons_b:
 " nfuse nellx (NCons y nelly) = nappend nellx nelly " 
by (simp add: nfuse_def1)

lemma nfuse_simps [simp]: 
 shows nhd_nfuse: "nhd(nfuse nellx nelly) = 
                   (if is_NNil nellx then
                       (if is_NNil nelly then nhd nellx else nlast nellx)
                    else nhd nellx)
                     " 
  and  ntl_nfuse: "ntl(nfuse nellx nelly) = 
                  (if is_NNil nellx then 
                     (if is_NNil nelly then ntl nellx else ntl nelly) 
                   else (if is_NNil nelly then ntl nellx 
                         else nappend (ntl nellx) (ntl nelly))) " 
by (simp_all add: nfuse_def1)

lemma nfuse_nbutlast: 
 assumes " nlast nellx = nfirst nelly" 
         " \<not> is_NNil nellx" 
         " \<not> is_NNil nelly" 
  shows  " nfuse nellx nelly = nappend (nbutlast nellx) nelly" 
using assms
by (metis nappend_NCons nappend_NNil nappend_assoc nappend_nbutlast_nlast_id nappend_snocn 
    nellist.collapse(2) nellist.sel(3) nfuse_def1 nhd_snocn)

lemma nfuse_nlength:
  shows    " nlength (nfuse nellx nelly) = (nlength nellx) + (nlength nelly)"
  unfolding nfuse_def1 
  by(cases nelly) (simp, simp add: eSuc_plus_1)

lemma nfuse_nnth:
assumes "i\<le> nlength (nfuse nellx nelly)"
        "nlast nellx = nfirst nelly"
shows   "(i \<le> nlength nellx \<longrightarrow> nnth (nfuse nellx nelly) i = nnth nellx i)
         \<and> 
         (nlength nellx < i \<and> i\<le> nlength (nfuse nellx nelly) \<longrightarrow> 
           nnth (nfuse nellx nelly) i = nnth nelly (i - (the_enat (nlength nellx))))"
unfolding nfuse_def1
by (cases nelly)
   (auto simp add: nnth_nappend,
    metis Suc_diff_Suc  enat_iless enat_ord_simps(2) ndropn_Suc_NCons ndropn_nfirst the_enat.simps)

lemma nfuse_nnth_a:
 assumes " j \<le> nlength nelly"
         " nlast nellx = nfirst nelly "
         "nfinite nellx" 
 shows   " nnth (nfuse nellx nelly) ((the_enat(nlength nellx)) +j) = (nnth nelly j)"
 using assms unfolding nfuse_def1
 by (cases "is_NNil nelly")
    (simp_all,
     metis assms(2) assms(3) is_NNil_def is_NNil_imp_nfinite ndropn_nlast ndropn_nfirst 
     ndropn_nnth nnth_NNil,
     metis enat_le_plus_same(2) gen_nlength_def nappend_NNil ndropn_nlast ndropn_nappend2 
     ndropn_nnth nellist.case_eq_if nellist.collapse(2) nfinite_nlength_enat nfirst_def 
     nlength_code the_enat.simps)

lemma nfuse_nappend:
 assumes " nlast nellx = nfirst nelly " 
 shows   " nfuse nellx nelly = 
            (if nfinite nellx then 
              (if is_NNil nellx then nelly else nappend (ntaken (the_enat(epred(nlength nellx))) nellx) nelly)
             else nellx) " 
proof (cases nelly)
 case (NNil x1)
 then show ?thesis  
 proof (cases "nfinite nellx")
 case True
 then show ?thesis 
 proof (cases nellx)
  case (NNil x1)
  then show ?thesis using assms 
   by (simp add: nfuse_def1)    
      (metis nappend_snocn nellist.collapse(1) nellist.collapse(2) nhd_nappend nhd_snocn)
  next
  case (NCons x21 x22)
  then show ?thesis  unfolding nfuse_def1 using assms NNil NCons True
   by (auto simp add: nfirst_def)
      (metis True add_diff_cancel_left' assms eSuc_enat ndropn_nfirst ndropn_nlast 
       nellist_split_2_last nfinite_nlength_enat nlast_NCons nlength_NCons plus_1_eq_Suc 
       the_enat.simps zero_less_Suc)
  qed
  next
 case False
 then show ?thesis
 by (simp add: NNil nfuse_def1) 
 qed
 next
 case (NCons x21 x22)
 then show ?thesis 
 proof (cases "nfinite nellx")
  case True
  then show ?thesis 
    unfolding nfuse_def1
     proof auto
      show "is_NNil nelly \<Longrightarrow> is_NNil nellx \<Longrightarrow> nellx = nelly" 
        by (simp add: NCons)
      show " nfinite nellx \<Longrightarrow> is_NNil nelly \<Longrightarrow> \<not> is_NNil nellx \<Longrightarrow>
             nellx = nappend (ntaken (the_enat (epred (nlength nellx))) nellx) nelly"
        using assms NCons True by simp  
      show "\<not> is_NNil nelly \<Longrightarrow> is_NNil nellx \<Longrightarrow> nappend nellx (ntl nelly) = nelly"
         using assms NCons True 
         by (metis nappend_NNil nellist.collapse(1) nellist.sel(5) nnth_0 ntaken_0 ntaken_nlast)
      show " nfinite nellx \<Longrightarrow> \<not> is_NNil nelly \<Longrightarrow> \<not> is_NNil nellx \<Longrightarrow> 
             nappend nellx (ntl nelly) = nappend (ntaken (the_enat (epred (nlength nellx))) nellx) nelly " 
         using assms NCons True 
      by (cases nellx)
         ( auto,
          metis  True add_diff_cancel_left' eSuc_enat nappend_NCons nappend_NNil nappend_assoc 
          ndropn_eq_NNil ndropn_nlast nellist.sel(5) nellist_split_2_last nfinite_nlength_enat 
          nlast_NNil nlast_ntl nlength_NCons nnth_0  ntaken_nlast ntl_ntaken_0 plus_1_eq_Suc 
          the_enat.simps zero_less_Suc)
    qed     
  next
  case False
  then show ?thesis
  by (simp add: nappend_inf nfuse_def1) 
 qed
qed   

lemma nfuse_leftneutral : 
    "nfuse (NNil (nfirst nell)) nell = nell"
by (simp add: nfuse_nappend)

lemma nfuse_rightneutral : 
    "nfuse nell (NNil (nlast nell)) = nell"
unfolding nfuse_def
by simp

lemma nfirst_nfuse :
 assumes " nlast nellx = nfirst nelly"
 shows   " nfirst (nfuse nellx nelly) = nfirst nellx"
using assms unfolding nfuse_def1 by transfer auto

lemma nlast_nfuse :
 assumes "nlast nellx = nfirst nelly"
         "nfinite nellx" 
 shows   " nlast (nfuse nellx nelly) = nlast nelly"
using assms unfolding nfuse_def1 by transfer (auto, metis lhd_LCons_ltl llast_LCons llast_lappend)

lemma nfuseassoc : 
 shows   " (nfuse nellx (nfuse nelly nellz)) = (nfuse (nfuse nellx nelly) nellz)"
unfolding nfuse_def1 
by transfer (auto simp add: lappend_assoc, simp add: lappend_ltl)

lemma ntaken_nfuse :
  assumes  " nlast nellx = nfirst nelly "
           "nfinite nellx"
  shows    " (ntaken (the_enat (nlength nellx)) (nfuse nellx nelly)) = nellx"
proof (cases "is_NNil nelly")
case True
then show ?thesis using assms by (metis ndropn_eq_NNil ndropn_nlast nfuse_def1 ntaken_all)
next
case False
then show ?thesis using assms 
by (metis add.left_neutral enat_le_plus_same(2) nfinite_nlength_enat nfuse_def1 ntaken_all
     ntaken_nappend1 the_enat.simps)
qed

lemma ndropn_nfuse :
 assumes   "nlast nellx = nfirst nelly "
           "nfinite nellx" 
 shows     "(ndropn (the_enat (nlength nellx)) (nfuse nellx nelly)) = nelly"
proof (cases "is_NNil nelly")
case True
then show ?thesis using assms by (metis ndropn_nlast nfuse_def1 nfuse_leftneutral) 
next
case False
then show ?thesis using assms
by (metis enat_le_plus_same(2) gen_nlength_def ndropn_nappend2 ndropn_nlast nfinite_nlength_enat
           nfuse_def1 nfuse_leftneutral nlength_code the_enat.simps) 
qed

lemma nfuse_ntaken_ndropn_nlength :
assumes "n \<le> nlength nell"
shows   "nlength (nfuse (ntaken n nell) (ndropn n nell)) = nlength nell "
using assms 
by (metis dual_order.order_iff_strict enat_add_sub_same infinity_ileE less_eqE min.absorb1 
    ndropn_nlength nfuse_nlength ntaken_nlength)

lemma nfuse_ntaken_ndropn_nnth :
assumes "n \<le> nlength nell"
        "i \<le> nlength nell"
shows   "   nnth (nfuse (ntaken n nell) (ndropn n nell)) i = nnth nell i "
using assms 
nfuse_nnth[of "i" "(ntaken n nell)" "(ndropn n nell)"]  
    nfuse_ntaken_ndropn_nlength[of "n" "nell"]
by (metis dual_order.strict_iff_order enat_ord_simps(1) le_add_diff_inverse min.orderE ndropn_nfirst 
    ndropn_nnth not_less ntaken_nlast ntaken_nlength ntaken_nnth the_enat.simps)

lemma nfuse_ntaken_ndropn:
assumes "n\<le> nlength nell"
shows   "nfuse (ntaken n nell) (ndropn n nell) = nell"
using assms 
by (simp add: nfuse_ntaken_ndropn_nlength nfuse_ntaken_ndropn_nnth nellist_eq_nnth_eq)

lemma nfuse_nnth_var:
assumes "  enat i \<le> nlength (nfuse nellx nelly)"  
        " nlast nellx = nfirst nelly  " 
shows   " (enat i \<le> nlength nellx \<longrightarrow> nnth (nfuse nellx nelly) i = nnth nellx i) \<and>
          (nlength nellx \<le> enat i \<and> enat i \<le> nlength (nfuse nellx nelly) \<longrightarrow> 
           nnth (nfuse nellx nelly) i = nnth nelly (i - the_enat (nlength nellx))) " 
 using nfuse_nnth assms
 by (metis cancel_comm_monoid_add_class.diff_cancel dual_order.strict_iff_order 
     ndropn_nfuse nlength_eq_enat_nfiniteD nnth_zero_ndropn the_enat.simps) 
 
lemma nset_nfuse:
 "nset (nfuse nellx nelly) =
  (if nfinite nellx then 
     (if is_NNil nelly then nset nellx else nset nellx \<union> nset (ntl nelly)) 
   else nset nellx)"
by (simp add: nfuse_def1 nset_nappend) 

lemma nsubn_nfuse:
 assumes " (enat k) \<le> n "
         " (enat n) \<le> m"
         " (enat m) \<le> nlength nell"
 shows   "nfuse (nsubn nell k n) (nsubn nell n m) = (nsubn nell k m)"
using assms 
proof -
 have 1: " nlast(nsubn nell k n) = (nnth nell n)"
   by (metis assms(1) enat_ord_simps(1) le_add_diff_inverse2 nsubn_def1 ntaken_ndropn_nlast)
 have 2: " nfirst(nsubn nell n m) = (nnth nell n)"
   by (simp add: ndropn_nfirst nsubn_def1 ntaken_ndropn_nfirst_a)
 have 3: "nlength (nsubn nell k n) = n -k"
   using assms nsubn_nlength[of nell k n]
   by (metis dual_order.trans enat_minus_mono1 idiff_enat_enat min_absorb1)  
 have 4: "nlength (nsubn nell n m) = m -n"
   by (metis assms(3) enat_minus_mono1 idiff_enat_enat min_def nsubn_nlength)    
 have 5: "nlength (nsubn nell k m) = m -k"
   by (metis assms(3) enat_minus_mono1 idiff_enat_enat min_absorb1 nsubn_nlength) 
 have 6: " nlength(nfuse (nsubn nell k n) (nsubn nell n m)) = nlength(nsubn nell k m)"
   unfolding nsubn_def
   by (metis "3" "4" "5" Nat.add_diff_assoc2 assms(1) assms(2) enat_ord_simps(1) nfuse_nlength 
       nsubn_def ordered_cancel_comm_monoid_diff_class.add_diff_inverse plus_enat_simps(1)) 
 have 7: " (\<forall> i. i \<le> nlength(nsubn nell k m) \<longrightarrow>
              (nnth (nfuse (nsubn nell k n) (nsubn nell n m)) i) = (nnth nell (k+i)))"
   proof 
     fix i
     show "i \<le> nlength(nsubn nell k m) \<longrightarrow>
              (nnth (nfuse (nsubn nell k n) (nsubn nell n m)) i) = (nnth nell (k+i))  "
     proof -
       have 41: "nlength (nsubn nell k m) = (m-k) "
         using "5" by blast         
       have 42: "i \<le> nlength (nsubn nell k m) \<longrightarrow> nnth (nfuse (nsubn nell k n) (nsubn nell n m)) i =
                  (if i \<le> n -k then (nnth (nsubn nell k n) i) 
                                             else (nnth (nsubn nell n m) (i-(n-k))))"
         by (simp add: "1" "2" "3" "6" nfuse_nnth)
       have 43: " i \<le> (m-k) \<longrightarrow> nnth (nfuse (nsubn nell k n) (nsubn nell n m)) i = 
                  (if i\<le> (n-k) then (nnth nell (k+i)) else (nnth nell (n+(i-(n-k)))))"
         using assms 42 5 unfolding nsubn_def1 
         by (auto simp add: ntaken_nnth) 
             (metis enat_ord_simps(1) min.bounded_iff,
              metis enat_ord_simps(1) min.bounded_iff)
       have 44: "i \<le> (m-k) \<longrightarrow> nnth (nfuse (nsubn nell k n) (nsubn nell n m)) i = 
                  (nnth nell (k+i))  "
         by (metis "43" Nat.diff_diff_right Nat.le_diff_conv2 add.commute assms(1) enat_ord_simps(1) 
             le_add_diff_inverse nat_le_linear)
       show ?thesis 
         by (simp add: "41" "44")
     qed
   qed
  have 8: "(\<forall> i. i \<le> nlength(nsubn nell k m) \<longrightarrow> (nnth (nsubn nell k m) i) = (nnth nell (k+i))) "
    using assms by (simp add: nsubn_def1 ntaken_nnth)
  show ?thesis 
    by (simp add: "6" "7" "8" nellist_eq_nnth_eq)
qed

lemma nmap_nfuse: 
 "nmap f (nfuse nellx nelly) = nfuse (nmap f nellx) (nmap f nelly) "
by (simp add: nfuse_def1 nmap_nappend_distrib)
 
lemma nzip_nfuse: 
 assumes "nlength \<sigma>1 = nlength l1"
         "nlength \<sigma>2 = nlength l2"
         "nfirst \<sigma>2 = nlast \<sigma>1" 
         " nfinite \<sigma>1"
         " nfirst l2 = nlast l1" 
 shows " (nzip (nfuse \<sigma>1 \<sigma>2) (nfuse l1 l2)) = nfuse (nzip \<sigma>1 l1) (nzip \<sigma>2 l2)"
using assms
apply transfer 
by (simp add: epred_inject lfuse_def lzip_lappend)

subsection \<open>nridx and nidx\<close>

lemma nridx_nidx: 
 "nridx (<) nell = nidx nell" 
apply transfer
by (simp add: ridx_lidx)

lemma nridx_expand:
 "nridx R nell \<longleftrightarrow> (\<forall>i. (Suc i) \<le> nlength nell \<longrightarrow> R (nnth nell i) (nnth nell (Suc i)))"
by transfer
  (auto simp add: min_def Suc_ile_eq, 
   metis co.enat.exhaust_sel eSuc_enat ileI1 iless_Suc_eq llength_eq_0 ridx_def, 
   metis Extended_Nat.eSuc_mono co.enat.exhaust_sel eSuc_enat llength_eq_0 order_le_less ridx_def)


lemma nidx_expand:
 "nidx nell \<longleftrightarrow> (\<forall>i. (Suc i) \<le> nlength nell \<longrightarrow> nnth nell i < nnth nell (Suc i))"
using nridx_nidx nridx_expand by blast

lemma nridx_NCons:
 " nridx R (NCons x nell) \<longleftrightarrow> 
   (\<forall>n. (Suc n) \<le> eSuc (nlength nell)  \<longrightarrow> R (nnth (NCons x nell) n) (nnth (NCons x nell) (Suc n)))"
using nridx_expand[of R "(NCons x nell)"]
by auto

lemma nidx_NCons:
 " nidx  (NCons x nell) \<longleftrightarrow> 
   (\<forall>n. (Suc n) \<le> eSuc (nlength nell)  \<longrightarrow>  (nnth (NCons x nell) n) < (nnth (NCons x nell) (Suc n)))"
using nridx_NCons[of "(<)" x nell] nridx_nidx by blast

lemma nridx_LCons_conv:
 "(\<forall>n. (Suc n) \<le> eSuc (nlength nell)  \<longrightarrow> R (nnth (NCons x nell) n) (nnth (NCons x nell) (Suc n))) \<longleftrightarrow>
  R x (nfirst nell) \<and>
  (\<forall>n. 0\<le>n \<and> (Suc n) \<le>  (nlength nell)  \<longrightarrow> R (nnth nell n) (nnth nell (Suc n))) " 
proof -
 have 1: "(\<forall>n. (Suc n) \<le> eSuc (nlength nell)  \<longrightarrow> R (nnth (NCons x nell) n) (nnth (NCons x nell) (Suc n))) \<longleftrightarrow>
          (\<forall>n. 0\<le>n \<and> (Suc n) \<le> eSuc (nlength nell)  \<longrightarrow> R (nnth (NCons x nell) n) (nnth (NCons x nell) (Suc n))) "
   by blast
 have 2: "(\<forall>n. 0\<le>n \<and> (Suc n) \<le> eSuc (nlength nell)  \<longrightarrow> R (nnth (NCons x nell) n) (nnth (NCons x nell) (Suc n))) \<longleftrightarrow>
          R x (nfirst nell) \<and>
          (\<forall>n. 1\<le>n \<and> (Suc n) \<le> eSuc (nlength nell)  \<longrightarrow> R (nnth (NCons x nell) n) (nnth (NCons x nell) (Suc n)))" 
  by (metis "1" One_nat_def add_diff_cancel_left' eSuc_ile_mono enat_le_plus_same(1) gen_nlength_def le_SucE le_add1 
     ndropn_Suc_NCons ndropn_nfirst ndropn_nfuse nfuse_leftneutral nlast_NNil nlength_NNil nlength_code 
     nlength_eq_enat_nfiniteD nnth_0 one_eSuc one_enat_def plus_1_eq_Suc the_enat_0 zero_enat_def)
 have 3: "R x (nfirst nell) \<and>
          (\<forall>n. 1\<le>n \<and> (Suc n) \<le> eSuc (nlength nell)  \<longrightarrow> R (nnth (NCons x nell) n) (nnth (NCons x nell) (Suc n))) \<longleftrightarrow>
          R x (nfirst nell) \<and>
          (\<forall>n. 1\<le>n \<and> (n) \<le>  (nlength nell)  \<longrightarrow> R (nnth (NCons x nell) n) (nnth (NCons x nell) (Suc n)))" 
    by (metis eSuc_enat eSuc_ile_mono)
 have 4: "R x (nfirst nell) \<and>
          (\<forall>n. 1\<le>n \<and> (n) \<le>  (nlength nell)  \<longrightarrow> R (nnth (NCons x nell) n) (nnth (NCons x nell) (Suc n))) \<longleftrightarrow>
         R x (nfirst nell) \<and>
          (\<forall>n. 0\<le>(n-1) \<and> (Suc(n-1)) \<le>  (nlength nell)  \<longrightarrow> R (nnth  nell (n-1)) (nnth nell (Suc (n-1))))" 
   by (metis add.commute add.right_neutral diff_add le_add1 nnth_Suc_NCons plus_1_eq_Suc)
 have 5: "R x (nfirst nell) \<and>
          (\<forall>n. 0\<le>(n-1) \<and> (Suc(n-1)) \<le>  (nlength nell)  \<longrightarrow> R (nnth  nell (n-1)) (nnth nell (Suc (n-1)))) \<longleftrightarrow>
          R x (nfirst nell) \<and>
          (\<forall>n. 0\<le>n \<and> (Suc n) \<le>  (nlength nell)  \<longrightarrow> R (nnth  nell n) (nnth nell (Suc n)))"
     by (metis diff_Suc_1) 
 show ?thesis 
 using "1" "2" "3" "4" "5" by presburger
qed

lemma nidx_LCons_conv:
 "(\<forall>n. (Suc n) \<le> eSuc (nlength nell)  \<longrightarrow>  (nnth (NCons x nell) n) < (nnth (NCons x nell) (Suc n))) \<longleftrightarrow>
   x < (nfirst nell) \<and>
  (\<forall>n. 0\<le>n \<and> (Suc n) \<le>  (nlength nell)  \<longrightarrow>  (nnth nell n) < (nnth nell (Suc n))) " 
by (metis nridx_LCons_conv)

lemma nridx_LCons_1 [simp]:
 "nridx R (NCons x nell) \<longleftrightarrow>  ( R x (nfirst nell) \<and> nridx R nell )"
by (metis nridx_LCons_conv nridx_NCons nridx_expand zero_le)

lemma nidx_LCons_1 [simp]:
 "nidx  (NCons x nell) \<longleftrightarrow>  (  x < (nfirst nell) \<and> nidx nell )"
by (metis nridx_LCons_1 nridx_nidx)

lemma nridx_less:
 assumes " nridx R nell"
         " Suc(n+k) \<le> nlength nell"
         "transp R" 
 shows   " R (nnth nell n) (nnth nell (Suc(n+k)))"
using assms
proof (induct k)
case 0
then show ?case 
by (simp add: nridx_expand)
next
case (Suc k)
then show ?case 
by (metis add_Suc_right dual_order.trans eSuc_enat ile_eSuc nridx_expand transpE)
qed

lemma nidx_less:
 assumes " nidx nell"
         " Suc(n+k) \<le> nlength nell"
 shows   " nnth nell n < nnth nell (Suc(n+k))"
using assms
by (simp add: nridx_less nridx_nidx)

lemma nridx_less_eq:
 assumes " nridx R nell"
         " k \<le> j"
         " j \<le> nlength nell"
         "transp R" 
         "reflp R" 
 shows   " R (nnth nell k)  (nnth nell j)"
proof (cases "k=j")
case True
then show ?thesis using assms by (meson  reflpE)
next
case False
then show ?thesis using assms 
by (metis (full_types) Suc_diff_Suc add.left_commute le_add_diff_inverse nridx_less order_neq_le_trans 
    plus_1_eq_Suc)
qed

lemma nidx_less_eq:
 assumes " nidx  nell"
         " k \<le> j"
         " j \<le> nlength nell" 
 shows   "  (nnth nell k) \<le>  (nnth nell j)"
using assms 
by (metis Orderings.order_eq_iff antisym_conv2 less_iff_Suc_add nidx_less order.strict_implies_order)

lemma nridx_less_last:
 assumes " nridx R nell"
         " Suc i < k"
         " nlength nell = (enat k) "
         "transp R" 
 shows   " R (nnth nell i) (nnth nell (k-1))"
using assms less_imp_Suc_add nridx_less by fastforce

lemma nidx_less_last:
 assumes " nidx nell"
         " Suc i < k"
         " nlength nell = (enat k) "
 shows   " nnth nell i < nnth nell (k-1)"
using assms less_imp_Suc_add nidx_less by fastforce

lemma nidx_less_last_1:
 assumes  " nidx nell"
          " i<nlength nell" 
          "nlength nell = (enat k)" 
 shows    " nnth nell i < nnth nell (k) "
using assms 
by (metis enat_ord_simps(2) less_imp_Suc_add linorder_le_cases nridx_less nridx_nidx transp_on_less)


lemma nridx_gr_first:
 assumes " nridx R nell"
         " 0<i"
         " i \<le> nlength nell"
         "transp R" 
 shows   " R (nnth nell 0)  (nnth nell i)"
using assms nridx_less[of R nell 0 "i-1"] by simp

lemma nidx_gr_first:
 assumes " nidx nell"
         " 0<i"
         " i \<le> nlength nell"
 shows   " (nnth nell 0) < nnth nell i"
using assms nidx_less[of nell 0 "i-1"] 
by simp


lemma nridx_ntake_a: 
 assumes "nridx R nell"
         "n \<le> nlength nell"
 shows   "nridx R (ntake n nell) " 
using assms 
by transfer
   (metis co.enat.exhaust_sel eSuc_ile_mono llength_eq_0 ridx_ltake_a)

lemma nidx_ntake_a: 
 assumes "nidx  nell"
         "n \<le> nlength nell"
 shows   "nidx  (ntake n nell) " 
using assms 
using nridx_ntake_a nridx_nidx by blast

lemma nridx_nappend_nfinite: 
assumes " nfinite nell1 "
shows   " nridx R (nappend nell1 nell2) \<longleftrightarrow> 
          nridx R nell1 \<and> (R (nlast nell1)  (nfirst nell2)) \<and> nridx R nell2 " 
using assms
by transfer
   (simp add: ridx_lappend_lfinite)

lemma nidx_nappend_nfinite: 
assumes " nfinite nell1 "
shows   " nidx  (nappend nell1 nell2) \<longleftrightarrow> 
          nidx  nell1 \<and> ( (nlast nell1) < (nfirst nell2)) \<and> nidx  nell2 " 
using assms
by (metis nridx_nappend_nfinite nridx_nidx)

lemma nidx_nfuse: 
 assumes " nfinite nell1"
         " nidx nell1"
         " nnth nell1 0 = 0 "
         " nidx nell2 "
         "nnth nell2 0 = nlast nell1"
 shows " nidx (nfuse nell1 nell2) " 
using assms 
proof (cases "is_NNil nell2")
case True
then show ?thesis unfolding nfuse_def1 
by (simp add: assms(2))
next
case False
then show ?thesis 
    proof -
     have 1: "nlast nell1 = nfirst nell2" 
       by (metis assms(5) ndropn_0 ndropn_nfirst)
     have 2: "nidx (ntl nell2)"
       by (metis False assms(4) eSuc_enat ileI1 iless_Suc_eq nellist.collapse(2) nidx_expand 
           nlength_NCons nnth_ntl) 
     have 3: "nlast nell1 < nfirst(ntl nell2)" 
         by (metis False assms(4) assms(5) eSuc_enat ileI1 iless_Suc_eq ndropn_0 ndropn_nfirst 
             nellist.collapse(2) nhd_conv_nnth nidx_expand nlength_NCons nnth_ntl zero_enat_def zero_le)
     have 4: "nidx (nappend nell1 (ntl nell2))" 
        by (simp add: "2" "3" assms(1) assms(2) nidx_nappend_nfinite)
     show ?thesis using False unfolding nfuse_def1 
     using "4" by auto
    qed 
qed


lemma nridx_ndropn: 
 assumes "nridx R nell"
         "n \<le> nlength nell" 
 shows   "nridx R (ndropn n nell) " 
using assms
by transfer
   (metis co.enat.exhaust_sel iless_Suc_eq ldrop_enat llength_eq_0 min.orderE nless_le 
    ridx_ldrop the_enat.simps)

lemma nidx_ndropn: 
 assumes "nidx  nell"
         "n \<le> nlength nell" 
 shows   "nidx  (ndropn n nell) " 
using assms
using nridx_ndropn nridx_nidx by blast

lemma nridx_ntake_all: 
 assumes "\<And>n. n\<le> nlength nell \<Longrightarrow> nridx R (ntake (enat n) nell) "
 shows   "nridx R nell"
using assms
by (auto simp add: nridx_expand)
   (metis Suc_ile_eq  linorder_le_cases ntake_nnth ntake_nnth order_less_imp_le ) 

lemma nidx_ntake_all: 
 assumes "\<And>n. n\<le> nlength nell \<Longrightarrow> nidx (ntake (enat n) nell) "
 shows   "nidx nell" 
using assms 
using nridx_nidx nridx_ntake_all by blast

lemma nridx_ntake: 
 assumes "nridx R (ntake n nell) "
         "n \<le> nlength nell" 
         "k \<le> n"
 shows   "nridx R (ntake (enat k) nell)" 
using assms
using nridx_ntake_a by fastforce

lemma nidx_ntake: 
 assumes "nidx  (ntake n nell) "
         "n \<le> nlength nell" 
         "k \<le> n"
 shows   "nidx  (ntake (enat k) nell)" 
using assms 
using nridx_nidx nridx_ntake by blast

lemma nidx_imp_ndistinct:
 assumes "nidx  nell"
 shows  "ndistinct nell"
using assms
apply transfer
using lidx_imp_ldistinct by auto


lemma ndistinct_Ex1:
 assumes "ndistinct nell"
         " x \<in> nset nell"
 shows   " \<exists>!i. i \<le> nlength nell \<and> (nnth nell i) = x"
using assms 
by transfer
   (auto,
    metis co.enat.exhaust_sel iless_Suc_eq ldistinct_Ex1 llength_eq_0 min.orderE the_enat.simps,
    metis co.enat.exhaust_sel iless_Suc_eq ldistinct_conv_lnth llength_eq_0)

lemma nidx_nset_eq:
 assumes " nidx nellx"
         " nidx nelly"
         "nset nellx = nset nelly"
 shows   "nellx = nelly"
using assms
by transfer
   (simp add: bi_unique_cr_nellist_help lidx_lset_eq nidx.rep_eq)

lemma nridx_nfuse_nfirst_nlast: 
 assumes " nridx R nell1 "
         " (nnth nell1 0 ) = (0::nat) "
         " nridx R nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "nfinite nell "
         " nlast nell1 = cp "
 shows " nlast nell1 = nfirst(nmap (\<lambda>x. x+cp) nell2) " 
using assms 
by (metis add.commute add.right_neutral ndropn_0 ndropn_nfirst nnth_nmap zero_enat_def zero_le)

lemma nidx_nfuse_nfirst_nlast: 
 assumes " nidx  nell1 "
         " (nnth nell1 0 ) = (0::nat) "
         " nidx  nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "nfinite nell "
         " nlast nell1 = cp "
 shows " nlast nell1 = nfirst(nmap (\<lambda>x. x+cp) nell2) " 
using assms 
by (metis add.commute add.right_neutral ndropn_0 ndropn_nfirst nnth_nmap zero_enat_def zero_le)

lemma nridx_nfuse_nnth_cp: 
 assumes " nridx R nell1 "
         " (nnth nell1 0 ) = 0 "
         " nridx R nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "nfinite nell2" 
         "nfinite nell "
         " nlast nell1 = cp "
         " nlast nell2 = the_enat((nlength nell)) - cp "
         " i \<le> (nlength nell2)" 
         "cp \<le> nlength nell " 
 shows "nnth (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) (the_enat((nlength nell1)) + i) = cp + (nnth nell2 i) "
proof -
 have 1: "nlast nell1 = nfirst (nmap (\<lambda>x. x+cp) nell2)"
   by (metis add_0 assms(4) assms(8) ndropn_0 ndropn_nfirst nnth_nmap zero_enat_def zero_le) 
 have 2: "nnth (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) (the_enat((nlength nell1)) + i) = 
          nnth (nmap (\<lambda>x. x+cp) nell2) i"
      by (simp add: "1" assms nfuse_nnth_a)
 have 3: " nnth (nmap (\<lambda>x. x+cp) nell2) i = cp + (nnth nell2 i)"
   by (simp add: assms) 
 show ?thesis
 by (simp add: "2" "3")
qed

lemma nidx_nfuse_nnth_cp: 
 assumes " nidx  nell1 "
         " (nnth nell1 0 ) = 0 "
         " nidx  nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "nfinite nell2" 
         "nfinite nell "
         " nlast nell1 = cp "
         " nlast nell2 = the_enat((nlength nell)) - cp "
         " i \<le> (nlength nell2)" 
         "cp \<le> nlength nell " 
 shows "nnth (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) (the_enat((nlength nell1)) + i) = cp + (nnth nell2 i) "
using assms nridx_nfuse_nnth_cp nridx_nidx by blast

lemma nridx_nfuse_nnth_cp_a: 
 assumes " nridx R nell1 "
         " (nnth nell1 0 ) = (0::nat) "
         " nridx R nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "nfinite nell2" 
         "nfinite nell "
         " nlast nell1 = cp "
         " nlast nell2 = the_enat((nlength nell)) - cp "
         " i \<le> ((nlength nell1)) + (nlength nell2)" 
         " the_enat((nlength nell1)) \<le> i " 
         "cp \<le> nlength nell" 
 shows "nnth (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) ( i) = cp + (nnth nell2 (i -the_enat((nlength nell1)))) "
proof -
 have 1: "i = (the_enat ( (nlength nell1)) + (i - the_enat ( (nlength nell1))))"
   by (simp add: assms)
 have 2: "enat ((i::nat) - the_enat ( (nlength nell1))) \<le> nlength nell2" 
   using assms 
    by (metis  enat_add_sub_same enat_minus_mono1 enat_ord_simps(1) 
        idiff_enat_enat infinity_ileE nfinite_conv_nlength_enat the_enat.simps)
 show ?thesis using 1 2 assms nridx_nfuse_nnth_cp[of R nell1 nell2 nell cp  "(i - the_enat ( (nlength nell1)))"]
   by presburger
qed


lemma nidx_nfuse_nnth_cp_a: 
 assumes " nidx  nell1 "
         " (nnth nell1 0 ) = (0::nat) "
         " nidx  nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "nfinite nell2" 
         "nfinite nell "
         " nlast nell1 = cp "
         " nlast nell2 = the_enat((nlength nell)) - cp "
         " i \<le> ((nlength nell1)) + (nlength nell2)" 
         " the_enat((nlength nell1)) \<le> i " 
         "cp \<le> nlength nell" 
 shows "nnth (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) ( i) = cp + (nnth nell2 (i -the_enat((nlength nell1)))) "
using assms nridx_nidx nridx_nfuse_nnth_cp_a by blast

lemma nridx_nfuse_nnth_cp_nlast: 
 assumes " nridx R nell1 "
         " (nnth nell1 0 ) = 0 "
         " nridx R nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "nfinite nell2" 
         "nfinite nell "
         " nlast nell1 = cp "
         " nlast nell2 = the_enat( (nlength nell)) - cp "
         " i \<le> (nlength nell2)" 
         " cp \<le> nlength nell" 
shows " nlast (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) = (the_enat ((nlength nell))) " 
proof -
 have 1: "nlast (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) = nlast (nmap (\<lambda>x. x+cp) nell2)"  
  using assms 
   by (metis add_0  ndropn_0 ndropn_nfirst nlast_nfuse nnth_nmap zero_enat_def zero_le)
 have 2:  "nlast (nmap (\<lambda>x. x+cp) nell2) = cp + (nlast nell2)" 
   by (simp add: assms(6))
 have 3: "cp + (nlast nell2) = (the_enat ((nlength nell)))"
   using assms 
   by (metis add.commute diff_add enat_ord_simps(1) nfinite_conv_nlength_enat the_enat.simps)
 show ?thesis 
 by (simp add: "1" "3" add.commute assms(6))
qed

lemma nidx_nfuse_nnth_cp_nlast: 
 assumes " nidx  nell1 "
         " (nnth nell1 0 ) = 0 "
         " nidx  nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "nfinite nell2" 
         "nfinite nell "
         " nlast nell1 = cp "
         " nlast nell2 = the_enat( (nlength nell)) - cp "
         " i \<le> (nlength nell2)" 
         " cp \<le> nlength nell" 
shows " nlast (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) = (the_enat ((nlength nell))) " 
using assms nridx_nidx nridx_nfuse_nnth_cp_nlast by blast

lemma nridx_nfuse_nnth_cp_infinite: 
 assumes " nridx R nell1 "
         " (nnth nell1 0 ) = (0::nat) "
         " nridx R nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "\<not>nfinite nell2" 
         "\<not>nfinite nell "
         " nlast nell1 = cp "
 shows "nnth (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) (the_enat((nlength nell1)) + i) = cp + (nnth nell2 i) "
proof -
 have 1: "nlast nell1 = nfirst(nmap (\<lambda>x. x+cp) nell2) "
   by (metis assms(1) assms(2) assms(3) assms(4) assms(5) assms(8)  nridx_nfuse_nfirst_nlast)
 have 2: "nnth (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) (the_enat((nlength nell1)) + 0) = nlast nell1 "
   using assms by (metis "1" add.right_neutral  ntaken_nfuse ntaken_nlast)
 have 3: "nfirst(nmap (\<lambda>x. x+cp) nell2) = cp + (nnth nell2 0)" 
   using "1" assms by auto
 have 8: "i\<le> (nlength nell2) "
   by (metis assms(6) enat_ile nfinite_conv_nlength_enat wlog_linorder_le)
 have 10: " (nlength nell1) \<le> enat (the_enat ( (nlength nell1)) + (i::nat))"
   using assms(5) nfinite_nlength_enat by fastforce 
 have 11: "enat (the_enat ( (nlength nell1)) + i) \<le> nlength (nfuse nell1 (nmap (\<lambda>x::nat. x + cp) nell2))" 
   by (metis "10" assms(6) enat_less_enat_plusI2 enat_ord_code(4) enat_the_enat leD nfuse_nlength 
       nlength_eq_enat_nfiniteD nlength_nmap order_less_imp_le)
 have 12: "(the_enat ( (nlength nell1)) + i - the_enat ((nlength nell1))) = i" 
   by auto
 have 4: "nnth (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) (the_enat((nlength nell1)) + i ) = 
          (nnth (nmap (\<lambda>x. x+cp) nell2)  i)"
  by (simp add: "1" "8" assms nfuse_nnth_a) 
 have 5: "(nnth (nmap (\<lambda>x. x+cp) nell2)  i) = cp + (nnth nell2 i)" 
  by (simp add: "8")
 show ?thesis 
 using "4" "5" by presburger
qed

lemma nidx_nfuse_nnth_cp_infinite: 
 assumes " nidx nell1 "
         " (nnth nell1 0 ) = 0 "
         " nidx nell2 "
         "(nnth nell2 0 ) = 0 " 
         "nfinite nell1 "
         "\<not>nfinite nell2" 
         "\<not>nfinite nell " 
         " nlast nell1 = cp "
 shows "nnth (nfuse nell1 (nmap (\<lambda>x. x+cp) nell2)) (the_enat((nlength nell1)) + i) = cp + (nnth nell2 i) "
using assms nridx_nidx nridx_nfuse_nnth_cp_infinite by blast

lemma nidx_nfuse_nidx:
 assumes "nidx  nell1"
         "nnth nell1 0 = 0 " 
         "nidx  nell2"
         "nnth nell2 0 = 0 " 
         "nfinite nell1"   
         "nlast nell1  = cp"
         "nfinite nell2"  
         "nfinite nell" 
         "nlast nell2  = the_enat((nlength nell)) -cp"
         "cp \<le> nlength nell"
 shows   "nidx (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) \<and> (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) 0) = 0"
proof -
 have 1: "nlast nell1 = nfirst(nmap (\<lambda>x. x+ cp) nell2)"
  using assms(1) assms(2) assms(3) assms(4) assms(5) assms(6) nidx_nfuse_nfirst_nlast by blast
 have 2: "nfirst (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) = nfirst nell1" 
   by (simp add: "1" nfirst_nfuse)
 have 4: "\<And>j. j\<le>nlength nell1 \<Longrightarrow> nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j= nnth nell1 j"
    using assms by (simp add: "1" add_increasing2 nfuse_nlength nfuse_nnth)
 have 40: "\<exists> k1. nlength nell1 = (enat k1) " 
   by (simp add: assms(5) nfinite_nlength_enat)
 obtain k1 where 41: "nlength nell1 = (enat k1)" 
   using 40 by blast
 have 5: "\<And>j.  (nlength nell1) \<le>j \<and> j \<le>(nlength nell1) + nlength nell2 \<Longrightarrow>
          nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j = 
          cp + (nnth nell2 (j- the_enat((nlength nell1)))) "   
    using assms nidx_nfuse_nnth_cp_a[of nell1 nell2 nell cp] 
       by (metis "41" enat_ord_simps(1) the_enat.simps)  
 have 45: "\<And>j.  k1 \<le>j \<and> j \<le> (enat (k1)) + nlength nell2 \<Longrightarrow>
          nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j = 
          cp + (nnth nell2 (j- (k1)))"
    by (simp add: "41" "5" order_less_imp_le)
 have 50: "nlength(nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) = (nlength nell1) + nlength nell2"
   by (simp add: nfuse_nlength)
 have 51: "\<And>j. enat (Suc j) \<le> nlength nell1 \<Longrightarrow>
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j) <
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) (Suc j))"
  using "4" Suc_ile_eq assms(1) nidx_expand by auto   
 have 52: "\<And>j.  k1 \<le>j \<and> (Suc j) \<le> enat (k1) + nlength nell2 \<Longrightarrow>
                cp + (nnth nell2 (j- (k1))) <
                cp + (nnth nell2 ((Suc j)- (k1)))"
        using assms(3) unfolding nidx_def  
        by (metis Nat.add_diff_assoc add_strict_left_mono assms(3) enat.simps(3) enat_add_sub_same 
            enat_minus_mono1 idiff_enat_enat nidx_expand plus_1_eq_Suc)       
 have 6: "\<And>j. enat (Suc j) \<le> nlength(nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) \<Longrightarrow>
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j) <
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) (Suc j))"
      proof -
         fix j 
         assume a: "enat (Suc j) \<le> nlength(nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2))"
         show "(nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j) < (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) (Suc j))"  
         proof -
          have 61: "enat (Suc j) \<le> nlength nell1 \<Longrightarrow>
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j) < (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) (Suc j))" 
            using "51" by blast
          have 62: "k1 \<le>j \<and> (Suc j) \<le> nlength(nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) \<Longrightarrow>
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j) < (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) (Suc j))" 
            using "41" "5" "50" "52" Suc_ile_eq by force
          show ?thesis
          using "41" "61" "62" a by fastforce
         qed
     qed
 show ?thesis
 unfolding nidx_expand 
    using "4" "6" assms zero_enat_def by force 
qed

lemma nidx_nfuse_nidx_infinite:
 assumes "nidx  nell1"
         "nnth nell1 0 = 0 " 
         "nidx  nell2"
         "nnth nell2 0 = 0 " 
         "nfinite nell1"   
         "nlast nell1  = cp"
         "\<not>nfinite nell2" 
         "\<not>nfinite nell" 
 shows   "nidx (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) \<and> (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) 0) = 0"
proof -
 have 1: "nlast nell1 = nfirst(nmap (\<lambda>x. x+ cp) nell2)"
   by (metis add_0 assms(4) assms(6) assms(7) enat_le_plus_same(1) enat_le_plus_same(2) enat_the_enat 
       infinity_ileE ndropn_0 ndropn_nfirst nfinite_conv_nlength_enat nnth_nmap)
 have 2: "nfirst (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) = nfirst nell1" 
   by (simp add: "1" nfirst_nfuse)
 have 4: "\<And>j. j\<le>nlength nell1 \<Longrightarrow> nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j= nnth nell1 j"
   by (simp add: "1" add_increasing2 nfuse_nlength nfuse_nnth)
 have 40: "\<exists> k1. nlength nell1 = (enat k1) " 
   by (simp add: assms(5) nfinite_nlength_enat)
 obtain k1 where 41: "nlength nell1 = (enat k1)" 
   using 40 by blast
 have 5: "\<And>j.  (nlength nell1) \<le>j \<and> j \<le> (nlength nell1) + nlength nell2 \<Longrightarrow>
          nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j = 
          cp + (nnth nell2 (j- the_enat((nlength nell1)))) "   
    using assms nidx_nfuse_nnth_cp_infinite[of nell1 nell2 nell cp  ]
    by (metis "41" enat_ord_simps(1) le_add_diff_inverse the_enat.simps)
 have 45: "\<And>j.  k1 \<le>j \<and> j \<le> (enat (k1)) + nlength nell2 \<Longrightarrow>
          nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j = 
          cp + (nnth nell2 (j- (k1)))"
       using "41" "5" enat_ord_simps(1) the_enat.simps by presburger
 have 50: "nlength(nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) = (nlength nell1) + nlength nell2"
     by (simp add: nfuse_nlength)
 have 51: "\<And>j. enat (Suc j) \<le> nlength nell1 \<Longrightarrow>
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j) <
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) (Suc j))"
    using "4" Suc_ile_eq assms(1) nidx_expand by auto
 have 52: "\<And>j.  k1 \<le>j \<and> (Suc j) \<le> enat (k1) + nlength nell2 \<Longrightarrow>
                cp + (nnth nell2 (j- (k1))) <
                cp + (nnth nell2 ((Suc j)- (k1)))"
       by (metis Nat.add_diff_assoc add_strict_left_mono assms(3) enat.simps(3) enat_add_sub_same 
           enat_minus_mono1 idiff_enat_enat nidx_expand plus_1_eq_Suc)
 have 53: "\<And>j.  k1 \<le>j \<and> (Suc j) \<le> enat (k1) + nlength nell2 \<Longrightarrow>
                 nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j <
                 nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) (Suc j)" 
       by (metis "45" "52" Suc_ile_eq nle_le not_less_eq_eq order_less_imp_le)
 have 6: "\<And>j. enat (Suc j) \<le> nlength(nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) \<Longrightarrow>
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) j) <
               (nnth (nfuse nell1 (nmap (\<lambda>x. x+ cp) nell2)) (Suc j))"
  by (metis "41" "50" "51" "53" enat_ord_simps(1) le_SucE linorder_le_cases)   
 show ?thesis unfolding nidx_expand 
   using "4" "6" assms zero_enat_def by fastforce
qed
   

lemma nsubn_nfuse_nidx:
 assumes "nidx nl" 
         "nfinite nl"
         "nfinite nell " 
         "nlast nl = (nlength nell) "
         " (Suc i) \<le> (nlength nl) "
 shows   " nfuse (nsubn nell (nnth nl i) (nnth nl (Suc i))) (nsubn nell (nnth nl (Suc i)) (nlast nl)) =
           (nsubn nell (nnth nl i) (nlast nl)) " 
proof -
 have 1: "(nnth nl i) \<le> (nnth nl (Suc i))"
    by (simp add: assms(1) assms(5) nidx_less_eq)   
 have 2: " nlast nl = (nnth nl (the_enat( (nlength nl)))) "
      by (simp add: assms(2) nnth_nlast) 
 have 3: "1 \<le> nlength nl "
    by (metis assms(5) dual_order.trans enat_0_iff(1) iless_Suc_eq le_zero_eq linorder_le_cases 
        nat.simps(3) one_eSuc order_neq_le_trans) 
 have 4: "(nnth nl (Suc i)) \<le> (nlast nl)"
    by (metis "2" assms(1) assms(2) assms(5) dual_order.eq_iff enat_ord_simps(1) 
        nfinite_conv_nlength_enat nidx_less_eq the_enat.simps)
 have 5: "enat (nnth nl (Suc i)) \<le> nlength nell"
   by (metis "4" assms(4) enat_ord_simps(1))
 have 6: "nlast (nsubn nell (nnth nl i) (nnth nl (Suc i))) = (nnth nell (nnth nl (Suc i)))"
     by (simp add: "1" nsubn_def1 ntaken_nlast)
 have 7: "nfirst (nsubn nell (nnth nl (Suc i)) (nlast nl)) = (nnth nell (nnth nl (Suc i))) "
 by (simp add: nsubn_def1 ntaken_ndropn_nfirst)
 show ?thesis 
 by (simp add: "1" "5" assms(4) nsubn_nfuse)
qed

lemma nidx_nfuse_split:
 assumes "nlast nell1 = nfirst nell2"
 shows   "nridx R (nfuse nell1 nell2) \<longleftrightarrow> 
          (if nfinite nell1 then nridx R nell1 \<and> nridx R nell2 else nridx R nell1)"
proof (cases "nfinite nell1")
case True
then show ?thesis 
by (metis assms nappend_nbutlast_nlast_id nbutlast_nfinite nellist.collapse(2) nellist.disc(1) 
    nfuse_def1 nfuse_leftneutral nhd_nfuse nlast_NNil nridx_LCons_1 nridx_nappend_nfinite)
next
case False
then show ?thesis by (simp add: assms nfuse_nappend)
qed


lemma nidx_all_le_nlast: 
assumes "nidx nell"
        "nfinite nell"
        "j\<le> nlength nell"
shows   "nnth nell j \<le> nlast nell" 
using assms 
by (metis nfinite_conv_nlength_enat nidx_less_last_1 nnth_nlast order.order_iff_strict the_enat.simps)

lemma nidx_shiftm :
  assumes "nidx  nell "
          " nnth nell 0 = k"
 shows    "nidx  (nmap (\<lambda>x .x- k) nell) \<and>  nnth (nmap (\<lambda>x .x- k) nell) 0 = 0 \<and>   k \<le> (nnth nell 0)"
using assms zero_enat_def
by (auto simp add:  nidx_expand   )
   (metis Suc_ile_eq add_diff_inverse_nat add_gr_0 assms(1) diff_less_mono nidx_gr_first
     nnth_nmap order_less_imp_le zero_less_Suc zero_less_diff)

lemma nidx_nsubn:
 assumes "k\<le>n"
         "n \<le> nlength nell"
         "nidx  nell"
         " nnth nell 0 = 0 "
 shows   "nidx  (nsubn nell k n ) \<and> nnth (nsubn nell k n ) 0 = (nnth nell k)"
using assms unfolding nidx_expand  nsubn_def1 
by (auto simp add: ntaken_nnth) 
   (simp add: Nat.le_diff_conv2 add.commute order_subst2)


lemma nidx_ntaken_niterates_Suc: 
 " nidx (ntaken n (niterates Suc 0))" 
proof -
 have 1: " nidx (ntaken n (niterates Suc 0)) = 
       (\<forall>i::nat.
        enat (Suc i) \<le> nlength (ntaken n (niterates Suc (0::nat))) \<longrightarrow>
        nnth (ntaken n (niterates Suc (0::nat))) i < 
        nnth (ntaken n (niterates Suc (0::nat))) (Suc i)) "
    unfolding nidx_expand by auto 
 have 2: "(\<forall>i::nat.
        enat (Suc i) \<le> nlength (ntaken n (niterates Suc (0::nat))) \<longrightarrow>
        nnth (ntaken n (niterates Suc (0::nat))) i < 
        nnth (ntaken n (niterates Suc (0::nat))) (Suc i))"
    proof 
     fix i
     show "enat (Suc i) \<le> nlength (ntaken n (niterates Suc (0::nat))) \<longrightarrow>
       nnth (ntaken n (niterates Suc (0::nat))) i < 
       nnth (ntaken n (niterates Suc (0::nat))) (Suc i)" 
     proof (induct i arbitrary: n)
     case 0
     then show ?case 
         by (simp add: ntaken_nnth)  
     next
     case (Suc i)
     then show ?case by (simp add: ntaken_nnth)
     qed 
   qed
  show ?thesis 
  using "1" "2" by fastforce
qed

lemma forall_nappend_single:
 assumes "nfinite l"  
 shows " (\<forall> i< nlength (nappend l (NNil x)). P i (Suc i)) \<longleftrightarrow>
         (P (the_enat(nlength l)) (Suc(the_enat(nlength l))) \<and> (\<forall> i< nlength l. P i (Suc i))) "
proof (cases "nlength l = 0")
case True
then show ?thesis using assms
by (metis enat_0_iff(1) gr_implies_not_zero iless_eSuc0 nappend_NNil
     ndropn_0 ndropn_nlast nlength_NCons nlength_NNil the_enat_0) 
next
case False
then show ?thesis using assms 
by (simp add: enat_the_enat nfinite_conv_nlength_enat)
   (metis dual_order.order_iff_strict iless_Suc_eq plus_1_eSuc(2) the_enat.simps)
qed


lemma forall_ncons_split: 
 assumes "nfinite l"
 shows "(\<forall>i. enat i < nlength (NCons 0 l) \<longrightarrow> 
           P (nnth (NCons 0 l) i) (nnth (NCons 0 l) (Suc i))) \<longleftrightarrow>
          P 0 (nnth l 0) \<and>
        (\<forall>i. enat i < nlength l \<longrightarrow> 
             P (nnth  l i) (nnth l (Suc i)))"
using assms 
by (auto simp  add: zero_enat_def )
   (metis nnth_0 zero_enat_def zero_le,  
    metis eSuc_enat ileI1 nnth_Suc_NCons, 
    metis Suc_ile_eq Suc_pred less_le_not_le nle_le nnth_0 nnth_Suc_NCons zero_le)


lemma forall_ncons_split_alt: 
 assumes "\<not>nfinite l"
 shows "(\<forall>i. enat i < nlength (NCons 0 l) \<longrightarrow> 
           P (nnth (NCons 0 l) i) (nnth (NCons 0 l) (Suc i))) \<longleftrightarrow>
          P 0 (nnth l 0) \<and>
        (\<forall>i. enat i < nlength l \<longrightarrow> 
             P (nnth  l i) (nnth l (Suc i)))"
using assms 
by auto 
   (metis nnth_0 zero_enat_def zero_le, 
    metis eSuc_enat ileI1 nnth_Suc_NCons, 
    metis Suc_ile_eq Suc_pred le_zero_eq less_le_not_le nle_le nnth_0 nnth_Suc_NCons)

lemma nidx_Ncons_shift: 
 assumes "nfinite l" 
         "nidx l"
 shows   "nidx (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l))" 
proof -
 have 1: " nidx (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) \<longleftrightarrow>
           0 < nfirst (nmap (\<lambda>x. x+ Suc ia) l) \<and> 
           nidx (nmap (\<lambda>x. x+ Suc ia) l)" 
     by auto
 have 2: " 0 < nfirst (nmap (\<lambda>x. x+ Suc ia) l)" 
     by (metis Zero_not_Suc add_Suc_right bot_nat_0.not_eq_extremum enat_defs(1) ndropn_0 
         ndropn_nfirst nnth_nmap zero_le)    
 have 3: "nidx (nmap (\<lambda>x. x+ Suc ia) l)"
    using nidx_expand Suc_ile_eq assms(2) by force 
 show ?thesis 
 using "2" "3" by auto
qed

lemma nidx_Ncons_shift_alt: 
 assumes "\<not>nfinite l" 
         "nidx l"
 shows   "nidx (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l))" 
proof -
 have 1: " nidx (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) \<longleftrightarrow>
           0 < nfirst (nmap (\<lambda>x. x+ Suc ia) l) \<and> 
           nidx (nmap (\<lambda>x. x+ Suc ia) l)" 
     by auto
 have 2: " 0 < nfirst (nmap (\<lambda>x. x+ Suc ia) l)" 
     by (metis Zero_not_Suc add_Suc_right bot_nat_0.not_eq_extremum enat_defs(1) ndropn_0
          ndropn_nfirst nnth_nmap zero_le)    
 have 3: "nidx (nmap (\<lambda>x. x+ Suc ia) l)"
    using nidx_expand Suc_ile_eq assms(2) by force 
 show ?thesis 
 using "2" "3" by auto
qed

 lemma infinite_nidx_imp_infinite_interval: 
assumes "\<not> nfinite l "
        "nidx l"
        "(nnth l 0) = 0"
        "\<forall>i. (nnth l i) \<le> nlength s"
shows  "\<not> nfinite s" 
proof 
 assume "nfinite s"
 thus False
   using assms
  proof (induct s  rule: nfinite_induct)
  case (NNil y)
  then show ?case
  by (metis dual_order.antisym enat_ord_simps(1) linorder_linear nfinite_ntaken nidx_gr_first nlength_NNil 
      not_gr_zero ntaken_all zero_le zero_less_Suc) 
  next
  case (NCons x nell)
  then show ?case 
    proof -
     have 1: "\<And>j. (nnth l j) < (nnth l (Suc j)) "
       by (metis assms(1) assms(2) linorder_le_cases nfinite_ntaken nidx_expand ntaken_all)
     have 2: "\<forall>i. enat (nnth l i) \<le> nlength (NCons x nell) \<Longrightarrow> False" 
      by (metis "1" NCons.hyps(2) assms(1) assms(2) assms(3) dual_order.strict_iff_order 
          enat_ord_simps(1) iless_Suc_eq linorder_not_le nlength_NCons)
     show ?thesis 
     using "2" NCons.prems(4) by auto
  qed 
 qed
qed


subsection \<open>nlastnfirst\<close>


lemma nlastnfirst_def2: 
"nlastnfirst =  llastlfirst \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist) " 
by (simp add: map_fun_def nlastnfirst_def)

lemma nlastnfirst_NNil[simp]:
 " nlastnfirst (NNil x)"
apply transfer
by simp

lemma nlastnfirst_LCons[simp]:      
 shows "nlastnfirst (NCons nell nells) \<longleftrightarrow> 
        nlast nell = nfirst (nfirst nells) \<and> nlastnfirst nells" 
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nlastnfirst (NCons nell nells) = 
          (llastlfirst \<circ> ?n2l) (NCons nell nells) "
   by (simp add: nlastnfirst_def2) 
 have 2: "(llastlfirst \<circ> ?n2l) (NCons nell nells) = 
          (llast (llist_of_nellist nell) = lfirst (lfirst (?n2l nells)) \<and> llastlfirst (?n2l nells))"
    by simp
 have 3: "llastlfirst (?n2l nells) = nlastnfirst nells"
    by (simp add: nlastnfirst.rep_eq)  
 have 4: "llast (llist_of_nellist nell) = nlast nell"
   by (metis llast_linfinite nfinite_def nlast_llast nlast_not_nfinite) 
 have 5: "lfirst (lfirst (?n2l nells)) = nfirst (nfirst nells)"
    by (simp add: lfirst_def llist_of_nellist.simps(2) nfirst_def) 
 show ?thesis 
 using "1" "2" "3" "4" "5" by presburger
qed


lemma nlastnfirst_def1: 
shows " nlastnfirst nells = 
   (\<forall>i. (Suc i)\<le> nlength nells \<longrightarrow> nlast(nnth nells i) = nfirst(nnth nells (Suc i))) " 
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nlastnfirst nells = (llastlfirst \<circ> ?n2l) nells"
    by (simp add: nlastnfirst_def2) 
 have 2: "(llastlfirst \<circ> ?n2l) nells = llastlfirst (?n2l nells)"
      by simp
 have 3: "llastlfirst (?n2l nells) \<longleftrightarrow>
          (\<forall>i. (Suc i) < llength (?n2l nells) \<longrightarrow>
             llast (lnth (?n2l nells) i) = lfirst  (lnth (?n2l nells) (Suc i)))"
      using llastlfirst_def by blast  
 have 4: "epred (llength (?n2l nells)) = nlength nells" 
         by simp
 have 5: "\<And>xs j. j\<le> nlength xs \<longrightarrow> nnth xs j = lnth (llist_of_nellist xs) j"
   unfolding nnth_def  by auto 
 have 6: "\<And>lx j . j<llength lx \<and> \<not> lnull lx\<longrightarrow> lnth lx j = nnth (nellist_of_llist lx) j " 
    by (metis "5" co.enat.exhaust_sel iless_Suc_eq llength_eq_0 nellist_of_llist_inverse nlength.abs_eq)
 have 7: "\<And>xs. nlast xs = llast (llist_of_nellist xs) " 
    by (metis llast_linfinite nfinite_def nlast_llast nlast_not_nfinite)
 have 8: "\<And>xs. nfirst xs = lfirst (llist_of_nellist xs) " 
    by (simp add: lfirst_def llist_of_nellist.simps(2) nfirst_def)
 have 9: "\<And>lx. \<not> lnull lx \<Longrightarrow> llast lx = nlast (nellist_of_llist lx)" 
   by (simp add: "7")
 have 10: "\<And>lx. \<not> lnull lx \<Longrightarrow> lfirst lx = nfirst (nellist_of_llist lx)" 
   by (simp add: "8")
 have 11: "\<And> j. j < llength (?n2l nells) \<longrightarrow>
             llast (lnth (?n2l nells) j) =
             nlast (nellist_of_llist  (lnth (?n2l nells) j)) " 
     by (simp add: "9")
 have 12: "\<And> j. (enat j) < llength (?n2l nells) \<longrightarrow>
              nlast (nellist_of_llist  (lnth (?n2l nells) j)) =
              nlast (nellist_of_llist  (llist_of_nellist (lnth (llist_of_nellist nells) j)))" 
    by simp
 have 13: "\<And> j. (enat j) < llength (?n2l nells) \<longrightarrow>
             nlast (nellist_of_llist  (llist_of_nellist (lnth (llist_of_nellist nells) j))) =
             nlast (lnth (llist_of_nellist nells) j)"
     by auto 
 have 14: "\<And> j. (enat j) < llength (?n2l nells) \<longrightarrow>
            nlast (lnth (llist_of_nellist nells) j) = nlast(nnth nells j) "
     by (simp add: "6") 
 have 15: "\<And> j. j < llength (?n2l nells) \<longrightarrow>
            llast (lnth (?n2l nells) j) = 
            nlast(nnth nells j)"
    using "11" "12" "13" "14" by presburger 
 have 16: "\<And> j. (Suc j) < llength (?n2l nells) \<longrightarrow>
             lfirst  (lnth (?n2l nells) (Suc j)) =
             nfirst (nellist_of_llist (lnth (?n2l nells) (Suc j)))   " 
    by (simp add: "10")
 have 17: "\<And> j. (Suc j) < llength (?n2l nells) \<longrightarrow>
            nfirst (nellist_of_llist (lnth (?n2l nells) (Suc j))) =
            nfirst (nellist_of_llist (llist_of_nellist (lnth  (llist_of_nellist nells) (Suc j)))) " 
     by simp
 have 18: "\<And> j. (Suc j) < llength (?n2l nells) \<longrightarrow> 
            nfirst (nellist_of_llist (llist_of_nellist (lnth  (llist_of_nellist nells) (Suc j)))) =
            nfirst (lnth  (llist_of_nellist nells) (Suc j))"
     by auto 
 have 19: "\<And> j. (Suc j) < llength (?n2l nells) \<longrightarrow> 
            nfirst (lnth  (llist_of_nellist nells) (Suc j)) = nfirst (nnth nells (Suc j))"
     by (simp add: "6") 
 have 20: "\<And> j. (Suc j) < llength (?n2l nells) \<longrightarrow>
           lfirst  (lnth (?n2l nells) (Suc j)) = nfirst (nnth nells (Suc j)) " 
     using "16" "17" "18" "19" by presburger
 have 21: "\<And> j. (Suc j) < llength (?n2l nells) \<longleftrightarrow>
                 (Suc j) \<le> nlength nells" 
    by (metis "4" co.enat.exhaust_sel dual_order.strict_iff_order enat_0_iff(1) epred_0 iless_Suc_eq 
        nat.simps(3))
 have 22: "(\<forall>i. (Suc i) < llength (?n2l nells) \<longrightarrow>
             llast (lnth (?n2l nells) i) = lfirst  (lnth (?n2l nells) (Suc i))) 
          \<longleftrightarrow>
           (\<forall>i. (Suc i)\<le> nlength nells \<longrightarrow> nlast(nnth nells i) = nfirst(nnth nells (Suc i))) " 
    by (metis "15" "20" "21" Suc_ile_eq order_less_imp_le)
 have 23: "llastlfirst (?n2l nells) \<longleftrightarrow> 
           (\<forall>i. (Suc i)\<le> nlength nells \<longrightarrow> nlast(nnth nells i) = nfirst(nnth nells (Suc i)))" 
    using "22" "3" by presburger
 show ?thesis using "1" "23" by auto
qed
 

lemma nlastnfirst_nridx: 
 "nlastnfirst nells = nridx (\<lambda> a b. nlast a = nfirst b) nells "
by (simp add: nlastnfirst_def1 nridx_expand)

lemma nlastnfirst_nappend_nfinite:
 assumes "nfinite nellxs" 
 shows   "nlastnfirst (nappend nellxs nellys) \<longleftrightarrow>
          nlastnfirst nellxs \<and> nlastnfirst nellys \<and> nlast(nlast nellxs) = nfirst(nfirst nellys) " 
using assms
by (metis (mono_tags, lifting) nlastnfirst_nridx nridx_nappend_nfinite)



subsection \<open>nfusecat\<close>

lemma nfusecat_def2: 
"nfusecat = nellist_of_llist \<circ> lfusecat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist) " 
by (simp add: map_fun_def nfusecat_def)

lemma not_null_lfusecat: 
 "\<not>lnull((lfusecat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells)" 
by (simp add: llist_of_nellist.code)

lemma nfusecat_def3:
" ((llist_of_nellist \<circ> nfusecat) nells) =
  ((lfusecat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells) " 
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "llist_of_nellist \<circ> nfusecat = 
          llist_of_nellist \<circ> nellist_of_llist \<circ> lfusecat \<circ> ?n2l" 
   by (simp add: nfusecat_def2 rewriteL_comp_comp)
 have 2: " \<not>lnull((lfusecat \<circ> ?n2l) nells) " 
  using not_null_lfusecat by auto
  have 3: "(llist_of_nellist \<circ> nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells =
          (lfusecat \<circ> ?n2l) nells" 
   using 2 nellist_of_llist_inverse by simp 
 show ?thesis 
 by (metis "1" "3")
qed   


lemma nfusecat_NNil[simp]: 
 " nfusecat (NNil nell) = nell " 
apply transfer
by simp

lemma nfusecat_NCons[simp]:
 "nfusecat (NCons nell nells) = nfuse nell (nfusecat nells) " 
apply transfer
by (simp add: lfuse_def llist.case_eq_if) 


lemma nfusecat_nfirst: 
assumes "(\<exists>x1. nells = NNil x1)"
 shows "  nfusecat nells = nfirst nells"
proof -
 obtain nell where 1: " nells = NNil nell" 
  using assms by auto
 have 2: " nfusecat nells = nell " 
   by (simp add: "1")
 have 3: " nfirst nells = nell " 
   by (metis "1" ndropn_0 ndropn_nfirst nnth_NNil)
 show ?thesis 
 by (simp add: "2" "3")
qed

lemma nfusecat_NCons_nfirst: 
assumes "(\<forall>nell. nells \<noteq> NNil nell) "
shows "   nfusecat nells = nfuse (nfirst nells) (nfusecat (ntl nells)) " 
by (metis assms nellist_split_2_first  nlast_NNil nfusecat_NCons not_le_imp_less ntaken_0 
    ntaken_all ntaken_nlast zero_enat_def)

lemma nfusecat_expand: 
 "nfusecat nells = 
  (if is_NNil nells then nfirst nells else nfuse (nfirst nells) (nfusecat (ntl nells)))   " 
unfolding is_NNil_def
using nfusecat_NCons_nfirst nfusecat_nfirst by simp blast

lemma nfusecat_expand_case: 
 " nfusecat nells = (case nells of (NNil nell) \<Rightarrow> nell |
                      (NCons nell' nells1) \<Rightarrow> nfuse nell' (nfusecat nells1)) " 
by (metis is_NNil_def nellist.case_eq_if nellist.collapse(2) nlast_NNil nfusecat_NCons nfusecat_NNil)


lemma nmap_nfusecat: 
 " nmap f (nfusecat nells) = (nfusecat (nmap ( nmap f ) nells))"
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nmap f (nfusecat nells) = 
         nmap f ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells) " 
   by (simp add: nfusecat_def2)
 have 2: " nmap f ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells) =
           nellist_of_llist (lmap f ((lfusecat \<circ>?n2l) nells))"
  using nmap_nellist_of_llist not_null_lfusecat by auto
 have 3: "(lmap f ((lfusecat \<circ> ?n2l) nells)) =
          lfusecat (lmap (lmap f) ((lmap llist_of_nellist \<circ> llist_of_nellist) nells))"
    by (simp add: lmap_lfusecat) 
 have 4: "(nfusecat (nmap ( nmap f ) nells)) =
          (nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (nmap ( nmap f ) nells)"
    by (simp add: nfusecat_def2) 
 have 8: "(lmap (lmap f) (lmap llist_of_nellist (llist_of_nellist nells))) = 
          (lmap llist_of_nellist (lmap (nmap f) (llist_of_nellist nells)))" 
  using lmap_llist_of_nellist_nmap by blast
 show ?thesis 
 using "1" "2" "3" "4" "8" by fastforce
qed

lemma nfirst_nfuse_var: 
 "nfirst (nfuse nellx nelly) = nfirst nellx" 
by (metis nfuse_def1 nlast_NNil ntaken_0 ntaken_nappend1 zero_enat_def zero_le)


lemma nfirst_nfusecat_nfirst:
  shows " nfirst(nfusecat nells) = nfirst(nfirst nells) " 
proof (cases nells)
case (NNil nell)
then show ?thesis
by (simp add: nfusecat_expand)
next
case (NCons nell nells1)
then show ?thesis 
   proof -
    have 1: "nfusecat (NCons nell nells1) = nfuse nell (nfusecat nells1)"
      by (simp) 
    have 2: "nfirst (nfuse nell (nfusecat nells1)) = nfirst nell"
     using nfirst_nfuse_var by auto
    have 3: "nfirst(nfirst (NCons nell nells1)) = nfirst nell" 
      by (metis "1" nfirst_nfuse_var nfusecat_expand)
    show ?thesis 
    using "1" "2" "3" NCons by fastforce
  qed
qed

lemma nfirst_nfusecat: 
  shows " nfirst(nfusecat (NCons nell nells)) = nfirst nell "
by (simp add: nfirst_nfuse_var )


lemma nfusecat_nlength_eq_zero_conv: 
 " nlength (nfusecat nells) = 0 \<longleftrightarrow> (\<forall> nell \<in> nset nells. nlength nell = 0)  " 
proof -
let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nlength (nfusecat nells) = 
          nlength ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells) "
  by (simp add: nfusecat_def2) 
 have 2: "nlength ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells) =
          epred (llength (lfusecat (?n2l nells)))"
     by simp 
        (metis lbutlast_snoc llength_lbutlast nellist_of_llist_a_inverse nlength.rep_eq) 
 have 3: "nlength (nfusecat nells) = 0 \<longleftrightarrow> 
           epred (llength (lfusecat (?n2l nells))) = 0"
    using "1" "2" by presburger
 have 4: "llength (?n2l nells) > 0 "
      by simp 
 have 5: "(llength (lfusecat (?n2l nells))) > 0 "
       by (simp add: lfusecat_not_lnull_var)
 have 6: "epred (llength (lfusecat (?n2l nells))) = 0 \<longleftrightarrow>
           (llength (lfusecat (lmap llist_of_nellist (llist_of_nellist nells)))) = 1"
    by (metis "5" comp_apply epred_1 epred_inject not_iless0 zero_one_enat_neq(1))
 have 7: "(llength (lfusecat (?n2l nells))) \<le> 1 \<longleftrightarrow>
          (\<forall> nell\<in> lset (?n2l nells). llength nell \<le> 1) "
     using lfusecat_all_empty_or_LNil lset_lltl_llength_var by blast
 have 8: "(\<forall> nell\<in> lset (?n2l nells). llength nell > 0)"
   by simp 
 have 9: "(llength (lfusecat (?n2l nells))) = 1 \<longleftrightarrow>
          (\<forall> nell\<in> lset (?n2l nells). llength nell = 1)"
    by (metis "5" "7" "8" ileI1 one_eSuc order_antisym_conv) 
 have 10: "(\<forall> nell\<in> lset (?n2l nells). llength nell = 1) \<longleftrightarrow>
           (\<forall> nell \<in> nset nells. nlength nell = 0)"
   by simp 
    (metis co.enat.exhaust_sel epred_1  llength_eq_0 llist_of_nellist_not_lnull nlength.rep_eq 
     zero_neq_one) 
 show ?thesis using 3 6 9 10 by simp
qed


lemma is_NNil_nfusecat_a: 
 assumes "\<forall>i. i\<le>nlength nells \<longrightarrow> is_NNil (nnth nells i)"
 shows " is_NNil(nfusecat nells)"
using assms nfusecat_nlength_eq_zero_conv[of nells ]
by (metis in_nset_conv_nnth is_NNil_def nle_le nlength_NNil ntaken_0 ntaken_all zero_enat_def)


lemma is_NNil_nfusecat_b: 
 assumes "is_NNil(nfusecat nells) "
 shows "\<forall>i. i\<le>nlength nells \<longrightarrow> is_NNil (nnth nells i) " 
using assms nfusecat_nlength_eq_zero_conv[of nells]
by (metis in_nset_conv_nnth nellist.collapse(1) nellist.collapse(2) nlength_NCons nlength_NNil 
    zero_ne_eSuc)

lemma ntl_lfusecat :
 shows   "ntl(nfusecat nells) = 
          (if is_NNil nells then ntl (nfirst nells) else
           (if is_NNil (nfirst nells) then 
              if is_NNil (nfusecat (ntl nells)) 
              then ntl (nfirst nells) 
              else ntl (nfusecat (ntl nells))
           else 
              if is_NNil (nfusecat (ntl nells)) 
              then ntl (nfirst nells) 
              else nappend (ntl (nfirst nells)) (ntl (nfusecat (ntl nells)))))
             "
proof (cases nells)
case (NNil nell)
then show ?thesis by (metis nellist.disc(1) nfusecat_NNil nfusecat_expand)
next
case (NCons nell nells1)
then show ?thesis
  using nfusecat_NCons[of nell nells1] ntl_nfuse[of nell "(nfusecat nells1)"] by simp
    (metis ndropn_0 ndropn_nfirst nnth_0)
qed

lemma  nfusecat_nappend: 
assumes "nfinite llx"     
shows "nfusecat (nappend llx lly) = nfuse (nfusecat llx) (nfusecat lly) "
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nfusecat (nappend llx lly) = 
          ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (nappend llx lly))"
   by (simp add: nfusecat_def2)  
 have 2: "((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (nappend llx lly)) = 
          nellist_of_llist (lfusecat (?n2l (nappend llx lly)))"
    by simp 
 have 3: "(llist_of_nellist (nappend llx lly)) = 
          lappend (llist_of_nellist llx) (llist_of_nellist lly)"
    by (simp add: llist_of_nellist_nappend) 
 have 4: " (lmap llist_of_nellist (lappend (llist_of_nellist llx) (llist_of_nellist lly))) =
           lappend (?n2l llx) (?n2l lly)"
    using lmap_lappend_distrib by auto 
 have 5: "(lfusecat (lappend (?n2l llx) (?n2l lly))) =
          lfuse (lfusecat (?n2l llx)) (lfusecat (?n2l lly))"
     using assms lfinite_lmap lfusecat_lappend nfinite_def by (metis comp_def) 
 have 6: "nellist_of_llist (lfuse (lfusecat (?n2l llx)) (lfusecat (?n2l lly))) =
          nfuse (nfusecat llx) (nfusecat lly)"
    by (simp add: llist_of_nellist.code nfuse.abs_eq nfusecat_def2) 
 show ?thesis 
 using "1" "2" "3" "4" "5" "6" by simp
qed

lemma nfusecat_split: 
 assumes " (Suc n) \<le> nlength nells" 
shows "nfusecat nells = nfuse (nfusecat (ntaken n nells)) (nfusecat (ndropn (Suc n) nells)) " 
using assms 
by (metis nappend_ntaken_ndropn nfinite_ntaken nfusecat_nappend)

lemma nfusecat_split_1: 
 assumes " (Suc n) \<le> nlength nells" 
shows "nfusecat nells = nfuse (nfusecat (ntake n nells)) (nfusecat (ndropn (Suc n) nells)) " 
using assms
by (simp add: nfusecat_split ntake_eq_ntaken)


lemma nfuse_nfinite:  
 shows   "nfinite (nfuse nellx nelly) \<longleftrightarrow> nfinite nellx \<and> nfinite nelly" 
apply transfer
using lfuse_lfinite by blast

lemma nfusecat_nfinite:
assumes "\<forall>nell \<in> nset nells. nlength nell > 0 "
         "\<forall>nell \<in> nset nells. nfinite nell" 
         "nlastnfirst nells" 
shows "nfinite (nfusecat nells) \<longleftrightarrow> nfinite nells " 
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nfinite (nfusecat nells) \<longleftrightarrow> 
          nfinite (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells))"
   by (simp add: nfusecat_def2) 
 have 2: "nfinite (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells)) \<longleftrightarrow>
          lfinite (llist_of_nellist (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells)))"
    using nfinite_def by blast 
 have 3: "lfinite (llist_of_nellist (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells))) \<longleftrightarrow>
          lfinite (lfusecat (?n2l nells))"
    by (simp add: lbutlast_lfinite) 
 have 4: "\<forall>nell \<in> lset (llist_of_nellist nells). nlength nell > 0"
   using assms(1) by force
 have 5: "\<forall>nell \<in> lset (llist_of_nellist nells). epred(llength (llist_of_nellist nell)) > 0"
    using "4" by fastforce 
 have 6: "\<forall>nell \<in> lset (llist_of_nellist nells). \<not> lnull (ltl (llist_of_nellist nell))"
   by (metis "5" epred_llength less_numeral_extra(3) llength_LNil llist.collapse(1))     
 have 7: "\<forall>nell \<in> lset (?n2l nells). \<not> lnull (ltl nell)"
   by (simp add: "6")   
 have 8: "\<forall>nell \<in> lset (?n2l nells). lfinite nell"
    using assms(2) nfinite_def by fastforce
 have 9: "llastlfirst (?n2l nells)" 
   using assms(3) nlastnfirst.rep_eq by auto
 have 10: "lfinite (lfusecat (?n2l nells)) \<longleftrightarrow> lfinite (?n2l nells)"
     using "7" "8" "9" lfusecat_lfinite by blast 
 have 11: "lfinite (?n2l nells) \<longleftrightarrow> nfinite nells" 
    by (simp add: nfinite_def)
 show ?thesis using 1 2 3 10 11 by presburger
qed

lemma nlastfirst_nfusecat_nlast:
 assumes "nfinite nells"  
         "nlastnfirst nells"
         "\<forall>nell \<in> nset nells. nfinite nell " 
 shows   " nlast(nfusecat nells) = nlast(nlast nells)"
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nlast(nfusecat nells) = 
          nlast (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells))"
    by (simp add: nfusecat_def2) 
 have 2: "nlast (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells)) =
          llast (llist_of_nellist (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells)))"
    by (metis (no_types, lifting) llast_linfinite nfinite_def nlast_llast nlast_not_nfinite) 
 have 3: "llast (llist_of_nellist (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells))) =
          llast (lfusecat (?n2l nells))"
    by (metis (no_types, lifting) "2" comp_def lbutlast.disc_iff(1) nellist_of_llist_a_inverse 
        nlast_nellist_of_llist_a_lnull not_lnull_eq_lappend_lbutlast_llast) 
 have 4: "lfinite (?n2l nells)"
    using assms(1) lfinite_lmap nfinite_def by auto 
 have 5: "\<not>lnull (?n2l nells)" 
    by simp
 have 6: "\<forall>nell \<in> lset (?n2l nells). \<not>lnull nell"
    by simp
 have 7: "llastlfirst (?n2l nells)"
    using assms(2) nlastnfirst.rep_eq by auto
 have 8: "\<forall>nell \<in> lset (?n2l nells). lfinite nell "
   using assms(3) nfinite_def by auto 
 have 9: "llast (lfusecat (?n2l nells)) = llast(llast (?n2l nells))"
   using "4" "5" "6" "7" "8" llastfirst_lfusecat_llast by blast
 have 10: "llast(llast (?n2l nells)) = nlast (nlast nells)"
   by (metis assms(1) assms(3) comp_apply llast_lmap llist_of_nellist_not_lnull nfinite_def 
       nlast_llast nset_nlast)
 show ?thesis using 1 2 3 9 10 by presburger
qed


lemma nfusecat_nlength_nfinite:
 assumes "nfinite nells"
         "\<forall>nell \<in> nset nells. nfinite nell" 
         "nlastnfirst nells" 
 shows  " nlength(nfusecat nells) = 
          (\<Sum> i = 0 .. (the_enat((nlength nells))) . (nlength (nnth nells i)))" 
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: " nlength(nfusecat nells) = nlength (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells))"
   by (simp add: nfusecat_def2) 
 have 2: "nlength (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells)) =
          epred (llength (llist_of_nellist (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells))))"
    using nlength.rep_eq by blast 
 have 3: " epred (llength (llist_of_nellist (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells)))) =
           epred (llength (lfusecat (?n2l nells)))" 
    using one_eSuc plus_1_eSuc(2) by simp   
 have 4: "lfinite (?n2l nells)"
   using assms(1) lfinite_lmap nfinite_def by auto
 have 5: "\<not>lnull (?n2l nells)"
   by simp
 have 6: "\<forall>nell \<in> lset (?n2l nells). \<not> lnull nell"
   by simp
 have 7: "\<forall>nell \<in> lset (?n2l nells). lfinite nell" 
   using assms(2) nfinite_def by auto
 have 8: "llastlfirst (?n2l nells)" 
   using assms(3) nlastnfirst.rep_eq by auto
 have 9: "epred (llength (lfusecat (?n2l nells))) =
          epred (eSuc(\<Sum> i = 0 .. (the_enat(epred(llength (?n2l nells)))) . 
                  epred(llength (lnth (?n2l nells) i)))) "
   by (metis "4" "5" "6" "7" "8" lfusecat_llength_lfinite)
 have 10: "epred (eSuc(\<Sum> i = 0 .. (the_enat(epred(llength (?n2l nells)))) . 
                  epred(llength (lnth (?n2l nells) i)))) =
           ((\<Sum> i = 0 .. (the_enat(epred(llength (?n2l nells)))) . 
                  epred(llength (lnth (?n2l nells) i)))) "
   using epred_eSuc by blast
 have 11: "((epred(llength (?n2l nells)))) = ((nlength nells))"
   by simp
 have 12: "\<And>j. j \<le> ((epred(llength (?n2l nells))))\<longrightarrow>
               epred(llength (lnth (?n2l nells) j)) = nlength (nnth nells j)"  
     by (metis "5" co.enat.exhaust_sel comp_apply iless_Suc_eq llength_eq_0 llength_llist_of_nellist 
         llength_lmap lnth_llist_of_nellist lnth_lmap min_def the_enat.simps)
 have 13: "((\<Sum> i = 0 .. (the_enat(epred(llength (?n2l nells)))) . 
             epred(llength (lnth (?n2l nells) i)))) =
           ((\<Sum> i = 0 ..  (the_enat(nlength nells)). nlength (nnth nells i)))"
    using "12" assms(1) nfinite_nlength_enat by force
 show ?thesis using 1 2 3 9 10 13 by metis
qed

lemma nlastnfirst_ntaken: 
 assumes "n\<le>nlength nells"
         "nlastnfirst nells" 
 shows   "nlastnfirst (ntaken n nells)"
using assms 
by (metis Suc_ile_eq antisym_conv2 nappend_ntaken_ndropn nfinite_ntaken nlastnfirst_nappend_nfinite 
    ntaken_all) 




lemma nlastnfirst_ntake: 
 assumes "n\<le>nlength nells"
         "nlastnfirst nells"
 shows   "nlastnfirst (ntake n nells)"
proof (cases n)
case (enat nat)
then show ?thesis by (metis assms nlastnfirst_ntaken ntake_eq_ntaken)
next
case infinity
then show ?thesis
by (simp add: assms ntake_all)
qed


lemma nfusecat_ntake:
 assumes " (enat n) \<le> nlength nells" 
        "\<forall> nell \<in> nset nells. nfinite nell" 
        "nlastnfirst nells" 
  shows " nfusecat (ntake n  nells) =
          ntake ( ((\<Sum> i = 0 .. n . (nlength (nnth nells i))))) 
                (nfusecat nells) " 
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nfusecat (ntake n  nells) = 
         (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (ntake n nells))) " 
   by (simp add: nfusecat_def2)
 have 2: "(((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (ntake n nells))) = 
          nellist_of_llist (lfusecat (?n2l (ntake n nells)))"
    by simp 
 have 3: "llist_of_nellist(ntake n nells) = ltake (eSuc n) (llist_of_nellist nells)  "
     by (metis co.enat.distinct(1) llist_of_nellist_inverse_b llist_of_nellist_not_lnull 
         ltake.disc(2) nellist_of_llist_inverse ntake.abs_eq) 
 have 4: "(?n2l (ntake n nells)) = ltake (eSuc n) (?n2l nells)"
     using "3" by auto 
 have 5: "\<not> lnull (?n2l nells)" 
    by simp 
 have 6: " (Suc n) \<le> llength (?n2l nells)"
 by (metis "5" Suc_ile_eq assms(1) co.enat.collapse comp_apply iless_Suc_eq llength_eq_0 llength_lmap 
     nlength.rep_eq)
 have 7:  "\<forall> nell \<in> lset (?n2l nells). \<not> lnull nell"
    by simp
 have 8:  "\<forall> nell \<in> lset (?n2l nells). lfinite nell" 
   using assms(2) nfinite_def by fastforce
 have 9:    "llastlfirst (?n2l nells)" 
   using assms(3) nlastnfirst.rep_eq by auto
 have 10: "(lfusecat (ltake (eSuc n) (?n2l nells))) =
          ltake (if (Suc n) =0 then 0 else eSuc(\<Sum> i = 0 .. n . 
                epred(llength (lnth (?n2l nells) i)))) 
                (lfusecat (?n2l nells))" 
   using lfusecat_ltake[of "(?n2l nells)" "Suc n"  ]
         "5" "6" "7" "8" "9" diff_Suc_1 eSuc_enat by presburger
 have 11: "(if (Suc n) =0 then 0 else eSuc(\<Sum> i = 0 .. n . 
               epred(llength (lnth (?n2l nells) i)))) =
           ( eSuc(\<Sum> i = 0 .. n . epred(llength (lnth (?n2l nells) i))))" 
     using nat.simps(3) by presburger
 have 111: "\<And>j. j \<le>n \<longrightarrow> (min (enat j) (epred (llength (llist_of_nellist nells)))) = j "
     by (metis assms(1) enat_ord_simps(1) llength_llist_of_nellist min.bounded_iff min_def)  
 have 12: "\<And> j. j \<le>n \<longrightarrow> 
            epred(llength (lnth (?n2l nells) j)) = nlength ( nnth nells j) "
     using 5 111 assms lnth_llist_of_nellist[of nells]
     by (metis (full_types) co.enat.exhaust_sel iless_Suc_eq llength_eq_0 llength_lmap lnth_lmap 
         min.order_iff nlength.rep_eq o_apply the_enat.simps) 
 have 13: "(lfusecat (?n2l nells)) = llist_of_nellist (nfusecat nells)"
    by (simp add: lfusecat_not_lnull_var nfusecat_def2)
 have 14: "ltake (if (Suc n) =0 then 0 else eSuc(\<Sum> i = 0 .. n . 
                  epred(llength (lnth (?n2l nells) i)))) 
                (lfusecat (?n2l nells)) =
           ltake ( eSuc(\<Sum> i = 0 .. n .  nlength ( nnth nells i))) (llist_of_nellist (nfusecat nells))  "
    using "12" "13" by auto   
 have 15: "nellist_of_llist (ltake ( eSuc(\<Sum> i = 0 .. n .  
               nlength ( nnth nells i))) (llist_of_nellist (nfusecat nells))) =
          ntake ((\<Sum> i = 0 .. n .  nlength ( nnth nells i))) (nfusecat nells) " 
    by (metis llist_of_nellist_inverse_b llist_of_nellist_not_lnull ntake.abs_eq)
 show ?thesis
 by (metis "10" "14" "15" "2" "4" nfusecat_def2)
qed
   
lemma nfusecat_ndropn:
 assumes" n < nlength nells" 
        "\<forall> nell \<in> nset nells. nfinite nell" 
        "nlastnfirst nells" 
  shows " nfusecat (ndropn n nells) =
          ndropn (the_enat(if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . nlength(nnth nells i))))
                (nfusecat nells) "    
proof - 
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nfusecat (ndropn n nells) = 
          (((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (ndropn n nells)))"
   by (simp add: nfusecat_def2) 
 have 2: "(((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (ndropn n nells))) = 
          nellist_of_llist (lfusecat (lmap llist_of_nellist (ldropn n (llist_of_nellist nells))))" 
   using assms by simp
 have 4: "nellist_of_llist (lfusecat (lmap llist_of_nellist (ldropn n (llist_of_nellist nells)))) =
          nellist_of_llist (lfusecat (lmap llist_of_nellist (ldrop n (llist_of_nellist nells))))" 
   by (simp add: ldrop_enat)
 have 5: "(lmap llist_of_nellist (ldrop n (llist_of_nellist nells))) = ldrop n (?n2l nells)"
    by (simp)
 have 6: "\<not> lnull (?n2l nells)   "  
   by simp
 have 7: " n < llength (?n2l nells)"
 by (metis "5" assms(1) ldrop_enat llist.map_disc_iff llist_of_nellist_ndropn llist_of_nellist_not_lnull 
     lnull_ldrop min_def nlength.rep_eq not_le_imp_less order_less_imp_le the_enat.simps) 
 have 8: "\<forall> nell \<in> lset (?n2l nells). \<not> lnull nell"
   by simp
 have 9: "\<forall> nell \<in> lset (?n2l nells). lfinite nell" 
   using assms(2) nfinite_def by fastforce
 have 10: "llastlfirst (?n2l nells)" 
   using assms(3) nlastnfirst.rep_eq by auto
 have 11: " lfusecat (ldrop (enat n) (?n2l nells)) =
          ldrop (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . 
                  epred(llength (lnth (?n2l nells) i))))
                (lfusecat (?n2l nells)) " 
     using "10" "6" "7" "8" "9" lfusecat_ldrop by blast
 have 111: "\<And>j. 0< j \<and> j <n-1 \<longrightarrow> 
                (min (enat j) (epred (llength (llist_of_nellist nells)))) = j"
      by (metis "7" comp_apply epred_enat epred_min less_imp_of_nat_less llength_lmap min.absorb3 
          min_less_iff_conj of_nat_eq_enat)
 have 112: "n-1 < llength (?n2l nells)"
      by (metis "7" One_nat_def Suc_ile_eq Suc_pred epred_0 epred_enat not_gr_zero order_less_imp_le zero_enat_def) 
 have 12: "\<And>j. 0< n \<and> j < n-1 \<longrightarrow> 
               epred(llength (lnth (?n2l nells) j)) = nlength(nnth nells j)" 
    using 6 7 111  lnth_llist_of_nellist[of nells] 
    by simp 
     (metis canonically_ordered_monoid_add_class.lessE diff_le_self dual_order.strict_trans1 
      enat_less_enat_plusI enat_min_eq_0_iff lnth_lmap nlength.rep_eq not_gr_zero the_enat.simps 
      zero_enat_def)
 have 13: "(lfusecat (?n2l nells)) = llist_of_nellist (nfusecat nells) " 
    by (simp add: lfusecat_not_lnull_var nfusecat_def2)
 have 131: "\<And>j . j< n \<Longrightarrow> (min (enat j) (epred (llength (llist_of_nellist nells)))) =j "
       by (metis (full_types) assms(1) min.assoc min.orderE min_enat_simps(1) nlength.rep_eq 
           order_less_imp_le) 
 have 132: "(if n = 0 then 0 else \<Sum>i = 0..n - 1. epred (llength (lnth (?n2l nells) i))) =
            (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . nlength(nnth nells i)))" 
      using sum.cong[of "{0..n-1}" "{0..n-1}" "\<lambda>i. epred (llength (lnth (?n2l nells) i)) " 
             "\<lambda>i. nlength(nnth nells i)"  ] assms 7 131 12 lnth_llist_of_nellist[of nells]
     by (metis "112" atLeastAtMost_iff comp_apply diff_less less_numeral_extra(1) llength_lmap 
           lnth_lmap nlength.rep_eq not_gr_zero order_neq_le_trans the_enat.simps)
 have 14: "ldrop (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . epred(llength (lnth (?n2l nells) i))))
                (lfusecat (?n2l nells)) = 
           ldrop (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . nlength(nnth nells i))) (llist_of_nellist (nfusecat nells))"
        using "13" "132" by presburger
 have 15: "0<n \<Longrightarrow> (\<And> j.  j \<le>n-1  \<longrightarrow>  epred(llength (lnth (?n2l nells) j)) < \<infinity>)" 
      using 7 9
      by (metis (no_types, opaque_lifting) "112" enat_ord_simps(1) enat_ord_simps(4) epred_llength 
          in_lset_conv_lnth lfinite_ltl llength_eq_infty_conv_lfinite order_le_less_trans)  
 have 16: "0< n \<Longrightarrow> ((\<Sum> i = 0 .. (n-1) . epred(llength (lnth (?n2l nells) i)))) < \<infinity>" 
    using 15
     proof (induct n)
     case 0
     then show ?case by simp
     next
     case (Suc n)
     then show ?case 
        proof (cases "n=0")
        case True
        then show ?thesis using Suc by simp
        next
        case False
        then show ?thesis using Suc 
        by simp 
           (metis One_nat_def Suc_pred' dual_order.refl plus_enat_simps(1) sum.atLeast0_atMost_Suc)
        qed
     qed
  have 17: "\<exists> m. (enat m) = (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . 
             epred(llength (lnth (?n2l nells) i))))" 
     by (metis "16" less_infinityE not_gr_zero zero_enat_def)     
  obtain m where 18: "(enat m) = (if n = 0 then 0 else 
                       (\<Sum> i = 0 .. (n-1) . epred(llength (lnth (?n2l nells) i))))"
     using 17 by blast
  have 19: "ldrop (if n = 0 then 0 else 
                   (\<Sum> i = 0 .. (n-1) . epred(llength (lnth (?n2l nells) i))))
                (lfusecat (?n2l nells)) = 
            ldropn m (lfusecat (?n2l nells))  " 
      using 18 ldrop_enat[of m "(lfusecat (?n2l nells))"] 
         by presburger 
  have 20: " enat m \<le> epred (llength (lfusecat (?n2l nells))) " 
       by (metis (no_types, lifting) "11" "19" "6" "7" "8" co.enat.exhaust_sel iless_Suc_eq in_lset_ldropD
           lfusecat_not_lnull_var linorder_not_le llength_eq_0 lnull_ldrop lnull_ldropn)
  have 21: "nellist_of_llist (ldropn m (lfusecat (?n2l nells))) = ndropn m (nfusecat nells) "
       using 20
       by (metis "13" llist_of_nellist_inverse_b llist_of_nellist_not_lnull min_def ndropn.abs_eq 
           the_enat.simps)  
  have 22: "m = (the_enat(if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . nlength(nnth nells i))))"  
      by (metis (mono_tags, lifting) "132" "18" the_enat.simps)  
  show ?thesis using 1 2 4 5 11 19 21 22 by (metis)
qed

lemma nridx_nfusecat_ntake: 
 assumes "nridx R (nfusecat nells)"
         " (enat n) \<le> nlength nells" 
         "nlastnfirst nells"
         "\<forall> nell \<in> nset nells. nfinite nell"
 shows   " nridx R (nfusecat (ntake n nells))"
using assms by (metis nfusecat_ntake nle_le nridx_ntake_a ntake_all)

lemma nridx_nfusecat_ndrop: 
 assumes "nridx R (nfusecat nells)"
         " (enat n) < nlength nells" 
         "nlastnfirst nells"
         "\<forall> nell \<in> nset nells. nfinite nell" 
 shows   " nridx R (nfusecat (ndropn n nells))"
proof -
let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nridx R (nfusecat (ndropn n nells)) \<longleftrightarrow> 
          nridx R ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (ndropn n nells))"
    by (simp add: nfusecat_def2)
 have 2: " nridx R ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (ndropn n nells)) \<longleftrightarrow>
           ridx R (llist_of_nellist ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (ndropn n nells)))"
    using nridx.rep_eq by blast
 have 3: "ridx R (llist_of_nellist ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) (ndropn n nells))) \<longleftrightarrow>
          ridx R ( (( lfusecat \<circ> ?n2l) (ndropn n nells)))" 
   using not_null_lfusecat
   by (metis (no_types, lifting) comp_apply nridx.abs_eq nridx.rep_eq)
 have 4: "llastlfirst (?n2l nells)"
   using assms nlastnfirst.rep_eq by auto
 have 5: "\<forall> nell \<in> lset (?n2l nells). lfinite nell" 
   using assms nfinite_def by fastforce
 have 6: "( (( lfusecat \<circ> ?n2l) (ndropn n nells))) =
          lfusecat (lmap llist_of_nellist (ldropn n (llist_of_nellist nells)))" 
   by (simp add: assms(2))
 have 7: "(lmap llist_of_nellist (ldropn n (llist_of_nellist nells))) =
          ldropn n (lmap llist_of_nellist (llist_of_nellist nells))"
    by auto 
 have 8: "ridx R (lfusecat (?n2l nells))" 
    by (metis (no_types, lifting) assms(1) comp_apply nfusecat_def2 nridx.abs_eq ridx_expand_1)
 have 9: "enat n < llength (?n2l nells)"
    by (metis assms(2) co.enat.exhaust_sel comp_apply iless_Suc_eq llength_eq_0 llength_lmap 
        llist_of_nellist_not_lnull nlength.rep_eq order_less_imp_le) 
 have 10: "ridx R (lfusecat (ldropn n (?n2l nells)))" 
     using ridx_lfusecat_ldrop[of R "(?n2l nells)" n]
     by (metis (mono_tags, lifting) "4" "5" "8" "9" comp_apply in_lset_conv_lnth ldrop_enat llength_lmap
           llist_of_nellist_not_lnull lnth_lmap)
 show ?thesis 
 using "1" "10" "2" "3" "6" "7" by (metis comp_apply)      
qed 


lemma nridx_nfusecat:
 assumes "nlastnfirst nells"
         "\<forall> nell \<in> nset nells. 0 < nlength nell"
         "\<forall> nell \<in> nset nells. nfinite nell"         
 shows   "nridx R (nfusecat nells) \<longleftrightarrow> (\<forall>i.  i \<le> nlength nells \<longrightarrow>  nridx R (nnth nells i) )"
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nridx R (nfusecat nells) \<longleftrightarrow> 
          nridx R ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells)"
   by (simp add: nfusecat_def)
 have 2: "nridx R ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells) \<longleftrightarrow>
          ridx R (llist_of_nellist ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells))" 
   using nridx.rep_eq by blast
 have 3: "ridx R (llist_of_nellist ((nellist_of_llist \<circ> lfusecat \<circ> ?n2l) nells)) \<longleftrightarrow>
          ridx R ( (( lfusecat \<circ> ?n2l) nells))" 
   using not_null_lfusecat nridx.abs_eq by fastforce
 have 4: "llastlfirst (?n2l nells)"
   using assms(1) nlastnfirst.rep_eq by auto
 have 5:  "\<forall> nell \<in> lset (?n2l nells). 1 < llength nell"
    by (metis assms(2) epred_1 comp_apply imageE less_numeral_extra(3) llist.set_map 
        llist_of_nellist_not_lnull lset_llist_of_nellist_a nlength.rep_eq not_lnull_llength 
        order_neq_le_trans)
 have 6: "\<forall> nell \<in> lset (?n2l nells). lfinite nell" 
   using assms(3) nfinite_def by fastforce
 have 7: "ridx R ( (( lfusecat \<circ> ?n2l) nells)) \<longleftrightarrow> 
          (\<forall>i.  i < llength (?n2l nells) \<longrightarrow>  ridx R (lnth (?n2l nells) i) )"
     using "4" "5" "6" ridx_lfusecat by fastforce
 have 8: "epred(llength (?n2l nells)) = nlength nells " 
   by simp
 have 9: "\<And>j. j < llength (?n2l nells) \<longrightarrow> 
              (lnth (?n2l nells) j) = llist_of_nellist(nnth nells j) "
   by simp 
      (metis  eSuc_epred eSuc_ile_mono gr_implies_not_zero ileI1  lnth_llist_of_nellist 
       min_def the_enat.simps)
 have 10: " (\<forall>i.  i < llength (?n2l nells) \<longrightarrow>  ridx R (lnth (?n2l nells) i) ) \<longleftrightarrow>
            (\<forall>i.  i \<le> nlength (nells) \<longrightarrow>  nridx R (nnth nells i) )"
        by (metis (mono_tags, lifting) "8" "9" co.enat.exhaust_sel comp_apply iless_Suc_eq 
            llength_eq_0 llist.map_disc_iff llist_of_nellist_not_lnull nridx.rep_eq)
 show ?thesis using 1 2 3 10 7 by blast
qed

lemma nfusecat_split_nsubn: 
 assumes "nlastnfirst nells"
         "\<forall> nell \<in> nset nells. 0 < nlength nell"
         "\<forall> nell \<in> nset nells. nfinite nell"   
 shows "(\<forall> i. i\<le> nlength nells \<longrightarrow> nridx (\<lambda> a b. f (nsubn \<sigma> a b))  (nnth nells i)) \<longleftrightarrow>
        nridx (\<lambda> a b. f (nsubn \<sigma> a b)) (nfusecat nells) "
 using assms nridx_nfusecat by blast

lemma nidx_nfusecat:
 assumes "nlastnfirst nells"
         "\<forall> nell \<in> nset nells. 0 < nlength nell"
         "\<forall> nell \<in> nset nells. nfinite nell"         
 shows   "nidx (nfusecat nells) \<longleftrightarrow> (\<forall>i.  i \<le> nlength nells \<longrightarrow>  nidx (nnth nells i) )"
using assms nridx_nfusecat nridx_nidx by blast

lemma nnth_sum_expand: 
 assumes "nlastnfirst nells"
         "\<forall> nell \<in> nset nells. 0 < nlength nell"
         "\<forall> nell \<in> nset nells. nfinite nell"   
         "(enat i) \<le> nlength nells" 
         "nfinite nells" 
 shows " (\<Sum> i = 0 .. (the_enat((nlength nells))) . (nlength (nnth nells i))) =
         (nlength (nnth nells i)) + 
        (\<Sum> j \<in> {k. k\<noteq>i  \<and> k\<le> (the_enat((nlength nells)))} . (nlength (nnth nells j))) "
proof -
 have 1: "{k. k\<le> (the_enat((nlength nells)))} = insert i {k. k\<noteq>i  \<and> k\<le> (the_enat((nlength nells)))}" 
   using assms by auto 
         (metis enat_ord_simps(1) nfinite_nlength_enat the_enat.simps)
 have 2: "{0.. (the_enat((nlength nells)))} = {k. k\<le> (the_enat((nlength nells)))}" 
    by auto 
 have 3: "(\<Sum> i = 0 .. (the_enat((nlength nells))) . (nlength (nnth nells i))) = 
          (\<Sum> i \<in>{k. k\<le> (the_enat((nlength nells)))} . (nlength (nnth nells i))) " 
       using "2" by presburger
 have 4: "(\<Sum> i \<in>{k. k\<le> (the_enat((nlength nells)))} .(nlength (nnth nells i))) =
          (\<Sum> j \<in>(insert i {k. k\<noteq>i  \<and> k\<le> (the_enat((nlength nells)))}) . (nlength (nnth nells j))) " 
      using "1" by presburger
 have 5: "(\<Sum> j \<in>(insert i {k. k\<noteq>i  \<and> k\<le> (the_enat((nlength nells)))}) . (nlength (nnth nells j))) =
          (nlength (nnth nells i)) + 
        (\<Sum> j \<in> {k. k\<noteq>i  \<and> k\<le> (the_enat((nlength nells)))} . (nlength (nnth nells j)))" 
     by auto
 show ?thesis 
 using "3" "4" "5" by presburger
qed

lemma nfusecat_nlength_a: 
 assumes "nlastnfirst nells"
         "\<forall> nell \<in> nset nells. 0 < nlength nell"
         "\<forall> nell \<in> nset nells. nfinite nell"   
         " i \<le> nlength nells"
         "(enat j) \<le> nlength (nnth nells i)"
 shows " (enat j) \<le> nlength (nfusecat nells)"
proof (cases "nfinite nells")
case True
then show ?thesis 
  proof -
   have 2: "nlength (nfusecat nells) = 
            (\<Sum> i = 0 .. (the_enat((nlength nells))) . (nlength (nnth nells i)))" 
    using True assms nfusecat_nlength_nfinite by fastforce
   have 3: "nlength (nnth nells i) \<le> 
            (\<Sum> i = 0 .. (the_enat((nlength nells))) . (nlength (nnth nells i)))"
      using nnth_sum_expand[of nells i] assms 
      by (metis (no_types, lifting) True le_iff_add)  
   show ?thesis using assms 2 3 by force  
 qed
next
case False
then show ?thesis 
  proof -
   have 5: "\<not> nfinite (nfusecat nells) " 
     using False assms nfusecat_nfinite by blast
   show ?thesis 
   using 5 
   by (simp add: nfinite_conv_nlength_enat)
  qed
qed 


subsection \<open>nkfilter\<close>

lemma nkfilter_def1:
  assumes  " (\<exists> x \<in> nset nell. P x)" 
  shows    " nkfilter P n nell = nmap snd (nfilter (P\<circ>fst) (nzip nell (niterates Suc n)))"
using assms 
apply transfer
by (simp add: kfilter_def) 
  (metis imageE llist.set_map lmap_fst_lzip)

lemma nmap_fst_nzip:
 " (nmap fst (nzip nell (niterates Suc n))) = nell"
apply transfer
  by (simp add: lmap_fst_lzip)

lemma nfilter_nkfilter_1: 
  assumes  " (\<exists> x \<in> nset nell. P x)" 
 shows     " (nmap fst (nfilter (P\<circ>fst) (nzip nell (niterates Suc n)))) =
              (nfilter P nell)"
using assms
by (metis nfilter_nmap nmap_fst_nzip)
 

lemma nkfilter_NNil [simp]:
  shows "P b \<Longrightarrow> nkfilter P n (NNil b)  = (NNil n)"
by transfer auto 

lemma nkfilter_NCons [simp]:
 assumes  " (\<exists> x \<in> nset nell. P x)" 
 shows    " nkfilter P n (NCons x nell) = 
           (if P x then NCons n (nkfilter P (Suc n) nell) else nkfilter P (Suc n) nell)"
using assms
by transfer
   (auto simp add: kfilter_lnull_conv)

lemma nkfilter_NCons_a [simp]:
assumes " \<not>(\<exists> x \<in> nset nell. P x)"  
        "P x"
shows   " nkfilter P n (NCons x nell) = (NNil n)"
using assms
by transfer
   (auto simp add: kfilter_lnull_conv)

lemma nkfilter_nlength:
 assumes "\<exists> x \<in> nset nell. P x"
 shows   " nlength(nkfilter P n nell) = nlength(nfilter P nell)"
using assms
by transfer
   (auto simp add: kfilter_lnull_conv kfilter_llength)

lemma nkfilter_upperbound:
 assumes "\<exists> x \<in> nset nell. P x"
         " i \<le> nlength (nkfilter P n nell)"
 shows   " nnth (nkfilter P n nell) i \<le> n + nlength nell" 
using assms
by transfer
   (auto, 
    simp add: kfilter_lnull_conv,
    metis co.enat.exhaust_sel iadd_Suc_right iless_Suc_eq kfilter_upperbound llength_eq_0)

lemma nkfilter_lowerbound:
 assumes "\<exists> x \<in> nset nell. P x"
         " i \<le> nlength (nkfilter P n nell)"
 shows   " n \<le> nnth (nkfilter P n nell) i "
 using assms
by transfer
  (auto,
   metis co.enat.collapse iless_Suc_eq kfilter_lnth_n_zero_a llength_eq_0)

lemma nkfilter_mono:
 assumes "\<exists> x \<in> nset nell. P x"
         " (Suc i) \<le> nlength (nkfilter P n nell)"
 shows   " nnth (nkfilter P n nell) i < nnth (nkfilter P n nell) (Suc i)"
using assms
by transfer
   (auto,
    metis diff_Suc_1 eSuc_epred epred_enat epred_le_epredI lidx_kfilter_gr iless_Suc_eq leD lessI 
    llength_eq_0 min.cobounded1 min_def the_enat.simps)

lemma nkfilter_nfilter:
 assumes "\<exists> x \<in> nset nell. P x"
         " i \<le> nlength (nkfilter P n nell)"
 shows   " (nnth nell ((nnth (nkfilter P n nell) i) - n)) = (nnth (nfilter P nell) i)"
using assms
apply transfer
proof (auto simp add: kfilter_lnull_conv split:if_split_asm)
 fix nella :: "'a llist" and Pa :: "'a \<Rightarrow> bool" and ia :: nat and na :: nat and x :: 'a
 assume a1: "x \<in> lset nella"
 assume a2: "Pa x"
 assume a3: "enat ia \<le> epred (llength (kfilter Pa na nella))"
 assume a4: "\<not> lnull nella"
 have f5: "\<forall>e ea. \<not> (e::enat) \<le> ea \<or> min e ea = e"
 by (simp add: min_def_raw)
 have f6: "\<forall>n p na as. \<not> enat n < llength (kfilter p na (as::'a llist)) \<or>
             enat (lnth (kfilter p na as) n) < enat na + llength as"
 by (meson kfilter_upperbound)
 have f7: "min (enat ia) (epred (llength (kfilter Pa na nella))) = enat ia"
 using f5 a3 by blast
 then have f8: "(enat ia < llength (kfilter Pa 0 nella)) = 
               (enat (the_enat (min (enat ia) (epred (llength (kfilter Pa na nella))))) < 
                llength (kfilter Pa na nella))"
 by (simp add: kfilter_llength)
 have f9: "\<forall>n p na as. \<not> enat n < llength (kfilter p na (as::'a llist)) \<or> 
              lnth (kfilter p na as) n - na = lnth (kfilter p 0 as) n"
 using kfilter_lnth_n_zero by blast
 have "eSuc (epred (llength nella)) = enat 0 + llength nella"
 using a4 by (metis co.enat.exhaust_sel gen_llength_def llength_code llength_eq_0)
 then have "enat (the_enat (min (enat ia) (epred (llength (kfilter Pa na nella))))) < 
           llength (kfilter Pa na nella) \<longrightarrow> 
           enat (lnth (kfilter Pa na nella) 
                      (the_enat (min (enat ia) (epred (llength (kfilter Pa na nella))))) - na)
           < eSuc (epred (llength nella))"
 using f9 f8 f7 f6 the_enat.simps by presburger
 then have f10: "enat (the_enat (min (enat ia) (epred (llength (kfilter Pa na nella))))) <
                 llength (kfilter Pa na nella) \<longrightarrow> 
                enat (lnth (kfilter Pa na nella) 
                           (the_enat (min (enat ia) (epred (llength (kfilter Pa na nella))))) - na)
                 \<le> epred (llength nella)"
 using iless_Suc_eq by blast
 have f11: "\<forall>n p na as. \<not> enat n < llength (kfilter p na as) \<or> lnth as (lnth (kfilter p na as) n - na) =
                (lnth (lfilter p as) n::'a)"
 using lfilter_kfilter by blast
 have f12: "the_enat (min (enat ia) (epred (llength (lfilter Pa nella)))) = 
           the_enat (min (enat ia) (epred (llength (kfilter Pa na nella))))"
 by (simp add: kfilter_llength)
 have "\<not> lnull (kfilter Pa na nella)"
 using a1 a2 kfilter_not_lnull_conv by auto
 then have "enat (the_enat (min (enat ia) (epred (llength (kfilter Pa na nella))))) < 
           llength (kfilter Pa na nella)"
 using f7 a3 by (metis co.enat.exhaust_sel iless_Suc_eq llength_eq_0 the_enat.simps)
 then show "\<not> lnull nella \<Longrightarrow>
       enat ia \<le> epred (llength (kfilter Pa na nella)) \<Longrightarrow>
       x \<in> lset nella \<Longrightarrow>
       Pa x \<Longrightarrow>
       lnth nella (the_enat (min (enat (lnth (kfilter Pa na nella) ia - na)) (epred (llength nella)))) =
       lnth (lfilter Pa nella) (the_enat (min (enat ia) (epred (llength (lfilter Pa nella)))))"
 using f12 f11 f10 f5 by simp
qed

lemma in_nkfilter_nset:
 assumes "\<exists> x \<in> nset nell. P x"
 shows " x \<in> nset(nkfilter P n nell) \<longleftrightarrow> x \<in> { n+i | i. i \<le> nlength nell \<and> P (nnth nell i)} "
using assms
by transfer
   (auto simp add: kfilter_lnull_conv  min_def,
    metis co.enat.exhaust_sel gen_llength_def iless_Suc_eq kfilter_holds kfilter_llength_n_zero 
       kfilter_lnth_n_zero kfilter_lowerbound kfilter_upperbound ldistinct_Ex1 ldistinct_kfilter 
       le_add_diff_inverse llength_code llength_eq_0,
    metis add.commute co.enat.exhaust_sel iless_Suc_eq kfilter_holds_not_a llength_eq_0)

lemma nkfilter_nset:
 assumes "\<exists> x \<in> nset nell. P x"
 shows " nset(nkfilter P n nell) = { n+i | i. i \<le> nlength nell \<and> P (nnth nell i)} "
using assms in_nkfilter_nset[of nell P _ n] by blast  

lemma nlength_nkfilter_le [simp]:
 assumes "\<exists> x \<in> nset nell. P x"
shows "nlength (nkfilter P n nell) \<le> nlength nell"
using assms
by transfer
    (auto simp add: kfilter_lnull_conv epred_le_epredI kfilter_llength llength_lfilter_ile)

lemma nkfilter_nleast:
 assumes "\<exists> x \<in> nset xs. P x"
 shows   " nnth (nkfilter P n xs) 0 = n+(nleast P xs)"
using assms
by transfer
   (auto simp add: kfilter_not_lnull_conv  kfilter_lnull_conv,
    metis enat_min_eq_0_iff kfilter_lnth_zero kfilter_lnull_conv the_enat_0 zero_enat_def)

lemma ndistinct_nkfilter:
 assumes "\<exists> x \<in> nset nell. P x"
 shows   "ndistinct(nkfilter P n nell)"
using assms
by transfer
  (auto simp add: kfilter_lnull_conv ldistinct_kfilter)

lemma nkfilter_ndropn_nset:
 assumes "\<exists> x \<in> nset (ndropn k nell). P x"
         " k \<le> nlength nell " 
 shows " nset(nkfilter P n (ndropn k nell)) = { n+i|i. i\<le>nlength nell - k \<and> P (nnth nell (k+i))}"
using assms nkfilter_nset[of "(ndropn k nell)" P n] 
ndropn_nnth[of _ nell k] by auto 

lemma nkfilter_ndropn_nset_b:
 assumes "\<exists> x \<in> nset (ndropn k nell). P x"
         " k \<le> nlength nell " 
 shows " nset(nkfilter P n (ndropn k nell)) = { n+ i |i. i\<le> nlength nell - k  \<and> P (nnth nell (i+k))}"
proof -
 have 1: "{ n+i|i. i\<le>nlength nell - k \<and> P (nnth nell (k+i))} = 
          { n+ i |i. i\<le> nlength nell - k  \<and> P (nnth nell (i+k))}" 
   using assms by (auto simp add: add.commute)
 show ?thesis using assms by (simp add: "1"  nkfilter_ndropn_nset)
qed

lemma nfilter_nkfilter_ntaken_nlength_0:
 assumes " P (nnth nell ( (nnth (nkfilter P n nell) k) -n )) "
         " k \<le> nlength (nfilter P nell) "
 shows   " (\<exists> x \<in> nset nell. P x) "
using assms
by (metis enat_ile in_nset_conv_nnth linorder_le_cases ntaken_all ntaken_nlast)

lemma nkfilter_nlength_n_zero:
 assumes " (\<exists> x \<in> nset nell. P x) "
 shows   " nlength(nkfilter P n nell) = nlength(nkfilter P 0 nell)"
using assms by (simp add: nkfilter_nlength)

lemma nkfilter_nnth_n_zero:
 assumes " (\<exists> x \<in> nset nell. P x) "
         " k \<le> nlength(nkfilter P n nell) "
 shows   " (nnth (nkfilter P n nell) k) - n = (nnth (nkfilter P 0 nell) k) "
using assms 
by transfer
  (auto split: if_split_asm,
   metis kfilter_lnull_conv,
   metis kfilter_lnull_conv,
   metis co.enat.exhaust_sel iless_Suc_eq kfilter_llength_n_zero kfilter_lnth_n_zero llength_eq_0 
   min.orderE the_enat.simps)

lemma nkfilter_n_zero:
 assumes " (\<exists> x \<in> nset nell. P x) "
 shows " (nkfilter P n nell) = nmap (\<lambda>i. i+n) (nkfilter P 0 nell)"
proof -
 have 1: "nlength(nkfilter P n nell) = nlength (nmap (\<lambda>i. i+n) (nkfilter P 0 nell))" 
   using assms nkfilter_nlength_n_zero by fastforce
 have 2: "\<And> k. k \<le> nlength (nkfilter P n nell) \<longrightarrow> 
           nnth (nkfilter P n nell) k = nnth (nmap (\<lambda>i. i+n) (nkfilter P 0 nell)) k"
   using 1 assms nkfilter_nnth_n_zero[of nell P _ n]  
   by (metis diff_add nkfilter_lowerbound nlength_nmap nnth_nmap)
 show ?thesis by (simp add: "1" "2" nellist_eq_nnth_eq)
qed

lemma nkfilter_n_zero_a:
 assumes " (\<exists> x \<in> nset nell. P x) "
 shows " (nkfilter P 0 nell) = nmap (\<lambda>i. i-n) (nkfilter P n nell)"
proof -
 have 1: "nlength(nkfilter P 0 nell) = nlength (nmap (\<lambda>i. i-n) (nkfilter P n nell))" 
   by (metis assms nkfilter_nlength_n_zero nlength_nmap)
 have 2: "\<And>k. k\<le> nlength(nkfilter P 0 nell) \<longrightarrow> 
               nnth (nkfilter P 0 nell) k = nnth (nmap (\<lambda>i. i-n) (nkfilter P n nell)) k"
   by (simp add: "1" assms nkfilter_nnth_n_zero)
 show ?thesis by (simp add: "1" "2" nellist_eq_nnth_eq)
qed

lemma nkfilter_holds:
 assumes " (\<exists> x \<in> nset nell. P x) "
         " y \<in> nset(nkfilter P n nell)"
 shows   " P (nnth nell (y-n))"
using assms  in_nkfilter_nset[of nell P]  by force

lemma nkfilter_holds_not:
 assumes " (\<exists> x \<in> nset nell. P x) "
         " y \<in> {i+n |i. i\<le> nlength nell} - (nset (nkfilter P n nell))"
 shows  " \<not> P (nnth nell (y-n)) "
using assms nkfilter_nset[of nell P n] by auto 

lemma nkfilter_holds_a:
 assumes " (\<exists> x \<in> nset nell. P x) "
         "i \<le> nlength nell"
         " (i+n) \<in> nset(nkfilter P n nell)"
 shows " P (nnth nell i) "
using assms nkfilter_holds[of nell P "i+n" n ] by simp

lemma nkfilter_holds_not_a:
 assumes " (\<exists> x \<in> nset nell. P x) "
         " i \<le> nlength nell"
         " P (nnth nell i) "
 shows   " (i+n) \<in>  nset(nkfilter P n nell)"
using assms by (simp add: in_nkfilter_nset)

lemma nkfilter_holds_b:
 assumes " (\<exists> x \<in> nset nell. P x) "
         " i \<le> nlength nell"
 shows   " (i+n) \<in> nset(nkfilter P n nell) = P (nnth nell i)"
using assms by (meson nkfilter_holds_a nkfilter_holds_not_a)

lemma nkfilter_holds_c:
 assumes " (\<exists> x \<in> nset nell. P x) "
         " n \<le> i" 
         " i-n \<le> nlength nell"
 shows   " i \<in> nset(nkfilter P n nell) = P (nnth nell (i-n))"
using assms 
by (metis diff_add idiff_enat_enat nkfilter_holds_b) 

lemma nkfilter_holds_not_b:
  assumes " (\<exists> x \<in> nset nell. P x) "
          " n \<le> i"
          " i-n \<le> nlength nell"
  shows   " i\<notin> nset (nkfilter P n nell) = (\<not> P (nnth nell (i-n))) "
using assms 
by (simp add: nkfilter_holds_c)

lemma nkfilter_disjoint_nset_coset:
 assumes " (\<exists> x \<in> nset nell. P x) "
 shows " ({i+n|i. i \<le> nlength nell} - nset(nkfilter P n nell)) \<inter> nset (nkfilter P n nell) = {}"
using assms by (simp add: inf.commute)


lemma nidx_nkfilter_expand:
 assumes " (\<exists> x \<in> nset nell. P x) "
         "(Suc i) \<le> nlength(nkfilter P n nell)"
 shows   " nnth (nkfilter P n nell) i < nnth (nkfilter P n nell) (Suc i)"
using assms by (simp add: nkfilter_mono)

lemma nidx_nkfilter:
 assumes " (\<exists> x \<in> nset nell. P x) "
 shows   "nidx (nkfilter P n nell)"
using assms nidx_nkfilter_expand[of nell P _ n]  nidx_expand[of "(nkfilter P n nell)" ] 
by blast

lemma nidx_nkfilter_gr_eq:
 assumes " (\<exists> x \<in> nset nell. P x) "
         " k \<le> j"
         " j \<le> nlength (nkfilter P n nell) " 
 shows   " nnth (nkfilter P n nell) k \<le> nnth (nkfilter P n nell) j"
using assms  nidx_nkfilter[of nell P n]  nidx_less_eq[of "nkfilter P n nell" k j]
by blast

lemma nidx_nkfilter_gr:
 assumes " (\<exists> x \<in> nset nell. P x) "    
 shows   "(\<forall> j. k<j \<and> j \<le> nlength (nkfilter P n nell) \<longrightarrow>
                 nnth (nkfilter P n nell) k < nnth(nkfilter P n nell) j) "
using assms nidx_nkfilter[of nell P n] nidx_less[of "nkfilter P n nell" ]
using less_imp_Suc_add by blast 

lemma nidx_nkfilter_less_eq:
 assumes " (\<exists> x \<in> nset nell. P x) "  
         "k \<le> nlength (nkfilter P n nell)"
 shows   "\<forall>j. j \<le> k \<longrightarrow> nnth (nkfilter P n nell) j \<le> nnth (nkfilter P n nell) k"
using assms by (simp add: nidx_nkfilter_gr_eq) 

lemma nkfilter_not_before:
  assumes " (\<exists> x \<in> nset nell. P x) " 
          " i < (nnth (nkfilter P 0 nell) 0) "
   shows  " \<not> P (nnth nell i) "
using assms
by transfer  
   (auto simp add: min_def split: if_split_asm,
    meson kfilter_not_before llength_eq_0 not_gr_zero,
    metis dual_order.strict_trans kfilter_not_before less_enatE llength_eq_0 not_gr_zero 
    not_le_imp_less the_enat.simps,
    meson le_less_linear less_enatE less_nat_zero_code,
    meson le_less_linear less_enatE not_less0)

lemma nkfilter_n_not_before:
  assumes " (\<exists> x \<in> nset (ndropn n nell). P x) "
          " n \<le> nlength nell"
          " n \<le> i"  
          " i < (nnth (nkfilter P n (ndropn n nell) ) 0) "
   shows  " \<not> P (nnth nell i) "
using assms
apply transfer
unfolding min_def 
using zero_enat_def
proof (auto split: if_split_asm)
 show "\<And>n nell P i x.
       enat 0 = 0 \<Longrightarrow>
       \<not> lnull nell \<Longrightarrow>
       enat n \<le> epred (llength nell) \<Longrightarrow>
       n \<le> i \<Longrightarrow>
       \<not> lnull (kfilter P n (ldropn n nell)) \<Longrightarrow>
       i < lnth (kfilter P n (ldropn n nell)) 0 \<Longrightarrow>
       x \<in> lset (ldropn n nell) \<Longrightarrow> P x \<Longrightarrow> enat i \<le> epred (llength nell) \<Longrightarrow> P (lnth nell i) \<Longrightarrow> False"
 by (metis co.enat.exhaust_sel iless_Suc_eq kfilter_n_not_before llength_eq_0 not_gr_zero) 
next
 fix na :: nat and nella :: "'b llist" and Pa :: "'b \<Rightarrow> bool" and ia :: nat and x :: 'b
 assume a1: "ia < lnth (kfilter Pa na (ldropn na nella)) 0"
 assume a2: "\<not> lnull (kfilter Pa na (ldropn na nella))"
 assume a3: "enat na \<le> epred (llength nella)"
 assume a4: "\<not> lnull nella"
 assume a5: "\<not> enat ia \<le> epred (llength nella)"
 assume a6: "Pa (lnth nella (the_enat (epred (llength nella))))"
 have f7: "\<forall>p n bs. lnull (kfilter p n (bs::'b llist)) \<or> lnth (kfilter p n bs) 0 = n + lleast p bs"
 by (meson kfilter_lnth_zero)
 then have f8: "lnth (kfilter Pa na (ldropn na nella)) 0 = na + lleast Pa (ldropn na nella)"
 using a2 by blast
 have f9: "0 < llength (kfilter Pa na (ldropn na nella))"
 using a2 by simp
 have f10: "na \<le> the_enat (epred (llength nella))" 
  by (metis a3 a5 enat_ord_code(4) enat_ord_simps(1) enat_the_enat less_imp_le)
 have f11: "the_enat (epred (llength nella)) < lnth (kfilter Pa na (ldropn na nella)) 0"
   by (metis a1 a5 dual_order.strict_trans enat_ord_simps(2) enat_ord_simps(3) enat_the_enat 
       not_le_imp_less)
 then show False using kfilter_n_not_before[of Pa na nella "(the_enat (epred (llength nella)))"]
   using f10 f11 
   by (metis a3 a4 a6 co.enat.exhaust_sel f9 iless_Suc_eq llength_eq_0)
qed

lemma nkfilter_not_after:
assumes " (\<exists> x \<in> nset nell. P x) " 
        " nnth (nkfilter P 0 nell) k < i "
        " nlength(nkfilter P 0 nell) = (enat k)"
        " i \<le> nlength nell"
 shows  " \<not> P (nnth nell i)"
using assms
by transfer
   (auto simp add: min_def split: if_split_asm,
    metis co.enat.exhaust_sel diff_Suc_1 eSuc_enat iless_Suc_eq kfilter_not_after llength_eq_0 
    not_gr_zero)

lemma nkfilter_n_not_after:
assumes " (\<exists> x \<in> nset (ndropn n nell). P x) "
        " n \<le> nlength nell"  
        " nnth (nkfilter P n (ndropn n nell)) k < i "
        " nlength(nkfilter P n (ndropn n nell)) = (enat k)"
        " i \<le> nlength nell"
shows  " \<not> P (nnth nell i)"
using assms 
by transfer
   (auto split: if_split_asm,
    metis diff_Suc_1 eSuc_enat eSuc_epred iless_Suc_eq kfilter_n_not_after llength_eq_0  not_gr_zero )

lemma nkfilter_not_between:
 assumes " (\<exists> x \<in> nset nell. P x) " 
         " nnth (nkfilter P 0 nell) k < i "
         " i < nnth (nkfilter P 0 nell) (Suc k)"
         " (Suc k) \<le> nlength (nkfilter P 0 nell)"
 shows   " \<not> P (nnth nell i) "
using assms
by transfer
   (auto simp add: min_def split: if_split_asm,
    metis co.enat.exhaust_sel iless_Suc_eq kfilter_not_between llength_eq_0,
    metis co.enat.exhaust_sel gen_llength_def iless_Suc_eq kfilter_upperbound le_less_trans 
    llength_code llength_eq_0 min.cobounded2 min.strict_order_iff min_enat_simps(1),
    meson Suc_ile_eq less_imp_le,
    meson Suc_ile_eq dual_order.strict_implies_order)


lemma ntaken_nset:
 assumes "k\<le> nlength nell"
 shows   " nset (ntaken k nell) = { (nnth nell i) | i. i \<le> k}"
using assms 
by (auto simp add: in_nset_conv_nnth)
   (metis min.orderE ntaken_nnth,
    metis min.orderE ntaken_nnth)
  

lemma ndropn_nset:
 assumes "k\<le> nlength nell"
 shows   " nset (ndropn k nell) = { (nnth nell i) | i. k \<le> i \<and> i \<le> nlength nell}"
using assms
by (auto simp add: in_nset_conv_nnth)
   (metis add.commute enat_min_eq le_add1 plus_enat_simps(1),
    metis enat_minus_mono1 idiff_enat_enat le_add_diff_inverse)


lemma nfilter_nkfilter_ntaken_nidx_a:
 assumes "\<exists> x \<in> nset nell. P x" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (ntaken k (nkfilter P n nell)) "
using assms 
by (simp add: nidx_expand nidx_nkfilter_gr ntaken_nnth)

lemma nfilter_nkfilter_ndropn_nidx_a:
 assumes "\<exists> x \<in> nset nell. P x" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (ndropn k (nkfilter P n nell)) "
using assms
by transfer
   (auto simp add: kfilter_lnull_conv,
    metis (no_types, lifting) gr_implies_not_zero kfilter_llength leI lfilter_kfilter_ldropn_lidx_a 
    lidx_def llength_eq_0 lnull_ldropn)

lemma nfilter_nkfilter_ntaken_nidx_b:
 assumes "P (nnth nell ( (nnth (nkfilter P 0 nell) k)))" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell)) "
 using assms nidx_nkfilter[of "(ntaken (nnth (nkfilter P 0 nell) k) nell)" P 0]
 by (metis nfinite_ntaken nset_nlast ntaken_nlast) 
 
lemma nfilter_nkfilter_ntaken_nidx_b_1:
 assumes "\<exists> x \<in> nset nell. P x" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell)) "
 using assms nfilter_nkfilter_ntaken_nidx_b[of P nell k]  nkfilter_holds[of nell P]  
by (metis in_nset_conv_nnth nkfilter_nlength nkfilter_nnth_n_zero)

lemma nfilter_nkfilter_ntaken_nidx_b_2:
 assumes "P (nnth nell ( (nnth (nkfilter P n nell) k) -n))" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (nkfilter P n (ntaken (nnth (nkfilter P n nell) k) nell)) "
 using assms nidx_nkfilter[of "(ntaken (nnth (nkfilter P n nell) k) nell)" P n]
by (metis (mono_tags, lifting) diff_le_self linorder_le_cases mem_Collect_eq 
    nfilter_nkfilter_ntaken_nlength_0 ntaken_all ntaken_nset)

 
lemma nfilter_nkfilter_ntaken_nidx_b_3:
 assumes "\<exists> x \<in> nset nell. P x" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (nkfilter P n (ntaken (nnth (nkfilter P n nell) k) nell)) "
using assms nfilter_nkfilter_ntaken_nidx_b_2[of P nell n k] nkfilter_holds
by (metis in_nset_conv_nnth nkfilter_nlength)

lemma nfilter_nkfilter_ndropn_nidx_b:
 assumes "P (nnth nell ( (nnth (nkfilter P 0 nell) k)))" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) k) nell)) "
using assms nidx_nkfilter[of "(ndropn (nnth (nkfilter P 0 nell) k) nell)" P 0]
by (metis diff_add diff_zero in_nset_conv_nnth ndropn_nnth zero_enat_def zero_le) 

lemma nfilter_nkfilter_ndropn_nidx_b_1:
 assumes "\<exists> x \<in> nset nell. P x" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) k) nell)) "
 using assms nfilter_nkfilter_ndropn_nidx_b[of P nell k]  nkfilter_holds[of nell P] 
by (metis in_nset_conv_nnth nkfilter_nlength nkfilter_nnth_n_zero)

lemma nfilter_nkfilter_ndropn_nidx_b_2:
 assumes "P (nnth nell ( (nnth (nkfilter P n nell) k) -n))" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (nkfilter P ((nnth (nkfilter P n nell) k)-n) 
                            (ndropn ((nnth (nkfilter P n nell) k)-n) nell) ) "
proof -
 have 1: "enat (nnth (nkfilter P n nell) k) -n \<le> nlength nell"
   using nkfilter_upperbound[of nell P]  nkfilter_nnth_n_zero[of nell P k n]  assms
   by (metis gen_nlength_def idiff_enat_enat nfilter_nkfilter_ntaken_nlength_0 
       nkfilter_nlength nlength_code) 
 have 2: "nset (ndropn ((nnth (nkfilter P n nell) k)-n) nell) =
            { (nnth nell i) |i. ((nnth (nkfilter P n nell) k)-n) \<le> i \<and> i \<le> nlength nell } " 
   using ndropn_nset[of "(nnth (nkfilter P n nell) k)-n" nell]  1 by simp
 have 3: "\<exists>x\<in>nset (ndropn ((nnth (nkfilter P n nell) k)-n) nell). P x" 
   using "1" "2" assms(1) by auto
 show ?thesis using 3 
   nidx_nkfilter[of "(ndropn ((nnth (nkfilter P n nell) k)-n) nell)" P "(nnth (nkfilter P n nell) k)-n"]
   by blast
qed

lemma nfilter_nkfilter_ndropn_nidx_b_3:
 assumes "\<exists> x \<in> nset nell. P x" 
         " k \<le> nlength (nfilter P nell)" 
 shows   " nidx (nkfilter P ((nnth (nkfilter P n nell) k)-n) 
                            (ndropn ((nnth (nkfilter P n nell) k)-n) nell)) "
using assms nfilter_nkfilter_ndropn_nidx_b_2 
by (metis in_nset_conv_nnth nkfilter_holds nkfilter_nlength)

lemma ntaken_nkfilter_ntaken_nset_eq:
 assumes "P (nnth nell ( (nnth (nkfilter P 0 nell) k)) )" 
         "k \<le> nlength (nfilter P nell) "
 shows   " nset(ntaken k (nkfilter P 0 nell)) =
           nset(nkfilter P 0 (ntaken ((nnth (nkfilter P 0 nell) k)) nell)) "
proof -
 have 1: "( (nnth (nkfilter P 0 nell) k)) \<le> nlength nell "
   using assms nkfilter_upperbound[of nell P k 0]
   by (metis diff_zero gen_nlength_def nfilter_nkfilter_ntaken_nlength_0 nkfilter_nlength nlength_code) 
 have 2: "\<exists> x \<in> nset nell. P x"
   using assms by (metis "1" exists_Pred_nnth_nset)
 have 3: "{i. i \<le> nlength(ntaken (nnth (nkfilter P 0 nell) k) nell) \<and> 
            P (nnth (ntaken (nnth (nkfilter P 0 nell) k) nell) i) }
           = {i. i \<le> (nnth (nkfilter P 0 nell) k) \<and> P (nnth nell i)}  "
   by (auto simp add: ntaken_nnth "1" order_subst2)      
 have 4: "\<exists> x \<in> nset(ntaken (nnth (nkfilter P 0 nell) k) nell). P x "
   using "1" assms(1) ntaken_nset by fastforce 
 have 5: "{i. i \<le> (nnth (nkfilter P 0 nell) k) \<and> P (nnth nell i)} =
          {i. i \<le> (nnth (nkfilter P 0 nell) k) \<and> i \<in> nset(nkfilter P 0 nell)}"
   using 4 1 2 nkfilter_holds_b
   by (metis (mono_tags, opaque_lifting) add_cancel_left_right enat_ord_simps(1) order.trans) 
 have 6: "{i. i \<le> (nnth (nkfilter P 0 nell) k) \<and> i \<in> nset(nkfilter P 0 nell)} =
          {i. i\<le> (nnth (nkfilter P 0 nell) k) \<and> 
            i\<in> { (nnth (nkfilter P 0 nell) j) | j. j \<le> nlength (nkfilter P 0 nell)}}"
   by (simp add: nset_conv_nnth)
 have 7: "{i. i\<le> (nnth (nkfilter P 0 nell) k) \<and> 
            i\<in> { (nnth (nkfilter P 0 nell) j) | j. j \<le> nlength (nkfilter P 0 nell)}} =
          {(nnth (nkfilter P 0 nell) j) |j. j \<le> k} "
   by (auto simp add: assms "2" nidx_nkfilter_less_eq nkfilter_nlength)
      (metis "2" dual_order.antisym le_cases nidx_nkfilter_gr_eq nkfilter_nlength,
      metis assms(2) dual_order.trans enat_ord_simps(1))
 have 8: "k \<le> nlength (nkfilter P 0 nell)"
   by (simp add: "2" assms(2) nkfilter_nlength) 
 have 9: "{(nnth (nkfilter P 0 nell) j) |j. j \<le> k} = nset(ntaken k (nkfilter P 0 nell))"
   using ntaken_nset[of k "(nkfilter P 0 nell)"] 8 by auto
 have 91: "min (enat (nnth (nkfilter P 0 nell) k)) (nlength nell) = (nnth (nkfilter P 0 nell) k)"
   by (simp add: "1" min_def)
 have 10: "nset(nkfilter P 0 (ntaken ((nnth (nkfilter P 0 nell) k)) nell)) =
           {i. i \<le> (nnth (nkfilter P 0 nell) k) \<and> 
               P (nnth (ntaken (nnth (nkfilter P 0 nell) k) nell) i) } "
   using nkfilter_nset[of "(ntaken (nnth (nkfilter P 0 nell) k) nell)" P 0 ]
   1 4 91 ntaken_nlength[of "(nnth (nkfilter P 0 nell) k)" nell]
   by auto      
 have 11: "nlength((ntaken ((nnth (nkfilter P 0 nell) k)) nell)) = (nnth (nkfilter P 0 nell) k) "
   by (simp add: "1") 
 have 12: "{i. i \<le> (nnth (nkfilter P 0 nell) k) \<and> 
               P (nnth (ntaken (nnth (nkfilter P 0 nell) k) nell) i) } =
           {i. i \<le> nlength((ntaken ((nnth (nkfilter P 0 nell) k)) nell)) \<and> 
               P (nnth (ntaken (nnth (nkfilter P 0 nell) k) nell) i) }" 
   using 11 by auto
 show ?thesis 
 using "10" "12" "3" "5" "6" "7" "9" by auto
qed

lemma ntaken_nkfilter_ntaken_nset_eq_1:
 assumes "\<exists> x \<in> nset nell. P x" 
         "k \<le> nlength (nfilter P nell) "
 shows   " nset(ntaken k (nkfilter P 0 nell)) =
           nset(nkfilter P 0 (ntaken ((nnth (nkfilter P 0 nell) k)) nell)) "
using assms ntaken_nkfilter_ntaken_nset_eq[of P nell k]
nkfilter_holds[of nell P "(nnth (nkfilter P 0 nell) k)" 0]  
by (metis in_nset_conv_nnth nkfilter_nlength nkfilter_nnth_n_zero) 
       
lemma ndropn_nkfilter_ndropn_nset_eq:
 assumes "P (nnth nell ( (nnth (nkfilter P 0 nell) k)) )" 
         "k \<le> nlength (nfilter P nell) "
 shows   " nset(ndropn k (nkfilter P 0 nell)) =
           nset(nkfilter P (nnth (nkfilter P 0 nell) k) (ndropn ((nnth (nkfilter P 0 nell) k)) nell)) "
proof -
 have 1: "(nnth (nkfilter P 0 nell) k) \<le> nlength nell"
   using nkfilter_upperbound[of nell P k 0]  assms
   by (metis diff_zero gen_nlength_def nfilter_nkfilter_ntaken_nlength_0 nkfilter_nlength nlength_code) 
 have 2: " \<exists> x \<in> nset nell. P x"
   using assms by (metis "1" exists_Pred_nnth_nset)
 have 4: "\<exists> x \<in> nset(ndropn (nnth (nkfilter P 0 nell) k) nell). P x" 
   using "1" assms(1) ndropn_nset by fastforce 
 have 10: "nset(nkfilter P (nnth (nkfilter P 0 nell) k) 
                            (ndropn ((nnth (nkfilter P 0 nell) k)) nell)) =
           {(nnth (nkfilter P 0 nell) k) + i| i. i \<le> nlength nell - (nnth (nkfilter P 0 nell) k) \<and> 
               P (nnth nell (i+ (nnth (nkfilter P 0 nell) k)))  }" 
   using 1 
   nkfilter_ndropn_nset_b[of "(nnth (nkfilter P 0 nell) k)" nell P "(nnth (nkfilter P 0 nell) k)"]
   "4" by linarith 
 have 5: "{(nnth (nkfilter P 0 nell) k) + i| i. i \<le> nlength nell - (nnth (nkfilter P 0 nell) k) \<and> 
               P (nnth nell (i+ (nnth (nkfilter P 0 nell) k)))  } =
          {(nnth (nkfilter P 0 nell) k) + i| i. i \<le> nlength nell - (nnth (nkfilter P 0 nell) k) \<and> 
             (i+ (nnth (nkfilter P 0 nell) k)) \<in> nset(nkfilter P 0 nell)}"
    proof (auto simp add: add.commute) 
     fix i :: nat
     assume a1: "enat i \<le> nlength nell - enat (nnth (nkfilter P 0 nell) k)"
     assume a2: "P (nnth nell (i + nnth (nkfilter P 0 nell) k))"
     have f0: "enat (i + (nnth (nkfilter P 0 nell) k)) \<le> nlength nell" 
     using "1" a1 enat_min_eq by auto
     show "i + nnth (nkfilter P 0 nell) k \<in> nset (nkfilter P 0 nell)"
     by (metis Nat.add_0_right a2 f0 in_nset_conv_nnth nkfilter_holds_b) 
    next 
     fix i
     assume b0: "enat i \<le> nlength nell - enat (nnth (nkfilter P 0 nell) k) "
     assume b1: "i + nnth (nkfilter P 0 nell) k \<in> nset (nkfilter P 0 nell)"
     show " P (nnth nell (i + nnth (nkfilter P 0 nell) k)) "
     using nkfilter_holds[of nell P ] 2 by (metis b1 minus_nat.diff_0)
   qed
 have 51: "{(nnth (nkfilter P 0 nell) k) + i| i. i \<le> nlength nell - (nnth (nkfilter P 0 nell) k) \<and> 
             (i+ (nnth (nkfilter P 0 nell) k)) \<in> nset(nkfilter P 0 nell)} =
           {(nnth (nkfilter P 0 nell) k) + i| i. 
             (nnth (nkfilter P 0 nell) k) \<le> i + (nnth (nkfilter P 0 nell) k) \<and> 
             i + (nnth (nkfilter P 0 nell) k) \<le> nlength nell  \<and> 
             (i+ (nnth (nkfilter P 0 nell) k)) \<in> nset(nkfilter P 0 nell)}" 
   by auto
      (metis "2" add.left_neutral in_nset_conv_nnth nkfilter_upperbound zero_enat_def,
       metis add_diff_cancel_right' enat_minus_mono1 idiff_enat_enat)
 have 52: "{(nnth (nkfilter P 0 nell) k) + i| i. 
             (nnth (nkfilter P 0 nell) k) \<le> i + (nnth (nkfilter P 0 nell) k) \<and> 
             i + (nnth (nkfilter P 0 nell) k) \<le> nlength nell  \<and> 
             (i+ (nnth (nkfilter P 0 nell) k)) \<in> nset(nkfilter P 0 nell)} =
           {j. (nnth (nkfilter P 0 nell) k) \<le>j \<and> j \<le> nlength nell \<and> j \<in> nset(nkfilter P 0 nell)} " 
   by (metis (no_types, lifting) add.commute diff_add)
 have 53 : "{j. (nnth (nkfilter P 0 nell) k) \<le>j \<and> j \<le> nlength nell \<and> j \<in> nset(nkfilter P 0 nell)} =
            {j. (nnth (nkfilter P 0 nell) k) \<le>j \<and> j \<le> nlength nell \<and> j \<in> 
                {(nnth (nkfilter P 0 nell) jj) |jj. jj \<le> nlength (nkfilter P 0 nell)}}"
   by (simp add: nset_conv_nnth)
 have 54: "{j. (nnth (nkfilter P 0 nell) k) \<le>j \<and> j \<le> nlength nell \<and> j \<in> 
                {(nnth (nkfilter P 0 nell) jj) |jj. jj \<le> nlength (nkfilter P 0 nell)}} =
           {(nnth (nkfilter P 0 nell) j) |j. k \<le> j \<and> j \<le> nlength(nkfilter P 0 nell)} "
   using assms 2 
   by (auto,
       metis dual_order.antisym le_cases nidx_less_eq nidx_nkfilter nkfilter_nlength,
       meson nidx_nkfilter_gr_eq,
       metis gen_nlength_def nkfilter_upperbound nlength_code)      
 have 8: " k \<le> nlength (nkfilter P 0 nell)" 
    by (simp add: "2" assms(2) nkfilter_nlength) 
 have 9: "{(nnth (nkfilter P 0 nell) j) |j. k \<le> j \<and> j \<le> nlength(nkfilter P 0 nell)} =
          nset(ndropn k (nkfilter P 0 nell))"
   by (simp add: "8" ndropn_nset) 
 show ?thesis 
   using "10" "5" "51" "52" "53" "54" "9" by auto
qed

lemma ndropn_nkfilter_ndropn_nset_eq_1:
 assumes "\<exists> x \<in> nset nell. P x" 
         "k \<le> nlength (nfilter P nell) "
 shows   " nset(ndropn k (nkfilter P 0 nell)) =
           nset(nkfilter P (nnth (nkfilter P 0 nell) k) (ndropn ((nnth (nkfilter P 0 nell) k)) nell)) "
using assms ndropn_nkfilter_ndropn_nset_eq[of P nell k]
by (metis diff_zero in_nset_conv_nnth nkfilter_holds nkfilter_nlength) 
 
lemma nkfilter_nkfilter_ntaken:
 assumes "\<exists> x \<in> nset nell. P x" 
         "k \<le> nlength (nfilter P nell) "
 shows   " ntaken k (nkfilter P 0 nell) =
           (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell)) "
using assms 
by (simp add: nfilter_nkfilter_ntaken_nidx_a nfilter_nkfilter_ntaken_nidx_b_1 nidx_nset_eq 
    ntaken_nkfilter_ntaken_nset_eq_1)

lemma nkfilter_nkfilter_ndropn:
 assumes "\<exists> x \<in> nset nell. P x" 
         "k \<le> nlength (nfilter P nell) "
 shows   "ndropn k (nkfilter P 0 nell) =
          (nkfilter P (nnth (nkfilter P 0 nell) k) (ndropn (nnth (nkfilter P 0 nell) k) nell))"
proof -
 have 1: "P (nnth nell ( (nnth (nkfilter P 0 nell) k)) ) "
   using nkfilter_holds[of nell P ] assms
   by (metis diff_zero in_nset_conv_nnth nkfilter_nlength)
 show ?thesis
   by (metis "1" assms(1) assms(2) diff_zero ndropn_nkfilter_ndropn_nset_eq 
       nfilter_nkfilter_ndropn_nidx_a nfilter_nkfilter_ndropn_nidx_b_3 nidx_nset_eq) 
qed



lemma nkfilter_nmap_nfilter:
 assumes "\<exists> x \<in> nset nell. P x" 
 shows   "nmap (\<lambda>n. nnth nell n) (nkfilter P 0 nell) = nfilter P nell"
 using assms nellist_eq_nnth_eq[of "nmap (\<lambda>n. nnth nell n) (nkfilter P 0 nell)" "nfilter P nell"]
nkfilter_nfilter[of nell P _ 0]  nkfilter_nnth_n_zero[of nell P _ 0] 
by (simp add: nkfilter_nlength) 

lemma nfilter_nkfilter_ntaken:
 assumes "P (nnth nell ( (nnth (nkfilter P 0 nell) k)) )"
         "k \<le> nlength (nkfilter P 0 nell)"  
 shows   " ntaken k (nfilter P nell) = 
           nfilter P (ntaken (nnth (nkfilter P 0 nell) k) nell) "
proof -
 have 1: "\<exists> x \<in> nset nell. P x"
   using assms 
   by (metis in_nset_conv_nnth min.cobounded1 min_def nfinite_ntaken nset_nlast ntaken_all
       ntaken_nlast)
 have 2: " nfilter P nell = nmap (\<lambda>n. nnth nell n) (nkfilter P 0 nell) "
   using 1  assms by (simp add: nkfilter_nmap_nfilter)
 have 3: "ntaken k (nfilter P nell) = ntaken k (nmap (\<lambda>n. nnth nell n) (nkfilter P 0 nell))" 
   using 2 by simp
 have 4: "ntaken k (nmap (\<lambda>n. nnth nell n) (nkfilter P 0 nell)) =
          nmap (\<lambda>n. nnth nell n) (ntaken k (nkfilter P 0 nell))"
   by simp
 have 5: "\<exists> x \<in> nset (ntaken (nnth (nkfilter P 0 nell) k) nell). P x "
   using assms by (metis nfinite_ntaken nset_nlast ntaken_nlast) 
 have 6: "(nfilter P (ntaken (nnth (nkfilter P 0 nell) k) nell)) =
          nmap (\<lambda>s. nnth (ntaken (nnth (nkfilter P 0 nell) k) nell) s) 
               (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell))  " 
   using 5 by (simp add: nkfilter_nmap_nfilter)
 have 7: "nmap (\<lambda>n. nnth nell n) (ntaken k (nkfilter P 0 nell)) =
          nmap (\<lambda>n. nnth nell n) (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell))"
   by (simp add: "1" "2" assms(2) nkfilter_nkfilter_ntaken) 
 have 70: "enat k \<le> nlength (nfilter P nell)"
   using "2" assms by auto
 have 71: "(\<And>z. z \<in> nset (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell)) \<Longrightarrow>
       (\<lambda>n. nnth nell n) z = (\<lambda>s. nnth (ntaken (nnth (nkfilter P 0 nell) k) nell) s)  z)"
   using assms 1 70 ntaken_nkfilter_ntaken_nset_eq[of P nell k] 
        ntaken_nset[of k "(nkfilter P 0 nell)"] mem_Collect_eq
     nidx_nkfilter_gr_eq[of nell P _ _ 0]
   proof simp
     fix z :: nat
     assume a1: "enat k \<le> nlength (nkfilter P 0 nell)"
     assume a2: "\<And>k j. \<lbrakk>k \<le> j; enat j \<le> nlength (nkfilter P 0 nell)\<rbrakk> \<Longrightarrow> 
                        nnth (nkfilter P 0 nell) k \<le> nnth (nkfilter P 0 nell) j"
     assume a3: "z \<in> nset (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell))"
     assume "{nnth (nkfilter P 0 nell) i |i. i \<le> k} = 
              nset (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell))"
     then have "\<exists>n. z = nnth (nkfilter P 0 nell) n \<and> n \<le> k"
     using a3 by blast
     then have "z \<le> nnth (nkfilter P 0 nell) k"
     using a2 a1 by meson
     then show "nnth nell z = nnth (ntaken (nnth (nkfilter P 0 nell) k) nell) z"
     by (simp add: ntaken_nnth)
   qed
 have 8: "nmap (\<lambda>n. nnth nell n) 
               (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell)) =
          nmap (\<lambda>s. nnth (ntaken (nnth (nkfilter P 0 nell) k) nell) s) 
               (nkfilter P 0 (ntaken (nnth (nkfilter P 0 nell) k) nell))"
   using 1 5 assms nellist.map_cong0
   using "71" by blast 
 show ?thesis 
 by (simp add: "3" "6" "7" "8")
qed

lemma nfilter_nkfilter_ntaken_1:
 assumes "\<exists> x \<in> nset nell. P x"
         "k \<le> nlength (nkfilter P 0 nell)"  
 shows   " ntaken k (nfilter P nell) = 
           nfilter P (ntaken (nnth (nkfilter P 0 nell) k) nell) "
using assms nfilter_nkfilter_ntaken[of P nell k]  
by (metis in_nset_conv_nnth nkfilter_holds nkfilter_nnth_n_zero)

lemma nkfilter_nmap_shift:
 assumes "\<exists> x \<in> nset nell. P x"
 shows " nmap (\<lambda>s. nnth nell (s+n)) (nkfilter P 0 nell) =
         nmap (\<lambda>s. nnth nell s) (nkfilter P n nell)"
proof -
 have 1: "nlength (nmap (\<lambda>s. nnth nell (s+n)) (nkfilter P 0 nell)) =
          nlength (nmap (\<lambda>s. nnth nell s) (nkfilter P n nell))"
   by (simp add: assms nkfilter_nlength) 
 have 2: "\<And>i. i\<le> nlength (nmap (\<lambda>s. nnth nell (s+n)) (nkfilter P 0 nell)) \<longrightarrow>
              nnth (nmap (\<lambda>s. nnth nell (s+n)) (nkfilter P 0 nell)) i =
              nnth nell ((nnth (nkfilter P 0 nell) i)+n)"
   by simp
 have 3: "\<And>i. i\<le> nlength (nmap (\<lambda>s. nnth nell s) (nkfilter P n nell)) \<longrightarrow>
              nnth (nmap (\<lambda>s. nnth nell s) (nkfilter P n nell)) i =
              nnth nell (nnth (nkfilter P n nell) i)"
   by simp
 have 4: " \<And>i. i\<le> nlength (nmap (\<lambda>s. nnth nell (s+n)) (nkfilter P 0 nell)) \<longrightarrow>
              nnth nell ((nnth (nkfilter P 0 nell) i)+n) = nnth nell (nnth (nkfilter P n nell) i)"
   using nkfilter_n_zero[of nell P n]
   by (simp add: assms)
 show ?thesis using "1" "4" nellist_eq_nnth_eq by force
qed

lemma nkfilter_nmap_shift_ndropn:
 assumes "\<exists> x \<in> nset (ndropn (nnth (nkfilter P 0 nell) k) nell). P x"
         "k \<le> nlength (nkfilter P 0 nell)" 
 shows " nmap (\<lambda>s. nnth nell (s+(nnth (nkfilter P 0 nell) k))) 
              (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) k) nell)) =
         nmap (\<lambda>s. nnth nell s) (nkfilter P (nnth (nkfilter P 0 nell) k) 
                                            (ndropn (nnth (nkfilter P 0 nell) k) nell))"
using assms 
      nellist_eq_nnth_eq[of "nmap (\<lambda>s. nnth nell (s+(nnth (nkfilter P 0 nell) k))) 
                                  (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) k) nell))" 
                            "nmap (\<lambda>s. nnth nell s) (nkfilter P (nnth (nkfilter P 0 nell) k) 
                                  (ndropn (nnth (nkfilter P 0 nell) k) nell))"]  
using nkfilter_n_zero[of "(ndropn (nnth (nkfilter P 0 nell) k) nell)" P 
                          "(nnth (nkfilter P 0 nell) k)"]
by simp 

lemma nfilter_nkfilter_ndropn:
 assumes "P (nnth nell ( (nnth (nkfilter P 0 nell) k)) )"
         "k \<le> nlength (nkfilter P 0 nell)"  
 shows   " ndropn k (nfilter P nell) = 
           nfilter P (ndropn (nnth (nkfilter P 0 nell) k) nell) "
proof -
 have 1: "\<exists>x \<in> nset nell. P x"
   using assms 
   by (metis in_nset_conv_nnth min.cobounded1 min_def nfinite_ntaken nset_nlast ntaken_all ntaken_nlast)
 have 2: "(nfilter P nell) = nmap (\<lambda>n. nnth nell n) (nkfilter P 0 nell)" 
   by (simp add: "1" nkfilter_nmap_nfilter)
 have 3: "ndropn k (nfilter P nell) = ndropn k (nmap (\<lambda>n. nnth nell n) (nkfilter P 0 nell))"
   by (simp add: "2") 
 have 4: "ndropn k (nmap (\<lambda>n. nnth nell n) (nkfilter P 0 nell)) =
          nmap (\<lambda>n. nnth nell n) (ndropn k (nkfilter P 0 nell)) "
   using ndropn_nmap by blast
 have 5: "\<exists> x \<in> nset(ndropn (nnth (nkfilter P 0 nell) k) nell). P x"
   by (metis add.commute add.left_neutral assms(1) in_nset_conv_nnth ndropn_nnth zero_enat_def 
       zero_le) 
 have 6: "nfilter P (ndropn (nnth (nkfilter P 0 nell) k) nell) =
          nmap (\<lambda>s. nnth (ndropn (nnth (nkfilter P 0 nell) k) nell) s)
               (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) k ) nell))" 
   by (metis "5" nkfilter_nmap_nfilter)
 have 7: "nmap (\<lambda>s. nnth (ndropn (nnth (nkfilter P 0 nell) k) nell) s)
               (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) k ) nell)) =
          nmap (\<lambda>s. nnth nell (s+(nnth (nkfilter P 0 nell) k)))
               (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) k ) nell))" 
   by (simp add: add.commute)
 have 8: "nmap (\<lambda>s. nnth nell (s+(nnth (nkfilter P 0 nell) k)))
               (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) k ) nell)) =
          nmap (\<lambda>s. nnth nell s) 
               (nkfilter P (nnth (nkfilter P 0 nell) k) (ndropn (nnth (nkfilter P 0 nell) k) nell))"
   using "5" assms(2) nkfilter_nmap_shift_ndropn by fastforce 
 have 9: "nmap (\<lambda>s. nnth nell s) 
               (nkfilter P (nnth (nkfilter P 0 nell) k) (ndropn (nnth (nkfilter P 0 nell) k) nell)) =
          nmap (\<lambda>s. nnth nell s) 
               (ndropn k (nkfilter P 0 nell))"
   by (simp add: "1" "2" assms(2) nkfilter_nkfilter_ndropn) 
 show ?thesis using "3" "4" "6" "7" "8" "9" by auto
qed

lemma nfilter_nkfilter_ndropn_1:
 assumes "\<exists>x \<in> nset nell. P x"
         "k \<le> nlength (nkfilter P 0 nell)"  
 shows   " ndropn k (nfilter P nell) = 
           nfilter P (ndropn (nnth (nkfilter P 0 nell) k) nell) "
using assms nfilter_nkfilter_ndropn 
by (metis in_nset_conv_nnth minus_nat.diff_0 nkfilter_holds)



lemma nfilter_nkfilter_nsubn: 
assumes "P (nnth nell ( (nnth (nkfilter P 0 nell) i)) ) "
        "enat i \<le> nlength (nkfilter P 0 nell)"  
        "P (nnth nell ( (nnth (nkfilter P 0 nell) j)) ) "
        "enat j \<le> nlength (nkfilter P 0 nell)"  
        " i \<le> j" 
shows "nsubn (nfilter P nell ) i j =
       (nfilter P (nsubn nell 
                         (nnth (nkfilter P 0 nell) i)
                         (nnth (nkfilter P 0 nell) j)
                  )
        )
        "  
proof -
 have 0: "nsubn (nfilter P nell ) i j = ntaken (j - i) (ndropn i (nfilter P nell))" 
    using nsubn_def1 by blast
 have 1: "ntaken (j - i) (ndropn i (nfilter P nell)) = 
          ntaken (j - i) (nfilter P (ndropn (nnth (nkfilter P 0 nell) i) nell))"
   by (simp add: assms(1) assms(2) nfilter_nkfilter_ndropn) 
 have 4: "((nnth (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) i) nell)) (j - i)) +
                             (nnth (nkfilter P 0 nell) i)) =
          ((nnth (nkfilter P (nnth (nkfilter P 0 nell) i) (ndropn (nnth (nkfilter P 0 nell) i) nell)) (j - i)) )"
    proof -
    have f1: "\<And>n ns. (nfirst (ndropn n ns)::nat) = nlast (ntaken n ns)"
    by (metis (lifting) ndropn_0 ndropn_nfirst ndropn_nnth ntaken_ndropn_nlast)
    have "\<And>n f ns. nlast (ntaken n (nmap f ns)) = (f (nlast (ntaken n ns)::nat)::nat)"
    by simp
    then show ?thesis
    using f1 by (metis  assms(1) in_nset_conv_nnth ndropn_0 ndropn_nfirst nkfilter_n_zero zero_enat_def zero_le)
    qed
 have 5: "((nnth (nkfilter P (nnth (nkfilter P 0 nell) i) (ndropn (nnth (nkfilter P 0 nell) i) nell)) (j - i)) ) =
          (nnth (nkfilter P 0 nell) j) "
  using assms nkfilter_nkfilter_ndropn[of nell P i] 
     by (metis in_nset_conv_nnth le_add_diff_inverse linorder_le_cases linorder_not_less ndropn_nnth 
         nfinite_ntaken nkfilter_nlength nnth_beyond nset_nlast ntaken_all)
 have 10: "\<exists>x\<in>nset (ndropn (nnth (nkfilter P 0 nell) i) nell). P x" 
    by (metis assms(1) in_nset_conv_nnth le_zero_eq nle_le nnth_zero_ndropn zero_enat_def)
 have 11: "ndropn i (nfilter P nell) = nfilter P (ndropn (nnth (nkfilter P 0 nell) i) nell)" 
    by (simp add: assms(1) assms(2) nfilter_nkfilter_ndropn)
 have 12: "nlength (ndropn i (nfilter P nell)) = nlength (nfilter P (ndropn (nnth (nkfilter P 0 nell) i) nell))"
    by (simp add: "11") 
 have 13: "nlength (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) i) nell)) =
          nlength (nfilter P (ndropn (nnth (nkfilter P 0 nell) i) nell)) " 
   by (simp add: "10" nkfilter_nlength)
 have 14: "enat i \<le> nlength (nfilter P  nell)" 
     by (metis assms(1) assms(2) in_nset_conv_nnth linorder_le_cases nfinite_ntaken nkfilter_nlength nset_nlast 
         ntaken_all ntaken_nlast)
 have 15: "enat j \<le> nlength (nfilter P  nell)" 
    by (metis "14" assms(1) assms(4) diff_zero nfilter_nkfilter_ntaken_nlength_0 nkfilter_nlength)
 have 16: "enat (j - i) \<le> nlength (ndropn i (nfilter P nell))"
     by (metis "15" enat_minus_mono1 idiff_enat_enat ndropn_nlength)
 have 2: "ntaken (j - i) (nfilter P (ndropn (nnth (nkfilter P 0 nell) i) nell)) =
          (nfilter P (ntaken (nnth (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) i) nell)) (j - i))
                   (ndropn (nnth (nkfilter P 0 nell) i) nell)))"
   by (metis "10" "12" "13" "16" nfilter_nkfilter_ntaken_1)
 have 3: "(nfilter P (ntaken (nnth (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) i) nell)) (j - i))
                   (ndropn (nnth (nkfilter P 0 nell) i) nell))) =
          (nfilter P (nsubn nell 
                            (nnth (nkfilter P 0 nell) i)
                            ((nnth (nkfilter P 0 (ndropn (nnth (nkfilter P 0 nell) i) nell)) (j - i)) +
                             (nnth (nkfilter P 0 nell) i))
                            ))   " 
    by (simp add: ntaken_ndropn)
  show ?thesis using 0 1 2 3 4 5 by auto
qed

lemma nfilter_nkfilter_nsubn_zero: 
assumes "P (nnth nell ( (nnth (nkfilter P 0 nell) j)) ) "
        "enat j \<le> nlength (nkfilter P 0 nell)"  
shows "nsubn (nfilter P nell ) 0 j =
       (nfilter P (nsubn nell 
                         0
                         (nnth (nkfilter P 0 nell) j)
                  )
        )
        "  
by (simp add: assms(1) assms(2) nfilter_nkfilter_ntaken nsubn_def1)



lemma nkfilter_nnth_aa:
 assumes "\<exists>x \<in> nset nell. P x"
         " n \<le> nlength (nfilter P nell)"
 shows   "P (nnth (nfilter P nell) n)"
using assms in_nset_conv_nnth nkfilter_holds[of nell P _ 0] nkfilter_nfilter[of nell P _ 0]
by (metis nkfilter_nlength)

lemma nfilter_nlength_zero_conv_a:
 assumes "\<exists>x \<in> nset nell. P x"
         "nlength(nfilter P nell) = 0"
 shows   "(\<exists> k \<le> nlength nell. P (nnth nell k) \<and> 
            (\<forall>j\<le> nlength nell. j \<noteq>k \<longrightarrow> \<not> P (nnth nell j)))"
using assms 
apply transfer
proof (auto simp add: epred_llength min_def split: if_split_asm)
 fix nella :: "'a llist" and Pa :: "'a \<Rightarrow> bool" and x :: 'a
 assume a1: "\<not> lnull nella"
 assume a2: "Pa x"
 assume a3: "x \<in> lset nella"
 assume a4: "lnull (ltl (lfilter Pa nella))"
 show " \<exists>k. (enat k \<le> llength (ltl nella) \<longrightarrow>
            Pa (lnth nella k) \<and> (\<forall>j. enat j \<le> llength (ltl nella) \<longrightarrow> j \<noteq> k \<longrightarrow> \<not> Pa (lnth nella j))) \<and>
           enat k \<le> llength (ltl nella) " 
 proof -
  have 1: "LCons (lhd nella) (ltl nella) = nella"
    by (metis a2 a3 lfilter_LNil llist.disc(1) llist.exhaust_sel lnull_lfilter)
  have 2: "llength nella = eSuc (llength (ltl nella))"
    by (metis "1" llength_LCons)
  have 3: "\<not> lnull (lfilter Pa nella)"
    by (meson a2 a3 lnull_lfilter)  
  show ?thesis
    by (metis "2" "3" a4 iless_Suc_eq lfilter_llength_one_conv_a lhd_LCons_ltl llength_LCons llength_LNil 
        llist.collapse(1) one_eSuc) 
  qed   
qed

lemma nfilter_nlength_zero_conv_c:
 "(\<exists> k \<le> nlength nell. P (nnth nell k) \<and> 
            (\<forall>j\<le> nlength nell. j \<noteq>k \<longrightarrow> \<not> P (nnth nell j))) =
  (\<exists> k \<le> nlength nell. P (nnth nell k) \<and> 
            (\<forall>j\<le> nlength nell. j<k \<or> k<j \<longrightarrow> \<not> P (nnth nell j)))"
using antisym_conv3 by  auto

lemma nfilter_nlength_zero_conv_d:
 " (\<exists> k \<le> nlength nell. P (nnth nell k) \<and> 
            (\<forall>j\<le> nlength nell. j<k \<or> k<j \<longrightarrow> \<not> P (nnth nell j))) \<longleftrightarrow>
   (\<exists> k \<le> nlength nell. P (nnth nell k) \<and> 
          (\<forall>j. j<k  \<longrightarrow> \<not> P (nnth nell j)) \<and>
          (\<forall>j\<le> nlength nell. k<j \<longrightarrow> \<not> P (nnth nell j)))"
by (meson enat_ord_simps(2) le_cases le_less_trans)
    
lemma nfilter_nlength_zero_conv_b:
 assumes "(\<exists> k \<le> nlength nell. P (nnth nell k) \<and> 
            (\<forall>j\<le> nlength nell. j \<noteq>k \<longrightarrow> \<not> P (nnth nell j)))"
 shows   " (\<exists>x \<in> nset nell. P x) \<and> nlength(nfilter P nell) = 0"
using assms 
by  transfer
    (auto split: if_split_asm,
     metis co.enat.exhaust_sel iless_Suc_eq in_lset_conv_lnth llength_eq_0,
     metis co.enat.exhaust_sel co.enat.sel(2) iless_Suc_eq lfilter_llength_one_conv_b llength_eq_0 
     one_eSuc)

lemma nfilter_nlength_zero_conv:
 "((\<exists>x \<in> nset nell. P x) \<and> nlength(nfilter P nell) = 0) \<longleftrightarrow>
   (\<exists> k \<le> nlength nell. P (nnth nell k) \<and> 
            (\<forall>j\<le> nlength nell. j \<noteq>k \<longrightarrow> \<not> P (nnth nell j)))"
using nfilter_nlength_zero_conv_a[of nell P]  nfilter_nlength_zero_conv_b[of nell P]
by blast

lemma nfilter_nlength_zero_conv_1:
 "((\<exists>x \<in> nset nell. P x) \<and> nlength(nfilter P nell) = 0) \<longleftrightarrow> 
  (\<exists> k \<le> nlength nell. P (nnth nell k) \<and> 
            (\<forall>j\<le> nlength nell. j<k \<or> k<j \<longrightarrow> \<not> P (nnth nell j)))"
using nfilter_nlength_zero_conv[of nell P] nfilter_nlength_zero_conv_c[of nell P]
by blast

lemma nfilter_nlength_zero_conv_2:
 "((\<exists>x \<in> nset nell. P x) \<and> nlength(nfilter P nell) = 0) \<longleftrightarrow> 
   (\<exists> k \<le> nlength nell. P (nnth nell k) \<and> 
          (\<forall>j. j<k  \<longrightarrow> \<not> P (nnth nell j)) \<and>
          (\<forall>j\<le> nlength nell. k<j \<longrightarrow> \<not> P (nnth nell j))) "
using nfilter_nlength_zero_conv_1[of nell P] nfilter_nlength_zero_conv_d[of nell P]
by blast

lemma nfilter_ndropns_nmap_help_0:
 assumes "\<exists> x \<in> nset nell. P x"
         " j \<le> nnth(nkfilter P 0 nell) 0 "
 shows   "(nfilter P (ndropn (nnth (nkfilter P 0 nell) 0) nell)) = (nfilter P (ndropn j nell))"
 using assms 
  proof (induction j arbitrary: nell)
  case 0
  then show ?case
  by (metis ndropn_0 nfilter_nkfilter_ndropn_1 zero_enat_def zero_le) 
  next
  case (Suc j)
  then show ?case 
    proof (cases nell)
    case (NNil x1)
    then show ?thesis
    by simp 
    next
    case (NCons x21 x22)
    then show ?thesis 
       proof -
        have 1: "is_NNil x22 \<Longrightarrow> 
                 nfilter P (ndropn (nnth (nkfilter P 0 nell) 0) nell) = nfilter P (ndropn (Suc j) nell) "
          by (metis NCons Suc.prems(2) Suc_le_D ndropn_Suc_NCons ndropn_is_NNil)
        have 2: "P x21 \<and> \<not>is_NNil x22 \<Longrightarrow> 
                 nfilter P (ndropn (nnth (nkfilter P 0 nell) 0) nell) = nfilter P (ndropn (Suc j) nell)" 
          using Suc NCons 
          by simp
             (metis Suc.prems(1) dual_order.strict_trans1 nkfilter_not_before nnth_0 zero_less_Suc)
        have 3: "\<not>P x21 \<and> \<not>is_NNil x22 \<Longrightarrow> 
                 nfilter P (ndropn (nnth (nkfilter P 0 nell) 0) nell) = nfilter P (ndropn (Suc j) nell)" 
          using Suc NCons by (simp add: nkfilter_nleast)
        show ?thesis
          using "1" "2" "3" by blast   
      qed   
    qed
qed

lemma nfilter_nappend_ntaken:
 assumes "\<exists> x \<in> nset (ntaken k nell). P x"
         " k\<le> nlength nell"
 shows "nfilter P (ntaken k nell) = 
        ntaken (the_enat (nlength(nfilter P (ntaken k nell)))) (nfilter P nell)"
using assms
apply transfer
proof auto
 show " \<And>k nell P x.
        \<not> lnull nell \<Longrightarrow>
        enat k \<le> epred (llength nell) \<Longrightarrow>
        x \<in> lset (ltake (enat (Suc k)) nell) \<Longrightarrow>
        P x \<Longrightarrow>
        \<forall>x\<in>lset nell. \<not> P x \<Longrightarrow>
        lfilter P (ltake (enat (Suc k)) nell) =
        ltake (enat (Suc (the_enat (epred (llength (lfilter P (ltake (enat (Suc k)) nell))))))) nell" 
  by (metis dual_order.strict_trans1 in_lset_conv_lnth llength_ltake lprefix_lnthD ltake_is_lprefix 
    min.cobounded2)
next
 fix k 
 fix nell :: "'a llist" 
 fix P
 fix x
 fix xa
 assume a0: "\<not> lnull nell "
 assume a1: "enat k \<le> epred (llength nell) "
 assume a2: "x \<in> lset (ltake (enat (Suc k)) nell) "
 assume a3: "P x "
 assume a4: "xa \<in> lset nell "
 assume a5: "P xa "      
 show "lfilter P (ltake (enat (Suc k)) nell) =
       ltake (enat (Suc (the_enat (epred (llength (lfilter P (ltake (enat (Suc k)) nell))))))) (lfilter P nell)"
 proof (cases "k = epred (llength nell)")
 case True
 then show ?thesis
  proof -
  have f1: "eSuc (enat k) = llength nell"
  by (simp add: True a0)
  then have "llength (lfilter P nell) \<noteq> 0"
  by (metis (no_types) a2 a3 eSuc_enat llength_eq_0 lnull_lfilter ltake_all order_refl)
  then show ?thesis
   using f1 by (metis True co.enat.exhaust_sel eSuc_enat enat_the_enat epred_le_epredI infinity_ileE 
   llength_lfilter_ile ltake_all order_refl)
  qed
 next
 case False
 then show ?thesis
  proof -
  have f1: "llength nell \<noteq> 0"
  using a0 llength_eq_0 by blast
  have g1: "\<not> lnull (lfilter P (ltake (enat (Suc k)) nell))"
  by (meson a2 a3 lnull_lfilter)
  then show ?thesis
  using f1
  proof -
   have f1: "enat k < epred (llength nell)"
   using False a1 order.not_eq_order_implies_strict by blast
   have f2: "\<forall>e. e = 0 \<or> e = eSuc (epred e)"
   by (metis co.enat.exhaust_sel)
   then have g2: "enat (Suc k) < llength nell"
   using f1 by (metis (no_types) Suc_ile_eq \<open>llength nell \<noteq> 0\<close> iless_Suc_eq)
   then show ?thesis
   using f2
   using lfilter_lappend_ltake[of "Suc k"]
   proof -
    have "eSuc (enat k) = min (enat (Suc k)) (llength nell)"
    by (metis \<open>enat (Suc k) < llength nell\<close> eSuc_enat min.strict_order_iff)
    then have "eSuc (enat k) = llength (ltake (enat (Suc k)) nell)"
    by simp
    then show ?thesis
    by (metis g1 g2 co.enat.exhaust_sel eSuc_enat eSuc_infinity enat_the_enat infinity_ileE
        lfilter_lappend_ltake llength_eq_0 llength_lfilter_ile)
   qed 
  qed
  qed
 qed
qed



lemma nappend_nfilter_nfinite:
 assumes "\<exists> x \<in> nset (nellx). P x"
         "\<exists> x \<in> nset (nelly). P x"
         "nfinite nellx" 
 shows   " nfilter P (nappend nellx nelly) = nappend (nfilter P nellx) (nfilter P nelly) "
using assms
by transfer auto

lemma nfilter_nappend_ndropn:
 assumes "\<exists> x \<in> nset (ndropn (Suc k) nell). P x"
         "\<exists> x \<in> nset (ntaken k nell). P x" 
         " (Suc k) \<le> nlength nell"
 shows "nfilter P (ndropn (Suc k) nell) = 
        ndropn (the_enat (eSuc(nlength(nfilter P (ntaken k nell))))) (nfilter P nell)"
proof -
 have 1: "nfinite (nfilter P (ntaken k nell))" 
   by (metis Suc_ile_eq assms(3) dual_order.strict_implies_order enat_ile length_nfilter_le 
       min.absorb1 nfinite_conv_nlength_enat ntaken_nlength)
 have 2: "nfilter P nell = nappend (nfilter P (ntaken k nell)) (nfilter P (ndropn (Suc k) nell)) "
   using assms nappend_nfilter_nfinite[of "(ntaken k nell)" P "(ndropn (Suc k) nell)"] 
         nappend_ntaken_ndropn[of k nell]  nfinite_ntaken[of k nell] by simp
 have 3: "ndropn (the_enat (eSuc(nlength(nfilter P (ntaken k nell))))) 
                   (nappend (nfilter P (ntaken k nell)) (nfilter P (ndropn (Suc k) nell))) 
          =  (nfilter P (ndropn (Suc k) nell))" 
   by (metis "1" antisym_conv2 gr_zeroI ile_eSuc iless_Suc_eq less_imp_le ndropn_0 
        ndropn_nappend3 nfinite_conv_nlength_enat one_enat_def plus_1_eSuc(2) plus_enat_simps(1) 
        the_enat.simps zero_less_diff)
 show ?thesis 
   by (simp add: "2" "3")
qed

lemma nkfilter_nappend_ntaken:
 assumes "\<exists> x \<in> nset (ntaken k nell). P x"
         " k\<le> nlength nell"
 shows "nkfilter P n (ntaken k nell) = 
        ntaken (the_enat (nlength(nkfilter P n (ntaken k nell)))) (nkfilter P n nell)"
using assms
apply transfer
proof (auto simp add: kfilter_not_lnull_conv kfilter_lnull_conv)
 show "\<And>k nell P n x.
       \<not> lnull nell \<Longrightarrow>
       enat k \<le> epred (llength nell) \<Longrightarrow>
       x \<in> lset (ltake (enat (Suc k)) nell) \<Longrightarrow>
       P x \<Longrightarrow>
       \<forall>x\<in>lset nell. \<not> P x \<Longrightarrow>
       kfilter P n (ltake (enat (Suc k)) nell) =
       ltake (enat (Suc (the_enat (epred (llength (kfilter P n (ltake (enat (Suc k)) nell))))))) (iterates Suc n)" 
  by (meson lset_ltake subsetD)
 next
 fix ka :: nat and nella :: "'a llist" and Pa :: "'a \<Rightarrow> bool" and na :: nat and x :: 'a and xa :: 'a
 assume a1: "Pa x"
 assume a2: "Pa xa"
 assume a3: "xa \<in> lset nella"
 assume a4: "x \<in> lset (ltake (enat (Suc ka)) nella)"
 have f5: "\<And>e. e = enat 0 \<or> eSuc (epred e) = e"
 by (metis (no_types) co.enat.exhaust_sel zero_enat_def)
 have f6: "\<And>as. min \<infinity> (llength (as::'a llist)) = llength as"
 by simp
 have f7: "eSuc \<infinity> = \<infinity>"
 by simp
 have f8: "(enat (Suc (the_enat (epred (llength (kfilter Pa na (ltake (enat (Suc ka)) nella))))))) =
          (llength (kfilter Pa na (ltake (enat (Suc ka)) nella)))"
  proof -
   have f1: "\<forall>n. \<not> \<infinity> \<le> enat n"
   by (meson infinity_ileE)
   have "\<not> lnull (kfilter Pa na (ltake (enat (Suc ka)) nella))"
   by (metis (no_types) a1 a4 kfilter_not_lnull_conv)
   then show ?thesis
   using f1
   by (metis co.enat.exhaust_sel dual_order.trans eSuc_enat eSuc_ile_mono enat_the_enat kfilter_llength 
      llength_eq_0 llength_lfilter_ile llength_ltake min.cobounded1)
  qed
 moreover
 { assume "llength nella \<noteq> \<infinity>"
   then have "ltake (enat (Suc ka)) nella = nella \<longrightarrow> 
              ltake (llength (lfilter Pa (ltake (enat (Suc ka)) nella))) (kfilter Pa na nella) = 
              kfilter Pa na (ltake (enat (Suc ka)) nella) \<and>
               epred (llength (lfilter Pa (ltake (enat (Suc ka)) nella))) \<noteq> \<infinity>"
   using f7 f6 f5 a3 a2 by (metis (no_types) kfilter_llength llength_eq_0 llength_lfilter_ile 
    lnull_lfilter ltake_all min.orderE not_le_imp_less not_less_iff_gr_or_eq zero_enat_def)
  moreover
  { assume "ltake (enat (Suc ka)) nella \<noteq> nella"
    then have "enat (Suc ka) < llength nella \<and> min (enat (Suc ka)) (llength nella) \<noteq> \<infinity> \<or> 
               enat (Suc ka) < llength nella \<and> llength (lfilter Pa (ltake (enat (Suc ka)) nella)) \<noteq> eSuc \<infinity>"
    by (metis (no_types) enat_ord_code(4) ltake_all min.orderE not_le_imp_less not_less_iff_gr_or_eq) }
   ultimately have "enat (Suc ka) < llength nella \<and> min (enat (Suc ka)) (llength nella) \<noteq> \<infinity> \<or>  
                    enat (Suc ka) < llength nella \<and> llength (lfilter Pa (ltake (enat (Suc ka)) nella)) \<noteq> eSuc \<infinity> \<or>
                     ltake (llength (lfilter Pa (ltake (enat (Suc ka)) nella))) (kfilter Pa na nella) = 
                     kfilter Pa na (ltake (enat (Suc ka)) nella) \<and> 
                     epred (llength (lfilter Pa (ltake (enat (Suc ka)) nella))) \<noteq> \<infinity>"
   by fastforce }
  ultimately show "kfilter Pa na (ltake (enat (Suc ka)) nella) = 
                   ltake (enat (Suc (the_enat (epred (llength (kfilter Pa na (ltake (enat (Suc ka)) nella)))))))
                         (kfilter Pa na nella)"
  using f6 f5 a4 a1
  kfilter_lappend_ltake[of "(enat (Suc ka))" nella Pa na ]
  by (metis enat_ord_code(4) kfilter_llength)
qed

lemma nfilter_ndropns_nmap_help_1:
 assumes "\<exists> x \<in> nset nell. P x"
         " j \<le> nnth(nkfilter P 0 nell) (Suc 0) "
         " nnth (nkfilter P 0 nell) 0 < j" 
         " (Suc 0) \<le> nlength(nfilter P nell)"
 shows   "(nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc 0)) nell)) =
          (nfilter P (ndropn j nell))"
using assms
proof 
 (induction j arbitrary: nell )
 case 0
 then show ?case
 by blast 
 next
 case (Suc j)
 then show ?case 
     proof (cases nell)
     case (NNil x1)
     then show ?thesis
     by auto 
     next
     case (NCons x21 x22)
     then show ?thesis 
        proof -
         have 1: "is_NNil x22 \<Longrightarrow> 
                  nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc 0)) nell) = 
                  nfilter P (ndropn (Suc j) nell)"
           by (metis NCons Suc.prems(2) Suc_le_D ndropn_Suc_NCons ndropn_is_NNil) 
         have 2: "P x21 \<and> \<not> is_NNil x22 \<Longrightarrow>
                  nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc 0)) nell) = 
                  nfilter P (ndropn (Suc j) nell) " 
           using Suc NCons 
           proof simp
           assume a1: "P x21 \<and> \<not> is_NNil x22"
           assume a2: "enat (Suc 0) \<le> nlength (nfilter P (NCons x21 x22))"
           assume a3: "nell = NCons x21 x22"
           assume a4: "Suc j \<le> nnth (nkfilter P 0 (NCons x21 x22)) (Suc 0)"
           have f5: "\<forall>as p a n. ((\<exists>a. (a::'a) \<in> nset as \<and> p a) \<or> \<not> p a) \<or> 
                     nkfilter p n (NCons a as) = NNil n"
           by (metis (no_types) nkfilter_NCons_a)
           have f6: "\<forall>as p n. (\<forall>a. (a::'a) \<notin> nset as \<or> \<not> p a) \<or> 
                       nlength (nkfilter p n as) = nlength (nfilter p as)"
           by (meson nkfilter_nlength)
           have f7: "\<forall>p as. (\<forall>a. (a::'a) \<notin> nset as \<or> \<not> p a) = (\<forall>a. a \<notin> nset as \<or> \<not> p a)"
           by meson
           have f8: "nlength (nkfilter P 0 (NCons x21 x22)) = 
                          nlength (nfilter P (NCons x21 x22))"
           by (meson a1 nellist.set_intros(2) nkfilter_nlength)
           have f9: "\<forall>p as. (\<forall>a. (a::'a) \<notin> nset as \<or> \<not> p a) = (\<forall>a. a \<notin> nset as \<or> \<not> p a)"
           by meson
           have f10: "(\<exists>a. a \<in> nset x22 \<and> P a) \<longrightarrow> 
                       nkfilter P 0 (NCons x21 x22) = NCons 0 (nkfilter P (Suc 0) x22)"
           using a1 by (simp add: Bex_def_raw)
           then have f11: "(\<exists>a. a \<in> nset x22 \<and> P a) \<longrightarrow> 
                            nnth (nkfilter P (Suc 0) x22) 0 - Suc 0 = nnth (nkfilter P 0 x22) 0"
           using f9 f8 f7 a2 by (metis Suc_ile_eq iless_Suc_eq nkfilter_nnth_n_zero nlength_NCons)
           have f14: "j < nnth (nkfilter P 0 (NCons x21 x22)) (Suc 0)"
           using a4 Suc_le_eq by blast
           have "\<exists>a. a \<in> nset x22 \<and> P a"
           using f8 f5 a2 a1 by (metis Suc_ile_eq gr_implies_not_zero nlength_NNil)
           then have "(\<exists>a. a \<in> nset x22 \<and> P a) \<and> nfilter P (ndropn (nnth (nkfilter P 0 x22) 0) x22) = 
                         nfilter P (ndropn j x22)"
           using f14   f11 a4
           proof -
           have "j \<le> nnth (nkfilter P 0 x22) 0"
           using \<open>\<exists>a. a \<in> nset x22 \<and> P a\<close> f10 f11 f14 by fastforce
           then show ?thesis
           by (simp add: Bex_def_raw \<open>\<exists>a. a \<in> nset x22 \<and> P a\<close> nfilter_ndropns_nmap_help_0)
           qed  
           then show "nfilter P (ndropn (nnth (nkfilter P 0 (NCons x21 x22)) (Suc 0)) (NCons x21 x22)) = 
                      nfilter P (ndropn j x22)"
           using   f11 a4
           by (metis One_nat_def Suc_le_D diff_Suc_1 f10 ndropn_Suc_NCons nnth_Suc_NCons)
        qed 
      have 3: "\<not>P x21 \<and> \<not> is_NNil x22 \<Longrightarrow>
                  nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc 0)) nell) = 
                  nfilter P (ndropn (Suc j) nell) " 
        using Suc NCons
        proof simp
           assume a1: "\<And>nell. \<lbrakk>\<exists>a\<in>nset nell. P a; j \<le> nnth (nkfilter P 0 nell) (Suc 0); 
                                nnth (nkfilter P 0 nell) 0 < j; enat (Suc 0) \<le> nlength (nfilter P nell)\<rbrakk> \<Longrightarrow> 
                               nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc 0)) nell) = 
                               nfilter P (ndropn j nell)"
           assume a2: "enat (Suc 0) \<le> nlength (nfilter P x22)"
           assume a3: "Suc j \<le> nnth (nkfilter P (Suc 0) x22) (Suc 0)"
           assume a4: "\<exists>x\<in>nset x22. P x"
           assume a5: "nnth (nkfilter P (Suc 0) x22) 0 < Suc j"
           have f12: "nnth (nkfilter P 0 x22) (Suc 0) = 
                           ( (nnth (nkfilter P (Suc 0) x22) (Suc 0)) - (Suc 0))"
           using nkfilter_nnth_n_zero[of x22 P "Suc 0" "Suc 0" ]
           by (simp add: a2 a4 nkfilter_nlength)  
           then have f13: "j \<le> nnth (nkfilter P 0 x22) (Suc 0)"
           using a3 by linarith
           have f14: "enat 0 \<le> nlength (nfilter P x22)"
           using a2 by (metis One_nat_def one_enat_def order.trans zero_enat_def zero_le_one)
           have f15: "nlength (nkfilter P (Suc 0) x22) = nlength (nfilter P x22)"
           by (simp add: a4 nkfilter_nlength)
           then have "Suc 0 \<le> nnth (nkfilter P (Suc 0) x22) 0"
           by (simp add: a4 f14 nkfilter_lowerbound)
           then have "Suc (nnth (nkfilter P 0 x22) 0) \<le> j"
           by (metis a4 a5 add_Suc leD nkfilter_nleast not_less_eq_eq)
           then have "nfilter P (ndropn (nnth (nkfilter P 0 x22) (Suc 0)) x22) = 
                      nfilter P (ndropn j x22)"
           by (simp add: Suc.IH Suc_le_lessD a2 a4 f13)
           then show "nfilter P (ndropn (nnth (nkfilter P (Suc 0) x22) (Suc 0)) (NCons x21 x22)) = 
                      nfilter P (ndropn j x22)"
           using ndropn_Suc_NCons 
           by (metis One_nat_def a2 a4 f12 f15 le_add_diff_inverse nkfilter_lowerbound plus_1_eq_Suc)
         qed
       show ?thesis
         using "1" "2" "3" by blast   
     qed
 qed
qed

lemma nfilter_ndropns_nmap_help_j:
 assumes "\<exists> x \<in> nset nell. P x"
         " j \<le> nnth(nkfilter P 0 nell) (Suc i) "
         " nnth (nkfilter P 0 nell) i < j" 
         " (Suc i) \<le> nlength(nfilter P nell)"
 shows   "(nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc i)) nell)) =
          (nfilter P (ndropn j nell))"
using assms
proof (induction j arbitrary: nell i)
case 0
then show ?case
by blast 
next
case (Suc j)
then show ?case 
  proof (cases nell)
  case (NNil x1)
  then show ?thesis
  by simp 
  next
  case (NCons x21 x22)
  then show ?thesis 
     proof -
      have 1: "is_NNil x22 \<Longrightarrow> 
                nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc i)) nell) = 
                nfilter P (ndropn (Suc j) nell) "
        by (metis NCons Suc.prems(2) Suc_le_D ndropn_Suc_NCons ndropn_is_NNil)
      have 2: "\<not> is_NNil x22 \<and> i = 0 \<Longrightarrow>
                nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc i)) nell) = 
                nfilter P (ndropn (Suc j) nell) "
        using Suc nfilter_ndropns_nmap_help_1[of nell P ]  by metis
      have 3: "P x21 \<and> \<not> is_NNil x22 \<and> 0< i \<Longrightarrow>
                nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc i)) nell) = nfilter P (ndropn (Suc j) nell)" 
        using Suc NCons
        proof simp
        assume a1: "\<And>nell i. \<lbrakk>\<exists>a\<in>nset nell. P a; j \<le> nnth (nkfilter P 0 nell) (Suc i); 
                              nnth (nkfilter P 0 nell) i < j; enat (Suc i) \<le> nlength (nfilter P nell)\<rbrakk> \<Longrightarrow> 
                        nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc i)) nell) = 
                        nfilter P (ndropn j nell)"
        assume a2: "P x21 \<and> \<not> is_NNil x22 \<and> 0 < i"
        assume a3: "enat (Suc i) \<le> nlength (nfilter P (NCons x21 x22))"
        assume a4: "nell = NCons x21 x22"
        assume a5: "Suc j \<le> nnth (nkfilter P 0 (NCons x21 x22)) (Suc i)"
        assume a6: "nnth (nkfilter P 0 (NCons x21 x22)) i < Suc j"
        show "nfilter P (ndropn (nnth (nkfilter P 0 (NCons x21 x22)) (Suc i)) (NCons x21 x22)) = 
                   nfilter P (ndropn j x22)"
        proof -
         have f0: "\<exists>a\<in>nset x22. P a"
           by (metis a2 a3 dual_order.antisym enat.inject nat.distinct(1) nfilter_NCons_a 
               nlength_NNil zero_enat_def zero_le) 
         have f1: "nlength (nkfilter P 0 (NCons x21 x22)) = nlength (nfilter P (NCons x21 x22))"
           using a4 by (metis (no_types) Suc(2) nkfilter_nlength)
         have f2: "nkfilter P 0 (NCons x21 x22) = NCons 0 (nkfilter P (Suc 0) x22)"
           by (metis a2 a3 dual_order.antisym enat.inject f1 nat.distinct(1) nkfilter_NCons 
               nkfilter_NCons_a nlength_NNil zero_enat_def zero_le)
         have f3: "nnth (nkfilter P 0 (NCons x21 x22)) (Suc i) =
                   nnth ( (nkfilter P (Suc 0) x22)) (i)"
           by (simp add: f2)
         have f4: "enat i \<le> nlength (nkfilter P (Suc 0) x22) "
           by (metis Suc_ile_eq a3 f1 f2 iless_Suc_eq nlength_NCons)
         have f5: "nnth ( (nkfilter P (Suc 0) x22)) (i) = (Suc 0) +nnth ( (nkfilter P 0 x22)) (i)" 
           using nkfilter_nnth_n_zero[of x22 P i "Suc 0" ] 
           using a5 f0 f3 f4 by linarith
         have f6: "ndropn (nnth (nkfilter P 0 (NCons x21 x22)) (Suc i)) (NCons x21 x22) =
                   ndropn ((Suc 0) +nnth ( (nkfilter P 0 x22)) (i)) (NCons x21 x22)" 
           using f3 f5 by presburger
         have f7: "ndropn ((Suc 0) +nnth ( (nkfilter P 0 x22)) (i)) (NCons x21 x22) =
                   ndropn (nnth ( (nkfilter P 0 x22)) (i)) x22 " 
           using ndropn_Suc_NCons by auto
         have f8: "j \<le> nnth (nkfilter P 0 x22) (i)"
           using a5 f3 f5 by linarith 
         have f11: "nnth (nkfilter P 0 (NCons x21 x22)) i  = nnth ( (nkfilter P (Suc 0) x22)) (i-1)"
           by (metis One_nat_def Suc_pred a2 f2 nnth_Suc_NCons)
         have f12: "enat (i - 1) \<le> nlength (nkfilter P (Suc 0) x22)"
           by (metis One_nat_def Suc_ile_eq Suc_pred a2 f4 less_imp_le)
         have f13: "(Suc 0) \<le> nnth ( (nkfilter P (Suc 0) x22)) (i-1)"
           by (meson f0 f12 nkfilter_lowerbound) 
         have f14: "nnth ( (nkfilter P (Suc 0) x22)) (i-1) = (Suc 0) +nnth ( (nkfilter P 0 x22)) (i-1)"
           using nkfilter_nnth_n_zero[of x22 P "i-1" "Suc 0" ] f12 f0  f13 by (metis le_add_diff_inverse)
         have f9: "nnth (nkfilter P 0 x22) (i-1) < j" 
           using a6 f11 f14 by linarith
         have f10: " i \<le> nlength (nfilter P x22)" 
           by (metis f0 f4 nkfilter_nlength)
         show ?thesis using a1[of x22 "i-1"] 
           by (metis Suc_diff_1 a2 f0 f10 f3 f5 f7 f8 f9) 
       qed
      qed
      have 4: "\<not>P x21 \<and> \<not> is_NNil x22 \<and> 0< i \<Longrightarrow>
                nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc i)) nell) = 
                nfilter P (ndropn (Suc j) nell)" 
        using Suc NCons 
        proof simp
         assume a0: "\<not> P x21 \<and> \<not> is_NNil x22 \<and> 0 < i " 
         assume a2: "\<And>nell i. \<lbrakk>\<exists>a\<in>nset nell. P a; j \<le> nnth (nkfilter P 0 nell) (Suc i); 
                             nnth (nkfilter P 0 nell) i < j; enat (Suc i) \<le> nlength (nfilter P nell)\<rbrakk> \<Longrightarrow>
                      nfilter P (ndropn (nnth (nkfilter P 0 nell) (Suc i)) nell) =
                      nfilter P (ndropn j nell)"
         assume a1: "\<exists>x\<in>nset x22. P x"
         assume a4: "Suc j \<le> nnth (nkfilter P (Suc 0) x22) (Suc i)"
         assume a5: "nnth (nkfilter P (Suc 0) x22) i < Suc j"
         assume a3: "enat (Suc i) \<le> nlength (nfilter P x22)"
         assume a6: "nell = NCons x21 x22" 
         show "nfilter P (ndropn (nnth (nkfilter P (Suc 0) x22) (Suc i)) (NCons x21 x22)) = 
              nfilter P (ndropn j x22)"
         proof - 
          have f1: "nlength (nkfilter P 0 (NCons x21 x22)) = nlength (nfilter P (NCons x21 x22))"
          by (metis NCons Suc.prems(1) nkfilter_nlength)
          have f2: "nkfilter P 0 (NCons x21 x22) =  (nkfilter P (Suc 0) x22)" 
          using NCons Suc.prems(2) a1 a5 by auto
          have f3: "nnth (nkfilter P 0 (NCons x21 x22)) (Suc i) =
                  nnth (nkfilter P (Suc 0) x22) (Suc i)" 
          by (simp add: f2)
          have f4: "enat (Suc i) \<le> nlength (nkfilter P (Suc 0) x22)" 
          using NCons Suc.prems(4) f1 f2 by auto
          have f5: "nnth ( (nkfilter P (Suc 0) x22)) (Suc i) = (Suc 0) +nnth ( (nkfilter P 0 x22)) (Suc i)" 
          by (metis One_nat_def Suc_le_D a1 a4 diff_Suc_1 f4 nkfilter_nnth_n_zero plus_1_eq_Suc)
          have f6: "(ndropn (nnth (nkfilter P (Suc 0) x22) (Suc i)) (NCons x21 x22)) =
                   ndropn ( (Suc 0) +nnth ( (nkfilter P 0 x22)) (Suc i)) (NCons x21 x22)"
          by (simp add: f5) 
          have f7: "ndropn ( (Suc 0) +nnth ( (nkfilter P 0 x22)) (Suc i)) (NCons x21 x22) =
                  ndropn ( nnth ( (nkfilter P 0 x22)) (Suc i)) (x22)" 
          by simp
          have f8: "j \<le> nnth (nkfilter P 0 x22) (Suc i)"
          using a4 f5 by force
          have f11: "nnth (nkfilter P 0 (NCons x21 x22)) i  = nnth ( (nkfilter P (Suc 0) x22)) (i)"
          by (simp add: f2)
          have f12: "enat (i ) \<le> nlength (nkfilter P (Suc 0) x22)"
          using Suc_ile_eq f4 by auto  
          have f13: "(Suc 0) \<le> nnth ( (nkfilter P (Suc 0) x22)) (i)"
          by (simp add: a1 f12 nkfilter_lowerbound)
          have f14: "nnth ( (nkfilter P (Suc 0) x22)) (i) = (Suc 0) +nnth ( (nkfilter P 0 x22)) (i)"
          by (metis a1 f12 f13 le_add_diff_inverse nkfilter_nnth_n_zero)
          have f9: "nnth (nkfilter P 0 x22) (i) < j" 
          using a5 f14 by auto 
          show ?thesis
          using Suc.IH a1 a3 f6 f7 f8 f9 by presburger
      qed
     qed
     show ?thesis
     using "1" "2" "3" "4" by fastforce 
    qed
  qed
qed

lemma nfilter_ndropns_nmap:
 assumes "\<exists> x \<in> nset (ndropns nell). P x"
         "(Suc i) \<le> nlength(nfilter P (ndropns nell))"
 shows " ndropn (Suc i) (nmap (\<lambda>s. nnth s 0) (nfilter P (ndropns nell))) =
         (nmap (\<lambda>s. nnth s 0) 
               (nfilter P (ndropns (ndropn (Suc (nnth (nkfilter P 0 (ndropns nell)) i)) nell)))) "
proof -
 have 1: "ndropn (Suc i) (nmap (\<lambda>s. nnth s 0) (nfilter P (ndropns nell))) =
          nmap (\<lambda>s. nnth s 0) (ndropn (Suc i) (nfilter P (ndropns nell)))"
   by (simp add: ndropn_nmap) 
 have 2: "(Suc (nnth (nkfilter P 0 (ndropns nell)) i)) \<le> nlength (ndropns nell)"
   using assms nkfilter_nkfilter_ntaken[of "(ndropns nell)" P  i] 
   by (metis Suc_ile_eq antisym_conv2 less_imp_le min.orderE nkfilter_nlength not_le_imp_less 
       ntaken_all ntaken_nlength)
 have 3: "(nfilter P (ndropns (ndropn (Suc (nnth (nkfilter P 0 (ndropns nell)) i)) nell))) =
          (nfilter P (ndropn (Suc (nnth (nkfilter P 0 (ndropns nell)) i)) (ndropns nell))) "
   by (simp add: "2" ndropn_ndropns) 
 have 4: "(ndropn (Suc i) (nfilter P (ndropns nell))) =
          (nfilter P (ndropn (nnth (nkfilter P 0 (ndropns nell)) (Suc i)) (ndropns nell))) "
   by (simp add: assms(1) assms(2) nfilter_nkfilter_ndropn_1 nkfilter_nlength)
 have 5: " (Suc (nnth (nkfilter P 0 (ndropns nell)) i)) \<le> (nnth (nkfilter P 0 (ndropns nell)) (Suc i)) " 
   by (simp add: Suc_leI assms(1) assms(2) nidx_nkfilter_expand nkfilter_nlength)
 have 6: " i < nlength(nfilter P (ndropns nell))"
   using Suc_ile_eq assms(2) by blast 
 have 7: "nnth (nkfilter P 0 (ndropns nell)) i < (Suc (nnth (nkfilter P 0 (ndropns nell)) i))"
   by simp
 have 8: "(nfilter P (ndropn (nnth (nkfilter P 0 (ndropns nell)) (Suc i)) (ndropns nell))) =
          (nfilter P (ndropn (Suc (nnth (nkfilter P 0 (ndropns nell)) i)) (ndropns nell)))"
   using 5 6 7 assms
   nfilter_ndropns_nmap_help_j[of "ndropns nell" P "(Suc (nnth (nkfilter P 0 (ndropns nell)) i))" i ]
   by blast   
 show ?thesis 
   using "1" "3" "4" "8" by auto
qed

lemma ndropns_nfilter_nnth_exist_ndropn:
 assumes "\<exists> x \<in> nset (ndropns nell). P x"
         " j \<le> nlength (nfilter P (ndropns nell)) "
 shows " (\<exists>i. enat i \<le> nlength nell \<and> nnth (nfilter P (ndropns nell)) j = ndropn i nell)"
proof -
 have 1: "nnth (nfilter P (ndropns nell)) j \<in> nset (ndropns nell)" 
   using assms in_nset_conv_nnth nfilter_nnth by fastforce
 show ?thesis using "1"using in_nset_ndropns by blast
qed

lemma nfilter_ndropns_nnth_bound:
 assumes "(\<exists> ys \<in> nset (ndropns xs). P ys) "
         " j \<le> nlength (nfilter P (ndropns xs))"
 shows  " nlength ((nnth (nfilter P (ndropns xs)) j)) \<le> nlength xs"
using assms ndropns_nfilter_nnth_exist_ndropn[of xs P j] 
by (metis add.commute enat.simps(3) enat_add_sub_same enat_le_plus_same(1) less_eqE ndropn_nlength) 

lemma ndropns_nfilter_ndropn:
 assumes "(Suc k) \<le> nlength nell"
         "\<exists> x \<in> nset (ndropns (ndropn (Suc k) nell)). P x"
         "\<exists>x\<in>nset (ntaken k (ndropns nell)). P x" 
 shows  "(nfilter P (ndropns (ndropn (Suc k) nell))) =  
         (ndropn (the_enat (eSuc(nlength(nfilter P (ntaken k (ndropns nell)))))) (nfilter P (ndropns nell))) "
using assms 
ndropn_ndropns[of "Suc k" nell] ndropns_nlength[of nell] nfilter_nappend_ndropn[of k "ndropns nell" P]    
by simp

lemma ndropns_nfilter_ndropn_a:
 assumes " k \<le> nlength(nfilter P (ndropns nell))"
         " \<exists> x \<in> nset (ndropns nell). P x"
 shows   " ndropn k (nfilter P (ndropns nell)) =
           nfilter P (ndropns (ndropn (nnth (nkfilter P 0 (ndropns nell)) k) nell)) "
using assms ndropn_ndropns   nfilter_nkfilter_ndropn_1[of "ndropns nell" P k] 
 nkfilter_upperbound[of "ndropns nell" P k 0] 
by (metis (mono_tags, lifting) add.left_neutral nkfilter_nlength zero_enat_def) 

lemma nfilter_nlength_imp:
 assumes " \<exists> x \<in> nset nell. P x \<and> Q x"
 shows   " nlength (nfilter (\<lambda>x. P x \<and> Q x) nell) \<le> nlength (nfilter P nell)"
using assms by transfer (auto simp add: lfilter_nlength_imp epred_le_epredI)

lemma nkfilter_chop: 
assumes "nlast nellx = nfirst nelly"
        " P (nlast nellx)"
        "nfinite nellx" 
shows "nkfilter P k (nfuse nellx nelly) = 
       nfuse (nkfilter P k nellx)  (nkfilter P (k+(the_enat (nlength nellx))) nelly)"
using assms 
proof (auto simp add: nfuse_def1)
show "nlast nellx = nfirst nelly \<Longrightarrow>
    P (nfirst nelly) \<Longrightarrow>
    nfinite nellx \<Longrightarrow>
    is_NNil nelly \<Longrightarrow>
    \<not> is_NNil (nkfilter P (k + the_enat (nlength nellx)) nelly) \<Longrightarrow> 
    nkfilter P k nellx = nappend (nkfilter P k nellx) (ntl (nkfilter P (k + the_enat (nlength nellx)) nelly))"
  by transfer  auto
show "nlast nellx = nfirst nelly \<Longrightarrow>
    P (nfirst nelly) \<Longrightarrow>
    nfinite nellx \<Longrightarrow> \<not> is_NNil nelly \<Longrightarrow> 
    is_NNil (nkfilter P (k + the_enat (nlength nellx)) nelly) \<Longrightarrow> 
    nkfilter P k (nappend nellx (ntl nelly)) = nkfilter P k nellx"  
  apply transfer 
  proof (auto simp add: kfilter_lappend_lfinite lappend_lnull2)
  fix nellxa :: "'a llist" and nellya :: "'a llist" and Pa :: "'a \<Rightarrow> bool" and ka :: nat and b :: nat
  assume a1: "(if lnull (kfilter Pa (ka + the_enat (epred (llength nellxa))) nellya) 
               then iterates Suc (ka + the_enat (epred (llength nellxa)))
               else kfilter Pa (ka + the_enat (epred (llength nellxa))) nellya) = LCons b LNil"
  assume a2: "\<not> lnull nellya"
  assume a3: "Pa (lhd nellya)"
  assume a4: "\<not> lnull (kfilter Pa (ka + the_enat (llength nellxa)) (ltl nellya))"
  have f5: "lnull (LNil::nat llist)"
  using llength_LNil llength_eq_0 by blast
  have f6: "nellya = LCons (lhd nellya) (ltl nellya)"
   using a2 by auto 
  then have "Pa (lhd nellya)"
    using a3 by auto
  then have False
   by (metis a1 a2 a4 eq_LConsD f6 kfilter_code(2) kfilter_lnull_conv llength_LNil llength_eq_0 llist.set_sel(1))
  then show "lappend (kfilter Pa ka nellxa) (kfilter Pa (ka + the_enat (llength nellxa)) (ltl nellya)) = iterates Suc ka"
  by meson
  next
   fix nellxa :: "'b llist" and nellya :: "'b llist" and Pa :: "'b \<Rightarrow> bool" and ka :: nat and b :: nat
   assume a1: "(if lnull (kfilter Pa (ka + the_enat (epred (llength nellxa))) nellya) 
                then iterates Suc (ka + the_enat (epred (llength nellxa)))
                else kfilter Pa (ka + the_enat (epred (llength nellxa))) nellya) = LCons b LNil"
   assume a2: "\<not> lnull nellya"
   assume a3: "Pa (lhd nellya)"
   assume a4: "\<not> lnull (kfilter Pa (ka + the_enat (llength nellxa)) (ltl nellya))"
   have f5: "lnull (LNil::nat llist)"
   using llength_LNil llength_eq_0 by blast
   have f6: "nellya = LCons (lhd nellya) (ltl nellya)"
   using a2 by auto
   then have "Pa (lhd nellya)"
   using a3 by auto
   then have False
   using f6 f5 a4 a1 by (metis kfilter_LCons kfilter_llength_n_zero llength_eq_0 ltl_simps(2) not_lnull_conv)
   then show "lappend (kfilter Pa ka nellxa) (kfilter Pa (ka + the_enat (llength nellxa)) (ltl nellya)) = 
                  kfilter Pa ka nellxa"
    by meson
   qed
next
show "nlast nellx = nfirst nelly \<Longrightarrow>
    P (nfirst nelly) \<Longrightarrow>
    nfinite nellx \<Longrightarrow>
    \<not> is_NNil nelly \<Longrightarrow>
    \<not> is_NNil (nkfilter P (k + the_enat (nlength nellx)) nelly) \<Longrightarrow>
    nkfilter P k (nappend nellx (ntl nelly)) = 
    nappend (nkfilter P k nellx) (ntl (nkfilter P (k + the_enat (nlength nellx)) nelly)) " 
    apply transfer
    proof (auto simp add: lappend_iterates kfilter_lnull_conv split:if_split_asm)
     show f1:"\<And>nellx nelly P k x xa.
       \<not> lnull nellx \<Longrightarrow>
       \<not> lnull nelly \<Longrightarrow>
       llast nellx = lhd nelly \<Longrightarrow>
       P (lhd nelly) \<Longrightarrow>
       lfinite nellx \<Longrightarrow>
       \<forall>b. nelly \<noteq> LCons b LNil \<Longrightarrow>
       \<forall>b. kfilter P (k + the_enat (epred (llength nellx))) nelly \<noteq> LCons b LNil \<Longrightarrow>
       x \<in> lset nelly \<Longrightarrow> P x \<Longrightarrow> \<forall>x\<in>lset nellx. \<not> P x \<Longrightarrow> P xa \<Longrightarrow> xa \<in> lset (ltl nelly) \<Longrightarrow>
          kfilter P k (lappend nellx (ltl nelly)) = iterates Suc k" 
    by (metis in_lset_lappend_iff lappend_lbutlast_llast_id lbutlast_lfinite llist.set_intros(1))
    next
     show f2:"\<And>nellx nelly P k x xa xb.
       \<not> lnull nellx \<Longrightarrow>
       \<not> lnull nelly \<Longrightarrow>
       llast nellx = lhd nelly \<Longrightarrow>
       P (lhd nelly) \<Longrightarrow>
       lfinite nellx \<Longrightarrow>
       \<forall>b. nelly \<noteq> LCons b LNil \<Longrightarrow>
       \<forall>b. kfilter P (k + the_enat (epred (llength nellx))) nelly \<noteq> LCons b LNil \<Longrightarrow>
       x \<in> lset nelly \<Longrightarrow>
       P x \<Longrightarrow>
       xa \<in> lset nellx \<Longrightarrow>
       P xa \<Longrightarrow>
       P xb \<Longrightarrow>
       xb \<in> lset nellx \<Longrightarrow> 
       kfilter P k (lappend nellx (ltl nelly)) = 
       lappend (kfilter P k nellx) (ltl (kfilter P (k + the_enat (epred (llength nellx))) nelly))"
       proof -
       fix nellxa :: "'b llist" and nellya :: "'b llist" and Pa :: "'b \<Rightarrow> bool" and
           ka :: nat and x :: 'b and xa :: 'b and xb :: 'b
       assume a1: "\<not> lnull nellya"
       assume a2: "Pa (lhd nellya)"
       assume a3: "llast nellxa = lhd nellya"
       assume a4: "\<not> lnull nellxa"
       assume a5: "lfinite nellxa"
       have f6: "nellya = LCons (lhd nellya) (ltl nellya)"
       using a1 by auto
       have f7: "LCons (lhd nellya) (ltl nellya) \<noteq> LNil \<and> lhd (LCons (lhd nellya) (ltl nellya)) = lhd nellya \<and> 
                 ltl (LCons (lhd nellya) (ltl nellya)) = ltl nellya"
       by auto
       then have f8: "kfilter Pa (the_enat (epred (llength nellxa)) + ka) (LCons (lhd nellya) (ltl nellya)) =
                      LCons (the_enat (epred (llength nellxa)) + ka) 
                            (kfilter Pa (Suc (the_enat (epred (llength nellxa)) + ka)) (ltl nellya))"
       using f6 a2 by (metis (no_types) kfilter_LCons)
       have f9: "\<forall>bs p n bsa. \<not> lfinite (bs::'b llist) \<or> kfilter p n (lappend bs bsa) = 
                  lappend (kfilter p n bs) (kfilter p (n + the_enat (llength bs)) bsa)"
       by (meson kfilter_lappend_lfinite)
       have f10: "lfinite (lbutlast nellxa)"
       using a5 by (meson lbutlast_lfinite)
       have f11: "lappend (kfilter Pa ka (lbutlast nellxa)) 
                          (kfilter Pa (ka + the_enat (llength (lbutlast nellxa))) LNil) = 
                  kfilter Pa ka (lbutlast nellxa)"
       by simp
       have "LCons (the_enat (epred (llength nellxa)) + ka) LNil = 
             kfilter Pa (the_enat (epred (llength nellxa)) + ka) (LCons (lhd nellya) LNil)"
       using a2 by simp
       then have f12: "lappend (lappend (kfilter Pa ka (lbutlast nellxa)) 
                                        (kfilter Pa (ka + the_enat (llength (lbutlast nellxa))) LNil)) 
                               (LCons (the_enat (epred (llength nellxa)) + ka) LNil) = kfilter Pa ka nellxa"
       using f11 f10 f9 a5 a4 a3 by (metis add.commute lappend_lbutlast_llast_id_lfinite llength_lbutlast)
       have "ltl (kfilter Pa (ka + the_enat (epred (llength nellxa))) nellya) =
             kfilter Pa (Suc (the_enat (epred (llength nellxa)) + ka)) (ltl nellya)"
       using f8 f6 by (simp add: add.commute)
       then show "kfilter Pa ka (lappend nellxa (ltl nellya)) = 
                  lappend (kfilter Pa ka nellxa) (ltl (kfilter Pa (ka + the_enat (epred (llength nellxa))) nellya))"
       using f12 f11 f10 f9 f8 f7 f6 a5 a4 a3 
       by (metis add.commute lappend_lbutlast_llast_id_lfinite lappend_snocL1_conv_LCons2 llength_lbutlast)
       qed 
   show "\<And>nellx nelly P k x xa xb.
       \<not> lnull nellx \<Longrightarrow>
       \<not> lnull nelly \<Longrightarrow>
       llast nellx = lhd nelly \<Longrightarrow>
       P (lhd nelly) \<Longrightarrow>
       lfinite nellx \<Longrightarrow>
       \<forall>b. nelly \<noteq> LCons b LNil \<Longrightarrow>
       \<forall>b. kfilter P (k + the_enat (epred (llength nellx))) nelly \<noteq> LCons b LNil \<Longrightarrow>
       x \<in> lset nelly \<Longrightarrow>
       P x \<Longrightarrow>
       xa \<in> lset nellx \<Longrightarrow>
       P xa \<Longrightarrow>
       P xb \<Longrightarrow>
       xb \<in> lset (ltl nelly) \<Longrightarrow>
       kfilter P k (lappend nellx (ltl nelly)) = 
       lappend (kfilter P k nellx) (ltl (kfilter P (k + the_enat (epred (llength nellx))) nelly)) "
       by (simp add: f2)
    qed   
qed

lemma nleast_conv:
 assumes "\<exists>x \<in> nset nellx. P x"
 shows   " nleast P nellx = (LEAST na. na \<le> nlength nellx \<and> P (nnth nellx na))"
using assms  
by transfer
   (auto simp add: min_def lleast_def,
    metis co.enat.exhaust_sel iless_Suc_eq llength_eq_0)

lemma nfilter_chop: 
assumes "nlast nellx = nfirst nelly"
        " P (nlast nellx)"
        "nfinite nellx" 
shows "nfilter P (nfuse nellx nelly) = nfuse (nfilter P nellx)  (nfilter P nelly)"
proof (cases "is_NNil nelly")
case True
then show ?thesis by (metis nfilter_NNil nfuse_def1 nfuse_leftneutral) 
next
case False
then show ?thesis 
  proof (cases "is_NNil (nfilter P nelly)")
  case True
  then show ?thesis unfolding nfuse_def1 using assms nfilter_nappend2[of "ntl nelly" P nellx] 
   nfilter_expand[of nelly P]  
   by (auto simp add: nfirst_def nellist.case_eq_if )
    (metis nellist.discI(2) nellist.set_sel(2)  nset_nlast)
  next
  case False
  then show ?thesis 
    proof -
     have 1: "\<exists>x\<in>nset nellx. P x "
       using assms nset_nlast by blast 
     have 2: "\<exists>x\<in>nset (ntl nelly). P x "
       using False assms 
       by (metis nellist.collapse(1) nellist.collapse(2) nellist.disc(1) nfilter_NCons_a
         nfilter_NNil nnth_0 ntaken_0 ntaken_nlast)
     have 3: "\<not>is_NNil (nfilter P nelly) \<Longrightarrow> 
               nfilter P (nappend nellx (ntl nelly)) = nappend (nfilter P nellx) (nfilter P (ntl nelly))"
       by (simp add: "1" "2" assms(3) nappend_nfilter_nfinite)
     have 4: "is_NNil (nfilter P nelly) \<Longrightarrow> nfilter P nellx = nappend (nfilter P nellx) (nfilter P (ntl nelly))"
       by (simp add: False)
     have 5: "nfilter P (nfuse nellx nelly) = nfilter P  (nappend nellx (ntl nelly))" 
       unfolding nfuse_def1 
       by (metis (full_types) False nellist.collapse(1) nfilter_NNil)
     have 6: " (ntl (nfilter P nelly)) =  (nfilter P (ntl nelly))"
       using assms 2 False nfilter_expand[of "nelly" P]
       by (metis  in_nset_ntlD nellist.case_eq_if nellist.sel(5) nfirst_def) 
     show ?thesis 
       by (metis "3" "5" "6" False nfuse_def1)
   qed  
  qed    
qed


lemma nfilter_chop1:
assumes " n\<le> nlength nellx"
        "  P (nlast (ntaken n nellx))"
shows  "nfilter P nellx = nfuse (nfilter P (ntaken n nellx))  (nfilter P (ndropn n nellx))"
using assms 
by (metis nfuse_ntaken_ndropn ndropn_nfirst nfilter_chop nfinite_ntaken ntaken_nlast)

lemma nfilter_chop1_ntaken:
assumes " n\<le> nlength nellx"
        "  P (nlast (ntaken n nellx))"
shows  "ntaken (the_enat (nlength(nfilter P (ntaken n nellx)))) (nfilter P nellx) = 
         (nfilter P (ntaken n nellx))  "
using assms nfilter_nappend_ntaken by (metis nfinite_ntaken nset_nlast)

lemma nfilter_nlast:
 assumes " n\<le> nlength nellx"
        "  P (nlast (ntaken n nellx))"
 shows  "nlast(nfilter P (ntaken n nellx)) = (nlast(ntaken n nellx))"
using assms
proof (induction n arbitrary: nellx)
case 0
then show ?case by simp
next
case (Suc n)
then show ?case 
  proof (cases nellx)
  case (NNil x1)
  then show ?thesis by auto
  next
  case (NCons x21 x22)
  then show ?thesis using Suc 
  by auto 
     (metis Suc_ile_eq iless_Suc_eq nfilter_NCons nfinite_ntaken nlast_NCons nlength_NCons 
      nset_nlast ntaken_Suc_NCons)
  qed
qed

lemma nfilter_nfirst:
 assumes " P (nfirst(ndropn n nellx))"
 shows   "nfirst(nfilter P (ndropn n nellx)) = (nfirst (ndropn n nellx)) "
using assms
proof (induction n arbitrary: nellx)
case 0
then show ?case 
  by (auto simp add: ndropn_nfirst)
     (metis nellist.set_intros(1) nfilter_NNil nfilter_nappend_ntaken nlast_NNil nlength_NNil
      ntaken_0 the_enat_0 zero_enat_def zero_le)
next
case (Suc n)
then show ?case 
   by (metis ndropn_ndropn plus_1_eq_Suc)
qed

lemma nfilter_chop1_ndropn:
assumes " n\<le> nlength nellx"
        "  P (nlast (ntaken n nellx))"
shows  "ndropn (the_enat (nlength(nfilter P (ntaken n nellx)))) (nfilter P nellx) = 
        (nfilter P (ndropn n nellx))  "
proof -
 have 1: "(nfilter P nellx) = nfuse (nfilter P (ntaken n nellx))  (nfilter P (ndropn n nellx))"
   by (simp add: assms nfilter_chop1) 
 have 2: "nlast(nfilter P (ntaken n nellx)) = nfirst (nfilter P (ndropn n nellx))"
   by (metis assms(1) assms(2) ndropn_nfirst nfilter_nfirst nfilter_nlast ntaken_nlast)
 have 3: "ndropn (the_enat (nlength(nfilter P (ntaken n nellx)))) 
                 (nfuse (nfilter P (ntaken n nellx))  (nfilter P (ndropn n nellx))) =
          (nfilter P (ndropn n nellx))"
   by (metis "2" assms(1) enat_the_enat infinity_ileE length_nfilter_le min_absorb1 ndropn_nfuse 
       nfinite_conv_nlength_enat ntaken_nlength) 
 show ?thesis by (simp add: "1" "3")        
qed

lemma nkfilter_chop1:
  assumes "(enat n) \<le> nlength nellx"
          " P (nlast (ntaken n nellx))"
  shows "nkfilter P k nellx = nfuse (nkfilter P k (ntaken n nellx)) (nkfilter P (k+n) (ndropn n nellx))"
using assms nfuse_ntaken_ndropn[of n nellx] nkfilter_chop[of "(ntaken n nellx)" "(ndropn n nellx)" P k]
by (metis min.orderE ndropn_nfirst nfinite_ntaken ntaken_nlast ntaken_nlength the_enat.simps) 

lemma nkfilter_nlast:
 assumes " n\<le> nlength nellx"
        "  P (nlast (ntaken n nellx))"
 shows  "nlast(nkfilter P k (ntaken n nellx)) = k+n"
using assms
proof (induction n arbitrary: k nellx)
case 0
then show ?case by simp
next
case (Suc n)
then show ?case 
 proof (cases nellx)
 case (NNil x1)
 then show ?thesis
 using Suc.prems(1) enat_0_iff(1) by auto 
 next
 case (NCons x21 x22)
 then show ?thesis 
   using Suc 
   proof auto
   assume a1: "P (nlast (ntaken n x22))"
   assume a2: "\<And>nellx k. \<lbrakk>enat n \<le> nlength nellx; P (nlast (ntaken n nellx))\<rbrakk> \<Longrightarrow> 
               nlast (nkfilter P k (ntaken n nellx)) = k + n"
   assume "enat (Suc n) \<le> eSuc (nlength x22)"
   then have "enat n < eSuc (nlength x22)"
   using Suc_ile_eq by blast
   then have f3: "enat n \<le> nlength x22"
   by (meson iless_Suc_eq)
   have "\<exists>a. a \<in> nset (ntaken n x22) \<and> P a"
   using a1 nfinite_ntaken nset_nlast by blast
   then show "nlast (nkfilter P k (NCons x21 (ntaken n x22))) = Suc (k + n)"
   using f3 a2 a1 by (simp add: Bex_def_raw)
   qed 
 qed
qed

lemma nkfilter_nfirst:
 assumes " P (nfirst(ndropn n nellx))"
 shows   "nfirst(nkfilter P k (ndropn n nellx)) = k "
 using assms
proof (induction n arbitrary: k nellx)
case 0
then show ?case 
 by (metis enat_defs(1) nellist.set_intros(1) nkfilter_NNil nkfilter_nappend_ntaken nlength_NNil 
     nnth_NNil ntaken_0 the_enat.simps zero_le)
next
case (Suc n)
then show ?case 
proof (cases nellx)
case (NNil x1)
then show ?thesis using Suc
by (metis ndropn_ndropn plus_1_eq_Suc) 
next
case (NCons x21 x22)
then show ?thesis using Suc
by auto 
qed
qed
 
lemma nkfilter_chop1_ndropn:
assumes " n\<le> nlength nellx"
        "  P (nlast (ntaken n nellx))"
shows  "ndropn (the_enat (nlength(nkfilter P k (ntaken n nellx)))) (nkfilter P k nellx) = 
        (nkfilter P (k+n) (ndropn n nellx))  "
proof -
 have 1: "(nkfilter P k nellx) = nfuse (nkfilter P k (ntaken n nellx))  (nkfilter P (k+n) (ndropn n nellx))"
   by (simp add: assms nkfilter_chop1) 
 have 2: "nlast(nkfilter P k (ntaken n nellx)) = nfirst (nkfilter P (k+n) (ndropn n nellx))"
   by (metis assms(1) assms(2) ndropn_nfirst nkfilter_nfirst nkfilter_nlast ntaken_nlast)    
 have 3: "ndropn (the_enat (nlength(nkfilter P k (ntaken n nellx)))) 
                 (nfuse (nkfilter P k (ntaken n nellx))  (nkfilter P (k+n) (ndropn n nellx))) =
          (nkfilter P (k+n) (ndropn n nellx))"
   by (metis "2" assms(2) enat_the_enat infinity_ileE ndropn_nfuse nfinite_conv_nlength_enat 
       nfinite_ntaken nlength_nkfilter_le nset_nlast)
  show ?thesis by (simp add: "1" "3")        
qed

lemma nkfilter_chop1_ntaken:
assumes " n\<le> nlength nellx"
        "  P (nlast (ntaken n nellx))"
shows  "ntaken (the_enat (nlength(nkfilter P k (ntaken n nellx)))) (nkfilter P k nellx) = 
        (nkfilter P k (ntaken n nellx))  "
proof -
 have 1: "(nkfilter P k nellx) = nfuse (nkfilter P k (ntaken n nellx))  (nkfilter P (k+n) (ndropn n nellx))"
   by (simp add: assms nkfilter_chop1) 
 have 2: "nlast(nkfilter P k (ntaken n nellx)) = nfirst (nkfilter P (k+n) (ndropn n nellx))"
   by (metis assms(1) assms(2) ndropn_nfirst nkfilter_nfirst nkfilter_nlast ntaken_nlast)
 have 3: "ntaken (the_enat (nlength(nkfilter P k (ntaken n nellx)))) 
                 (nfuse (nkfilter P k (ntaken n nellx))  (nkfilter P (k+n) (ndropn n nellx))) =
          (nkfilter P k (ntaken n nellx))"
   by (metis "1" assms(1) assms(2) nfinite_ntaken nkfilter_nappend_ntaken nset_nlast)   
 show ?thesis by (simp add: "1" "3")        
qed

lemma nfilter_nsubn: 
 assumes "enat j \<le> nlength nellx "
         "P (nnth nellx j) "
         "enat i \<le> nlength nellx"
         "P (nnth nellx i)"
         "i \<le> j "
         "enat ni = (nlength(nfilter P (ntaken i nellx)))" 
         "enat nj = (nlength(nfilter P (ntaken j nellx))) " 
 shows "(nfilter P (nsubn nellx i j)) = (nsubn (nfilter P nellx) ni nj) "
proof -
 have 1: "(nfilter P (nsubn nellx i j)) = (nfilter P (ntaken (j-i) (ndropn i nellx)))" 
   by (simp add: nsubn_def1)
 have 2: "(nfilter P (ntaken (j-i) (ndropn i nellx))) =
          ntaken (the_enat (nlength(nfilter P (ntaken (j-i) (ndropn i nellx))))) 
                 (nfilter P (ndropn i nellx))" 
  by (metis assms(2) assms(5) enat_ile le_add_diff_inverse2 linorder_le_cases nfilter_chop1_ntaken 
      ntaken_all ntaken_ndropn_nlast)
 have 3: "ntaken (the_enat (nlength(nfilter P (ntaken (j-i) (ndropn i nellx))))) 
                 (nfilter P (ndropn i nellx)) =
          ntaken (the_enat (nlength(nfilter P (ntaken (j-i) (ndropn i nellx))))) 
                 (ndropn (the_enat (nlength(nfilter P (ntaken i nellx)))) (nfilter P nellx))" 
  by (simp add: assms(3) assms(4) nfilter_chop1_ndropn ntaken_nlast)
 have 4: "ntaken (the_enat (nlength(nfilter P (ntaken (j-i) (ndropn i nellx))))) 
                 (ndropn (the_enat (nlength(nfilter P (ntaken i nellx)))) (nfilter P nellx)) =
       nsubn (nfilter P nellx)
      (the_enat (nlength(nfilter P (ntaken i nellx))))
      ((the_enat (nlength(nfilter P (ntaken (j-i) (ndropn i nellx)))))+
       (the_enat (nlength(nfilter P (ntaken i nellx)))))" 
  using ntaken_ndropn by blast
 have 5: "((the_enat (nlength(nfilter P (ntaken (j-i) (ndropn i nellx)))))+
           (the_enat (nlength(nfilter P (ntaken i nellx))))) =
          ((the_enat (nlength(nfilter P (nsubn nellx i j))))+
           (the_enat (nlength(nfilter P (ntaken i nellx))))) "
  using "1" by auto 
 have 6: "ntaken j nellx = nfuse (ntaken i nellx) (nsubn nellx i j)"
   using nsubn_nfuse[of 0 i j nellx] 
   by (simp add: assms(1) assms(5) nsubn_def1)
 have 7: "nlength (nfilter P (nfuse (ntaken i nellx) (nsubn nellx i j))) =
          (nlength (nfilter P (ntaken i nellx)) + nlength (nfilter P (nsubn nellx i j))) " 
    by (simp add: assms(4) ndropn_nfirst nfilter_chop nfuse_nlength nsubn_def1 ntaken_nfirst ntaken_nlast) 
 have 70: "nfinite (nfilter P (nfuse (ntaken i nellx) (nsubn nellx i j)))" 
    by (metis "6" assms(1) assms(2) nfilter_chop1_ntaken nfinite_ntaken ntaken_nlast)
 have 71: "nfinite (nfilter P (ntaken i nellx))"
   by (metis assms(3) assms(4) nfilter_chop1_ntaken nfinite_ntaken ntaken_nlast) 
 have 72: "nfinite (nsubn nellx i j)"
    by (simp add: nsubn_def1)
 have 8: "nj =
          ((the_enat (nlength(nfilter P (nsubn nellx i j))))+ ni)"
    by (metis "6" "7" add.commute assms(6) assms(7) enat_le_plus_same(2) enat_the_enat infinity_ileE 
        plus_enat_simps(1) the_enat.simps)
 show ?thesis 
 using "1" "2" "3" "4" "8" by (metis assms(6) the_enat.simps)
qed

lemma nfilter_nsubn_zero: 
 assumes "enat j \<le> nlength nellx "
         "P (nnth nellx j) "
 shows "(nfilter P (nsubn nellx 0 j)) = 
        (nsubn (nfilter P nellx)
               0
               (the_enat (nlength(nfilter P (ntaken j nellx))))
        )
        "
  using assms 
  by (simp add: nfilter_chop1_ntaken nsubn_def1 ntaken_nlast)


subsection \<open>nrev\<close>

lemma is_NNil_nrev[simp]: 
 " is_NNil (nrev xs) = is_NNil xs "
apply transfer 
by (simp add: is_LEmpty_llength)

lemma nrev_NNil[simp]: 
 "nrev (NNil a) = (NNil a) " 
apply transfer
by force

lemma nrev_NCons[simp]:
 "nrev (NCons a xs) = (if nfinite xs then nappend (nrev xs) (NNil a) else (NCons a xs))" 
apply transfer
unfolding lrev_def 
by auto
   (metis lfinite_code(2) list_of_LCons lrev_LCons lrev_def rev.simps(2))

lemma ntl_nrev[simp]:
 " nlength xs > 1 \<Longrightarrow> 
   ntl (nrev xs) =  
   (if nfinite xs then 
     (  nappend (ntl(nrev (ntl xs))) (NNil (nhd xs))) 
     else ntl xs ) " 
apply transfer
by (auto simp add: is_LEmpty_llength epred_llength)
   (simp add: llength_ltl)



lemma ntl_nrev_one[simp]:
 " nlength xs = 1 \<Longrightarrow> 
   ntl (nrev xs) = (NNil (nhd xs)) " 
apply transfer
apply (auto simp add: is_LEmpty_llength )
apply (metis epred_1 epred_llength lappend_lnull1 llength_eq_0 llength_lrev)
apply (simp add: epred_llength)
by (metis co.enat.sel(2) eSuc_infinity llength_eq_infty_conv_lfinite) 

lemma ntl_nrev_zero[simp]:
assumes " nlength xs = 0 " 
shows " ntl (nrev xs) = xs "  
by (metis assms ndropn_0 ndropn_nlast nellist.sel(4) nlength_eq_enat_nfiniteD nrev_NNil
     the_enat_0 zero_enat_def)

lemma nmap_nrev:
 "nmap f (nrev xs) = (nrev (nmap f xs)) " 
apply transfer
by (simp add: lmap_lrev)

lemma nfinite_nrev:
 " nfinite(nrev xs) = nfinite xs " 
apply transfer
by simp

lemma nset_nrev:
 "nset(nrev xs) = nset xs " 
apply transfer
by (simp add: lset_lrev)

lemma nlength_nrev[simp]:
 "nlength (nrev xs) = nlength xs" 
apply transfer
by simp

lemma lnth_lrev_help: 
"\<not> lnull xs \<Longrightarrow> lfinite xs \<Longrightarrow> enat i \<le> epred (llength xs) \<Longrightarrow> lnth (lrev xs) i = lnth xs (the_enat (min (enat (the_enat (epred (llength xs)) - i)) (epred (llength xs))))"
using lnth_lrev[of xs i] 
unfolding min_def
apply auto
apply (metis co.enat.exhaust_sel diff_diff_left enat_the_enat epred_enat iless_Suc_eq 
        llength_eq_0 llength_eq_infty_conv_lfinite plus_1_eq_Suc the_enat.simps)
by (metis co.enat.exhaust_sel diff_le_self eSuc_infinity enat_ord_simps(1) enat_the_enat 
    llength_eq_0 llength_eq_infty_conv_lfinite)

lemma nnth_nrev:
assumes "nfinite xs "
        " (enat i) \<le> nlength xs"
shows " (nnth (nrev xs) i ) = (nnth xs ((the_enat (nlength xs)) -i)) "
using assms
apply transfer
apply auto
using lnth_lrev_help by blast

lemma nappend_nrev_nfinite: 
assumes "nfinite xs"
        "nfinite ys"
shows "nrev (nappend xs ys) = nappend (nrev ys) (nrev xs)"
using assms 
apply transfer
by (simp add: lappend_lrev_lfinite)


lemma nappend_nrev_infa: 
assumes "nfinite xs"
        "\<not>nfinite ys"
shows "nrev (nappend xs ys) = nappend xs ys" 
using assms
apply transfer
by simp

lemma nappend_nrev__infb: 
assumes "\<not>nfinite xs"
shows "nrev (nappend xs ys) = nappend (nrev xs) (nrev ys)" 
using assms 
apply transfer
by (simp add: lappend_inf)

lemma nrev_nrev: 
 "nrev (nrev xs) = xs " 
apply transfer
by (simp add: lrev_lrev)

lemma nrev_ntaken: 
 assumes "nfinite xs" 
          "enat k \<le> nlength xs" 
 shows " nrev (ntaken k xs) = ndropn (the_enat(nlength xs) -k) (nrev xs)"
using assms 
proof (transfer)
fix xs k
show "\<not> lnull xs \<and> xs = xs \<Longrightarrow>
       lfinite xs \<Longrightarrow>
       enat k \<le> epred (llength xs) \<Longrightarrow>
       \<not> lnull
           (if lfinite (ltake (enat (Suc k)) xs)
            then lrev (ltake (enat (Suc k)) xs)
            else ltake (enat (Suc k)) xs) \<and>
       (if lfinite (ltake (enat (Suc k)) xs)
        then lrev (ltake (enat (Suc k)) xs)
        else ltake (enat (Suc k)) xs) =
       ldropn
        (the_enat
          (min (enat
                 (the_enat (epred (llength xs)) - k))
            (epred
              (llength
                (if lfinite xs then lrev xs
                 else xs)))))
        (if lfinite xs then lrev xs else xs)" 
proof -
 assume a0: "\<not> lnull xs \<and> xs = xs" 
 assume a1: "lfinite xs " 
 assume a2: "enat k \<le> epred (llength xs)" 
 show "
    \<not> lnull
        (if lfinite (ltake (enat (Suc k)) xs) then lrev (ltake (enat (Suc k)) xs)
         else ltake (enat (Suc k)) xs) \<and>
    (if lfinite (ltake (enat (Suc k)) xs) then lrev (ltake (enat (Suc k)) xs) else ltake (enat (Suc k)) xs) =
    ldropn
     (the_enat
       (min (enat (the_enat (epred (llength xs)) - k)) (epred (llength (if lfinite xs then lrev xs else xs)))))
     (if lfinite xs then lrev xs else xs) "  
 proof -
 have 1: "enat (Suc k) \<noteq> 0"  
   by (metis eSuc_enat zero_ne_eSuc)
 have 2: " llength xs = 1 \<Longrightarrow> ?thesis "
   using a0 a1 a2 by (simp add: "1" ltake_all one_enat_def)
 have 3: " enat (Suc k) < llength xs \<Longrightarrow> 
          (the_enat (min (enat (the_enat (epred (llength xs)) - k)) (epred (llength xs)))) = 
          (enat (the_enat (llength xs) - Suc k)) " 
    using a0 a1 a2 by simp
  (metis diff_diff_left diff_le_self enat_ord_simps(1) enat_the_enat epred_enat 
   llength_eq_infty_conv_lfinite min.absorb1 plus_1_eq_Suc the_enat.simps) 
 
 have 4: "enat (Suc k) < llength xs \<Longrightarrow> ?thesis"
   using a0 a1 a2  lrev_ltake[of xs "Suc k"]   apply simp using 3
   by (metis "1" ldrop_enat llength_lrev lnull_ldrop lnull_lrev ltake.disc(2))  
 have 5: " llength xs =  enat (Suc k) \<Longrightarrow> ?thesis " 
    using a0 a1 a2 by (simp add: "1" ltake_all)
 have 6: " llength xs = 1 \<or> enat (Suc k) < llength xs \<or> llength xs =  enat (Suc k)"
   using a0 a1 a2 apply simp
    by (metis Suc_ile_eq co.enat.exhaust_sel iless_Suc_eq llength_eq_0 order_neq_le_trans)
 show ?thesis using a0 1 2 4 5  6 by blast
 qed
 qed
qed


lemma nrev_ndropn: 
 assumes "nfinite xs" 
          "enat k \<le> nlength xs" 
 shows " nrev (ndropn k xs) = ntaken (the_enat(nlength xs) -k) (nrev xs) "
using assms 
proof transfer
fix xs k
show "\<not> lnull xs \<and> xs = xs \<Longrightarrow>
       lfinite xs \<Longrightarrow>
       enat k \<le> epred (llength xs) \<Longrightarrow>
       \<not> lnull
           (if lfinite (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
            then lrev (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
            else ldropn (the_enat (min (enat k) (epred (llength xs)))) xs) \<and>
       (if lfinite (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
        then lrev (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
        else ldropn (the_enat (min (enat k) (epred (llength xs)))) xs) =
       ltake (enat (Suc (the_enat (epred (llength xs)) - k))) (if lfinite xs then lrev xs else xs)" 
proof -
 assume a0: "\<not> lnull xs \<and> xs = xs" 
 assume a1: "lfinite xs"
 assume a2: "enat k \<le> epred (llength xs)"  
 show "\<not> lnull
           (if lfinite (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
            then lrev (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
            else ldropn (the_enat (min (enat k) (epred (llength xs)))) xs) \<and>
       (if lfinite (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
        then lrev (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
        else ldropn (the_enat (min (enat k) (epred (llength xs)))) xs) =
       ltake (enat (Suc (the_enat (epred (llength xs)) - k))) (if lfinite xs then lrev xs else xs)" 
 proof -
 have 1: "\<not> lnull
           (if lfinite (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
            then lrev (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)
            else ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)"
   using a0 a1 a2 apply simp
   by (metis co.enat.exhaust_sel iless_Suc_eq leD llength_eq_0)
 have 2: "lfinite (ldropn (the_enat (min (enat k) (epred (llength xs)))) xs)"
   using a1 lfinite_ldropn by blast
 have 3: "(the_enat (min (enat k) (epred (llength xs)))) = enat k"
   using a0 a1 a2 by simp
 have 4: "(enat (the_enat (llength xs) - k))  = (enat (Suc (the_enat (epred (llength xs)) - k)))"
    using a0 a1 a2 apply simp
    using "1" llength_eq_infty_conv_lfinite by fastforce
 have 5: "llength xs = 1 \<or> enat (Suc k) < llength xs \<or> llength xs =  enat (Suc k)"
   using a0 a1 a2 apply simp
   by (metis Suc_ile_eq co.enat.exhaust_sel iless_Suc_eq llength_eq_0 order_neq_le_trans)  
 have 6: "llength xs = 1 \<Longrightarrow> ?thesis"
      using a0 a1 a2 by (simp add: ltake_all one_enat_def)
 have 7: " enat (Suc k) < llength xs \<Longrightarrow> ?thesis" 
  using lrev_ldrop[of xs k]  a0 a1 a2 apply simp
  by (metis "4" Suc_ile_eq dual_order.strict_iff_order ldrop_enat linorder_not_less)  
 have 8: "llength xs =  enat (Suc k) \<Longrightarrow> ?thesis"
   using a0 a1 a2 apply simp
   by (metis "4" a2 add_diff_cancel_left' cancel_comm_monoid_add_class.diff_cancel eSuc_enat epred_enat 
       iless_Suc_eq ldrop_enat lrev_ldrop plus_1_eq_Suc the_enat.simps)  
 show ?thesis
 using "5" "6" "7" "8" by fastforce 
 qed
qed
qed
  
   

lemma nrev_nsubn: 
 assumes "nfinite xs" 
         " enat i \<le> enat j "
         "enat j \<le> nlength xs" 
 shows " nrev (nsubn xs i j ) = nsubn  (nrev xs)((the_enat(nlength xs)) -j) ((the_enat(nlength xs)) - i) " 
using assms
proof transfer
fix xs i j
assume a0: "\<not> lnull xs \<and> xs = xs " 
assume a1: "lfinite xs " 
assume a2: "enat i \<le> enat j " 
assume a3: "enat j \<le> epred (llength xs) " 
show " \<not> lnull
           (if lfinite (lsubc i j xs)
            then lrev (lsubc i j xs)
            else lsubc i j xs) \<and>
       (if lfinite (lsubc i j xs)
        then lrev (lsubc i j xs) else lsubc i j xs) =
       lsubc (the_enat (epred (llength xs)) - j)
        (the_enat (epred (llength xs)) - i)
        (if lfinite xs then lrev xs else xs)" 
 proof -
 have 1: "\<not> lnull (if lfinite (lsubc i j xs) then lrev (lsubc i j xs) else lsubc i j xs)"  
   using a0 a1 a2 a3 apply simp
   by (metis co.enat.exhaust_sel iless_Suc_eq llength_eq_0 llength_lsubc zero_ne_eSuc)  
 have 2: "lfinite (lsubc i j xs)"
    using a0 a1 a2 a3 by (simp add: lsubc_def)  
 have 3: "(the_enat (epred (llength xs)) - j) = (the_enat (llength xs) - (j + 1))"
    using a0 a1 a2 a3 lfinite_llength_enat by fastforce  
 have 4: "(the_enat (epred (llength xs)) - i) = (the_enat (llength xs) - (i + 1))" 
    using a0 a1 a2 a3 lfinite_llength_enat by fastforce  
 have 5: "enat j < llength xs" 
   using a0 a1 a2 a3 apply simp
   by (metis co.enat.exhaust_sel iless_Suc_eq llength_eq_0)
 have 6: "lrev (lsubc i j xs) = lsubc (the_enat (epred (llength xs)) - j)
        (the_enat (epred (llength xs)) - i)
        (if lfinite xs then lrev xs else xs)" 
    using lrev_lsubc[of xs i j]   5 a0 a1 a2 a3 3 4 by presburger 
 show ?thesis by (meson "1" "2" "6") 
 qed
qed
 

lemma nlast_nrev: 
shows "nlast (nrev xs) = (if nfinite xs then nfirst xs else nlast xs  ) "
apply transfer
by (simp add: lfirst_def llast_lrev)

lemma nfirst_nrev: 
shows " nfirst (nrev xs)  = (if nfinite xs then nlast xs else nfirst xs) "
apply transfer
apply simp
by (metis lfirst_def lfirst_lrev)




lemma nellist_all2_nrev:
 "  nellist_all2 P (nrev xs) (nrev ys) \<longleftrightarrow>  nellist_all2 P xs ys" 
apply transfer
using llist_all2_lfiniteD llist_all2_lrev  apply simp
by fastforce




end





subsection \<open>Transfer rule\<close>

context includes lifting_syntax
begin
lemma ndropns_transfer2 [transfer_rule]:
 " (nellist_all2 A ===> nellist_all2 (nellist_all2 A)) ndropns ndropns"
unfolding rel_fun_def
by (auto intro: nellist_all2_ndropnsI  )

end

end