(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)
section \<open>Monitors\<close>

theory Monitor
imports First

begin


text \<open>
The RV monitors language is  introduced plus the algebraic properties of the monitor operators. 
\<close>

subsection \<open>Syntax\<close>
(*sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="cvc4 e z3 vampire spass  " ]
*)
(*
declare [[show_types]]


declare [[show_consts]]
*)

datatype ('a ::world) monitor =
   mFIRST_d "'a formula"             ("(FIRST _)" [84] 83)
 | mUPTO_d "'a monitor" "'a monitor" ("(_ UPTO _)" [84,84] 83)
 | mTHRU_d "'a monitor" "'a monitor" ("(_ THRU _)" [84,84] 83)
 | mTHEN_d "'a monitor" "'a monitor" ("(_ THEN _)" [84,84] 83)
 | mWITH_d "'a monitor" "'a formula" ("(_ WITH _)" [84,84] 83)


fun MON :: "('a::world) monitor \<Rightarrow> 'a formula" 
where  "(MON (FIRST f))  = LIFT(\<rhd> f)"
     | "(MON (a UPTO b)) = LIFT(\<rhd>((MON a) \<or> (MON b) )) "
     | "(MON (a THRU b)) = LIFT(\<rhd>(di(MON a) \<and> di(MON b)))"
     | "(MON (a THEN b)) = LIFT((MON a)\<frown>(MON b))"
     | "(MON (a WITH f)) = LIFT((MON a) \<and> f)"  

syntax
 "_MON" :: "'a monitor \<Rightarrow> lift" ("(\<M> _)" [80] 80)

translations
 "_MON" == "CONST MON"  

definition eq_d :: "('a:: world) monitor \<Rightarrow> 'a monitor \<Rightarrow> bool" ("( _ \<simeq> _)" [84,84] 83)
where
 "eq_d a b \<equiv> (\<turnstile> (\<M> a) = (\<M> b))" 

lemma MonEqRefl:
 "a \<simeq> a"
by (simp add: eq_d_def)

lemma MonEqSym:
 assumes "a \<simeq> b"
 shows   "b \<simeq> a"
using assms by (metis eq_d_def inteq_reflection)

lemma MonEqTrans:
 assumes "a \<simeq> b"
         "b \<simeq> c"
 shows   "a \<simeq> c"
using assms(1) assms(2) by (metis eq_d_def inteq_reflection)

lemma MonEq:
 "(a \<simeq> b) = (\<turnstile> (\<M> a) = (\<M> b))"
by (simp add: eq_d_def)

lemma MonEqSubstWith:
 assumes "a \<simeq> b"
 shows   "(a WITH f) \<simeq> (b WITH f)"
using assms by (metis MON.simps(5) eq_d_def inteq_reflection lift_and_com)  

lemma MonEqSubstThen:
 assumes "a1 \<simeq> b1"
         "a2 \<simeq> b2"
 shows   " (a1 THEN a2) \<simeq> (b1 THEN b2)"
using assms by (simp add: SChopEqvSChop eq_d_def)

lemma MonEqSubstUpto:
 assumes "a1 \<simeq> b1"
         "a2 \<simeq> b2"
 shows   " (a1 UPTO a2) \<simeq> (b1 UPTO b2)"
using assms by (metis (mono_tags, lifting) MON.simps(2) eq_d_def int_eq MonEqRefl)

lemma MonEqSubstThru:
 assumes "a1 \<simeq> b1"
         "a2 \<simeq> b2"
 shows   " (a1 THRU a2) \<simeq> (b1 THRU b2)"
using assms by (metis (mono_tags, lifting) MON.simps(3) eq_d_def int_eq MonEqRefl)

subsection \<open>Derived Monitors\<close>

definition HALT_d :: "('a :: world) formula \<Rightarrow> 'a monitor"
where "HALT_d w \<equiv> FIRST(LIFT(fin (init w)))"

definition LEN_d :: " nat \<Rightarrow> ('a ::world) monitor" 
where
 "LEN_d k \<equiv> FIRST (LIFT(len k))"

definition EMPTY_d :: "('a:: world) monitor"
where
 "EMPTY_d \<equiv> FIRST (LIFT(empty))"

definition SKIP_d :: "('a:: world) monitor"
where
 "SKIP_d \<equiv> FIRST (LIFT (skip))"

syntax
  "_HALT_d"  :: "lift \<Rightarrow> 'a monitor"        ("(HALT _)" [84] 83)
  "_LEN_d"   :: "nat \<Rightarrow> 'a monitor"         ("(LEN _)" [84] 83)
  "_EMPTY_d" :: "'a monitor"                ("(EMPTY)" )
  "_SKIP_d"  :: "'a monitor"                ("(SKIP)")
 
syntax (ASCII)
  "_HALT_d"   :: "lift \<Rightarrow> 'a monitor"        ("(HALT _)" [84] 83)
  "_LEN_d"    :: "nat \<Rightarrow> 'a monitor"         ("(LEN _)" [84] 83)
  "_EMPTY_d"  :: "'a monitor"                ("(EMPTY)")
  "_SKIP_d"   :: "'a monitor"                ("(SKIP)")
 

translations
 "_HALT_d"  \<rightleftharpoons> "CONST HALT_d" 
 "_LEN_d"   \<rightleftharpoons> "CONST LEN_d"
 "_EMPTY_d" \<rightleftharpoons> "CONST EMPTY_d"
 "_SKIP_d"  \<rightleftharpoons> "CONST SKIP_d"


definition GUARD_d :: "('a::world) formula \<Rightarrow> 'a monitor "
where
 "GUARD_d w \<equiv>  (EMPTY WITH LIFT(init w))"  

primrec TIMES_d :: "('a :: world) monitor \<Rightarrow> nat \<Rightarrow> 'a monitor" 
where
  TIMES_0  : "TIMES_d a 0       = EMPTY"
| TIMES_Suc: "TIMES_d a (Suc k) = (a THEN (TIMES_d a k))"

syntax
 "_GUARD_d" :: "lift \<Rightarrow> 'a monitor"             ("(GUARD _)" [84] 83)
 "_TIMES_d" :: "['a monitor,nat] \<Rightarrow> 'a monitor" ("(_ TIMES _)" [84,84] 83)

syntax (ASCII)
 "_GUARD_d" :: "lift \<Rightarrow> 'a monitor"             ("(GUARD _)" [84] 83)
 "_TIMES_d" :: "['a monitor,nat] \<Rightarrow> 'a monitor" ("(_ TIMES _)" [84,84] 83)

translations
  "_GUARD_d" \<rightleftharpoons> "CONST GUARD_d"
  "_TIMES_d" \<rightleftharpoons> "CONST TIMES_d" 


definition FAIL_d :: "('a:: world) monitor" 
where 
 "FAIL_d \<equiv> GUARD (#False)"

definition ALWAYS_d :: "('a :: world) monitor \<Rightarrow> 'a formula \<Rightarrow> 'a monitor"
where 
 "ALWAYS_d a w \<equiv>  (a WITH LIFT((bi (finite\<longrightarrow>fin (init w)))))"

definition SOMETIME_d :: "('a :: world) monitor \<Rightarrow> 'a formula \<Rightarrow> 'a monitor"
where 
 "SOMETIME_d a w \<equiv>  (a WITH LIFT((di (finite \<and> fin (init w)))))"
 
definition LIMIT_d :: "('a :: world) formula \<Rightarrow> 'a formula" 
where 
 "LIMIT_d f \<equiv> LIFT(bs (\<not> f))"  (*  \<equiv> (\<not> di (f;skip))  *)

definition UNTIL_d :: "('a :: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a monitor"
  where
 "UNTIL_d w1 w2 \<equiv> (HALT w2) WITH (LIFT(bm w1))"

syntax
 "_FAIL_d"     :: "'a monitor"                      ("FAIL")
 "_ALWAYS_d"   :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("(_ ALWAYS _)" [84,84] 83)
 "_SOMETIME_d" :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("(_ SOMETIME _)" [84,84] 83)
 "_LIMIT_d"    :: "lift \<Rightarrow> lift"                    ("(Limit _)" [84] 83)
 "_UNTIL_d"    :: "[lift,lift] \<Rightarrow> 'a monitor"       ("(_ UNTIL _)" [84,84] 83)

syntax (ASCII)
 "_FAIL_d"     :: "'a monitor"                      ("FAIL")
 "_ALWAYS_d"   :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("(_ ALWAYS _)" [84,84] 83)
 "_SOMETIME_d" :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("(_ SOMETIME _)" [84,84] 83)
 "_LIMIT_d"    :: "lift \<Rightarrow> lift"                    ("(Limit _)" [84] 83)
 "_UNTIL_d"    :: "[lift,lift] \<Rightarrow> 'a monitor"       ("(_ UNTIL _)" [84,84] 83)

translations
 "_FAIL_d"     \<rightleftharpoons> "CONST FAIL_d"
 "_ALWAYS_d"   \<rightleftharpoons> "CONST ALWAYS_d"
 "_SOMETIME_d" \<rightleftharpoons> "CONST SOMETIME_d"
 "_LIMIT_d"    \<rightleftharpoons> "CONST LIMIT_d"
 "_UNTIL_d"    \<rightleftharpoons> "CONST UNTIL_d" 
 

definition WITHIN_d :: "('a :: world) monitor \<Rightarrow> 'a formula \<Rightarrow> 'a monitor"
where 
 "WITHIN_d a f \<equiv> (a WITH LIFT(Limit f))"

syntax
 "_WITHIN_d" :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("(_ WITHIN _)" [84,84] 83)

syntax (ASCII)
 "_WITHIN_d" :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("(_ WITHIN _)" [84,84] 83)

translations
 "_WITHIN_d" \<rightleftharpoons> "CONST WITHIN_d"


definition AND_d :: "('a :: world) monitor \<Rightarrow> 'a monitor \<Rightarrow> 'a monitor"
where 
  "AND_d a b \<equiv> (a WITH LIFT(\<M> b)) "

definition ITERATE_d :: "('a :: world) monitor \<Rightarrow> 'a monitor \<Rightarrow> 'a monitor"
where
  "ITERATE_d a b \<equiv> (a WITH (LIFT (schopstar (\<M> b))))"


syntax
 "_AND_d"     :: "['a monitor,'a monitor] \<Rightarrow> 'a monitor" ("(_ AND _)" [84,84] 83)
 "_ITERATE_d" :: "['a monitor,'a monitor] \<Rightarrow> 'a monitor" ("(_ ITERATE _)" [84,84] 83)

syntax (ASCII)
 "_AND_d"     :: "['a monitor,'a monitor] \<Rightarrow> 'a monitor" ("(_ AND _)" [84,84] 83)
 "_ITERATE_d" :: "['a monitor,'a monitor] \<Rightarrow> 'a monitor" ("(_ ITERATE _)" [84,84] 83)

translations
 "_AND_d"     \<rightleftharpoons> "CONST AND_d"
 "_ITERATE_d" \<rightleftharpoons> "CONST ITERATE_d" 


definition STAR_d :: "('a :: world) monitor \<Rightarrow> 'a formula \<Rightarrow> 'a monitor"
where
 "STAR_d a f \<equiv> ((FIRST LIFT(\<diamond> f)) ITERATE (a))"

definition REPEAT_d :: "('a :: world) monitor \<Rightarrow> 'a formula \<Rightarrow> 'a monitor"
where 
  "REPEAT_d a w \<equiv> ((HALT w) ITERATE (a WITH LIFT(keep(\<not> (init w)))))"

syntax
 "_STAR_d"   :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("(_ STAR _)" [84,84] 83)
 "_REPEAT_d" :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("( _ REPEATUNTIL _)" [84,84] 83)

syntax (ASCII)
 "_STAR_d"   :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("(_ STAR _)" [84,84] 83)
 "_REPEAT_d" :: "['a monitor,lift] \<Rightarrow> 'a monitor" ("( _ REPEATUNTIL _)" [84,84] 83)

translations
 "_STAR_d"   \<rightleftharpoons> "CONST STAR_d" 
 "_REPEAT_d" \<rightleftharpoons> "CONST REPEAT_d" 



subsection \<open>Monitor Laws\<close>

lemma MFixFst:
 "\<turnstile> (\<M> a) = \<rhd> (\<M> a)"
proof
 (induct a )
 case (mFIRST_d x)
 then show ?case 
  proof -
   have 1: "\<turnstile> (\<M> (FIRST x)) = \<rhd> x"  by simp 
   have 2: "\<turnstile> \<rhd> x = \<rhd> (\<rhd> x)" using FstFixFst by fastforce
   have 3: "\<turnstile>  \<rhd> (\<rhd> x) = \<rhd>(\<M> (FIRST x))" by simp  
   from 1 2 3 show ?thesis by fastforce
  qed
 next
  case (mUPTO_d a1 a2)
  then show ?case 
  proof -
  have 1: "\<turnstile> (\<M> (a1 UPTO a2)) = \<rhd>( (\<M> a1) \<or> (\<M> a2))" 
       by (simp )
  have 2: "\<turnstile>  \<rhd>( (\<M> a1) \<or> (\<M> a2)) = \<rhd>(\<rhd>((\<M> a1) \<or> (\<M> a2)))" 
       using FstFixFst by fastforce
  have 3: "\<turnstile> \<rhd>(\<rhd>((\<M> a1) \<or> (\<M> a2))) = \<rhd>(\<M> (a1 UPTO a2))" 
       using "2" by simp 
  from 1 2 3 show ?thesis by fastforce
 qed  
next
  case (mTHRU_d a1 a2)
  then show ?case 
  proof -
   have 1: "\<turnstile> (\<M> (a1 THRU a2)) = \<rhd>( di (\<M> a1) \<and> di(\<M> a2))" 
        by (simp)
   have 2: "\<turnstile> \<rhd>( di (\<M> a1) \<and> di(\<M> a2)) = \<rhd>(\<rhd>(di(\<M> a1) \<and> di(\<M> a2)))" 
        using FstFixFst by fastforce
   have 3: "\<turnstile> \<rhd>(\<rhd>( di (\<M> a1) \<and> di(\<M> a2))) = \<rhd>(\<M> (a1 THRU a2))" 
        using "2" by simp 
   from 1 2 3 show ?thesis by fastforce
 qed
next
  case (mTHEN_d a1 a2)
  then show ?case 
  proof -
   have 1: "\<turnstile> (\<M> (a1 THEN a2)) = (\<M> a1) \<frown> (\<M> a2) " 
        by (simp)
   have 2: "\<turnstile> (\<M> a1) \<frown> (\<M> a2) = \<rhd>(\<M> a1) \<frown> \<rhd>(\<M> a2)" 
        using SChopEqvSChop mTHEN_d.hyps(1) mTHEN_d.hyps(2) by blast
   have 3: "\<turnstile> \<rhd>(\<M> a1) \<frown> \<rhd>(\<M> a2) = \<rhd>(\<rhd>(\<M> a1) \<frown> (\<M> a2))" 
        using FstFstChopEqvFstChopFst
        by (metis FstImpFinite Prop10 inteq_reflection schop_d_def)
   have 4: "\<turnstile> \<rhd>(\<rhd>(\<M> a1) \<frown> (\<M> a2)) = \<rhd>((\<M> a1) \<frown> (\<M> a2))" 
        using FstEqvRule LeftSChopEqvSChop mTHEN_d.hyps(1) by (metis inteq_reflection)
   have 5: "\<turnstile> \<rhd>((\<M> a1) \<frown> (\<M> a2)) = \<rhd>(\<M> (a1 THEN a2))" 
        using "4" by simp  
   from 1 2 3 4 5 show ?thesis by fastforce
  qed
next
  case (mWITH_d a x2)
  then show ?case 
  proof -
   have 1: "\<turnstile> (\<M> (a WITH x2)) = ((\<M> a) \<and> ( x2))" 
        by (simp )
   have 2: "\<turnstile> ((\<M> a) \<and> ( x2)) = (\<rhd>(\<M> a) \<and> ( x2))" 
        using mWITH_d.hyps by fastforce
   have 3: "\<turnstile> (\<rhd>(\<M> a) \<and> ( x2)) = \<rhd>(\<rhd>(\<M> a) \<and> ( x2))"  
        using FstFstAndEqvFstAnd by fastforce
   have 4: "\<turnstile> \<rhd>(\<rhd>(\<M> a) \<and> ( x2)) = \<rhd>((\<M> a) \<and> ( x2))" 
        using "2" FstEqvRule by fastforce
   have 5: "\<turnstile> \<rhd>((\<M> a) \<and> ( x2)) = \<rhd>(\<M> (a WITH x2))" 
        using "4" by simp 
   from 1 2 3 4 5 show ?thesis by (metis inteq_reflection)
 qed
qed

lemma MGuardFalseEqvFalse:
 "\<turnstile>  \<M>(GUARD #False) = #False"
proof -
  have 1: "\<turnstile> \<M>(GUARD #False) = \<M>(EMPTY WITH LIFT(init #False))" by (simp add: GUARD_d_def)
  have 2: "\<turnstile> \<M>(EMPTY WITH LIFT(init #False)) = (\<M>(EMPTY) \<and> (init #False))" by (simp )
  have 3: "\<turnstile> #False = (init #False)" by (simp add:init_defs Valid_def)
  have 4: "\<turnstile> (\<M>(EMPTY) \<and> (init #False)) = (\<M>(EMPTY) \<and> #False)" using "3" by auto
  have 5: "\<turnstile> (\<M>(EMPTY) \<and> #False) = #False" by simp
  have 6: "\<turnstile> (\<M>(EMPTY) \<and> (init #False)) = #False" using "4" "5" by simp
  have 7: "\<turnstile> \<M>(EMPTY WITH LIFT(init #False)) = #False" using "2" "6" by fastforce
  have 8: "\<turnstile> \<M>(GUARD #False) = #False" using "1" "7" by fastforce
  from 8 show ?thesis by auto
qed

lemma MFirstFalseEqvFalse:
 "\<turnstile>  \<M>(FIRST LIFT #False) = #False"
proof -
  have 1: "\<turnstile> \<M>(FIRST LIFT #False) = \<rhd> #False" by (simp )
  have 2:  "\<turnstile> \<M>(FIRST LIFT #False) = #False" using FstFalse  by fastforce
 from 2 show ?thesis by auto
qed

lemma MFailAlt:
 "\<turnstile>  \<M> FAIL = #False"
proof -
  have 1: "\<turnstile>  \<M> FAIL =  \<M> (GUARD (#False)) " by (simp add: FAIL_d_def)
  have 2: "\<turnstile> \<M>(GUARD (#False)) = #False" using MGuardFalseEqvFalse by auto
 from 1 2 show ?thesis by fastforce
qed

lemma MFailEqvFirstFalseWithinEmpty:
 "  FAIL \<simeq>  ((FIRST LIFT #False) WITHIN empty) "
proof -
  have 1: "\<turnstile> \<M> ( (FIRST LIFT #False) WITHIN ( empty )) =  
             \<M>((FIRST LIFT #False) WITH LIFT(Limit empty) )" 
       by (simp add: WITHIN_d_def)
  have 2: "\<turnstile>  \<M>((FIRST LIFT #False) WITH LIFT(Limit empty) ) =  
              (\<M>(FIRST LIFT #False) \<and> (Limit empty )) " 
       by (simp )
  have 3: "\<turnstile>  \<M>((FIRST LIFT #False) WITH LIFT(Limit empty) ) = #False" 
       using MFirstFalseEqvFalse by auto
  have 4: "\<turnstile> \<M>( (FIRST LIFT #False) WITHIN ( empty )) = #False" 
       using "1" "3" by fastforce
  have 5: "\<turnstile> \<M>( FAIL) = #False" 
       using MFailAlt by simp
 from 4 5 show ?thesis using MonEq by (metis int_eq)
 qed

lemma MEmptyAlt:
 "\<turnstile>  \<M> EMPTY = empty"
proof -
  have 1: "\<turnstile> \<M> ( EMPTY) = \<M>( (FIRST LIFT empty))" by (simp add: EMPTY_d_def)
  have 2: "\<turnstile> \<M> ( (FIRST LIFT empty)) = \<rhd> empty" by (simp)
  have 3: "\<turnstile> \<rhd> empty = empty" using FstEmpty by auto
 from 1 2 3 show ?thesis by fastforce
qed

lemma MSkipAlt:
 "\<turnstile>  \<M> SKIP = skip"
proof -
  have 1: "\<turnstile>  \<M> SKIP = \<M> (FIRST LIFT skip)" by (simp add: SKIP_d_def)
  have 2: "\<turnstile> \<M> (FIRST LIFT skip) = \<rhd> skip" by (simp)
  have 3: "\<turnstile> \<rhd> skip = skip" using FstSkip by simp
 from 1 2 3 show ?thesis by fastforce
qed

lemma MGuardAlt:
 "\<turnstile>  \<M> (GUARD(w)) = (empty \<and> init w)"
proof -
  have 1: "\<turnstile>  \<M>(GUARD(w)) = \<M>(EMPTY WITH ( LIFT (init w)))" by (simp add:GUARD_d_def)
  have 2: "\<turnstile> \<M>(EMPTY WITH ( LIFT(init w))) = (\<M>( EMPTY) \<and> ( init w))" by (simp)
  have 3: "\<turnstile> (\<M>( EMPTY) \<and> ( init w)) = (empty \<and> ( init w))" using MEmptyAlt by fastforce
  have 4: "\<turnstile>  (empty \<and> ( init w)) = (empty \<and> init w)" by simp
 from 1 2 3 4 show ?thesis by fastforce
qed

lemma MLengthAlt:
 "\<turnstile> \<M> (LEN(k)) = len(k)"
proof -
  have 1: "\<turnstile> \<M>(LEN(k)) = \<M>(FIRST LIFT(len(k)))" by (simp add:LEN_d_def)
  have 2: "\<turnstile> \<M>(FIRST LIFT(len(k))) = \<rhd>(len(k))" by (simp)
  have 3: "\<turnstile> \<rhd>(len(k)) = len(k)" using FstLenEqvLen by blast
 from 1 2 3 show ?thesis by fastforce
qed

lemma MAlwaysAlt:
  "\<turnstile> \<M>(a ALWAYS w) = (\<M>(a) \<and> \<box> (init w))"
proof -
  have 1: "\<turnstile> \<M>(a ALWAYS w) = \<M>(a WITH LIFT(bi (finite\<longrightarrow>fin (init w))))" 
       by (simp add: ALWAYS_d_def)
  have 2: "\<turnstile> \<M>(a WITH LIFT(bi (finite\<longrightarrow>fin (init w)))) = (\<M>(a) \<and> (bi (finite\<longrightarrow>fin (init w))))" 
       by (simp)
  have 3: "\<turnstile> (\<M>(a) \<and> (bi (finite\<longrightarrow>fin (init w)))) = (\<M>(a) \<and> \<box> (init w))" 
      using BoxStateEqvBiFinState by fastforce 
  from 1 2 3 show ?thesis by fastforce
qed


lemma MSometimeAlt:
 "\<turnstile> \<M>(a SOMETIME w) = (\<M>(a) \<and> \<diamond> (init w))"
proof -
  have 1: "\<turnstile> \<M>(a SOMETIME w) = \<M>(a WITH LIFT(di (finite \<and> fin (init w))))" 
       by (simp add: SOMETIME_d_def)
  have 2: "\<turnstile> \<M>(a WITH LIFT(di (finite \<and> fin (init w)))) = (\<M>(a) \<and> (di (finite \<and> fin (init w))))" 
       by (simp)
  have 3: "\<turnstile> \<M>(a WITH LIFT(di (finite \<and> fin (init w)))) = (\<M>(a) \<and>  \<diamond> (init w))" 
       using DiamondStateEqvDiFinState  by fastforce
  from 1 2 3 show ?thesis by fastforce
qed

lemma MWithinAlt:
 "\<turnstile> \<M>(a WITHIN f) = (\<M>(a) \<and> (bs (\<not> f)))"
proof -
  have 1: "\<turnstile> \<M>(a WITHIN f) = \<M>(a WITH LIFT(bs (\<not> f)))" 
       by (simp add: WITHIN_d_def LIMIT_d_def)
  have 2: "\<turnstile> \<M>(a WITH LIFT(bs (\<not> f))) = (\<M>(a) \<and> (bs (\<not> f)))" 
       by (simp)
  from 1 2 show ?thesis by fastforce
qed

lemma MTimesAlt:
 "\<turnstile> \<M>(a TIMES k) = fpower (\<M>(a)) k"
proof 
 (induct k)
 case 0
 then show ?case 
  proof -
   have 1: "\<turnstile>  \<M> (a TIMES 0) =  \<M> EMPTY" by simp
   have 2: "\<turnstile>  \<M> EMPTY = empty " using MEmptyAlt by simp
   have 3: "\<turnstile> empty = fpower (\<M> a) 0" by (simp add: fpower_d_def)
   from 1 2 3 show ?thesis by fastforce
  qed
 next
 case (Suc k)
 then show ?case 
  proof -
   have 1: "\<turnstile>  \<M>( a TIMES Suc k) = \<M>(a THEN (a TIMES k) ) " 
        by simp
   have 2: "\<turnstile> \<M>(a THEN (a TIMES k) ) = (\<M> a)\<frown>(\<M> (a TIMES k))" 
        by (simp)
   have 3: "\<turnstile> (\<M> a)\<frown>(\<M>(a TIMES k)) = (\<M> a)\<frown>(fpower (\<M> a) k)" 
        using RightSChopEqvSChop Suc.hyps by blast
   have 4: "\<turnstile> (\<M> a)\<frown>(fpower (\<M> a) k) = fpower (\<M> a) (Suc k)" 
      by (simp add: fpower_d_def schop_d_def)  
   from 1 2 3 4 show ?thesis by fastforce
  qed
 qed

lemma MUptoAlt:
 "\<turnstile> \<M>(a UPTO b) = 
    (( (\<M> a) \<and> bi (\<not>(\<M> b)) \<and> finite) \<or> ((\<M> b) \<and> bi (\<not>(\<M> a)) \<and> finite) \<or> ((\<M> a) \<and> (\<M> b))) "
proof -
  have 1: "\<turnstile> \<M> (a UPTO b) = \<rhd>((\<M> a) \<or> (\<M> b))" 
        by (simp)
  have 2: "\<turnstile> \<rhd>((\<M> a) \<or> (\<M> b)) = ((\<rhd>(\<M> a) \<and>  (bs (\<not>(\<M> b)))) \<or> (\<rhd>(\<M> b) \<and>  (bs (\<not>(\<M> a))))) " 
       using FstWithOrEqv by blast
  have 3: "\<turnstile> ((\<rhd>(\<M> a) \<and>  (bs (\<not>(\<M> b)))) \<or> (\<rhd>(\<M> b) \<and>  (bs (\<not>(\<M> a))))) =
            (((\<M> a) \<and> ((\<M> b) \<or> \<not>(\<M> b)) \<and>  (bs (\<not>(\<M> b)))) \<or> 
            ((\<M> b) \<and> ((\<M> a) \<or> \<not>(\<M> a)) \<and>  (bs (\<not>(\<M> a)))))"  
       using MFixFst by fastforce
  have 4: "\<turnstile> (((\<M> a) \<and> ((\<M> b) \<or> \<not>(\<M> b)) \<and>  (bs (\<not>(\<M> b)))) \<or> 
            ((\<M> b) \<and> ((\<M> a) \<or> \<not>(\<M> a)) \<and>  (bs (\<not>(\<M> a))))) = 
            (((\<M> a) \<and> ( ((\<M> b) \<and> bs (\<not>(\<M> b))) \<or> (\<not>(\<M> b) \<and> bs (\<not>(\<M> b)))) ) \<or>
            ((\<M> b) \<and> ( ((\<M> a) \<and> bs (\<not>(\<M> a))) \<or> (\<not>(\<M> a) \<and> bs (\<not>(\<M> a)))) ))" 
       by auto
  have 5: "\<turnstile> (((\<M> a) \<and> ( ((\<M> b) \<and> bs (\<not>(\<M> b))) \<or> (\<not>(\<M> b) \<and> bs (\<not>(\<M> b)))) ) \<or>
            ((\<M> b) \<and> ( ((\<M> a) \<and> bs (\<not>(\<M> a))) \<or> (\<not>(\<M> a) \<and> bs (\<not>(\<M> a)))) )) =
            (((\<M> a) \<and> ( (\<rhd>(\<M> b)) \<or> (\<not>(\<M> b) \<and> bs (\<not>(\<M> b)))) ) \<or>
            ((\<M> b) \<and> ( (\<rhd>(\<M> a)) \<or> (\<not>(\<M> a) \<and> bs (\<not>(\<M> a)))) ))" 
       by (simp add: first_d_def)
  have 6: "\<turnstile> (((\<M> a) \<and> ( (\<rhd>(\<M> b)) \<or> (\<not>(\<M> b) \<and> bs (\<not>(\<M> b)))) ) \<or>
            ((\<M> b) \<and> ( (\<rhd>(\<M> a)) \<or> (\<not>(\<M> a) \<and> bs (\<not>(\<M> a)))) )) =
            (((\<M> a) \<and> ( ((\<M> b)) \<or> (\<not>(\<M> b) \<and> bs (\<not>(\<M> b)))) ) \<or>
            ((\<M> b) \<and> ( ((\<M> a)) \<or> (\<not>(\<M> a) \<and> bs (\<not>(\<M> a)))) )) " 
       using MFixFst by fastforce
  have 7: "\<turnstile> (\<not>(\<M> b) \<and> bs (\<not>(\<M> b))) = (bi(\<not>(\<M> b)) \<and> finite)" 
       using AndBsEqvBi by blast
  have 8: "\<turnstile> (\<not>(\<M> a) \<and> bs (\<not>(\<M> a))) = (bi(\<not>(\<M> a)) \<and> finite)" 
       using AndBsEqvBi by blast
  have 9: "\<turnstile> (((\<M> a) \<and> ( ((\<M> b)) \<or> ((\<not>(\<M> b)) \<and> bs(\<not>(\<M> b)))) ) \<or>
            ((\<M> b) \<and> ( ((\<M> a)) \<or> ((\<not>(\<M> a)) \<and> bs(\<not>(\<M> a)))) )) =
            (((\<M> a) \<and> ( ((\<M> b)) \<or> ( bi(\<not>(\<M> b)) \<and> finite   )) ) \<or>
            ((\<M> b) \<and> ( ((\<M> a)) \<or> ( bi(\<not>(\<M> a)) \<and> finite)) ))" 
       using "7" "8" by fastforce
  have 10: "\<turnstile>(((\<M> a) \<and> ( ((\<M> b)) \<or> ( bi(\<not>(\<M> b)) \<and> finite)) ) \<or>
            ((\<M> b) \<and> ( ((\<M> a)) \<or> ( bi(\<not>(\<M> a)) \<and> finite)) )) = 
            ((( (\<M> a) \<and> (\<M> b)) \<or> ( (\<M> a) \<and> bi(\<not>(\<M> b)) \<and> finite)) \<or> 
            (( (\<M> b) \<and> (\<M> a)) \<or> ( (\<M> b) \<and> bi(\<not>(\<M> a)) \<and> finite))) "  
       by auto
  have 11: "\<turnstile> ((( (\<M> a) \<and> (\<M> b)) \<or> ( (\<M> a) \<and> bi(\<not>(\<M> b)) \<and> finite)) \<or> 
            (( (\<M> b) \<and> (\<M> a)) \<or> ( (\<M> b) \<and> bi(\<not>(\<M> a)) \<and> finite))) =
           (( (\<M> a) \<and> bi (\<not>(\<M> b)) \<and> finite) \<or> ((\<M> b) \<and> bi (\<not>(\<M> a)) \<and> finite) \<or> ((\<M> a) \<and> (\<M> b))) " 
       by auto
  show ?thesis 
  by (metis "10" "11" "2" "3" "4" "5" "6" "9" MON.simps(2) int_eq) 
qed

lemma MThruAlt:
 "\<turnstile> \<M>(a THRU b) = (((\<M> a) \<and> di(\<M> b)) \<or> ((\<M> b) \<and> di(\<M> a)))"
proof -
  have 1: "\<turnstile> \<M>(a THRU b) = \<rhd>(di(\<M> a) \<and> di(\<M> b))" 
       by (simp)
  have 2: "\<turnstile> \<rhd>(di(\<M> a) \<and> di(\<M> b)) = ((\<rhd>(\<M> a) \<and> di(\<M> b)) \<or> (\<rhd>(\<M> b) \<and> di(\<M> a)))" 
       using FstDiAndDiEqv by auto
  have 3: "\<turnstile> ((\<rhd>(\<M> a) \<and> di(\<M> b)) \<or> (\<rhd>(\<M> b) \<and> di(\<M> a))) =
            (((\<M> a) \<and> di(\<M> b)) \<or> ((\<M> b) \<and> di(\<M> a)))" 
       using MFixFst by fastforce
 from 1 2 3 show ?thesis by fastforce
qed

lemma MHaltAlt:
 "\<turnstile> \<M>(HALT w) = (halt(init w) \<and> finite)"
proof -
  have 1: "\<turnstile> \<M>(HALT w) = \<M>(FIRST LIFT(fin (init  w)))" by (simp add: HALT_d_def)
  have 2: "\<turnstile> \<M>(FIRST LIFT(fin (init  w))) = \<rhd> (fin (init  w))" by (simp)
  have 3: "\<turnstile> \<rhd> (fin (init  w)) = (halt(init w) \<and> finite)" using HaltStateEqvFstFinState by fastforce
  from 1 2 3 show ?thesis by fastforce
qed

lemma MFailUpto:
"(FAIL UPTO a) \<simeq> ( a)"
proof -
  have 1: "\<turnstile>  \<M>(FAIL UPTO a) = \<rhd>( (\<M> FAIL) \<or> (\<M> a))" by (simp)
  have 2: "\<turnstile>  (\<M> FAIL \<or> \<M> a) = (#False \<or> \<M> a)" using MFailAlt by auto
  have 3: "\<turnstile> \<rhd>(\<M> FAIL \<or> (\<M> a)) = \<rhd>(#False \<or> (\<M> a))" using 2 FstEqvRule by blast
  have 4: "\<turnstile> (#False \<or> (\<M> a)) = \<M> a" by simp
  have 5: "\<turnstile> \<rhd>(#False \<or> (\<M> a)) =  \<rhd>(\<M> a)" using 4 FstEqvRule by blast
  have 6: "\<turnstile> \<rhd>(\<M> a) = \<M> a" using MFixFst by fastforce
  from 1 2 3 4 5 6 show ?thesis using MonEq by (metis int_eq)
qed

lemma MFailThru:
 " (FAIL THRU ( a)) \<simeq>   FAIL"
proof -
  have 1: "\<turnstile> \<M> (FAIL THRU ( a)) = \<rhd>(di(\<M> FAIL) \<and> di(\<M> a))" 
       by (simp)
  have 2: "\<turnstile> \<rhd>(di (\<M> FAIL) \<and> di(\<M> a)) = \<rhd>(di (#False) \<and> di(\<M> a)) " 
       using MFailAlt by (metis "1" int_eq)
  have 3: "\<turnstile> di #False = #False" 
       by (simp add: di_defs Valid_def)
  hence 4: "\<turnstile> \<rhd>(di (#False) \<and> di(\<M> a)) = \<rhd>( (#False) \<and> di(\<M> a))" 
       by (metis "2" inteq_reflection)  
  have 5: "\<turnstile> \<rhd>( (#False) \<and> di(\<M> a)) = \<rhd>#False" 
       using FstEqvRule by fastforce
  have 6: "\<turnstile> \<rhd>#False = #False" using FstFalse 
       by auto
  have 7: "\<turnstile> #False = \<M> FAIL" 
       using MFailAlt by auto
 from 1 2 4 5 6 7 show ?thesis using MonEq by (metis int_eq)
qed

lemma MFailAnd:
" (FAIL AND a) \<simeq>   FAIL"
proof -
  have 1: "\<turnstile>  \<M> (FAIL AND a) = (\<M> FAIL \<and> (\<M> a))" by (simp add: AND_d_def)
  have 2: "\<turnstile> (\<M> FAIL \<and> (\<M> a)) = (#False \<and> (\<M> a))" using MFailAlt by fastforce
  have 3: "\<turnstile> (#False \<and> (\<M> a)) = #False" by auto
  have 4: "\<turnstile>  \<M>(FAIL AND a) = #False" using 1 2 3 by fastforce
  have 5: "\<turnstile> #False =  \<M> FAIL" using MFailAlt by auto
  from 1 2 3 4 5 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThenFail:
 " (a THEN FAIL) \<simeq> FAIL"
proof -
  have 1: "\<turnstile>  \<M> (a THEN FAIL) = (\<M> a)\<frown>(\<M> FAIL)" by (simp)
  have 2: "\<turnstile> (\<M> a)\<frown>(\<M> FAIL) = (\<M> a)\<frown>#False " by (simp add: MFailAlt RightSChopEqvSChop)
  have 3: "\<turnstile> (\<M> a)\<frown>#False = #False"by (simp add: SChopRightFalse)
  have 4: "\<turnstile> #False =  \<M> FAIL" using MFailAlt by auto
 from 1 2 3 4 show ?thesis using MonEq by (metis int_eq)
qed

lemma MFailThen:
 " ( FAIL THEN a) \<simeq>  FAIL"
proof -
  have 1: "\<turnstile> \<M>( FAIL THEN a) = (\<M> FAIL)\<frown>(\<M> a)" by (simp)
  have 2: "\<turnstile> (\<M> FAIL)\<frown>(\<M> a) = #False\<frown>(\<M> a) " using MFailAlt using LeftSChopEqvSChop by blast
  have 3: "\<turnstile> #False\<frown>(\<M> a) = #False" by (simp add: schop_d_def chop_d_def Valid_def)
  have 4: "\<turnstile> #False = \<M> FAIL" using MFailAlt by auto
 from 1 2 3 4 show ?thesis  using MonEq by (metis int_eq)
qed

lemma MFailWith:
 " ( FAIL WITH f) \<simeq>  FAIL"
proof -
  have 1: "\<turnstile> \<M> (FAIL WITH f) = ((\<M> FAIL) \<and> f)" by (simp)
  have 2: "\<turnstile> ((\<M> FAIL) \<and> f) = (#False \<and> f)" using MFailAlt by auto
  have 3: "\<turnstile> (#False \<and> f) = #False" by simp
  have 4: "\<turnstile> #False =  \<M> FAIL" using MFailAlt by auto
 from 1 2 3 4 show ?thesis using MonEq by (metis int_eq)
qed

lemma MWithFalse:
 " (a WITH (LIFT(#False))) \<simeq>  FAIL"
proof -
  have 1: "\<turnstile> \<M> (a WITH LIFT(#False)) = ((\<M> a) \<and> #False)" by (simp)
  have 2: "\<turnstile> ((\<M> a) \<and> #False) = \<M> FAIL" using MFailAlt by auto
 from 1 2 show ?thesis using MonEq by (metis int_eq)
qed

lemma MWithTrue:
 " (a WITH (LIFT(#True))) \<simeq>  a"
proof -
  have 1: "\<turnstile> \<M> (a WITH LIFT(#True)) = ((\<M> a) \<and> #True)" by (simp)
  have 2: "\<turnstile> ((\<M> a) \<and> #True) = \<M> a" by simp
 from 1 2 show ?thesis using MonEq by (metis int_eq)
qed

lemma MEmptyUpto:
 " (EMPTY UPTO a) \<simeq>  EMPTY"
proof -
  have 1: "\<turnstile>  \<M> (EMPTY UPTO a) = \<rhd>(\<M> EMPTY \<or> (\<M> a))" by (simp)
  have 2: "\<turnstile> \<M> EMPTY  = empty " using MEmptyAlt by auto
 hence 3: "\<turnstile>  (\<M> EMPTY \<or> (\<M> a)) = (empty \<or> (\<M> a))" by auto
 hence 4: "\<turnstile>  \<rhd>(\<M> EMPTY \<or> \<M> a) =  \<rhd>(empty \<or> \<M> a)" using FstEqvRule by blast
  have 5: "\<turnstile> \<rhd>(empty \<or> \<M> a) = empty" using FstEmptyOrEqvEmpty by blast
  have 6: "\<turnstile> empty = \<M> EMPTY" using MEmptyAlt by auto
 from 1 4 5 6 show ?thesis using MonEq by (metis int_eq)
qed

lemma MEmptyThru:
 " (EMPTY THRU a) \<simeq> ( a)"
proof -
  have 1: "\<turnstile>  \<M>(EMPTY THRU a) = \<rhd>(di(\<M> EMPTY) \<and> di(\<M> a))" by (simp)
  have 2: "\<turnstile> di(\<M> EMPTY) = di empty " using MEmptyAlt DiEqvDi by blast
 hence 3: "\<turnstile> (di(\<M> EMPTY) \<and> di(\<M> a)) = (di empty \<and> di(\<M> a))" by auto
 hence 4: "\<turnstile> (di empty \<and> di(\<M> a)) = di(\<M> a)" using DiEmpty by auto
  have 5: "\<turnstile> (di(\<M> EMPTY) \<and> di(\<M> a)) = di(\<M> a)" using "3" "4" by fastforce
 hence 6: "\<turnstile> \<rhd>(di(\<M> EMPTY) \<and> di(\<M> a)) = \<rhd>(di(\<M> a))" using FstEqvRule by blast
  have 7: "\<turnstile> \<rhd>(di(\<M> a)) = \<rhd>(\<M> a)" using FstDiEqvFst by blast
  have 8: "\<turnstile> \<rhd>(\<M> a) = (\<M> a)" using MFixFst by fastforce
 from 1 6 7 8 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThenEmpty: 
 "( a THEN EMPTY) \<simeq> (a WITH (LIFT finite))"
proof -
  have 1: "\<turnstile> \<M>( a THEN EMPTY) = (\<M> a)\<frown> (\<M> EMPTY)" by (simp)
  have 2: "\<turnstile> (\<M> a)\<frown> (\<M> EMPTY) = (\<M> a)\<frown> empty" by (simp add: MEmptyAlt RightSChopEqvSChop)
  have 3: "\<turnstile> (\<M> a)\<frown> empty = ((\<M> a) \<and> finite)" by (simp add: ChopEmpty schop_d_def)
 from 1 2 3 show ?thesis using MonEq by (metis MON.simps(5) inteq_reflection)
qed

lemma MEmptyThen: 
 "( EMPTY THEN a) \<simeq> a"
proof -
  have 1: "\<turnstile> \<M>( EMPTY THEN a) = (\<M> EMPTY)\<frown>(\<M> a)" by (simp)
  have 2: "\<turnstile> (\<M> EMPTY)\<frown>(\<M> a) = empty\<frown>(\<M> a)" by (simp add: MEmptyAlt LeftSChopEqvSChop)
  have 3: "\<turnstile> empty\<frown>(\<M> a) = (\<M> a)" by (simp add: EmptySChop)
 from 1 2 3 show ?thesis using MonEq by (metis int_eq)
qed

lemma MEmptyIterate: 
 "( EMPTY ITERATE b) \<simeq>  EMPTY"
proof -
  have 1: "\<turnstile> \<M>( EMPTY ITERATE b) = \<M>( EMPTY WITH LIFT (schopstar (\<M> b)))" 
       by (simp add: ITERATE_d_def)
  have 2: "\<turnstile> \<M> (EMPTY WITH LIFT (schopstar (\<M> b))) = (\<M> EMPTY \<and> (schopstar (\<M> b)))" 
       by (simp)
  have 3: "\<turnstile> (\<M> EMPTY \<and> (schopstar (\<M> b)))  = (empty \<and> (schopstar (\<M> b))) " 
       using MEmptyAlt by auto
  have 4: "\<turnstile> (empty \<and> (schopstar (\<M> b)))  = (empty \<and> (empty \<or> (((\<M> b) \<and> more)\<frown>(schopstar (\<M> b)))))" 
       using SChopstarEqv by fastforce
  have 5: "\<turnstile> (empty \<and> (empty \<or> (((\<M> b) \<and> more)\<frown>(schopstar (\<M> b))))) = empty" 
       by auto
  have 6: "\<turnstile>  \<M>(EMPTY ITERATE b) =  \<M> EMPTY" 
       using 1 2 3 4 5 MEmptyAlt by fastforce
 from 6 show ?thesis using MonEq by (metis int_eq)
qed

lemma MIterateIdemp: 
 "(a ITERATE a) \<simeq>  (a)"
proof -
  have 1: "\<turnstile>  \<M>(a ITERATE  a) =  \<M> (a WITH LIFT (schopstar (\<M> a)))" by (simp add: ITERATE_d_def)
  have 2: "\<turnstile>  \<M>(a WITH LIFT (schopstar (\<M> a))) = ((\<M> a) \<and> (schopstar (\<M> a)))"  by (simp)
  have 3: "\<turnstile> ((\<M> a) \<and> (schopstar (\<M> a))) = (\<rhd>(\<M> a) \<and> (schopstar (\<rhd>(\<M> a)))) " using MFixFst
    by (metis "2" inteq_reflection) 
  have 4: "\<turnstile> (\<rhd>(\<M> a) \<and> (schopstar (\<rhd>(\<M> a)))) = \<rhd>(\<M> a)" using FstAndFstStarEqvFst by fastforce
  have 5: "\<turnstile> \<rhd>(\<M> a) = \<M> a" using MFixFst by fastforce
  from 1 2 3 4 5 show ?thesis using MonEq by (metis int_eq)
qed

lemma MUptoIdemp:
 "(a UPTO a) \<simeq>  (a)"
proof -
  have 1: "\<turnstile>  \<M>(a UPTO a) = \<rhd>((\<M> a) \<or> (\<M> a))"  by auto
  have 2: "\<turnstile> \<rhd>((\<M> a) \<or> (\<M> a)) = \<rhd>(\<M> a)" using FstEqvRule by fastforce
  have 3: "\<turnstile> \<rhd>(\<M> a) = (\<M> a)" using MFixFst by fastforce
 from 1 2 3 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThruIdemp:
 "( a THRU a) \<simeq> (a)"
proof -
  have 1: "\<turnstile>  \<M>( a THRU a) = \<rhd>( di(\<M> a) \<and> di(\<M> a))"  by auto
  have 2: "\<turnstile> \<rhd>( di(\<M> a) \<and> di(\<M> a)) = \<rhd>(di (\<M> a))" using FstEqvRule by fastforce
  have 3: "\<turnstile> \<rhd>(di (\<M> a)) = \<rhd>(\<M> a)" using FstDiEqvFst by blast
  have 4: "\<turnstile> \<rhd>(\<M> a) = (\<M> a)" using MFixFst by fastforce
 from 1 2 3 4 show ?thesis using MonEq by (metis int_eq)
qed

lemma MAndIdemp:
 "(a AND a) \<simeq> (a)"
proof -
  have 1: "\<turnstile> \<M>(a AND a) = ((\<M> a) \<and> (\<M> a))" by (simp add: AND_d_def)
  have 2: "\<turnstile> ((\<M> a) \<and> (\<M> a)) = (\<M> a)" by fastforce
 from 1 2 show ?thesis using MonEq by (metis int_eq)
qed

lemma MWithIdemp:
 "( (a WITH f)  WITH f) \<simeq> (a WITH f)"
proof -
  have 1: "\<turnstile> \<M>( (a WITH f)  WITH f) = (((\<M> a) \<and> ( f)) \<and> ( f))" by auto
  have 2: "\<turnstile> (((\<M> a) \<and> ( f)) \<and> ( f)) = ((\<M> a) \<and> ( f))" by fastforce
  have 3: "\<turnstile> ((\<M> a) \<and> ( f)) = \<M>(a WITH f)"  by auto
 from 1 2 3 show ?thesis using MonEq by (metis int_eq)
qed

lemma MUptoCommut:
 "(a UPTO b) \<simeq> (b UPTO a)"
proof -
  have 1: "\<turnstile> \<M>(a UPTO b) = \<rhd>((\<M> a) \<or> (\<M> b))" by (simp)
  have 2: "\<turnstile> ((\<M> a) \<or> (\<M> b)) = ((\<M> b) \<or> (\<M> a))" by auto
 hence 3: "\<turnstile> \<rhd>((\<M> a) \<or> (\<M> b)) = \<rhd>((\<M> b) \<or> (\<M> a)) " using FstEqvRule by blast
  have 4: "\<turnstile> \<rhd>((\<M> b) \<or> (\<M> a)) = \<M>(b UPTO a)" by auto
 from 1 3 4 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThruCommut:
 "(a THRU b) \<simeq> (b THRU a)"
proof -
  have 1: "\<turnstile> \<M>(a THRU b) = \<rhd>(di(\<M> a) \<and> di(\<M> b))" by (simp)
  have 2: "\<turnstile> (di(\<M> a) \<and> di(\<M> b)) = (di(\<M> b) \<and> di(\<M> a)) " by auto
 hence 3: "\<turnstile> \<rhd>(di(\<M> a) \<and> di(\<M> b)) = \<rhd>(di(\<M> b) \<and> di(\<M> a))" using FstEqvRule by blast
  have 4: "\<turnstile> \<rhd>(di(\<M> b) \<and> di(\<M> a)) = \<M>(b THRU a)" by auto
 from 1 3 4 show ?thesis using MonEq by (metis int_eq)
qed

lemma MAndCommut:
 "(a AND b) \<simeq> (b AND a)"
proof -
  have 1: "\<turnstile> \<M>(a AND b) = ((\<M> a) \<and> (\<M> b))" by (simp add: AND_d_def)
  have 2: "\<turnstile> ((\<M> a) \<and> (\<M> b)) = ((\<M> b) \<and> (\<M> a)) " by auto
  have 3: "\<turnstile> ((\<M> b) \<and> (\<M> a)) = \<M>(b AND a)" by (simp add: AND_d_def)
 from 1 2 3 show ?thesis using MonEq by (metis int_eq)
qed

lemma MWithCommut:
 "((a WITH f) WITH g) \<simeq> ((a WITH g) WITH f) "
proof -
  have 1: "\<turnstile> \<M>((a WITH f) WITH g) = (((\<M> a) \<and> (f)) \<and> (g))"  by auto
  have 2: "\<turnstile> (((\<M> a) \<and> (f)) \<and> (g)) = (((\<M> a) \<and> (g)) \<and> (f))" by auto
  have 3: "\<turnstile> (((\<M> a) \<and> (g)) \<and> (f)) = \<M>((a WITH g) WITH f)" by auto
 from 1 2  3 show ?thesis using MonEq by (metis int_eq)
qed

lemma MWithAbsorp:
 "((a WITH f) WITH g) \<simeq> (a WITH LIFT(f \<and> g)) "
proof -
  have 1: "\<turnstile> \<M>((a WITH f) WITH g) = (((\<M> a) \<and> (f)) \<and> (g))" by auto
  have 2: "\<turnstile> (((\<M> a) \<and> (f)) \<and> (g)) =  ((\<M> a) \<and> (f \<and> g))" by auto
 from 1 2 show ?thesis by (simp add: MonEq)
qed

lemma MUptoAssoc:
 "((a UPTO b) UPTO c) \<simeq> (a UPTO (b UPTO c))"
proof -
  have 1: "\<turnstile> \<M>((a UPTO b) UPTO c) = \<rhd>(\<M>(a UPTO b) \<or> (\<M> c))" 
       by (simp)
  have 2: "\<turnstile> \<rhd>(\<M>(a UPTO b) \<or> (\<M> c)) = \<rhd>(\<rhd>((\<M> a) \<or> (\<M> b)) \<or> (\<M> c))" 
       by auto
  have 3: "\<turnstile> \<rhd>(\<rhd>((\<M> a) \<or> (\<M> b)) \<or> (\<M> c)) = \<rhd>(((\<M> a) \<or> (\<M> b)) \<or> (\<M> c))"  
       using FstFstOrEqvFstOrL by blast
  have 4: "\<turnstile> (((\<M> a) \<or> (\<M> b)) \<or> (\<M> c)) = ((\<M> a) \<or> ((\<M> b) \<or> (\<M> c)))" 
       by auto
  hence 5: "\<turnstile> \<rhd>(((\<M> a) \<or> (\<M> b)) \<or> (\<M> c)) = \<rhd>((\<M> a) \<or> ((\<M> b) \<or> (\<M> c)))"  
       using FstEqvRule by blast
  have 6: "\<turnstile> \<rhd>((\<M> a) \<or> ((\<M> b) \<or> (\<M> c))) = \<rhd>((\<M> a) \<or> \<rhd>((\<M> b) \<or> (\<M> c)))" 
       using FstFstOrEqvFstOrR  by fastforce
  have 7: "\<turnstile> \<rhd>((\<M> a) \<or> \<rhd>((\<M> b) \<or> (\<M> c))) = \<rhd>((\<M> a) \<or> \<M>(b UPTO c))" 
       by auto
  have 8: "\<turnstile> \<rhd>((\<M> a) \<or> \<M>(b UPTO c)) = \<M>(a UPTO (b UPTO c))" 
       by auto
 from 1 2 3 5 6 7 8 show ?thesis using MonEq by (metis int_eq)
qed

lemma DiAndFiniteEqvDiFst: 
 "\<turnstile> di (f \<and> finite) = di(\<rhd> f) " 
proof -
 have 1: "\<turnstile> (\<rhd> f \<and> finite) = \<rhd> f" 
   by (meson FstImpFinite Prop10 Prop11)
 have 2: "\<turnstile> (f \<and> finite);#True = (\<rhd> f \<and> finite);#True" 
   using DfEqvDfFst FstImpFinite unfolding df_d_def schop_d_def by auto
 show ?thesis 
 by (metis "1" "2" di_d_def inteq_reflection)
qed   

lemma FstEqvFstAndFinite: 
 "\<turnstile> \<rhd> f = \<rhd> (f \<and> finite) " 
by (metis FstDfEqvFst FstDiEqvFst di_d_def inteq_reflection itl_def(15) schop_d_def)
 

lemma DiAndFiniteEqvDfAndFinite: 
 "\<turnstile> (di f \<and> finite) = (df f \<and> finite) " 
by (auto simp add: Valid_def di_defs df_defs finite_defs) 

lemma FstDiAndDiEqvFstDfAndDf: 
 "\<turnstile> \<rhd>(di f \<and> di g) = \<rhd>(df f \<and> df g) " 
proof -
 have 1: "\<turnstile>  \<rhd>(di f \<and> di g) = \<rhd> ((di f \<and> di g) \<and> finite) " 
   by (simp add: FstEqvFstAndFinite)
 have 2: "\<turnstile> ((di f \<and> di g) \<and> finite) = ((df f \<and> df g) \<and> finite) " 
  using DiAndFiniteEqvDfAndFinite[of f] DiAndFiniteEqvDfAndFinite[of g]
    by fastforce
 have 3: "\<turnstile> \<rhd>((di f \<and> di g) \<and> finite) = \<rhd>((df f \<and> df g) \<and> finite) "
   by (simp add: "2" FstEqvRule)
 have 4: "\<turnstile> \<rhd>((df f \<and> df g) \<and> finite) = \<rhd>(df f \<and> df g) "
  by (meson FstEqvFstAndFinite Prop11)
 show ?thesis 
 by (metis "1" "3" "4" inteq_reflection)
qed


lemma MThruAssoc:
 "((a THRU b) THRU c) \<simeq> (a THRU (b THRU c))"
proof -
  have 1: "\<turnstile> \<M>((a THRU b) THRU c) = \<rhd>(di(\<rhd>(di(\<M> a) \<and> di(\<M> b))) \<and> di(\<M> c))" 
       by auto
  have 2: "\<turnstile> di(\<rhd>(di(\<M> a) \<and> di(\<M> b))) = di( (di(\<M> a) \<and> di(\<M> b)) \<and> finite) " 
       using DiAndFiniteEqvDiFst by fastforce
  have 3: "\<turnstile> di((di(\<M> a) \<and> di(\<M> b)) \<and> finite) = (di((\<M> a) \<and> finite) \<and> di((\<M> b) \<and> finite)) " 
   by (metis DfDfAndEqvDf DiAndFiniteEqvDiFst MFixFst df_d_def di_d_def inteq_reflection schop_d_def) 
  have 4: "\<turnstile> di(\<rhd>(di(\<M> a) \<and> di(\<M> b))) = (di((\<M> a) \<and> finite) \<and> di((\<M> b) \<and> finite))" 
       using "2" "3" by fastforce
  hence 5: "\<turnstile> (di(\<rhd>(di(\<M> a) \<and> di(\<M> b)))  \<and> di((\<M> c) \<and> finite)) = 
              (di((\<M> a) \<and> finite) \<and> (di((\<M> b) \<and> finite)  \<and> di((\<M> c) \<and> finite)))" 
       by auto
  have 6: "\<turnstile> (di((\<M> b) \<and> finite)  \<and> di((\<M> c) \<and> finite)) = di ((di(\<M> b)  \<and> di(\<M> c)) \<and> finite)" 
       by (metis DfDfAndEqvDf DfEqvDiAndFinite DiAndFiniteEqvDiFst MFixFst inteq_reflection)
  have 7: "\<turnstile> di ((di(\<M> b)  \<and> di(\<M> c)) \<and> finite) = di (\<rhd>(di(\<M> b)  \<and> di(\<M> c)))" 
    by (simp add: DiAndFiniteEqvDiFst)
  have 8: "\<turnstile> (di((\<M> b) \<and> finite)  \<and> di((\<M> c) \<and> finite)) =
              di (\<rhd>(di(\<M> b)  \<and> di(\<M> c)))" 
       using "6" "7" by fastforce
  hence 9: "\<turnstile>(di((\<M> a) \<and> finite) \<and> (di((\<M> b) \<and> finite)  \<and> di((\<M> c) \<and> finite))) =
              (di((\<M> a) \<and> finite) \<and> di (\<rhd>(di(\<M> b)  \<and> di(\<M> c))))" 
       by auto
  have 10: "\<turnstile> (di(\<rhd>(di(\<M> a) \<and> di(\<M> b)))  \<and> di((\<M> c) \<and> finite)) = 
             (di((\<M> a) \<and> finite) \<and> di (\<rhd>(di(\<M> b)  \<and> di(\<M> c)))) " 
       using "5" "9" by fastforce
  hence 11: "\<turnstile> \<rhd>(di(\<rhd>(di(\<M> a) \<and> di(\<M> b)))  \<and> di((\<M> c) \<and> finite)) = 
             \<rhd>(di((\<M> a) \<and> finite) \<and> di (\<rhd>(di(\<M> b)  \<and> di(\<M> c)))) "  
       using FstEqvRule by fastforce
  have 12: "\<turnstile> \<rhd>(di((\<M> a) \<and> finite) \<and> di (\<rhd>(di(\<M> b)  \<and> di(\<M> c)))) = \<M>(a THRU (b THRU c)) " 
      by (metis DiAndFiniteEqvDiFst MFixFst MON.simps(3) inteq_reflection)
 from 1 11 12 show ?thesis using MonEq 
 by (metis DiAndFiniteEqvDiFst MFixFst inteq_reflection)
qed

lemma MAndAssoc:
 "((a AND b) AND c) \<simeq> (a AND (b AND c))"
proof -
  have 1: "\<turnstile> \<M>((a AND b) AND c) = ((\<M> a) \<and> (\<M> b) \<and> (\<M> c))" 
       using AND_d_def by (metis MON.simps(5) MWithAbsorp eq_d_def)
  have 2: "\<turnstile> ((\<M> a) \<and> (\<M> b) \<and> (\<M> c)) = \<M> (a AND (b AND c))" 
       using AND_d_def by (simp add: AND_d_def)
 from 1 2 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThenAssoc:
 "((a THEN b) THEN c) \<simeq> (a THEN (b THEN c))"
proof -
  have 1: "\<turnstile> \<M>((a THEN b) THEN c) = ((\<M> a)\<frown>(\<M> b))\<frown>(\<M> c) " by auto
  have 2: "\<turnstile> ((\<M> a)\<frown>(\<M> b))\<frown>(\<M> c) = (\<M> a)\<frown>((\<M> b)\<frown>(\<M> c))" by (meson Prop11 SChopAssoc)
  have 3: "\<turnstile> (\<M> a)\<frown>((\<M> b)\<frown>(\<M> c)) = \<M>(a THEN (b THEN c))" by auto
 from 1 2 3 show ?thesis using MonEq by (metis int_eq)
qed

lemma MUptoThruAbsorp:
 "(a UPTO (a THRU b)) \<simeq>  a"
proof -
  have 1: "\<turnstile> \<M>(a UPTO (a THRU b)) = \<rhd>((\<M> a) \<or> \<rhd>(di(\<M> a) \<and> di(\<M> b) ))" 
       by simp
  have 2: "\<turnstile> \<rhd>((\<M> a) \<or> \<rhd>(di(\<M> a) \<and> di(\<M> b) )) =
            \<rhd>((\<M> a) \<or> (di(\<M> a) \<and> di(\<M> b) ))" 
       using FstFstOrEqvFstOrR by auto
  have 3: "\<turnstile>  ((\<M> a) \<or> (di(\<M> a) \<and> di(\<M> b) )) =
             (((\<M> a) \<or> di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)))" 
       by auto
  have 4: "\<turnstile> (((\<M> a) \<or> di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b))) =
            ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)))" 
       using OrDiEqvDi by fastforce
  have 5: "\<turnstile> ((\<M> a) \<or> (di(\<M> a) \<and> di(\<M> b) )) =
            ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)))" 
       using "3" "4" by auto
  hence 6: "\<turnstile> \<rhd> ((\<M> a) \<or> (di(\<M> a) \<and> di(\<M> b) )) =
             \<rhd> ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b))) " 
       using FstEqvRule by blast
  have 7: "\<turnstile> \<rhd> ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b))) =
            ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and> 
            bs (\<not>( (di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b))))) " 
       by (auto simp add: first_d_def)
  have 8: "\<turnstile> ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b))) = 
            ((di(\<M> a) \<and> (\<M> a)) \<or> (di(\<M> a) \<and> di(\<M> b)))" 
       by auto
  hence 9: "\<turnstile> (\<not>((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)))) = 
            (\<not>((di(\<M> a) \<and> (\<M> a)) \<or> (di(\<M> a) \<and> di(\<M> b))))" 
       by fastforce
  have 10: "\<turnstile> (\<not>((di(\<M> a) \<and> (\<M> a)) \<or> (di(\<M> a) \<and> di(\<M> b)))) =
             (\<not>(((\<M> a)) \<or> (di(\<M> a) \<and> di(\<M> b))))" 
       using AndDiEqv using "5" by auto
  have 11: "\<turnstile> (\<not>(((\<M> a)) \<or> (di(\<M> a) \<and> di(\<M> b)))) =
              (\<not>(\<M> a) \<and> \<not>(di(\<M> a) \<and> di(\<M> b)))" 
       by auto
  have 12: "\<turnstile> (\<not>((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)))) = 
              (\<not>(\<M> a) \<and> \<not>(di(\<M> a) \<and> di(\<M> b)))" 
       using "9" "10" "11" by auto
  hence 13: "\<turnstile> bs (\<not>((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)))) = 
              bs (\<not>(\<M> a) \<and> \<not>(di(\<M> a) \<and> di(\<M> b)))" 
       using BsEqvRule by blast
  have 14: "\<turnstile> bs ((\<not>(\<M> a)) \<and> \<not>(di(\<M> a) \<and> di(\<M> b))) =
             (bs ((\<not>(\<M> a))) \<and> bs(\<not>(di(\<M> a) \<and> di(\<M> b))))" 
       using BsAndEqv by fastforce
  have 141: "\<turnstile> bs (\<not>((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)))) = 
              (bs ((\<not>(\<M> a))) \<and> bs(\<not>(di(\<M> a) \<and> di(\<M> b))))" 
       using "13" "14" by fastforce
  hence 15: "\<turnstile> ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and> 
              bs (\<not>((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b))))) = 
              ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and>
              bs ((\<not>(\<M> a))) \<and> bs(\<not>(di(\<M> a) \<and> di(\<M> b)))) " 
       by auto
  have 16: "\<turnstile> ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and>
              bs ((\<not>(\<M> a))) \<and> bs(\<not>(di(\<M> a) \<and> di(\<M> b)))) =
             ((bs ((\<not>(\<M> a))) \<and> di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and>  
              bs(\<not>(di(\<M> a) \<and> di(\<M> b))))" 
       by auto
  have 17: "\<turnstile> ((bs ((\<not>(\<M> a))) \<and> di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and>  
              bs(\<not>(di(\<M> a) \<and> di(\<M> b)))) =
             ((\<rhd>(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and>  
              bs(\<not>(di(\<M> a) \<and> di(\<M> b)))) " 
       using FstEqvBsNotAndDi by fastforce
  have 18: "\<turnstile> ((\<rhd>(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and>  
              bs(\<not>(di(\<M> a) \<and> di(\<M> b)))) = 
             (((\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and> 
              bs(\<not>(di(\<M> a) \<and> di(\<M> b))))" 
       using MFixFst by fastforce
  have 19: "\<turnstile> (((\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and>  
              bs(\<not>(di(\<M> a) \<and> di(\<M> b)))) =
             (((\<M> a)) \<and>  bs(\<not>(di(\<M> a) \<and> di(\<M> b))))" 
       by auto
  have 20: "\<turnstile>   (\<not>(di(\<M> a) \<and> di(\<M> b))) = (\<not>(di(\<M> a)) \<or> \<not>(di(\<M> b)) )"  
       by auto
  have 21: "\<turnstile> (\<not>(di(\<M> a)) \<or> \<not>(di(\<M> b)) ) = ((bi (\<not>(\<M> a))) \<or> (bi (\<not>(\<M> b))) )" 
       by (simp add: bi_d_def)
  have 22: "\<turnstile> (\<not>(di(\<M> a) \<and> di(\<M> b))) = ((bi (\<not>(\<M> a))) \<or> (bi (\<not>(\<M> b))))  " 
       using "20" "21" by auto
  hence 23: "\<turnstile> bs (\<not>(di(\<M> a) \<and> di(\<M> b))) = bs ((bi (\<not>(\<M> a))) \<or> (bi(\<not>(\<M> b)))) " 
       using BsEqvRule by blast
  have 24: "\<turnstile> bs ((bi (\<not>(\<M> a))) \<or> (bi (\<not>(\<M> b)))) = bs (\<not>(\<M> a)) \<or> bs (\<not>(\<M> b))" 
       using BsOrBsEqvBsBiOrBi  by fastforce
  have 25: "\<turnstile> bs (\<not>(di(\<M> a) \<and> di(\<M> b))) = (bs (\<not>(\<M> a)) \<or> bs (\<not>(\<M> b)))" 
       using "23" "24" using BsOrBsEqvBsBiOrBi by fastforce
  hence 26: "\<turnstile> ((\<M> a) \<and> bs (\<not>(di(\<M> a) \<and> di(\<M> b)))) = 
              ((\<M> a) \<and> (bs (\<not>(\<M> a)) \<or> bs (\<not>(\<M> b))))" 
       by auto
  have 27: "\<turnstile> ((\<M> a) \<and> (bs (\<not>(\<M> a)) \<or> bs (\<not>(\<M> b)))) = 
            (\<rhd>(\<M> a) \<and> (bs (\<not>(\<M> a)) \<or> bs (\<not>(\<M> b)))) " 
       using MFixFst by fastforce
  have 28: "\<turnstile> (\<rhd>(\<M> a) \<and> (bs (\<not>(\<M> a)) \<or> bs (\<not>(\<M> b)))) =
             ((\<M> a) \<and> bs (\<not>(\<M> a)) \<and> (bs (\<not>(\<M> a)) \<or> bs (\<not>(\<M> b))))" 
       by (auto simp add: first_d_def)
  have 29: "\<turnstile> ((\<M> a) \<and> bs (\<not>(\<M> a)) \<and> (bs (\<not>(\<M> a)) \<or> bs (\<not>(\<M> b)))) =
             ((\<M> a) \<and> bs (\<not>(\<M> a)))" 
       by auto
  have 30: "\<turnstile> ((\<M> a) \<and> bs (\<not>(\<M> a))) = \<rhd>(\<M> a)" 
       by (simp add: first_d_def)
  have 31: "\<turnstile> \<rhd>(\<M> a) = (\<M> a)" 
       using MFixFst by fastforce
  have 32: "\<turnstile> \<M>(a UPTO (a THRU b)) = 
            ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and> 
            bs (\<not>( (di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)))))" 
       using "1" "2" "6" "7" by fastforce
  have 33: "\<turnstile> ((di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b)) \<and> 
            bs (\<not>( (di(\<M> a)) \<and> ((\<M> a) \<or> di(\<M> b))))) =
             (((\<M> a)) \<and>  bs(\<not>(di(\<M> a) \<and> di(\<M> b))))" 
       using "15" "16" "17" "18" "19" by (metis int_eq)
  have 34: "\<turnstile> (((\<M> a)) \<and>  bs(\<not>(di(\<M> a) \<and> di(\<M> b)))) = (\<M> a) " 
       using "26" "27" "28" "29" "30" "31" by (metis int_eq)
 from 32 33 34 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThruUptoAbsorp:
 "(a THRU (a UPTO b)) \<simeq> (a) "
proof -
  have 1: "\<turnstile> \<M>(a THRU (a UPTO b)) = \<rhd>(di(\<M> a) \<and> di(\<rhd>((\<M> a) \<or> (\<M> b))))" 
       by simp
  have 2: "\<turnstile>  \<rhd>(di(\<M> a) \<and> di(\<rhd>((\<M> a) \<or> (\<M> b)))) = 
              \<rhd>(di(\<M> a) \<and> di(((\<M> a) \<or> (\<M> b))))"
   by (metis DfEqvDfFst FstDiAndDiEqvFstDfAndDf inteq_reflection)
  have 3: "\<turnstile> \<rhd>(di(\<M> a) \<and> di(((\<M> a) \<or> (\<M> b)))) = 
            \<rhd>(di(\<M> a) \<and> (di(\<M> a) \<or> di(\<M> b)))" 
      by (metis DiOrEqv FstEqvRule inteq_reflection lift_and_com) 
  have 4: "\<turnstile> (di(\<M> a) \<and> (di(\<M> a) \<or> di(\<M> b))) = (di(\<M> a))" 
        by auto
  hence 5: "\<turnstile> \<rhd>(di(\<M> a) \<and> (di(\<M> a) \<or> di(\<M> b))) = \<rhd>(di(\<M> a))" 
       using FstEqvRule by blast
  have 6: "\<turnstile> \<rhd>(di(\<M> a)) = \<rhd>(\<M> a)" 
       using FstDiEqvFst by blast
  have 7: "\<turnstile> \<rhd>(\<M> a) = (\<M> a)" 
       using MFixFst by fastforce
 from 1 2 3 5 6 7 show ?thesis using MonEq by (metis int_eq)
qed

lemma MUptoThruDistrib:
 "(a UPTO (b THRU c)) \<simeq> ((a UPTO b) THRU (a UPTO c))"
proof -
  have 1: "\<turnstile> \<M>((a UPTO b) THRU (a UPTO c)) = 
            \<rhd>( di(\<rhd>((\<M> a) \<or> (\<M> b))) \<and> di(\<rhd>((\<M> a) \<or> (\<M> c))) ) " 
       by simp
  have 2: "\<turnstile> ( di(\<rhd>((\<M> a) \<or> (\<M> b))) \<and> di(\<rhd>((\<M> a) \<or> (\<M> c))) ) =
            ( di(((\<M> a) \<or> (\<M> b)) \<and> finite) \<and> di(((\<M> a) \<or> (\<M> c)) \<and> finite) )" 
       using DiAndFiniteEqvDiFst by fastforce
  have 3: "\<turnstile> ( di(((\<M> a) \<or> (\<M> b)) \<and> finite) \<and> di(((\<M> a) \<or> (\<M> c)) \<and> finite) ) =
            (( di((\<M> a) \<and> finite) \<or> di((\<M> b) \<and> finite)) \<and> (di((\<M> a) \<and> finite) \<or> di((\<M> c) \<and> finite)))"
            by (metis (no_types, lifting) DiDiAndEqvDi DiOrEqv FiniteOr inteq_reflection)  
  have 4: "\<turnstile> (( di((\<M> a) \<and> finite) \<or> di((\<M> b) \<and> finite)) \<and> (di((\<M> a) \<and> finite) \<or> di((\<M> c) \<and> finite))) =
           (di((\<M> a) \<and> finite) \<or> (di((\<M> b) \<and> finite) \<and> di((\<M> c) \<and> finite))) " 
       by auto
  have 5: "\<turnstile> ( di(\<rhd>((\<M> a) \<or> (\<M> b))) \<and> di(\<rhd>((\<M> a) \<or> (\<M> c))) ) =
            (di((\<M> a) \<and> finite) \<or> (di((\<M> b) \<and> finite) \<and> di((\<M> c) \<and> finite)))" 
        using "2" "3" "4" by fastforce
  hence 6: "\<turnstile> \<rhd>( di(\<rhd>((\<M> a) \<or> (\<M> b))) \<and> di(\<rhd>((\<M> a) \<or> (\<M> c))) ) =
            \<rhd>( di(\<M> a) \<or> (di(\<M> b) \<and> di(\<M> c)) ) "
            by (metis DiAndFiniteEqvDiFst MFixFst int_simps(1) inteq_reflection)       
  have 7: "\<turnstile> \<rhd>( di(\<M> a) \<or> (di(\<M> b) \<and> di(\<M> c)) ) =
            \<rhd>( \<rhd>(di(\<M> a)) \<or> \<rhd>(di(\<M> b) \<and> di(\<M> c)) ) " 
       using FstFstOrEqvFstOr by fastforce
  have 8: "\<turnstile> \<rhd>(di(\<M> a)) = \<rhd>((\<M> a))" 
       using FstDiEqvFst  by blast
  have 9: "\<turnstile> \<rhd>((\<M> a)) = (\<M> a) " 
       using MFixFst by fastforce
  have 10: "\<turnstile> \<rhd>(di(\<M> a)) = (\<M> a)" 
       using "8" "9" by fastforce
  hence 11: "\<turnstile> (\<rhd>(di(\<M> a)) \<or> \<rhd>(di(\<M> b) \<and> di(\<M> c))) = 
              ((\<M> a) \<or> \<rhd>(di(\<M> b) \<and> di(\<M> c)))" 
       by auto
  hence 12: "\<turnstile> \<rhd>(\<rhd>(di(\<M> a)) \<or> \<rhd>(di(\<M> b) \<and> di(\<M> c))) = 
              \<rhd>((\<M> a) \<or> \<rhd>(di(\<M> b) \<and> di(\<M> c)))" 
       using FstEqvRule by blast
  have 13: "\<turnstile>\<rhd>((\<M> a) \<or> \<rhd>(di(\<M> b) \<and> di(\<M> c))) = \<M>(a UPTO (b THRU c)) " 
       by simp
 from 1 6 7 12 13 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThruUptoDistrib:
 "(a THRU (b UPTO c)) \<simeq> ((a THRU b) UPTO (a THRU c))"
proof -
  have 1: "\<turnstile> \<M>((a THRU b) UPTO (a THRU c)) =
          \<rhd>(\<rhd>(di(\<M> a) \<and> di(\<M> b)) \<or> \<rhd>(di(\<M> a) \<and> di(\<M> c)))" 
          by simp
  have 2: "\<turnstile> \<rhd>(\<rhd>(di(\<M> a) \<and> di(\<M> b)) \<or> \<rhd>(di(\<M> a) \<and> di(\<M> c))) =
            \<rhd>((di(\<M> a) \<and> di(\<M> b)) \<or> (di(\<M> a) \<and> di(\<M> c)))" 
          using FstFstOrEqvFstOr by auto
  have 3: "\<turnstile> ((di((\<M> a) \<and> finite) \<and> di((\<M> b) \<and> finite)) \<or> (di((\<M> a) \<and> finite) \<and> di((\<M> c) \<and> finite))) =
            (di((\<M> a) \<and> finite) \<and> (di((\<M> b) \<and> finite) \<or> di((\<M> c) \<and> finite)))" by auto
  have 4: "\<turnstile> (di((\<M> a) \<and> finite) \<and> (di((\<M> b) \<and> finite) \<or> di((\<M> c) \<and> finite))) =
            (di((\<M> a) \<and> finite) \<and> di( ((\<M> b) \<or> (\<M> c)) \<and> finite ) )" using DiOrEqv
            by (metis "3" FiniteOr inteq_reflection) 
  have 5: "\<turnstile> (di((\<M> a) \<and> finite) \<and> di( ((\<M> b) \<or> (\<M> c)) \<and> finite)) =
            (di((\<M> a) \<and> finite) \<and> di(\<rhd>( (\<M> b) \<or> (\<M> c))))"
            using DiAndFiniteEqvDiFst by fastforce 
  have 6: "\<turnstile> ((di((\<M> a) \<and> finite) \<and> di((\<M> b) \<and> finite)) \<or> (di((\<M> a) \<and> finite) \<and> di((\<M> c) \<and> finite))) =
            (di((\<M> a) \<and> finite) \<and> di(\<rhd>( (\<M> b) \<or> (\<M> c))))"
 using "3" "4" "5" by fastforce
 hence 7: "\<turnstile> \<rhd>((di(\<M> a) \<and> di(\<M> b)) \<or> (di(\<M> a) \<and> di(\<M> c))) =
            \<rhd>(di(\<M> a) \<and> di(\<rhd>( (\<M> b) \<or> (\<M> c)))) "
            by (metis DiAndFiniteEqvDiFst MFixFst int_simps(20) inteq_reflection) 
  have 8: "\<turnstile> \<rhd>(di(\<M> a) \<and> di(\<rhd>( (\<M> b) \<or> (\<M> c)))) =
            \<M>(a THRU (b UPTO c))" by simp
 from 1 2 7 8 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThruUptoRDistrib:
 "((a THRU b) UPTO c) \<simeq> ((a UPTO c) THRU (b UPTO c))"
proof -
  have 1: "((a THRU b) UPTO c) \<simeq> (c UPTO (a THRU b))" 
       using MUptoCommut by auto
  have 2: "(c UPTO (a THRU b)) \<simeq> ((c UPTO a) THRU (c UPTO b))" 
       using MUptoThruDistrib by auto
  have 3: "(c UPTO a) \<simeq> (a UPTO c)" 
       using MUptoCommut by auto
  have 4: "(c UPTO b) \<simeq> (b UPTO c)" 
       using MUptoCommut by auto
  have 5:  "((c UPTO a) THRU (c UPTO b)) \<simeq> ((a UPTO c) THRU (c UPTO b))" 
       using 3  by (simp add: MonEqRefl MonEqSubstThru) 
  have 6:  "((a UPTO c) THRU (c UPTO b)) \<simeq> ((c UPTO b) THRU (a UPTO c))" 
       using MThruCommut by auto
  have 7:  "((c UPTO b) THRU (a UPTO c)) \<simeq> ((b UPTO c) THRU (a UPTO c))" 
       using 4 by (simp add: MonEqRefl MonEqSubstThru)
  from 1 2 5 6 7 show ?thesis using MThruCommut MonEq by (metis int_eq)
qed

lemma MUptoThruRDistrib:
 "((a UPTO b) THRU c) \<simeq> ((a THRU c) UPTO (b THRU c))"
proof -
  have 1: "((a UPTO b) THRU c) \<simeq> (c THRU (a UPTO b))" 
       using MThruCommut by auto
  have 2: "(c THRU (a UPTO b)) \<simeq> ((c THRU a) UPTO (c THRU b))" 
       using MThruUptoDistrib by auto
  have 3: "(c THRU a) \<simeq> (a THRU c)" 
       using MThruCommut by auto
  have 4: "(c THRU b) \<simeq> (b THRU c)" 
       using MThruCommut by auto
  have 5:  "((c THRU a) UPTO (c THRU b)) \<simeq> ((a THRU c) UPTO (c THRU b))" 
       using 3 by (simp add: MonEqRefl MonEqSubstUpto)
  have 6:  "((a THRU c) UPTO (c THRU b)) \<simeq> ((c THRU b) UPTO (a THRU c))" 
       using MUptoCommut by auto
  have 7:  "((c THRU b) UPTO (a THRU c)) \<simeq> ((b THRU c) UPTO (a THRU c))" 
       using 4 by (simp add: MonEqRefl MonEqSubstUpto)
  from 1 2 5 6 7 show ?thesis using MUptoCommut MonEq by (metis int_eq)
qed

lemma MWithAndDistrib:
 "((a AND b) WITH f) \<simeq> ((a WITH f) AND (b WITH f))"
proof -
  have 1: "\<turnstile> \<M>((a AND b) WITH f) = (\<M>(a AND b) \<and> f)" 
       by (simp)
  have 2: "\<turnstile> \<M>(a AND b) = \<M>(a WITH LIFT(\<M> b))" 
       by (simp add: AND_d_def)
  have 3: "\<turnstile> (\<M>(a AND b) \<and> f) = (\<M>(a WITH LIFT(\<M> b)) \<and> f)" 
       using "2" by auto
  have 4: "\<turnstile> \<M>(a WITH (LIFT( (\<M> b) \<and> f))) = (\<M>(a) \<and> \<M>(b) \<and> f)" 
       by simp
  have 5: "\<turnstile> (\<M>(a) \<and> \<M>(b) \<and> f) = ((\<M>(a) \<and> f) \<and> (\<M>(b) \<and> f))" 
       by auto
  have 6: "\<turnstile> ((\<M>(a) \<and> f) \<and> (\<M>(b) \<and> f)) = (\<M>(a WITH f) \<and> \<M>( b WITH f))" 
       by simp
  have 7: "\<turnstile> (\<M>(a WITH f) \<and> \<M>(b WITH f)) = \<M>((a WITH f) WITH LIFT(\<M>(b WITH f)))" 
       by simp
  have 8: "\<turnstile> \<M>((a WITH f) WITH LIFT(\<M>(b WITH f))) = \<M>((a WITH f) AND (b WITH f))" 
       by (simp add: AND_d_def)
  from 1 2 3 4 5 6 7 8 show ?thesis using MonEq by (metis AND_d_def MWithAbsorp int_eq)
qed

lemma MHaltWithAndDistrib:
 "(((HALT w) WITH f) AND ((HALT w) WITH g)) \<simeq> ((HALT w) WITH LIFT(f \<and> g))"
proof -
  have 1: "\<turnstile> \<M>(((HALT w) WITH f) AND ((HALT w) WITH g)) = 
             \<M>(((HALT w) WITH f) WITH LIFT(\<M>((HALT w) WITH g)))" 
       by (simp add: AND_d_def)
  have 2: "\<turnstile> \<M>(((HALT w) WITH f) WITH LIFT(\<M>((HALT w) WITH g))) = 
             (\<M>(HALT w) \<and> f \<and> \<M>(HALT w) \<and> g)" 
       by auto
  have 3: "\<turnstile> (\<M>(HALT w) \<and> f \<and> \<M>(HALT w) \<and> g) = (\<M>(HALT w) \<and> f \<and> g)"  
       by auto
  have 4: "\<turnstile> (\<M>(HALT w) \<and> f \<and> g) = \<M>((HALT w) WITH LIFT(f \<and> g))" 
       by auto
  from 1 2 3 4 show ?thesis using MonEq by (metis int_eq)
qed

lemma MHaltWithUptoHaltWithEqvHaltWithOr:
 "(((HALT w) WITH f) UPTO ((HALT w) WITH g)) \<simeq> ((HALT w) WITH LIFT(f \<or> g))"
proof -
  have 1: "\<turnstile> \<M>(((HALT w) WITH f) UPTO ((HALT w) WITH g)) = 
             \<rhd>(\<M>((HALT w) WITH f) \<or> \<M>((HALT w) WITH g))" 
       by (simp)
  have 2: "\<turnstile> \<rhd>(\<M>((HALT w) WITH f) \<or> \<M>((HALT w) WITH g)) = 
             \<rhd>((\<M>(HALT w) \<and> f) \<or> (\<M>(HALT w) \<and> g))" 
       by auto
  have 3: "\<turnstile> ((\<M>(HALT w) \<and> f) \<or> (\<M>(HALT w) \<and> g)) = (\<M>(HALT w) \<and> (f \<or> g))" 
       by auto 
  have 4: "\<turnstile> \<rhd>((\<M>(HALT w) \<and> f) \<or> (\<M>(HALT w) \<and> g)) = \<rhd>(\<M>(HALT w) \<and> (f \<or> g))"
       using "3" FstEqvRule by blast 
  have 5: "\<turnstile> \<rhd>(\<M>(HALT w) \<and> (f \<or> g)) = \<rhd>(\<M>((HALT w) WITH LIFT(f \<or> g)))" 
       by simp
  have 6: "\<turnstile> \<M>(((HALT w) WITH LIFT(f \<or> g))) = \<rhd>(\<M>((HALT w) WITH LIFT(f \<or> g)))" 
       using MFixFst by blast
  from 1 2 3 4 5 6 show ?thesis using MonEq by (metis int_eq)
qed

lemma MHaltWithThruHaltWithEqvHaltWithAndHaltWith:
 "(((HALT w) WITH f) THRU ((HALT w) WITH g)) \<simeq> (((HALT w) WITH f) AND ((HALT w) WITH g))"
proof -
  have 1: "\<turnstile> \<M>(((HALT w) WITH f) THRU ((HALT w) WITH g)) = 
             \<rhd>( di(\<M>(HALT w) \<and> f) \<and> di(\<M>(HALT w) \<and> g) )" 
       by simp
  have 2: "\<turnstile> (di(\<M>(HALT w) \<and> f) \<and> di(\<M>(HALT w) \<and> g)) = 
             (di((halt(init w) \<and> finite) \<and> f) \<and> di((halt(init w) \<and> finite) \<and> g))" 
       using MHaltAlt by (metis DiDiAndEqvDi inteq_reflection) 
  have 3: "\<turnstile> (di((halt(init w)  \<and> finite)\<and> f) \<and> di((halt(init w) \<and> finite) \<and> g)) = 
             di( (halt(init w) \<and> finite)\<and> f \<and> g)" 
       by (metis FstDiamondStateEqvHalt LFstAndDistrD inteq_reflection) 
  have 4: "\<turnstile> di( (halt(init w) \<and> finite)\<and> f \<and> g) = di(\<M>(HALT w) \<and> f \<and> g)"
       by (metis "3" MHaltAlt inteq_reflection) 
  have 5: "\<turnstile> (di(\<M>(HALT w) \<and> f) \<and> di(\<M>(HALT w) \<and> g)) = di(\<M>(HALT w) \<and> f \<and> g)" 
       using 2 3 4 by fastforce
  have 6: "\<turnstile> \<rhd>(di(\<M>(HALT w) \<and> f) \<and> di(\<M>(HALT w) \<and> g)) = \<rhd>(di(\<M>(HALT w) \<and> f \<and> g))" 
       using 5 FstEqvRule by blast
  have 7: "\<turnstile> \<rhd>(di(\<M>(HALT w) \<and> f \<and> g)) = \<rhd>(\<M>(HALT w) \<and> f \<and> g)" 
       using FstDiEqvFst by fastforce
  have 8: "\<turnstile> \<rhd>(\<M>(HALT w) \<and> f \<and> g) = \<rhd>(\<M>((HALT w) WITH LIFT(f \<and> g)))" 
       by simp
  have 9: "\<turnstile> \<M>((HALT w) WITH LIFT(f \<and> g)) = \<rhd>(\<M>((HALT w) WITH LIFT(f \<and> g)))" 
       using MFixFst by blast
  have 10: "\<turnstile> \<M>(((HALT w) WITH f) THRU ((HALT w) WITH g)) = \<M>((HALT w) WITH LIFT(f \<and> g))" 
       using 1 2 3 4 5 6 7 8 9 int_eq by metis
  have 11: "\<turnstile> \<M>(((HALT w) WITH f) AND ((HALT w) WITH g)) = \<M>((HALT w) WITH LIFT(f \<and> g))" 
       using MHaltWithAndDistrib using eq_d_def by blast
  have 12: "\<turnstile> \<M>((HALT w) WITH LIFT(f \<and> g)) = \<M>(((HALT w) WITH f) AND ((HALT w) WITH g))" 
       using 11 by fastforce
  from 10 12 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThenAndDistrib:
 "(a THEN (b AND c)) \<simeq> ((a THEN b) AND (a THEN c))"
proof -
  have 1: "\<turnstile> \<M>(a THEN (b AND c)) = (\<M>(a)) \<frown> (\<M>(b AND c))" 
       by simp
  have 2: "\<turnstile> (\<M>(a)) \<frown> (\<M>(b AND c)) = (\<M>(a)) \<frown> (\<M>(b) \<and> \<M>(c))"
       by (simp add: AND_d_def) 
  have 3: "\<turnstile> (\<M>(a)) \<frown> (\<M>(b) \<and> \<M>(c)) =  \<rhd> (\<M>(a)) \<frown> (\<M>(b) \<and> \<M>(c))"
       by (simp add: LeftSChopEqvSChop MFixFst)
  have 4: "\<turnstile> \<rhd> (\<M>(a)) \<frown> (\<M>(b) \<and> \<M>(c)) =  ((\<rhd> (\<M>(a)) \<frown> (\<M>(b))) \<and> (\<rhd> (\<M>(a)) \<frown> (\<M>(c))))"
       by (metis LFstAndDistrB Prop11 schop_d_def)
  have 5: "\<turnstile>((\<rhd> (\<M>(a)) \<frown> (\<M>(b))) \<and> (\<rhd> (\<M>(a)) \<frown> (\<M>(c)))) = 
            (( (\<M>(a)) \<frown> (\<M>(b))) \<and> ( (\<M>(a)) \<frown> (\<M>(c))) ) " using MFixFst 
            by (metis "4" inteq_reflection)
  have 6: "\<turnstile> (( (\<M>(a)) \<frown> (\<M>(b))) \<and> ( (\<M>(a)) \<frown> (\<M>(c))) ) =
             (\<M>(a THEN b) \<and> \<M>(a THEN c))  " 
       by simp
  have 7: "\<turnstile> (\<M>(a THEN b) \<and> \<M>(a THEN c)) = \<M>((a THEN b) AND (a THEN c))"
        by (simp add: AND_d_def) 
  from 1 2 3 4 5 6 7 show ?thesis using MonEq by (metis int_eq)
qed

lemma MThenUptoDistrib:
 "(a THEN (b UPTO c)) \<simeq> ((a THEN b) UPTO (a THEN c))"
proof -
 have 1: "\<turnstile> (\<M> (a THEN (b UPTO c))) = ((\<M> a)\<frown>(\<rhd>((\<M> b) \<or> (\<M> c))))" 
      by simp
 have 2: "\<turnstile> ((\<M> a)\<frown>(\<rhd>((\<M> b) \<or> (\<M> c)))) = (\<rhd>(\<M> a)\<frown>(\<rhd>((\<M> b) \<or> (\<M> c))))" 
      by (simp add: MFixFst LeftSChopEqvSChop)
 have 3: "\<turnstile> (\<rhd>(\<M> a)\<frown>(\<rhd>((\<M> b) \<or> (\<M> c)))) = ((\<rhd>(\<rhd>(\<M> a)\<frown>((\<M> b) \<or> (\<M> c)))))"
      by (metis FstFstChopEqvFstChopFst FstImpFinite Prop10 inteq_reflection schop_d_def) 
 have 4: "\<turnstile> \<rhd>(\<M> a)\<frown>((\<M> b) \<or> (\<M> c)) = (\<M> a)\<frown>((\<M> b) \<or> (\<M> c))" 
      using MFixFst by (metis LeftSChopEqvSChop inteq_reflection)
 have 5: "\<turnstile> (\<M> a)\<frown>((\<M> b) \<or> (\<M> c)) = ((\<M> a)\<frown>(\<M> b) \<or> (\<M> a)\<frown>(\<M> c))" 
      by (simp add: SChopOrEqv)
 have 6: "\<turnstile> ((\<M> a)\<frown>(\<M> b) \<or> (\<M> a)\<frown>(\<M> c)) = (\<M>(a THEN b) \<or> \<M>(a THEN c))" 
      by simp
 have 7: "\<turnstile> \<rhd>(\<M> a)\<frown>((\<M> b) \<or> (\<M> c)) = (\<M>(a THEN b) \<or> \<M>(a THEN c))  "
      using "6" "5" "4" by fastforce
 have 8: "\<turnstile> \<rhd>(\<rhd>(\<M> a)\<frown>((\<M> b) \<or> (\<M> c))) = \<rhd>(\<M>(a THEN b) \<or> \<M>(a THEN c)) "
      using "7" by (simp add: FstEqvRule)
 have 9: "\<turnstile> \<rhd>(\<M>(a THEN b) \<or> \<M>(a THEN c)) = \<M>((a THEN b) UPTO (a THEN c)) "
      by simp
 from 9 7 1 2 3 show ?thesis by (metis eq_d_def inteq_reflection)
qed

lemma MThenthruDistrib:
 "(a THEN (b THRU c)) \<simeq> ((a THEN b) THRU (a THEN c))"
proof -
 have 1: "\<turnstile> \<M>(a THEN (b THRU c)) = (\<M> a)\<frown>\<rhd>(di(\<M> b) \<and> di(\<M> c))"
      by simp
 have 2: "\<turnstile> (\<M> a)\<frown>\<rhd>(di(\<M> b) \<and> di(\<M> c)) = \<rhd>(\<M> a)\<frown>\<rhd>(di(\<M> b) \<and> di(\<M> c))"
      by (simp add: MFixFst LeftSChopEqvSChop)
 have 3: "\<turnstile> \<rhd>(\<M> a)\<frown>\<rhd>(di(\<M> b) \<and> di(\<M> c)) = \<rhd>(\<rhd>(\<M> a)\<frown>(di(\<M> b) \<and> di(\<M> c) ))"
      by (metis FstFstChopEqvFstChopFst FstImpFinite Prop10 inteq_reflection schop_d_def)
 have 4: "\<turnstile> \<rhd>(\<M> a)\<frown>(di(\<M> b) \<and> di(\<M> c) ) = (\<rhd>(\<M> a)\<frown>di(\<M> b) \<and> \<rhd>(\<M> a)\<frown>di(\<M> c))"
      by (metis LFstAndDistrB inteq_reflection schop_d_def) 
 have 5: "\<turnstile> (\<rhd>(\<M> a)\<frown>di(\<M> b) \<and> \<rhd>(\<M> a)\<frown>di(\<M> c)) = ((\<M> a)\<frown>di(\<M> b) \<and> (\<M> a)\<frown>di(\<M> c))" 
      using MFixFst by (metis "4" int_eq)
 have 6: "\<turnstile> (\<M> a)\<frown>di(\<M> b) = (\<M> a)\<frown>((\<M> b)\<frown>#True)"
      by (metis DiAndFiniteEqvDiFst MFixFst RightSChopEqvSChop di_d_def inteq_reflection schop_d_def)
 have 7: "\<turnstile> (\<M> a)\<frown>((\<M> b)\<frown>#True) = ((\<M> a)\<frown>(\<M> b))\<frown>#True"
      by (simp add: SChopAssoc)
 have 8: "\<turnstile> ((\<M> a)\<frown>(\<M> b))\<frown>#True = di((\<M> a)\<frown>(\<M> b))"
      by (metis DiAndFiniteEqvDiFst MFixFst MON.simps(4) di_d_def inteq_reflection schop_d_def) 
 have 9: "\<turnstile> (\<M> a)\<frown>di(\<M> b) = di((\<M> a)\<frown>(\<M> b))"
      using "8" "7" "6" by fastforce
 have 10: "\<turnstile> (\<M> a)\<frown>di(\<M> c) = (\<M> a)\<frown>((\<M> c)\<frown>#True)"
      by (metis DiAndFiniteEqvDiFst MFixFst di_d_def int_simps(20) inteq_reflection schop_d_def)
 have 11: "\<turnstile> (\<M> a)\<frown>((\<M> c)\<frown>#True) = ((\<M> a)\<frown>(\<M> c))\<frown>#True"
      by (simp add: SChopAssoc)
 have 12: "\<turnstile> ((\<M> a)\<frown>(\<M> c))\<frown>#True = di((\<M> a)\<frown>(\<M> c))"
      by (metis (no_types, lifting) "10" "11" ChopAssoc di_d_def inteq_reflection schop_d_def)
 have 13: "\<turnstile> (\<M> a)\<frown>di(\<M> c) = di((\<M> a)\<frown>(\<M> c))"
      using "12" "11" "10" by fastforce
 have 14: "\<turnstile> ((\<M> a)\<frown>di(\<M> b) \<and> (\<M> a)\<frown>di(\<M> c)) = (di((\<M> a)\<frown>(\<M> b)) \<and> di((\<M> a)\<frown>(\<M> c))) "
      using "13" "9" by fastforce
 have 15: "\<turnstile> (di((\<M> a)\<frown>(\<M> b)) \<and> di((\<M> a)\<frown>(\<M> c))) = (di(\<M>(a THEN b)) \<and> di(\<M>(a THEN c)))"
      by simp
 have 16: "\<turnstile> \<rhd>(\<M> a)\<frown>(di(\<M> b) \<and> di(\<M> c) ) = (di(\<M>(a THEN b)) \<and> di(\<M>(a THEN c))) "  
      using "15" "14" "4" "5" by fastforce
 have 17: "\<turnstile> \<rhd>(\<rhd>(\<M> a)\<frown>(di(\<M> b) \<and> di(\<M> c) )) = \<rhd>(di(\<M>(a THEN b)) \<and> di(\<M>(a THEN c)))" 
      using "16" by (simp add: FstEqvRule)  
 have 18: "\<turnstile> \<rhd>(di(\<M>(a THEN b)) \<and> di(\<M>(a THEN c))) = \<M>((a THEN b) THRU (a THEN c))"
      by simp
 from 18 16 1 2 3 show ?thesis by (metis eq_d_def int_eq)
qed




end
