(*  Title     : A Shallow Encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

chapter \<open>CDT-PITL\<close>   

theory CDT_PITL_Deep

imports 
  NELList_Extras 
begin
(*
 sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                        isar_proofs=false,
                    provers="z3 cvc5 e spass vampire" ]
*)

text \<open>
This theory contains a deep embedding of PITL with the CDT operators of Yde Venema\cite{venema91} 
without chop-star and chop-omega. 
\<close>

section \<open>Syntax\<close>

subsection \<open>Primitive formulae\<close>

datatype  cdt =
    false_d                         ("ifalse")
  | atom_d "nat"                    ("($_)" [100] 99) 
  | iimp_d " cdt" " cdt"            ("( _ iimp _ )" [26,25] 25)
  | empty_d                         ("(empty)") 
  | c_d "cdt" " cdt"                ("( _ cee _ )" [84,84] 83)
  | d_d "cdt" "cdt"                 ("( _ dee _ )" [84,84] 83)
  | t_d "cdt" "cdt"                 ("( _ tee _ )" [84,84] 83)

subsection \<open>Derived Boolean Operators\<close>

definition inot_d ("(inot _)" [90] 90) 
where 
  "inot f \<equiv> f iimp ifalse"

definition itrue_d ("itrue") 
where 
  "itrue \<equiv> inot  ifalse"

definition ior_d ("(_ ior _)" [31,30] 30) 
where 
  "f ior g \<equiv> (inot f) iimp g"

definition iand_d ("(_ iand _)" [36,35] 35)
where 
  "f iand g \<equiv> inot (inot f ior inot g)"

definition ieqv_d ( "(_ ieqv _) " [21,20] 20)
where 
  "f ieqv g \<equiv> ( ( f iimp g) iand  (g iimp f))" 
 
subsection \<open>ifinite and inf\<close>

definition finite_d ("ifinite") 
where 
  "ifinite \<equiv> itrue cee empty"

definition inf_d ("inf") 
where 
  "inf \<equiv> inot ifinite"


subsection \<open>weak chop\<close>

definition chop_d ("( _ ; _ )" [84,84] 83)
 where " f ; g \<equiv> (f cee g) ior (f iand inf)" 

subsection \<open>more,  skip, next, wnext, prev and wprev \<close>

definition more_d ("more") 
where 
  "more \<equiv> inot empty"

definition skip_d ("skip") 
where 
  "skip \<equiv> more iand inot(more cee more)"

definition next_d ("(next _)")
 where "next f \<equiv> skip cee f" 

definition wnext_d ("(wnext _)")
 where "wnext f \<equiv> (empty ior next f)"

definition prev_d ("(prev _)")
 where "prev f \<equiv> f ; skip"  

definition wprev_d ("(wprev _)")
 where "wprev f \<equiv> (empty ior prev f)"


subsection \<open>Box and Diamond Operators\<close>

definition diamond_d ("(diamond _)" [88] 87)
where 
  "diamond f \<equiv> itrue cee f"

definition box_d ("(box _)" [88] 87)
where 
  "box f \<equiv> inot (diamond (inot f))"

definition df_d ("(df _)" [88] 87)
where 
  "df f \<equiv> f cee itrue"

definition bf_d ("(bf _)" [88] 87)
where 
  "bf f \<equiv> inot (df (inot f))"

definition ldiamond_d ("(ldiamond _)" [88] 87)
where "ldiamond f \<equiv> f dee itrue" 

definition  lbox_d ("(lbox _)" [88] 87)
where
 "lbox f = inot (ldiamond (inot f))" 

definition rdiamond_d ("(rdiamond _)" [88] 87)
where "rdiamond f \<equiv> (f  tee ifinite) "

definition  rbox_d ("(rbox _)" [88] 87)
where
 "rbox f = inot (rdiamond (inot f))" 


subsection \<open>Initial and Final Operators\<close>

definition init_d ("(init _)" [88] 87)
where
 " init f \<equiv> ((empty iand f) cee itrue)"

definition fin_d ("(fin _)" [88] 87)
where
  "fin f \<equiv> diamond ( empty iand f)"

subsection \<open>Horizontal\<close>

definition 
  "hor q = ( (lbox $q) iand (box $q) iand (ifinite iimp rbox (fin $q))) " 

subsection \<open>Compass operators\<close>

definition ediamond_d ("(ediamond _)" [88] 87)
where 
  "ediamond f \<equiv> more cee f"

definition wdiamond_d ("(wdiamond _)" [88] 87)
where 
  "wdiamond f \<equiv> more dee f"

definition sdiamond_d ("(sdiamond _)" [88] 87)
where 
  "sdiamond f \<equiv>f cee more"

definition ndiamond_d ("(ndiamond _)" [88] 87)
where 
  "ndiamond f \<equiv> more tee f"

definition ewdiamond_d ("(ewdiamond _)" [88] 87)
where 
  "ewdiamond f \<equiv> ediamond f ior f ior wdiamond f"

definition nsdiamond_d ("(nsdiamond _)" [88] 87)
where 
  "nsdiamond f \<equiv> sdiamond f ior f ior ndiamond f"

definition  ebox_d ("(ebox _)" [88] 87)
where
 "ebox f = inot (ediamond (inot f))" 

definition  wbox_d ("(wbox _)" [88] 87)
where
 "wbox f = inot (wdiamond (inot f))" 

definition  sbox_d ("(sbox _)" [88] 87)
where
 "sbox f = inot (sdiamond (inot f))" 

definition  nbox_d ("(nbox _)" [88] 87)
where
 "nbox f = inot (ndiamond (inot f))"

definition lex_d ("(lex)")
where "lex \<equiv> wbox ifalse" 

definition rex_d ("(rex)")
where "rex \<equiv> nbox ifalse" 

definition  ewbox_d ("(ewbox _)" [88] 87)
where
 "ewbox f = inot (ewdiamond (inot f))" 

definition  nsbox_d ("(nsbox _)" [88] 87)
where
 "nsbox f = inot (nsdiamond (inot f))" 

definition ahor_d ("(ahor _)" [88] 87)
where " ahor f \<equiv> ewbox f iand nbox (ewbox(inot f)) iand sbox(ewbox (inot f)) " 

definition aver_d ("(aver _)" [88] 87)
where " aver f \<equiv> nsbox f iand ebox (nsbox(inot f)) iand wbox(nsbox (inot f)) "




section \<open>Semantics\<close>

subsection \<open>Intervals\<close>

type_synonym interval = " nat set nellist " 


(* -------------------- Venema C T D operators ----------------------------- *)

(*

++++++++++++++++++++++++++++++++++++ f1 cee f2 +++++++++++++++++++++++++
 ----------------------------------------------------------------- global

---------| l                           r |------------------------   f1 cee f2
         |-------------------------------| c

         |------------------| c1
                         c2 |------------|

---------|                 |--------------------------------------  f1
         |-----------------|


---------------------------|           |-------------------------- f2
                           |-----------|

+++++++++++++++++++++++++++++++++++++ f1 tee f2 +++++++++++++++++++++++++

---------| l                        r |----------------------------  f1 tee f2
         |----------------------------| c


                                   r1 |-------------|
                                                 r2 |-------------

--------------------------------------|             |-------------  f1
                                      |-------------|

----------|                                         |-------------  f2
          |-----------------------------------------|


+++++++++++++++++++++++++++++++++++++++ f1 dee f2 +++++++++++++++++++++++

---------------------| l                            r |------------ f1 dee f2
                   c |--------------------------------| 
                      
------| l1
      |--------------| l2

------|              |--------------------------------------------- f1
      |--------------|

------|                                                |----------- f2
      |------------------------------------------------|

*)


fun semantics_cdt :: "[ interval, nat, enat,  cdt] \<Rightarrow> bool" ("(_ _ _ \<Turnstile>\<^sub>i _)" [80,10] 10)
where
  "( g i j \<Turnstile>\<^sub>i ifalse) = False  " 
| "( g i j \<Turnstile>\<^sub>i $p)     = ((enat i)\<le> nlength g \<and> p \<in> (nnth g i)) "  
| "( g i j \<Turnstile>\<^sub>i F iimp G) = ( ( g i j \<Turnstile>\<^sub>i F) \<longrightarrow> ( g i j \<Turnstile>\<^sub>i G)) " 
| "( g i j \<Turnstile>\<^sub>i empty) = ( (enat i) = j ) " 
| "( g i j \<Turnstile>\<^sub>i F cee G) = (\<exists> k. (enat i) \<le> (enat k) \<and> (enat k) \<le> j \<and> ( g i k \<Turnstile>\<^sub>i F) \<and> ( g k j \<Turnstile>\<^sub>i G) ) " 
| "( g i j \<Turnstile>\<^sub>i F tee G) = (\<exists> k l. j< \<infinity> \<and> l = the_enat j \<and> j \<le> k \<and> k \<le> nlength g \<and> ( g l k \<Turnstile>\<^sub>i F) \<and> ( g i k \<Turnstile>\<^sub>i G) ) "    
| "( g i j \<Turnstile>\<^sub>i F dee G) = (\<exists> k. 0 \<le> k \<and> k \<le> i \<and> ( g k i \<Turnstile>\<^sub>i F) \<and> ( g k j \<Turnstile>\<^sub>i G) ) " 



  
primrec fvars :: "cdt \<Rightarrow> nat set"
where 
  "fvars ifalse        = {} "
| "fvars ($p)          = {p} "
| "fvars (f iimp g)    = (fvars f) \<union> (fvars g) "
| "fvars empty         = {} "
| "fvars (f cee g)     = (fvars f) \<union> (fvars g) "
| "fvars (f tee g)     = fvars f \<union> (fvars g)"
| "fvars (f dee g)     = fvars f \<union> (fvars g)"  
 

definition upd :: " interval \<Rightarrow> nat \<Rightarrow> (bool nellist) \<Rightarrow> interval " 
where 
 " upd \<sigma> n l = nmap (\<lambda> (x,y). (if y then insert n x else x - {n})) (nzip \<sigma> l)" 


lemma upd_NCons: 
 "upd (NCons x xs) n (NCons y ys) = 
  (if y then NCons (insert n x) (upd xs n ys) else  NCons (x - {n}) (upd xs n ys))" 
by (simp add: upd_def)

lemma upd_NNil: 
 "upd (NNil a) n (NNil b) = 
  (if b then  (NNil (insert n a)) else (NNil (a - {n}))) " 
by (simp add: nfirst_eq_nnth_zero nnth_NNil upd_def)

lemma nnth_upd: 
assumes "nlength \<sigma> = nlength l"
        " (enat k) \<le> nlength \<sigma>"
shows  " (nnth (upd \<sigma> n l) k) = (if (nnth l k) then (insert n (nnth \<sigma> k)) else (nnth \<sigma> k) - {n}) " 
using assms 
by (simp add: upd_def nnth_nzip)

lemma nlength_upd: 
 assumes "nlength \<sigma> = nlength l"
 shows "nlength ((upd \<sigma> n l)) = nlength \<sigma>" 
using assms by (simp add: upd_def)

   
lemma upd_swap: 
 assumes "nlength \<sigma> = nlength l1"
         "nlength \<sigma> = nlength l2" 
         " x1 \<noteq> x2 " 
 shows " upd (upd \<sigma> x1 l1) x2 l2 = (upd (upd \<sigma> x2 l2) x1 l1) " 
proof -
 have 1: "nlength (upd (upd \<sigma> x1 l1) x2 l2) = nlength (upd (upd \<sigma> x2 l2) x1 l1)"
 using assms nlength_upd by auto
 have 2: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow>
           (nnth (upd (upd \<sigma> x1 l1) x2 l2) i) = 
         (if (nnth l2 i) then insert x2 (nnth (upd \<sigma> x1 l1) i) else (nnth (upd \<sigma> x1 l1) i) - {x2})" 
   using assms by (simp add: nlength_upd nnth_upd) 
 have 3: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow>
           (nnth (upd (upd \<sigma> x2 l2) x1 l1) i) = 
          (if (nnth l1 i) then insert x1 (nnth (upd \<sigma> x2 l2) i) else (nnth (upd \<sigma> x2 l2) i) - {x1})" 
   using assms by (simp add: nlength_upd nnth_upd) 
 have 4: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow>
           (if (nnth l2 i) then insert x2 (nnth (upd \<sigma> x1 l1) i) else (nnth (upd \<sigma> x1 l1) i) - {x2}) =
           (if (nnth l1 i) then insert x1 (nnth (upd \<sigma> x2 l2) i) else (nnth (upd \<sigma> x2 l2) i) - {x1}) " 
    using assms nnth_upd nlength_upd by auto
 show ?thesis
 by (simp add: "1" "2" "3" "4" nellist_eq_nnth_eq) 
qed

lemma upd_absorb: 
 assumes "nlength \<sigma> = nlength l1"
         "nlength \<sigma> = nlength l2" 
         " x1 = x2 " 
 shows " upd (upd \<sigma> x1 l1) x2 l2 = (upd \<sigma> x2 l2) "
proof -
 have 1: "nlength (upd (upd \<sigma> x1 l1) x2 l2) = nlength (upd \<sigma> x2 l2)"
 using assms nlength_upd by auto  
 have 2: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow>
           (nnth (upd (upd \<sigma> x1 l1) x2 l2) i) = 
         (if (nnth l2 i) then insert x2 (nnth (upd \<sigma> x1 l1) i) else (nnth (upd \<sigma> x1 l1) i) - {x2})" 
   using assms by (simp add: nlength_upd nnth_upd)
 have 3: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow> 
           (nnth (upd \<sigma> x2 l2) i) =
           (if (nnth l2 i) then insert x2 (nnth \<sigma> i) else (nnth \<sigma> i) - {x2})"
    using assms  by (simp add: nlength_upd nnth_upd)
 have 4: "\<And> i. i \<le> nlength (upd (upd \<sigma> x1 l1) x2 l2) \<Longrightarrow> 
            (if (nnth l2 i) then insert x2 (nnth (upd \<sigma> x1 l1) i) else (nnth (upd \<sigma> x1 l1) i) - {x2}) =
            (if (nnth l2 i) then insert x2 (nnth \<sigma> i) else (nnth \<sigma> i) - {x2})"
   using assms  nnth_upd nlength_upd by auto
 show ?thesis
 by (simp add: "1" "2" "3" "4" nellist_eq_nnth_eq) 
qed
   
lemma upd_nappend: 
 assumes "nlength \<sigma>1 = nlength l1"
         "nlength \<sigma>2 = nlength l2"
 shows " upd (nappend \<sigma>1 \<sigma>2) x (nappend l1 l2) = 
         (nappend (upd \<sigma>1 x l1) (upd \<sigma>2 x l2)) " 
using assms
by (simp add: nmap_nappend_distrib nzip_nappend upd_def)

 
lemma ntaken_upd:
 assumes "nlength \<sigma> = nlength l"
         "j \<le> nlength \<sigma> "
 shows "ntaken j (upd \<sigma> x l) = (upd (ntaken j \<sigma>) x (ntaken j l)) "
using assms 
by (simp add: ntaken_nzip upd_def)


lemma ndropn_upd:
 assumes "nlength \<sigma> = nlength l"
         "j \<le> nlength \<sigma> "
 shows "ndropn j (upd \<sigma> x l) = (upd (ndropn j \<sigma>) x (ndropn j l)) "
using assms ndropn_nzip upd_def by (simp add: ndropn_nmap)

 
lemma nsubn_upd:
 assumes "nlength \<sigma> = nlength l"
         " (enat j) \<le> enat k" 
         " (enat k) \<le> nlength \<sigma> "
 shows " nsubn (upd \<sigma> x l) j k  = (upd (nsubn  \<sigma> j k) x (nsubn  l j k))" 
proof -
 have 1: "nsubn (upd \<sigma> x l) j k = ntaken (k - j) (ndropn j (upd \<sigma> x l))" 
   using assms unfolding nsubn_def1 by simp
 have 2: " (ndropn j (upd \<sigma> x l)) = (upd (ndropn j \<sigma>) x (ndropn j l))"  
   using assms by (meson dual_order.trans ndropn_upd)  
 have 3: " ntaken (k - j) (upd (ndropn j \<sigma>) x (ndropn j l)) =
           (upd (ntaken (k-j) (ndropn j \<sigma>)) x (ntaken (k-j) (ndropn j l)))"
    using assms  ntaken_upd[of "(ndropn j \<sigma>)" "(ndropn j l)" "(k - j)" x]
    by (metis ndropn_nlength nle_le nlength_upd ntaken_all)
 have 4: "(upd (ntaken (k-j) (ndropn j \<sigma>)) x (ntaken (k-j) (ndropn j l))) =
          (upd (nsubn  \<sigma> j k) x (nsubn  l j k))" 
    unfolding nsubn_def1 by simp
 show ?thesis using "1" "2" "3" "4" by presburger 
qed       
 
lemma nfirst_upd: 
 assumes " nlength \<sigma> = nlength l" 
 shows "nfirst (upd \<sigma> x l) = (if nfirst l then insert x (nfirst \<sigma>) else (nfirst \<sigma>) - {x} )"
 by (metis assms nfirst_eq_nnth_zero nnth_upd zero_enat_def zero_le) 

lemma nlast_upd: 
 assumes " nlength \<sigma> = nlength l" 
         "nfinite \<sigma> "
 shows " nlast (upd \<sigma> x l) = (if nlast l then insert x (nlast \<sigma>) else (nlast \<sigma>) - {x} ) " 
using assms 
by (metis dual_order.order_iff_strict nfinite_conv_nlength_enat nlength_upd nnth_nlast nnth_upd the_enat.simps)

lemma nfinite_upd: 
 assumes " nlength \<sigma> = nlength l " 
 shows   "nfinite (upd \<sigma> x l) \<longleftrightarrow> nfinite \<sigma> \<and> nfinite l" 
using assms 
by (simp add: nfinite_conv_nlength_enat nlength_upd)

lemma upd_nfuse: 
 assumes "nlength \<sigma>1 = nlength l1"
         "nlength \<sigma>2 = nlength l2"
         "nfirst \<sigma>2 = nlast \<sigma>1" 
         " nfinite \<sigma>1"
         " nfirst l2 = nlast l1" 
 shows " upd (nfuse \<sigma>1 \<sigma>2) x (nfuse l1 l2) = 
         (nfuse (upd \<sigma>1 x l1) (upd \<sigma>2 x l2)) " 
using assms by (simp add: nfuse_nlength nmap_nfuse nzip_nfuse upd_def)

lemma nrev_upd: 
  assumes "nfinite g" 
          "nlength g = nlength l" 
  shows "nrev (upd g n l ) = (upd (nrev g) n (nrev l)) " 
proof -
 have 1: "nlength (nrev (upd g n l )) = nlength (upd (nrev g) n (nrev l)) "
  by (simp add: assms(2) nlength_upd)
 have 2: "\<And>i. i \<le> nlength (nrev (upd g n l )) \<longrightarrow>
           (nnth (nrev (upd g n l )) i) = 
           (nnth (upd (nrev g) n (nrev l)) i)"
 using assms by (auto simp add:  nlength_upd nnth_nrev nnth_upd nfinite_conv_nlength_enat)
  show ?thesis
 using "1" "2" nellist_eq_nnth_eq by blast  
qed   



lemma upd_exist: 
 "\<exists> r2 ur2. nlength r2 = nlength r1 \<and> nlength ur2 = nlength r1 \<and> r1 = upd r2 n ur2 " 
proof -
 have 1: "\<exists> ur1. nlength ur1 = nlength r1 \<and> (\<forall> i. i\<le> nlength r1 \<longrightarrow> (nnth ur1 i) = ( n \<in> (nnth r1 i)) )" 
   using nlength_nmap nnth_nmap by blast
 have 2: "\<exists> r3. nlength r3 = nlength r1 \<and> (\<forall> i. i \<le> nlength r1 \<longrightarrow> (nnth r3 i) = (nnth r1 i) -{n} )" 
  by (metis nlength_nmap nnth_nmap Set.remove_eq)
 obtain ur1 where 3: " nlength ur1 = nlength r1 \<and> (\<forall> i. i\<le> nlength r1 \<longrightarrow> (nnth ur1 i) = ( n \<in> (nnth r1 i)) )"
   using 1 by blast
 obtain r3 where 4: "nlength r3 = nlength r1 \<and> (\<forall> i. i \<le> nlength r1 \<longrightarrow> (nnth r3 i) = (nnth r1 i) -{n} )"
   using 2 by blast
 have 5: " r1 = upd r3 n ur1 " 
 by (metis "3" "4" Diff_empty Diff_insert0 insert_Diff nellist_eq_nnth_eq nlength_upd nnth_upd)  
 show ?thesis 
 using "3" "4" "5" by auto
qed

lemma not_fvar_upd: 
 assumes " n \<notin> fvars f"
         " nlength ug = nlength g" 
         " i \<le> j" 
         " j \<le> nlength g" 
 shows " (g i j \<Turnstile>\<^sub>i f) = ((upd g n ug) i j \<Turnstile>\<^sub>i f) " 
using assms
proof (induction f arbitrary: g i j ug)
case false_d
then show ?case by auto
next
case (atom_d x)
then show ?case 
by (simp add: nlength_upd nnth_upd)
next
case (iimp_d f1 f2)
then show ?case by simp
next
case empty_d
then show ?case using nlength_upd by auto
next
case (c_d f1 f2)
then show ?case 
  proof -
   have c1: "(g i j \<Turnstile>\<^sub>i  f1 cee f2 ) = 
             (\<exists>k\<ge>i. enat k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i f1) \<and> (g k j \<Turnstile>\<^sub>i f2))"
   by simp
   have c2: "((upd g n ug) i j \<Turnstile>\<^sub>i  f1 cee f2 ) = 
            (\<exists>k\<ge>i. enat k \<le> j \<and> ((upd g n ug) i (enat k) \<Turnstile>\<^sub>i f1) \<and> ((upd g n ug) k j \<Turnstile>\<^sub>i f2))"
    by simp
   have c3: "(\<exists>k\<ge>i. enat k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i f1) \<and> (g k j \<Turnstile>\<^sub>i f2)) \<Longrightarrow>
             (\<exists>k\<ge>i. enat k \<le> j \<and> ((upd g n ug) i (enat k) \<Turnstile>\<^sub>i f1) \<and> ((upd g n ug) k j \<Turnstile>\<^sub>i f2))"
   using c_d by auto  
  have c5: "(\<exists>k\<ge>i. enat k \<le> j \<and> ((upd g n ug) i (enat k) \<Turnstile>\<^sub>i f1) \<and> ((upd g n ug) k j \<Turnstile>\<^sub>i f2))
            \<Longrightarrow> (\<exists>k\<ge>i. enat k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i f1) \<and> (g k j \<Turnstile>\<^sub>i f2)) "
   using c_d by auto
   show ?thesis   using c1 c2 c3 c5 by argo 
  qed
next
case (d_d f1 f2)
then show ?case 
   proof -
    have d1: "(g i j \<Turnstile>\<^sub>i  f1 dee f2 ) =
               (\<exists>k\<le>i. (g k (enat i) \<Turnstile>\<^sub>i f1) \<and> (g k j \<Turnstile>\<^sub>i f2))"
     by simp
    have d2: "((upd g n ug) i j \<Turnstile>\<^sub>i  f1 dee f2 ) =
               (\<exists>k\<le>i. ((upd g n ug) k (enat i) \<Turnstile>\<^sub>i f1) \<and> ((upd g n ug) k j \<Turnstile>\<^sub>i f2))"
      by simp
    have d3: "(\<exists>k\<le>i. (g k (enat i) \<Turnstile>\<^sub>i f1) \<and> (g k j \<Turnstile>\<^sub>i f2)) \<Longrightarrow>
              (\<exists>k\<le>i. ((upd g n ug) k (enat i) \<Turnstile>\<^sub>i f1) \<and> ((upd g n ug) k j \<Turnstile>\<^sub>i f2))"
    using d_d 
     by (metis Un_iff dual_order.trans fvars.simps(7) of_nat_eq_enat of_nat_mono)
    have d4: "(\<exists>k\<le>i. ((upd g n ug) k (enat i) \<Turnstile>\<^sub>i f1) \<and> ((upd g n ug) k j \<Turnstile>\<^sub>i f2)) \<Longrightarrow>
              (\<exists>k\<le>i. (g k (enat i) \<Turnstile>\<^sub>i f1) \<and> (g k j \<Turnstile>\<^sub>i f2))"
    using d_d
    by (metis Un_iff  dual_order.trans enat_ord_simps(1) fvars.simps(7))        
    show ?thesis
    using d1 d2 d3 d4 by argo
   qed
next
case (t_d f1 f2)
then show ?case 
  proof - 
   have t1: "(g i j \<Turnstile>\<^sub>i  f1 tee f2 ) =
              ((\<exists>i. j = enat i) \<and> (\<exists>k\<ge>j. k \<le> nlength g \<and> (g (the_enat j) k \<Turnstile>\<^sub>i f1) \<and> (g i k \<Turnstile>\<^sub>i f2)))" 
   by simp
  have t2: "((upd g n ug) i j \<Turnstile>\<^sub>i  f1 tee f2 ) =
     ((\<exists>i. j = enat i) \<and> (\<exists>k\<ge>j. k \<le> nlength (upd g n ug) \<and> ((upd g n ug) (the_enat j) k \<Turnstile>\<^sub>i f1) \<and> ((upd g n ug) i k \<Turnstile>\<^sub>i f2)))" 
   by simp
  have t3: "((\<exists>i. j = enat i) \<and> (\<exists>k\<ge>j. k \<le> nlength g \<and> (g (the_enat j) k \<Turnstile>\<^sub>i f1) \<and> (g i k \<Turnstile>\<^sub>i f2))) \<Longrightarrow>
            ((\<exists>i. j = enat i) \<and> (\<exists>k\<ge>j. k \<le> nlength (upd g n ug) \<and> ((upd g n ug) (the_enat j) k \<Turnstile>\<^sub>i f1) \<and> ((upd g n ug) i k \<Turnstile>\<^sub>i f2)))"
   using t_d by simp
    (metis (no_types, lifting) nlength_upd order_trans the_enat.simps) 
  have t4: "((\<exists>i. j = enat i) \<and> (\<exists>k\<ge>j. k \<le> nlength (upd g n ug) \<and> ((upd g n ug) (the_enat j) k \<Turnstile>\<^sub>i f1) \<and> ((upd g n ug) i k \<Turnstile>\<^sub>i f2)))
            \<Longrightarrow>
            ((\<exists>i. j = enat i) \<and> (\<exists>k\<ge>j. k \<le> nlength g \<and> (g (the_enat j) k \<Turnstile>\<^sub>i f1) \<and> (g i k \<Turnstile>\<^sub>i f2)))"
  using  t_d by simp
   (metis dual_order.trans nlength_upd the_enat.simps) 
  show ?thesis 
  using t1 t2 t3 t4 by argo 
 qed
qed

lemma inot_defs :
 shows " (g i j \<Turnstile>\<^sub>i inot f ) = Not (g i j \<Turnstile>\<^sub>i f) " 
by (simp add: inot_d_def)

lemma ior_defs : 
  " (g i j \<Turnstile>\<^sub>i f1 ior f2) = ( (g i j \<Turnstile>\<^sub>i f1) \<or> (g i j \<Turnstile>\<^sub>i f2))"
by (metis inot_defs ior_d_def semantics_cdt.simps(3))

lemma iand_defs :
  "  (g i j \<Turnstile>\<^sub>i f1 iand f2) = ( (g i j \<Turnstile>\<^sub>i f1) \<and> (g i j \<Turnstile>\<^sub>i f2))"
  using iand_d_def inot_defs ior_defs by presburger

lemma ieqv_defs : 
  " (g i j \<Turnstile>\<^sub>i f1 ieqv f2) = ( (g i j \<Turnstile>\<^sub>i f1) = (g i j \<Turnstile>\<^sub>i f2))"
  using iand_defs ieqv_d_def by auto

lemma itrue_defs :
  " (g i j \<Turnstile>\<^sub>i itrue) "
by (simp add: inot_d_def itrue_d_def)

lemma empty_defs : 
 "  (g i j \<Turnstile>\<^sub>i empty) = (i = j)   " 
by simp

lemma more_defs: 
 " (g i j \<Turnstile>\<^sub>i more) \<longleftrightarrow> (i \<noteq> j) " 
 unfolding more_d_def
 using empty_defs i0_less inot_defs by presburger 

lemma skip_defs: 
assumes "i \<le>j"
        " j \<le> nlength g"  
shows " (g i j \<Turnstile>\<^sub>i skip) \<longleftrightarrow> j = enat (Suc i)"
proof -
 have 1: "(g i j \<Turnstile>\<^sub>i skip) \<longleftrightarrow> (g i j \<Turnstile>\<^sub>i more iand inot ( more cee more ))" 
  unfolding skip_d_def by blast
 have 2: "(g i j \<Turnstile>\<^sub>i more iand inot ( more cee more )) \<longleftrightarrow>
          (g i j \<Turnstile>\<^sub>i more) \<and> (g i j \<Turnstile>\<^sub>i inot ( more cee more ))"
   using  iand_defs by presburger 
 have 3: " (g i j \<Turnstile>\<^sub>i inot ( more cee more )) \<longleftrightarrow> Not (g i j \<Turnstile>\<^sub>i  ( more cee more ))  " 
  using inot_defs by presburger 
 have 4: "(g i j \<Turnstile>\<^sub>i  ( more cee more )) \<longleftrightarrow> 
     (\<exists>k\<ge>i. enat k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i more) \<and> (g k j \<Turnstile>\<^sub>i more))"
    by simp   
 have 5: " (\<exists>k\<ge>i. enat k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i more) \<and> (g k j \<Turnstile>\<^sub>i more)) \<longleftrightarrow>
           (\<exists>k\<ge>i. enat k \<le> j \<and> i \<noteq> (enat k) \<and> k \<noteq> j ) "
   using more_defs by presburger
 have 7: " (\<exists>k\<ge>i. enat k \<le> j \<and> i \<noteq> (enat k) \<and> k \<noteq> j ) \<Longrightarrow>
             j > enat (Suc i)" 
   by (metis Suc_ile_eq assms(1) enat_ord_simps(1) nle_le order_neq_le_trans)
 have 8: "j > enat (Suc i) \<Longrightarrow> (\<exists>k\<ge>i. enat k \<le> j \<and> i \<noteq> (enat k) \<and> k \<noteq> j )"
   by (metis enat.inject le_add2 linorder_neq_iff n_not_Suc_n order_less_imp_le plus_1_eq_Suc)
 have 9: "(\<exists>k\<ge>i. enat k \<le> j \<and> i \<noteq> (enat k) \<and> k \<noteq> j ) \<longleftrightarrow>
          j > enat (Suc i)"
   using 7 8 by blast
 have 10: "(g i j \<Turnstile>\<^sub>i  ( more cee more )) \<longleftrightarrow> (j > enat (Suc i))"
    using "4" "5" "7" "8" by argo
 have 11: "(g i j \<Turnstile>\<^sub>i more) \<and> (g i j \<Turnstile>\<^sub>i inot ( more cee more )) \<longleftrightarrow>
           i \<noteq>j \<and> Not(j > enat (Suc i))" 
    using "10" "3"  more_defs by presburger
 have 12: "i \<noteq>j \<and> Not(j > enat (Suc i)) \<longleftrightarrow>  j = enat (Suc i)" 
    by (metis Suc_ile_eq assms(1) dual_order.irrefl order_neq_le_trans)
   show ?thesis 
   using "1" "11" "12" "2" by presburger
qed        

lemma skip_cee_itrue_defs: 
 assumes "i \<le> j"
         "j \<le> nlength g"  
 shows "(g i j \<Turnstile>\<^sub>i (skip cee itrue)) = (i < j) "
proof -
 have 1: "(g i j \<Turnstile>\<^sub>i (skip cee itrue)) \<Longrightarrow> (i < j)"
   using assms nless_le skip_defs by fastforce    
 have 2: " (i < j) \<Longrightarrow> (g i j \<Turnstile>\<^sub>i (skip cee itrue))"
   using assms 
   by (metis eSuc_enat ileI1 ile_eSuc itrue_defs order.trans semantics_cdt.simps(5) skip_defs)
 show ?thesis using 1 2 by blast
qed

lemma itrue_cee_skip_defs: 
 assumes "i \<le> j"
         " j \<le> nlength g" 
 shows "(g i j \<Turnstile>\<^sub>i (itrue cee skip)) = (i < j \<and> (\<exists> k. j = (enat k))) "  
proof -
 have 1: "(g i j \<Turnstile>\<^sub>i (itrue cee skip)) \<Longrightarrow> (i < j \<and> (\<exists> k. j = (enat k)))"
    using assms by simp
     (metis enat_ord_simps(1) iand_defs more_defs nle_le order_neq_le_trans skip_d_def skip_defs) 
 have 2: "(i < j \<and> (\<exists> k. j = (enat k)))  \<Longrightarrow> (g i j \<Turnstile>\<^sub>i (itrue cee skip))"
   proof -
    assume a0: "(i < j \<and> (\<exists> k. j = (enat k)))"  
    show "(g i j \<Turnstile>\<^sub>i (itrue cee skip))"
    proof -
     have 21: "(g i j \<Turnstile>\<^sub>i (itrue cee skip)) = (\<exists>k\<ge>i. enat k \<le> j  \<and> j = (enat (Suc k)))"
       using assms 
       by (meson enat_ord_simps(1) itrue_defs semantics_cdt.simps(5) skip_defs)
    have 22: "(\<exists>k\<ge>i. enat k \<le> j  \<and> j = (enat (Suc k)))"
     using assms a0 
     using Nat.lessE by fastforce 
    show ?thesis 
    using "21" "22" by blast
   qed
  qed
 show ?thesis using 1 2 by blast
qed

lemma ifinite_defs:
 assumes "i \<le> j"
         " j \<le> nlength g" 
 shows   "(g i j \<Turnstile>\<^sub>i ifinite) \<longleftrightarrow> (\<exists>k. j = (enat k))" 
using assms  unfolding finite_d_def itrue_d_def inot_d_def 
by auto


lemma inf_defs: 
assumes " i \<le> j"
        " j \<le> nlength g"  
shows "(g i j \<Turnstile>\<^sub>i inf) \<longleftrightarrow> j = \<infinity>" 
using assms
unfolding inf_d_def by (simp add: ifinite_defs inot_defs) 

lemma chop_defs:
 assumes " i \<le> j"
        " j \<le> nlength g" 
 shows "(g i j \<Turnstile>\<^sub>i f1 ; f2 )  = 
         (\<exists> k. (enat i) \<le> (enat k) \<and> (enat k) \<le> j \<and> ( g i k \<Turnstile>\<^sub>i f1) \<and> ( g k j \<Turnstile>\<^sub>i f2) ) \<or>
         ( j = \<infinity> \<and> ( g i j \<Turnstile>\<^sub>i f1) )" 
using assms apply (simp add: iand_defs ior_defs chop_d_def inf_defs) 
by auto

lemma diamond_defs:
assumes " i \<le> j"
        " j \<le> nlength g" 
 shows "(g i j \<Turnstile>\<^sub>i diamond f) \<longleftrightarrow> 
        (\<exists> k. i \<le> k \<and> k \<le> j \<and> ( g k j  \<Turnstile>\<^sub>i f))" 
using assms
unfolding diamond_d_def 
by (simp add: inot_defs itrue_defs)

lemma box_defs:
assumes " i \<le> j"
        " j \<le> nlength g" 
 shows "(g i j \<Turnstile>\<^sub>i box f) \<longleftrightarrow> 
        (\<forall> k. i \<le> k \<and> k \<le> j \<longrightarrow> ( g k j  \<Turnstile>\<^sub>i f))" 
using assms unfolding box_d_def 
by (simp add: inot_defs diamond_defs) 
 blast

lemma fin_defs: 
assumes " i \<le> j"
        " j \<le> nlength g"  
shows "(g i j \<Turnstile>\<^sub>i (fin $p)) \<longleftrightarrow> ((\<exists> k. j =(enat k))  \<and> p \<in> (nnth g (the_enat j))) " 
using assms
unfolding fin_d_def
by (auto simp add: diamond_defs iand_defs ) 

lemma df_defs:
assumes " i \<le> j"
        " j \<le> nlength g" 
 shows "(g i j \<Turnstile>\<^sub>i df f) \<longleftrightarrow> 
        (\<exists> k. i \<le> k \<and> k \<le> j  \<and> (g i (enat k) \<Turnstile>\<^sub>i f))" 
using assms
unfolding df_d_def 
by (simp add: inot_defs itrue_defs)

lemma atom_diamond_defs:
assumes "i \<le> j"
        " j \<le> nlength g" 
 shows "(g i j \<Turnstile>\<^sub>i diamond $q) \<longleftrightarrow> (\<exists> k. i \<le> k \<and> k \<le> j \<and> q \<in> (nnth g k))" 
using assms diamond_defs
by auto

lemma atom_df_fin_defs:
assumes " i \<le> j"
        " j \<le> nlength g" 
 shows "(g i j \<Turnstile>\<^sub>i df (fin $q)) \<longleftrightarrow> (\<exists> k. i \<le> k \<and> k \<le> j  \<and> q \<in> (nnth g k))" 
using assms df_defs fin_defs
by auto

lemma atom_box_defs:
 assumes " i \<le> j"
         " j \<le> nlength g"
 shows "(g i j \<Turnstile>\<^sub>i box $q) \<longleftrightarrow> (\<forall> k. i \<le> k \<and> k \<le> j  \<longrightarrow> q \<in> (nnth g k))" 
using assms box_defs
by auto

lemma atom_fin_defs:
 assumes " i \<le> j"
         " j \<le> nlength g" 
 shows "(g i j \<Turnstile>\<^sub>i  fin $q) \<longleftrightarrow> ((\<exists>k. j = (enat k)  \<and> q \<in> (nnth g (the_enat j)))) " 
using assms
using fin_defs nnth_nlast by blast
 
lemma atom_lbox_defs: 
 assumes " i \<le> j"
         " j \<le> nlength g"  
 shows "(g i j \<Turnstile>\<^sub>i (lbox $q) ) \<longleftrightarrow> 
        (\<forall> k.  k \<le> i  \<longrightarrow> q \<in> (nnth g k)) "
using assms by (auto simp add: inot_defs lbox_d_def ldiamond_d_def itrue_defs) 
 (meson enat_ord_simps(1) order_trans)

lemma atom_rbox_defs: 
 assumes " i \<le> j"
         " j \<le> nlength g"         
 shows "(g i j \<Turnstile>\<^sub>i ifinite iimp rbox (fin $q) ) \<longleftrightarrow> 
      ( (\<exists> k . j = (enat k)) \<longrightarrow>  (\<forall> k. j \<le> k \<and> k \<le> nlength g \<longrightarrow> q \<in> (nnth g k))) "
using assms 
by (auto simp add: rbox_d_def rdiamond_d_def inot_defs itrue_defs iand_defs ifinite_defs 
        fin_d_def diamond_d_def ) 

lemma hor_defs: 
assumes " i \<le> j"
        " j \<le> nlength g"  
shows "(g i j \<Turnstile>\<^sub>i (hor q)) \<longleftrightarrow> 
        (\<forall> k. k \<le> nlength g \<longrightarrow> q \<in> (nnth g k))  " 
using assms 
using atom_box_defs hor_def iand_defs atom_lbox_defs atom_rbox_defs
by simp
 (meson dual_order.trans enat_ile enat_ord_simps(1) nle_le) 

lemma ediamond_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i ediamond f) = (\<exists>k. i < k \<and> enat k \<le> j \<and> (g k j \<Turnstile>\<^sub>i f))" 
using assms 
apply (simp add: ediamond_d_def more_d_def inot_defs)
using dual_order.strict_iff_order by blast 

lemma wdiamond_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i wdiamond f) = (\<exists>k. k < i \<and> (g k j \<Turnstile>\<^sub>i f)) " 
using assms 
apply (simp add: wdiamond_d_def more_d_def  inot_defs)
using order.order_iff_strict by blast

lemma sdiamond_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i sdiamond f) = (\<exists>k. i \<le> k \<and>  enat k < j \<and> (g i (enat k) \<Turnstile>\<^sub>i f) ) " 
using assms 
apply (simp add: sdiamond_d_def more_d_def  inot_defs)
by (metis linorder_neq_iff order_le_imp_less_or_eq order_less_imp_le)

lemma ndiamond_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i ndiamond f) = 
       (j < \<infinity> \<and> (\<exists>k. j< k  \<and> k \<le> nlength g  \<and> (g i k \<Turnstile>\<^sub>i f))) " 
using assms 
apply (simp add: ndiamond_d_def more_d_def  inot_defs)
by (metis linorder_neq_iff order_le_imp_less_or_eq order_less_imp_le the_enat.simps)

lemma ewdiamond_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i ewdiamond f) =  
        (\<exists>k.  enat k \<le> j \<and> (g k j \<Turnstile>\<^sub>i f))   " 
using assms  
apply (simp add: ewdiamond_d_def ediamond_defs wdiamond_defs ior_defs)
by (metis enat_ord_simps(2) linorder_less_linear order_less_imp_not_less order_less_le_subst2
  verit_comp_simplify1(3))

lemma nsdiamond_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i nsdiamond f) =
       ( (\<exists>k. i \<le> k \<and> k \<le> nlength g \<and> (g i k \<Turnstile>\<^sub>i f) ) ) " 
using assms 
apply (auto simp add: ior_defs nsdiamond_d_def ndiamond_defs sdiamond_defs)
apply (meson dual_order.trans enat_ord_simps(1) order_less_imp_le)
apply (metis assms(1) dual_order.trans order_less_imp_le)
apply (metis enat_ord_simps(1) enat_ord_simps(4) ifinite_defs inf_d_def inf_defs inot_defs)
by (metis enat_ord_simps(1) ifinite_defs inf_d_def inf_defs inot_defs le_iff_add 
    not_less_iff_gr_or_eq order_neq_le_trans plus_eq_infty_iff_enat)



lemma ebox_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i ebox f) = (\<forall>k. i < k \<and> enat k \<le> j \<longrightarrow> (g k j \<Turnstile>\<^sub>i f))" 
using assms ediamond_defs ebox_d_def inot_defs by auto 

lemma ebox_ifalse_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i ebox ifalse) = (i = j)"
using assms ebox_defs[of i j g ifalse]
by (metis Suc_ile_eq antisym_conv2 enat_ord_simps(1) leD lessI semantics_cdt.simps(1)) 
 
lemma wbox_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i wbox f) = (\<forall>k. k < i \<longrightarrow> (g k j \<Turnstile>\<^sub>i f)) " 
using assms wdiamond_defs wbox_d_def inot_defs by auto

lemma lex_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i lex) = (i=0) "
using assms using lex_d_def wbox_defs
by (metis antisym_conv1 le_add1 le_add_same_cancel1 less_nat_zero_code semantics_cdt.simps(1))

lemma sbox_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i sbox f) = (\<forall>k. i \<le> k \<and>  enat k < j \<longrightarrow> (g i (enat k) \<Turnstile>\<^sub>i f) ) " 
using assms sdiamond_defs sbox_d_def inot_defs by auto

lemma sbox_ifalse_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i sbox ifalse) = (i = j) " 
using assms sbox_defs[of i j g ifalse]
by fastforce

lemma nbox_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i nbox f) = 
       (j < \<infinity> \<longrightarrow> (\<forall>k. j< k  \<and> k \<le> nlength g  \<longrightarrow> (g i k \<Turnstile>\<^sub>i f))) " 
using assms ndiamond_defs nbox_d_def inot_defs by auto

lemma rex_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i rex) = (j = nlength g) " 
using assms 
by (metis enat_ord_simps(4) rex_d_def linorder_not_le nbox_defs order.order_iff_strict semantics_cdt.simps(1))

lemma ewbox_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i ewbox f) =  
        (\<forall>k.  enat k \<le> j \<longrightarrow> (g k j \<Turnstile>\<^sub>i f))   " 
using assms ewdiamond_defs ewbox_d_def inot_defs by auto

lemma ebox_ifalse_iand_lex_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i ebox ifalse iand lex) = (i = 0 \<and> i = j)    "
using assms ebox_ifalse_defs iand_defs lex_d_def lex_defs zero_enat_def by auto

lemma nsbox_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i nsbox f) =
       ( (\<forall>k. i \<le> k \<and> k \<le> nlength g \<longrightarrow> (g i k \<Turnstile>\<^sub>i f) ) ) " 
using assms nsdiamond_defs nsbox_d_def inot_defs by auto

lemma rex_iand_sbox_ifalse_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i rex iand sbox ifalse) = (i = j \<and> j = nlength g) " 
using assms iand_defs rex_d_def rex_defs sbox_ifalse_defs by auto

lemma nbox_ewbox_inot_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i nbox (ewbox inot f)) = 
       (j < \<infinity> \<longrightarrow> 
        (\<forall>k. j < k \<and> k \<le> nlength g \<longrightarrow>
          (\<forall>k1.  enat k1 \<le> k \<longrightarrow> \<not>(g k1 k \<Turnstile>\<^sub>i f))
          ) )  " 
using assms ewbox_defs inot_defs nbox_defs[of i j g "(ewbox inot f)"] 
apply simp
by (meson order.order_iff_strict order_trans)

 
lemma sbox_ewbox_inot_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i sbox (ewbox inot f)) =
        (\<forall>k. i \<le> k \<and> enat k < j \<longrightarrow> 
         (\<forall>k1.  enat k1 \<le> (enat k) \<longrightarrow> \<not> (g k1 (enat k) \<Turnstile>\<^sub>i  f))
         ) " 
using assms sbox_defs[of i j g "(ewbox inot f)"]
using ewbox_defs inot_defs by auto

lemma ahor_defs: 
assumes  "i \<le> j" 
         "j\<le>nlength g " 
shows "(g i j \<Turnstile>\<^sub>i ahor f) = 
        ( (\<forall>k.  enat k \<le> j \<longrightarrow> (g k j \<Turnstile>\<^sub>i f)) \<and>
          (j < \<infinity> \<longrightarrow> 
             (\<forall>k. j < k \<and> k \<le> nlength g \<longrightarrow>
               (\<forall>k1.  enat k1 \<le> k \<longrightarrow> \<not>(g k1 k \<Turnstile>\<^sub>i f))
             ) ) \<and>
          (\<forall>k. i \<le> k \<and> enat k < j \<longrightarrow> 
            (\<forall>k1.  enat k1 \<le> (enat k) \<longrightarrow> \<not>(g k1 (enat k) \<Turnstile>\<^sub>i f))
          ) 
         )"
using assms ahor_d_def ewbox_defs nbox_ewbox_inot_defs sbox_ewbox_inot_defs
using iand_defs by presburger


lemma ahor_defs_alt: 
assumes "i \<le> j" 
         "j\<le>nlength g " 
 shows "(g i j \<Turnstile>\<^sub>i ahor f) = 
        (\<forall>k k1 . i \<le>k \<and>  enat k1 \<le> k \<and> k \<le> nlength g \<longrightarrow> (k = j \<longleftrightarrow> (g k1 k \<Turnstile>\<^sub>i f))) " 
using assms ahor_defs apply auto
apply (metis enat_ord_simps(1) enat_ord_simps(4) less_infinityE)
apply (metis enat_ord_simps(1) enat_ord_simps(4) leI less_infinityE nle_le)
apply (metis assms(1) dual_order.strict_trans2 less_le_not_le)
by (metis dual_order.trans enat_ord_simps(1) linorder_neq_iff order_less_imp_le)



subsection \<open>Validity\<close>



definition valid :: " cdt \<Rightarrow> bool" ("(\<turnstile>\<^sub>i _)" 5)
where "(\<turnstile>\<^sub>i f) = (\<forall> g i j. (enat i) \<le> j \<and> j \<le> nlength g  \<longrightarrow> (g i j \<Turnstile>\<^sub>i f)) " 


lemma cdt_valid [simp] :
  " (\<turnstile>\<^sub>i f) = (\<forall> g i j. (enat i) \<le>j \<and> j \<le> nlength g \<longrightarrow> (g i j \<Turnstile>\<^sub>i f))"
by (simp add: valid_def)

lemma cdt_ieq:
  " (\<turnstile>\<^sub>i f1 ieqv f2) = (\<forall> g i j. i \<le> j \<and> j \<le> nlength g \<longrightarrow> ((g i j \<Turnstile>\<^sub>i f1) = (g i j \<Turnstile>\<^sub>i f2))) "
  using ieqv_defs valid_def by presburger

lemma more_iand_help: 
assumes "i \<le> j"
        " j \<le> nlength g" 
shows  "g i j \<Turnstile>\<^sub>i (more iand ifinite) iimp (skip cee itrue) iand (itrue cee skip) " 
using assms skip_cee_itrue_defs itrue_cee_skip_defs iand_defs more_defs ifinite_defs
apply simp
by force 


lemma Venema_discrete: 
 "\<turnstile>\<^sub>i (more iand ifinite) iimp (skip cee itrue) iand (itrue cee skip)" 
using more_iand_help valid_def by presburger

lemma Venema_B1a: 
 "\<turnstile>\<^sub>i (f1 ior f2) cee g ieqv (f1 cee g ior f2 cee g) " 
unfolding valid_def 
by (auto simp add: ieqv_defs ior_defs ) 

lemma Venema_B1b: 
 "\<turnstile>\<^sub>i f cee (g1 ior g2) ieqv (f cee g1 ior f cee g2) " 
unfolding valid_def 
by (auto simp add: ieqv_defs ior_defs) 
 
lemma Venema_B2a: 
 "\<turnstile>\<^sub>i (f1 ior f2) tee g ieqv (f1 tee g ior f2 tee g)" 
unfolding valid_def 
by (auto simp add: ieqv_defs ior_defs ) 

lemma Venema_B2b: 
 "\<turnstile>\<^sub>i f tee (g1 ior g2) ieqv (f tee g1 ior f tee g2)" 
unfolding valid_def 
by (auto simp add: ieqv_defs ior_defs ) 

lemma Venema_B3a: 
 "\<turnstile>\<^sub>i (f1 ior f2) dee g ieqv (f1 dee g ior f2 dee g) " 
unfolding valid_def 
by (auto simp add: ieqv_defs ior_defs ) 

lemma Venema_B3b: 
 "\<turnstile>\<^sub>i f dee (g1 ior g2) ieqv (f dee g1 ior f dee g2) " 
unfolding valid_def 
by (auto simp add: ieqv_defs ior_defs ) 

lemma Venema_C1: 
 "\<turnstile>\<^sub>i (inot(f tee g)) cee f  iand ifinite iimp inot g"
by (auto simp add: inot_defs   iand_defs ifinite_defs) 

lemma Venema_C2: 
 "\<turnstile>\<^sub>i (inot(f tee g)) dee g  iimp inot (f )"
by (auto simp add:  inot_defs iand_defs inf_d_def ifinite_defs ) 

lemma Venema_C3:  
 "\<turnstile>\<^sub>i f tee (inot (g cee f)) iimp inot g" 
by (auto simp add: inot_defs   ) 

lemma Venema_D: 
 "\<turnstile>\<^sub>i more cee itrue  ieqv more "
by (auto simp  add: ieqv_defs more_defs inot_defs itrue_defs skip_defs  )
 (metis Suc_ile_eq Suc_n_not_le_n linorder_linear order.strict_iff_order)

lemma Venema_E1: 
 "\<turnstile>\<^sub>i empty cee f ieqv f "
by (auto simp add: ieqv_defs ) 

lemma Venema_E2: 
 "\<turnstile>\<^sub>i empty tee f ieqv (f iand ifinite) " 
by (auto simp add:  ifinite_defs ieqv_defs iand_defs )

lemma Venema_E3: 
 "\<turnstile>\<^sub>i f tee empty ieqv (f iand empty)" 
by (auto simp add:  ieqv_defs iand_defs ifinite_defs )

lemma Venema_E4: 
 "\<turnstile>\<^sub>i empty dee f ieqv f" 
by (auto simp add:    ieqv_defs )

lemma Venema_E5: 
 "\<turnstile>\<^sub>i f cee empty ieqv (f iand ifinite)" 
by (auto simp add:    ifinite_defs ieqv_defs iand_defs  )

lemma Venema_F: 
 "\<turnstile>\<^sub>i ( (empty iand f) cee itrue iand ( (empty iand g) cee itrue ) cee itrue) iimp (empty iand g) cee itrue" 
by (auto simp add:   iand_defs itrue_defs )

lemma cee_assoc_1: 
 assumes "(g i j \<Turnstile>\<^sub>i  f1 cee ( f2 cee f3 ) ) " 
 shows " (g i j \<Turnstile>\<^sub>i  ( f1 cee f2 ) cee f3 ) " 
using assms dual_order.trans by auto 

lemma cee_assoc_2: 
 assumes " (g i j \<Turnstile>\<^sub>i  ( f1 cee f2 ) cee f3 ) " 
 shows " (g i j \<Turnstile>\<^sub>i  f1 cee ( f2 cee f3 ) )"
using assms
by (meson order.trans semantics_cdt.simps(5)) 

lemma Venema_G1: 
 " \<turnstile>\<^sub>i f cee (g cee h) ieqv (f cee g) cee h"
unfolding valid_def ieqv_defs
using cee_assoc_1 cee_assoc_2 
by blast

lemma Venema_G2a: 
 "\<turnstile>\<^sub>i f tee (g cee h)  iimp ( g cee (f tee h) ior ( h tee f ) tee g) " 
by ( simp add: ior_defs  ) 
 (metis nle_le order_trans the_enat.simps)

lemma Venema_G2b: 
 "\<turnstile>\<^sub>i ( g cee (f tee h) ior ( h tee f ) tee g) iimp f tee (g cee h)   " 
by (simp add: ior_defs )
 (metis enat_ord_simps(1) order_trans the_enat.simps)

lemma Venema_G3: 
 " \<turnstile>\<^sub>i f cee ( g tee h) iimp g tee ( f cee h ) " 
by (auto ) 
 (meson enat_ord_simps(1) order.trans)

lemma Venema_G4: 
 " \<turnstile>\<^sub>i ( (f iand ifinite) tee g) cee h iimp (( h dee f) tee g ior g cee ( f dee h )) " 
unfolding valid_def
apply (auto simp add: ior_defs iand_defs ifinite_defs )
using ifinite_defs apply fastforce
by (metis dual_order.trans enat_ile enat_ord_simps(1) nle_le the_enat.simps)

lemma Venema_MP: 
 assumes "\<turnstile>\<^sub>i f"
         "\<turnstile>\<^sub>i f iimp g" 
 shows   "\<turnstile>\<^sub>i g" 
using assms unfolding valid_def by simp

lemma Venema_Gen_1a: 
 assumes "\<turnstile>\<^sub>i f"
 shows   "\<turnstile>\<^sub>i inot((inot f) cee g) " 
using assms unfolding valid_def 
apply (simp add: inot_defs) 
by force

lemma Venema_Gen_1b: 
 assumes "\<turnstile>\<^sub>i g"
 shows   "\<turnstile>\<^sub>i inot(g cee (inot g)) " 
using assms unfolding valid_def
by (simp add: inot_defs) 

lemma Venema_Gen_2a: 
 assumes "\<turnstile>\<^sub>i f" 
 shows   "\<turnstile>\<^sub>i inot((inot f) tee g) " 
using assms unfolding valid_def
apply (simp add: inot_defs) 
by force

lemma Venema_Gen_2b: 
 assumes "\<turnstile>\<^sub>i g" 
 shows   "\<turnstile>\<^sub>i inot(f  tee (inot g)) " 
using assms unfolding valid_def
apply (simp add: inot_defs) 
by (meson dual_order.trans)

lemma Venema_Gen_3a: 
 assumes "\<turnstile>\<^sub>i f" 
 shows   "\<turnstile>\<^sub>i inot((inot f) dee g) " 
using assms unfolding valid_def 
apply (simp add: inot_defs) 
by force


lemma Venema_Gen_3b: 
 assumes "\<turnstile>\<^sub>i g" 
 shows   "\<turnstile>\<^sub>i inot(f dee (inot g)) " 
using assms unfolding valid_def
apply (simp add: inot_defs) 
by (meson enat_ord_simps(1) order_trans)


lemma hor_rule: 
 assumes " q \<notin> fvars f" 
         " \<forall> g i j. 
              ( i \<le>j \<and> j \<le> nlength g \<and> 
                (\<forall> k. k \<le> nlength g \<longrightarrow> q \<in> (nnth g k)) ) 
              \<longrightarrow> (g i j \<Turnstile>\<^sub>i f) " 
         " i \<le> j " 
         " j \<le> nlength g" 
 shows " (g i j \<Turnstile>\<^sub>i f)"
proof -
 have 1: "\<exists> ul. nlength ul = nlength  g\<and> (\<forall> i. i \<le> nlength g \<longrightarrow> (nnth ul i) = True)"  
  by (metis inj_enat nlength_nmap nnth_nmap) 
 obtain ug where 4: " nlength ug = nlength g \<and> (\<forall> i. i \<le> nlength g \<longrightarrow> (nnth ug i) = True)"
   using 1 by blast 
 have 8: "(g i j \<Turnstile>\<^sub>i f) \<longleftrightarrow>  ((upd g q ug) i j \<Turnstile>\<^sub>i f) "  
   using not_fvar_upd[of q f ug g i j]
   using "4" assms by blast
 show ?thesis using assms 
 by (simp add: "4" "8" nlength_upd nnth_upd)
qed


lemma Venema_Consistency: 
 assumes "\<turnstile>\<^sub>i hor(q) iimp f "
         " q \<notin> fvars f" 
 shows   "\<turnstile>\<^sub>i f" 
using hor_rule[of q f] hor_defs assms by force

definition valid_finite :: " cdt \<Rightarrow> bool" ("(\<turnstile>\<^sub>f _)" 5)
where "(\<turnstile>\<^sub>f f) = (\<forall> g i j. (enat i) \<le> j \<and> j \<le> nlength g \<and> nfinite g \<longrightarrow> (g i j \<Turnstile>\<^sub>i f)) " 

lemma valid_finite: 
 assumes "\<turnstile>\<^sub>i f"
 shows "\<turnstile>\<^sub>f f" 
using assms  valid_finite_def
using valid_def by blast

primrec rva :: "cdt \<Rightarrow> cdt" ("(rva _)" [85] 85)
where
  "rva ifalse = ifalse "
| "rva ($p) = (fin $p) " 
| "rva (f iimp g) = ((rva f) iimp (rva g)) " 
| "rva empty = empty " 
| "rva (f cee g) = ( (rva g) cee (rva f)) " 
| "rva (f tee g) = ( (rva f) dee (rva g)) "
| "rva (f dee g) = ( (rva f) tee (rva g)) "
 
lemma rva_sem: 
assumes "nfinite g" 
        "i \<le> j"
        " j \<le> nlength g "
shows " (g i j \<Turnstile>\<^sub>i (rva f)) = ( (nrev g) (the_enat(nlength g) -j) (nlength g - i) \<Turnstile>\<^sub>i f) " 
using assms
proof (induction f arbitrary: g i j)
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case apply (simp add: fin_defs) 
by (metis diff_diff_cancel diff_le_self enat_ord_simps(1) nfinite_conv_nlength_enat nnth_nrev the_enat.simps) 
next
case (iimp_d f1 f2)
then show ?case by simp
next
case empty_d
then show ?case  apply (simp ) 
by (metis diff_diff_cancel dual_order.trans enat_ord_simps(1) idiff_enat_enat nfinite_conv_nlength_enat the_enat.simps)
next
case (c_d f1 f2)
then show ?case 
 proof -
  have c1: "(g i j \<Turnstile>\<^sub>i (rva (f1 cee f2))) = (g i j \<Turnstile>\<^sub>i (rva f2) cee (rva f1)) "
    by simp
  have c2: "(g i j \<Turnstile>\<^sub>i (rva f2) cee (rva f1)) = 
            (\<exists>k\<ge>i. k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i rva f2) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f1))"
    by simp
  have c3: " ((nrev g) (the_enat (nlength g) - j) (nlength g - enat i) \<Turnstile>\<^sub>i  f1 cee f2 )  = 
          (\<exists>k\<ge>the_enat (nlength g) - j. enat k \<le> nlength g - enat i \<and>
              ((nrev g) (the_enat (nlength g) - j) (enat k) \<Turnstile>\<^sub>i f1) \<and> ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2))  " 
   by simp
  have c4: "(\<exists>k\<ge>i. k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i rva f2) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f1)) \<Longrightarrow>
            (\<exists>k\<ge>the_enat (nlength g) - j. enat k \<le> nlength g - enat i \<and>
              ((nrev g) (the_enat (nlength g) - j) (enat k) \<Turnstile>\<^sub>i f1) \<and> ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2)) "
   proof -
    assume ca0: "(\<exists>k\<ge>i. k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i rva f2) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f1))"
    show "(\<exists>k\<ge>the_enat (nlength g) - j. enat k \<le> nlength g - enat i \<and>
              ((nrev g) (the_enat (nlength g) - j) (enat k) \<Turnstile>\<^sub>i f1) \<and> ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2))" 
    proof -

     obtain k where c100: "k\<ge>i \<and> k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i rva f2) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f1)"
      using ca0 by blast
     have c101: " ( (nrev g) (the_enat(nlength g) - k) (nlength g - i) \<Turnstile>\<^sub>i f2)"
      by (meson c100 c_d.IH(2) c_d.prems(1) c_d.prems(3) enat_ord_simps(1) order_trans)
     have c102: " ( (nrev g) (the_enat(nlength g) - j) (nlength g - k) \<Turnstile>\<^sub>i f1) "
       using c100 c_d.IH(1) c_d.prems(1) c_d.prems(3) by blast
   
     show ?thesis 
     by (metis c100 c101 c102 c_d.prems(1) diff_le_mono2 enat_ord_simps(1) idiff_enat_enat nfinite_conv_nlength_enat the_enat.simps)
    qed
   qed
 have c5: "(\<exists>k\<ge>the_enat (nlength g) - j. enat k \<le> nlength g - enat i \<and>
              ((nrev g) (the_enat (nlength g) - j) (enat k) \<Turnstile>\<^sub>i f1) \<and> ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2)) \<Longrightarrow>
          (\<exists>k\<ge>i. k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i rva f2) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f1)) "
  proof -
   assume ca0: "(\<exists>k\<ge>the_enat (nlength g) - j. enat k \<le> nlength g - enat i \<and>
              ((nrev g) (the_enat (nlength g) - j) (enat k) \<Turnstile>\<^sub>i f1) \<and> ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2))"
   show "(\<exists>k\<ge>i. k \<le> j \<and> (g i (enat k) \<Turnstile>\<^sub>i rva f2) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f1))"
   proof -
    obtain k where c200: "k\<ge>the_enat (nlength g) - j \<and> enat k \<le> nlength g - enat i \<and>
              ((nrev g) (the_enat (nlength g) - j) (enat k) \<Turnstile>\<^sub>i f1) \<and> ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2)"
    using ca0 by blast
   have c201: "i \<le> the_enat (nlength g) - k" 
    by (metis c200 c_d.prems(1) c_d.prems(2) c_d.prems(3) diff_diff_cancel diff_le_mono2 dual_order.trans enat_ord_simps(1) idiff_enat_enat nfinite_conv_nlength_enat the_enat.simps) 
   have c2011: " the_enat (nlength g) - (the_enat (nlength g) - k) = k " 
   by (metis c200 c201 c_d.prems(1) diff_diff_cancel diff_is_0_eq enat.distinct(2) enat_ord_simps(1) enat_the_enat idiff_0_right nfinite_conv_nlength_enat nle_le zero_enat_def)  
   have c202: " nlength g - enat (the_enat (nlength g) - k) = (enat k) " 
    by (metis c200 c_d.prems(1) diff_diff_cancel diff_le_self dual_order.trans enat_ord_simps(1) idiff_enat_enat nfinite_conv_nlength_enat the_enat.simps)
    have c203: "the_enat (nlength g) - k \<le> j"
    using c200 by linarith
    have c204: "(g (the_enat (nlength g) -k)  (enat j) \<Turnstile>\<^sub>i rva f1)" using c_d.IH(1)[of g "(the_enat (nlength g) -k)" j ] 
      using c200 c202 c203 c_d.prems(1) c_d.prems(3) by presburger
    have c205: "(g i (enat (the_enat (nlength g) -k)) \<Turnstile>\<^sub>i (rva f2))"  using c_d.IH(2)[of g i  "(the_enat (nlength g) -k)" ]
      by (metis c200 c201 c2011 c203 c_d.prems(1) c_d.prems(3) enat_ord_simps(1) order.trans)
    show ?thesis
    using c201 c203 c204 c205 by blast
   qed
  qed
 show ?thesis using c1 c2 c3 c4 c5 by blast
 qed 
next
case (d_d f1 f2)
then show ?case 
 proof -
  have d1: "(g i j \<Turnstile>\<^sub>i rva ( f1 dee f2 )) = 
            (\<exists>k\<ge>enat j. k \<le> nlength g \<and> (g j k \<Turnstile>\<^sub>i rva f1) \<and> (g i k \<Turnstile>\<^sub>i rva f2)) "
   by simp 
  have d2: "((nrev g) (the_enat (nlength g) - j) (nlength g - enat i) \<Turnstile>\<^sub>i  f1 dee f2 ) =
              (\<exists>k\<le>the_enat (nlength g) - j. ((nrev g) k (enat (the_enat (nlength g) - j)) \<Turnstile>\<^sub>i f1) \<and>
                 ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2))  " 
    by simp
  have d3: " (\<exists>k\<ge>enat j. k \<le> nlength g \<and> (g j k \<Turnstile>\<^sub>i rva f1) \<and> (g i k \<Turnstile>\<^sub>i rva f2)) \<Longrightarrow>
              (\<exists>k\<le>the_enat (nlength g) - j. ((nrev g) k (enat (the_enat (nlength g) - j)) \<Turnstile>\<^sub>i f1) \<and>
                 ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2)) " 
   proof -
    assume da0: "(\<exists>k\<ge>enat j. k \<le> nlength g \<and> (g j k \<Turnstile>\<^sub>i rva f1) \<and> (g i k \<Turnstile>\<^sub>i rva f2))"
    show "(\<exists>k\<le>the_enat (nlength g) - j. ((nrev g) k (enat (the_enat (nlength g) - j)) \<Turnstile>\<^sub>i f1) \<and>
                 ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2))"
    proof -
     obtain k where d100: "k\<ge>enat j \<and> k \<le> nlength g \<and> (g j k \<Turnstile>\<^sub>i rva f1) \<and> (g i k \<Turnstile>\<^sub>i rva f2)"
     using da0 by blast
     have d103: "(the_enat(nlength g -k)) \<le>the_enat (nlength g) - j" 
     by (metis d100 d_d.prems(1) diff_le_mono2 enat_ile enat_ord_simps(1) idiff_enat_enat nfinite_conv_nlength_enat the_enat.simps)     
     have d101: "((nrev g) (the_enat(nlength g -k)) (enat (the_enat (nlength g) - j)) \<Turnstile>\<^sub>i  f1)"
       using d_d.IH(1)[of g j _]  d100
       by (metis d_d.prems(1) enat_ile enat_ord_simps(1) idiff_enat_enat nfinite_conv_nlength_enat
        the_enat.simps) 
      have d102: "((nrev g) (the_enat(nlength g -k)) (nlength g - enat i) \<Turnstile>\<^sub>i f2)" 
     using d100 d_d.IH(2) d_d.prems(1) d_d.prems(2) enat_ile nfinite_nlength_enat by fastforce
     show ?thesis
     using d101 d102 d103 by blast 
    qed
   qed
  have d4: "(\<exists>k\<le>the_enat (nlength g) - j. ((nrev g) k (enat (the_enat (nlength g) - j)) \<Turnstile>\<^sub>i f1) \<and>
                 ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2)) \<Longrightarrow>
            (\<exists>k\<ge>enat j. k \<le> nlength g \<and> (g j k \<Turnstile>\<^sub>i rva f1) \<and> (g i k \<Turnstile>\<^sub>i rva f2))"
   proof -
    assume da0: "(\<exists>k\<le>the_enat (nlength g) - j. ((nrev g) k (enat (the_enat (nlength g) - j)) \<Turnstile>\<^sub>i f1) \<and>
                 ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2))"
    show "(\<exists>k\<ge>enat j. k \<le> nlength g \<and> (g j k \<Turnstile>\<^sub>i rva f1) \<and> (g i k \<Turnstile>\<^sub>i rva f2))"
    proof -
     obtain k where d200: "k\<le>the_enat (nlength g) - j \<and> ((nrev g) k (enat (the_enat (nlength g) - j)) \<Turnstile>\<^sub>i f1) \<and>
                 ((nrev g) k (nlength g - enat i) \<Turnstile>\<^sub>i f2)"
      using da0 by blast
     have d201: "j \<le> the_enat (nlength g) - k" 
     using d200 d_d.prems(1) d_d.prems(3) nfinite_nlength_enat by fastforce
     have d202: " enat (the_enat (nlength g) - k) \<le> nlength g" 
     by (metis diff_le_self enat_ord_simps(1) enat_ord_simps(3) enat_the_enat)
     have d203: "the_enat (nlength g) - (the_enat (nlength g) - k) = k "
     by (meson d200 diff_diff_cancel diff_le_self dual_order.trans) 
     have d204: "(g j (enat (the_enat(nlength g) - k)) \<Turnstile>\<^sub>i rva f1) " 
       using  d_d.IH(1)[of g j "(the_enat(nlength g) - k)" ]
       by (metis d200 d201 d202 d_d.prems(1) diff_diff_cancel diff_is_0_eq' enat.distinct(1) enat_the_enat idiff_enat_enat nfinite_conv_nlength_enat nle_le)       
     have d205: "(g i (nlength g - k) \<Turnstile>\<^sub>i rva f2)" 
       using  d_d.IH(2)[of g i "(the_enat(nlength g) - k)" ]
       by (metis d200 d201 d202 d203 d_d.prems(1) d_d.prems(2) dual_order.trans enat.distinct(1) enat_the_enat idiff_enat_enat nfinite_conv_nlength_enat) 
       show ?thesis
       by (metis d201 d202 d204 d205 d_d.prems(1) enat_ord_simps(1) idiff_enat_enat nfinite_conv_nlength_enat the_enat.simps)
    qed
   qed 
  show ?thesis using d1 d2 d3 d4 by blast
 qed  
next
case (t_d f1 f2)
then show ?case 
  proof -
   have t1: "(g i j \<Turnstile>\<^sub>i rva ( f1 tee f2 )) = 
            (\<exists>k\<le>i. (g k (enat i) \<Turnstile>\<^sub>i rva f1) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f2))" 
     by simp
   have t2: "((nrev g) (the_enat (nlength g) - j) (nlength g - enat i) \<Turnstile>\<^sub>i  f1 tee f2 ) = 
            (\<exists>k\<ge>nlength g - enat i. k \<le> nlength g \<and> ((nrev g) (the_enat (nlength g - enat i)) k \<Turnstile>\<^sub>i f1) \<and>
              ((nrev g) (the_enat (nlength g) - j) k \<Turnstile>\<^sub>i f2)) " 
     apply simp 
     apply auto 
     by (metis enat_ord_code(5) enat_the_enat nfinite_conv_nlength_enat t_d.prems(1))

   have t3: "(\<exists>k\<le>i. (g k (enat i) \<Turnstile>\<^sub>i rva f1) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f2)) \<Longrightarrow>
             (\<exists>k\<ge>nlength g - enat i. k \<le> nlength g \<and> ((nrev g) (the_enat (nlength g - enat i)) k \<Turnstile>\<^sub>i f1) \<and>
              ((nrev g) (the_enat (nlength g) - j) k \<Turnstile>\<^sub>i f2))"
   proof -
    assume ta0: "(\<exists>k\<le>i. (g k (enat i) \<Turnstile>\<^sub>i rva f1) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f2))"
    show "(\<exists>k\<ge>nlength g - enat i. k \<le> nlength g \<and> ((nrev g) (the_enat (nlength g - enat i)) k \<Turnstile>\<^sub>i f1) \<and>
              ((nrev g) (the_enat (nlength g) - j) k \<Turnstile>\<^sub>i f2))"
    proof -
     obtain k where t100: "k\<le>i \<and> (g k (enat i) \<Turnstile>\<^sub>i rva f1) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f2)"
     using ta0 by blast
     have t101: "((nrev g) (the_enat (nlength g - enat i)) (nlength g - k) \<Turnstile>\<^sub>i f1) " 
        by (metis enat_ord_simps(1) idiff_enat_enat nfinite_conv_nlength_enat order_subst2 t100 t_d.IH(1) t_d.prems(1) t_d.prems(2) t_d.prems(3) the_enat.simps)
     have t102: "((nrev g) (the_enat (nlength g) - j) (nlength g -k) \<Turnstile>\<^sub>i f2) " 
        by (meson order.trans t100 t_d.IH(2) t_d.prems(1) t_d.prems(2) t_d.prems(3))
     have t103: "(nlength g -k) \<ge>nlength g - enat i " using t100 
     by (metis diff_le_mono2 enat_ord_simps(1) idiff_enat_enat nfinite_conv_nlength_enat t_d.prems(1))
     have t104: "(nlength g -k) \<le> nlength g  "   using t100 
     by (metis enat.distinct(2) enat_add_sub_same enat_le_plus_same(2) enat_minus_mono1)
     show ?thesis 
      using t101 t102 t103 t104 by blast
     qed
    qed 
   have t4: "(\<exists>k\<ge>nlength g - enat i. k \<le> nlength g \<and> ((nrev g) (the_enat (nlength g - enat i)) k \<Turnstile>\<^sub>i f1) \<and>
              ((nrev g) (the_enat (nlength g) - j) k \<Turnstile>\<^sub>i f2)) \<Longrightarrow>
             (\<exists>k\<le>i. (g k (enat i) \<Turnstile>\<^sub>i rva f1) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f2))"
    proof -
     assume ta0: " (\<exists>k\<ge>nlength g - enat i. k \<le> nlength g \<and> ((nrev g) (the_enat (nlength g - enat i)) k \<Turnstile>\<^sub>i f1) \<and>
              ((nrev g) (the_enat (nlength g) - j) k \<Turnstile>\<^sub>i f2))"
     show "(\<exists>k\<le>i. (g k (enat i) \<Turnstile>\<^sub>i rva f1) \<and> (g k (enat j) \<Turnstile>\<^sub>i rva f2))"
     proof -
      obtain k where t200: "k\<ge>nlength g - enat i \<and> k \<le> nlength g \<and> ((nrev g) (the_enat (nlength g - enat i)) k \<Turnstile>\<^sub>i f1) \<and>
              ((nrev g) (the_enat (nlength g) - j) k \<Turnstile>\<^sub>i f2)"
      using ta0 by blast
      have t201: " (the_enat(nlength g) -k) \<le> i" using t200 
      using enat_ile nfinite_nlength_enat t_d.prems(1) by fastforce
      have t202: "nlength g - enat (the_enat (nlength g - k)) = k " 
      by (metis add.commute enat_add_sub_same enat_ord_code(4) enat_the_enat leD le_iff_add t200)
      have t203: "the_enat (nlength g) - i = (the_enat (nlength g - enat i))" 
      by (metis enat.distinct(2) enat.inject enat_the_enat idiff_enat_enat nfinite_conv_nlength_enat t_d.prems(1))
      have t2031: "((nrev g) (the_enat (nlength g) - i) (nlength g - enat (the_enat (nlength g - k))) \<Turnstile>\<^sub>i f1)" 
       using t200 t202 t203 by presburger
      have t2032: "nfinite g \<and> the_enat (nlength g - k) \<le> i \<and> enat i \<le> nlength g"  
       by (metis enat_ile enat_ord_simps(1) nfinite_conv_nlength_enat order.trans t201 t_d.prems(1) t_d.prems(2) t_d.prems(3) the_enat.simps) 
      have t204: "(g (the_enat(nlength g -k))  (enat i) \<Turnstile>\<^sub>i rva f1) "
        using t_d.IH(1)[of g "(the_enat(nlength g -k))"  i] using t200 t201 t202 t2031 t2032 by blast
       have t205: " (g (the_enat(nlength g -k)) (enat j) \<Turnstile>\<^sub>i rva f2)"
          using t_d.IH(2)[of g "(the_enat(nlength g -k))" j] 
          using t200 t202 t2032 t_d.prems(2) t_d.prems(3) by auto
       show ?thesis
      using t201 t202 t203 t204 t205 
      using t2032 by blast
     qed
    qed
   show ?thesis using t1 t2 t3 t4 by blast
  qed  
qed

lemma rva_atom: 
assumes "nfinite g" 
        "i \<le> j "
        " j \<le> nlength g"
shows " (g i j \<Turnstile>\<^sub>i (rva ($p))) = (g i j \<Turnstile>\<^sub>i fin ($p)) "
using assms by simp

lemma rva_fin_atom: 
assumes  "nfinite g" 
        "i \<le> j "
        " j \<le> nlength g" 
shows " (g i j \<Turnstile>\<^sub>i (rva ( (fin ($p))))) = (g i j \<Turnstile>\<^sub>i $p) " 
using assms unfolding fin_d_def diamond_d_def iand_d_def ior_d_def inot_d_def itrue_d_def
apply simp  
unfolding fin_d_def diamond_d_def iand_d_def ior_d_def inot_d_def itrue_d_def 
by simp

lemma rva_rva: 
 assumes  "nfinite g" 
        "i \<le> j "
        " j \<le> nlength g" 
shows " (g i j \<Turnstile>\<^sub>i (rva (rva f))) = (g i j \<Turnstile>\<^sub>i f) " 
using assms
proof (induction f arbitrary: g i j )
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case apply simp 
using rva_fin_atom 
using semantics_cdt.simps(2) by presburger
next
case (iimp_d f1 f2)
then show ?case by simp
next
case empty_d
then show ?case by simp
next
case (c_d f1 f2)
then show ?case apply simp  
by (meson dual_order.trans enat_ord_simps(1))
next
case (d_d f1 f2)
then show ?case apply simp 
by (meson dual_order.trans enat_ord_simps(1))
next
case (t_d f1 f2)
then show ?case apply simp 
by (metis (no_types, lifting) dual_order.trans the_enat.simps)
qed


lemma rva_valid_1: 
assumes "\<And> g i j. enat i \<le> j \<and> j \<le> nlength g \<and> nfinite g \<Longrightarrow> (g i j \<Turnstile>\<^sub>i f)"
        "enat i \<le> j " 
        " enat j \<le> nlength g "
        " nfinite g " 
shows "   (g i j \<Turnstile>\<^sub>i rva f)"
using assms using rva_sem[of g i j f] 
by (metis diff_diff_left diff_le_self enat_ord_simps(1) idiff_enat_enat le_iff_add nfinite_conv_nlength_enat nlength_nrev the_enat.simps)


lemma rva_valid_2: 
assumes "\<And> g i j. enat i \<le> j \<and> j \<le> nlength g \<and> nfinite g \<Longrightarrow> (g i j \<Turnstile>\<^sub>i rva f)"
        "enat i \<le> j " 
        " enat j \<le> nlength g "
        " nfinite g "   
shows "(g i j \<Turnstile>\<^sub>i f)" 
using assms  rva_sem[of ]
by simp (meson assms(2) rva_rva rva_valid_1) 

lemma rva_valid: 
 "(\<turnstile>\<^sub>f f) = (\<turnstile>\<^sub>f rva f) " 
unfolding valid_finite_def 
by (metis enat_ile nfinite_nlength_enat rva_valid_1 rva_valid_2)

   
end


